{
    VolMan3.pas - VOL file management routines for AGI V3 games
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

unit VolMan3;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, StdCtrls, Grids, Outline, DirOutln, ComCtrls, ViewEdit1, defines,
  ExtCtrls;

function ReadV3Resource(ResourceType1,ResourceID1:byte) : TResource;
procedure AddV3Resource(ResourceToAdd:TResource;ResType1,ResNum1:byte);
procedure RebuildV3VOLfiles;

implementation

uses ResourceWin1, VolRebuild1, Main1, VolMan2;

{*************************************************************}
function EncodeDecodeLogicMessages(LogicResource:TResource) : boolean;
{*************************************************************}
var MessageSectionStart : word;
    MessagesStart       : word;
    NumMessages         : byte;
    CurPos              : word;
begin
  EncodeDecodeLogicMessages := False;
  if LogicResource.Size > 2 then
  begin
    MessageSectionStart := LogicResource.Data^[0] + LogicResource.Data^[1]*256 + 2;
    if MessageSectionStart <= LogicResource.Size-1 then
    begin
      NumMessages := LogicResource.Data^[MessageSectionStart];
      if NumMessages = 0 then EncodeDecodeLogicMessages := True
      else if MessageSectionStart <= LogicResource.Size-5 then
      begin
        MessagesStart := MessageSectionStart + 3 + NumMessages*2;
        if MessagesStart <= LogicResource.Size-1 then
        begin
          for CurPos := MessagesStart to LogicResource.Size-1 do
            LogicResource.Data^[CurPos] := LogicResource.Data^[CurPos] XOR ord(EncryptionKey[(CurPos-MessagesStart) mod 11 + 1]);
          EncodeDecodeLogicMessages := True;
        end;
      end;
    end;
  end;
end;


{*************************************************************}
function DecompressResource(CompressedResource:TResource;DecompSize:word) : TResource;
{*************************************************************}
const MAXBITS = 12;
      TABLE_SIZE = 18041;
      START_BITS = 9;

type prefix_code_array = array[0..TABLE_SIZE] of word;
     append_character_array = array[0..TABLE_SIZE] of char;
     prefix_code_type = prefix_code_array;
     append_character_type = append_character_array;

var DecompResource : TResource;
    InPos, OutPos : word;

    BITS, MAX_VALUE, MAX_CODE : integer;
    prefix_code : prefix_code_type;
    append_character : append_character_type;
    input_bit_count : integer;
    input_bit_buffer : longint;

    next_code, new_code, old_code : word;

  function ReadInputChar : char;
  begin
    if InPos < CompressedResource.Size then
    begin
      ReadInputChar := chr(CompressedResource.Data^[InPos]);
      Inc(InPos);
    end
    else
    begin
      ReadInputChar := chr(0);
      ShowMessage('Error: Read beyond end of compressed resource.');
      Halt(1);
    end;
  end;

  procedure WriteOutputChar(TheChar:char);
  begin
    if OutPos < DecompSize then
    begin
      DecompResource.Data^[OutPos] := ord(TheChar);
      Inc(OutPos);
    end;
  end;


  function SetBITS(Value:integer) : boolean;
  // Purpose: To adjust the number of bits used to store codes to the value passed in.
  begin
   if (Value = MAXBITS) then SetBITS := True
   else
    begin
     BITS := Value;
     MAX_VALUE := (1 SHL BITS) - 1;
     MAX_CODE := MAX_VALUE - 1;
     SetBITS := False;
    end;
  end;

  function decode_string(AString:string; code: word) : string;
  var i : integer;
  begin
    i := 0;
    while (code > 255) do
    begin
      if code > TABLE_SIZE then
      begin
        ShowMessage('FATAL ERROR: Invalid code ('+IntToStr(code)+') in decode_string.');
        Halt(1);
      end
      else
      begin
        AString := concat(AString, append_character[code]);
        code := prefix_code[code];
        i := i + 1;
        if (i > 4000) then
        begin
          ShowMessage('Fatal error during code expansion.');
          Halt(1);
        end;
      end;
    end;
    AString := concat(AString, Chr(ord(code)));
    decode_string := AString;
  end;

  function input_code : word;
  var return_value:word;
      CharacterInput : char;
      AWord:word;
  begin
    while (input_bit_count <= 24) and (InPos < CompressedResource.Size) do
    begin
      CharacterInput := ReadInputChar;
      AWord := ORD(CharacterInput);
      input_bit_buffer := input_bit_buffer or (AWord SHL input_bit_count);
      input_bit_count := input_bit_count + 8;
     end;
     return_value := (input_bit_buffer and $7FFF) MOD (1 SHL BITS);
     input_bit_buffer := input_bit_buffer SHR BITS;
     input_bit_count := input_bit_count - BITS;
     input_code := return_value;
  end;

  procedure expand;
  var character, counter, index : integer;
      AString : string;
      AChar :char;
      BITSFull : boolean;
  begin
    BITSFull := SetBITS(START_BITS);   {Set initial value of BITS}

    next_code := 257;                {Next available code to define. SIERRA start at 257}
                                     {256 is apparently the table flush code}
    counter := 0;                    {counter is a pacifier}

    old_code := input_code;          {Read in the first code.}
    AChar := chr(ord(old_code));
    character := old_code;           {Initialise the character variable.}
    new_code := input_code;

    while (OutPos < DecompSize) and (new_code <> $101) do
    begin
      if (counter = 1000) then    { Make it look like somethings being done }
        counter := 0;
      counter := counter + 1;

      if (new_code = $100) then   { Restart LZW process }
      begin
        next_code := 258;
        BITSFull := SetBITS(START_BITS);
        old_code := input_code;          {Read in the first code.}
        AChar := chr(ord(old_code));
        character := old_code;           {Initialise the character variable.}
        WriteOutputChar(AChar);            {Send out the first code.}
        new_code := input_code;
      end
      else
      begin
        if (new_code >= next_code) then
          AString := decode_string(Chr(character), old_code)
        else
          AString := decode_string('', new_code);
        character := ord(AString[length(AString)]);
        for index := length(AString) downto 1 do
          WriteOutputChar(AString[index]);
        if (next_code > MAX_CODE) then
          BITSFull := SetBITS(BITS + 1);
        prefix_code[next_code] := old_code;
        append_character[next_code] := Chr(character);
        next_code := next_code + 1;
        old_code := new_code;
//        if InPos = CompressedResource.Size - 5 then
//          ShowMessage(AString);
        AString := '';
        new_code := input_code;
      end;
    end;
  end;

begin
  DecompResource.Size := DecompSize;
  GetMem(DecompResource.Data,DecompResource.Size);
  InPos := 0;
  OutPos := 0;

  input_bit_buffer := 0;
  input_bit_count := 0;
  Expand;

  FreeMem(CompressedResource.Data,CompressedResource.Size);
  DecompressResource.Size := DecompResource.Size;
  DecompressResource.Data := DecompResource.Data;
end;

{*************************************************************}
function DecompressPicture(CompressedResource:TResource) : TResource;
{*************************************************************}
var DecompPicture  : TResource;
    DecompResource : TResource;
    InPos, OutPos  : word;
    CurByte        : byte;
    BufferByte     : byte;
    IsOffset       : boolean;
    ThisByte       : byte;

  function ReadByte : byte;
  begin
    if InPos < CompressedResource.Size then
    begin
      ReadByte := CompressedResource.Data[InPos];
      Inc(InPos);
    end
    else ReadByte := 0;
  end;

  procedure WriteByte(TheByte:byte);
  begin
    DecompPicture.Data[OutPos] := TheByte;
    Inc(OutPos);
    if OutPos >= MaxResourceSize then
    begin
      ShowMessage('FATAL ERROR: Decompressed picture too big (>'+IntToStr(MaxResourceSize)+' bytes)');
      Halt;
    end;
  end;

begin
  DecompPicture.Size := MaxResourceSize;
  GetMem(DecompPicture.Data,DecompPicture.Size);
  InPos := 0;
  OutPos := 0;
  IsOffset := False;
  BufferByte := $00;
  repeat
  begin
    CurByte := ReadByte;
    if IsOffset then
    begin
      BufferByte := BufferByte + CurByte SHR 4;
      ThisByte := BufferByte;
      BufferByte := CurByte SHL 4;
    end
    else ThisByte := CurByte;
    WriteByte(ThisByte);


    if ((ThisByte = $F0) or (ThisByte = $F2)) and (InPos < CompressedResource.Size - 1) then
    begin
      if IsOffset then
      begin
        WriteByte(BufferByte SHR 4);
        IsOffset := False;
      end
      else
      begin
        CurByte := ReadByte;
        WriteByte(CurByte SHR 4);
        BufferByte := CurByte SHL 4;
        IsOffset := True;
      end;
    end;

  end; until InPos >= CompressedResource.Size;


  DecompResource.Size := OutPos;
  GetMem(DecompResource.Data,DecompResource.Size);
  Move(DecompPicture.Data^,DecompResource.Data^,DecompResource.Size);
  FreeMem(DecompPicture.Data,DecompPicture.Size);
  FreeMem(CompressedResource.Data,CompressedResource.Size);
  DecompressPicture.Data := DecompResource.Data;
  DecompressPicture.Size := DecompResource.Size;
end;


{*************************************************************}
function ReadV3Resource(ResourceType1,ResourceID1:byte) : TResource;
{*************************************************************}
const NoError             : byte = 0;
      FileError           : byte = 1;
      InvalidResError     : byte = 2;
      InvalidLocError     : byte = 3;
      DecompError         : byte = 4;

var ResourceFile          : file of byte;
    lsbyte,msbyte         : byte;
    ResourceData          : TResource;
    ErrorType             : byte;
    DecompSize            : word;
    DecompResource        : TResource;
    VolNumByte            : byte;
    ResourceIsPicture     : boolean;

begin
  ErrorType := NoError;
  AssignFile(ResourceFile,GameDir+ResourceInfo[ResourceType1,ResourceID1].Filename);
  {$I-}
  Reset(ResourceFile);
  {$I+}
  if IOResult <> 0 then
  begin
    ErrorType := FileError;
  end;
  if ErrorType = NoError then
    if ResourceInfo[ResourceType1,ResourceID1].Loc > FileSize(ResourceFile) then
      ErrorType := InvalidLocError;
  if ErrorType = NoError then
  begin
    Seek(ResourceFile,ResourceInfo[ResourceType1,ResourceID1].Loc);
    Read(ResourceFile,msbyte);
    Read(ResourceFile,lsbyte);
  end;
  if ErrorType = NoError then
  if (not ((msbyte=$12) and (lsbyte=$34))) then
    ErrorType := InvalidResError;
  if ErrorType = NoError then
  begin
    Read(ResourceFile,VolNumByte);
    ResourceIsPicture := ((VolNumByte AND $80) = $80);
    Read(ResourceFile,lsbyte,msbyte);
    DecompSize := msbyte * 256 + lsbyte;
    Read(ResourceFile,lsbyte,msbyte);
    ResourceData.Size := msbyte * 256 + lsbyte;
    GetMem(ResourceData.Data,ResourceData.Size);
    BlockRead(ResourceFile,ResourceData.Data^,ResourceData.Size);
  end;

  if ErrorType = NoError then
  begin
    if ResourceIsPicture then
    begin
      DecompResource := DecompressPicture(ResourceData);
      ReadV3Resource.Data := DecompResource.Data;
      ReadV3Resource.Size := DecompResource.Size;
    end
    else if ResourceData.Size <> DecompSize then
    begin
      DecompResource := DecompressResource(ResourceData,DecompSize);  // this frees memory from ResourceData and allocates memory to DecompResource
      if ResourceType1 = LOGIC then EncodeDecodeLogicMessages(DecompResource);
      ReadV3Resource.Data := DecompResource.Data;
      ReadV3Resource.Size := DecompResource.Size;
    end
    else
    begin  // no compression
      ReadV3Resource.Data := ResourceData.Data;
      ReadV3Resource.Size := ResourceData.Size;
    end;
  end;
{  if ErrorType = FileError then
  begin
    ShowMessage('Error reading file '+GameDir+ResourceInfo[ResourceType1,ResourceID1].Filename);
    Halt;
  end;
  if ErrorType = InvalidResError then
    begin
      ShowMessage('Error: Resource signature not found');
      Halt;
    end;
  if ErrorType = InvalidLocError then
    begin
      ShowMessage('Error: Specified resource location is past end of file');
      Halt;
    end;
  if ErrorType = DecompError then
    begin
      ShowMessage('Error: Could not decompress resource');
      Halt;
    end;}
  if ErrorType <> FileError then Close(ResourceFile);
  if ErrorType <> NoError then ReadV3Resource.Size := 0;
end;

{*************************************************************}
procedure AddV3Resource(ResourceToAdd:TResource;ResType1,ResNum1:byte);
{*************************************************************}
const NoError            : byte = 0;
      VolFileAccessError : byte = 2;
      DirFileAccessError : byte = 3;
      PVTooBigError      : byte = 5;


var VolFile         : file;
    DirFile         : file of byte;
    ResHeader       : array[1..7] of byte;
    ErrorType       : byte;
    DirByte         : array[1..3] of byte;
    NewResource     : boolean;
    tmp1            : integer;
    tmp2            : integer;
    PatchVol        : byte;
    DirOffset       : word;
    lsbyte, msbyte  : byte;

  procedure OpenPatchVol;
  var PatchVolFilename   : string;
  begin
    PatchVolFilename := GameID + 'VOL.' + IntToStr(PatchVol);
    AssignFile(VolFile,GameDir+PatchVolFilename);
    {$I-}
    if FileExists(GameDir+PatchVolFilename) then Reset(VolFile,1)
    else Rewrite(VolFile,1);
    {$I+}
    if IOResult <> 0 then
      begin
        ErrorType := VolFileAccessError;
        ShowMessage('Error accessing file '+PatchVolFilename);
      end;
  end;

  procedure RewriteDirFile;
  type TDirData      = array[0..767] of byte;
  var DirFile2       : file;
      Offset         : array[0..3] of word;
      Size           : array[0..3] of word;
      CurResType     : word;
      lsbyte, msbyte : byte;
      DirData        : array[0..3] of TDirData;
      OffsetArray    : array[0..7] of byte;
  begin
    Seek(DirFile,0);
    for CurResType := 3 downto 0 do
    begin
      Seek(DirFile,CurResType*2);
      Read(DirFile,lsbyte);
      Read(DirFile,msbyte);
      Offset[CurResType] := lsbyte + msbyte*256;
      if CurResType = 3 then Size[CurResType] := FileSize(DirFile) - Offset[CurResType]
      else Size[CurResType] := Offset[CurResType+1] - Offset[CurResType];
      if (Offset[CurResType]>FileSize(DirFile)) or (Size[CurResType]>768) or (Offset[CurResType]+Size[CurResType]>FileSize(DirFile)) then
      begin
        ShowMessage('FATAL ERROR: DIR file is invalid.');
        Halt;
      end;
    end;
    Close(DirFile);
    Assign(DirFile2,GameDir+GameID+'DIR');
    Reset(DirFile2,1);
    for CurResType := 0 to 3 do
    begin
      Seek(DirFile2,Offset[CurResType]);
      FillChar(DirData[CurResType],$FF,768);
      BlockRead(DirFile2,DirData[CurResType],Size[CurResType]);
    end;
    CloseFile(DirFile2);
    Rewrite(DirFile2,1);
    OffsetArray[0] := 8;
    OffsetArray[1] := 0;
    Offset[0] := 8;
    for CurResType := 1 to 3 do
    begin
      Offset[CurResType] := Offset[CurResType-1] + 768;
      OffsetArray[CurResType*2] := Offset[CurResType] mod 256;
      OffsetArray[CurResType*2+1] := Offset[CurResType] div 256;
    end;
    BlockWrite(DirFile2,OffsetArray,8);
    for CurResType := 0 to 3 do
    begin
      BlockWrite(DirFile2,DirData[CurResType],768);
    end;
    CloseFile(DirFile2);
    Reset(DirFile);
  end;

begin
  ErrorType := NoError;
  PatchVol := 0;
{open the patch volume, if we can}
  if ErrorType = NoError then
    OpenPatchVol;
{open the DIR file, if we can}
  if ErrorType = NoError then
  begin
    AssignFile(DirFile,GameDir+GameID+'DIR');
    {$I-}
    Reset(DirFile);
    {$I+}
    if IOResult <> 0 then
      begin
        ErrorType := DirFileAccessError;
        ShowMessage('Error accessing file '+GameID+'DIR');
      end;
  end;
  repeat
  begin
    if (ErrorType = NoError) and (FileSize(VolFile) + ResourceToAdd.Size > 1048000) then
    begin
      CloseFile(VolFile);
      PatchVol := PatchVol + 1;
      OpenPatchVol;
    end;
  end; until (ErrorType <> NoError) or (FileSize(VolFile) + ResourceToAdd.Size <= 1048000);
{write the resource to the patch volume and update the DIR file}
  if ErrorType = NoError then
  begin
    RewriteDirFile;
    Seek(VolFile,FileSize(VolFile));
    DirByte[1] := PatchVol*$10 + FilePos(VolFile) div $10000;
    DirByte[2] := (FilePos(VolFile) mod $10000) div $100;
    DirByte[3] := FilePos(VolFile) mod $100;
    ResourceInfo[ResType1,ResNum1].Loc := FilePos(VolFile);
    ResourceInfo[ResType1,ResNum1].Filename := GameID+'VOL.'+IntToStr(PatchVol);
    NewResource := not ResourceInfo[ResType1,ResNum1].Exists;
    ResourceInfo[ResType1,ResNum1].Exists := True;
    ResHeader[1] := $12;
    ResHeader[2] := $34;
    ResHeader[3] := PatchVol;
    ResHeader[4] := ResourceToAdd.Size mod 256;
    ResHeader[5] := ResourceToAdd.Size div 256;
    ResHeader[6] := ResourceToAdd.Size mod 256;  // no compression so compressed size
    ResHeader[7] := ResourceToAdd.Size div 256;  // and uncompressed size are the same
    BlockWrite(VolFile,ResHeader,7);
    BlockWrite(VolFile,ResourceToAdd.Data^,ResourceToAdd.Size);
    if ResType1 = LOGIC then Seek(DirFile,0)
    else if ResType1 = PICTURE then Seek(DirFile,2)
    else if ResType1 = VIEW then Seek(DirFile,4)
    else Seek(DirFile,6);
    Read(DirFile,lsbyte); Read(DirFile,msbyte); DirOffset := lsbyte + msbyte*256;
    Seek(DirFile,DirOffset+ResNum1*3);
    Write(DirFile,DirByte[1]);
    Write(DirFile,DirByte[2]);
    Write(DirFile,DirByte[3]);
  end;
{finish off}
  if ErrorType <> VolFileAccessError then
  begin
    CloseFile(VolFile);
    if ErrorType <> DirFileAccessError then CloseFile(DirFile);
  end;

{update main window}
  if (NewResource) and (SelectedResourceType = ResType1) then
  begin
    if ResNum1 > ResourceNum[ResType1,NumResources[ResType1]-1] then tmp1 := NumResources[ResType1]
    else
    begin
      tmp1 := 0;
      if ResourceNum[ResType1,0] < ResNum1 then
      repeat tmp1 := tmp1 + 1;
      until ResourceNum[ResType1,tmp1] > ResNum1;
      for tmp2 := NumResources[ResType1] - 1 downto tmp1 do
        ResourceNum[ResType1,tmp2+1] := ResourceNum[ResType1,tmp2];
    end;
    NumResources[ResType1] := NumResources[ResType1] + 1;
    ResourceNum[ResType1,tmp1] := ResNum1;
    if (ResourceWin.ResourceList.Items.Count>0) and (tmp1<NumResources[ResType1]-1) then
      ResourceWin.ResourceList.Items.Insert(tmp1,ResourceName(ResType1,ResNum1))
    else ResourceWin.ResourceList.Items.Add(ResourceName(ResType1,ResNum1));
    ResourceWin.ResourceList.ItemIndex := tmp1;
    ResourceWin.ResourceListClick(ResourceWin);
  end;
  ResourceWin.SelectResource(ResType1,ResNum1);
  FreeMem(ResourceToAdd.Data,ResourceToAdd.Size);
end;


{*************************************************************}
procedure RebuildV3VOLfiles;
{*************************************************************}
const NoError            : byte = 0;
      FileAccessError    : byte = 1;
      MaxVOLFileSize     : longint = 1023*1024; {1023k - not quite 1024k, just to be safe}
      ResourceSig        : array[1..2] of byte = ($12,$34);


var RBResType            : byte;
    RBRes                : byte;
    RBResNum             : byte;
    TempResource         : TResource;
    TempStr              : String;
    VolFileNum           : byte;
    VolFileName          : String;
    VolFile              : file;
    NumVolFiles          : byte;
    DirFileName          : String;
    DirFile              : file of byte;
    ResHeader            : array[1..7] of byte;
    byte1, byte2, byte3  : byte;
    ErrorType            : byte;
    ReqdFile             : array[0..19] of byte;
    NumErrorsOccured     : word;
    DirOffset            : array[0..3] of word;
    NewResourceInfo      : array[0..3,0..255] of TResourceInfo;

begin
  VOLRebuildCanceled := False;
  VOLRebuild.NumErrorsDisplay.Caption := '0';
  VOLRebuild.ActionDisplay.Caption := 'Starting';
  VOLRebuild.ProgressGauge.MinValue := 0;
  VOLRebuild.ProgressGauge.MaxValue := NumResources[0] + NumResources[1] + NumResources[2] + NumResources[3];
  VOLRebuild.ProgressGauge.Progress := 0;
  VOLRebuild.Show;
  Main.Enabled := False;
  ResourceWin.Enabled := False;

  ErrorType := NoError;
  ChDir(GameDir);
  for VolFileNum := 0 to 15 do
  if FileExists(GameDir+GameID+'VOLNEW.'+IntToStr(VolFileNum)) then
  begin
    if not DeleteFile(GameDir+GameID+'VOLNEW.'+IntToStr(VolFileNum)) then
    begin
      ErrorType := FileAccessError;
      ShowMessage('Error deleting file "'+GameID+'VOLNEW.'+IntToStr(VolFileNum)+'"');
    end;
  end;
  if FileExists(GameDir+GameID+'DIR'+'.NEW') then
  begin
    if not DeleteFile(GameDir+GameID+'DIR'+'.NEW') then
    begin
      ErrorType := FileAccessError;
      ShowMessage('Error deleting file "'+GameID+'DIR'+'.NEW"');
    end;
  end;
  if ErrorType = NoError then
  begin
    ResHeader[1] := $12;
    ResHeader[2] := $34;
    NumErrorsOccured := 0;
    VolFileNum := 0;
    VolFileName := GameDir + GameID + 'VOLNEW.0';
    AssignFile(VolFile,VolFileName);
    Rewrite(VolFile,1);
    if ErrorType = NoError then
    begin
      DirFileName := GameID + 'DIR' + '.NEW';
      AssignFile(DirFile,DirFileName);
      {$I-}
      ReWrite(DirFile);
      {$I+}
      if IOResult <> 0 then
      begin
        ShowMessage('Error creating file '+DirFileName);
        ErrorType := FileAccessError;
      end
      else
      begin
        for RBResType := 0 to 3 do
        begin
          DirOffset[RBResType] := 8 + RbResType*$300;
          byte1 := DirOffset[RBResType] mod $100;
          byte2 := DirOffset[RBResType] div $100;
          Write(DirFile,byte1);
          Write(DirFile,byte2);
        end;
        byte1 := $FF;
        for RBResType := 0 to 3 do
          for RBResNum := 0 to 255 do
          begin
            Write(DirFile,byte1);
            Write(DirFile,byte1);
            Write(DirFile,byte1);
          end;
      end;
    end;
    for RBResType := 0 to 3 do
    begin
      if ErrorType = NoError then
      begin
        if NumResources[RBResType] > 0 then
        begin
          for RBRes := 0 to NumResources[RBResType] - 1 do
          begin
            if not VolRebuildCanceled then
            begin
              RBResNum := ResourceNum[RBResType,RBRes];
              VOLRebuild.ActionDisplay.Caption := 'Adding resource: ' + ResourceName(RBResType,RBResNum);
              VOLRebuild.Panel2.Repaint;
              Application.ProcessMessages;
              TempResource := ReadResource(RBResType,RBResNum);
              if TempResource.Size = 0 then
              begin
                NumErrorsOccured := NumErrorsOccured + 1;
                VOLRebuild.NumErrorsDisplay.Caption := IntToStr(NumErrorsOccured);
              end
              else
              begin
                if FileSize(VolFile) + TempResource.Size + 7 > MaxVOLFileSize then
                begin
                  Close(VolFile);
                  VolFileNum := VolFileNum + 1;
                  VolFileName := GameDir + GameID + 'VOLNEW.' + IntToStr(VolFileNum);
                  AssignFile(VolFile,VolFileName);
                  Rewrite(VolFile,1);
                end;
                NewResourceInfo[RBResType,RBResNum].Loc := FilePos(VolFile);
                NewResourceInfo[RbResType,RBResNum].Filename := GameID + 'VOL.'+IntToStr(VolFileNum);
                byte1 := VolFileNum*$10 + FilePos(VolFile) div $10000;
                byte2 := (FilePos(VolFile) mod $10000) div $100;
                byte3 := FilePos(VolFile) mod $100;
                ResHeader[3] := VolFileNum;
                ResHeader[4] := TempResource.Size mod 256;
                ResHeader[5] := TempResource.Size div 256;
                ResHeader[6] := TempResource.Size mod 256;  // resource is not compressed so compressed
                ResHeader[7] := TempResource.Size div 256;  // size and uncompressed size are the same
                BlockWrite(VolFile,ResHeader,7);
                BlockWrite(VolFile,TempResource.Data^,TempResource.Size);
                FreeMem(TempResource.Data,TempResource.Size);
                Seek(DirFile,DirOffset[RBResType]+RBResNum*3);
                Write(DirFile,byte1);
                Write(DirFile,byte2);
                Write(DirFile,byte3);
              end;
              VOLRebuild.ProgressGauge.Progress := VOLRebuild.ProgressGauge.Progress + 1;
            end; {if not VolRebuildCanceled}
          end; {for RBRes := 0 to NumResources[RBResType] - 1 do}
        end; {if NumResources[RBResType] > 0}
      end; {if ErrorType = NoError}
    end;
    CloseFile(VolFile);
    CloseFile(DirFile);
    NumVolFiles := VolFileNum + 1;
  end;

  if ErrorType = NoError then
  begin
    for VolFileNum := 0 to 15 do
    if (not VolRebuildCanceled) and FileExists(GameDir+GameID+'VOL.'+IntToStr(VolFileNum)) and (not CanAccessFile(GameDir+GameID+'VOL.'+IntToStr(VolFileNum))) then
    begin
      ShowMessage('Error replacing file "'+GameID+'VOL.'+IntToStr(VolFileNum)+'". Check that you are not running the game.');
      VolRebuildCanceled := True;
    end;
    if (not VolRebuildCanceled) and FileExists(GameDir+GameID+'DIR') and (not CanAccessFile(GameDir+GameID+'DIR')) then
    begin
      ShowMessage('Error replacing file "'+GameID+'DIR". Check that you are not running the game.');
      VolRebuildCanceled := True;
    end;

    if VolRebuildCanceled then
    begin  // delete NEW files
      for VolFileNum := 0 to 15 do
      if FileExists(GameDir+GameID+'VOLNEW.'+IntToStr(VolFileNum)) then
      begin
        if not DeleteFile(GameDir+GameID+'VOLNEW.'+IntToStr(VolFileNum)) then
        begin
          ShowMessage('Error deleting file "'+GameID+'VOLNEW.'+IntToStr(VolFileNum)+'"');
        end;
      end;
      for RBResType := 0 to 3 do
      if FileExists(GameDir+GameID+'DIR.NEW') then
      begin
        if not DeleteFile(GameDir+GameID+'DIR.NEW') then
        begin
          ShowMessage('Error deleting file "'+GameID+'DIR"');
        end;
      end;
    end {if VolRebuildCanceled}
    else
    begin  // delete old files and replace with NEW ones
      for VolFileNum := 0 to 15 do
      if FileExists(GameDir+GameID+'VOL.'+IntToStr(VolFileNum)) then
      begin
        if not DeleteFile(GameDir+GameID+'VOL.'+IntToStr(VolFileNum)) then
        begin
          ErrorType := FileAccessError;
          ShowMessage('Error deleting file "'+GameID+'VOL.'+IntToStr(VolFileNum)+'".');
        end;
      end;
      if FileExists(GameDir+GameID+'DIR') then
      begin
        if not DeleteFile(GameDir+GameID+'DIR') then
        begin
          ErrorType := FileAccessError;
          ShowMessage('Error deleting file "'+GameID+'DIR".');
        end;
      end;
      if ErrorType = NoError then
      begin
        for VolFileNum := 0 to NumVolFiles - 1 do
        begin
          AssignFile(VolFile,GameDir+GameID+'VOLNEW.'+IntToStr(VolFileNum));
          Rename(VolFile,GameID+'VOL.'+IntToStr(VolFileNum));
        end;
        begin
          AssignFile(DirFile,GameDir+GameID+'DIR'+'.NEW');
          Rename(DirFile,GameID+'DIR');
        end;
      end;
      if ErrorType = NoError then
      for RBResType := 0 to 3 do
        if NumResources[RBResType] > 0 then
        for RBResNum := 0 to NumResources[RBResType]-1 do
        begin
          ResourceInfo[RBResType,ResourceNum[RBResType,RBResNum]].Loc := NewResourceInfo[RBResType,ResourceNum[RBResType,RBResNum]].Loc;
          ResourceInfo[RBResType,ResourceNum[RBResType,RBResNum]].Filename := NewResourceInfo[RbResType,ResourceNum[RBResType,RBResNum]].Filename;
        end;
    end; {if not VolRebuildCanceled}
  end; {if ErrorType = NoError}

  Main.Enabled := True;
  ResourceWin.Enabled := True;
  VOLRebuild.Close;
end;


end.
