{
    VolMan2.pas - VOL file management routines for AGI V2 games
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

{$A+,B-,C+,D+,E-,F-,G+,H+,I+,J+,K-,L+,M-,N+,O-,P+,Q-,R-,S-,T-,U-,V+,W-,X+,Y+,Z1}
{$MINSTACKSIZE $00004000}
{$MAXSTACKSIZE $00100000}
{$IMAGEBASE $00400000}
{$APPTYPE GUI}
{Volume managing routines for AGI version 2}
unit VolMan2;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, StdCtrls, Grids, Outline, DirOutln, ComCtrls, ViewEdit1, defines,
  ExtCtrls;

function CanAccessFile(Filename:string) : boolean;
function ReadResource(ResourceType1,ResourceID1:byte) : TResource; // v3 support
function ResourceIndex(ResType,ResNum:byte) : integer;
function GetResourceSize(ResType,ResNum:byte) : integer;  // v3 support
procedure ExtractResource;
procedure AddResourceAsk;
procedure AddResource(ResourceToAdd:TResource;ResType1,ResNum1:byte);  // v3 support
function DeleteResource(ResType1,ResNum1:byte;Confirm:boolean) : boolean;  // v3 support
procedure RenumberResource(ResType:Byte;OldNum,NewNum:Byte);  // v3 support
procedure RebuildVOLfiles;  // NO v3 support
function GetAGIVersionNumber : double;
function FindAGIV3GameID(dir:string) : string;

implementation

uses main1,GetResourceName1,VolRebuild1, ResourceWin1, PreviewWin1, VolMan3;

{*************************************************************}
function CanAccessFile(Filename:string) : boolean;
{*************************************************************}
var TestFile : file;
begin
  CanAccessFile := False;
  if FileExists(FileName) then
  begin
    AssignFile(TestFile,Filename);
    {$I-}
    Reset(TestFile,1);
    {$I+}
    If IOResult = 0 then
    begin
      CloseFile(TestFile);
      CanAccessFile := True;
    end;
  end;
end;

{*************************************************************}
function ReadResource(ResourceType1,ResourceID1:byte) : TResource;
{*************************************************************}
const NoError             : byte = 0;
      FileError           : byte = 1;
      InvalidResError     : byte = 2;
      InvalidLocError     : byte = 3;

var ResourceFile          : file of byte;
    lsbyte,msbyte         : byte;
    ResourceData          : TResource;
    ErrorType             : byte;

begin
  if GameIsV3 then ReadResource := ReadV3Resource(ResourceType1,ResourceID1)
  else
  begin
    ErrorType := NoError;
    AssignFile(ResourceFile,GameDir+ResourceInfo[ResourceType1,ResourceID1].Filename);
    {$I-}
    Reset(ResourceFile);
    {$I+}
    if IOResult <> 0 then
    begin
      ErrorType := FileError;
    end;
    if ErrorType = NoError then
      if ResourceInfo[ResourceType1,ResourceID1].Loc > FileSize(ResourceFile) then
        ErrorType := InvalidLocError;
    if ErrorType = NoError then
    begin
      Seek(ResourceFile,ResourceInfo[ResourceType1,ResourceID1].Loc);
      Read(ResourceFile,msbyte);
      Read(ResourceFile,lsbyte);
    end;
    if ErrorType = NoError then
    if (not ((msbyte=$12) and (lsbyte=$34))) then
      ErrorType := InvalidResError;
    if ErrorType = NoError then
    begin
      Seek(ResourceFile,ResourceInfo[ResourceType1,ResourceID1].Loc+3);
      Read(ResourceFile,lsbyte);
      Read(ResourceFile,msbyte);
      ResourceData.Size := msbyte * 256 + lsbyte;
      GetMem(ResourceData.Data,ResourceData.Size);
      BlockRead(ResourceFile,ResourceData.Data^,ResourceData.Size);
      ReadResource.Data := ResourceData.Data;
      ReadResource.Size := ResourceData.Size;
    end;
  {  if ErrorType = FileError then
    begin
      ShowMessage('Error reading file '+GameDir+ResourceInfo[ResourceType1,ResourceID1].Filename);
      Halt;
    end;
    if ErrorType = InvalidResError then
      begin
        ShowMessage('Error: Resource signature not found');
        Halt;
      end;
    if ErrorType = InvalidLocError then
      begin
        ShowMessage('Error: Specified resource location is past end of file');
        Halt;
      end;}
    if ErrorType <> FileError then Close(ResourceFile);
    if ErrorType <> NoError then ReadResource.Size := 0;
  end;
end;

{*************************************************************}
function ResourceIndex(ResType,ResNum:byte) : integer;
{returns the index number of a certain resource, if it is exits}
{i.e. ResourceNum[ResourceIndex(ResType,ResNum)] := ResNum}
{if the resource does not exits, it returns -1}
{*************************************************************}

var TmpIndex : byte;

begin
  ResourceIndex := -1;
  if NumResources[ResType] > 0 then
    for TmpIndex := 0 to NumResources[ResType]-1 do
      if ResourceNum[ResType,TmpIndex] = ResNum then ResourceIndex := TmpIndex;
end;

{*************************************************************}
function GetResourceSize(ResType,ResNum:byte) : integer;
{*************************************************************}
var VOLFile : file of byte;
    lsbyte, msbyte : byte;
begin
  GetResourceSize := -1;
  if ResourceInfo[ResType,ResNum].Exists then
  begin
    AssignFile(VOLFile,ResourceInfo[ResType,ResNum].Filename);
    {$I-}
    Reset(VOLFile);
    {$I+}
    if IOResult = 0 then
    begin
      if FileSize(VOLFile) >= ResourceInfo[ResType,ResNum].Loc+5 then
      begin
        Seek(VOLFile,ResourceInfo[ResType,ResNum].Loc);
        Read(VOLFile,lsbyte,msbyte);
        if (lsbyte = $12) and (msbyte = $34) then
        begin
          Read(VOLFile,lsbyte);
          Read(VOLFile,lsbyte,msbyte);
          GetResourceSize := msbyte*256 + lsbyte;
        end;
      end;
      Close(VOLFile);
    end;
  end;
end;

{*************************************************************}
procedure ExtractResource;
{*************************************************************}
var ExtractedFileName     : string;
    ExtractedFile         : File;
    TempResource          : TResource;
    ErrorType             : byte;

begin
  ExtractedFileName := ResourceName(SelectedResourceType,ResourceNum[SelectedResourceType,ResourceWin.ResourceList.ItemIndex]);
  if Pos(' ',ExtractedFilename) > 0 then ExtractedFilename[Pos(' ',ExtractedFilename)] := '.';
  TempResource := ReadResource(SelectedResourceType,ResourceNum[SelectedResourceType,ResourceWin.ResourceList.ItemIndex]);
  if TempResource.Size = 0 then
    ShowMessage('Error: Could not read resource.')
  else
  begin
    Main.ResourceExtractDialog.Filename := ExtractedFileName;
    if Main.ResourceExtractDialog.Execute then
    begin
      ExtractedFileName := Main.ResourceExtractDialog.Filename;
      if (not FileExists(ExtractedFilename)) or (FileExists(ExtractedFilename) and
          AskYesNo(ExtractFilename(ExtractedFilename),'File exists. Overwrite?')) then
      begin
        AssignFile(ExtractedFile,ExtractedFileName);
        {$I-}
        Rewrite(ExtractedFile,1);
        {$I+}
        if IOResult <> 0 then
          ShowMessage('Error: Could not write to file')
        else
        begin
          BlockWrite(ExtractedFile,TempResource.Data^,TempResource.Size);
          CloseFile(ExtractedFile);
        end;
      end;
    end;
    FreeMem(TempResource.Data,TempResource.Size);
  end;
end;

{*************************************************************}
procedure AddResourceAsk;
{*************************************************************}
var ThisResourceName      : string;
    ResourceTypeStr       : string;
    ResourceNumStr        : string;
    TempResourceNum       : integer;
    Code                  : integer;
    TempFile              : file;

begin
  GetResourceName.NameEntered := False;
  GetResourceName.ValidType := False;
  GetResourceName.ValidNumber := False;
  Main.ResourceAddDialog.Filename := '';
  Main.ResourceAddDialog.Execute;
  if Length(Main.ResourceAddDialog.Filename) > 0 then
  begin
    ResourceTypeStr := UpperCase(Copy(ExtractFileName(Main.ResourceAddDialog.Filename),1,Pos('.',ExtractFileName(Main.ResourceAddDialog.Filename))-1));
    GetResourceName.LogicRadio.Checked := False;
    GetResourceName.PictureRadio.Checked := False;
    GetResourceName.ViewRadio.Checked := False;
    GetResourceName.SoundRadio.Checked := False;
    if ResourceTypeStr = 'LOGIC' then
    begin
      GetResourceName.LogicRadio.Checked := True;
      GetResourceName.ResType := LOGIC;
      GetResourceName.ValidType := True;
    end
    else if (ResourceTypeStr = 'PIC') or (ResourceTypeStr = 'PICTURE') then
    begin
      GetResourceName.PictureRadio.Checked := True;
      GetResourceName.ResType := PICTURE;
      GetResourceName.ValidType := True;
    end
    else if ResourceTypeStr = 'VIEW' then
    begin
      GetResourceName.ViewRadio.Checked := True;
      GetResourceName.ResType := VIEW;
      GetResourceName.ValidType := True;
    end
    else if (ResourceTypeStr = 'SND') or (ResourceTypeStr = 'SOUND')then
    begin
      GetResourceName.SoundRadio.Checked := True;
      GetResourceName.ResType := SOUND;
      GetResourceName.ValidType := True;
    end;
    ResourceNumStr := copy(ExtractFileExt(Main.ResourceAddDialog.Filename),2,3);
    Val(ResourceNumStr,TempResourceNum,Code);
    if (Code=0) and (TempResourceNum>=0) and (TempResourceNum<=255) then
    begin
      GetResourceName.NumberEdit.Text := IntToStr(TempResourceNum);
      GetResourceName.ValidNumber := True;
    end
    else
      GetResourceName.NumberEdit.Text := '';
    if GetResourceName.ValidNumber and GetResourceName.ValidType then
      GetResourceName.NameDisplay.Caption := ResourceName(GetResourceName.ResType,GetResourceName.ResNumber)
    else GetResourceName.NameDisplay.Caption := '';
    AssignFile(TempFile,Main.ResourceAddDialog.FileName);
    {$I-}
    Reset(TempFile);
    {$I+}
    if IOResult <> 0 then
      ShowMessage('Error accessing file')
    else
    begin
      CloseFile(TempFile);
      GetResourceName.Filename := Main.ResourceAddDialog.Filename;
      GetResourceName.ShowModal;
    end;
  end;
end;

{*************************************************************}
procedure AddResource(ResourceToAdd:TResource;ResType1,ResNum1:byte);
{*************************************************************}
const NoError            : byte = 0;
      VolFileAccessError : byte = 2;
      DirFileAccessError : byte = 3;
      PVTooBigError      : byte = 5;


var VolFile         : file;
    DirFile         : file of byte;
    ResHeader       : array[1..5] of byte;
    ErrorType       : byte;
    DirByte         : array[1..3] of byte;
    NewResource     : boolean;
    tmp1            : integer;
    tmp2            : integer;
    PatchVol        : byte;

  procedure OpenPatchVol;
  var PatchVolFilename   : string;
  begin
    PatchVolFilename := 'VOL.' + IntToStr(PatchVol);
    AssignFile(VolFile,GameDir+PatchVolFilename);
    {$I-}
    if FileExists(GameDir+PatchVolFilename) then Reset(VolFile,1)
    else Rewrite(VolFile,1);
    {$I+}
    if IOResult <> 0 then
      begin
        ErrorType := VolFileAccessError;
        ShowMessage('Error accessing file '+PatchVolFilename);
      end;
  end;

begin
  if GameIsV3 then AddV3Resource(ResourceToAdd,ResType1,ResNum1)
  else
  begin
    ErrorType := NoError;
    PatchVol := 0;
  {open the patch volume, if we can}
    if ErrorType = NoError then
      OpenPatchVol;
  {open the DIR file, if we can}
    if ErrorType = NoError then
    begin
      AssignFile(DirFile,GameDir+ResTypeAbbrv[ResType1]+'DIR');
      {$I-}
      Reset(DirFile);
      {$I+}
      if IOResult <> 0 then
        begin
          ErrorType := DirFileAccessError;
          ShowMessage('Error accessing file '+ResTypeAbbrv[ResType1]+'DIR');
        end;
    end;
    repeat
    begin
      if (ErrorType = NoError) and (FileSize(VolFile) + ResourceToAdd.Size > 1048000) then
      begin
        CloseFile(VolFile);
        PatchVol := PatchVol + 1;
        OpenPatchVol;
      end;
    end; until (ErrorType <> NoError) or (FileSize(VolFile) + ResourceToAdd.Size <= 1048000);
  {write the resource to the patch volume and update the DIR file}
    if ErrorType = NoError then
    begin
      if ResNum1*3 > FileSize(DirFile) then
      begin
        DirByte[1] := $FF;
        Seek(DirFile,FileSize(DirFile));
        repeat Write(DirFile,DirByte[1])
        until FilePos(DirFile)=ResNum1*3;
      end;
      Seek(VolFile,FileSize(VolFile));
      DirByte[1] := PatchVol*$10 + FilePos(VolFile) div $10000;
      DirByte[2] := (FilePos(VolFile) mod $10000) div $100;
      DirByte[3] := FilePos(VolFile) mod $100;
      ResourceInfo[ResType1,ResNum1].Loc := FilePos(VolFile);
      ResourceInfo[ResType1,ResNum1].Filename := 'VOL.'+IntToStr(PatchVol);
      NewResource := not ResourceInfo[ResType1,ResNum1].Exists;
      ResourceInfo[ResType1,ResNum1].Exists := True;
      ResHeader[1] := $12;
      ResHeader[2] := $34;
      ResHeader[3] := PatchVol;
      ResHeader[4] := ResourceToAdd.Size mod 256;
      ResHeader[5] := ResourceToAdd.Size div 256;
      BlockWrite(VolFile,ResHeader,5);
      BlockWrite(VolFile,ResourceToAdd.Data^,ResourceToAdd.Size);
      Seek(DirFile,ResNum1*3);
      Write(DirFile,DirByte[1]);
      Write(DirFile,DirByte[2]);
      Write(DirFile,DirByte[3]);
    end;
  {finish off}
    if ErrorType <> VolFileAccessError then
    begin
      CloseFile(VolFile);
      if ErrorType <> DirFileAccessError then CloseFile(DirFile);
    end;

  {update main window}
    ResourceWin.ResTypeSelect.ItemIndex := ResType1;
    ResourceWin.ResTypeSelect.OnChange(Application);
    if (NewResource) and (SelectedResourceType = ResType1) then
    begin
      if ResNum1 > ResourceNum[ResType1,NumResources[ResType1]-1] then tmp1 := NumResources[ResType1]
      else
      begin
        tmp1 := 0;
        if ResourceNum[ResType1,0] < ResNum1 then
        repeat tmp1 := tmp1 + 1;
        until ResourceNum[ResType1,tmp1] > ResNum1;
        for tmp2 := NumResources[ResType1] - 1 downto tmp1 do
          ResourceNum[ResType1,tmp2+1] := ResourceNum[ResType1,tmp2];
      end;
      NumResources[ResType1] := NumResources[ResType1] + 1;
      ResourceNum[ResType1,tmp1] := ResNum1;
      if (ResourceWin.ResourceList.Items.Count>0) and (tmp1<NumResources[ResType1]-1) then
        ResourceWin.ResourceList.Items.Insert(tmp1,ResourceName(ResType1,ResNum1))
      else ResourceWin.ResourceList.Items.Add(ResourceName(ResType1,ResNum1));
      ResourceWin.ResourceList.ItemIndex := tmp1;
      ResourceWin.ResourceListClick(ResourceWin);
    end;
    ResourceWin.SelectResource(ResType1,ResNum1);
    FreeMem(ResourceToAdd.Data,ResourceToAdd.Size);
  end;  // if GameIsV3
end;

{*************************************************************}
function DeleteResource(ResType1,ResNum1:byte;Confirm:boolean) : boolean;
{*************************************************************}
var DirFile        : file of byte;
    DirFilename    : string;
    DirByte1       : byte;
    tmp1           : byte;
    lsbyte, msbyte : byte;
    DirOffset      : word;
    SelNum         : word;

begin
  DeleteResource := False;
  SelNum := ResourceIndex(ResType1,ResNum1);
  if SelNum < 0 then
    ShowMessage('Internal error: Could not locate resource in index. Resource not deleted.')
  else
  begin
    if (not Confirm) or (Application.MessageBox('Are you sure you want to delete this resource?','Confirm Delete',MB_YESNO) = IDYES) then
    begin
      if GameIsV3 then DirFilename := GameID+'DIR'
      else DirFilename := ResTypeAbbrv[ResType1]+'DIR';
      AssignFile(DirFile,GameDir+DirFilename);
      {$I-}
      Reset(DirFile);
      {$I+}
      if IOResult <> 0 then
        ShowMessage('Error reading file "'+DirFilename+'".')
      else
      begin
        DirByte1 := $FF;
        if GameIsV3 then
        begin
          if FileSize(DirFile) < 8 then
          begin ShowMessage('FATAL ERROR: '+DirFilename+' file invalid.'); Halt; end;
          if ResType1 = LOGIC then Seek(DirFile,0)
          else if ResType1 = PICTURE then Seek(DirFile,2)
          else if ResType1 = VIEW then Seek(DirFile,4)
          else Seek(DirFile,6);
          Read(DirFile,lsbyte);
          Read(DirFile,msbyte);
          DirOffset := lsbyte + msbyte*256;
          if FileSize(DirFile) < DirOffset+ResNum1*3+2 then
          begin ShowMessage('FATAL ERROR: '+DirFilename+' file invalid.'); Halt; end;
          Seek(DirFile,DirOffset+ResNum1*3);
        end
        else
        begin
          if FileSize(DirFile) < ResNum1*3+2 then
          begin ShowMessage('FATAL ERROR: '+DirFilename+' file invalid.'); Halt; end;
          Seek(DirFile,ResNum1*3);
        end;
        Write(DirFile,DirByte1);
        Write(DirFile,DirByte1);
        Write(DirFile,DirByte1);
        ResourceInfo[ResType1,ResNum1].Exists := False;
        if SelNum < NumResources[ResType1] - 1 then
          for tmp1 := SelNum to NumResources[ResType1] - 2 do
          begin
            ResourceNum[ResType1,tmp1] := ResourceNum[ResType1,tmp1+1];
          end;
        ResourceWin.ResourceList.Items.Delete(SelNum);
        NumResources[ResType1] := NumResources[ResType1] - 1;
        CloseFile(DirFile);
        ResourceWin.ResourceExtract.Enabled := False;
        ResourceWin.ResourceDelete.Enabled := False;
        ResourceWin.ResourceRenum.Enabled := False;
        ResourceWin.StatusBar1.Panels[0].Text := '';
        PreviewWin.ViewPreviewPanel.Visible := False;
        PreviewWin.PicPreviewPanel.Visible := False;
        PreviewWin.ErrorMessageLabel.Visible := False;
        if (ResType1 = LOGIC) and (FileExists(GameDir+'src\Logic'+IntToStr(ResNum1)+'.txt')) then
          DeleteFile(GameDir+SourceDir+'\Logic'+IntToStr(ResNum1)+'.txt');
        DeleteResource := True;
      end;
    end;
  end;
end;



{*************************************************************}
procedure RenumberResource(ResType:Byte;OldNum,NewNum:Byte);
{*************************************************************}
const EmptyByte        : byte = $FF;

var ResourceExists     : boolean;
    ConfirmRenumber    : boolean;
{    NewResourceIndex   : boolean;
    tmp1, tmp2         : byte;
    DirFilename        : string;
    DirFile            : file of byte;
    DirByte1, DirByte2, DirByte3 : byte;}
    TempResource      : TResource;
    OldResourceIndex  : integer;

begin
  // Note: I have changed this so it uses DeleteResource and AddResource.
  // I have left the old renumbering code in there in case I need it, but
  // it doesn't have version 3 support.
  if OldNum = NewNum then
    ShowMessage('Resource number not changed!')
  else
  begin
    if ResourceIndex(ResType,NewNum) = -1 then ResourceExists := False
    else ResourceExists := True;
    if ResourceExists then ConfirmRenumber := AskYesNo('Confirm Renumber','Resource '+ResourceName(ResType,NewNum)+' already exists. Replace it?')
    else ConfirmRenumber := True;
    if ConfirmRenumber then
    begin
      TempResource := ReadResource(ResType,OldNum);
      if TempResource.Size = 0 then
        ShowMessage('Error reading resource. Resource number not changed.')
      else
      begin
        if DeleteResource(ResType,OldNum,False) then
          AddResource(TempResource,ResType,NewNum);  // AddResource will free the memory allocated to TempResorce
      end;
(*      DirFilename := ResTypeAbbrv[ResType] + 'DIR';
      AssignFile(DirFile,GameDir+DirFilename);
      {$I-}
      Reset(DirFile);
      {$I+}
      if IOResult <> 0 then ShowMessage('Error: Could not open '+DirFilename)
      else
      begin
        ResourceInfo[ResType,NewNum] := ResourceInfo[ResType,OldNum];
        ResourceInfo[ResType,OldNum].Exists := False;
        if ResourceExists then
        begin
          ResourceWin.ResourceList.Items.Delete(ResourceIndex(ResType,OldNum));
          if ResourceIndex(ResType,OldNum) < NumResources[ResType]-1 then
          begin
            for tmp2 := ResourceIndex(ResType,OldNum) to NumResources[ResType]-2 do
              ResourceNum[ResType,tmp2] := ResourceNum[ResType,tmp2+1];
          end;
          NumResources[ResType] := NumResources[ResType] - 1;
          ResourceWin.ResourceList.ItemIndex := -1;
          ResourceWin.ResourceList.ItemIndex := ResourceIndex(ResType,NewNum);
        end {if ResourceExists}
        else
        begin
          tmp1 := 0;
          if NewNum > ResourceNum[ResType,NumResources[ResType]-1] then tmp1 := NumResources[ResType]
          else
            if ResourceNum[ResType,0] < NewNum then
              repeat tmp1 := tmp1 + 1;
              until ResourceNum[ResType,tmp1] > NewNum;
          for tmp2 := NumResources[ResType] - 1 downto tmp1 do
            ResourceNum[ResType,tmp2+1] := ResourceNum[ResType,tmp2];
          NumResources[ResType] := NumResources[ResType] + 1;
          ResourceNum[ResType,tmp1] := NewNum;
          ResourceWin.ResourceList.Items.Insert(tmp1,ResourceName(ResType,NewNum));
          ResourceWin.ResourceList.Items.Delete(ResourceIndex(ResType,OldNum));
          if ResourceIndex(ResType,OldNum) < NumResources[ResType]-1 then
          begin
            for tmp2 := ResourceIndex(ResType,OldNum) to NumResources[ResType]-1 do
              ResourceNum[ResType,tmp2] := ResourceNum[ResType,tmp2+1];
          end;
          NumResources[ResType] := NumResources[ResType] - 1;
          ResourceWin.ResourceList.ItemIndex := ResourceIndex(ResType,NewNum);
        end; {if not ResourceExists}
        if FileSize(DirFile) < NewNum then
        begin
          Seek(DirFile,FileSize(DirFile)*3);
          for tmp1 := FileSize(DirFile)*3 to NewNum*3 do
            Write(DirFile,EmptyByte);
        end;
        Seek(DirFile,OldNum*3);
        Read(DirFile,DirByte1); Read(DirFile,DirByte2); Read(DirFile,DirByte3);
        Seek(DirFile,OldNum*3);
        Write(DirFile,EmptyByte); Write(DirFile,EmptyByte); Write(DirFile,EmptyByte);
        Seek(DirFile,NewNum*3);
        Write(DirFile,DirByte1); Write(DirFile,DirByte2); Write(DirFile,DirByte3);
        CloseFile(DirFile);
      end; {if IOResult = 0} *)
    end; {if ConfirmRenumber}
  end; {if OldNum <> NewNum}
end;

{*************************************************************}
procedure RebuildVOLfiles;
{*************************************************************}
const NoError            : byte = 0;
      FileAccessError    : byte = 1;
      MaxVOLFileSize     : longint = 1023*1024; {1023k - not quite 1024k, just to be safe}
      ResourceSig        : array[1..2] of byte = ($12,$34);


var RBResType            : byte;
    RBRes                : byte;
    RBResNum             : byte;
    TempResource         : TResource;
    TempStr              : String;
    VolFileNum           : byte;
    VolFileName          : String;
    VolFile              : file;
    NumVolFiles          : byte;
    DirFileName          : String;
    DirFile              : file of byte;
    ResHeader            : array[1..5] of byte;
    byte1, byte2, byte3  : byte;
    ErrorType            : byte;
    ReqdFile             : array[0..19] of byte;
    NumErrorsOccured     : word;
    NewResourceInfo      : array[0..3,0..255] of TResourceInfo;

begin
  if GameIsV3 then RebuildV3VolFiles
  else
  begin
    VOLRebuildCanceled := False;
    VOLRebuild.NumErrorsDisplay.Caption := '0';
    VOLRebuild.ActionDisplay.Caption := 'Starting';
    VOLRebuild.ProgressGauge.MinValue := 0;
    VOLRebuild.ProgressGauge.MaxValue := NumResources[0] + NumResources[1] + NumResources[2] + NumResources[3];
    VOLRebuild.ProgressGauge.Progress := 0;
    VOLRebuild.Show;
    Main.Enabled := False;
    ResourceWin.Enabled := False;

    ErrorType := NoError;
    ChDir(GameDir);
    for VolFileNum := 0 to 15 do
    if FileExists(GameDir+'VOLNEW.'+IntToStr(VolFileNum)) then
    begin
      if not DeleteFile(GameDir+'VOLNEW.'+IntToStr(VolFileNum)) then
      begin
        ErrorType := FileAccessError;
        ShowMessage('Error deleting file "VOLNEW.'+IntToStr(VolFileNum)+'"');
      end;
    end;
    for RBResType := 0 to 3 do
    if FileExists(GameDir+ResTypeAbbrv[RBResType]+'DIR'+'.NEW') then
    begin
      if not DeleteFile(GameDir+ResTypeAbbrv[RBResType]+'DIR'+'.NEW') then
      begin
        ErrorType := FileAccessError;
        ShowMessage('Error deleting file "'+ResTypeAbbrv[RBResType]+'DIR'+'.NEW"');
      end;
    end;
    if ErrorType = NoError then
    begin
      ResHeader[1] := $12;
      ResHeader[2] := $34;
      NumErrorsOccured := 0;
      VolFileNum := 0;
      VolFileName := GameDir + 'VOLNEW.0';
      AssignFile(VolFile,VolFileName);
      Rewrite(VolFile,1);
      for RBResType := 0 to 3 do
      begin
        if ErrorType = NoError then
        begin
          DirFileName := ResTypeAbbrv[RBResType] + 'DIR' + '.NEW';
          AssignFile(DirFile,DirFileName);
          {$I-}
          ReWrite(DirFile);
          {$I+}
          if IOResult <> 0 then
          begin
            ShowMessage('Error creating file '+DirFileName);
            ErrorType := FileAccessError;
          end;
        end;
        if ErrorType = NoError then
        begin
          byte1 := $FF;
          if NumResources[RBResType] > 0 then
          begin
            for RBResNum := 0 to ResourceNum[RBResType, NumResources[RBResType]-1] do
            begin
              write(DirFile,byte1);
              write(DirFile,byte1);
              write(DirFile,byte1);
            end;
            Seek(DirFile,0);
            for RBRes := 0 to NumResources[RBResType] - 1 do
            begin
              if not VolRebuildCanceled then
              begin
                RBResNum := ResourceNum[RBResType,RBRes];
                VOLRebuild.ActionDisplay.Caption := 'Adding resource: ' + ResourceName(RBResType,RBResNum);
                VOLRebuild.Panel2.Repaint;
                Application.ProcessMessages;
                TempResource := ReadResource(RBResType,RBResNum);
                if TempResource.Size = 0 then
                begin
                  NumErrorsOccured := NumErrorsOccured + 1;
                  VOLRebuild.NumErrorsDisplay.Caption := IntToStr(NumErrorsOccured);
                end
                else
                begin
                  if FileSize(VolFile) + TempResource.Size + 5 > MaxVOLFileSize then
                  begin
                    Close(VolFile);
                    VolFileNum := VolFileNum + 1;
                    VolFileName := GameDir + 'VOLNEW.' + IntToStr(VolFileNum);
                    AssignFile(VolFile,VolFileName);
                    Rewrite(VolFile,1);
                  end;
                  NewResourceInfo[RBResType,RBResNum].Loc := FilePos(VolFile);
                  NewResourceInfo[RBResType,RBResNum].Filename := 'VOL.'+IntToStr(VolFileNum);
                  byte1 := VolFileNum*$10 + FilePos(VolFile) div $10000;
                  byte2 := (FilePos(VolFile) mod $10000) div $100;
                  byte3 := FilePos(VolFile) mod $100;
                  ResHeader[3] := VolFileNum;
                  ResHeader[4] := TempResource.Size mod 256;
                  ResHeader[5] := TempResource.Size div 256;
                  BlockWrite(VolFile,ResHeader,5);
                  BlockWrite(VolFile,TempResource.Data^,TempResource.Size);
                  FreeMem(TempResource.Data,TempResource.Size);
                  Seek(DirFile,RBResNum*3);
                  Write(DirFile,byte1);
                  Write(DirFile,byte2);
                  Write(DirFile,byte3);
                end;
                VOLRebuild.ProgressGauge.Progress := VOLRebuild.ProgressGauge.Progress + 1;
              end; {if not VolRebuildCanceled}
            end; {for RBRes := 0 to NumResources[RBResType] - 1 do}
          end; {if NumResources[RBResType] > 0}
          CloseFile(DirFile);
        end; {if ErrorType = NoError}
      end;
      CloseFile(VolFile);
      NumVolFiles := VolFileNum + 1;
    end;

    if ErrorType = NoError then
    begin
      for VolFileNum := 0 to 15 do
      if (not VolRebuildCanceled) and FileExists(GameDir+'VOL.'+IntToStr(VolFileNum)) and (not CanAccessFile(GameDir+'VOL.'+IntToStr(VolFileNum))) then
      begin
        ShowMessage('Error replacing file "VOL.'+IntToStr(VolFileNum)+'". Check that you are not running the game.');
        VolRebuildCanceled := True;
      end;
      for RBResType := 0 to 3 do
      if (not VolRebuildCanceled) and FileExists(GameDir+ResTypeAbbrv[RBResType]+'DIR') and (not CanAccessFile(GameDir+ResTypeAbbrv[RBResType]+'DIR')) then
      begin
        ShowMessage('Error replacing file "'+ResTypeAbbrv[RBResType]+'DIR". Check that you are not running the game.');
        VolRebuildCanceled := True;
      end;

      if VolRebuildCanceled then
      begin  // delete NEW files
        for VolFileNum := 0 to 15 do
        if FileExists(GameDir+'VOLNEW.'+IntToStr(VolFileNum)) then
        begin
          if not DeleteFile(GameDir+'VOLNEW.'+IntToStr(VolFileNum)) then
          begin
            ShowMessage('Error deleting file "VOLNEW.'+IntToStr(VolFileNum)+'"');
          end;
        end;
        for RBResType := 0 to 3 do
        if FileExists(GameDir+ResTypeAbbrv[RBResType]+'DIR.NEW') then
        begin
          if not DeleteFile(GameDir+ResTypeAbbrv[RBResType]+'DIR.NEW') then
          begin
            ShowMessage('Error deleting file "'+ResTypeAbbrv[RBResType]+'DIR"');
          end;
        end;
      end {if VolRebuildCanceled}
      else
      begin  // delete old files and replace with NEW ones
        for VolFileNum := 0 to 15 do
        if FileExists(GameDir+'VOL.'+IntToStr(VolFileNum)) then
        begin
          if not DeleteFile(GameDir+'VOL.'+IntToStr(VolFileNum)) then
          begin
            ErrorType := FileAccessError;
            ShowMessage('Error deleting file "VOL.'+IntToStr(VolFileNum)+'".');
          end;
        end;
        for RBResType := 0 to 3 do
        if FileExists(GameDir+ResTypeAbbrv[RBResType]+'DIR') then
        begin
          if not DeleteFile(GameDir+ResTypeAbbrv[RBResType]+'DIR') then
          begin
            ErrorType := FileAccessError;
            ShowMessage('Error deleting file "'+ResTypeAbbrv[RBResType]+'DIR".');
          end;
        end;
        if ErrorType = NoError then
        begin
          for VolFileNum := 0 to NumVolFiles - 1 do
          begin
            AssignFile(VolFile,GameDir+'VOLNEW.'+IntToStr(VolFileNum));
            Rename(VolFile,'VOL.'+IntToStr(VolFileNum));
          end;
          for RBResType := 0 to 3 do
          begin
            AssignFile(DirFile,GameDir+ResTypeAbbrv[RBResType]+'DIR'+'.NEW');
            Rename(DirFile,ResTypeAbbrv[RBResType]+'DIR');
          end;
        end;
        if ErrorType = NoError then
        for RBResType := 0 to 3 do
          if NumResources[RBResType] > 0 then
          for RBResNum := 0 to NumResources[RBResType]-1 do
          begin
            ResourceInfo[RBResType,ResourceNum[RBResType,RBResNum]].Loc := NewResourceInfo[RBResType,ResourceNum[RBResType,RBResNum]].Loc;
            ResourceInfo[RBResType,ResourceNum[RBResType,RBResNum]].Filename := NewResourceInfo[RbResType,ResourceNum[RBResType,RBResNum]].Filename;
          end;
      end; {if not VolRebuildCanceled}
    end; {if ErrorType = NoError}

    Main.Enabled := True;
    ResourceWin.Enabled := True;
    VOLRebuild.Close;
  end;  // if GameIsV3
end;

{**********************************************************************}
function GetAGIVersionNumber : double;
{this code adapted from agiver by Jeremy Hayes}
{**********************************************************************}

type
  VersionNumType=string[9];

var
  VersionNumBuffer : VersionNumType;
  InFilename       : string;
  InFile           : file;
  AGIData          : TResource;
  ISVerNum         : boolean;
  x                : integer;
  VerLen           : byte;
  VerNumText       : string;
  VerNum           : double;
  code             : integer;
  ResPos           : word;

begin
  InFilename := GameDir+'AGIDATA.OVL';
  VersionNumBuffer := 'A_CDE_GHI';
  GetAGIVersionNumber := 2.917;  // This is what we use if we can't find the version number.
                                 // Version 2.917 is the most common interpreter and
                                 // the one that all the "new" AGI games should be based on.
  AGIData.Size := 0;
  AssignFile(InFile,InFileName);
  {$I-}
  Reset(InFile,1);
  {$I+}
  if IOResult = 0 then
  begin
    if FileSize(InFile) < MaxResourceSize then
    begin
      AGIData.Size := FileSize(InFile);
      GetMem(AGIData.Data,AGIData.Size);
      BlockRead(InFile,AGIData.Data^,AGIData.Size);
    end;
    CloseFile(InFile);
  end;  // if IOResult <> 0


  if AGIData.Size > 0 then
  begin
    ResPos := 0;
    VerLen := 0;
    while (ResPos < AGIData.Size) and (VerLen = 0) do
    begin
      VersionNumBuffer := Copy(VersionNumBuffer,2,8) + chr(AGIData.Data^[ResPos]);
      Inc(ResPos);
      ISVerNum:=TRUE;
      If VersionNumBuffer[2]='.' then
      begin
        if (not (VersionNumBuffer[1] in ['0'..'9'])) then ISVerNum := False;
        for x := 3 to 5 do
          if (not (VersionNumBuffer[x] in ['0'..'9'])) then ISVerNum:=FALSE;
        if ISVerNum and (VersionNumBuffer[1]<='2') then VerLen:=5;  // 2.xxx format
        If VersionNumBuffer[6]<>'.' then ISVerNum := False;
        for x := 7 to 9 do  // 3.xxx.xxx format
          if (not (VersionNumBuffer[x] in ['0'..'9'])) then ISVerNum:=FALSE;
        If ISVerNum
          then VerLen:=9;
      end
      else ISVerNum := False;
      if VerLen>0 then
      begin
        if VersionNumBuffer[6] = '.' then
          VerNumText := Copy(VersionNumBuffer,1,5) + Copy(VersionNumBuffer,7,3) // remove second .
        else VerNumText := Copy(VersionNumBuffer,1,VerLen);
        Val(VerNumText,VerNum,code);
        if code = 0 then GetAGIVersionNumber := VerNum;
      end;
    end;  // while not EOF(InFile)
    FreeMem(AGIData.Data,AGIData.Size);
  end; // if AGIData.Size > 0
end;

{**********************************************************************}
function FindAGIV3GameID(dir:string) : string;
{**********************************************************************}
var Found : integer;
    SearchRec : TSearchRec;
    Filename : string;
    GameID1 : string;
begin
  GameID1 := 'V2';
  Found := FindFirst(dir+'*.*', faAnyFile, SearchRec);
  while (Found = 0) and (GameID1 = 'V2') do
  begin
    Filename := UpperCase(SearchRec.Name);
    if (pos('DIR',Filename) > 0) and (pos('.',Filename) = 0) then
    begin
      if FileExists(dir+Copy(Filename,1,pos('DIR',Filename)-1)+'VOL.0') then
        GameID1 := Copy(Filename,1,pos('DIR',Filename)-1);
    end;
    Found := FindNext(SearchRec);
  end;
  FindClose(SearchRec);
  FindAGIV3GameID := GameID1;
end;

end.
