{
    LogCompile.pas - LOGIC compiler
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

unit LogCompile;

interface

uses Classes, SysUtils, Dialogs, Defines;

function CompileLogic(LogEditorNum:integer) : TResource;

implementation

uses AGICommands, LogEdit1, WordsEditMain1, objed;

{******************************************************************************}
function CompileLogic(LogEditorNum:integer) : TResource;
{******************************************************************************}
const MaxBlockDepth = 12;
      MaxLabels = 255;
      UseTypeChecking = True;
      MaxDefines = 255;

type TLogicLabel = record
                     Name : string;
                     Loc : word;
                   end;


var CompiledLogic : TResource;
    OutputLogic : TResource;
    ResPos : word;
    LogicSize : word;
    InputLines : TStringList;  // lines come from editor window
    EditLines : TStringList;  // same as InputLines but with include files inserted
    RealLineNum : array[0..65535] of word;  // read line number in the
    LineFile : array[0..65535] of word;  // number of include file the line comes from, or 0 if from editor window
    IncludeFilenames : TStringList;

    ErrorOccured : boolean;
    CurChar : word;

    Messages : TStringList;
    MessageExists : array[0..255] of boolean;

    Labels : array[1..MaxLabels] of TLogicLabel;
    NumLabels : byte;

    DefineNames : TStringList;
    DefineValues : TStringList;
    DefineNameLength : array[0..255] of word;
    NumDefines : word;

    Words : TWordList;
    CurWordGroup : word;
    CurWord : word;

    OBJECTFile : TOBJECTFile;
    OBJECTFileEncrypted : boolean;
    InvObjects : TStringList;
    NumInvObjects : word;
    CurInvObject : word;

  {**************************************************}
  procedure WriteByte(TheByte:byte);
  {**************************************************}
  begin
    if ResPos < CompiledLogic.Size then
    begin
      CompiledLogic.Data^[ResPos] := TheByte;
      ResPos := ResPos + 1;
      if ResPos > LogicSize then LogicSize := ResPos;
    end;
  end;

  {**************************************************}
  procedure WriteByteAtLoc(TheByte:byte;Loc:Word);
  {**************************************************}
  begin
    if Loc < CompiledLogic.Size then
    begin
      CompiledLogic.Data^[Loc] := TheByte;
      if Loc > LogicSize then LogicSize := Loc;
    end;
  end;

  {**************************************************}
  procedure WriteLSMSWord(TheWord:word);
  {**************************************************}
  begin
    WriteByte(TheWord mod 256);
    WriteByte(TheWord div 256);
  end;

  {**************************************************}
  procedure ShowError(Line:word;ErrorMsg:string);
  {**************************************************}
  var SelStart : integer;
      CurLine : word;
      LineNum : word;
      TextEditorNum : word;
  begin
    if not ErrorOccured then
    begin
      LineNum := RealLineNum[Line];
      if (LineFile[Line] = 0) or (Line > EditLines.Count) then
      begin  // error is in logic in editor window
        if Line > EditLines.Count then
        begin
          LogEditWin[LogEditorNum].EditMemo.SelStart := Length(LogEditWin[LogEditorNum].EditMemo.Text);
          LogEditWin[LogEditorNum].EditMemo.SelLength := 0;
          LogEditWin[LogEditorNum].StatusBar1.Panels[0].Text := 'Error (at end of file): '+ErrorMsg
        end
        else
        begin
          SelStart := 0;
          if LineNum > 0 then
            for CurLine := 0 to LineNum - 1 do
              SelStart := SelStart + Length(LogEditWin[LogEditorNum].EditMemo.Lines[CurLine]) + 2;
          LogEditWin[LogEditorNum].EditMemo.SelStart := SelStart;
          LogEditWin[LogEditorNum].EditMemo.SelLength := Length(LogEditWin[LogEditorNum].EditMemo.Lines[LineNum]);
          LogEditWin[LogEditorNum].StatusBar1.Panels[0].Text := 'Error (line '+IntToStr(LineNum)+'): '+ErrorMsg;
        end;
      end
      else
      begin
        if LineFile[Line] > IncludeFilenames.Count then
          ShowMessage('Internal error: Error found in unknown include file.')
        else
        begin
          TextEditorNum := EditTextFile(IncludeFilenames[LineFile[Line]-1]);
          if (not (TextEditorNum in [1..MaxLogEditWins])) or (not LogEditWinUsed[TextEditorNum]) then
            ShowMessage('Error: Could not display error (file "'+IncludeFilenames[LineFile[Line]-1]+'" could not be opened).')
          else
          begin
            SelStart := 0;
            if LineNum > 0 then
              for CurLine := 0 to LineNum - 1 do
                SelStart := SelStart + Length(LogEditWin[TextEditorNum].EditMemo.Lines[CurLine]) + 2;
            LogEditWin[TextEditorNum].EditMemo.SelStart := SelStart;
            LogEditWin[TextEditorNum].EditMemo.SelLength := Length(LogEditWin[TextEditorNum].EditMemo.Lines[LineNum]);
            if LineNum > LogEditWin[TextEditorNum].EditMemo.Lines.Count - 1 then
              LogEditWin[TextEditorNum].StatusBar1.Panels[0].Text := 'Error in '+ExtractFileName(IncludeFilenames[LineFile[Line]-1])+' (at end of file): '+ErrorMsg
            else LogEditWin[TextEditorNum].StatusBar1.Panels[0].Text := 'Error in '+ExtractFileName(IncludeFilenames[LineFile[Line]-1])+' (line '+IntToStr(LineNum)+'): '+ErrorMsg;
          end;
        end;  // if LineFile[Line] > IncludeFilenames.Count
      end;  // if LineFile[Line] = 0
      ErrorOccured := True;
    end;
  end;

  {**************************************************}
  function NumLength(LineNum,StartPos:word) : integer;
  {**************************************************}
  var CurPos : integer;
      LineLength : integer;
  begin
    LineLength := Length(EditLines[LineNum]);
    CurPos := StartPos - 1;
    repeat
      CurPos := CurPos + 1;
    until (CurPos > LineLength) or (not (EditLines[LineNum][CurPos] in ['0'..'9']));
    NumLength := CurPos - StartPos;
  end;

  {**************************************************}
  function ReadNum(LineNum,StartPos:word) : integer;
  {**************************************************}
  var NumText : string;
      Num : integer;
      code : integer;
  begin
    NumText := copy(EditLines[LineNum],StartPos,NumLength(LineNum,StartPos));
    if NumText = '' then code := 1
    else Val(NumText,Num,code);
    if code = 0 then ReadNum := Num
    else ReadNum := -1;
  end;

  {**************************************************}
  procedure RemoveComments(var Lines:TStringList);
  {**************************************************}
  var CurLine : word;
      CurChar : integer;
      ThisLine : string;
      InQuotes : boolean;
      CommentDepth : word;
      CommentStart : word;
      RestOfLineIgnoredFrom : word;
  begin
    CommentDepth := 0;
    if Lines.Count > 0 then
    for CurLine := 0 to Lines.Count - 1 do
    begin
      ThisLine := Lines[CurLine];
      if Length(ThisLine) > 0 then
      begin
        InQuotes := False;
        CurChar := 0;
        CommentStart := 1;
        RestOfLineIgnoredFrom := 0;
        repeat
        begin
          CurChar := CurChar + 1;
          if (ThisLine[CurChar]='"') and ((CurChar=1) or (ThisLine[CurChar-1]<>'\')) then
            InQuotes := not InQuotes;
          if (not InQuotes) then
          begin
            if ((Copy(ThisLine,CurChar,2)='//') or (Copy(ThisLine,CurChar,1)='[')) and (RestOfLineIgnoredFrom=0) then
              RestOfLineIgnoredFrom := CurChar;
            if (Copy(ThisLine,CurChar,2)='/*') then
            begin
              if CommentDepth = 0 then CommentStart := CurChar;
              Inc(CommentDepth);
            end
            else if (Copy(ThisLine,CurChar,2)='*/') then
            begin
              if CommentDepth = 0 then
              begin
                CommentDepth := 1;
                CommentStart := CurChar;
              end;
              Dec(CommentDepth);
              if CommentDepth = 0 then
              begin
                if RestOfLineIgnoredFrom > 0 then
                  RestOfLineIgnoredFrom := RestOfLineIgnoredFrom - (CurChar-CommentStart+2);
                Thisline := Copy(ThisLine,1,CommentStart-1) + Copy(ThisLine,CurChar+2,9999);
                CurChar := CommentStart;
              end;
            end;
          end;  // if (not InQuotes)
        end; until CurChar >= Length(ThisLine);
        if RestOfLineIgnoredFrom > 0 then
          ThisLine := Copy(ThisLine,1,RestOfLineIgnoredFrom-1);
        if CommentDepth > 0 then
          ThisLine := Copy(ThisLine,1,CommentStart-1);
      end;
      Lines[CurLine] := ThisLine;
    end;
//    LogEditWin[LogEditorNum].EditMemo.Lines.Assign(EditLines);
  end;

  {**************************************************}
  procedure AddIncludes;
  {**************************************************}
  var IncludeStrings  : TStringList;  // strings read from include file
      IncludeLines    : TStringList;
      IncludeFilename : string;
      CurLine         : word;  // current line in EditLines (the output)
      CurInputLine    : word;  // current line in InputLines (the input from the editor window)
      CurIncludeLine  : word;  // current line in IncludeLines (the include file)
      FilenameStart   : word;
      LinePos         : word;
      CurEditorNum    : word;
  begin
    IncludeFilenames := TStringList.Create;
    EditLines := TStringList.Create;
    IncludeLines := TStringList.Create;  // only temporary, freed at end of AddIncludes
    CurLine := 0;
    CurInputLine := 0;
    repeat
    begin
      EditLines.Add(InputLines[CurInputLine]);
      CurLine := EditLines.Count - 1;
      RealLineNum[CurLine] := CurInputLine;
      LineFile[CurLine] := 0;
      if Lowercase(Copy(InputLines[CurInputLine],1,8)) = '#include' then
      begin
        if Copy(InputLines[CurInputLine],9,1) <> ' ' then
          ShowError(CurLine,'" " expected after #include.')
        else
        begin
          FilenameStart := 9;
          while (FilenameStart <= Length(InputLines[CurInputLine])) and (InputLines[CurInputLine][FilenameStart] = ' ') do
            Inc(FilenameStart);
          if FilenameStart >= Length(InputLines[CurInputLine]) then
            ShowError(CurLine,'Include filename expected.')
          else
          begin
            if Copy(InputLines[CurInputLine],FilenameStart,1) = '"' then
            begin  // quote marks around filename
              Inc(FilenameStart);
              LinePos := FilenameStart;
              while (LinePos <= Length(InputLines[CurInputLine]))
                and (InputLines[CurInputLine][LinePos] <> '"') do
                Inc(LinePos);
              if (LinePos > Length(InputLines[CurInputLine]))
                and (InputLines[CurInputLine][LinePos-1] <> '"') then
                ShowError(CurLine,'" expected after filename.');
              IncludeFilename := Copy(InputLines[CurInputLine],FilenameStart,LinePos-FilenameStart);
              Inc(LinePos);
              while (LinePos <= Length(InputLines[CurInputLine]))
                and (InputLines[CurInputLine][LinePos] = ' ') do
                Inc(LinePos);
              if LinePos <= Length(InputLines[CurInputLine]) then
                ShowError(CurLine,'Nothing allowed on line after filename.');
            end    // quote marks around filename
            else
            begin  // no quote marks around filename
              LinePos := FilenameStart;
              while (LinePos <= Length(InputLines[CurInputLine]))
                and (InputLines[CurInputLine][LinePos] <> ' ') do
                Inc(LinePos);
              IncludeFilename := Copy(InputLines[CurInputLine],FilenameStart,LinePos-FilenameStart);
              while (LinePos <= Length(InputLines[CurInputLine]))
                and (InputLines[CurInputLine][LinePos] = ' ') do
                Inc(LinePos);
              if LinePos <= Length(InputLines[CurInputLine]) then
                ShowError(CurLine,'Filenames containing spaces need quote marks around them.');
            end;  // no quote marks around filename
            EditLines[CurLine] := '';
            if ExtractFilename(IncludeFilename) <> IncludeFilename then
              ShowError(CurLine,'Only files in the src directory can be included.')
            else
            begin
              IncludeFilename := GameDir+'src\'+IncludeFilename;
              if not FileExists(IncludeFilename) then
                ShowError(CurLine,'Could not find file "'+ExtractFilename(IncludeFilename)+'" in src directory.')
              else
              begin
                for CurEditorNum := 1 to MaxLogEditWins do
                  if LogEditWinUsed[CurEditorNum] and (not LogEditWin[CurEditorNum].ContentsIsLogic)
                  and (LogEditWin[CurEditorNum].TextFilename = IncludeFilename)
                  and LogEditWin[CurEditorNum].Modified then
                    LogEditWin[CurEditorNum].SaveTextFile;
                IncludeFilenames.Add(IncludeFilename);
                IncludeLines.Clear;
                IncludeLines.LoadFromFile(IncludeFilename);
                RemoveComments(IncludeLines);
                if IncludeLines.Count > 0 then
                  for CurIncludeLine := 0 to IncludeLines.Count - 1 do
                  begin
                    EditLines.Add(IncludeLines[CurIncludeLine]);
                    CurLine := EditLines.Count - 1;
                    RealLineNum[CurLine] := CurIncludeLine;
                    LineFile[CurLine] := IncludeFilenames.Count;
                  end;
              end;  // if FileExists(IncludeFilename)
            end;  // if ExtractFilename(IncludeFilename) = IncludeFilename
          end;  // if FilenameStart < Length(InputLines[CurInputLine]))
        end;  // if Copy(InputLines[CurInputLine],9,1) = ' '
      end;  // if Lowercase(Copy(InputLines[CurInputLine],1,8)) = '#include'
      Inc(CurInputLine);
    end; until CurInputLine >= InputLines.Count;
    IncludeLines.Free;
    InputLines.Free;
  end;

  {**************************************************}
  procedure ReadDefines;
  {**************************************************}
  var
    ThisDefineName : string;
    ThisDefineValue : string;
    DefineNameStartPos : word;
    DefineValueStartPos : word;
    LinePos : word;
    LineLength : word;
    LowerCaseLine : string;
    CurLine : word;
    CurCommand : word;
    CurChar : word;
    CurDefine : word;
    FoundDefine : boolean;
    InQuotes : boolean;

  begin
    NumDefines := 0;
    for CurLine := 0 to EditLines.Count - 1 do
    begin
      if LowerCase(Copy(EditLines[CurLine],1,7)) = '#define' then
      begin
        if Copy(EditLines[CurLine],8,1) <> ' ' then
          ShowError(CurLine,'" " expected after #define.')
        else if NumDefines >= MaxDefines then
          ShowError(CurLine,'Too many defines (max '+IntToStr(MaxDefines)+').')
        else
        begin
          LowerCaseLine := LowerCase(EditLines[CurLine]);
          LineLength := Length(EditLines[CurLine]);
          LinePos := 8;
          while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do Inc(LinePos);
          DefineNameStartPos := LinePos;
          while (LinePos <= LineLength) and (LowerCaseLine[LinePos] in ['a'..'z','0'..'9','_','.']) do Inc(LinePos);
          if LowerCaseLine[LinePos] <> ' ' then ShowError(CurLine,'"'+LowerCaseLine[LinePos]+'" not allowed in define name.');
          ThisDefineName := Copy(LowerCaseLine,DefineNameStartPos,LinePos-DefineNameStartPos);
          if NumDefines > 0 then
            for CurDefine := 0 to NumDefines-1 do
              if ThisDefineName = Definenames[CurDefine] then
                ShowError(CurLine,'"'+Copy(EditLines[CurLine],DefineNameStartPos,LinePos-DefineNameStartPos)+'" already defined.');


          CurCommand := 0;
          while (CurCommand <= NumAGICommands) and (ThisDefineName <> AGICommand[CurCommand].Name) do
            Inc(CurCommand);
          if ThisDefineName = AGICommand[CurCommand].Name then
            ShowError(CurLine,'Define name can not be a command name.')
          else
          begin
            CurCommand := 0;
            while (CurCommand <= NumTestCommands) and (ThisDefineName <> TestCommand[CurCommand].Name) do
              Inc(CurCommand);
            if ThisDefineName = TestCommand[CurCommand].Name then
              ShowError(CurLine,'Define name can not be a command name.')
          end;
          if (ThisDefineName = 'if') or (ThisDefineName = 'else') or (ThisDefineName = 'goto') then
            ShowError(CurLine,'"'+ThisDefineName+'" is not a valid define name.');
          for CurChar := 1 to Length(ThisDefineName) do
            if ThisDefineName[CurChar] in ['!',',','"','&','|',';',':','(',')'] then
              ShowError(CurLine,'Character "'+ThisDefineName[CurChar]+'" not allowed in define name.');

          while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do Inc(LinePos);
          if (LinePos <= LineLength) and (LowerCaseLine[LinePos] = '"') then
          begin  // string
            Inc(LinePos);
            DefineValueStartPos := LinePos;
            while (LinePos <= LineLength) and (not ((LowerCaseLine[LinePos]='"') and (LowerCaseLine[LinePos-1]<>'\'))) do
              Inc(LinePos);
            if not ((LowerCaseLine[LinePos]='"') and (LowerCaseLine[LinePos-1]<>'\')) then
              ShowError(CurLine,'" required at end of string.');
            ThisDefineValue := Copy(EditLines[CurLine],DefineValueStartPos,LinePos-DefineValueStartPos);
            ThisDefineValue := '"' + ThisDefineValue + '"';
            DefineNames.Add(ThisDefineName);
            DefineValues.Add(ThisDefineValue);
            DefineNameLength[NumDefines] := Length(DefineNames[NumDefines]);
            NumDefines := NumDefines + 1;
            Inc(LinePos);
            while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do Inc(LinePos);
            if (LinePos <= LineLength) and (LowerCaseLine[LinePos] <> ' ') then
              ShowError(CurLine,'Nothing allowed on line after define value.');
          end
          else if (LinePos <= LineLength) then
          begin  // other
            DefineValueStartPos := LinePos;
            while (LinePos <= LineLength) and (LowerCaseLine[LinePos] <> ' ') do
              Inc(LinePos);
            ThisDefineValue := Copy(EditLines[CurLine],DefineValueStartPos,LinePos-DefineValueStartPos);
            for CurChar := 1 to Length(ThisDefineValue) do
              if ThisDefineValue[CurChar] in ['!',',','"','&','|',';',':','(',')'] then
                ShowError(CurLine,'Character "'+ThisDefineValue[CurChar]+'" not allowed in define value.');
            DefineNames.Add(ThisDefineName);
            DefineValues.Add(ThisDefineValue);
            DefineNameLength[NumDefines] := Length(DefineNames[NumDefines]);
            NumDefines := NumDefines + 1;
            Inc(LinePos);
            while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do Inc(LinePos);
            if (LinePos <= LineLength) and (LowerCaseLine[LinePos] <> ' ') then
              ShowError(CurLine,'Nothing allowed on line after define value.');
          end
          else ShowError(CurLine,'Value required for define.');
          EditLines[CurLine] := ''; // set to empty so CompileCommands doesn't try to read it
        end;  // if NumDefines >= MaxDefines
      end;  // if LowerCase(copy(EditLines[CurLine],1,9)) = '#define '
(*      else if EditLines[CurLine] <> '' then
      begin
        LowerCaseLine := LowerCase(EditLines[CurLine]);
        LineLength := Length(EditLines[CurLine]);
        LinePos := 1;
        InQuotes := False;
        repeat
        begin
          FoundDefine := False;
          if NumDefines > 0 then
          begin
            CurDefine := 0;
            while (CurDefine < NumDefines) and (not FoundDefine) do
            begin
//              if Copy(LowerCaseLine,LinePos,1) = '"' then
              if (LinePos <= LineLength) and (LowerCaseLine[LinePos]='"') then
              begin
                if not InQuotes then InQuotes := True
//                else if Copy(LowerCaseLine,LinePos-1,1) <> '\' then InQuotes := False;
                else if (LinePos > 1) and (LowerCaseLine[LinePos-1]<>'\') then InQuotes := False;
              end;
//              if (not InQuotes) and (Copy(LowerCaseLine,LinePos,DefineNameLength[CurDefine]) = DefineNames[CurDefine]) then
//              begin
//                EditLines[CurLine] := Copy(EditLines[CurLine],1,LinePos-1) + DefineValues[CurDefine] + Copy(EditLines[CurLine],LinePos+DefineNameLength[CurDefine],LineLength);
//                LinePos := LinePos + Length(DefineValues[CurDefine]);
//                LowerCaseLine := LowerCase(EditLines[CurLine]);
//                LineLength := Length(EditLines[CurLine]);
//                FoundDefine := True;
//              end;
              Inc(CurDefine);
            end;
          end;
          if not FoundDefine then Inc(LinePos);
        end; until LinePos > LineLength;
      end; *)
    end;  // for CurLine := 0 to EditLines.Count - 1
  end;

  {**************************************************}
  procedure ReadPredefinedMessages;
  {**************************************************}
  var CurMessage : byte;
      CurLine : word;
      MessageNum : integer;
      MessageNumText : string;
      MessageNumStartPos : integer;

      MessageTextStartPos : integer;

      LineLength : integer;
      LowerCaseLine : string;
      LinePos : integer;
      code : integer;
  begin
    Messages := TStringList.Create;
    for CurMessage := 0 to 255 do
    begin
      Messages.Add('');
      MessageExists[CurMessage] := False;
    end;
    for CurLine := 0 to EditLines.Count - 1 do
    begin
      if LowerCase(Copy(EditLines[CurLine],1,8)) = '#message' then
      begin
        if Copy(EditLines[CurLine],9,1) <> ' ' then
          ShowError(CurLine,'" " expected after #message.')
        else
        begin
          LowerCaseLine := LowerCase(EditLines[CurLine]);
          LineLength := Length(EditLines[CurLine]);
          LinePos := 9;
          while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do Inc(LinePos);
          MessageNumStartpos := LinePos;
          while (LinePos <= LineLength) and (LowerCaseLine[LinePos] in ['0'..'9']) do
            Inc(LinePos);
          MessageNumText := Copy(LowerCaseLine,MessageNumStartPos,LinePos-MessageNumStartPos);
          if MessageNumText = '' then code := -1
          else
            Val(MessageNumText,MessageNum,Code);
          if (code <> 0) or (not (MessageNum in [1..255])) then
            ShowError(CurLine,'Invalid message number (must be 1-255).')
          else
          begin
            while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do Inc(LinePos);
            if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> '"') then
              ShowError(CurLine,'" required at start of string.')
            else
            begin
              Inc(LinePos);
              MessageTextStartPos := LinePos;
              while (LinePos <= LineLength) and (not ((LowerCaseLine[LinePos]='"') and (LowerCaseLine[LinePos-1]<>'\'))) do
                Inc(LinePos);
              if not ((LowerCaseLine[LinePos]='"') and (LowerCaseLine[LinePos-1]<>'\')) then
                ShowError(CurLine,'" required at end of string.')
              else
              begin
                Messages[MessageNum] := Copy(EditLines[CurLine],MessageTextStartPos,LinePos-MessageTextStartPos);
                MessageExists[MessageNum] := True;
                Inc(LinePos);
                while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do Inc(LinePos);
                if (LinePos <= LineLength) and (LowerCaseLine[LinePos] <> ' ') then
                  ShowError(CurLine,'Nothing allowed on line after message.');
                EditLines[CurLine] := ''; // set to empty so CompileCommands doesn't try to read it
              end;
            end;  // if (LinePos <= LineLength) and (LowerCaseLine[LinePos] = '"')
          end;  // if (code = 0) and (MessageNum in [1..255])
        end;  // if Copy(EditLines[CurLine],9,1) = ' '
      end;  // if LowerCase(Copy(EditLines[CurLine],1,8)) = '#message'
    end;  // for CurLine := 0 to EditLines.Count - 1
  end;

  {**************************************************}
  procedure WriteMessageSection;
  {**************************************************}
  var
      MessageSectionStart : word;
      MessageSectionEnd : word;
      MessageLoc : array[1..255] of word;
      CurChar : word;
      CurMessage : integer;
      NumMessages : word;
      EncryptionStart : word;
      ThisMessageLength : word;

      function WriteEncByte(TheByte:byte) : byte;
      begin
        WriteByte(TheByte XOR ord(EncryptionKey[(ResPos-EncryptionStart) mod 11 + 1]));
      end;

  begin
    MessageSectionStart := ResPos;
    NumMessages := 0;
    CurMessage := 256;
    repeat
      CurMessage := CurMessage - 1;
    until MessageExists[CurMessage] or (CurMessage = 0);
    NumMessages := CurMessage;
    WriteByte(NumMessages);
    ResPos := MessageSectionStart + 3 + NumMessages*2;
    EncryptionStart := ResPos;
    for CurMessage := 1 to NumMessages do
      if MessageExists[CurMessage] then
      begin
        ThisMessageLength := Length(Messages[CurMessage]);
        MessageLoc[CurMessage] := ResPos - MessageSectionStart - 1;
        if ThisMessageLength > 0 then
        begin
          CurChar := 1;
          repeat
          begin
            if (Messages[CurMessage][CurChar] = '\') and (CurChar<ThisMessageLength) then
            begin
              if Messages[CurMessage][CurChar+1] = 'n' then
                begin WriteEncByte($0A); Inc(CurChar); end
              else if Messages[CurMessage][CurChar+1] = '"' then
                begin WriteEncByte($22); Inc(CurChar); end
              else if Messages[CurMessage][CurChar+1] = '\' then
                begin WriteEncByte($5C); Inc(CurChar); end
              else WriteEncByte($5C);
            end
            else
            WriteEncByte(ord(Messages[CurMessage][CurChar]));
            CurChar := CurChar + 1;
          end; until CurChar > Length(Messages[CurMessage]);
        end;  // if ThisMessageLength > 0
        WriteEncByte($00);
      end
      else MessageLoc[CurMessage] := 0;
    MessageSectionEnd := ResPos - MessageSectionStart - 1;
    ResPos := MessageSectionStart + 1;
    WriteLSMSWord(MessageSectionEnd);
    for CurMessage := 1 to NumMessages do
      WriteLSMSWord(MessageLoc[CurMessage]);
    Respos := 0;
    WriteLSMSWord(MessageSectionStart-2);  // in the file, the message section start is relative to start of actual code
    Messages.Free;
  end;

  {**************************************************}
  procedure ReadLabels;
  {**************************************************}
  var CurLine : word;
      LinePos : word;
      LineLength : word;
      LowerCaseLine : string;
      LabelStartPos : word;
      CurLabel : word;
      LabelName : string;
      CurDefineName : word;
  begin
    NumLabels := 0;
    for CurLine := 0 to EditLines.Count - 1 do
    if EditLines[CurLine] <> '' then
    begin
      LineLength := Length(EditLines[CurLine]);
      LowerCaseLine := LowerCase(EditLines[CurLine]);
      LinePos := 1;
      while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do
        Inc(LinePos);
      LabelStartPos := LinePos;
      while (LinePos <= LineLength) and (LowerCaseLine[LinePos] in ['a'..'z','0'..'9','_','.']) do
        Inc(LinePos);
      if (LinePos > LabelStartPos) and (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ':') then
      begin
        LabelName := Copy(LowerCaseLine,LabelStartPos,LinePos-LabelStartPos);
        CurLabel := 1;
        while (CurLabel <= NumLabels) and (Labels[CurLabel].Name <> LabelName) do
          Inc(CurLabel);
        if CurLabel <= NumLabels then
          ShowError(CurLine,'Label "'+Copy(EditLines[CurLine],LabelStartPos,LinePos-LabelStartPos)+'" already defined.')
        else if NumLabels >= MaxLabels then
          ShowError(CurLine,'Too many labels (max '+IntToStr(MaxLabels)+').')
        else if (LabelName = 'if') or (LabelName = 'else') or (LabelName = 'goto') then
          ShowError(CurLine,'"'+LabelName+'" is not a valid label name.')
        else
        begin
          CurDefineName := 0;
          while (CurDefineName <= NumDefines - 1) and (LabelName <> DefineNames[CurDefineName]) and (LabelName + ':' <> DefineNames[CurDefineName]) do
            Inc(CurDefineName);
          if (CurDefineName <= NumDefines - 1) and ((LabelName = DefineNames[CurDefineName]) or (LabelName + ':' = DefineNames[CurDefineName])) then
            ShowError(CurLine,'Can''t have a label with the same name a a define.');
          Inc(NumLabels);
          Labels[NumLabels].Name := LabelName;
          Labels[NumLabels].Loc := 0;
        end;
      end;
    end;
  end;

  {**************************************************}
  procedure CompileCommands;
  {**************************************************}
  const MaxGotos = 255;
  type TLogicGoto = record
                      LabelNum : byte;
                      DataLoc : word;
                    end;

  var
      CurLine : integer;
      FinishedReading : boolean;
      CurChar : word;
      LineLength : word;
      LowerCaseLine : string;
      LinePos : integer;

      CommandName : string;
      CommandNameStartPos : integer;
      CommandNum : integer;

      InIf : boolean;
      NumCommandsInIfStatement : byte;
      NumCommandsInIfBrackets : byte;
      AwaitingNextTestCommand : boolean;
      InIfBrackets : boolean;

      BlockStartDataLoc : array[1..MaxBlockDepth] of word;
      BlockDepth : integer;
      BlockIsIf : array[1..MaxBlockDepth] of boolean;
      BlockLength : array[1..MaxBlockDepth] of word;

      CurLabel : word;
      EncounteredLabel : boolean;

      Gotos : array[1..MaxGotos] of TLogicGoto;
      NumGotos : word;
      GotoData : word;
      CurGoto : word;

      NOTOn : boolean;

      LastCommandWasReturn : boolean;

      {************************}
      procedure NextLine;
      {************************}
      var NumLines : integer;
          FoundLine : boolean;
      begin
        NumLines := EditLines.Count;
        FoundLine := False;
        repeat
        begin
          repeat
            CurLine := CurLine + 1;
          until (CurLine > NumLines - 1) or (EditLines[CurLine] <> '');
          if CurLine > NumLines - 1 then
            FinishedReading := True
          else
          begin
            LinePos := 1;
            LowerCaseLine := LowerCase(EditLines[CurLine]);
            LineLength := Length(EditLines[CurLine]);
            while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do
              Inc(LinePos);
            if Linepos <= LineLength then FoundLine := True;
          end;
        end; until FoundLine or FinishedReading;
      end;

      {************************}
      procedure SkipSpaces;
      {************************}
      begin
        while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do
          Inc(LinePos);
        if LinePos > LineLength then NextLine;
      end;

      {************************}
      function MessageNum(TheMessage:string) : byte;
      {************************}
      var CurMessage : word;
      begin
        CurMessage := 0;
        repeat
          Inc(CurMessage);
        until (CurMessage > 255) or (MessageExists[CurMessage] and (Messages[CurMessage] = TheMessage));
        if CurMessage > 255 then MessageNum := 0
        else MessageNum := CurMessage;
      end;

      {************************}
      function AddMessage(TheMessage:string) : byte;
      // Adds a message to the message list regardles of whether or not it
      // already exists. Returns the number of the added message, or 0 if
      // there are already 255 messages.
      {************************}
      var CurMessage : word;
      begin
        CurMessage := 0;
        repeat
          Inc(CurMessage);
        until (CurMessage > 255) or (not MessageExists[CurMessage]);
        if CurMessage > 255 then AddMessage := 0
        else
        begin
          MessageExists[CurMessage] := True;
          Messages[CurMessage] := TheMessage;
          AddMessage := CurMessage;
        end;
      end;

      {************************}
      function ReadString : string;
      {************************}
      var TheString : string;
          StringStartPos : word;  // pos of first character after "
      begin
        ReadString := '';
        if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> '"') then
          ShowError(CurLine,'" required at start of string.')
        else
        begin
          StringStartPos := LinePos + 1;
          repeat
            Inc(LinePos);
          until (LinePos>LineLength) or ((LowerCaseLine[LinePos]='"') and (LowerCaseLine[LinePos-1]<>'\'));
          TheString := Copy(EditLines[CurLine],StringStartPos,LinePos-StringStartPos);
          if TheString = '' then ShowError(CurLine,'Strings must have at least one character.');
          if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> '"') then
            ShowError(CurLine,'" required at end of string.');
          Inc(LinePos);
          ReadString := TheString;
        end;
      end;

      {************************}
      function ReplaceDefine(InText:string) : string;
      {************************}
      var CurDefine : integer;
          FoundDefine : boolean;
          LowerCaseText : string;
      begin
        LowerCaseText := LowerCase(InText);
        FoundDefine := False;
        if NumDefines > 0 then
        begin
          CurDefine := 0;
          repeat
          begin
            if LowerCaseText = DefineNames[CurDefine] then
            begin
              ReplaceDefine := DefineValues[CurDefine];
              FoundDefine := True;
            end;
            Inc(CurDefine);
          end until (CurDefine >= NumDefines) or FoundDefine;
        end;
        if not FoundDefine then ReplaceDefine := InText;
      end;

      {************************}
      procedure ReadArgs(CommandIsIf:boolean;CmdNum:byte);
      {************************}
      const MaxSaidArgs = 40;
      var ArgValue : integer;
          CurArg : byte;

          SaidArgs                : array[1..MaxSaidArgs] of integer;
          NumSaidArgs             : byte;
          CurSaidArg              : byte;
          FinishedReadingSaidArgs : boolean;
          ThisWord                : string;
          FoundWordGroupNum       : boolean;
          CurWord                 : word;

          ThisCommand             : CommandStruct;
          ThisArgTypePrefix       : string[2];

          ThisMessage             : string;
          ThisMessageNum          : byte;

          ThisInvObjectName       : string;
          ThisInvObjectNum        : word;
          CurInvObject            : integer;

          ArgText                 : string;
          LowerCaseArgText        : string;
          ArgTextPos              : integer;
          ArgTextLength           : integer;
          PrevLine                : word;

          procedure ReadArgText;         // do not use for string - does not take quotes into account
          var ArgTextStartPos : word;
              InQuotes : boolean;
          begin
            InQuotes := False;
            SkipSpaces;
            ArgTextStartPos := LinePos;
            while (LinePos <= LineLength) and (not ((LowerCaseLine[LinePos] in [',',')']) and (not InQuotes))) do
            begin
              if LowerCaseLine[LinePos] = '"' then
              begin
                if not InQuotes then InQuotes := True
                else if LowerCaseLine[LinePos-1] <> '\' then InQuotes := False;
              end;
              Inc(LinePos);
            end;
            ArgText := ReplaceDefine(Copy(EditLines[CurLine],ArgTextStartPos,LinePos-ArgTextStartPos));
            LowerCaseArgText := LowerCase(ArgText);
            ArgTextLength := Length(ArgText);
            ArgTextPos := 1;
          end;

          procedure ReadArgValue;
          var
              ValueText : string;
              code : integer;
              StartPos : integer;
          begin
            StartPos := ArgTextPos;
            while (ArgTextPos <= ArgTextLength) and (ArgText[ArgTextPos] in ['0'..'9']) do
              Inc(ArgTextPos);
            ValueText := Copy(ArgText,StartPos,ArgTextPos-StartPos);
            if ValueText = '' then ArgValue := -1
            else
            begin
              Val(ValueText,ArgValue,code);
              if code <> 0 then ArgValue := -1;
            end;
          end;

      begin
        SkipSpaces;
        if (LinePos > LineLength) or (EditLines[CurLine][LinePos] <> '(') then
          ShowError(CurLine,'"(" expected.')
        else
        begin
          Inc(LinePos);
          if (CmdNum = 14) and CommandIsIf then // said test command
          begin
            NumSaidArgs := 0;
            FinishedReadingSaidArgs := False;
            repeat
            begin
              ReadArgText;
              NumSaidArgs := NumSaidArgs + 1;
              if Copy(ArgText,1,1) = '"' then
              begin
                ArgValue := 0;
                repeat
                  Inc(ArgTextPos);
                until (ArgTextPos > ArgTextLength) or (ArgText[ArgTextPos] = '"');
                ThisWord := LowerCase(Copy(ArgText,2,ArgTextPos-2));
                if ArgTextpos > ArgTextLength then
                  ShowError(CurLine,'" required at end of word.')
                else
                begin  // find word group number
                  FoundWordGroupNum := False;
                  if Words.NumGroups > 0 then
                  begin
                    CurWordGroup := 0;
                    repeat
                    begin
                      for CurWord := 0 to Words.WordGroup[CurWordGroup].Words.Count - 1 do
                        if Words.WordGroup[CurWordGroup].Words[CurWord] = ThisWord then
                        begin
                          ArgValue := Words.WordGroup[CurWordGroup].GroupNum;
                          FoundWordGroupNum := True;
                        end;
                      CurWordGroup := CurWordGroup + 1;
                    end; until (CurWordGroup > Words.NumGroups-1) or FoundWordGroupNum;
                    if not FoundWordGroupNum then
                      ShowError(CurLine,'Unknown word "'+ThisWord+'".');
                  end;
                end;  // find word group number
              end
              else ReadArgValue;
              SaidArgs[NumSaidArgs] := ArgValue;
              if (SaidArgs[NumSaidArgs] < 0) or (SaidArgs[NumSaidArgs] > 65535) then
              begin
                ShowError(CurLine,'Invalid word number for argument '+IntToStr(NumSaidArgs)+' (must be 0-65535).');
                SaidArgs[NumSaidArgs] := 0;
              end;
              if (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ',') then
              begin
                if  NumSaidArgs >= MaxSaidArgs then
                begin
                  ShowError(CurLine,'Too many arguments for said command.');
                  FinishedReadingSaidArgs := True;
                end;
              end
              else if (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ')') then
                FinishedReadingSaidArgs := True
              else ShowError(CurLine,'"," or ")" expected after argument '+IntToStr(NumSaidArgs)+'.');
              Inc(LinePos);
            end; until FinishedReadingSaidArgs or ErrorOccured;
            WriteByte(NumSaidArgs);
            for CurSaidArg := 1 to NumSaidArgs do
            begin
              WriteByte(SaidArgs[CurSaidArg] mod 256);
              WriteByte(SaidArgs[CurSaidArg] div 256);
            end;
          end
          else  // other command
          begin
            if CommandIsIf then ThisCommand := TestCommand[CmdNum]
            else ThisCommand := AGICommand[CmdNum];
            for CurArg := 1 to ThisCommand.NumArgs do
            begin
              SkipSpaces;
              ReadArgText;
              if (ThisCommand.ArgTypes[CurArg] = atMsg) and (ArgTextLength >= 1) and (ArgText[1] = '"') then
              begin  // argument is message and given as string
                ThisMessage := '';
                ArgTextPos := 1;
                repeat
                begin
                  Inc(ArgTextPos);
                  if (ArgTextPos = ArgTextLength) and (ArgText[ArgTextPos]='"') and (ArgText[ArgTextPos-1]<>'\') then
                  begin
                    ThisMessage := ThisMessage + Copy(ArgText,2,ArgTextPos-2);
{                    while (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ' ') do
                     Inc(LinePos);
                    if LinePos > LineLength then}
                    PrevLine := CurLine;
                    SkipSpaces;
                    if CurLine > PrevLine then
                    begin
                      ReadArgText;
                      if (ArgTextLength=0) or (ArgText[1]<>'"') then
                        ShowError(CurLine,'" expected.');
                      ArgTextPos := 2;
                    end;
                  end;
                end;
                until (ArgTextPos > ArgTextLength) or ((ArgText[ArgTextPos]='"') and (ArgText[ArgTextPos-1]<>'\'));
                if ArgTextPos > ArgTextLength then
                  ShowError(CurLine,'" required at end of string.');
//                ThisMessage := ThisMessage + Copy(ArgText,2,ArgTextPos-2);
                Inc(ArgTextPos);
                ThisMessageNum := MessageNum(ThisMessage);
                if ThisMessageNum > 0 then
                  WriteByte(ThisMessageNum)
                else
                begin
                  ThisMessageNum := AddMessage(ThisMessage);
                  if ThisMessageNum = 0 then
                    ShowError(CurLine,'Too many messages (max 255).')
                  else WriteByte(ThisMessageNum);
                end;
              end  // argument is message and given as string
              else
              if (ThisCommand.ArgTypes[CurArg] = atIObj) and (ArgTextLength >= 1) and (ArgText[1] = '"') then
              begin    // argument is inventory object and given as string
                ArgTextPos := 1;
                repeat
                  Inc(ArgTextPos);
                until (ArgTextPos > ArgTextLength) or ((ArgText[ArgTextPos]='"') and (ArgText[ArgTextPos-1]<>'\'));
                if ArgTextPos > ArgTextLength then
                  ShowError(CurLine,'" required at end of string.');
                ThisInvObjectName := LowerCase(Copy(ArgText,2,ArgTextPos-2));
                Inc(ArgTextPos);
                if ThisInvObjectName = '' then
                  ShowError(CurLine,'Object name must be at least one character.')
                else
                begin
                  CurInvObject := -1;
                  repeat
                    Inc(CurInvObject);
                  until (CurInvObject > NumInvObjects-1) or (InvObjects[CurInvObject] = ThisInvObjectName);
                  if CurInvObject > NumInvObjects-1 then
                    ShowError(CurLine,'Unknown inventory object "'+Copy(ArgText,2,ArgTextPos-2)+'".')
                  else
                  begin
                    ThisInvObjectNum := CurInvObject;
                    WriteByte(ThisInvObjectNum);
                  end;
                end;
              end    // argument is inventory object and given as string
              else
              begin  // normal argument
                ThisArgTypePrefix := ArgTypePrefix[ThisCommand.ArgTypes[CurArg]];

                if UseTypeChecking and (Copy(LowerCaseArgText,1,Length(ThisArgTypePrefix)) <> ThisArgTypePrefix) then
                  ShowError(CurLine,'Invalid or unknown argument type for argument '+IntToStr(CurArg)+' (should be a '+ArgTypeName[ThisCommand.ArgTypes[CurArg]]+').')
                else
                begin
                  if UseTypeChecking then Inc(ArgTextPos,Length(ThisArgTypePrefix))
                  else
                  begin
                    while (ArgTextPos <= ArgTextLength) and (not (LowerCaseArgText[ArgTextPos] in ['a'..'z'])) do
                      Inc(ArgTextPos);
                  end;
                  ReadArgValue;
                  if not (ArgValue in [0..255]) then
                    ShowError(CurLine,'Invalid or missing value for argument '+IntToStr(CurArg)+' (must be 0-255)')
                  else WriteByte(ArgValue);
                end;
              end;  // normal argument
              if CurArg < ThisCommand.NumArgs then
              begin
                if ArgTextPos <= ArgTextLength then
                  ShowError(CurLine,'"," expected after argument '+IntToStr(CurArg)+'.')
                else
                if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> ',') then
                  ShowError(CurLine,'"," expected after argument '+IntToStr(CurArg)+'.')
                else Inc(LinePos,1);
              end
              else if ArgTextPos <= ArgTextLength then
                ShowError(CurLine,'")" expected after argument '+IntToStr(CurArg)+'.')
            end;
            if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> ')') then
            begin
              if ThisCommand.NumArgs > 0 then
                ShowError(CurLine,'")" expected after argument '+IntToStr(ThisCommand.NumArgs)+'.')
              else ShowError(CurLine,'")" expected.');
            end
            else Inc(LinePos,1);
          end;
        end;
      end;

      {************************}
      function ReadText : string;
      {************************}
      var StartPos : word;
      begin
        StartPos := LinePos;
        while (LinePos <= LineLength) and (not (LowerCaseLine[LinePos] in ['(',' ',',',')',':'])) do
          Inc(LinePos);
        ReadText := Copy(LowerCaseLine,StartPos,LinePos-StartPos);
      end;

      {************************}
      function ReadPlainText : string;
      {************************}
      var StartPos : word;
      begin
        StartPos := LinePos;
        while (LinePos <= LineLength) and (LowerCaseLine[LinePos] in ['a'..'z','0'..'9','_','.']) do
          Inc(LinePos);
        ReadPlainText := Copy(LowerCaseLine,StartPos,LinePos-StartPos);
      end;

      {************************}
      function ReadExprText : string;
      {************************}
      var StartPos : word;
      begin
        StartPos := LinePos;
        while (LinePos <= LineLength) and (LowerCaseLine[LinePos] in ['=','+','-','*','/','>','<','!']) do
          Inc(LinePos);
        ReadExprText := Copy(LowerCaseLine,StartPos,LinePos-StartPos);
      end;

      {************************}
      procedure ReadCommandName;
      {************************}
      begin
        SkipSpaces;
        CommandNameStartPos := LinePos;
        CommandName := ReadText;
      end;

      {************************}
      function FindCommandNum(CommandIsIf:boolean;CmdName:string) : byte;
      {************************}
      var CmdNum : integer;
      begin
        if CommandIsIf then
        begin
          CmdNum := 0;
          repeat
            CmdNum := CmdNum + 1;
          until (CmdNum > NumTestCommands) or (CmdName = TestCommand[CmdNum].Name);
          if CmdNum > NumTestCommands then FindCommandNum := 255
          else FindCommandNum := CmdNum;
        end
        else
        begin
          CmdNum := -1;
          repeat
            CmdNum := CmdNum + 1;
          until (CmdNum > NumAGICommands) or (CmdName = AGICommand[CmdNum].Name);
          if CmdNum > NumAGICommands then FindCommandNum := 255
          else FindCommandNum := CmdNum;
        end;
      end;

      {************************}
      function AddSpecialIFSyntax : boolean;
      {************************}
      var
        arg1 : integer;
        arg2 : integer;
        arg2isvar : boolean;
        ArgText : string;
        expr    : string;
        code : integer;
        AddNOT : boolean;
        OldLinepos : word;

      begin
        AddSpecialIFSyntax := False;
        OldLinepos := LinePos;
        LinePos := LinePos - Length(CommandName);
        ArgText := ReplaceDefine(ReadPlainText);
        if Copy(ArgText,1,1) = 'v' then
        begin
          if NOTOn then ShowError(CurLine,'"!" not allowed before var.');
          Val(Copy(ArgText,2,Length(ArgText)-1),arg1,code);
          if (code <> 0) or (not (arg1 in [0..255])) then
            ShowError(CurLine,'Invalid number given or error in expression syntax.')
          else
          begin
            SkipSpaces;
            expr := ReadExprText;
            SkipSpaces;
            ArgText := ReplaceDefine(ReadPlainText);
            arg2isvar := (Copy(ArgText,1,1) = 'v');
            if arg2isvar then Val(Copy(ArgText,2,Length(ArgText)-1),arg2,code)
            else Val(ArgText,arg2,code);
            if (code <> 0) or (not (arg2 in [0..255])) then
              ShowError(CurLine,'Invalid number given or error in expression syntax.')
            else
            begin
              CommandNum := 0;
              AddNOT := False;
              if expr = '==' then CommandNum := $01 // equal
              else if expr = '<' then CommandNum := $03 // less
              else if expr = '>' then CommandNum := $05 // greater
              else if expr = '!=' then begin CommandNum := $01; AddNOT := True; end // !equal
              else if expr = '>=' then begin CommandNum := $03; AddNOT := True; end // !less
              else if expr = '<=' then begin CommandNum := $05; AddNOT := True; end // !greater
              else ShowError(CurLine,'Expression syntax error');
              if CommandNum > 0 then
              begin
                if arg2isvar then Inc(CommandNum);
                if AddNOT then WriteByte($FD);
                WriteByte(CommandNum);
                WriteByte(arg1);
                WriteByte(arg2);
                AddSpecialIFSyntax := True;
              end;
            end;
          end;
        end  // if Copy(ArgText,1,1) = 'v'
        else if Copy(ArgText,1,1) = 'f' then
        begin
          Val(Copy(ArgText,2,Length(ArgText)-1),arg1,code);
          if (code <> 0) or (not (arg1 in [0..255])) then
            ShowError(CurLine,'Invalid number given or error in expression syntax..')
          else
          begin
            WriteByte($07);  // isset
            WriteByte(arg1);
            AddSpecialIFSyntax := True;
          end;
        end  // if Copy(ArgText,1,1) = 'f'
        else Linepos := OldLinepos;
      end;

      {************************}
      function AddSpecialSyntax : boolean;
      {************************}
      var
        arg1 : integer;
        arg2 : integer;
        arg2isvar : boolean;
        arg3 : integer;
        arg3isvar : boolean;
        ArgText : string;
        expr    : string;
        expr2   : string;
        code : integer;
        AddNOT : boolean;
        arg2isstar : boolean;
        OldLinepos : integer;

      begin
        AddSpecialSyntax := False;
        OldLinePos := LinePos;
        LinePos := LinePos - Length(CommandName);
        if Copy(CommandName,1,1) = '*' then
        begin
          Inc(LinePos);
          ArgText := '*' + ReplaceDefine(ReadPlainText);
        end
        else ArgText := ReplaceDefine(ReadPlainText);
        if Copy(ArgText,1,1) = 'v' then
        begin
          Val(Copy(ArgText,2,Length(ArgText)-1),arg1,code);
          if (code <> 0) or (not (arg1 in [0..255])) then
            ShowError(CurLine,'Invalid number given or error in expression syntax.')
          else
          begin
            SkipSpaces; expr := ReadExprText;
            if expr = '++' then
            begin
              WriteByte($01); // increment
              WriteByte(arg1); AddSpecialSyntax := True;
            end
            else if expr = '--' then
            begin
              WriteByte($02); // decrement
              WriteByte(arg1); AddSpecialSyntax := True;
            end
            else
            begin
              if Copy(expr,Length(expr),1)='*' then
              begin
                expr := Copy(expr,1,Length(expr)-1);
                LinePos := LinePos - 1;
              end;
              SkipSpaces;
              arg2isstar := False;
              ArgText := ReadPlainText;
              if (ReadPlainText = '') and (Copy(LowerCaseLine,Linepos-Length(ArgText),1) = '*') then
              begin
                Inc(LinePos);
                ArgText := '*' + ReplaceDefine(ReadPlainText);
              end
              else ArgText := ReplaceDefine(ArgText);
              if (Copy(ArgText,1,1) = 'v') and (not arg2isstar) then
                arg2isvar := True
              else if (Copy(ArgText,1,2) = '*v') and (not arg2isstar) then
                arg2isstar := True;
              if arg2isvar then Val(Copy(ArgText,2,Length(ArgText)-1),arg2,code)
              else if arg2isstar then Val(Copy(ArgText,3,Length(ArgText)-2),arg2,code)
              else Val(ArgText,arg2,code);
              if (code <> 0) or (not (arg2 in [0..255])) then
                ShowError(CurLine,'Invalid number given or error in expression syntax.')
              else
              begin
                if (expr = '+=') and (not arg2isstar) then
                begin
                  if arg2isvar then WriteByte($06) {addv} else WriteByte($05); {addn}
                  WriteByte(arg1); WriteByte(arg2); AddSpecialSyntax := True;
                end
                else if (expr = '-=') and (not arg2isstar) then
                begin
                  if arg2isvar then WriteByte($08) {subv} else WriteByte($07); {subn}
                  WriteByte(arg1); WriteByte(arg2); AddSpecialSyntax := True;
                end
                else if (expr = '*=') and (not arg2isstar) then
                begin
                  if arg2isvar then WriteByte($A6) {mul.v} else WriteByte($A5); {mul.n}
                  WriteByte(arg1); WriteByte(arg2); AddSpecialSyntax := True;
                end
                else if (expr = '/=') and (not arg2isstar) then
                begin
                  if arg2isvar then WriteByte($A8) {div.v} else WriteByte($A7); {div.n}
                  WriteByte(arg1); WriteByte(arg2); AddSpecialSyntax := True;
                end
              else if expr = '=' then
              begin
                if (LinePos <= LineLength) and (EditLines[CurLine][LinePos] = ';') then
                begin  // must be assignn, assignv or rindirect
                  if arg2isvar then WriteByte($04) // assignv
                  else if arg2isstar then WriteByte($0A) // rindirect
                  else WriteByte($03); // assignv
                  WriteByte(arg1);
                  WriteByte(arg2);
                  AddSpecialSyntax := True;
                end
                else if arg2 <> arg1 then ShowError(CurLine,'Expression syntax error')
                else
                begin
                  SkipSpaces; expr2 := ReadExprText;
                  SkipSpaces;
                  ArgText := ReplaceDefine(ReadPlainText);
                  arg3isvar := (Copy(ArgText,1,1) = 'v');
                  if arg3isvar then Val(Copy(ArgText,2,Length(ArgText)-1),arg3,code)
                  else Val(ArgText,arg3,code);
                  if (code <> 0) or (not (arg3 in [0..255])) then
                    ShowError(CurLine,'Invalid number given or error in expression syntax.')
                  else
                  begin
                    if expr2 = '+' then
                    begin
                      if arg3isvar then WriteByte($06) {addv} else WriteByte($05); {addn}
                      WriteByte(arg1); WriteByte(arg3); AddSpecialSyntax := True;
                    end
                    else if expr2 = '-' then
                    begin
                      if arg3isvar then WriteByte($08) {subv} else WriteByte($07); {subn}
                      WriteByte(arg1); WriteByte(arg3); AddSpecialSyntax := True;
                    end
                    else if expr2 = '*' then
                    begin
                      if arg3isvar then WriteByte($A6) {mul.v} else WriteByte($A5); {mul.n}
                      WriteByte(arg1); WriteByte(arg3); AddSpecialSyntax := True;
                    end
                    else if expr2 = '/' then
                    begin
                      if arg3isvar then WriteByte($A8) {div.v} else WriteByte($A7); {div.n}
                      WriteByte(arg1); WriteByte(arg3); AddSpecialSyntax := True;
                    end
                    else ShowError(CurLine,'Expression syntax error');
                  end;  // if (code = 0) and (arg3 in [0..255])
                end;
              end  // if expr = '='
              else ShowError(CurLine,'Expression syntax error');
              end;
            end;  // if (expr <> '--') and (expr <> '++')
          end;  // if (code = 0) and (arg1 in [0..255])
        end // if Copy(ArgText,1,1) = 'v'
        else if Copy(ArgText,1,2) = '*v' then
        begin
          LinePos := LinePos - Length(CommandName) + 1;
          ArgText := ReplaceDefine(ReadPlainText);
          Val(Copy(ArgText,2,Length(ArgText)-1),arg1,code);
          if (code <> 0) or (not (arg1 in [0..255])) then
            ShowError(CurLine,'Invalid number given or error in expression syntax.')
          else
          begin
            SkipSpaces; expr := ReadExprText;
            if expr <> '=' then
              ShowError(CurLine,'Expression syntax error')
            else
            begin
              SkipSpaces;
              ArgText := ReplaceDefine(ReadPlainText);
              arg2isvar := (Copy(ArgText,1,1) = 'v');
              if arg2isvar then Val(Copy(ArgText,2,Length(ArgText)-1),arg2,code)
              else Val(ArgText,arg2,code);
              if (code <> 0) or (not (arg2 in [0..255])) then
                ShowError(CurLine,'Invalid number given or error in expression syntax.')
              else
              begin
                if arg2isvar then WriteByte($09) {lindirectv} else WriteByte($0B); {lindirectn}
                WriteByte(arg1); WriteByte(arg2); AddSpecialSyntax := True;
              end;
            end;
          end;  // if (code = 0) and (arg1 in [0..255])
        end  // if Copy(ArgText,1,2) = '*v'
        else LinePos := OldLinePos;
      end;

      {************************}
      function LabelNum(LabelName:string) : byte;
      {************************}
      var CurLabel : word;
      begin
        CurLabel := 1;
        while (CurLabel <= NumLabels) and (Labels[CurLabel].Name <> CommandName) do
          Inc(CurLabel);
        if CurLabel > NumLabels then LabelNum := 0
        else LabelNum := CurLabel;
      end;

      {************************}
      function LabelAtStartOfLine(LabelName:string) : boolean;
      {************************}
      var CurLinePos : word;
      begin
        CurLinePos := LinePos - Length(LabelName) - 2;
        while (CurLinePos > 0) and (LowerCaseLine[CurLinePos] = ' ') do
          Dec(CurLinePos);
        LabelAtStartOfLine := (CurLinePos = 0);
      end;

  {************************}
  begin
    InIf := False;
    BlockDepth := 0;
    NumGotos := 0;
    FinishedReading := False;
    CurLine := -1;
    NextLine;
    if FinishedReading then
      ShowError(CurLine,'Nothing to compile!')
    else
    repeat
    begin
      LastCommandWasReturn := False;
      if not InIf then
      begin
        if (LinePos <= LineLength) and (LowerCaseLine[LinePos] = '}') then
        begin
          Inc(LinePos);
          if BlockDepth = 0 then ShowError(CurLine,'"}" not at end of any command blocks.')
          else
          begin
            if ResPos = BlockStartDataLoc[BlockDepth] + 2 then
              ShowError(CurLine,'Command block must contain at least one command.');
            BlockLength[BlockDepth] := ResPos-BlockStartDataLoc[BlockDepth]-2;
            WriteByteAtLoc(BlockLength[BlockDepth] mod 256,BlockStartDataLoc[BlockDepth]);
            WriteByteAtLoc(BlockLength[BlockDepth] div 256,BlockStartDataLoc[BlockDepth]+1);
            BlockDepth := BlockDepth - 1;
            SkipSpaces;
            if (LinePos > LineLength) and (CurLine < EditLines.Count - 1) then NextLine;
            if Copy(LowerCaseLine,LinePos,4) = 'else' then
            begin
              Inc(LinePos,4);
              SkipSpaces;
              if not BlockIsIf[BlockDepth+1] then
                ShowError(CurLine,'"else" not allowed after command blocks that start with "else".')
              else if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> '{') then
                ShowError(CurLine,'"{" expected after else.')
              else
              begin
                Inc(LinePos);
                BlockDepth := BlockDepth + 1;
                BlockLength[BlockDepth] := BlockLength[BlockDepth] + 3;
                WriteByteAtLoc(BlockLength[BlockDepth] mod 256,BlockStartDataLoc[BlockDepth]);
                WriteByteAtLoc(BlockLength[BlockDepth] div 256,BlockStartDataLoc[BlockDepth]+1);
                BlockIsIf[BlockDepth] := True;
                WriteByte($FE);
                BlockStartDataLoc[BlockDepth] := ResPos;
                WriteByte($00);  // block length filled in later.
                WriteByte($00);
              end;
            end; //  if Copy(LowerCaseLine,LinePos,4) = 'else'
          end;  // if BlockDepth > 0
        end  // if LowerCaseLine[LinePos] = '}'
        else
        begin
          ReadCommandName;
          if CommandName = 'if' then
          begin
            WriteByte($FF);
            InIf := True;
            SkipSpaces;
            if (LinePos > LineLength) or (EditLines[CurLine][LinePos] <> '(') then
              ShowError(CurLine,'"(" expected at start of if statement.');
            Inc(LinePos,1);
            InIfBrackets := False;
            NumCommandsInIfStatement := 0;
            AwaitingNextTestCommand := True;
          end
          else if CommandName = 'else' then
          begin
            ShowError(CurLine,'"}" required before "else".');
          end
          else if CommandName = 'goto' then
          begin
            if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> '(') then
              ShowError(CurLine,'"(" expected.')
            else
            begin
              Inc(LinePos);
              ReadCommandName;
              CommandName := ReplaceDefine(CommandName);
              if LabelNum(CommandName) = 0 then
                ShowError(CurLine,'Unknown label "'+CommandName+'".')
              else if NumGotos >= MaxGotos then
                ShowError(CurLine,'Too many labels (max '+IntToStr(MaxLabels)+').')
              else
              begin
                Inc(NumGotos);
                Gotos[NumGotos].LabelNum := LabelNum(CommandName);
                WriteByte($FE);
                Gotos[NumGotos].DataLoc := ResPos;
                WriteByte($00);
                WriteByte($00);
                if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> ')') then
                  ShowError(CurLine,'")" expected after label name.');
                Inc(LinePos);
                if (LinePos > LineLength) or (LowerCaseLine[LinePos] <> ';') then
                  ShowError(CurLine,'";" expected after goto command.');
                Inc(LinePos);
              end;
            end;
          end
          else
          begin
            CommandNum := FindCommandNum(False,CommandName);
            EncounteredLabel := (LabelNum(CommandName) > 0);
            if EncounteredLabel and (LinePos <= LineLength) and (LowerCaseLine[LinePos] = ':') then
              Inc(LinePos)
            else EncounteredLabel := False;
            EncounteredLabel := EncounteredLabel and LabelAtStartOfLine(CommandName);
            if EncounteredLabel then
              Labels[LabelNum(CommandName)].Loc := ResPos
            else
            begin
              if (CommandNum = 255) then  // not found
              begin
                if not AddSpecialSyntax then
                  ShowError(CurLine,'Unknown action command "'+Copy(EditLines[CurLine],CommandNameStartPos,Length(CommandName))+'".')
              end
              else
              begin
                WriteByte(CommandNum);
                ReadArgs(False,CommandNum);
                if CommandNum = 0 then LastCommandWasReturn := True;
              end;
              if (LinePos > LineLength) or (EditLines[CurLine][LinePos] <> ';') then
                ShowError(CurLine,'";" expected after command.');
              Inc(LinePos);
            end;  // if we found a label
          end;  // command
        end;  // if LowerCaseLine[LinePos] <> '}'
      end;  // if not InIf
      if InIf then
      begin
        LastCommandWasReturn := False;
        if AwaitingNextTestCommand then
        begin
          if LowerCaseLine[LinePos] = '(' then
          begin
            if InIfBrackets then ShowError(CurLine,'Brackets too deep in if statement.');
            InIfBrackets := True;
            WriteByte($FC);
            NumCommandsInIfBrackets := 0;
            Inc(LinePos);
          end  // if LowerCaseLine[LinePos] = '('
          else if LowerCaseLine[LinePos] = ')' then
          begin
            if NumCommandsInIfStatement = 0 then
                ShowError(CurLine,'If statement must contain at least one command.')
            else if InIfBrackets and (NumCommandsInIfBrackets=0) then
                ShowError(CurLine,'Brackets must contain at least one command.')
            else ShowError(CurLine,'Expected statement but found closing bracket.');
            Inc(LinePos);
          end
          else
          begin
            NOTOn := False;
            if LowerCaseLine[LinePos] = '!' then
            begin
              NOTOn := True;
              Inc(LinePos);
            end;
            SkipSpaces;
            ReadCommandName;
//            CommandName := ReplaceDefine(CommandName);
            CommandNum := FindCommandNum(True,CommandName);
            if NOTOn then WriteByte($FD);
            if CommandNum = 255 then  // not found
            begin
              if not AddSpecialIFSyntax then
                ShowError(CurLine,'Unknown test command "'+Copy(EditLines[CurLine],CommandNameStartPos,Length(CommandName))+'".')
            end
            else
            begin
              WriteByte(CommandNum);
              ReadArgs(True,CommandNum);
            end;
            Inc(NumCommandsInIfStatement);
            if InIfBrackets then Inc(NumCommandsInIfBrackets);
            AwaitingNextTestCommand := False;
          end;
        end  // if AwaitingNextTestCommand
        else if LinePos <= LineLength then
        begin
          if LowerCaseLine[LinePos] = ')' then
          begin
            Inc(LinePos);
            if InIfBrackets then
            begin
              if NumCommandsInIfBrackets = 0 then
                ShowError(CurLine,'Brackets must contain at least one command.')
              else InIfBrackets := False;
              WriteByte($FC);
            end
            else
            begin
              if NumCommandsInIfStatement = 0 then
                ShowError(CurLine,'If statement must contain at least one command.')
              else
              begin
                SkipSpaces;
                if (LinePos > LineLength) or (EditLines[CurLine][LinePos] <> '{') then
                  ShowError(CurLine,'"{" expected after if statement.');
                Inc(LinePos);
                WriteByte($FF);
                if BlockDepth >= MaxBlockDepth then
                  ShowError(CurLine,'Too many nested blocks (max '+IntToStr(MaxBlockDepth)+').')
                else
                begin
                  BlockDepth := BlockDepth + 1;
                  BlockStartDataLoc[BlockDepth] := ResPos;
                  BlockIsIf[BlockDepth] := True;
                  WriteByte($00);   // block length filled in later.
                  WriteByte($00);
                end;
                InIf := False;
              end;
            end;
          end  // else if LowerCaseLine[LinePos] = ')'
          else if LowerCaseLine[LinePos] = '!' then
          begin
            ShowError(CurLine,'"!" can only be placed directly in front of a command.');
            Inc(LinePos);
          end
          else if Copy(LowerCaseLine,LinePos,2) = '&&' then
          begin
            if InIfBrackets then ShowError(CurLine,'"&&" not allowed within brackets.');
            AwaitingNextTestCommand := True;
            Inc(LinePos,2);
          end
          else if Copy(LowerCaseLine,LinePos,2) = '||' then
          begin
            if not InIfBrackets then
              ShowError(CurLine,'Commands to be ORred together must be placed within brackets.');
            AwaitingNextTestCommand := True;
            Inc(LinePos,2);
          end
          else
          begin
            if InIfBrackets then
              ShowError(CurLine,'Expected "||" or end of if statement.')
            else ShowError(CurLine,'Expected "&&" or end of if statement.');
          end;
        end; // if (not AwaitingNextTestCommand) and (LinePos <= LineLength)
      end;  // if InIf
      SkipSpaces;
      if ErrorOccured then FinishedReading := True
      else if LinePos > LineLength then NextLine;
    end; until FinishedReading;
    if (not LastCommandWasReturn) then ShowError(CurLine,'return command expected.');
    if InIf then
    begin
      if AwaitingNextTestCommand then
      begin
        if NumCommandsInIfStatement = 0 then ShowError(CurLine,'Expected test command.')
        else ShowError(CurLine,'Expected another test command or end of if statement.');
      end
      else
      begin
        if InIfBrackets then ShowError(CurLine,'Expected "||" or end of if statement.')
        else ShowError(CurLine,'Expected "&&" or end of if statement.');
      end;
    end
    else if BlockDepth > 0 then
      ShowError(CurLine,'"}" expected.');
    for CurGoto := 1 to NumGotos do
    begin
      GotoData := Labels[Gotos[CurGoto].LabelNum].Loc - Gotos[CurGoto].DataLoc - 2;
      WriteByteAtLoc(GotoData mod 256,Gotos[CurGoto].DataLoc);
      WriteByteAtLoc(GotoData div 256,Gotos[CurGoto].DataLoc+1);
    end;
  end;



{**************************************************}
begin
  CompileLogic.Size := 0;
  if LogEditWin[LogEditorNum].EditMemo.Lines.Count = 0 then
    ShowMessage('Nothing to compile!')
  else
  begin
    ErrorOccured := False;
    InputLines := TStringList.Create;
    InputLines.Assign(LogEditWin[LogEditorNum].EditMemo.Lines);
    CompiledLogic.Size := MaxResourceSize;
    GetMem(CompiledLogic.Data,CompiledLogic.Size);
    LogicSize := 0;

    ResPos := 2;
    Words := ReadWORDSTOKFile(GameDir+'WORDS.TOK');
    if Words.Exists then
    begin
      OBJECTFileEncrypted := False;
      OBJECTFile := ReadOBJECTFile(GameDir+'OBJECT',OBJECTFileEncrypted);
      if OBJECTFile.Exists then
      begin
        InvObjects := TStringList.Create;
        InvObjects.Assign(OBJECTFile.ItemNames);
        OBJECTFile.ItemNames.Free;
        NumInvObjects := InvObjects.Count;
        for CurInvObject := 0 to NumInvObjects - 1 do
        begin
          InvObjects[CurInvObject] := LowerCase(InvObjects[CurInvObject]);  // words already in lower case in file so we don't need to convert them
          if Length(InvObjects[CurInvObject]) > 0 then
            for CurChar := Length(InvObjects[CurInvObject]) downto 1 do
            if InvObjects[CurInvObject][CurChar] = '"' then
              InvObjects[CurInvObject] := Copy(InvObjects[CurInvObject],1,CurChar-1) + '\"' + Copy(InvObjects[CurInvObject],CurChar+1,255);
        end;

        DefineNames := TStringList.Create;
        DefineValues := TStringList.Create;

        RemoveComments(InputLines);
        if not ErrorOccured then AddIncludes;  // creates EditLines and frees InputLines
        if not ErrorOccured then ReadDefines;
        if not ErrorOccured then ReadPredefinedMessages;
        if not ErrorOccured then ReadLabels;
        if not ErrorOccured then CompileCommands;
        if not ErrorOccured then WriteMessageSection;

        if Words.NumGroups > 0 then
          for CurWordGroup := 0 to Words.NumGroups - 1 do
            Words.WordGroup[CurWordGroup].Words.Free;
        InvObjects.Free;
        DefineNames.Free;
        DefineValues.Free;
      end;
    end;
    if ErrorOccured then LogicSize := 0;
    OutputLogic.Size := LogicSize;
    if LogicSize > 0 then
    begin
      GetMem(OutputLogic.Data,OutputLogic.Size);
      Move(CompiledLogic.Data^,OutputLogic.Data^,OutputLogic.Size);
      CompileLogic.Data := OutputLogic.Data;
    end;
    CompileLogic.Size := OutputLogic.Size;
    FreeMem(CompiledLogic.Data,CompiledLogic.Size);
    EditLines.Free;
    IncludeFilenames.Free;
  end;
end;

end.
