/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

#pragma once

/*
 * A SoundPlayer class is responsible for playback of an SCI0 sound resource.
 * It posts playback events (see Messages.h) to the message queue of a window
 * handle passed in at construction time. A SoundPlayer subclass must also
 * implement a control interface consisting of start and stop operations.
 * Future versions of SoundPlayer may also require pause, fade, seek, volume,
 * or other operations.
 *
 * Using a sound player involves creating it, initializing it, and playing
 * sound resources with Start() and Stop(). The Initialize() method takes an
 * optional pointer to a DataStream with patch data. The format of patch data
 * is device specific and will not be used by all players. The player should
 * be closed with Cleanup() before being disposed.
 *
 * RealtimeSoundPlayer implements a timing method that plays the sound in
 * realtime in an independent thread. It calls virtual functions to handle
 * MIDI events.
 *
 * SoundPlayers may throw DeviceException from any method. If an exception is
 * thrown in a separate thread, it is stored in the player object for the main
 * thread.
 */

#include <string>

class SoundResource;
struct sound_event;
class DataStream;

class SoundPlayer {
public:
	SoundPlayer(HWND listener);
	virtual ~SoundPlayer();
	virtual void Initialize(DataStream *patch = 0) = 0;
	virtual void Cleanup() = 0;
	virtual void Start(SoundResource *sound, int filter) = 0;
	virtual void Stop() = 0;
	bool IsError();
	std::string GetError();
protected:
	HWND mListener;
	bool mError;
	std::string mErrorMessage;
};

class RealtimePlayer : public SoundPlayer {
public:
	// subclasses should never have to worry about these
	RealtimePlayer(HWND listener);
	virtual ~RealtimePlayer();
	virtual void Initialize(DataStream *patch = 0);
	virtual void Cleanup();
	virtual void Start(SoundResource *sound, int filter);
	virtual void Stop();
protected:
	// subclasses should implement these
	virtual void OpenDevice(DataStream *patch) = 0;
	virtual void CloseDevice() = 0;
	virtual void MapChannelVoices(int channel, int numVoices) = 0;
	virtual void HandleEvent(const sound_event &event) = 0;
protected:
	// subclasses should never have to worry about these
	void ThreadProc();
	static DWORD WINAPI StaticThreadProc(LPVOID lpvSelf);
	void CleanupThread();

	bool mInitialized;
	bool mPlaying;
	HANDLE mThreadHandle;
	DWORD mThreadId;
	SoundResource *mSound;
	int mFilter;
	volatile bool mPleaseStop;
};
