/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

// SoundDisplay.cpp : implementation file
//

#include "stdafx.h"
#include "Soundbox.h"
#include "SoundDisplay.h"
#include "SoundResource.h"

// CSoundDisplay

IMPLEMENT_DYNAMIC(CSoundDisplay, CWnd)
CSoundDisplay::CSoundDisplay()
: mShowNoteOns(true), mShowVoiceControls(true), mShowLoopPoint(true),
  mShowCues(true), mShowTimeRules(true), mSound(0), mChannelNumbersFont(0),
  mFilter(0), mPlayPosition(-1)
{
	mTimeRuleFont.CreateFont(15, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, "Verdana");
	mMemDc.CreateCompatibleDC(0);

	mTimeRulePen.CreatePen(PS_DOT, 1, RGB(0, 0, 0));
	mDirectCuePen.CreatePen(PS_SOLID, 1, RGB(0, 128, 0));
	mCumulativeCuePen.CreatePen(PS_SOLID, 1, RGB(128, 128, 0));
	mLoopPointPen.CreatePen(PS_SOLID, 3, RGB(0, 0, 128));
	mPlayPositionPen.CreatePen(PS_SOLID, 3, RGB(128, 0, 0));
	mNoteOnLitPen.CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
	mNoteOnDarkPen.CreatePen(PS_SOLID, 1, RGB(192, 192, 192));
	mVoiceControlLitPen.CreatePen(PS_SOLID, 1, RGB(128, 0, 0));
	mVoiceControlDarkPen.CreatePen(PS_SOLID, 1, RGB(255, 192, 192));
}

CSoundDisplay::~CSoundDisplay()
{
	if (mChannelNumbersFont) delete mChannelNumbersFont;
}

bool CSoundDisplay::GetShowNoteOns()
{
	return mShowNoteOns;
}

void CSoundDisplay::SetShowNoteOns(bool val)
{
	if (mShowNoteOns != val) {
		mShowNoteOns = val;
		PaintStaticDisplay();
		RedrawWindow();
	}
}

bool CSoundDisplay::GetShowVoiceControls()
{
	return mShowVoiceControls;
}

void CSoundDisplay::SetShowVoiceControls(bool val)
{
	if (mShowVoiceControls != val) {
		mShowVoiceControls = val;
		PaintStaticDisplay();
		RedrawWindow();
	}
}

bool CSoundDisplay::GetShowLoopPoint()
{
	return mShowLoopPoint;
}

void CSoundDisplay::SetShowLoopPoint(bool val)
{
	if (mShowLoopPoint != val) {
		mShowLoopPoint = val;
		PaintStaticDisplay();
		RedrawWindow();
	}
}

bool CSoundDisplay::GetShowCues()
{
	return mShowCues;
}

void CSoundDisplay::SetShowCues(bool val)
{
	if (mShowCues != val) {
		mShowCues = val;
		PaintStaticDisplay();
		RedrawWindow();
	}
}

bool CSoundDisplay::GetShowTimeRules()
{
	return mShowTimeRules;
}

void CSoundDisplay::SetShowTimeRules(bool val)
{
	if (mShowTimeRules != val) {
		mShowTimeRules = val;
		PaintStaticDisplay();
		RedrawWindow();
	}
}

void CSoundDisplay::SetSound(SoundResource *sound)
{
	mSound = sound;
	if (GetSafeHwnd()) {
		CalcDisplayParameters();
		PaintStaticDisplay();
		RedrawWindow();
	}
}

void CSoundDisplay::CalcDisplayParameters()
{
	RECT rect;
	GetClientRect(&rect);

	mWidth = rect.right;
	mHeight = rect.bottom;

	mHeaderSpace = 25;
	mFooterSpace = 50;

	mChannelHeight = (mHeight - (mHeaderSpace + mFooterSpace)) / 16;
	if (mChannelHeight < 12) mChannelHeight = 12;
	mChannelDisplayHeight = mChannelHeight * 2 / 3;

	// create a font for displaying channels
	if (mChannelNumbersFont) delete mChannelNumbersFont;
	mChannelNumbersFont = new CFont();
	mChannelNumbersFont->CreateFont(mChannelDisplayHeight, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, "Courier New");

	// figure out how much horizontal space a channel label will take
	CDC testDc;
	testDc.CreateCompatibleDC(0);
	testDc.SelectObject(*mChannelNumbersFont);
	SIZE sz = testDc.GetTextExtent("0000");
	mChannelNumberBarWidth = sz.cx;

	// make a flags bar approximately the same width but a multiple of 8
	mChannelFlagWidth = mChannelNumberBarWidth / 8;
	if (mChannelFlagWidth < 6) mChannelFlagWidth = 6;
	mChannelFlagsBarWidth = mChannelFlagWidth * 8;

	// put white space on either side of the event display
	mLeftGutterWidth = 50;
	mRightGutterWidth = 50;

	// how much horizontal space do we have to display events
	mDisplayX = mChannelNumberBarWidth + mChannelFlagsBarWidth + mLeftGutterWidth;
	mDisplayWidth = mWidth - (mDisplayX + mRightGutterWidth);
	if (mDisplayWidth < 100) mDisplayWidth = 100;

	// where and how long are display lines (time rules, cues, etc)
	mDisplayLineTopY = mHeaderSpace / 2;
	mDisplayLineBottomY = mHeight - ((mHeaderSpace - mDisplayLineTopY) + mFooterSpace / 2);
	mDisplayLineHeight = mDisplayLineBottomY - mDisplayLineTopY;

	// everything below this point requires a loaded sound
	if (mSound == 0) return;
	if (mSound->GetNumEvents() == 0) return;

	// don't let sound rules lie within 50 pixels of each other
	int soundTime = mSound->GetTotalTime();
	mRuleThreshold = 50 * soundTime / mDisplayWidth;

	// we'll aim for time rules every 100 pixels, then round them to the
	//   nearest second or tenth of a second as appropraite
	int desiredNumRules = mDisplayWidth / 100;
	int desiredRuleFrequency = soundTime / desiredNumRules;
	if (desiredRuleFrequency < mRuleThreshold) desiredRuleFrequency = mRuleThreshold;
	if (soundTime > 600) {
		// sound is longer than 10 seconds, so round time rules to the nearest second
		mRuleFrequency = desiredRuleFrequency - desiredRuleFrequency % 60;
	} else if (soundTime > 60) {
		// sound is longer than 1 second, so round time rules to the nearest tenth of a second
		mRuleFrequency = desiredRuleFrequency - desiredRuleFrequency % 6;
	}
}

void CSoundDisplay::PaintStaticDisplay()
{
	mMemDc.FillSolidRect(0, 0, mWidth, mHeight, RGB(255, 255, 255));
	PaintChannelNumbers(&mMemDc);
	PaintChannelFlags(&mMemDc);
	if (mShowTimeRules) PaintTimeRules(&mMemDc);
	PaintEvents(&mMemDc);
}

int CSoundDisplay::CalcX(int time)
{
	if (mSound == 0) return 0;
	if (mSound->GetNumEvents() == 0) return 0;
	if (mSound->GetTotalTime() == 0) return 0;

	// return the x position in the client area for a given time in ticks
	return mDisplayX + time * mDisplayWidth / mSound->GetTotalTime();
}

void CSoundDisplay::PaintChannelNumbers(CDC *dc)
{
	dc->FillSolidRect(0, 0, mChannelNumberBarWidth, mHeight, RGB(192, 192, 192));
	dc->SelectObject(mChannelNumbersFont);
	dc->SetTextColor(RGB(0, 0, 0));
	dc->SetBkMode(TRANSPARENT);

	char txt[16];
	for (int i = 0; i < 16; ++i) {
		sprintf(txt, " %2d", i);
		dc->TextOut(0, mHeaderSpace + i * mChannelHeight, txt);
	}
}

void CSoundDisplay::PaintChannelFlags(CDC *dc)
{
	dc->FillSolidRect(mChannelNumberBarWidth, 0, mChannelFlagsBarWidth, mHeight, RGB(64, 64, 64));

	if (mSound == 0) return;

	for (int i = 0; i < 16; ++i) {
		int flags = mSound->GetChannelPlayFlag(i);
		for (int j = 0; j < 8; ++j) {
			if (flags & (1 << j)) {
				dc->FillSolidRect(mChannelNumberBarWidth + (7 - j) * mChannelFlagWidth,
					mHeaderSpace + i * mChannelHeight, mChannelFlagWidth, mChannelDisplayHeight,
					CSoundboxApp::FILTER_COLORS_LIGHT[j]);
			}
		}
	}
}

void CSoundDisplay::PaintTimeRules(CDC *dc)
{
	// we can only have time rules if we have a non-empty loaded sound
	if (mSound == 0) return;
	if (mSound->GetNumEvents() == 0) return;

	// set up the DC
	dc->SelectObject(mTimeRuleFont);
	dc->SelectObject(mTimeRulePen);
	dc->SetTextColor(RGB(0, 0, 0));
	dc->SetBkColor(RGB(255, 255, 255));

	// paint time rules until we overrun the sound time
	int soundTime = mSound->GetTotalTime();
	for (int time = 0; time < soundTime - mRuleThreshold; time += mRuleFrequency) {
		PaintDisplayLine(dc, time);
		PaintTimeLabel(dc, time);
	}

	// paint a rule for the very end of the sound
	PaintDisplayLine(dc, soundTime);
	PaintTimeLabel(dc, soundTime);
}

void CSoundDisplay::PaintTimeLabel(CDC *dc, int time)
{
	// for efficiency, assume the DC has already been set up correctly

	int xCenter = CalcX(time);
	char txt[32];

	if (time > 6) {
		int minutes = time / 3600;
		time = time % 3600;
		int seconds = time / 60;
		time = time % 60;
		int tenths = time / 6;

		if (tenths == 0) {
			sprintf(txt, "%d:%02d", minutes, seconds);
		} else {
			sprintf(txt, "%d:%02d.%d", minutes, seconds, tenths);
		}
	} else if (time > 0) {
		sprintf(txt, "%dt", time);
	} else {
		sprintf(txt, "0");
	}

	SIZE extent = dc->GetTextExtent(txt);
	int x = xCenter - extent.cx / 2;
	int y = mDisplayLineBottomY + 10;

	dc->TextOut(x, y, txt);
}

void CSoundDisplay::PaintDisplayLine(CDC *dc, int time)
{
	// for efficiency, assume the DC has already been set up correctly
	int xPos = CalcX(time);
	dc->MoveTo(xPos, mDisplayLineTopY);
	dc->LineTo(xPos, mDisplayLineBottomY);
}

void CSoundDisplay::PaintEventLine(CDC *dc, int time, int channel)
{
	// for efficiency, assume the DC has already been set up correctly
	int xPos = CalcX(time);
	int yPos = mHeaderSpace + channel * mChannelHeight;
	dc->MoveTo(xPos, yPos);
	dc->LineTo(xPos, yPos + mChannelDisplayHeight);
}

void CSoundDisplay::PaintEvents(CDC *dc)
{
	if (mSound == 0) return;
	int n = mSound->GetNumEvents();

	for (int i = 0; i < n; ++i) {
		sound_event event = mSound->GetEvent(i);
		bool lit = (mSound->GetChannelPlayFlag(event.channel) & mFilter) != 0;

		if ((event.status == sound_event::NOTE_ON) && (event.param2 != 0)) {
			// note on event
			if (mShowNoteOns) {
				dc->SelectObject(lit ? mNoteOnLitPen : mNoteOnDarkPen);
				PaintEventLine(dc, event.absoluteTime, event.channel);
			}
		} else if (event.status == sound_event::CONTROL_CHANGE) {
			if (event.param1 == 0x4B) {
				// voice control
				if (mShowVoiceControls) {
					dc->SelectObject(lit ? mVoiceControlLitPen : mVoiceControlDarkPen);
					PaintEventLine(dc, event.absoluteTime, event.channel);
				}
			} else if (event.param1 == 0x60) {
				// cumulative cue
				if (mShowCues) {
					dc->SelectObject(mCumulativeCuePen);
					PaintDisplayLine(dc, event.absoluteTime);
				}
			}
		} else if ((event.status == sound_event::PATCH_CHANGE) && (event.channel == 15)) {
			if (event.param1 == 127) {
				// loop point
				if (mShowLoopPoint) {
					dc->SelectObject(mLoopPointPen);
					PaintDisplayLine(dc, event.absoluteTime);
				}
			} else {
				// direct cue
				if (mShowCues) {
					dc->SelectObject(mDirectCuePen);
					PaintDisplayLine(dc, event.absoluteTime);
				}
			}
		}
	}
}

void CSoundDisplay::SetFilter(int filter)
{
	mFilter = filter;
	PaintStaticDisplay();
	RedrawWindow();
}

void CSoundDisplay::SetPlayPosition(int position)
{
	if (mPlayPosition == position) return;   // effectless call

	RECT updateArea;
	updateArea.top = mDisplayLineTopY;
	updateArea.bottom = mDisplayLineBottomY;

	if (mPlayPosition == -1) {
		// we only need to redraw the area with the new play position
		int x = CalcX(position);
		updateArea.left = x - 1;
		updateArea.right = x + 3;
	} else if (position == -1) {
		// we only need to redraw the area with the old play position
		int x = CalcX(mPlayPosition);
		updateArea.left = x - 1;
		updateArea.right = x + 3;
	} else {
		// redraw a rectangle that covers both old and new
		int x1 = CalcX(position);
		int x2 = CalcX(mPlayPosition);
		updateArea.left = (x1 < x2) ? (x1 - 1) : (x2 - 1);
		updateArea.right = (x1 > x2) ? (x1 + 3) : (x2 + 3);
	}

	mPlayPosition = position;
	RedrawWindow(&updateArea);
}

void CSoundDisplay::PaintBuffer()
{
	CalcDisplayParameters();
	PaintStaticDisplay();
	RedrawWindow();
}


BEGIN_MESSAGE_MAP(CSoundDisplay, CWnd)
	ON_WM_PAINT()
	ON_WM_SIZE()
END_MESSAGE_MAP()



// CSoundDisplay message handlers


void CSoundDisplay::OnPaint()
{
	CPaintDC dc(this); // device context for painting

	// copy from the static buffer to the DC
	int x = dc.m_ps.rcPaint.left;
	int y = dc.m_ps.rcPaint.top;
	int width = dc.m_ps.rcPaint.right - x;
	int height = dc.m_ps.rcPaint.bottom - y;
	dc.BitBlt(x, y, width, height, &mMemDc, x, y, SRCCOPY);

	// overlay the play position if we're playing the sound
	if (mPlayPosition != -1) {
		int px = CalcX(mPlayPosition);
		dc.SelectObject(mPlayPositionPen);
		PaintDisplayLine(&dc, mPlayPosition);
	}
}

void CSoundDisplay::OnSize(UINT nType, int cx, int cy)
{
	CWnd::OnSize(nType, cx, cy);

	// create a new buffer bitmap to cover the new area
	HDC screenHDC = ::GetDC(0);
	CDC screenDC;
	screenDC.Attach(screenHDC);
	mMemBmp.DeleteObject();
	mMemBmp.CreateCompatibleBitmap(&screenDC, cx, cy);
	mMemDc.SelectObject(mMemBmp);
	screenDC.Detach();
	::ReleaseDC(0, screenHDC);

	// fill in the buffer
	CalcDisplayParameters();
	PaintStaticDisplay();
}
