/////////////////////////////////////////////////////
//
// PositionCtrlDlgUI.h
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides event handlers for user-interface
//    events such as button presses, etc., for the
//    Position Control dialog; also provides functions
//    that update the user interface as necessary to
//    reflect the current state of the application
//
// Portability information:
//    this file contains code written specifically for
//    a Win32 build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include "resource.h"
#include "BLGDefs.h"
#include "GeneralUI.h"
#include "PositionCtrlDlgUI.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in PositionCtrlDlgEventHandlers.cpp
extern HWND g_hwndPositionCtrlDlg;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in PositionCtrlDlg.cpp
extern EGOPOSITIONCONTROL g_epcTemp;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// OnIgnorePrevRoom
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the IDC_CHECK_IGNORE_PREV_ROOM
//    checkbox
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnIgnorePrevRoom()
{
    BOOL bIgnorePrevRoom = IsCheckboxChecked(g_hwndPositionCtrlDlg, 
                                             IDC_CHECK_IGNORE_PREV_ROOM);
    
    // the "Coming from room" edit box and the "Ignore previous room"
    // checkbox are mutually exclusive input items...disable IDC_EDIT_SRC_ROOM
    // if this checkbox is checked
    EnableWindow(GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_SRC_ROOM),
                 !bIgnorePrevRoom);

    TryToEnableOKButton();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEditSrcRoom
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_EDIT_SRC_ROOM
//    edit box
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEditSrcRoom(WPARAM wParam)
{
    // there can be multiple types of events for an edit box,
    // determine which kind this is and dispatch to an appropriate
    // handler
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        return OnEditSrcRoomENUpdate();
    }

    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnEditSrcRoomENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the EN_UPDATE notification for the
//    IDC_EDIT_SRC_ROOM edit box
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEditSrcRoomENUpdate()
{
    char szBuffer[MAX_DIGITS + 1];
    HWND hwndEdit = GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_SRC_ROOM);
    GetWindowText(hwndEdit, szBuffer, MAX_DIGITS + 1);
 
    // the "Coming from room" edit box and the "Ignore previous room"
    // checkbox are mutually exclusive input items...disable
    // IDC_CHECK_IGNORE_PREV_ROOM if the edit box is not empty

    if (szBuffer[0] == '\0')
    {
        EnableWindow(GetDlgItem(g_hwndPositionCtrlDlg, 
                                IDC_CHECK_IGNORE_PREV_ROOM),
                     TRUE);
        
    }
    else
    {
        EnableWindow(GetDlgItem(g_hwndPositionCtrlDlg,
                                IDC_CHECK_IGNORE_PREV_ROOM),
                     FALSE);
        
    }

    TryToEnableOKButton();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnOK
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the OK button for the
//    Position Control dialog
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnOK()
{
    char szBuffer[MAX_DIGITS + 1];
    int nTemp;
    HWND hwndEdit;

    hwndEdit = GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_SRC_ROOM);
    GetWindowText(hwndEdit, szBuffer, MAX_DIGITS + 1);

    // determine the source room (or if this is an absolute positioning
    // control item)
    if (szBuffer[0] == '\0')
    {
        if (IsCheckboxChecked(g_hwndPositionCtrlDlg, 
                              IDC_CHECK_IGNORE_PREV_ROOM))
        {
            g_epcTemp.nSrcRoom = ABSOLUTE_POSITIONING_SRC_ROOM;
        }
        else
        {
            // ok should be disabled in this case, if it's not, we got
            // a problem
            assert(FALSE);
        }
    }
    else
    {
        nTemp = atoi(szBuffer);
        g_epcTemp.nSrcRoom = nTemp;

        // bounds check the source room
        if (g_epcTemp.nSrcRoom < 0 ||
            g_epcTemp.nSrcRoom > AGI_MAX_VALUE)
        {
            MessageBox(g_hwndPositionCtrlDlg,
                       "The \"Coming from room\" field must have a value "
                       "between 0 and 255.", "Error", MB_OK);
            SetFocus(GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_SRC_ROOM));
            return TRUE;
        }
    }

    // get the x position for this position control item

    hwndEdit = GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_EGO_POS_X);
    GetWindowText(hwndEdit, szBuffer, MAX_DIGITS + 1);
    if (szBuffer[0] == '\0')
    {
        // ok should be disabled in this case
        assert(FALSE);
    }
    else
    {
        nTemp = atoi(szBuffer);
        g_epcTemp.nPosX = nTemp;

        // bounds check the x position
        if (g_epcTemp.nPosX < 0 ||
            g_epcTemp.nPosX > AGI_MAX_VALUE)
        {
            MessageBox(g_hwndPositionCtrlDlg,
                       "The x position must be between 0 and 255.",
                       "Error", MB_OK);
            SetFocus(GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_EGO_POS_X));
            return TRUE;
        }
    }

    // get the y position for this position control item

    hwndEdit = GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_EGO_POS_Y);
    GetWindowText(hwndEdit, szBuffer, MAX_DIGITS + 1);
    if (szBuffer[0] == '\0')
    {
        // ok should be disabled in this case
        assert(FALSE);
    }
    else
    {
        nTemp = atoi(szBuffer);
        g_epcTemp.nPosY = nTemp;

        // bounds check the y position
        if (g_epcTemp.nPosY < 0 ||
            g_epcTemp.nPosY > AGI_MAX_VALUE)
        {
            MessageBox(g_hwndPositionCtrlDlg,
                       "The y position must be between 0 and 255.",
                       "Error", MB_OK);
            SetFocus(GetDlgItem(g_hwndPositionCtrlDlg, IDC_EDIT_EGO_POS_Y));
            return TRUE;
        }
    }

    // if we're at this point, then there's no problems...end the dialog box
    EndDialog(g_hwndPositionCtrlDlg, IDOK);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// TryToEnableOKButton
//
/////////////////////////////////////////////////////
//
// Purpose:
//    looks at the current state of input in the
//    dialog box and determine if it is appropriate
//    to enable the OK button (note that this function
//    does not bounds check, because disabling the OK
//    button for out-of-bounds values would just make
//    the user wonder what's going on)
//
/////////////////////////////////////////////////////

void TryToEnableOKButton()
{
    // there are three checks that must hold in order for the OK button
    // to be enabled:
    // 1. there must be a source room -- either this or the absolute
    //    positioning box must be checked
    // 2. there must be an x position
    // 3. there must be a y position
    //
    // if any of these conditions fails, then the OK button is disabled
    BOOL bCheck1OK = FALSE;
    BOOL bCheck2OK = FALSE;
    BOOL bCheck3OK = FALSE;

    HWND hwndOK = GetDlgItem(g_hwndPositionCtrlDlg, IDOK);

    if (IsCheckboxChecked(g_hwndPositionCtrlDlg, IDC_CHECK_IGNORE_PREV_ROOM))
    {
        bCheck1OK = TRUE;
    }
    else if (GetWindowTextLength(GetDlgItem(g_hwndPositionCtrlDlg,
                                            IDC_EDIT_SRC_ROOM)) > 0)
    {
        bCheck1OK = TRUE;
    }

    if (!bCheck1OK)
    {
        EnableWindow(hwndOK, FALSE);
        return;
    }

    if (GetWindowTextLength(GetDlgItem(g_hwndPositionCtrlDlg, 
                                       IDC_EDIT_EGO_POS_X)) > 0)
    {
        bCheck2OK = TRUE;
    }

    if (!bCheck2OK)
    {
        EnableWindow(hwndOK, FALSE);
        return;
    }

    if (GetWindowTextLength(GetDlgItem(g_hwndPositionCtrlDlg, 
                                       IDC_EDIT_EGO_POS_Y)) > 0)
    {
        bCheck3OK = TRUE;
    }

    if (!bCheck3OK)
    {
        EnableWindow(hwndOK, FALSE);
        return;
    }

    EnableWindow(hwndOK, TRUE);
}

/////////////////////////////////////////////////////
//
// OnEditEgoPosX
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_EDIT_EGO_POS_X
//    edit box
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEditEgoPosX(WPARAM wParam)
{
    // there can be multiple types of events for an edit box,
    // determine which kind this is and dispatch to an appropriate
    // handler
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        return OnEditEgoPosXENUpdate();
    }

    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnEdtiEgoPosXENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the EN_UPDATE notification for the
//    IDC_EDIT_EGO_POS_X edit box
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEditEgoPosXENUpdate()
{
    TryToEnableOKButton();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEditEgoPosY
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the IDC_EDIT_EGO_POS_Y
//    edit box
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEditEgoPosY(WPARAM wParam)
{
    // there can be multiple types of events for an edit box,
    // determine which kind this is and dispatch to an appropriate
    // handler
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        return OnEditEgoPosYENUpdate();
    }

    return FALSE;

}

/////////////////////////////////////////////////////
//
// OnEditEgoPosYENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the EN_UPDATE notification for the
//    IDC_EDIT_EGO_POS_Y edit box
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEditEgoPosYENUpdate()
{
    TryToEnableOKButton();
    return TRUE;
}
