/////////////////////////////////////////////////////
//
// Persistence.h
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    defines the interface for the BLG to save
//    state information from one run to the next
//
// Portability information:
//    the code in this file should be portable
//    without much trouble
//
/////////////////////////////////////////////////////

// make sure this header can't be included twice
#if !defined(PERSISTENCE_H_INCLUDED)
#define PERSISTENCE_H_INCLUDED

#include "BLGDefs.h"

// use BLG_STATE_DEFINE_NAME_LIST to save or load the list of define names 
// that will be used by the BLG if the option "Use define names" is selected;
// in this case, the pvData parameter to the state info functions must
// be a pointer to a CDefineNameList object; the storage location of the
// define name list is tracked internally by the state info functions;
// although the specific location can be requested by the caller in the
// CDefineNameList object passed to SaveBLGStateInfo, the implementation
// is free to ignore this request and store the list wherever it likes; the
// caller should not be written so that it is dependent on where or how the
// define name list is stored
#define BLG_STATE_DEFINE_NAME_LIST 0

// use BLG_STATE_OPTIONS to save or load the BLG options as defined in
// the COptions class; if BLG_STATE_OPTIONS is the first parameter to
// one of the state info functions, then the pvData parameter to those
// functions must be a pointer to a valid COptions object
#define BLG_STATE_OPTIONS          1

// use BLG_STATE_LOOK_WORD_GROUP to save or load the word group in the
// user's WORDS.TOK file for the word "look" (in most cases, this will
// be 20, because that is what the template game uses, but if somebody
// uses a different group, they shouldn't have to set the value every
// time); if BLG_STATE_LOOK_WORD_GROUP is the first parameter to one
// of the state info functions, then the pvData parameter to those
// functions must be a pointer to an int
#define BLG_STATE_LOOK_WORD_GROUP  2

// use BLG_STATE_LAST_GAME_PATH to save or load the last path where
// a user chose to generate the BLG files; if BLG_STATE_LAST_GAME_PATH is
// the first parameter to one of the state info functions, then the pvData
// parameter to those functions must be a pointer to a null-terminated
// string big enough to hold _MAX_PATH + 1 characters including the null
// terminator
#define BLG_STATE_LAST_GAME_PATH   3

/////////////////////////////////////////////////////
//
// SaveBLGStateInfo
//
/////////////////////////////////////////////////////
//
// Purpose:
//    saves various information regarding the state
//    of the BLG so that the user doesn't have to
//    keep setting their options every time the BLG
//    is run
// Parameter ulInfoType:
//    a ULONG specifying what kind of information is
//    to be stored -- use the constants defined in
//    Persistence.h for this parameter
// Parameter pvData:
//    a pointer to a data type appropriate for the
//    kind of information that is being stored;
//    depending on the value of ulInfoType, must not
//    be NULL; see the list of ulInfoType constants
//    in Persistence.h for a description of what kind
//    of pointer is expected for various values of
//    ulInfoType
//
/////////////////////////////////////////////////////

void SaveBLGStateInfo(ULONG ulInfoType, void* pvData);

/////////////////////////////////////////////////////
//
// LoadBLGStateInfo
//
/////////////////////////////////////////////////////
//
// Purpose:
//    loads various information regarding the state
//    of the BLG so that the user doesn't have to
//    keep setting their options every time the BLG
//    is run
// Parameter ulInfoType:
//    a ULONG specifying what kind of information is
//    to be loaded -- use the constants defined in
//    Persistence.h for this parameter
// Parameter pvData:
//    a pointer to a data type appropriate for the
//    kind of information that is being loaded;
//    depending on the value of ulInfoType, must not
//    be NULL; see the list of ulInfoType constants
//    in Persistence.h for a description of what kind
//    of pointer is expected for various values of
//    ulInfoType
//
/////////////////////////////////////////////////////

void LoadBLGStateInfo(ULONG ulInfoType, void* pvData /* output parameter */);

#endif // !defined(PERSISTENCE_H_INCLUDED)