/////////////////////////////////////////////////////
//
// MainDlgUI.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides event handlers for user-interface
//    events such as button presses, etc., for the
//    main BLG dialog; also provides functions
//    that update the user interface as necessary to
//    reflect the current state of the application
//
// Portability information:
//    this file contains code written specifically
//    for a Windows build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cstdio>
#include <cassert>
#include <shlobj.h>
#include "resource.h"
#include "BLGDefs.h"
#include "Persistence.h"
#include "EgoPositionControlArray.h"
#include "AboutBox.h"
#include "PositionCtrlDlg.h"
#include "EdgeCodeAdvancedDlg.h"
#include "DefineNameList.h"
#include "DefineNameDlg.h"
#include "OptionsDlg.h"
#include "EntryLookDlg.h"
#include "FirstRoomCtrlDlg.h"
#include "LogicGeneration.h"
#include "SourceCodeGeneration.h"
#include "MessageBoxes.h"
#include "GeneralUI.h"
#include "MainDlgUI.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in MainDlgEventHandlers.cpp
extern HWND g_hwndMainDlg;
/////////////////////////////////////////////////////

CMainLogicOptions g_mlopt;
CEntryLookInfo g_elinfo;
CFirstRoomCtrl g_frctrl;
CEgoPositionControlArray g_aepc;
CEdgeCodeInfo g_ecinfo;
CDefineNameList g_deflist;
COptions g_options;

/////////////////////////////////////////////////////
//
// OnUseRoomNumPic
//
/////////////////////////////////////////////////////
//
// Purpose:
//    event handler for the Use room_no for pic
//    checkbox
// Return value:
//    TRUE if the function handled the event; FALSE
//    if Windows needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnUseRoomNumPic()
{
    // find out whether the checkbox is checked or unchecked
    // (note: it is an autocheckbox, so this is just finding
    // out what its current state is -- it doesn't need to be
    // set)
    BOOL bUseRoomNumChecked = IsCheckboxChecked(g_hwndMainDlg, 
        IDC_CHECK_USE_ROOM_NUM_PIC);

    // enable or disable the pic number edit box depending on
    // whether the user wants to use room_no to determine
    // the pic number or not
    EnableWindow(GetDlgItem(g_hwndMainDlg, IDC_EDIT_PIC_NUMBER),
        !bUseRoomNumChecked);

    if (bUseRoomNumChecked)
    {
        // if Use room_no for pic is checked, then update the
        // pic number edit box with the room number
        char szBuffer[MAX_DIGITS + 1];
        HWND hwndEdit = GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_NUMBER);
        GetWindowText(hwndEdit, szBuffer, MAX_DIGITS + 1);
        SetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_PIC_NUMBER),
            szBuffer);
    }
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnFirstRoom
//
/////////////////////////////////////////////////////
//
// Purpose:
//    event handler for the First Room checkbox
// Return value:
//    TRUE if the function handled the event; FALSE
//    if Windows needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnFirstRoom()
{
    // find out whether the checkbox is checked or unchecked
    // (note: it is an autocheckbox, so this is just finding
    // out what its current state is -- it doesn't need to be
    // set)
    BOOL bFirstRoomChecked = 
        IsCheckboxChecked(g_hwndMainDlg, IDC_CHECK_FIRST_ROOM);

    // enable or disable the First Room Controls button based on whether
    // this is the first room in the game or not
    EnableWindow(GetDlgItem(g_hwndMainDlg, IDC_BUTTON_FIRST_ROOM_CONTROLS),
        bFirstRoomChecked);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnDrawEgoInitially
//
/////////////////////////////////////////////////////
//
// Purpose:
//    event handler for the Draw ego initially checkbox
// Return value:
//    TRUE if the function handled the event; FALSE
//    if Windows needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnDrawEgoInitially()
{
    // nothing really needs to happen in response to this
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEditLogicNumber
//
/////////////////////////////////////////////////////
//
// Purpose:
//    event handler for the Logic number edit box
// Return value:
//    TRUE if the function handled the event; FALSE
//    if Windows needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEditLogicNumber(WPARAM wParam)
{
    // edit boxes can send notifications for
    // various reasons -- figure out which one
    // this is
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        // the text in the edit box is changing
        return OnEditLogicNumberENUpdate();
    }

    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnEditLogicNumberENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles EN_UPDATE notifications from the Logic
//    Number edit box
// Return value:
//    TRUE if event handled; FALSE if not
//
/////////////////////////////////////////////////////

BOOL OnEditLogicNumberENUpdate()
{
    // the only thing that needs to be done when the logic number changes
    // is to change the pic number, and then only if Use room_no for pic
    // is checked

    char szBuffer[MAX_DIGITS + 1];
    BOOL bPicNumSyncsWithLogicNum =
        IsCheckboxChecked(g_hwndMainDlg, IDC_CHECK_USE_ROOM_NUM_PIC);

    if (bPicNumSyncsWithLogicNum)
    {
        HWND hwndEdit = GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_NUMBER);
        GetWindowText(hwndEdit, szBuffer, MAX_DIGITS + 1);
        SetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_PIC_NUMBER),
                      szBuffer);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnAddPosControl
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Add button for positioning
//    controls
// Return value:
//    TRUE if event handled; FALSE if Windows should
//    handle it
//
/////////////////////////////////////////////////////

BOOL OnAddPosControl()
{
    EGOPOSITIONCONTROL epcNew;
    char szBuffer[100];
    HWND hwndLB;
    int nResponse;

    // get the window handle for the Ego Positioning list box
    hwndLB = GetDlgItem(g_hwndMainDlg, IDC_LIST_EGO_POS);

    // let the user fill out all the information
    if (PositionCtrlDlgDoModal(POSCTRLDLG_MODE_ADD, epcNew) == IDOK)
    {
        // if user clicked OK, then try to add the new positioning item

        // see if this new item is an abs. positioning item
        if (epcNew.nSrcRoom == ABSOLUTE_POSITIONING_SRC_ROOM)
        {
            // make sure there isn't already an absolute positioning item
            if (g_aepc.GetAbsolutePositioningItem() != NULL)
            {
                // if there is, see which one the user wants to keep (there
                // can be only one)
                nResponse = MessageBox(g_hwndMainDlg,
                                       "An absolute positioning item already "
                                       "exists for this logic and there can\n"
                                       "only be one per logic. Do you want "
                                       "to discard the old one and replace\n"
                                       "it with the new one?",
                                       "Question", MB_YESNO);

                if (nResponse == IDNO)
                {
                    // user said No

                    // the TRUE here simply indicates that the command
                    // to add a position control was handled; it has
                    // nothing to do with whether a control was added;
                    // in this case, the new control was dropped
                    return TRUE;
                }

                // user said Yes

                // delete the absolute positioning item that already exists
                SendMessage(hwndLB, LB_DELETESTRING,
                            static_cast<WPARAM>(g_aepc.GetElementCount()), 0);
            }

            g_aepc.SetAbsolutePositioningItem(&epcNew);

            sprintf(szBuffer, 
                    "Coming from any other room, position ego "
                    "at (%d, %d)", epcNew.nPosX, epcNew.nPosY);

            SendMessage(hwndLB, LB_INSERTSTRING,
                        static_cast<WPARAM>(g_aepc.GetElementCount()),
                        reinterpret_cast<LPARAM>(szBuffer));
        }
        else
        {
            // this is not an abs. positioning item, so just put it
            // in the list (not at the end, though, if there's an abs.
            // positioning item)
            g_aepc.Add(&epcNew);
        
            sprintf(szBuffer, "Coming from room %d, position ego at (%d, %d)",
                    epcNew.nSrcRoom, epcNew.nPosX, epcNew.nPosY);
        
            SendMessage(hwndLB, LB_INSERTSTRING, 
                        static_cast<WPARAM>(g_aepc.GetElementCount() - 1), 
                        reinterpret_cast<LPARAM>(szBuffer));
        }

        EnableClearPosControls();
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEditPosControl
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Edit button for positioning
//    controls
// Return value:
//    TRUE if event handled; FALSE if Windows should
//    handle it
//
/////////////////////////////////////////////////////

BOOL OnEditPosControl()
{
    HWND hwndLB;
    int nCurSel;
    char szBuffer[100];
    EGOPOSITIONCONTROL epcEdited;

    // get the list box window handle and the index of the item
    // that is to be edited
    hwndLB = GetDlgItem(g_hwndMainDlg, IDC_LIST_EGO_POS);
    nCurSel = SendMessage(hwndLB, LB_GETCURSEL, 0, 0);

    // make sure there is actually a selection before attempting to edit
    // the current selection
    if (nCurSel != LB_ERR)
    {
        // the absolute positioning item is a special case...test for it
        if ((g_aepc.GetAbsolutePositioningItem() != NULL) &&
            nCurSel == g_aepc.GetElementCount())
        {
            // get a copy of the absolute positioning item
            memcpy(&epcEdited, g_aepc.GetAbsolutePositioningItem(),
                   sizeof(EGOPOSITIONCONTROL));

            // let the user edit the item
            if (PositionCtrlDlgDoModal(POSCTRLDLG_MODE_EDIT, epcEdited) == 
                IDOK)
            {
                // if the user's accepted the changes they made, then
                // set the absolute positioning item and replace it in
                // the list box with the edited version
                if (epcEdited.nSrcRoom == ABSOLUTE_POSITIONING_SRC_ROOM)
                {
                    // if the item is still an absolute positioning item,
                    // then just set it as the absolute positioning item
                    g_aepc.SetAbsolutePositioningItem(&epcEdited);
                    sprintf(szBuffer, 
                            "Coming from any other room, position ego "
                            "at (%d, %d)", epcEdited.nPosX, epcEdited.nPosY);
                }
                else
                {
                    // if the item is no longer an abs. positioning item,
                    // then remove the abs. positioning item from g_aepc
                    // and put the item in the array
                    g_aepc.SetAbsolutePositioningItem(NULL);
                    g_aepc.Add(&epcEdited);
                    sprintf(szBuffer,
                            "Coming from room %d, position ego at (%d, %d)",
                            epcEdited.nSrcRoom, epcEdited.nPosX,
                            epcEdited.nPosY);
                }

                SendMessage(hwndLB, LB_DELETESTRING, 
                            static_cast<WPARAM>(nCurSel), 0);
                SendMessage(hwndLB, LB_INSERTSTRING,
                            static_cast<WPARAM>(nCurSel), 
                            reinterpret_cast<LPARAM>(szBuffer));
                SendMessage(hwndLB, LB_SETCURSEL,
                            static_cast<WPARAM>(nCurSel), 0);
            }
        }
        else
        {
            // get a copy of the selected positioning item
            memcpy(&epcEdited, &g_aepc[nCurSel], sizeof(EGOPOSITIONCONTROL));
            
            // let the user edit the item
            if (PositionCtrlDlgDoModal(POSCTRLDLG_MODE_EDIT, epcEdited) == 
                IDOK)
            {
                // editing a normal item can be slightly problematic if
                // the user changed it to an absolute positioning item;
                // first check to see if they did
                if (epcEdited.nSrcRoom != ABSOLUTE_POSITIONING_SRC_ROOM)
                {
                    // if they didn't, no problem...just put the edited
                    // element in the array where the old one way and
                    // change the list box entry
                    g_aepc.SetAt(&epcEdited, nCurSel);
                    sprintf(szBuffer, 
                            "Coming from room %d, position ego at (%d, %d)",
                            epcEdited.nSrcRoom, epcEdited.nPosX,
                            epcEdited.nPosY);
                    SendMessage(hwndLB, LB_DELETESTRING, 
                                static_cast<WPARAM>(nCurSel), 0);
                    SendMessage(hwndLB, LB_INSERTSTRING,
                                static_cast<WPARAM>(nCurSel), 
                                reinterpret_cast<LPARAM>(szBuffer));
                    SendMessage(hwndLB, LB_SETCURSEL,
                                static_cast<WPARAM>(nCurSel), 0);

                }
                else
                {
                    // if they did change the item to an abs. pos. item,
                    // then make sure there isn't already an abs. pos.
                    // item

                    if (g_aepc.GetAbsolutePositioningItem() != NULL)
                    {                        
                        // if there is, then see if the user wants to
                        // replace it
                        int nResponse = MessageBox(g_hwndMainDlg,
                                       "An absolute positioning item already "
                                       "exists for this logic and there can\n"
                                       "only be one per logic. Do you want "
                                       "to discard the old one and replace\n"
                                       "it with the new one?",
                                       "Question", MB_YESNO);

                        if (nResponse == IDYES)
                        {
                            SendMessage(hwndLB, LB_DELETESTRING, 
                                        static_cast<WPARAM>(
                                            g_aepc.GetElementCount()), 0);
                        }
                        else
                        {
                            return TRUE;
                        }
                    }

                    // remove the edited item from the array and set it as
                    // the absolute positioning item
                    g_aepc.RemoveAt(nCurSel, 1);
                    g_aepc.SetAbsolutePositioningItem(&epcEdited);

                    // remove the old entry from the list box and put the
                    // abs. pos. entry at the bottom
                    sprintf(szBuffer, 
                            "Coming from any other room, position ego "
                            "at (%d, %d)", epcEdited.nPosX, epcEdited.nPosY);
                    SendMessage(hwndLB, LB_DELETESTRING, 
                                static_cast<WPARAM>(nCurSel), 0);

                    nCurSel = g_aepc.GetElementCount();

                    SendMessage(hwndLB, LB_INSERTSTRING,
                                static_cast<WPARAM>(nCurSel), 
                                reinterpret_cast<LPARAM>(szBuffer));
                    SendMessage(hwndLB, LB_SETCURSEL,
                                static_cast<WPARAM>(nCurSel), 0);
                }            
            }
        }
    }
    else
    {
        // this shouldn't actually ever be reached, since the Edit button
        // should always be disabled when it's not applicable, but in case
        // it somehow is enabled when inapplicable, this will let the user
        // know why nothing happened when they clicked the Edit button
        MessageBox(g_hwndMainDlg, 
                   "Please select an item before clicking Edit.",
                   "Select An Item", MB_OK);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnDelPosControl
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Delete button for positioning
//    controls
// Return value:
//    TRUE if event handled; FALSE if Windows should
//    handle it
//
/////////////////////////////////////////////////////

BOOL OnDelPosControl()
{
    HWND hwndLB;
    int nCurSel;

    // get the list box window handle and the index of the item
    // that is to be edited
    hwndLB = GetDlgItem(g_hwndMainDlg, IDC_LIST_EGO_POS);
    nCurSel = SendMessage(hwndLB, LB_GETCURSEL, 0, 0);
    
    // make sure there is actually a selection before attempting to delete
    // the current selection
    if (nCurSel != LB_ERR)
    {
        char szBuffer[100];
        EGOPOSITIONCONTROL epc;

        // check if it is the absolute positioning item the user is trying 
        // to delete
        if (g_aepc.GetAbsolutePositioningItem() != NULL &&
            nCurSel == g_aepc.GetElementCount())
        {
            // confirm the deletion

            epc = *(g_aepc.GetAbsolutePositioningItem());
            sprintf(szBuffer, "Are you sure you want to delete this item:\n"
                              "Coming from any other room\n"
                              "Position ego at (%d, %d)",
                    epc.nPosX, epc.nPosY);

            if (MessageBox(g_hwndMainDlg, szBuffer, "Confirm Delete",
                           MB_YESNO | MB_ICONEXCLAMATION) == IDYES)
            {
                // if the user really wants to delete the item, then
                // do as they say
                SendMessage(hwndLB, LB_DELETESTRING,
                            static_cast<WPARAM>(g_aepc.GetElementCount()),
                            0);
                g_aepc.SetAbsolutePositioningItem(NULL);

                if (g_aepc.GetElementCount() == 0)
                {
                    EnableEditPosControl(FALSE);
                    EnableDelPosControl(FALSE);
                    EnableClearPosControls(FALSE);
                }

            }

        }
        else
        {
            // confirm the deletion

            epc = g_aepc[nCurSel];

            assert(epc.nSrcRoom >= 0);

            sprintf(szBuffer, "Are you sure you want to delete this item:\n"
                              "Coming from room %d\n"
                              "Position ego at (%d, %d)",
                    epc.nSrcRoom, epc.nPosX, epc.nPosY);

            if (MessageBox(g_hwndMainDlg, szBuffer, "Confirm Delete",
                           MB_YESNO | MB_ICONEXCLAMATION) == IDYES)
            {
                // if the user really wants to delete the item, then
                // do as they say

                SendMessage(hwndLB, LB_DELETESTRING,
                            static_cast<WPARAM>(nCurSel), 0);
                g_aepc.RemoveAt(nCurSel);

                if (g_aepc.GetElementCount() == 0)
                {
                    EnableEditPosControl(FALSE);
                    EnableDelPosControl(FALSE);
                    EnableClearPosControls(FALSE);
                }
            }
        }
    }
    else
    {
        // this shouldn't actually ever be reached, since the Delete button
        // should always be disabled when it's not applicable, but in case
        // it somehow is enabled when inapplicable, this will let the user
        // know why nothing happened when they clicked the Delete button
        MessageBox(g_hwndMainDlg, 
                   "Please select an item before clicking Delete.",
                   "Select An Item", MB_OK);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnClearPosControls
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Delete All button for
//    positioning controls
// Return value:
//    TRUE if event handled; FALSE if Windows should
//    handle it
//
/////////////////////////////////////////////////////

BOOL OnClearPosControls()
{
    // the user asked to delete ALL of the positioning controls...definitely
    // confirm before doing this

    if (MessageBox(g_hwndMainDlg, 
                   "This command will delete ALL the ego positioning "
                   "information that you have entered. Are you sure?",
                   "Confirm Delete", MB_YESNO | MB_ICONEXCLAMATION) == IDYES)
    {
        SendMessage(GetDlgItem(g_hwndMainDlg, IDC_LIST_EGO_POS),
                    LB_RESETCONTENT, 0, 0);
        g_aepc.RemoveAll();
        g_aepc.SetAbsolutePositioningItem(NULL);
        EnableEditPosControl(FALSE);
        EnableDelPosControl(FALSE);
        EnableClearPosControls(FALSE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnListEgoPos
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events related to the ego positioning
//    list box
// Return value:
//    TRUE if event handled; FALSE if Windows should
//    handle it
//
/////////////////////////////////////////////////////

BOOL OnListEgoPos(WPARAM wParam)
{
    // figure out what type of notification this is

    switch(HIWORD(wParam))
    {
    case LBN_SELCHANGE:
        // the user changed the selection
        return OnListEgoPosLBNSelChange();
    case LBN_DBLCLK:
        // the user double-clicked an item in the list
        return OnListEgoPosLBNDblClk();
    }

    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnListEgoPosLBNSelChange
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the LBN_SELCHANGE notification for the
//    ego positioning list box
// Return value:
//    TRUE if event handled; FALSE if Windows should
//    handle it
//
/////////////////////////////////////////////////////

BOOL OnListEgoPosLBNSelChange()
{
    // enable the Edit and Delete buttons if there is
    // a selection, disable them otherwise

    int nCurSel;
    nCurSel = SendMessage(GetDlgItem(g_hwndMainDlg, IDC_LIST_EGO_POS),
                          LB_GETCURSEL, 0, 0);
    EnableEditPosControl(nCurSel != LB_ERR);
    EnableDelPosControl(nCurSel != LB_ERR);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnListEgoPosLBNDblClk
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the LBN_DBLCLICK notification for the
//    ego positioning list box
// Return value:
//    TRUE if event handled; FALSE if Windows should
//    handle it
//
/////////////////////////////////////////////////////

BOOL OnListEgoPosLBNDblClk()
{
    // a double-click should be considered the same as
    // clicking the Edit button
    OnEditPosControl();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// EnableEditPosControl
//
/////////////////////////////////////////////////////
//
// Purpose:
//    enables or disables the Edit button for the
//    ego positioning controls
// Parameter bEnable:
//    specifies whether to enable or disable the button
//
/////////////////////////////////////////////////////

void EnableEditPosControl(BOOL bEnable)
{
    HWND hwndButton = GetDlgItem(g_hwndMainDlg, IDC_BUTTON_EDIT_POS_CONTROL);

    // NOTE: if the Edit button currently has the input focus when it
    //       is disabled, then the user loses the ability to navigate
    //       the BLG with the keyboard until they click the mouse or
    //       task switch!! -- therefore, make sure that doesn't
    //       happen by setting the input focus to the OK button when
    //       the Edit button is being disabled while it has the input
    //       focus
    if (GetFocus() == hwndButton)
    {
        SetFocus(GetDlgItem(g_hwndMainDlg, IDOK));
    }

    EnableWindow(hwndButton, bEnable);
}

/////////////////////////////////////////////////////
//
// EnableDelPosControl
//
/////////////////////////////////////////////////////
//
// Purpose:
//    enables or disables the Delete button for the
//    ego positioning controls
// Parameter bEnable:
//    specifies whether to enable or disable the button
//
/////////////////////////////////////////////////////

void EnableDelPosControl(BOOL bEnable)
{
    HWND hwndButton = GetDlgItem(g_hwndMainDlg, IDC_BUTTON_DEL_POS_CONTROL);

    // NOTE: if the Delete button currently has the input focus when it
    //       is disabled, then the user loses the ability to navigate
    //       the BLG with the keyboard until they click the mouse or
    //       task switch!! -- therefore, make sure that doesn't
    //       happen by setting the input focus to the OK button when
    //       the Delete button is being disabled while it has the input
    //       focus
    if (GetFocus() == hwndButton)
    {
        SetFocus(GetDlgItem(g_hwndMainDlg, IDOK));
    }

    EnableWindow(hwndButton, bEnable);

}

/////////////////////////////////////////////////////
//
// EnableClearPosControls
//
/////////////////////////////////////////////////////
//
// Purpose:
//    enables or disables the Delete All button for the
//    ego positioning controls
// Parameter bEnable:
//    specifies whether to enable or disable the button
//
/////////////////////////////////////////////////////

void EnableClearPosControls(BOOL bEnable)
{
    HWND hwndButton = GetDlgItem(g_hwndMainDlg, IDC_BUTTON_CLEAR_POS_CONTROLS);

    // NOTE: if the Delete All button currently has the input focus when it
    //       is disabled, then the user loses the ability to navigate
    //       the BLG with the keyboard until they click the mouse or
    //       task switch!! -- therefore, make sure that doesn't
    //       happen by setting the input focus to the OK button when
    //       the Delete All button is being disabled while it has the input
    //       focus
    if (GetFocus() == hwndButton)
    {
        SetFocus(GetDlgItem(g_hwndMainDlg, IDOK));
    }

    EnableWindow(hwndButton, bEnable);
}

/////////////////////////////////////////////////////
//
// OnEdgeControlAdvanced
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Advanced button for the
//    edge code controls
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnEdgeControlAdvanced()
{
    HWND hwndLeftEdgeGoto;
    HWND hwndRightEdgeGoto;
    HWND hwndBottomEdgeGoto;
    HWND hwndHorizonEdgeGoto;

    hwndLeftEdgeGoto    = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_LEFT_EDGE_GOTO);
    hwndRightEdgeGoto   = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_RIGHT_EDGE_GOTO);
    hwndBottomEdgeGoto  = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_BOTTOM_EDGE_GOTO);
    hwndHorizonEdgeGoto = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_HORIZON_EDGE_GOTO);

    GetEdgeCodeGotoValues();

    // bring up the Edge Code Advanced dialog
    if (EdgeCodeAdvancedDlgDoModal(g_ecinfo) == IDOK)
    {
        // if the user accepts the changes they made in the
        // EC Advanced dialog, some stuff needs to be checked

        // if the user checked to include an empty left edge
        // block, then the Goto Room box for the left needs to
        // be disabled; otherwise, enable it
        if (g_ecinfo.m_bEmptyLeft)
        {
            SetWindowText(hwndLeftEdgeGoto, "");
            EnableWindow(hwndLeftEdgeGoto, FALSE);
        }
        else
        {
            EnableWindow(hwndLeftEdgeGoto, TRUE);
        }

        // if the user checked to include an empty right edge
        // block, then the Goto Room box for the right needs to
        // be disabled; otherwise, enable it
        if (g_ecinfo.m_bEmptyRight)
        {
            SetWindowText(hwndRightEdgeGoto, "");
            EnableWindow(hwndRightEdgeGoto, FALSE);
        }
        else
        {
            EnableWindow(hwndRightEdgeGoto, TRUE);
        }

        // if the user checked to include an empty bottom edge
        // block, then the Goto Room box for the bottom needs to
        // be disabled; otherwise, enable it
        if (g_ecinfo.m_bEmptyBottom)
        {
            SetWindowText(hwndBottomEdgeGoto, "");
            EnableWindow(hwndBottomEdgeGoto, FALSE);
        }
        else
        {
            EnableWindow(hwndBottomEdgeGoto, TRUE);
        }

        // if the user checked to include an empty horizon edge
        // block, then the Goto Room box for the horizon needs to
        // be disabled; otherwise, enable it

        if (g_ecinfo.m_bEmptyHorizon)
        {
            SetWindowText(hwndHorizonEdgeGoto, "");
            EnableWindow(hwndHorizonEdgeGoto, FALSE);
        }
        else
        {
            EnableWindow(hwndHorizonEdgeGoto, TRUE);
        }

    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnButtonOptions
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Options button
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnButtonOptions()
{
    // nothing special to do here, just display the
    // Options dialog
    OptionsDlgDoModal(g_options);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnButtonBrowseForGamePath
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Browse button for the
//    game path
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnButtonBrowseForGamePath()
{
    // I'm not going to comment this code -- it's just standard
    // browse for folder stuff...look it up in the API docs
    // under SHBrowseForFolder

    LPITEMIDLIST pidl;
    BROWSEINFO bi;
    char szBuffer[_MAX_PATH + 1];

    bi.hwndOwner = g_hwndMainDlg;
    bi.pidlRoot = NULL;
    bi.pszDisplayName = szBuffer;
    bi.lpszTitle = "Choose your game's folder:";
    bi.ulFlags = BIF_RETURNONLYFSDIRS;
    bi.lpfn = NULL;

    pidl = SHBrowseForFolder(&bi);

    if (pidl)
    {
        if (SHGetPathFromIDList(pidl, szBuffer))
        {
            int nStrLen = strlen(szBuffer);
            if (szBuffer[nStrLen - 1] != '\\')
            {
                strcat(szBuffer, "\\");
            }

            SetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_GAME_PATH),
                          szBuffer);
        }
        else
        {
            MessageBox(g_hwndMainDlg,
                       "an error occurred setting the game path;\n",
                       "error", MB_OK);
            return TRUE;
        }
    }
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnButtonEnterLook
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Entry and Looking button
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnButtonEnterLook()
{
    // just display the Entry and Looking dialog
    EntryLookDlgDoModal(g_elinfo);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnButtonFirstRoomControls
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the First Room Controls button
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnButtonFirstRoomControls()
{
    // just display the First Room Controls dialog
    FirstRoomCtrlDlgDoModal(g_frctrl);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnMainDlgOK
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the OK button
// Return value:
//    TRUE if the event is handled; FALSE if Windows
//    needs to handle it
//
/////////////////////////////////////////////////////

BOOL OnMainDlgOK()
{
    char szBuffer[_MAX_PATH + 2];
    char szBuffer2[100];

    // retrieve and validate the logic number
    GetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_NUMBER),
                  szBuffer, MAX_DIGITS + 1);
    g_mlopt.m_nLogicNumber = atoi(szBuffer);

    if (g_mlopt.m_nLogicNumber < 0 ||
        g_mlopt.m_nLogicNumber > AGI_MAX_VALUE)
    {
        MessageBox(g_hwndMainDlg,
                   "The logic number must be between 0 and 255.",
                   "Error", MB_OK);
        SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_NUMBER));
        return TRUE;
    }

    // determine whether the reserved var room_no is being used to
    // decide the picture number
    g_mlopt.m_bUseRoomNumForPic = 
        IsCheckboxChecked(g_hwndMainDlg, IDC_CHECK_USE_ROOM_NUM_PIC);

    // if not, then get the picture number and validate it
    if (!g_mlopt.m_bUseRoomNumForPic)
    {
        GetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_PIC_NUMBER),
                      szBuffer, MAX_DIGITS + 1);
        g_mlopt.m_nPicNumber = atoi(szBuffer);

        if (g_mlopt.m_nPicNumber < 0 ||
            g_mlopt.m_nPicNumber > AGI_MAX_VALUE)
        {
            MessageBox(g_hwndMainDlg,
                       "The picture number must be between 0 and 255.",
                       "Error", MB_OK);
            SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_PIC_NUMBER));
            return TRUE;
        }
    }

    GetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_HORIZON),
                  szBuffer, MAX_DIGITS + 1);
    g_mlopt.m_nHorizon = atoi(szBuffer);

    if (g_mlopt.m_nHorizon < 0 ||
        g_mlopt.m_nHorizon > MAX_HORIZON_VALUE)
    {
        MessageBox(g_hwndMainDlg,
                   "The horizon value must be between 0 and 166 (inclusive).",
                   "Error", MB_OK);
        SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_HORIZON));
        return TRUE;
    }

    g_mlopt.m_bDrawEgoInitially =
        IsCheckboxChecked(g_hwndMainDlg, IDC_CHECK_DRAW_EGO_INITIALLY);
    g_mlopt.m_bFirstRoom =
        IsCheckboxChecked(g_hwndMainDlg, IDC_CHECK_FIRST_ROOM);

    // get the logic title and check for escaped quotes...they aren't
    // necessary in the logic title
    GetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_TITLE),
                  g_mlopt.m_szLogicTitle, MAX_LOGIC_TITLE_LEN + 1);

    if (CountEscapedQuotes(g_mlopt.m_szLogicTitle) > 0)
    {
        int nMsgBoxResponse;

        // if there were escaped quotes in the logic title, prompt
        // the user for what to do about it
        nMsgBoxResponse = EscapedQuoteMessageBox(g_hwndMainDlg,
                                                 "Logic title");

        switch(nMsgBoxResponse)
        {
        case EQMB_REMOVE_BACKSLASH:
            // the user chose to remove the backslash and include
            // the quotes as normal quotes for all escaped quotes
            ConvertEscapedQuotesToUnescapedQuotes(g_mlopt.m_szLogicTitle);
            SetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_TITLE),
                          g_mlopt.m_szLogicTitle);
            break;

        case EQMB_INCLUDE_BACKSLASH:
            // the user chose to leave the escaped quotes as escaped
            // quotes
            break;

        case EQMB_USER_EDIT:
            // the user chose to edit the logic title
            SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_TITLE));
            return TRUE;

        default:
            // should never get here
            assert(FALSE);
            break;
        }
    }

    // set the modified string immediately so that the user doesn't have
    // to be bothered with it again
    SetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_LOGIC_TITLE),
                  g_mlopt.m_szLogicTitle);

    // retrieve and validate the edge code goto values
    GetEdgeCodeGotoValues();

    // check the left goto values
    if ((g_ecinfo.m_nLeftGotoRoom < 0 ||
         g_ecinfo.m_nLeftGotoRoom > AGI_MAX_VALUE) &&
         g_ecinfo.m_nLeftGotoRoom != NO_GOTO_ROOM)
    {
        MessageBox(g_hwndMainDlg,
                   "The left edge goto room must be between 0 and 255.",
                   "Error", MB_OK);
        SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_LEFT_EDGE_GOTO));
        return TRUE;
    }

    // check the right goto values
    if ((g_ecinfo.m_nRightGotoRoom < 0 ||
         g_ecinfo.m_nRightGotoRoom > AGI_MAX_VALUE) &&
         g_ecinfo.m_nRightGotoRoom != NO_GOTO_ROOM)
    {
        MessageBox(g_hwndMainDlg,
                   "The right edge goto room must be between 0 and 255.",
                   "Error", MB_OK);
        SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_RIGHT_EDGE_GOTO));
        return TRUE;
    }

    // check the bottom goto values
    if ((g_ecinfo.m_nBottomGotoRoom < 0 ||
         g_ecinfo.m_nBottomGotoRoom > AGI_MAX_VALUE) &&
         g_ecinfo.m_nBottomGotoRoom != NO_GOTO_ROOM)
    {
        MessageBox(g_hwndMainDlg,
                   "The bottom edge goto room must be between 0 and 255.",
                   "Error", MB_OK);
        SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_BOTTOM_EDGE_GOTO));
        return TRUE;
    }

    // check the horizon goto values
    if ((g_ecinfo.m_nHorizonGotoRoom < 0 ||
         g_ecinfo.m_nHorizonGotoRoom > AGI_MAX_VALUE) &&
         g_ecinfo.m_nHorizonGotoRoom != NO_GOTO_ROOM)
    {
        MessageBox(g_hwndMainDlg,
                   "The horizon edge goto room must be between 0 and 255.",
                   "Error", MB_OK);
        SetFocus(GetDlgItem(g_hwndMainDlg, IDC_EDIT_HORIZON_EDGE_GOTO));
        return TRUE;
    }

    if (g_options.m_bGenerateCompiledCode ||
        (g_options.m_bGenerateSource && g_options.m_bWriteToFile))
    {
        // if there is a file to be generated, then
        // make sure the BLG knows WHERE to generate it

        GetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_GAME_PATH),
                      szBuffer, _MAX_PATH + 1);

        if (strlen(szBuffer) == 0)
        {
            OnButtonBrowseForGamePath();

            GetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_GAME_PATH),
                          szBuffer, _MAX_PATH + 1);

            if (strlen(szBuffer) == 0)
            {
                MessageBox(g_hwndMainDlg,
                           "The game path must be set in order to\n"
                           "generate files.", "Game Path Not Set",
                           MB_OK);

                return TRUE;
            }
        }

        // if we've gotten to this point, there's a directory
        // name in the Game Path box
        WIN32_FIND_DATA fd;
        HANDLE handle;

        if (strlen(szBuffer) >= (_MAX_PATH - strlen("src\\logic255.txt")))
        {
            MessageBox(g_hwndMainDlg,
                       "The selected game path is too long",
                       "Game Path Too Long",
                       MB_OK);

            return TRUE;
        }

        // TODO:
        //    the current implementation of testing the current directory
        //    to make sure it exists is inadequate; it works if the game
        //    folder's parent folder still exists, but if the parent folder
        //    does not exist, then the game folder cannot be created; an
        //    adequate implementation will need to parse the game path
        //    and create directories down the folder hierarchy

        // make sure the game path actually exists (the browse for
        // folder only allows folders that exist to be chosen, I think,
        // but the game path is also saved between runs of the BLG, and
        // it could be deleted between runs of the program)
        strcat(szBuffer, "*");
        handle = FindFirstFile(szBuffer, &fd);

        // take off the asterisk and the trailing backslash
        szBuffer[strlen(szBuffer) - 2] = '\0';

        if (handle == INVALID_HANDLE_VALUE)
        {
            if (MessageBox(g_hwndMainDlg,
                           "The specified game folder does not exist.\n\n"
                           "Do you want to create it now?", "Create Folder",
                           MB_YESNO) == IDYES)
            {
                if (!CreateDirectory(szBuffer, NULL))
                {
                    MessageBox(g_hwndMainDlg,
                               "Unable to create folder. Cannot generate "
                               "files.", "Error", MB_OK);
                    return TRUE;
                }

            }
            else
            {
                MessageBox(g_hwndMainDlg,
                           "Could not generate files.\n\nSpecified game "
                           "folder does not exist.", "Error", MB_OK);
                return TRUE;
            }

        }

        // put the backslash back on the end of the path name
        strcat(szBuffer, "\\");

        if (g_options.m_bGenerateSource && g_options.m_bWriteToFile)
        {
            strcat(szBuffer, "src\\*");
            handle = FindFirstFile(szBuffer, &fd);

            // take off the asterisk and the trailing backslash
            szBuffer[strlen(szBuffer) - 2] = '\0';

            if (handle == INVALID_HANDLE_VALUE)
            {
                if (MessageBox(g_hwndMainDlg,
                               "The game folder you have selected has no\n"
                               "src subfolder.\n\nDo you want to create "
                               "it now?", "Create Folder", MB_YESNO) == IDYES)
                {
                    if (!CreateDirectory(szBuffer, NULL))
                    {
                        MessageBox(g_hwndMainDlg,
                                   "Unable to create folder. Cannot generate "
                                   "files.", "Error", MB_OK);
                        return TRUE;
                    }
                }
                else
                {
                    MessageBox(g_hwndMainDlg,
                               "No src folder. Cannot generate files.",
                               "Error", MB_OK);
                    return TRUE;
                }
            }

            // finally, make sure the source file doesn't already exist
            strcat(szBuffer, "\\");
            sprintf(szBuffer2, "logic%d.txt", g_mlopt.m_nLogicNumber);
            strcat(szBuffer, szBuffer2);

            handle = FindFirstFile(szBuffer, &fd);

            if (handle != INVALID_HANDLE_VALUE)
            {
                if (MessageBox(g_hwndMainDlg,
                               "A source file with the specified logic number "
                               "already exists in the specified game path. Do "
                               "you want to overwrite?\n\nIf you say no, you "
                               "must change the logic number or game path to "
                               "generate a source file.",
                               "File Exists", MB_YESNO) == IDNO)
                {
                    return TRUE;
                }
            }

            // remove src, backslash, and file name from szBuffer
            szBuffer[strlen(szBuffer) - strlen("src\\") - strlen(szBuffer2)] = '\0';
        }

        if (g_options.m_bGenerateCompiledCode)
        {
            // see if the compiled file exists
            //strcat(szBuffer, "\\");
            sprintf(szBuffer2, "logic.%0.3d", g_mlopt.m_nLogicNumber);
            strcat(szBuffer, szBuffer2);

            handle = FindFirstFile(szBuffer, &fd);
            if (handle != INVALID_HANDLE_VALUE)
            {
                if (MessageBox(g_hwndMainDlg,
                               "A compiled file with the specified logic "
                               "number already exists in the specified game "
                               "path. Do you want to overwrite?\n\nIf you "
                               "say no, you must change the logic number or "
                               "game path to generate a compiled file.",
                               "File Exists", MB_YESNO) == IDNO)
                {
                    return TRUE;
                }
            }

            // remove file name from path
            szBuffer[strlen(szBuffer) - strlen(szBuffer2)] = '\0';
        }
    }

    // do the file generation
    if (g_options.m_bGenerateCompiledCode)
    {
        GenerateCompiledLogic(szBuffer);
    }

    if (g_options.m_bGenerateSource)
    {
        GenerateSourceFile(szBuffer);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// GetEdgeCodeGotoValues
//
/////////////////////////////////////////////////////
//
// Purpose:
//    retrieves the edge code goto values from the
//    four edge code edit boxes on the main dialog
//    and assigns their numeric values into the 
//    g_ecinfo object
//
/////////////////////////////////////////////////////

void GetEdgeCodeGotoValues()
{
    // pretty self-explanatory stuff here...retrieve the text
    // from the controls -- if there is text in the edit box,
    // convert it to an int and assign it to the appropriate
    // g_ecinfo member...if there is no text in the edir box,
    // assign NO_GOTO_ROOM to the appropriate g_ecinfo member

    char szBuffer[MAX_DIGITS + 1];
    HWND hwndLeftEdgeGoto;
    HWND hwndRightEdgeGoto;
    HWND hwndBottomEdgeGoto;
    HWND hwndHorizonEdgeGoto;

    hwndLeftEdgeGoto    = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_LEFT_EDGE_GOTO);
    hwndRightEdgeGoto   = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_RIGHT_EDGE_GOTO);
    hwndBottomEdgeGoto  = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_BOTTOM_EDGE_GOTO);
    hwndHorizonEdgeGoto = GetDlgItem(g_hwndMainDlg, 
                                     IDC_EDIT_HORIZON_EDGE_GOTO);

    GetWindowText(hwndLeftEdgeGoto, szBuffer, MAX_DIGITS + 1);

    if (strlen(szBuffer) != 0)
    {
        g_ecinfo.m_nLeftGotoRoom = atoi(szBuffer);
    }
    else
    {
        g_ecinfo.m_nLeftGotoRoom = NO_GOTO_ROOM;
    }

    GetWindowText(hwndRightEdgeGoto, szBuffer, MAX_DIGITS + 1);

    if (strlen(szBuffer) != 0)
    {
        g_ecinfo.m_nRightGotoRoom = atoi(szBuffer);
    }
    else
    {
        g_ecinfo.m_nRightGotoRoom = NO_GOTO_ROOM;
    }

    GetWindowText(hwndBottomEdgeGoto, szBuffer, MAX_DIGITS + 1);

    if (strlen(szBuffer) != 0)
    {
        g_ecinfo.m_nBottomGotoRoom = atoi(szBuffer);
    }
    else
    {
        g_ecinfo.m_nBottomGotoRoom = NO_GOTO_ROOM;
    }

    GetWindowText(hwndHorizonEdgeGoto, szBuffer, MAX_DIGITS + 1);

    if (strlen(szBuffer) != 0)
    {
        g_ecinfo.m_nHorizonGotoRoom = atoi(szBuffer);
    }
    else
    {
        g_ecinfo.m_nHorizonGotoRoom = NO_GOTO_ROOM;
    }
}

/////////////////////////////////////////////////////
//
// OnMainDlgCancel
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles clicks of the Exit button for the main
//    BLG dialog
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    should handle it
//
/////////////////////////////////////////////////////

BOOL OnMainDlgCancel()
{
    // this event handler saves persistent data and closes
    // the Base Logic Generator

    char szBuffer[_MAX_PATH + 1];
    SaveBLGStateInfo(BLG_STATE_DEFINE_NAME_LIST, &g_deflist);
    SaveBLGStateInfo(BLG_STATE_OPTIONS, &g_options);

    SaveBLGStateInfo(BLG_STATE_LOOK_WORD_GROUP, &g_elinfo.m_nLookWordGroup);

    GetWindowText(GetDlgItem(g_hwndMainDlg, IDC_EDIT_GAME_PATH),
                  szBuffer, _MAX_PATH + 1);
    SaveBLGStateInfo(BLG_STATE_LAST_GAME_PATH, szBuffer);

    EndDialog(g_hwndMainDlg, IDCANCEL);

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnMainDlgSysAbout
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles selection of the about box item from the
//    system menu
// Return value:
//    TRUE if the event was handled; FALSE if Windows
//    should handle it
//
/////////////////////////////////////////////////////

BOOL OnMainDlgSysAbout()
{
    AboutBoxDoModal();
    return TRUE;
}