/////////////////////////////////////////////////////
//
// EgoPositionControlArray.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    implementation of the CEgoPositionControlArray
//    class
//
// Portability information:
//    the code in this file should be portable without
//    modification; if it is not, feel free to modify
//    it so that it is portable; do not add non-portable
//    code to this file
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cstdlib>
#include <cassert>
#include "BLGExceptions.h"
#include "EgoPositionControlArray.h"

/////////////////////////////////////////////////////
//
// constructors
//
/////////////////////////////////////////////////////
//
// Purpose:
//    constructs a new CEgoPositionControlArray
//    object
// Parameter nInitialSize:
//    specifies how many ego position control
//    items the array can hold initially; defaults
//    to 10
// Parameter nGrowBy:
//    when the capacity of the array increases,
//    it is increased by nGrowBy elements; if
//    this parameter is < 1, then the array
//    grows by its initial size on each capacity
//    increase; defaults to growing by the initial
//    size
//
/////////////////////////////////////////////////////

CEgoPositionControlArray::CEgoPositionControlArray(int nInitialSize /* = 10 */,
                                                   int nGrowBy /* = -1 */)
{
    assert(nInitialSize > 0);

    // m_nArraySize is the 1-based count of the number of elements that
    // can be stored in the array before a resize (i.e., current capacity)
    m_nArraySize = nInitialSize;

    // obviously, the array needs to grow by a positive number of elements
    // when it is reallocated to fit more items; if the value passed in for
    // nGrowBy <= 0 then just use the array's initial size as the grow size
    if (nGrowBy <= 0)
    {
        m_nGrowBy = m_nArraySize;
    }
    else
    {
        m_nGrowBy = nGrowBy;
    }

    // make the array
    m_pepcTheArray = reinterpret_cast<PEGOPOSITIONCONTROL>(
                            malloc(m_nArraySize * sizeof(EGOPOSITIONCONTROL)));

    if (!m_pepcTheArray)
    {
        throw CMallocFailedException();
    }

    // m_nLastElement is the 0-based count of the actual array elements
    m_nLastElement = -1;

    // the absolute positioning item is a special item and is handled differently
    // from the other items -- it is not actually stored in the array but
    // in a separate EGOPOSITIONCONTROL; if it is not used, then the values
    // NO_POS_X and NO_POX_Y should be assigned to the struct's nPosX and
    // nPosY members, respectively
    m_epcAbsolutePos.nSrcRoom = ABSOLUTE_POSITIONING_SRC_ROOM;
    m_epcAbsolutePos.nPosX = NO_POS_X;
    m_epcAbsolutePos.nPosY = NO_POS_Y;
}

/////////////////////////////////////////////////////
//
// destructor
//
/////////////////////////////////////////////////////
//
// Purpose:
//    performs clean-up work when this 
//    CEgoPositionControlArray object is destroyed
//
/////////////////////////////////////////////////////

CEgoPositionControlArray::~CEgoPositionControlArray()
{
    if (m_pepcTheArray)
    {
        free(m_pepcTheArray);
    }
}

/////////////////////////////////////////////////////
//
// Add
//
/////////////////////////////////////////////////////
//
// Purpose:
//    adds ego positioning data to the end of the
//    array
// Parameter pepcNewElement:
//    pointer to a EGOPOSITIONCONTROL structure
//    containing the data to add to the array; the
//    structure itself is not added to the array;
//    its data is simply copied into the array's
//    private data array
//
/////////////////////////////////////////////////////

void CEgoPositionControlArray::Add(const PEGOPOSITIONCONTROL pepcNewElement)
{
    // don't try to add a NULL element to the array
    assert(pepcNewElement != NULL);

    // increment the record of the last element's index in a temporary variable
    // first, in case this operation for some reason fails
    int nLastElement = m_nLastElement;
    nLastElement++;

    // if the last element (a 0-based index) is equal to the array size (a 1-based
    // count), then the element to be added is going to be past the end of the
    // current array, so the array needs to grow
    if (nLastElement == m_nArraySize)
    {
        try
        {
            GrowArray();
        }
        catch(CMallocFailedException)
        {
            throw;
        }
    }

    // copy the EGOPOSITIONCONTROL's info
    memcpy(m_pepcTheArray + nLastElement, pepcNewElement, 
        sizeof(EGOPOSITIONCONTROL));

    // make the record of the last element in the array permanent
    m_nLastElement = nLastElement;
}

/////////////////////////////////////////////////////
//
// InsertAt
//
/////////////////////////////////////////////////////
//
// Purpose:
//    inserts ego position control data into the array
// Parameter pepcNewElement:
//    pointer to a EGOPOSITIONCONTROL structure
//    containing the data to add to the array; the
//    structure itself is not added to the array;
//    its data is simply copied into the array's
//    private data array
// Parameter nPos:
//    the 0-based index in the array where the
//    data should be added; if the index is beyond
//    the end of the array, this method has the
//    same effect as Add
//
/////////////////////////////////////////////////////

void CEgoPositionControlArray::InsertAt(
                                   const PEGOPOSITIONCONTROL pepcNewElement,
                                   int nPos)
{
    // don't try to insert a NULL element into the array
    assert(pepcNewElement != NULL);

    // increment the record of the last element's index in a temporary variable
    // first, in case this operation for some reason fails
    int nLastElement = m_nLastElement;
    nLastElement++;

    if (nPos >= nLastElement)
    {
        // use the Add method if the InsertAt operation is to be performed
        // past the end of the current array; there's no point in duplicating
        // code here
        try
        {
            Add(pepcNewElement);
            return;
        }
        catch(CMallocFailedException)
        {
            throw;
        }
    }
    else if (nPos < 0)
    {
        // can't insert an element into a negative position, so just assume
        // that 0 is the target position
        nPos = 0;
    }

    // if the last element (a 0-based index) is equal to the array size (a 1-based
    // count), then inserting an element will push the other elements past the
    // end of the current array, so the array needs to grow
    if (nLastElement == m_nArraySize)
    {
        try
        {
            GrowArray();
        }
        catch(CMallocFailedException)
        {
            throw;
        }
    }

    // determing the number of elements that need to be moved to perform
    // this insertion
    int nElementsToMove = nLastElement - nPos;

    // move the existing elements out of the way
    memmove(m_pepcTheArray + nPos + 1, m_pepcTheArray + nPos, 
        nElementsToMove * sizeof(EGOPOSITIONCONTROL));
    // insert the new element
    memcpy(m_pepcTheArray + nPos, pepcNewElement, sizeof(EGOPOSITIONCONTROL));

    // make the record of the last element in the array permanent
    m_nLastElement = nLastElement;
}

/////////////////////////////////////////////////////
//
// SetAt
//
/////////////////////////////////////////////////////
//
// Purpose:
//    copies ego position control data into the array,
//    overwriting the data that is currently there
// Parameter pepcNewElement:
//    pointer to a EGOPOSITIONCONTROL structure
//    containing the data to add to the array; the
//    structure itself is not added to the array;
//    its data is simply copied into the array's
//    private data array
// Parameter nPos:
//    the 0-based index in the array where the
//    data should be added; must not be out of
//    bounds
//
/////////////////////////////////////////////////////

void CEgoPositionControlArray::SetAt(const PEGOPOSITIONCONTROL pepcNewElement,
                                     int nPos)
{
    // don't try to put a NULL element into the array
    assert(pepcNewElement != NULL);

    // because this is a SetAt operation, the index where the new element
    // is to be placed must be within bounds
    if (nPos < 0 || nPos > m_nLastElement)
    {
        throw CArrayIndexOutOfBoundsException();
    }

    // this method overwrites whatever is currently in the position indicated
    // by nPos
    memcpy(m_pepcTheArray + nPos, pepcNewElement, sizeof(EGOPOSITIONCONTROL));
}

/////////////////////////////////////////////////////
//
// RemoveAt
//
/////////////////////////////////////////////////////
//
// Purpose:
//    removes ego position control data from the
//    array and moves data following the removed
//    data toward the front of the array
// Parameter nPos:
//    the 0-based index of the first item to be
//    removed
// Parameter nCount:
//    the number of items to remove
//
/////////////////////////////////////////////////////

void CEgoPositionControlArray::RemoveAt(int nPos, int nCount /* = 1 */)
{
    // can't remove stuff that's not in the array
    assert(nPos >= 0 && nPos <= m_nLastElement);
    
    // caller specified to remove 0 or fewer elements; ignore the request
    if (nCount <= 0)
    {
        return;
    }

    // check to make sure that deleting the specified number of elements will
    // not delete past the end of the array; if it will, then reduce the count
    // of elements to be deleted
    if ((nPos + nCount) - 1 > m_nLastElement)
    {
        nCount = (m_nLastElement - nPos) + 1;
    }

    // move the elements behind the elements to be removed to a position lower
    // in the array (this overwrites the elements that are to be removed)
    if ((nPos + nCount) - 1 <= m_nLastElement)
    {
        memmove(m_pepcTheArray + nPos, m_pepcTheArray + nPos + nCount,
                (m_nLastElement - ((nPos + nCount) - 1)) * 
                sizeof(EGOPOSITIONCONTROL));
    }

    // update the last element index to account for the remove operation
    m_nLastElement -= nCount;
    // zero out the memory that is past the last element index in the array,
    // to be sure that the elements were completely removed
    memset(m_pepcTheArray + m_nLastElement + 1, 0,
           (m_nArraySize - m_nLastElement - 1) * sizeof(EGOPOSITIONCONTROL));

}

/////////////////////////////////////////////////////
//
// RemoveAll
//
/////////////////////////////////////////////////////
//
// Purpose:
//    removes all data from the array and causes the
//    array object to report that there are no longer
//    any elements in the array
//
/////////////////////////////////////////////////////

void CEgoPositionControlArray::RemoveAll()
{
    // set the last element to -1 to indicate that there are no
    // elements currently in the array
    m_nLastElement = -1;
    // zero the array's memory
    memset(m_pepcTheArray, 0, m_nArraySize * sizeof(EGOPOSITIONCONTROL));
}

/////////////////////////////////////////////////////
//
// operator[]
//
/////////////////////////////////////////////////////
//
// Purpose:
//    accesses one of the elements of the array
// Parameter nPos:
//    the 0-based index of the item to return; must
//    not be out of bounds
// Return value:
//    a reference to the ego position control data
//    at position nPos; note that this is a reference
//    to the array's copy of the data; changes to
//    the returned data will affect the array
//
/////////////////////////////////////////////////////

EGOPOSITIONCONTROL& CEgoPositionControlArray::operator [](int nPos)
{
    try
    {
        return GetAt(nPos);
    }
    catch(CArrayIndexOutOfBoundsException)
    {
        throw;
    }
}

/////////////////////////////////////////////////////
//
// GetAt
//
/////////////////////////////////////////////////////
//
// Purpose:
//    accesses one of the elements of the array
// Parameter nPos:
//    the 0-based index of the item to return; must
//    not be out of bounds
// Return value:
//    a reference to the ego position control data
//    at position nPos; note that this is a reference
//    to the array's copy of the data; changes to
//    the returned data will affect the array
//
/////////////////////////////////////////////////////

EGOPOSITIONCONTROL& CEgoPositionControlArray::GetAt(int nPos)
{
    if (nPos < 0 || nPos > m_nLastElement)
    {
        throw CArrayIndexOutOfBoundsException();
    }

    return m_pepcTheArray[nPos];
}

/////////////////////////////////////////////////////
//
// SetSize
//
/////////////////////////////////////////////////////
//
// Purpose:
//    sets the capacity of this CEgoPositionControlArray
//    object; optionally sets the grow size, as well
// Parameter nNewSize:
//    the maximum number of elements that can be stored
//    in this CEgoPositionControlArray object; if this
//    number is less than the current capacity, then the
//    elements at the end of the array are dropped
// Parameter nGrowBy:
//    when the capacity of the array increases,
//    it is increased by nGrowBy elements; if
//    this parameter is < 1, then the array
//    grows by nNewSize on each capacity
//    increase; defaults to growing by nNewSize
// Return value:
//    the new size of the array
//
/////////////////////////////////////////////////////

int CEgoPositionControlArray::SetSize(int nNewSize, int nGrowBy /* = -1 */)
{
    // the new array size must be at least 1
    assert(nNewSize > 0);

    if (nNewSize == m_nArraySize)
    {
        // don't reallocate if the new size is the same as the current size;
        // set growby info (see constructor)
        if (nGrowBy <= 0)
        {
            m_nGrowBy = m_nArraySize;
        }
        else
        {
            m_nGrowBy = nGrowBy;
        }

        return m_nArraySize;
    }

    // a temp. variable to hold the pointer to the array's new memory
    // for error-checking prior to saving the reference permanently
    PEGOPOSITIONCONTROL pepcReallocedArray;
    // realloc the array
    pepcReallocedArray = 
        reinterpret_cast<PEGOPOSITIONCONTROL>(realloc(m_pepcTheArray, 
            nNewSize * sizeof(EGOPOSITIONCONTROL)));
    if (!pepcReallocedArray)
    {
        throw CMallocFailedException();
    }
    else
    {
        // realloc successful - save the reference
        m_pepcTheArray = pepcReallocedArray;
        if (nNewSize < m_nArraySize)
        {
            // if the new size is less than the current size, all of the
            // elements beyond the new size will be dropped
            m_nLastElement = nNewSize - 1;
        }
        // make the new size official
        m_nArraySize = nNewSize;

        // set growby info (see constructor)
        if (nGrowBy <= 0)
        {
            m_nGrowBy = m_nArraySize;
        }
        else
        {
            m_nGrowBy = nGrowBy;
        }
    }

    return m_nArraySize;
}

/////////////////////////////////////////////////////
//
// GrowArray
//
/////////////////////////////////////////////////////
//
// Purpose:
//    increases the capacity of the array
//
/////////////////////////////////////////////////////

void CEgoPositionControlArray::GrowArray()
{
    // calculate the new size for the array once it has grown;
    // store it in a temporary variable until it's known the operation
    // was successful
    int nNewSize = m_nArraySize + m_nGrowBy;

    // a temp. variable to hold the pointer to the array's new memory
    // for error-checking prior to saving the reference permanently
    PEGOPOSITIONCONTROL pepcReallocedArray;
    // realloc the array
    pepcReallocedArray = 
        reinterpret_cast<PEGOPOSITIONCONTROL>(realloc(m_pepcTheArray, 
            nNewSize * sizeof(EGOPOSITIONCONTROL)));

    if (!pepcReallocedArray)
    {
        throw CMallocFailedException();
    }
    else
    {
        // realloc successful - save the reference
        m_pepcTheArray = pepcReallocedArray;
        // make the new size official
        m_nArraySize = nNewSize;
    }
}

/////////////////////////////////////////////////////
//
// GetElementCount
//
/////////////////////////////////////////////////////
//
// Purpose:
//    returns the number of actual elements currently
//    in the array
// Return value:
//    see Purpose
//
/////////////////////////////////////////////////////

int CEgoPositionControlArray::GetElementCount()
{
    return m_nLastElement + 1;
}

/////////////////////////////////////////////////////
//
// GetSize
//
/////////////////////////////////////////////////////
//
// Purpose:
//    returns the current capacity of the array before
//    a grow will be required
// Return value:
//    see Purpose
//
/////////////////////////////////////////////////////

int CEgoPositionControlArray::GetSize()
{
    return m_nArraySize;
}

/////////////////////////////////////////////////////
//
// SetAbsolutePositioningItem
//
/////////////////////////////////////////////////////
//
// Purpose:
//    saves the single absolute positioning ego
//    position control that is allowed to exist
// Parameter pepcItem:
//    pointer to an EGOPOSITIONCONTROL structure
//    containing the data for the absolute positioning
//    item; the structure itself is not saved to
//    the CEgoPositionControlArray object;
//    its data is simply copied into the object's
//    private data area
//
/////////////////////////////////////////////////////

void CEgoPositionControlArray::SetAbsolutePositioningItem(
            const PEGOPOSITIONCONTROL pepcItem)
{
    if (pepcItem == NULL)
    {
        // if the item is NULL, then set the values for the EGOPOSITIONCONTROL
        // so that the object knows that there is no absolute positioning
        // item
        m_epcAbsolutePos.nPosX = NO_POS_X;
        m_epcAbsolutePos.nPosY = NO_POS_Y;
    }
    else
    {
        assert(pepcItem->nSrcRoom == ABSOLUTE_POSITIONING_SRC_ROOM);

        memcpy(&m_epcAbsolutePos, pepcItem, sizeof(EGOPOSITIONCONTROL));
    }
}

/////////////////////////////////////////////////////
//
// GetAbsolutePositioningItem
//
/////////////////////////////////////////////////////
//
// Purpose:
//    returns a pointer to this CEgoPositionControlArray
//    object's absolute positioning item
// Return value:
//    see Purpose
// Remarks:
//    this is a pointer to this object's absolute
//    positioning item; changes to the item pointed
//    to affect this object's copy of the item
//
/////////////////////////////////////////////////////

PEGOPOSITIONCONTROL CEgoPositionControlArray::GetAbsolutePositioningItem()
{
    if (m_epcAbsolutePos.nPosX == NO_POS_X ||
        m_epcAbsolutePos.nPosY == NO_POS_Y)
    {
        return NULL;
    }
    else
    {
        return &m_epcAbsolutePos;
    }
}