/* ------------------------------------------------------------------------- */
/*                                                                           */
/*   TITLE       : DEMOVMR1.C (MSC Version)                                  */
/*                                                                           */
/*   DESCRIPTION :                                                           */
/*       This program demonstrates how to perform voice recording using the  */
/*       SGVOICEM.LIB library. The voice recording is using the Conventional */
/*       memory method. The recording can be terminated by pressing ESC.     */
/*                                                                           */
/* ------------------------------------------------------------------------- */

#include  <io.h>
#include  <string.h>
#include  <stdlib.h>
#include  <dos.h>
#include  <bios.h>
#include  <stdio.h>
#include  <fcntl.h>
#include  <conio.h>

#include  "sgsdk.h"
#include  "sgc.h"

/* Function Prototypes */
void SaveVoiceFile(char *,char far *);
int Recording(char far *lpBuf, unsigned long lBufSize);
int WriteToFile(int Handle, char far *lpBuf, unsigned long lSize);
void RecordVoice(char *szFilename);


main()
{
    extern  char far * near voice_drv;
    unsigned version_number;
    int      iDSPPort=0x220, iIRQ=7, iDMA=1;

    /* Display SG SDK version number */
    version_number = sgGetVocLibVersion();
    printf("Sound Galaxy Developer Kit, ");
    printf("SGVOICEM Library Version ");
    printf("%d.%02d\n\n", (version_number>>8) & 0xff, version_number & 0xff);

    /* set to Sound Blaster mode */
    if (!sgSetMode(SB8MODE))
	printf("Set to Sound Blaster compatible mode.\n");

    /***
       Insert BLASTER environment and CARD detection here if desired;
       otherwise, use the configuration program for Sound Galaxy cards
       to set to the desired settings

       Set global variables to the desired IO address, IRQ and DMA channel
       Remember to use the supplied configuration program to configure
       your sound card
    ***/
    printf("Please wait...acquiring card settings...\n");

    iDSPPort = TestBaseAddr();
    if (iDSPPort<0x220 || iDSPPort>0x240)
    {
	printf("Wrong base I/O address setting: %d\n", iDSPPort);
	printf("Please use configuration utility.\n");
	exit(1);
    }
    sgSetBaseAddress(iDSPPort);

    iIRQ = sg_scan_int();
    if (iIRQ<=0)
    {
	printf("Wrong IRQ setting: %d\n", iIRQ);
	printf("Please use configuration utility.\n");
	exit(1);
    }
    sgSetIRQ(iIRQ);

    iDMA = sg_test_dma();
    if (iDMA<0)
    {
      printf("DMA setting is wrong!\n");  
      exit(1);
    }
    sgSetDMA(iDMA);

    printf("DSP Port Address = 0x%x\n", iDSPPort);
    printf("IRQ              = %d\n", iIRQ);
    printf("DMA Channel      = %d\n", iDMA);

    // set recording source & volume
    sgDSPAux_set_input_source(MICROPHONE);
    sgDSPAux_SetVolume(MICVOL,7,7);

    if (!sgDSPvm_init())
	{
	sgDSPvm_speaker(0);

	RecordVoice("DEMO.VOC");
	sgDSPvm_terminate();
	}
	else
	printf ("Error on library initialisation!\n");
	return (0);
}

/* ------------------------------------------------------------------------ */
/*  USAGE:                                                                  */
/*                                                                          */
/*   RecordVoice (char *szFilename)                                         */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Record voice into a file with filename specified.                  */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       szFileName :- File to be recorded.                                 */
/*                                                                          */
/*   EXIT:                                                                  */
/*       None                                                               */
/*                                                                          */
/* ------------------------------------------------------------------------ */

void RecordVoice (char *szFilename)
{
    unsigned        wSeg;
    char far        *lpVoiceBuf;
    unsigned long   lBufSize = 0x20000L;

    if (!_dos_allocmem((unsigned)((lBufSize+15) >> 4),&wSeg))
    {
	FP_SEG(lpVoiceBuf) = wSeg;
	FP_OFF(lpVoiceBuf) = 0;

	if (Recording(lpVoiceBuf,lBufSize))
	{
	    SaveVoiceFile(szFilename,lpVoiceBuf);
	}

	_dos_freemem(FP_SEG(lpVoiceBuf));
     }
     else
	printf("Memory allocation error.\n");
}

/* ------------------------------------------------------------------------ */
/*  USAGE:                                                                  */
/*                                                                          */
/*   Recording (char far *lpBuf, long lBufSize)                             */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Start recording voice.                                             */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       lpBuf :- buffer for voice recording.                               */
/*       lBufSize :- buffer size.                                           */
/*                                                                          */
/*   EXIT:                                                                  */
/*       Non-zero if successful, else returns 0.                            */
/*                                                                          */
/* ------------------------------------------------------------------------ */

#pragma optimize("l", off)
int Recording (char far *lpBuf, unsigned long lBufSize)
{
    int RetVal = 0;

    /* ready to record */
    printf("\n\nPress any key to start recording...\n");
    getch();

    /* turn off speaker */
    sgDSPvm_speaker(0);

    sgDSPAux_set_stereo(STEREO);
    if (sgDSPvm_input(lpBuf,lBufSize,(unsigned int)22050) == NO_ERROR)
    {
	RetVal = 1;
	printf("\nStart recording. Please wait .....");

	while (sg_voice_status)
	{
	    if (_bios_keybrd(_KEYBRD_READY))
	    {
		/* check for escape key */
		if (_bios_keybrd(_KEYBRD_READ) == 0x11b)
		{
		    sgDSPvm_stop();
		}
	    }
	}

	printf("\nRecording end.");
    }
    return(RetVal);
}
#pragma optimize("l", on)

/* ------------------------------------------------------------------------ */
/*  USAGE:                                                                  */
/*                                                                          */
/*   SaveVoiceFile (char *szFilename, char far *lpBuf)                      */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Save recorded voice from memory to file.                           */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       szFilename :- file name to be saved to.                            */
/*       lpBuf :- recorded voice buffer.                                    */
/*                                                                          */
/*   EXIT:                                                                  */
/*       None                                                               */
/*                                                                          */
/* ------------------------------------------------------------------------ */

void SaveVoiceFile(char *szFilename, char far *lpBuf)
{
    int             Handle;
    unsigned long   lVoiceSize;
    VOCHDR          stHeader;

    strcpy(stHeader.id,"Creative Voice File\x01A");
    stHeader.voice_offset = sizeof(VOCHDR);
    stHeader.version = 0x10a;
    stHeader.check_code = ~stHeader.version + 0x1234;

    if (_dos_creat(szFilename,_A_NORMAL,&Handle))
	printf("Create %s error.\n",szFilename);
    else
    {
	/* write voice header */
	if (WriteToFile(Handle,(char far*)&stHeader,(long)sizeof(VOCHDR)))
	{
	    /* get the recorded data block size */
	    lVoiceSize = *(lpBuf+3);
	    lVoiceSize <<= 16;
	    lVoiceSize += *(unsigned far *)(lpBuf+1);

	    /*  include the block type, block length  */
	    /*  and terminator block type             */

	    lVoiceSize += 5;

	    WriteToFile(Handle,lpBuf,lVoiceSize);
	}

	_dos_close(Handle);
    }
}

/* ------------------------------------------------------------------------ */
/*  USAGE:                                                                  */
/*                                                                          */
/*   WriteToFile (int Handle, char far *lpBuf, unsigned long lSize)         */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Write data from buffer to file.                                    */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       Handle :- File handle where data to be written to.                 */
/*       lpBuf :- buffer to be written to file.                             */
/*       lSize :- Size to be written to file.                               */
/*                                                                          */
/*   EXIT:                                                                  */
/*       Non-Zero if successful, else returns 0.                            */
/*                                                                          */
/* ------------------------------------------------------------------------ */

int WriteToFile (int Handle, char far *lpBuf, unsigned long lSize)
{
    int         RetVal = 1;
    unsigned    wByteToWrite, wByteWritten;

    while (lSize)
    {
	wByteToWrite = 0x8000;

	if (lSize < 0x8000)
	    wByteToWrite = (unsigned int)lSize;
	if (_dos_write(Handle,lpBuf,wByteToWrite,&wByteWritten))
	{
	    printf("Write file error.\n");
	    RetVal = 0;
	    break;
	}
	else
	{
	    if (wByteWritten != wByteToWrite)
	    {
		printf("Disk full.\n");
		RetVal = 0;
		break;
	    }
	    else
	    {
		if (FP_OFF(lpBuf) < 0x8000)
		    FP_OFF(lpBuf) += wByteWritten;
		else
		{
		    FP_OFF(lpBuf) += wByteWritten;
		    FP_SEG(lpBuf) += 0x1000;
		}

		lSize -= wByteWritten;
	    }
	}
    }
    return(RetVal);
}

/* EOF: DEMOVMR1.C */

