//***************************************************************************
//
// Copyright (c) 1992-93 Sierra Semiconductor Corp.
//
// FILE:    playmidi.c
//
// LANGUAGE:
//          Microsoft C/C++ Version 7.0
//
// DESCRIPTION:
//
//          Aria MIDI File Player
//
//          Compile:  cl /AL /Zp /Ox /c playmidi.c
//          Link:     link playmidi,,,aria_l;
//
// Aria and Aria Synthesizer are trademarks of Sierra Semiconductor Corp.
// QSound is a trademark of Archer Communications
//
//***************************************************************************

// $Header:   F:\projects\ariai\dos\archives\playmidi.c_v   1.5   03 Sep 1993 10:15:16   golds  $
// $Log:   F:\projects\ariai\dos\archives\playmidi.c_v  $
// 
//    Rev 1.5   03 Sep 1993 10:15:16   golds
// Recompiled with API library version 2.2
// 
//    Rev 1.4   13 Aug 1993 09:15:48   golds
// Added /O option for outputting MIDI to external port
// 
//    Rev 1.0   24 Jun 1993 12:51:22   golds
// Initial revision.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#ifdef TURBOC
#include <alloc.h>
#else
#include <malloc.h>
#endif
#include <dos.h>
#include <io.h>

#include "aria.h"
#include "ariamidi.h"

#define ESC    27
#define PGUP   573
#define PGDN   581
#define F1     559
#define F2     560
#define F3     561
#define F4     562
#define F5     563
#define F6     564
#define F7     565
#define HOME   571
#define END    579

#define MAXMIDICH    16       // Maximum number of MIDI channels
#define TRIES        0xFF     // MIDI port timeout

#define STOPPED      0        // MIDI playback done flag
#define PLAYING      1        // MIDI playback active flag
#define PAUSED       2        // MIDI playback paused flag

#define DEFAULT_VOL  0x4000   // Initial synthesizer volume
#define MAX_VOL      0x7FFF   // Maximum volume level
#define MAX_REVERB   127      // Maximum reverb level
#define VOL_INCR     1024     // Volume change increments
#define REVERB_INCR  8        // Reverb level change increments

#define TIMER_COUNT  256      // 8259 timer value
#define BIOS_UPDATE  256      // Time-of-day update period (65536/TIMER_COUNT)
#define TIMER_RES    0x36EDL  // 0.21455 msec
#define TEMPO_DEF    500000L  // Default tempo (in usec per quarter note)

#define TIMER_INTR   0x08     // 8259 interrupt vector (channel 0)

// Macro to convert a long integer from Motorola format to Intel format
#define CNVLONG(x) (((x)>>24)|((x)<<24)|(((x)>>8)&0x0000FF00L)|\
                   (((x)<<8)&0x00FF0000L))

// Macro to convert a short integer from Motorola format to Intel format
#define CNVWORD(x) (((x)>>8)|((x)<<8))

// MIDI file chunk types
#define MAXCHUNK     2
enum chunktypes {HDR_CHUNK, TRK_CHUNK};
static char FAR *chunkname[MAXCHUNK] = {"MThd", "MTrk"};

// MIDI file meta-events
#define SEQNUM       0x00
#define TEXTEVENT    0x01
#define COPYRIGHT    0x02
#define SEQTRKNAME   0x03
#define INSNAME      0x04
#define LYRIC        0x05
#define MARKER       0x06
#define CUEPOINT     0x07
#define CHPREFIX     0x20
#define ENDOFTRK     0x2F  // End of track
#define SETTEMPO     0x51  // Set tempo
#define SMPTEOFF     0x54
#define TIMESIG      0x58
#define KEYSIG       0x59
#define SEQSPECIFIC  0x7F
                           // These are ours:
#define SYSEXMSG1    0xFD  // System exclusive event type 1
#define SYSEXMSG2    0xFE  // System exclusive event type 2
#define MIDIMSG      0xFF  // MIDI event

#define READSIZE   16380   // Read buffer size
#define REVERBMODE   4     // Default reverb mode
#define REVERBLEVEL  64    // Default reverb level

#define BREAKINT     0x1B  // Control break interrupt
#define CTRLCINT     0x23  // Control C interrupt

// MIDI file chunk template
typedef struct chunkstruct
   {
   char  type[4];    // Must be one of the chunktypes listed above
   DWORD size;       // Size of chunk (originally stored in Motorola format!)
   } CHUNK, HUGE *CHUNKPTR, TRACK, HUGE *TRACKPTR;

// MIDI file header template
typedef struct headerstruct
   {
   char  type[4];    // Must be "MThd"
   DWORD size;       // Size of chunk (originally stored in Motorola format!)
   WORD  format;     // File format (originally stored in Motorola format!)
   WORD  ntrks;      // No. of tracks (originally stored in Motorola format!)
   WORD  division;   // Time mode (originally stored in Motorola format!)
   } HEADER, HUGE *HEADERPTR;

// Our event structure definition -
// These are linked together, one for each MIDI track
typedef struct eventstruct
   {
   BYTE   cmd;       // Event command -  00h to EFh = MIDI command
                     //                    F0h, F7h = SysEx messages
                     //                         FFh = Meta event
   BYTE   dat1;      // MIDI data byte 1
   BYTE   dat2;      // MIDI data byte 2
   BYTE   metatype;  // Meta-event type (valid when cmd is FFh,
                     //  otherwise metatype is set to FFh)
   DWORD  timing;    // Timing count
   DWORD  size;      // MIDI event size in bytes (no. of bytes to next event)
   DWORD  tempo;     // Tempo value
   HPBYTE eventptr;  // MIDI buffer pointer
   struct eventstruct HUGE *nexttrk;   // Next MIDI track structure
   } EVENT, HUGE *EVENTPTR;

FILE   *midistream=NULL;      // MIDI file stream
FILE   *patchstream=NULL;     // Patch file stream
HPBYTE patchbnk=NULL;         // Patch bank buffer
HPBYTE midibuf=NULL;          // MIDI file buffer
HPBYTE qbuf=NULL;             // QSound DSP file buffer
HPBYTE rbuf=NULL;             // Reverb DSP file buffer
HEADERPTR hdrptr;             // MIDI header pointer
EVENTPTR firsttrk=NULL;       // First MIDI track pointer
WORD   playstate = STOPPED;   // Current MIDI playback state
char   midifile[60];          // MIDI file name
char   patchfile[60];         // Patch file name
char   qfile[60];             // QSound DSP file name
char   reverbfile[60];        // Reverb DSP file name
WORD   rmode=REVERBMODE;      // Reverb mode
UINT   rlevel=REVERBLEVEL;    // Reverb level
BOOL   reverbactive=FALSE;    // Reverb active flag
BOOL   qsoundactive=FALSE;    // QSound active flag
LONG   midisize;              // Size in bytes of MIDI file
DWORD  timer_tick=0L;         // Tempo timer (in msec per tick)
DWORD  tempo;                 // Tempo (in usec per quarter note)
BOOL   timeout=FALSE;         // Timer tick flag
BOOL   chfilter[MAXMIDICH]=   // MIDI channel filter
       {TRUE,TRUE,TRUE,TRUE,TRUE,TRUE,TRUE,TRUE,
        TRUE,TRUE,FALSE,FALSE,FALSE,FALSE,FALSE,FALSE};
BOOL   midiout=FALSE;         // MIDI output flag
BYTE   perc = 9;              // Percussion channel
HEADER hrec;                  // MIDI header information
ARIACAPS AriaCap;             // Aria capabilities structure
VOID (__interrupt __far *oldTimer)();   // Old timer interrupt vector
VOID (interrupt FAR *oldctrlchandler)();
VOID (interrupt FAR *oldbreakhandler)();

// Function prototypes
static VOID pausePlayback (VOID);
static DWORD setTempo (WORD, DWORD);
static HEADERPTR getHeader (VOID);
static TRACKPTR getTrack (TRACKPTR);
static EVENTPTR getEvent (EVENTPTR);
static HPBYTE ReadVarLen (HPBYTE, LPDWORD);
static short getChunkType (CHUNKPTR);
static VOID process (VOID);
static HPBYTE loadQSound (LPBYTE);
static HPBYTE loadReverb (LPBYTE);
static VOID __interrupt __far timer_intr (VOID);
static VOID errexit (short, LPBYTE);
static BOOL putcmd (BYTE);
static BOOL putdata (BYTE);
VOID interrupt FAR newctrlchandler (VOID);
VOID interrupt FAR newbreakhandler (VOID);


main (short argc, LPSTR argv[])
   {
   EVENTPTR evtmp1, evtmp2;
   TRACKPTR trkptr;
   char FAR *strptr;
   LONG patchsize, j;
   short i;
   short ch;
   BOOL rdy=FALSE;
   static UINT vol=DEFAULT_VOL;

   //*** Print title and copyright notice ***

   printf ("\nPLAYMIDI - Aria Standard MIDI File Player (Version 1.5)");
   printf ("\nCopyright (c) 1992-93 Sierra Semiconductor Corp.\n");

   //*** Check for command line argument(s) ***

   if (argc < 2)
      errexit (5, NULL);

   //*** Initialize Aria system ***

   i = SystemInit (ARIA_SYNTH);
   if (i != 0)
      errexit (9, NULL);

   GetAriaCaps ((LPBYTE) &AriaCap);

   //*** Set the initial volume ***

   SetSynthVolume (vol, vol);

   //*** Get command line arguments ***

   i = 1;
   midifile[0]  = '\0';
   patchfile[0] = '\0';
   qfile[0]     = '\0';
   while (i < argc)
      {
      switch (argv[i][0])
         {
         case '/':
            switch (argv[i][1])
               {
               case 'Q':                     // QSound option
               case 'q':
                  if (reverbactive)          // Cannot use reverb if QSound
                     break;
                  strcpy (qfile, &argv[i][2]);
                  if (qfile[0] == '\0')      // Is no .DSP file specified?
                     {
                     if (!(AriaCap.DSPapps & QSOUND_MASK))
                        printf ("\nWARNING - QSound not available\n");
                     else
                        qsoundactive = TRUE;
                     }                       // Otherwise, load DSP app
                  else if ((qbuf = loadQSound ((LPBYTE) qfile)) == NULL)
                     errexit (10, (HPBYTE) qfile);
                  else
                     qsoundactive = TRUE;
                  break;

               case 'R':                     // Reverb option
               case 'r':
                  if (qsoundactive)          // Cannot use QSound if reverb
                     break;
                  strcpy (reverbfile, &argv[i][2]);
                  if (reverbfile[0] == '\0') // Is no .DSP file specified?
                     {
                     if (!(AriaCap.DSPapps & REVERB_MASK))
                        printf ("\nWARNING - Reverb not available\n");
                     else
                        reverbactive = TRUE;
                     }                       // Otherwise, load DSP app
                  else if ((rbuf = loadReverb ((LPBYTE) reverbfile)) == NULL)
                     errexit (10, (HPBYTE) reverbfile);
                  else
                     reverbactive = TRUE;
                  break;

               case 'M':
               case 'm':
                  switch (argv[i][2])
                     {
                     case '1':
                        for (ch = 0; ch <= 11; ++ch)
                           chfilter[ch] = FALSE;
                        for (ch = 12; ch <= 15; ++ch)
                           chfilter[ch] = TRUE;
                        perc = 15;
                        break;
                     case '2':
                        for (ch = 0; ch <= 11; ++ch)
                           chfilter[ch] = TRUE;
                        for (ch = 12; ch <= 15; ++ch)
                           chfilter[ch] = FALSE;
                        break;
                     case '0':
                     default:
                        for (ch = 0; ch <= 15; ++ch)
                           chfilter[ch] = TRUE;
                        break;
                     }
                  break;

               case 'O':      // MIDI output option
               case 'o':
                  midiout = TRUE;
                  break;

               case 'H':      // Help option
               case 'h':
               case '?':
                  errexit (5, NULL);

               default:       // Unknown option
                  break;
               }
            break;

         default:
            if (midifile[0] == '\0')         // 1st arg should be MIDI file
               strcpy (midifile, argv[i]);
            else if (patchfile[0] == '\0')
               strcpy (patchfile, argv[i]);  // 2nd arg should be patch bank
            break;
         }
      ++i;
      }

   if (patchfile[0] == '\0')  // No patch bank specified?
      {
      // Look for patch bank path environment variable (ARIABANK)

      if ((strptr = getenv ("ARIABANK")) != NULL)
         {
         strcpy (patchfile, strptr);
         if (patchfile[strlen(patchfile)-1] != '\\')
            strcat (patchfile, "\\");
         }

      if (AriaCap.ROMsize == 512)
         strcat (patchfile, "genmidi1.bnk");
      else if (AriaCap.ROMsize == 1024)
         strcat (patchfile, "genmidi2.bnk");
      else
         errexit (1, NULL);
      }

   // Open the patch bank file

   if ((patchstream = fopen (patchfile, "rb")) == NULL)
      errexit (2, (LPBYTE) patchfile);

   patchsize = filelength (fileno (patchstream));

   // Load the patch bank file

   if ((patchbnk = halloc ((patchsize>>1)+1L, 2)) == NULL)
      errexit (3, (LPBYTE) "patch bank file");
   j = 0L;
   while (fread ((VOID HUGE *) (patchbnk+j), 1, READSIZE, 
                 patchstream) == READSIZE)
      j += (LONG) READSIZE;
   fclose (patchstream);

   //*** Validate and initialize patch bank ***

   if (InitPatchBank (patchbnk, MAXOPERATORS) != 0)
      errexit (6, NULL);

   //*** Load MIDI file data ***

   if ((midistream = fopen (midifile, "rb")) == NULL)
      errexit (2, midifile);

   midisize = filelength (fileno (midistream));

   if ((midibuf = halloc ((midisize>>1)+1L, 2)) == NULL)
      errexit (3, (LPBYTE) "MIDI data");
   j = 0L;
   while (fread ((VOID HUGE *) (midibuf+j), 1, READSIZE, 
                 midistream) == READSIZE)
      j += (LONG) READSIZE;
   fclose (midistream);

   //*** Validate MIDI file ***

   if ((hdrptr = getHeader ()) == NULL)
      errexit (7, NULL);

   if (hrec.format > 1)
      errexit (8, NULL);

   //*** Load QSound application, if necessary

   if (qsoundactive)
      {
      if ((i = SelectDSPApp (QSOUND_ID, TRUE, (LPBYTE) qbuf)) != 0)
         errexit (11, "QSound");
      if (SetQSoundMode (QSYNTH) != 0)
         errexit (12, "QSound");
      }

   //*** Load reverb application, if necessary

   if (reverbactive)
      {
      if ((i = SelectDSPApp (REVERB_ID, TRUE, (LPBYTE) rbuf)) != 0)
         errexit (11, "Reverb");
      if (SetReverbMode (rmode) != 0)
         errexit (12, "Reverb");
      SetReverbLevel (rlevel, rlevel);
      }

   // Set up MIDI output port if requested

   if (midiout)
      {
      midibase = 0x330;       // Try 330h 
      putcmd (MRESET);
      if (!putcmd (MUART))    // If it didn't work,
         {
         midibase = 0x320;    //   Try 320h
         putcmd (MRESET);
         if (!putcmd (MUART))
            {
            midiout = FALSE;
            errexit (13, NULL);
            }
         }
      }

   // Set default tempo to 120 beats per minute

   timer_tick = setTempo (hrec.division, TEMPO_DEF);

   //*** Locate and allocate MIDI tracks ***

   trkptr = (TRACKPTR) hdrptr;
   for (i = 0; i < (short) hrec.ntrks; ++i)
      {
      if ((trkptr = getTrack (trkptr)) != NULL)
         {
         if ((evtmp1 = (EVENTPTR) _fmalloc ((size_t) sizeof (EVENT))) == NULL)
            errexit (3, (LPBYTE) "MIDI track buffers");

         evtmp1->size     = 0L;
         evtmp1->metatype = MIDIMSG;
         evtmp1->eventptr = ((HPBYTE) trkptr) + 8;
         evtmp1->nexttrk  = NULL;
         evtmp1 = getEvent (evtmp1);

         if (firsttrk == NULL)
            firsttrk = evtmp1;
         else
            evtmp2->nexttrk = evtmp1;
         evtmp2 = evtmp1;
         }
      else
         {
         printf ("\nWARNING - MIDI file missing track #%d\n", i+1);
         break;
         }
      }

   //*** Set up timer interrupt ***

   oldTimer = _dos_getvect (TIMER_INTR);
   DISABLE
   _dos_setvect (TIMER_INTR, timer_intr);

   // Reprogram timer to new speed

   OUTP (0x43, 0x36);
   OUTP (0x40, TIMER_COUNT & 0xFF);
   _asm nop
   OUTP (0x40, TIMER_COUNT >> 8);
   ENABLE

   //*** Play the file ***

   printf ("\nHit:  <Space bar> to PAUSE/RESUME");
   printf ("\n    <PgUp>/<PgDn> keys for VOLUME control");
   printf ("\n            <Esc> key to STOP");
   if (reverbactive)
      {
      printf ("\n             <Fn> keys for REVERB MODE (n = 1 to 7)");
      printf ("\n     <Home>/<End> keys for REVERB LEVEL control");
      }
   if (qsoundactive)
      {
      printf ("\n              <+> key to enable QSOUND");
      printf ("\n              <-> key to disable QSOUND");
      }

   DISABLE
   oldctrlchandler = GETVECT (CTRLCINT);
   oldbreakhandler = GETVECT (BREAKINT);
   SETVECT (CTRLCINT, newctrlchandler);
   SETVECT (BREAKINT, newbreakhandler);
   ENABLE

   printf ("\n\nPLAYING . . .");

   playstate = PLAYING;
   while (playstate != STOPPED)
      {
      while (timeout)   // Time to process midi messages
         {
         timeout = FALSE;
         process ();
         }

      if (kbhit ())     // Key hit?
         {
         switch (getch ())
            {
            case ' ':   // Space bar pauses and resumes playback
               if (playstate == PLAYING)
                  {
                  pausePlayback ();
                  printf ("\nMIDI playback is PAUSED . . .");
                  }
               else if (playstate == PAUSED)
                  {
                  playstate = PLAYING;
                  printf ("\nMIDI playback is RESUMED\n");
                  }
               break;

            case '+':   // Turn on QSound if loaded
               if (qsoundactive)
                  SetQSoundMode (QSYNTH);
               break;

            case '-':   // Turn off QSound if loaded
               if (qsoundactive)
                  SetQSoundMode (0);
               break;

            case 0:
               ch = getch () + 500;
               switch (ch)
                  {
                  case PGUP:  // Increase volume level
                     if ((vol += VOL_INCR) > MAX_VOL)
                        vol = MAX_VOL;
                     SetSynthVolume (vol, vol);
                     break;

                  case PGDN:  // Decrease volume level
                     if (vol < VOL_INCR)
                        vol = 0;
                     else
                        vol -= VOL_INCR;
                     SetSynthVolume (vol, vol);
                     break;

                  case HOME:  // Increase reverb level
                     if (reverbactive)
                        {
                        if ((rlevel += REVERB_INCR) > MAX_REVERB)
                           rlevel = MAX_REVERB;
                        SetReverbLevel (rlevel, rlevel);
                        }
                     break;

                  case END:   // Decrease reverb level
                     if (reverbactive)
                        {
                        if (rlevel < REVERB_INCR)
                           rlevel = 0;
                        else
                           rlevel -= REVERB_INCR;
                        SetReverbLevel (rlevel, rlevel);
                        }
                     break;

                  case F1:    // Change reverb mode
                  case F2:
                  case F3:
                  case F4:
                  case F5:
                  case F6:
                  case F7:
                     if (reverbactive)
                        {
                        rmode = (ch - F1) + 1;
                        SetReverbMode (rmode);
                        }
                     break;
                  }
               break;

            case ESC:    // Esc key halts playback
               pausePlayback ();
               playstate = STOPPED;
               printf ("\nMIDI playback has STOPPED\n");
               break;
            }
         }
      }

   //*** Reset timer interrupt and clock cycles back to 18.2 Hz ***

   DISABLE
   SETVECT (CTRLCINT, oldctrlchandler);
   SETVECT (BREAKINT, oldbreakhandler);

   OUTP (0x43, 0x36);
   OUTP (0x40, 0);
   _asm nop
   OUTP (0x40, 0);

   _dos_setvect (TIMER_INTR, oldTimer);
   ENABLE

   if (midiout)
      putcmd (MRESET);

   //*** Clean up memory and exit ***

   errexit (0, NULL);
   }


//***************************************************************************
// Suspend playback of synthesizer and any MIDI output
//***************************************************************************

static VOID pausePlayback (VOID)
   {
   register short i;

   playstate = PAUSED;
   // Send all notes off
   for (i = 0; i < 16; ++i)
      {
      MIDImessage ((BYTE) (0xB0 | i), 123, 0);
      if (midiout)
         {
         putdata ((BYTE) (0xB0 | i));
         putdata (123);
         putdata (0);
         }
      }
   }


//***************************************************************************
// Calculate new timer value
//
// The technique is to reprogram the PC timer to a known value (.21 msec/QN),
// then calculate the required tempo in msec's per quarter note.
// By accumulating tempo count each timer tick and waiting for it to
// become greater than or equal to our timer value, we can generate a
// MIDI tick each time.
//***************************************************************************

static DWORD setTempo (WORD division, DWORD newtempo)
   {
   DWORD n, d, rem;

   tempo = newtempo;
   d   = (DWORD) division * 1000L;
   n   = tempo / d;
   rem = ((tempo % d) << 11) / (d >> 5);

   return ((n << 16) | rem);  // This is the tempo value
   }


//***************************************************************************
// Locate header chunk in MIDI file
//***************************************************************************

static HEADERPTR getHeader (VOID)
   {
   HEADERPTR hptr;
   DWORD count=0L;
   DWORD size;

   hptr = (HEADERPTR) midibuf;
   while (getChunkType ((CHUNKPTR) hptr) != HDR_CHUNK)
      {
      size = CNVLONG (hptr->size) + 8L;
      hptr = (HEADERPTR) (((HPBYTE) (hptr)) + size);
      count += size;
      if (count >= (DWORD) midisize)
         return NULL;
      }

   // Convert Motorola (MSB first) to Intel format (LSB first)

   hrec.size     = CNVLONG (hptr->size);
   hrec.format   = CNVWORD (hptr->format);
   hrec.ntrks    = CNVWORD (hptr->ntrks);
   hrec.division = CNVWORD (hptr->division);

   return hptr;
   }


//***************************************************************************
// Locate next track chunk in MIDI file
//***************************************************************************

static TRACKPTR getTrack (TRACKPTR tptr)
   {
   DWORD count=0L;
   DWORD size;

   do
      {
      size = CNVLONG (tptr->size) + 8L;
      tptr = (TRACKPTR) (((HPBYTE) (tptr)) + size);
      count += size;
      if (count >= (DWORD) midisize)
         return NULL;
      } while (getChunkType ((CHUNKPTR) tptr) != TRK_CHUNK);

   return tptr;
   }


//***************************************************************************
// Locate next event in track and load information
//***************************************************************************

static EVENTPTR getEvent (EVENTPTR mptr)
   {
   HPBYTE   datptr;
   EVENTPTR nextptr;
   DWORD    val;
   static DWORD count;

   if (mptr->eventptr != NULL && mptr->metatype != ENDOFTRK)
      {
      // Set pointer to next event

      datptr = ((HPBYTE) mptr->eventptr) + mptr->size;

      // Get the next timing value

      datptr = ReadVarLen (datptr, (LPDWORD) &count);
      mptr->timing = count;

      nextptr = (EVENTPTR) datptr;
      mptr->eventptr = datptr;

      // Determine type of event

      if (nextptr->cmd <= 0xEF)  // Midi message
         {
         mptr->metatype = MIDIMSG;

         // Running status?
         if (nextptr->cmd < 0x80)
            {
            mptr->size = 1L;
            mptr->dat1 = nextptr->cmd;
            if (mptr->cmd < 0xC0 || mptr->cmd > 0xDF)
               {
               mptr->dat2 = nextptr->dat1;
               mptr->size++;
               }
            }
         else
            {
            mptr->size = 2L;
            mptr->cmd = nextptr->cmd;
            mptr->dat1 = nextptr->dat1;
            if (mptr->cmd < 0xC0 || mptr->cmd > 0xDF)
               {
               mptr->dat2 = nextptr->dat2;
               mptr->size++;
               }
            }
         }
      else  // Sysex or meta-event
         {
         switch (nextptr->cmd)
            {
            case 0xFF:  // Meta-event
               datptr++;
               mptr->metatype = *datptr;
               datptr = ReadVarLen (++datptr, (LPDWORD) &count);
               mptr->size = count;
               mptr->eventptr = datptr;

               switch (mptr->metatype)
                  {
                  case SETTEMPO: // Set tempo
                     val = ((DWORD) *datptr) << 16;
                     datptr++;
                     val |= (((DWORD) *datptr) << 8);
                     datptr++;
                     val |= (DWORD) *datptr;
                     mptr->tempo = val;
                     break;

                  default:       // Others not supported
                     break;
                  }
               break;

            case 0xF0:
               mptr->metatype = SYSEXMSG1;
               mptr->eventptr = ReadVarLen (++datptr, (LPDWORD) &count);
               mptr->size = count;
               break;

            case 0xF7:
               mptr->metatype = SYSEXMSG2;
               mptr->eventptr = ReadVarLen (++datptr, (LPDWORD) &count);
               mptr->size = count;
               break;

            case 0xF1:
            case 0xF3:
               mptr->metatype = MIDIMSG;
               mptr->cmd  = nextptr->cmd;
               mptr->dat1 = nextptr->dat1;
               mptr->size = 2L;
               break;

            case 0xF2:
               mptr->metatype = MIDIMSG;
               mptr->cmd  = nextptr->cmd;
               mptr->dat1 = nextptr->dat1;
               mptr->dat2 = nextptr->dat2;
               mptr->size = 3L;
               break;

            default:
               mptr->metatype = MIDIMSG;
               mptr->cmd  = nextptr->cmd;
               mptr->size = 1L;
               break;
            }
         }
   	}
   return mptr;
   }


//***************************************************************************
// Get next variable length timing count from MIDI file
//***************************************************************************

static HPBYTE ReadVarLen (HPBYTE datptr, LPDWORD count)
   {
   DWORD cnt;
   BYTE  c;

   if ((cnt = (DWORD) *datptr) & 0x80)
      {
      ++datptr;
      cnt &= 0x7F;
      do
         {
         cnt = (cnt << 7) + (DWORD) ((c = (BYTE) *datptr) & 0x7F);
         ++datptr;
         } while (c & 0x80);
      }
   else
      ++datptr;

   *count = cnt;

   return datptr;
   }


//***************************************************************************
// Determine MIDI file chunk type
//***************************************************************************

static short getChunkType (CHUNKPTR ptr)
   {
   short i;

   for (i = 0; i < MAXCHUNK; ++i)
      if (strncmp (chunkname[i], ptr->type, 4) == 0)
         return i;
   return 0xFFFF;
   }


//***************************************************************************
// Process MIDI event
//***************************************************************************

static VOID process (VOID)
   {
   EVENTPTR mptr;
   short  state=STOPPED;
   BYTE   i;
   static DWORD last_tick=0L;
   static DWORD midi_tick=0L;

   // If currently playing,

   if (playstate == PLAYING)
      {
      // Time to do something?

      if (midi_tick == 0L)
         {
         midi_tick = 0xFFFFFFFFL;

         // Check each track

         mptr = firsttrk;
         while (mptr != NULL)
            {
            // If the track has not finished playing,

            if (mptr->metatype != ENDOFTRK)
               {
               // Determine largest wait count for next events

               if (last_tick > 0L)
                  mptr->timing -= last_tick;

               // Time to play something?

               while (mptr->timing == 0L && mptr->metatype != ENDOFTRK)
                  {
                  switch (mptr->metatype)
                     {
                     case MIDIMSG:
                        i = (BYTE) (mptr->cmd & 0x0F);
                        if (mptr->cmd <= 0xEF && chfilter[i])
                           {
                           if (i == perc)
                              mptr->cmd = ((mptr->cmd & 0xF0) | 0x09);
                           MIDImessage (mptr->cmd, mptr->dat1, mptr->dat2);
                           if (midiout)
                              {
                              putdata (mptr->cmd);
                              putdata (mptr->dat1);
                              if (mptr->cmd < 0xC0 || mptr->cmd > 0xDF)
                                 putdata (mptr->dat2);
                              }
                           }
                        break;

                     case SETTEMPO:
                        timer_tick = setTempo (hrec.division, mptr->tempo);
                        break;

                     case SYSEXMSG1:
                     case SYSEXMSG2:
                        // do nothing
                        break;
                     }
                  mptr = getEvent (mptr);
                  }
               if (mptr->metatype != ENDOFTRK)
                  {
                  --mptr->timing;
                  if (mptr->timing < midi_tick)
                     midi_tick = mptr->timing;
                  state = PLAYING;
                  }
               }
            mptr = mptr->nexttrk;
            }
         playstate = state;
         last_tick = midi_tick;  // Save largest wait count
         }
      else
         --midi_tick;   // Decrement wait count
      }
   }


//***************************************************************************
// Load QSound DSP application from disk
//***************************************************************************

static HPBYTE loadQSound (LPBYTE qfile)
   {
   FILE *qstream;
   LONG qsize, j;
   HPBYTE qbuf;

   if ((qstream = fopen (qfile, "rb")) == NULL)
      return NULL;

   qsize = filelength (fileno (qstream));

   if ((qbuf = halloc ((qsize>>1)+1L, 2)) == NULL)
      errexit (3, (LPBYTE) "QSound DSP file");

   j = 0L;
   while (fread ((VOID HUGE *) (qbuf+j), 1, READSIZE, qstream) == READSIZE)
      j += (LONG) READSIZE;
   fclose (qstream);

   return qbuf;
   }


//***************************************************************************
// Load Reverb DSP application from disk
//***************************************************************************

static HPBYTE loadReverb (LPBYTE reverbfile)
   {
   FILE *reverbstream;
   LONG reverbsize, j;
   HPBYTE rbuf;

   if ((reverbstream = fopen (reverbfile, "rb")) == NULL)
      return NULL;

   reverbsize = filelength (fileno (reverbstream));

   if ((rbuf = halloc ((reverbsize>>1)+1L, 2)) == NULL)
      errexit (3, (LPBYTE) "Reverb DSP file");

   j = 0L;
   while (fread ((VOID HUGE *) (rbuf+j), 1, READSIZE, 
                 reverbstream) == READSIZE)
      j += (LONG) READSIZE;
   fclose (reverbstream);

   return rbuf;
   }


//***************************************************************************
// Interrupt Service Routine for timer
//***************************************************************************

static VOID __interrupt __far timer_intr (VOID)
   {
   static DWORD tick_count=0L;            // Timer tick accumulator
   static WORD  pending=0;                // Number of pending interrupts
   static WORD  bios_count=BIOS_UPDATE;   // Time-of-day service counter
   static BOOL  busy=FALSE;               // ISR busy flag-Reentrancy allowed

   if (busy)
      ++pending;
   else
      {
      busy = TRUE;

      // Add to our tick accumulator

      tick_count += (TIMER_RES * (DWORD) (pending+1));
      pending = 0;

      // Check to see if it's time for a MIDI tick

      if (tick_count >= timer_tick)
         {
         timeout = TRUE;

         // Reduce tick accumulator by timer tick value
         if (TIMER_RES > timer_tick)   // In this case, we must process
            tick_count = 0;            //               every time
         else
            tick_count -= timer_tick;
         }
      busy = FALSE;
      }

   // Time to chain to the real timer ISR ?

   if (--bios_count == 0)
      {
      bios_count = BIOS_UPDATE;
      _chain_intr (oldTimer);
      }
   outp (0x20, 0x20);   // End of interrupt
   }


//***************************************************************************
// Exit/Error handler
//***************************************************************************

static VOID errexit (short type, LPBYTE msg)
   {
   EVENTPTR tmp;

   switch (type)
      {
      case 0:
         printf ("\nAll done\n");
         break;

      case 1:
         printf ("\nERROR - Patch bank file name must be specified\n");
         break;

      case 2:
         printf ("\nERROR - Could not open %s\n", msg);
         break;

      case 3:
         printf ("\nERROR - Could not allocate memory for %s\n", msg);
         break;

      case 4:
         printf ("\nERROR - Could not read file %s\n", msg);
         break;

      case 5:
         printf ("\nSYNTAX:  PLAYMIDI midifile <bankfile> </Qdspfile> </R> </Mn>");
         printf ("\n\n");
         printf ("where, midifile = Standard MIDI file (format 0 or 1)\n");
         printf ("\n       ");
         printf ("bankfile = Optional Aria patch bank file\n");
         printf ("\n                  ");
         printf ("If no patch bank is specified,");
         printf ("\n                   ");
         printf ("GENMIDI1.BNK will be used for 512K ROM");
         printf ("\n                   ");
         printf ("GENMIDI2.BNK will be used for 1M ROM\n");
         printf ("\n       ");
         printf ("dspfile  = Optional QSound DSP application file\n");
         printf ("\n       ");
         printf ("/R       = Activate reverb if available");
         printf ("\n       ");
         printf ("/Mn      = MIDI channel filter mode:");
         printf ("\n       ");
         printf ("           if n=0 or blank, all channels play (10 is percussion)");
         printf ("\n       ");
         printf ("              n=1, channels 13 to 16 play (16 is percussion)");
         printf ("\n       ");
         printf ("              n=2 (DEFAULT), channels 1 to 10 play (10 is percussion)");
         break;

      case 6:
         printf ("\nERROR - Invalid patch bank file\n");
         break;

      case 7:
         printf ("\nERROR - Invalid Standard MIDI file\n");
         break;

      case 8:
         printf ("\nERROR - Cannot play MIDI file format type\n");
         break;

      case 9:
         printf ("\nERROR - Cannot initialize Aria system\n");
         break;

      case 10:
         printf ("\nERROR - Cannot load DSP application: %s\n", msg);
         break;

      case 11:
         printf ("\nERROR - Cannot initialize %s application\n", msg);
         break;

      case 12:
         printf ("\nERROR - Cannot initialize %s mode\n", msg);
         break;

      case 13:
         printf ("\nERROR - Cannot initialize MIDI output port\n");
         break;
      }

   // Free the track structure linked list

   while (firsttrk != NULL)
      {
      tmp = firsttrk;
      firsttrk = firsttrk->nexttrk;
      _ffree (tmp);
      }

   // Free the patch bank buffer and MIDI buffer

   if (patchbnk != NULL)
      hfree (patchbnk);
   if (midibuf != NULL)
      hfree (midibuf);
   if (qbuf != NULL)
      hfree (qbuf);
   if (rbuf != NULL)
      hfree (rbuf);

   // Return the Aria system to Sound Blaster mode

   SystemInit (DEFAULT_SYNTH);

   // All done

   exit (type);
   }


static BOOL putcmd (BYTE cmd)
   {
   WORD i;
   BYTE dat;

   DISABLE
   OUTP (MIDICMD, cmd);
   i = TRIES;
   while (--i)
      {
      dat = (BYTE) INP (MIDISTAT);
      if (!(dat & DSR))
         break;
      }
   if (!i)
      {
      ENABLE
      return FALSE;
      }
   dat = (BYTE) INP (MIDIDATA);
   if (dat != MACK)
      {
      ENABLE
      return FALSE;
      }
   ENABLE

   return TRUE;
   }


static BOOL putdata (BYTE mididat)
   {
   WORD i;
   BYTE dat;

   DISABLE
   i = TRIES;
   while (--i)
      {
      dat = (BYTE) INP (MIDISTAT);
      if (!(dat & DRR))
         break;
      }
   if (!i)
      {
      ENABLE
      return FALSE;
      }
   OUTP (MIDIDATA, mididat);
   ENABLE

   return TRUE;
   }


VOID interrupt FAR newctrlchandler (VOID)
   {
   }


VOID interrupt FAR newbreakhandler (VOID)
   {
   }

