/*
    setpanel.h - A user-space program to write to the ASUS DigiMatrix front panel.
    Copyright (c) 2005 Richard Taylor <richard@artaylor.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* Some info gathered about the registers:
00 - 1st Digit
01 - 2nd
02 - 3rd
03 - 4th
04 - 5th
05 - 6th

06 - Colon 1, 2, 3 (bottom, middle, both)
07 - Not sure... 
08 - Show play symbol
09 - Show pause symbol
10 - 
11 - HD Mode (HD Symbol)
12 - CD Mode (CD Symbol)
13 - FM Mode (FM + Radio on)
14 - AM Mode (AM + Radio on)
15 - Turns off Radio / Current mode? (Doesn't clear icon)

16 - Radio Next Preset
17 - Radio Prev Preset
18 - 
19 - Radio Seek Up

21 - Store station (select number)
22 - Store station (cancel)
25 - Store station (store)

30 - Hour (Hex e.g. 0x23 = 11pm)
31 - Minute

33 - Reads back the value written...
34 - Displays temperature (hex value written). 
35 - Go back to digit mode (resets to -- --:--)

36 - Volume control count (8-bit +/-)
37 - Radio: manual tune up (+0.5 MHz) 
38 - Radio: manual tune down (-0.5 MHz) 
39 - Radio: seek down
40 -

41 - Display time

Don't go above 86 - the I2C bus freezes! 
*/

#ifndef _SETPANEL_H
#define _SETPANEL_H

#define DM_VERSION "00.05"

// CPU Temp
#define IT8705_TEMP_REG 0x29
// MoBo Temp
//#define IT8705_TEMP_REG 0x28


// Address on the I2C bus of the AudioDL (OZ268) IC. 
#define AUDIODJ_I2C_ADDRESS 0x49

// Update intervals for temperature display and volume control in milli-seconds. 
#define TEMPERATURE_UPDATE_INTERVAL_MS 500
#define VOLUME_UPDATE_INTERVAL_MS 50
#define I2C_TRANSACTION_DELAY_MS 500


#ifndef FALSE
typedef enum _bool
{
    FALSE=0,
    TRUE
} BOOL;
#endif
    // First digit: 
    // 0x80 of 0x00 makes LED not light up for representative mode. 
    
    // All digits:
    // 0x0f = blank
    // 0x0e = 'C' (2nd Digit = 'D') CD MODE LED
    // 0x0d = 'P' (2nd Digit = '-') HD MODE LED
    // 0x0c = '-' (2nd Digit = 'M') Invalid MODE
    // 0x0b = 'A'                   AM MODE LED 
    // 0x0a = 'F'                   FM MODE LED
    // < 0x0a = Number.i

typedef enum _control
{
	SEEK_DOWN = 0,
	SEEK_UP,
	FINE_TUNE_DOWN,
	FINE_TUNE_UP,
	CHANNEL_PREV,
	CHANNEL_NEXT,
	STORE_START,
	STORE_CANCEL,
	STORE_COMMIT
} CONTROL;

typedef enum _radio_mode
{
    OFF_MODE = 0,
    AM_RADIO,
    FM_RADIO,
    CD_MODE,
    HD_MODE
} RADIO_MODE;

typedef enum _leds
{
    LED_NONE = 0x8f,
    LED_AM = 0x0b,
    LED_FM = 0x0a,
    LED_CD = 0x0e,
    LED_HD = 0x0d,
} LED;


// Writing to Reg 0x07 does nothing, 0x08 = PLAY LED, 0x09 = PAUSE LED.
typedef enum _pp
{
    PP_NONE = 0x07,
    PP_PLAY = 0x08,
    PP_PAUSE = 0x09
} PP;


// Dots are in reg 0x06
    // 0x00 = No Dots.
    // 0x01 = middle dot
    // 0x02 = bottom dot 
    // 0x03 = both dots
typedef enum _dots
{
    DOTS_NONE = 0,
    DOTS_MIDDLE,
    DOTS_BOTTOM,
    DOTS_BOTH
} DOTS;

void digimatrix_Control(int file, CONTROL control); 
void digimatrix_TurnOnRadio(int file, RADIO_MODE radio_mode);
void digimatrix_DisplayNumbers(int file, char digits[6], LED led, PP pp, DOTS dots, BOOL reinit);
void digimatrix_DisplayTemp(int file, int temp);
void digimatrix_UpdateTemp(int file, int temp);

#endif //_SETPANEL_H
