/*
 *  (C) Copyright 2004 Richard Taylor <richard@artaylor.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
	       
#include <sys/io.h>
#include <unistd.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <errno.h>

#include "it8712.h"

#define FAN_ON          	(0)
#define FAN_OFF         	(1)
#define FAN_AUTO        	(2)
#define FAN_INTELLIGENT 	(3)

#define FAN_MAX_PWM		(0x7f)

#define FAN_DEFAULT_MIN 	(30)
#define FAN_DEFAULT_GRADIENT  	(4)
#define FAN_DEFAULT_OFFSET	(30)

int debug = 0;

static double min      = FAN_DEFAULT_MIN;	// The minimum (percentage) that the fans will be set to
static double gradient = FAN_DEFAULT_GRADIENT;	// The rate at which the fan speed increases with temperaure (
static double offset   = FAN_DEFAULT_OFFSET;	// The Temperature at which we should be at minimum speed

static struct option longopts[] = {
	{ "min",         1, 0, 'm' },
	{ "gradient",    1, 0, 'g' },
	{ "offset",      1, 0, 'o' },
    { "auto",        0, 0, 'a' },
    { "intelligent", 0, 0, 'i' },
    { "set",         1, 0, 's' },
    { "read",        0, 0, 'r' },
    { "version",     0, 0, 'v' },
    { "help",        0, 0, 'h' },
	{ "debug",		 0, 0, 'd' },
    { NULL,          0, 0,  0  }
};
														
unsigned char read_byte(unsigned char addr)
{
        outb(addr, 0x295);
        return (inb(0x296));
}

int write_byte (unsigned char addr, unsigned char data)
{
	outb(addr, 0x295);
	outb(data, 0x296);

	if (read_byte(addr) != data)
	{
		printf("Error Writing addr: 0x%02X, wrote: 0x%02X, read: 0x%02X\n", addr, data, read_byte(addr));
		return -1;
	}
    return 0;
}

int set_fan (unsigned char state)
{
    unsigned char speed, temp1, temp2, speed_old = 0;
    double percentage = 0;

    if (ioperm(0x295, 2, 1) == -1)
    {
        printf("Error getting access\n");
        return -1;
    }

    temp1 = read_byte(0x29);
    temp2 = read_byte(0x2b);

    switch (state)
    {
    case FAN_AUTO:
        // Set all fans to Automatic operation
        write_byte(0x15, 0x80); // # Fan1
        write_byte(0x16, 0x80); // # Fan2
        write_byte(0x17, 0x80); // # Fan3 (unused)
	break;

    case FAN_ON:
        write_byte(0x15, 0x7f); // Fan1 ON
	write_byte(0x16, 0x7f); // Fan2 ON
	write_byte(0x17, 0x7f); // Fan3 ON
	break;

    case FAN_OFF:
        write_byte(0x15, 0x10); // Fan1 (almost) OFF
        write_byte(0x16, 0x10); // Fan2 (almost) OFF
        write_byte(0x17, 0x10); // Fan3 (almost) OFF
        break;

    case FAN_INTELLIGENT:
        while (1)
	{
            temp1 = read_byte(0x29);
            temp2 = read_byte(0x2b);
	
            if (temp2 > temp1)
            {
	        // Go with the average of the two temperatures. (we need to cool down the mainboard too)
	        temp1 = (temp1 + temp2) / 2;
            }
            
	    percentage = min + ( (temp1 - offset) * gradient);

            // We can't go more than 100%! 
            if (percentage > 100)
            {
                percentage = 100;
            }

            // Sorry, no slower than min :) 
            if (percentage < min)
            {
                percentage = min;
            }
	
            speed = (unsigned char) ((percentage * (double)FAN_MAX_PWM) / 100);
	
            write_byte(0x15, (speed&0x7f)); // # Fan1
            write_byte(0x16, (speed&0x7f)); // # Fan2

            if (speed != speed_old)
	    {
	       if (debug == 1)
				 printf("New Fan setting is: %d%%\n", (100 * (read_byte(0x15) & 0x7F)) / 0x7f);
		speed_old = speed;
            }

	    sleep(5);
	}
	break;

    default:
        write_byte(0x15, state);
	write_byte(0x16, state);
    	break;
    }

    return 0;
}

void usage(const char *argv0)
{
	printf(
"Usage: %s [OPTIONS]\n"
"\n"
"  The min, gradient and offset options must come first, otherwise they will be ignored!\n"
"  All temperatures are in Deg C, all fan speeds are in percentages.\n\n"
"  -m, --min                  Set the minimum fan speed\n"
"  -o, --offset               The temperature for minimum fan speed\n"
"  -g, --gradient             Set the rate at which speed increases with temperature\n"
"                                 e.g. grad =     (100 - min)    \n"
"                                             -------------------\n"
"                                             (max_temp - offset)\n"
"\n"
"  -r, --read                 Read the current fan setting\n"
"  -s, --set=VALUE            Set the current fan setting\n"
"  -a, --auto                 Set to H/W monitoring (as in BIOS)\n"
"  -i, --intelligent          Set the fan based on current temperature\n"
"\n"
"  -d, --debug				  Display current fan speed at 5s intervals for testing settings\n"
"  -v, --version              Print version\n"
"  -h, --help                 This help\n"
"\n", argv0);
}

int main(int argc, char **argv)
{
    int ch, longindex = 0;

    if (argc < 2) 
    {
        usage(argv[0]);
        exit(1);
    }

    if (ioperm(0x295, 2, 1) == -1)
    {
        printf("Error getting access - are you root?\n");
        return -1;
    }

    if (it8712_open())
    {
        printf("No IT8712 / IT8705 IC Found\n");
	return -1;
    }
    
    for (;;) 
    {
        if ((ch = getopt_long(argc, argv, "m:g:o:rs:aivhd", longopts, &longindex)) == -1)
	{
	    break;
	}

        switch (ch) 
        {
        case 'm':
            printf("\nSetting min to: %.02f%%\n", min = (atof(optarg)<0)?FAN_DEFAULT_MIN:(atof(optarg)>100)?FAN_DEFAULT_MIN:atof(optarg) );
            break;

        case 'g':
            printf("\nSetting gradient to: %.02f\n", gradient = (atof(optarg)<0)?FAN_DEFAULT_GRADIENT:(atof(optarg)>100)?FAN_DEFAULT_GRADIENT:atof(optarg) );
            break;

        case 'o':
            printf("\nSetting offset to: %.02fDegC\n", offset = (atof(optarg)<18)?FAN_DEFAULT_OFFSET:(atof(optarg)>55)?FAN_DEFAULT_OFFSET:atof(optarg) );
            break;
				
        case 'r':
            printf("\nCurrent Fan setting is: %d%%\n", (100 * (read_byte(0x15) & 0x7F)) / 0x7f);
            break;

        case 's':
            printf("\nSetting Fan to: %d%%\n", atoi(optarg));
	    set_fan( ( ((atoi(optarg)<5)?6:atoi(optarg)) * 0x7f) / 100);
            break;


        case 'a':
            printf("\nSetting Fan to AUTO\n");
            set_fan(FAN_AUTO);
            break;

        case 'i':
            printf("\nRunning in intelligent mode (update = 5s)\n");
            set_fan(FAN_INTELLIGENT);
            break;

        case 'v':
            printf("ASUS DigiMatrix Fan Controller Version 0.2 \n");
            break;

        case 'h':
            usage(argv[0]);
            break;

		case 'd':
			debug=1;
			break;

	default:
	    break;
        }
    }
    return 0;
}


