; CHIP3ASM.ASM
;
; This file contains assembler subroutines for Play3voi.
;

;
; Establish the segment ordering.  Code comes first.
;
_TEXT	SEGMENT PUBLIC 'CODE'
_TEXT	ENDS

_DATA 	SEGMENT PARA PUBLIC 'DATA'
_DATA 	ENDS

CONST	SEGMENT WORD PUBLIC 'CONST'
CONST	ENDS

_BSS	SEGMENT WORD PUBLIC 'BSS'
_BSS	ENDS

STACK 	SEGMENT PARA STACK 'STACK'
STACK 	ENDS

DGROUP	GROUP	_DATA,CONST,_BSS,STACK

;
; Local data.
;
_DATA 	SEGMENT PARA PUBLIC 'DATA'
;
; Global variables defined in Chip3voi.c and Play3voi.c.
;
EXTRN	_IOport:WORD		; SN76496 I/O port
EXTRN	_SampRate:WORD		; sampling rate of sound file (Hz)
EXTRN	_SoundBuf:DWORD		; far pointer to sample buffer
EXTRN	_AmplifyTable:WORD	; near pointer to amplification table
EXTRN	_PortTable:WORD		; near pointer to I/O port values table
EXTRN	_SoundLen:DWORD		; 32-bit number of samples in buffer
;
; Local data.
;
Pitclock	DD	1193181	; PIT clock rate
TESTSAMPLES	EQU	200	; number of samples to test speed
		;
		; Buffer full of zero samples for testing speed.
		;
TestBuf		DB	TESTSAMPLES DUP (0)
IntMask		DB	0	; default interrupt mask
Channel0Count	DW	0	; channel 0 count for speed test
Int8Happened	DB	0	; 1 if timer interrupt during speed test
_DATA 	ENDS

_TEXT		SEGMENT	PUBLIC 'CODE'
Int08Default	DD	0	; default Int 8 vector
;
; Replacement Int 8 handler.  Just sets a flag and returns.
;
Int08Hdlr	PROC	FAR
	PUSH	AX
	MOV	Int8Happened,1
	MOV	AL,20h
	OUT	20h,AL
	POP	AX
	IRET
Int08Hdlr	ENDP
;
; Replace_Int8() function.  Saves the default vector for Int 8 and sets the
; vector to the handler above.
;
Replace_Int8	PROC	NEAR
	PUSH	AX
	PUSH	BX
	PUSH	DS
	;
	; Get and save default vector.
	;
	XOR	AX,AX
	MOV	DS,AX
	MOV	BX,4*8
	MOV	AX,[BX]
	MOV	CS:Int08Default,AX
	MOV	AX,[BX+2]
	MOV	CS:Int08Default+2,AX
	;
	; Set new vector.
	;
	MOV	WORD PTR [BX],OFFSET Int08Hdlr
	MOV	[BX+2],CS
	POP	DS
	POP	BX
	POP	AX
	RET
Replace_Int8	ENDP
;
; Restore_Int8() function.  Restores the vector for Int 8 to the system
; default.
;
Restore_Int8	PROC	NEAR
	PUSH	AX
	PUSH	BX
	PUSH	DS
	;
	; Reset old vector.
	;
	XOR	AX,AX
	MOV	DS,AX
	MOV	BX,4*8
	MOV	AX,CS:Int08Default
	MOV	[BX],AX
	MOV	AX,CS:Int08Default+2
	MOV	[BX+2],AX
	POP	DS
	POP	BX
	POP	AX
	RET
Restore_Int8	ENDP
;
; Detect_XT() function.  This function tests for the presence of the timer
; channel 2 out bit at PPI port B (port 62h).  That bit is needed to time
; the intervals between samples.  XT's have it and AT's don't.  This function
; returns 0 if the bit is present and -1 if not.
;
_Detect_XT	PROC	NEAR
	PUBLIC	_Detect_XT
	;
	; Disable interrupts.
	;
	CLI
	;
	; Program PPI to clear bit 1 and set bit 0 (enable timer channel
	; 2 out, disable gate to speaker).
	;
	IN	AL,61h
	JMP	$+2
	AND	AL,0FDh
	OR	AL,1
	OUT	61h,AL
	;
	; Program timer channel 2 (sound/extra channel).  Use a count of
	; 500 - about 2386Hz, slow enough not to strain the slowest machine.
	;
	MOV	AL,0B6h	; timer channel 2, mode 3, LSB then MSB, binary
	OUT	43h,AL
	JMP	$+2
	MOV	AX,500	; put clock count in AX
	OUT	42h,AL	; send low byte to timer chip
	JMP	$+2
	MOV	AL,AH	; get high byte of clock count
	OUT	42h,AL	; send to timer chip
	JMP	$+2
	;
	; Set timer channel 0 to square wave mode at the default rate of
	; 18.2 Hz.  This shouldn't really be needed, but it does no harm
	; to make sure.
	;
	MOV	AL,36h	; timer channel 0, mode 3, LSB then MSB, binary
	OUT	43h,AL
	JMP	$+2
	MOV	AL,0
	OUT	40h,AL
	JMP	$+2
	OUT	40h,AL
	JMP	$+2
	;
	; Hook the timer interrupt.
	;
	CALL	Replace_Int8
	;
	; Turn off the "interrupt happened" flag.
	;
	MOV	Int8Happened,0
	;
	; Enable interrupts.
	;
	STI
	;
	; Wait until a timer interrupt occurs.
	;
DETECT_XT_LOOP1:
	CMP	Int8Happened,0
	JE	DETECT_XT_LOOP1
	;
	; Turn off the "interrupt happened" flag again.
	;
	MOV	Int8Happened,0
	;
	; Wait until either (1) we detect a rising edge on timer channel 2
	; out, or (2) another timer interrupt.  Even at only 2386Hz, (1)
	; is guaranteed to happen first on an XT, while on an AT, (1) will
	; never happen, so (2) occurs first.
	;
DETECT_XT_LOOP2:	; wait for timer channel 2 to go low (or interrupt)
	CMP	Int8Happened,1
	JE	DETECT_XT_DONE
	IN	AL,62h
	TEST	AL,20h
	JNZ	DETECT_XT_LOOP2
DETECT_XT_LOOP3:	; ... and high again (or interrupt)
	CMP	Int8Happened,1
	JE	DETECT_XT_DONE
	IN	AL,62h
	TEST	AL,20h
	JZ	DETECT_XT_LOOP3
	;
	; Disable interrupts.
	;
DETECT_XT_DONE:
	CLI
	;
	; Unhook the timer interrupt.
	;
	CALL	Restore_Int8
	;
	; Enable interrupts.
	;
	STI
	;
	; Set return code.
	;
	MOV	AL,Int8Happened		; AX=0 => XT; AX=1 => AT
	CBW
	NEG	AX			; AX=0 => XT; AX=-1 => AT
	RET
_Detect_XT	ENDP
;
; Test_Speed() function.  This function takes a sampling rate and returns
; 1 (TRUE) if the rate is playable on the machine or 0 (FALSE) if it is
; not.
;
_Test_Speed	PROC	NEAR
	PUBLIC	_Test_Speed
	PUSH	BP		; save caller's BP
	MOV	BP,SP		; rate is at BP+4
	;
	; Save other needed registers.
	;
	PUSH	SI
	PUSH	DI
	PUSH	ES
	;
	; Disable interrupts.
	;
	CLI
	;
	; Program PPI to clear bit 1 and set bit 0 (enable timer channel
	; 2 out, disable gate to speaker).
	;
	IN	AL,61h
	JMP	$+2
	AND	AL,0FDh
	OR	AL,1
	OUT	61h,AL
	;
	; Turn all SN76496 tone and noise channels off.
	;
	MOV	DX,_IOport
	MOV	AL,9Fh
	OUT	DX,AL
	MOV	AL,0BFh
	OUT	DX,AL
	MOV	AL,0DFh
	OUT	DX,AL
	MOV	AL,0FFh
	OUT	DX,AL
	;
	; Compute timer clocks per sample.
	;
	MOV	AX,WORD PTR Pitclock
	MOV	DX,WORD PTR Pitclock+2
	MOV	BX,[BP+4]
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	DX,AX	; DX = clocks per sample
	;
	; Program timer channel 2 (sound/extra channel).
	;
	MOV	AL,0B6h	; timer channel 2, mode 3, LSB then MSB, binary
	OUT	43h,AL
	JMP	$+2
	MOV	AL,DL	; get low byte of clock count
	OUT	42h,AL	; send to timer chip
	JMP	$+2
	MOV	AL,DH	; get high byte of clock count
	OUT	42h,AL	; send to timer chip
	JMP	$+2
	;
	; Program timer channel 0 (timer tick interrupt) for interrupt on
	; terminal count mode, but don't load a count yet.  Interrupt on
	; terminal count will cause one (and only one) interrupt when the
	; count reaches 0.  If the count reaches 0 and interrupt occurs
	; before we get to the end of the test, the sampling rate is too
	; high for the system.  We add 10% to the specified number of test
	; samples here to allow some leeway.
	;
	MOV	AX,(TESTSAMPLES*11)/10	; compute count and save
	MUL	DX
	MOV	Channel0Count,AX
	MOV	AL,30h	; timer channel 0, mode 0, LSB then MSB, binary
	OUT	43h,AL
	JMP	$+2
	;
	; Save the default interrupt mask, and mask out all interrupts but
	; the timer tick.
	;
	IN	AL,21h
	JMP	$+2
	MOV	IntMask,AL
	MOV	AL,0FEh
	OUT	21h,AL
	JMP	$+2
	;
	; Hook the timer interrupt.
	;
	CALL	Replace_Int8
	;
	; In the following, we have:
	;   ES:DI -> sound samples
	;   DS:BX -> amplification table
	;   DS:BP -> I/O port values lookup table
	;   CX = loop count
	;   DX = SN76496 I/O port number
	;   DF set for incrementing
	;
	MOV	DI,DS
	MOV	ES,DI
	MOV	DI,OFFSET TestBuf
	MOV	BX,_AmplifyTable
	MOV	BP,_PortTable
	MOV	DX,_IOport
	MOV	CX,TESTSAMPLES
	CLD
	;
	; Turn off the "interrupt happened" flag.
	;
	MOV	Int8Happened,0
	;
	; Load a count now into timer channel 0 to start counting.
	;
	MOV	AX,Channel0Count
	OUT	40h,AL
	JMP	$+2
	MOV	AL,AH
	OUT	40h,AL
	JMP	$+2
	;
	; Enable the timer interrupt.
	;
	STI
	;
	; Jump to loop start.
	;
	JMP	TEST_LOOP
	;
	; Adjust ES on segment wrap.  It's here to avoid unnecessary jumps.
	;
TEST_LOOP_SEGWRAP:
	MOV	SI,ES		; start new segment
	ADD	SI,1000h
	MOV	ES,SI
	JMP	TEST_LOOP_ADJUST
	;
	; Second loop:  do what remains.
	;
TEST_LOOP:
	MOV	AL,ES:[DI]		; get a sample
	INC	DI			; go to next
	JZ	TEST_LOOP_SEGWRAP	; if at end of segment, go adjust ES
TEST_LOOP_ADJUST:
	XLATB			; amplify sample
	XOR	AH,AH		; convert to word
	SHL	AX,1		; AX = offset in _PortTable for sample
	SHL	AX,1
	ADD	AX,BP		; DS:SI -> I/O port values for sample
	MOV	SI,AX
TEST_LOOP_WAIT1:		; wait for timer channel 2 to go low
	IN	AL,62h
	TEST	AL,20h
	JNZ	TEST_LOOP_WAIT1
TEST_LOOP_WAIT2:		; ... and high again
	IN	AL,62h
	TEST	AL,20h
	JZ	TEST_LOOP_WAIT2
	LODSW			; write to port
	OUT	DX,AL
	MOV	AL,AH
	OUT	DX,AL
	LODSB
	OUT	DX,AL
	LOOP	TEST_LOOP
	;
	; Disable interrupts.
	;
	CLI
	;
	; Restore the default interrupt mask.
	;
	MOV	AL,IntMask
	OUT	21h,AL
	JMP	$+2
	;
	; Unhook the timer interrupt.
	;
	CALL	Restore_Int8
	;
	; Reset timer channel 0 to square wave mode at the default rate of
	; 18.2 Hz.
	;
	MOV	AL,36h	; timer channel 0, mode 3, LSB then MSB, binary
	OUT	43h,AL
	JMP	$+2
	MOV	AL,0
	OUT	40h,AL
	JMP	$+2
	OUT	40h,AL
	JMP	$+2
	;
	; Enable interrupts.
	;
	STI
	;
	; Restore registers.
	;
	POP	ES
	POP	DI
	POP	SI
	;
	; Set return code.
	;
	MOV	AL,Int8Happened
	XOR	AL,1		; NOT Int8Happened == OK (TRUE)
	MOV	AH,0
	POP	BP		; restore caller's BP
	RET
_Test_Speed	ENDP
;
; Play_Internal() function.  This is where the sound is actually played.
; It assumes that all needed global variables have been set.
;
_Play_Internal	PROC	NEAR
	PUBLIC	_Play_Internal
	;
	; Save needed registers.
	;
	PUSH	SI
	PUSH	DI
	PUSH	BP
	PUSH	ES
	;
	; Set sound multiplexer for the 3-voice chip.
	;
	MOV	AX,8003h
	INT	1Ah
	;
	; Disable interrupts.
	;
	CLI
	;
	; Program PPI to clear bit 1 and set bits 0 and 5.
	;
	IN	AL,61h
	JMP	$+2
	AND	AL,0FDh
	OR	AL,21h
	OUT	61h,AL
	;
	; Turn all SN76496 tone and noise channels off.
	;
	MOV	DX,_IOport
	MOV	AL,9Fh
	OUT	DX,AL
	MOV	AL,0BFh
	OUT	DX,AL
	MOV	AL,0DFh
	OUT	DX,AL
	MOV	AL,0FFh
	OUT	DX,AL
	;
	; Reprogram all SN76496 frequency registers, setting the tone
	; frequency to the maximum - 111861Hz, well above the range of human
	; hearing.
	;
	MOV	AL,81h
	OUT	DX,AL
	MOV	AL,0
	OUT	DX,AL
	MOV	AL,0A1h
	OUT	DX,AL
	MOV	AL,0
	OUT	DX,AL
	MOV	AL,0C1h
	OUT	DX,AL
	MOV	AL,0
	OUT	DX,AL
	;
	; Compute timer clocks per sample.
	;
	MOV	AX,WORD PTR Pitclock
	MOV	DX,WORD PTR Pitclock+2
	MOV	BX,_SampRate
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	DX,AX	; DX = clocks per sample
	;
	; Program timer channel 2 (sound/extra channel).
	;
	MOV	AL,0B6h	; timer channel 2, mode 3, LSB then MSB, binary
	OUT	43h,AL
	JMP	$+2
	MOV	AL,DL	; get low byte of clock count
	OUT	42h,AL	; send to timer chip
	JMP	$+2
	MOV	AL,DH	; get high byte of clock count
	OUT	42h,AL	; send to timer chip
	JMP	$+2
	;
	; In the following, we have:
	;   ES:DI -> sound samples
	;   DS:BX -> amplification table
	;   DS:BP -> I/O port values lookup table
	;   DX = SN76496 I/O port number
	;   DF set for incrementing
	;
	LES	DI,DWORD PTR _SoundBuf
	MOV	BX,_AmplifyTable
	MOV	BP,_PortTable
	MOV	DX,_IOport
	CLD
	;
	; Get the first sample in AH.
	;
	MOV	AL,ES:[DI]
	XLATB
	MOV	AH,AL
	;
	; AL = current sample.
	;
	MOV	AL,0
	;
	; While the current sample is not the same as the first sample of
	; the file, play the current sample.
	;
RAMP_UP_LOOP:
	CMP	AL,AH		; are we at the first sample yet?
	JE	RAMP_UP_LPEND	; if so, go to end
	INC	AL		; increment current
	PUSH	AX		; save current, first on stack
	XOR	AH,AH		; get offset in port values table
	SHL	AX,1
	SHL	AX,1
	ADD	AX,BP
	MOV	SI,AX		; DS:SI -> port values for sample
	LODSW			; play the sample
	OUT	DX,AL
	MOV	AL,AH
	OUT	DX,AL
	LODSB
	OUT	DX,AL
	MOV	CX,512		; delay before playing the next sample
RAMP_UP_DELAY:
	LOOP	RAMP_UP_DELAY
	POP	AX		; get back current, first
	JMP	RAMP_UP_LOOP
RAMP_UP_LPEND:
	;
	; Set up for first loop - skip if fewer than 65536 samples.
	;
	MOV	CX,WORD PTR _SoundLen+2
	JCXZ	PLAY_DOLAST
	JMP	PLAY_LOOP1_OUTER
	;
	; Adjust ES on segment wrap.  It's here to avoid unnecessary jumps.
	;
PLAY_LOOP1_SEGWRAP:
	MOV	SI,ES		; start new segment
	ADD	SI,1000h
	MOV	ES,SI
	JMP	PLAY_LOOP1_ADJUST
	;
	; First loop:  do multiples of 65536 samples.
	;
PLAY_LOOP1_OUTER:
	PUSH	CX
	XOR	CX,CX		; do inner loop 65536 times
PLAY_LOOP1_INNER:
	MOV	AL,ES:[DI]		; get a sample
	INC	DI			; go to next
	JZ	PLAY_LOOP1_SEGWRAP	; if at end of segment, go adjust ES
PLAY_LOOP1_ADJUST:
	XLATB			; amplify sample
	XOR	AH,AH		; convert to word
	SHL	AX,1		; AX = offset in _PortTable for sample
	SHL	AX,1
	ADD	AX,BP		; DS:SI -> I/O port values for sample
	MOV	SI,AX
PLAY_LOOP1_WAIT1:		; wait for timer channel 2 to go low
	IN	AL,62h
	TEST	AL,20h
	JNZ	PLAY_LOOP1_WAIT1
PLAY_LOOP1_WAIT2:		; ... and high again
	IN	AL,62h
	TEST	AL,20h
	JZ	PLAY_LOOP1_WAIT2
	LODSW			; write to port
	OUT	DX,AL
	MOV	AL,AH
	OUT	DX,AL
	LODSB
	OUT	DX,AL
	LOOP	PLAY_LOOP1_INNER
	POP	CX
	LOOP	PLAY_LOOP1_OUTER
	;
	; Set up for second loop - skip if all done.
	;
PLAY_DOLAST:
	MOV	CX,WORD PTR _SoundLen
	JCXZ	PLAY_ALLDONE
	JMP	PLAY_LOOP2
	;
	; Adjust ES on segment wrap.  It's here to avoid unnecessary jumps.
	;
PLAY_LOOP2_SEGWRAP:
	MOV	SI,ES		; start new segment
	ADD	SI,1000h
	MOV	ES,SI
	JMP	PLAY_LOOP2_ADJUST
	;
	; Second loop:  do what remains.
	;
PLAY_LOOP2:
	MOV	AL,ES:[DI]		; get a sample
	INC	DI			; go to next
	JZ	PLAY_LOOP2_SEGWRAP	; if at end of segment, go adjust ES
PLAY_LOOP2_ADJUST:
	XLATB			; amplify sample
	XOR	AH,AH		; convert to word
	SHL	AX,1		; AX = offset in _PortTable for sample
	SHL	AX,1
	ADD	AX,BP		; DS:SI -> I/O port values for sample
	MOV	SI,AX
PLAY_LOOP2_WAIT1:		; wait for timer channel 2 to go low
	IN	AL,62h
	TEST	AL,20h
	JNZ	PLAY_LOOP2_WAIT1
PLAY_LOOP2_WAIT2:		; ... and high again
	IN	AL,62h
	TEST	AL,20h
	JZ	PLAY_LOOP2_WAIT2
	LODSW			; write to port
	OUT	DX,AL
	MOV	AL,AH
	OUT	DX,AL
	LODSB
	OUT	DX,AL
	LOOP	PLAY_LOOP2
	;
	; Get the last sample in AL and clear AH.
	;
PLAY_ALLDONE:
	OR	DI,DI
	JNZ	RAMP_DOWN_ESOK
	MOV	SI,ES
	SUB	SI,1000h
	MOV	ES,SI
RAMP_DOWN_ESOK:
	MOV	AL,ES:[DI-1]
	XLATB
	XOR	AH,AH
	;
	; While the current sample is not zero, play the current sample.
	;
RAMP_DOWN_LOOP:
	CMP	AL,AH		; are we at zero yet?
	JE	RAMP_DOWN_LPEND	; if so, go to end
	DEC	AL		; decrement current
	PUSH	AX		; save current on stack
	SHL	AX,1		; get offset in port values table
	SHL	AX,1
	ADD	AX,BP
	MOV	SI,AX		; DS:SI -> port values for sample
	LODSW			; play the sample
	OUT	DX,AL
	MOV	AL,AH
	OUT	DX,AL
	LODSB
	OUT	DX,AL
	MOV	CX,512		; delay before playing the next sample
RAMP_DOWN_DELAY:
	LOOP	RAMP_DOWN_DELAY
	POP	AX		; get back current
	JMP	RAMP_DOWN_LOOP
RAMP_DOWN_LPEND:
	;
	; Reenable interrupts.
	;
	STI
	;
	; Set sound multiplexer back to 8253 channel 2 (gate to speaker
	; still disabled).
	;
	MOV	AX,8000h
	INT	1Ah
	;
	; Restore registers.
	;
	POP	ES
	POP	BP
	POP	DI
	POP	SI
	RET
_Play_Internal	ENDP
_TEXT		ENDS
