// #########################################################
// === File #8 of 14 : pfmeter.cpp =========================
// ------------------- C++ code : BEGIN --------------------

// ==============================================================
//
//  Copyright (c) 2002 by Alex Vinokur.  This work and all works
//  derived from it may be copied and modified without any
//  restrictions other than that a copy of this copyright notice
//  must be included in any copy of this work or any derived work.
//
// ==============================================================
// #########################################################
// ## mailto:alexvn@go.to, mailto:alexv@hitechclub.com
// ## http://up.to/alexv, http://go.to/alexv_math
// #########################################################
// ==============================================================


// ##############################################################
//
//  SOFTWARE : C/C++ Program Perfometer
//  FILE     : pfmeter.cpp
//
//  DESCRIPTION :
//         Implementation of perfometer's common (non-template) classes
//
// ##############################################################


//===================
#include "pfmeter2.h"
//===================

//-----------------------------------------
//-
//-  Constants for IO Manipulation	
//-
//-----------------------------------------
static const string	space1	(1, ' ');
static const string	space2	(2, ' ');
static const string	space3	(3, ' ');
static const string	space5	(5, ' ');
static const string	space9	(9, ' ');

static const string	slash2	(2, '/');

static const string	colon1 (1, ':');

static const string	sulamit1 (1, '#');

static const string	pref1	("# ");

static const size_t	size12	= 12;
static const size_t	size25	= 25;
static const size_t	size27	= 27;
static const size_t	size43	= 43;
static const size_t	size45	= 45;
static const size_t	setw16	= 16;

//#########################################
//#
//#  Global (Non-Template) Functions 
//#  	IMPLEMENTATION
//#
//#########################################
//=============================
void MeasureFunc (TypeMeasuredFunc * const func_i)
{
  perf_ins.add_measured_func (func_i);
}

//=============================
void CreateComparisonGroup (const string& group_name_i)
{
  perf_ins.add_comparison_group (group_name_i);
}

//=============================
void CompareFunc(const string& group_name_i, TypeMeasuredFunc* const func_i)
{
  perf_ins.add_compared_func (group_name_i, func_i);
}

//=============================
void SetTotalTests(ulong tests_i)
{
  perf_ins.set_tests (tests_i);
}


//=============================
void SetDiscardingThreshold(float discarding_threshold_i)
{
  perf_ins.set_discarding_threshold (discarding_threshold_i);
}


//=============================
void SetScaleAndTotalIterations (ulong per_calls_i, ulong iterations_i)
{
  perf_ins.set_per_calls (per_calls_i);	// Must be before set_iterations()
  perf_ins.set_iterations (iterations_i);
}

//=============================
void SetDetailedMeasurementReportFlag(bool flag_i)
{
  perf_ins.set_detailed_measurement_report_flag(flag_i);
}

//=============================
void SetMeasurementReportFlag(bool flag_i)
{
  perf_ins.set_measurement_report_flag(flag_i);
}

//=============================
void SetTracingReportFlag(bool flag_i)
{
  perf_ins.set_tracing_report_flag (flag_i);
}

//=============================
bool ParseCommandLine(int argc, char** argv)
{
  return perf_ins.parse_command_line(argc, argv); 
}



//#########################################
//#
//#  Class-1 : ClassResultsFunc
//#  	METHODS
//#
//#########################################

//-----------------------------
size_t		ClassResultsFunc::max_prettyfuncname_len_s(0);
size_t		ClassResultsFunc::max_funcname_len_s(0);
size_t		ClassResultsFunc::max_filename_len_s(0);
size_t		ClassResultsFunc::max_descrmsg_len_s(0);
//-----------------------------

//=============================
bool operator< (
		const ClassResultsFunc&	ins1, 
		const ClassResultsFunc& ins2
		)
//=============================
{
  if (ins1.prettyfuncname_ < ins2.prettyfuncname_) return true;
  if (ins1.prettyfuncname_ > ins2.prettyfuncname_) return false;

  if (ins1.funcname_ < ins2.funcname_) return true;
  if (ins1.funcname_ > ins2.funcname_) return false;

  if (ins1.filename_ < ins2.filename_) return true;
  if (ins1.filename_ > ins2.filename_) return false;

  if (ins1.lineno_ < ins2.lineno_) return true;
  if (ins1.lineno_ > ins2.lineno_) return false;

  if (ins1.descrmsg_ < ins2.descrmsg_) return true;
  if (ins1.descrmsg_ > ins2.descrmsg_) return false;
 
  //----------------------------------------------------
  // IMPORTANT. funcptr_ must not take part in operator<
  //----------------------------------------------------


  return false;

} // operator<


//=============================
bool operator== (
		const ClassResultsFunc&	ins1, 
		const ClassResultsFunc& ins2
		)
//=============================
{
  if (!(ins1.prettyfuncname_ == ins2.prettyfuncname_))	return false;
  if (!(ins1.funcname_ == ins2.funcname_))		return false;
  if (!(ins1.filename_ == ins2.filename_))		return false;
  if (!(ins1.lineno_ == ins2.lineno_))			return false;
  if (!(ins1.descrmsg_ == ins2.descrmsg_))		return false;

  //-----------------------------------------------------
  // IMPORTANT. funcptr_ must not take part in operator==
  //-----------------------------------------------------

  return true;

} // operator==


//=============================
// Constructor-0
ClassResultsFunc::ClassResultsFunc() :
      		funcptr_ (NULL),		
      		prettyfuncname_ (""),		
      		funcname_ (""),		
      		filename_ (""),		
      		lineno_ (0),		
      		descrmsg_ ("")		
//=============================
{
  assert (0);
} // ClassResultsFunc


//=============================
// Constructor-1
ClassResultsFunc::ClassResultsFunc(
      	TypeMeasuredFunc*	const	funcptr_i,		
	const string&			prettyfuncname_i,
	const string&			funcname_i,
	const string&			filename_i,
	size_t				lineno_i,
	const string&			descrmsg_i
	) :
      		funcptr_ (funcptr_i),		
      		prettyfuncname_ (prettyfuncname_i),		
      		funcname_ (funcname_i),		
      		filename_ (filename_i),		
      		lineno_ (lineno_i),		
      		descrmsg_ (descrmsg_i)		
//=============================
{

  max_prettyfuncname_len_s = MAX_VALUE (
				max_prettyfuncname_len_s, 
				prettyfuncname_.size()
				);

  max_funcname_len_s = MAX_VALUE (
				max_funcname_len_s, 
				funcname_.size()
				);

  max_filename_len_s = MAX_VALUE (
				max_filename_len_s, 
				filename_.size()
				);
  max_descrmsg_len_s = MAX_VALUE (
				max_descrmsg_len_s, 
				descrmsg_.size()
				);

} // ClassResultsFunc



//=============================
// Destructor
ClassResultsFunc::~ClassResultsFunc()
//=============================
{
} // ~ClassResultsFunc


//=============================
string ClassResultsFunc::get_prettyfuncname () const
//=============================
{
  return prettyfuncname_;
}


//=============================
string ClassResultsFunc::get_funcname () const
//=============================
{
  return funcname_;
}

//=============================
string ClassResultsFunc::get_filename () const
//=============================
{
  return filename_;
}


//=============================
size_t ClassResultsFunc::get_lineno () const
//=============================
{
  return lineno_;
}


//=============================
string ClassResultsFunc::get_descr () const
//=============================
{
  return descrmsg_;
}

//=============================
TypeMeasuredFunc* ClassResultsFunc::get_funcptr () const
//=============================
{
  return funcptr_;
}

//=============================
string ClassResultsFunc::getstr_func_detailed_descr (
		vector<string>	resource_descr_i,
		count_t		func_serial_no_i, 
		count_t		total_funcs_i, 
		count_t		total_tests_i, 
		bool		detailed_report_flag_i, 
		const string&	msg_i
		) const

//=============================
{
ostringstream osstr;

  osstr	<< endl;

const string word_Resource ("Resource");

#ifdef OPTIMIZATION_LEVEL
  osstr << space3 << OPTIMIZATION_LEVEL << endl;
#endif

  osstr	<< space3
	<< string (size45, '#') 
	<< endl;

  osstr	<< space3
        << pref1
        << string (word_Resource.size(), '-')
	<< endl;

  osstr	<< space3
        << pref1
        << word_Resource
	<< endl;

  osstr	<< space3
        << pref1
        << string (word_Resource.size(), '-')
	<< endl;

  for (count_t i = 0; i < resource_descr_i.size(); i++)
  {
    osstr	<< space3
    		<< pref1
    		<< "\t"
    		<< resource_descr_i[i]
    		<< endl;
 }

  osstr	<< space3
	<< pref1
	<< string (size43, '=') 
	<< endl;

  if (!msg_i.empty())
  {
    osstr << space3 << msg_i << " ---> "; 
  }


  osstr	<< space3
	<< pref1 
	<< setw (setw16) 
	<< string ("Function-" + to_string (func_serial_no_i)).c_str()
	<< " : " 
	<< "of "
        << total_funcs_i
	<< endl;

  osstr	<< space3
	<< pref1 
	<< setw (setw16) 
	<< "PRETTY FUNCTION" 
	<< " : " 
	<< prettyfuncname_ 
	<< endl;

  osstr	<< space3
	<< pref1 
	<< setw (setw16) 
	<< "FUNCTION" 
	<< " : " 
	<< funcname_ 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "FILE" 
	<< " : " 
	<< filename_ 
	<< ", line#" 
	<< lineno_ 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "DESCRIPTION" 
	<< " : " 
	<< descrmsg_ 
	<< endl;


  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "PER-CALLS" 
	<< " : " 
	<< perf_ins.get_scale_per_calls() 
	<< endl;


  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "TOTAL TESTS" 
	<< " : " 
	<< total_tests_i 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< string (9, '-').c_str() 
	<< string (3, '-') 
	<< string (9, '-') 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< "Specific Performance is Runcost expended per " 
	<< perf_ins.get_scale_per_calls() 
	<< " iterations"
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< string (9, '-').c_str() 
	<< string (3, '-') 
	<< string (9, '-') 
	<< endl;

  osstr	<< space3
	<< pref1
	<< space5;

  if (detailed_report_flag_i)
  { 
    osstr << "Detailed measurement report";
  }
  else
  {
    osstr << "Brief measurement report";
  }
  osstr	<< endl;

  osstr	<< space3
	<< string (size45, '#') 
	<< endl;

 return osstr.str();   

} // ClassResultsFunc::getstr_func_detailed_descr



//=============================
size_t ClassResultsFunc::get_max_prettyfuncname_len_S () 
//=============================
{
  return max_prettyfuncname_len_s;
}


//=============================
size_t ClassResultsFunc::get_max_funcname_len_S () 
//=============================
{
  return max_funcname_len_s;
}


//=============================
size_t ClassResultsFunc::get_max_filename_len_S () 
//=============================
{
  return max_filename_len_s;
}



//=============================
size_t ClassResultsFunc::get_max_lineno_len_S () 
//=============================
{
  return 3;
}

//=============================
size_t ClassResultsFunc::get_max_descrmsg_len_S () 
//=============================
{
  return max_descrmsg_len_s;
}



//#########################################
//#
//#  Class-7 : BasicClassMeasurementCollection
//#  	METHODS
//#
//#########################################
//-----------------------------
vector<BasicClassMeasurementCollection*>	BasicClassMeasurementCollection::all_measurement_collections_s;
//-----------------------------

//=============================
// Constructor-0
BasicClassMeasurementCollection::BasicClassMeasurementCollection () 
			:
			typeid_name_(string()),
			total_measured_funcs_(0),
    			resource_description_ (vector<string>())
//=============================
{
}

//=============================
// Copy Constructor
BasicClassMeasurementCollection::BasicClassMeasurementCollection (const BasicClassMeasurementCollection& inst_i) 
//=============================
{
  assert (0);
}

//=============================
// Destructor
BasicClassMeasurementCollection::~BasicClassMeasurementCollection () 
//=============================
{
}



//=============================
void BasicClassMeasurementCollection::add_resource_type_if_need () 
//=============================
{

  if (find_if(
		all_measurement_collections_s.begin(), 
		all_measurement_collections_s.end(), 
		bind1st(
			mem_fun(&BasicClassMeasurementCollection::typeid_predicat), 
			this
			)
		)
      ==
      all_measurement_collections_s.end()
     )
  {
    all_measurement_collections_s.push_back(this);
  }
  
  
  assert (count_if(
		all_measurement_collections_s.begin(), 
		all_measurement_collections_s.end(), 
		bind1st(
			mem_fun(&BasicClassMeasurementCollection::typeid_predicat), 
			this
			)
		)
          == 1
         );

} // BasicClassMeasurementCollection::add_resource_type_if_need



//=============================
bool BasicClassMeasurementCollection::typeid_predicat (
		const BasicClassMeasurementCollection* const p_inst_i
		) const 
//=============================
{
  assert (p_inst_i);  
  return (typeid_name_ == p_inst_i->typeid_name_);
}



//=============================
size_t BasicClassMeasurementCollection::get_total_resource_types_s ()
//=============================
{
  return all_measurement_collections_s.size();
}


//=============================
void BasicClassMeasurementCollection::show_list_of_resources_s ()
//=============================
{
const char fill0 = '%';
const char fill1 = '=';
ostringstream osstr;
  
  cout << endl;
  cout << endl;
  cout << endl;
  cout << "\t" << string (42, fill1) << endl;
  cout << "\t" << string (2, fill0) << string (40, fill1) << endl;
  cout << "\t" << string (2, fill0) << endl;

  cout << "\t" << string (2, fill0) << space2 
       << "Total Measured Resources : " 
       << all_measurement_collections_s.size() 
       << endl;

  for (count_t i = 0; i < all_measurement_collections_s.size(); i++)
  {


    cout << "\t" 
         << string (2, fill0) 
         << space1 
         << endl; 

    cout << "\t" 
         << string (2, fill0) 
         << space2 
         << string (10, '-')
         << space1 
         << "Resource#"
         << (i + 1)
         << space1 
         << string (10, '-')
         << endl; 

    for (
	count_t j = 0; 
	        j < all_measurement_collections_s[i]->resource_description_.size(); 
	        j++
	        )
    {
      cout << "\t" 
           << string (2, fill0) 
           << space1; 
      cout << all_measurement_collections_s[i]->resource_description_[j];
      cout << endl;

    }

    cout << "\t" 
         << string (2, fill0) 
         << space1
         << "---> Total measured functions : "
         << all_measurement_collections_s[i]->total_measured_funcs_
         << endl; 
  }
  cout << "\t" << string (2, fill0) << "\t" << string(osstr.str().size(), '-') << endl;

  cout << "\t" << string (2, fill0) << endl;
  cout << "\t" << string (2, fill0) << string (40, fill1) << endl;
  cout << "\t" << string (42, fill1) << endl;
  cout << endl;


} // BasicClassMeasurementCollection::show_list_of_resources_s



//#########################################
//#
//#  Class-9 : ClassManager	
//#  	METHODS
//#
//#########################################

//-----------------------------
ClassManager		perf_ins;
count_t			ClassManager::inst_counter_s(0);
TypeMeasuredFunc*	ClassManager::cur_func_ptr_s(NULL);

//-----------------------------

//=============================
// Constructor-0
ClassManager::ClassManager ()
//=============================
{

  //---------------------
  show_logo ("START");
  //---------------------

  inst_counter_s++;
  assert (inst_counter_s == 1);
  //-----------------------
  set_measurement_report_flag ();
  set_detailed_measurement_report_flag ();
  set_tracing_report_flag ();

  set_discarding_threshold (); 
  set_tests (); 
  set_per_calls (); 	// Must be before  set_iterations()
  set_iterations (); 

  assert (scale_per_calls_ > 0);
  assert (total_iterations_ > 0);
  assert (scale_per_calls_ <= total_iterations_);

  assert (total_tests_ > 0);

} // ClassManager::ClassManager


//=============================
// Copy Constructor
ClassManager::ClassManager (const ClassManager& inst_i) 
//=============================
{
  assert (0);
} // ClassManager::ClassManager

//=============================
// Destructor
ClassManager::~ClassManager () 
//=============================
{
  //-----------
  perform_measurement ();
  report_measurement (); 
  delete_it (); 
  show_logo ("FINISH");
} // ClassManager::~ClassManager 



//=============================
void ClassManager::perform_measurement () const
//=============================
{

  for (count_t i = 0; i < total_tests_; i++)
  {
     perform_a_measurement (); 
  }

} // perform_measurement

//=============================
void ClassManager::perform_a_measurement () const
//=============================
{

  /*
  for_each(
	measured_funcs_.begin(), 
	measured_funcs_.end(), 
	self_caller<TypeMeasuredFunc*>
	);
  */


  cur_func_ptr_s = NULL;
  for (count_t cur_func_no = 0; 
	cur_func_no < measured_funcs_.size(); 
	cur_func_no++
	)
  {
    cur_func_ptr_s = measured_funcs_[cur_func_no];
    assert (cur_func_ptr_s);

    //-----------------------------
    measured_funcs_[cur_func_no] ();
    //-----------------------------

    cur_func_ptr_s = NULL;
  }


} // perform_a_measurement


//=============================
void ClassManager::report_measurement () const
//=============================
{

vector<BasicClassMeasurementCollection*> &all_measurement_collections = 
		BasicClassMeasurementCollection::all_measurement_collections_s;

map<string, vector<TypeMeasuredFunc*>, less<string> >::const_iterator	map_iter;

  BasicClassMeasurementCollection::show_list_of_resources_s ();

  for (count_t i = 0;
               i < all_measurement_collections.size();
               i++
      )
  {
   
    if (perf_ins.get_measurement_report_flag ())
    {
      // Measurement Report
      all_measurement_collections[i]->show_resource_detailed_descr(i + 1, "Performance Measurement", false);  
      all_measurement_collections[i]->show_measurement_collection();
    }


    //-----------------------
    // Comparison Report
    all_measurement_collections[i]->show_resource_detailed_descr(i + 1, "Performance Comparison", true);  

    count_t group_no = 0;
    for (
	  map_iter = comparison_groups_.begin();
	  map_iter != comparison_groups_.end();
	  map_iter++
	  )
    {
      group_no++;
      ostringstream osstr1;
      ostringstream osstr2;
      osstr1 << "Comparison Group#"
             << group_no
             << " of "
             << comparison_groups_.size()
             << " : "
             << map_iter->first;

      for (
	   count_t j = 0; 
	           j < all_measurement_collections[i]->resource_description_.size(); 
	           j++
	           )
      {
        osstr2 << "\t\t" 
               << "#" 
               << string (2, ' ') 
               << space1; 
        osstr2 << all_measurement_collections[i]->resource_description_[j];
        osstr2 << endl;
      }

      //-----------------------
      cout << endl;
      cout << endl;

#ifdef OPTIMIZATION_LEVEL
  cout << "\t\t" << OPTIMIZATION_LEVEL << endl;
#endif

      cout << "\t\t" 
	   << "#" 
	   << string (osstr1.str().size() + 1, '=') 
	   << endl;

      cout << "\t\t" 
	   << "#" 
	   << " " 
	   << osstr1.str() 
	   << endl;

      cout << "\t\t" 
	   << "#" 
           << string (osstr1.str().size() + 1, '-')
	   << endl;

      cout << osstr2.str(); 

      cout << "\t\t" 
	   << "#" 
	   << string (osstr1.str().size() + 1, '=') 
	   << endl;
      
      osstr1.str(string());
    
      osstr1 << string (3, ' ') 
             << ": " 
             << string (
		ClassResultsFunc::get_max_funcname_len_S() + 3
		+ 
		ClassResultsFunc::get_max_filename_len_S() + 3
		+ 
		ClassResultsFunc::get_max_lineno_len_S() + 3
		+ 
		ClassResultsFunc::get_max_descrmsg_len_S() + 4
		+
		12
		, '-'); 

      cout << osstr1.str() << endl;
      all_measurement_collections[i]->show_comparison_collection(map_iter->second);
      cout << osstr1.str() << endl;
    }
    cout << endl;
    cout << endl;

  }  // for (count_t i = 0; ...

} // ClassManager::report_measurement


//=============================
void ClassManager::add_measured_func (TypeMeasuredFunc * const func_i)
//=============================
{
  assert (func_i);
  if (find (
	     measured_funcs_.begin(), 
	     measured_funcs_.end(),
	     func_i
	     )
	     == measured_funcs_.end()
     )
  {
    measured_funcs_.push_back(func_i);
  }
} 


//=============================
void ClassManager::add_comparison_group (const string& group_name_i)
//=============================
{
  if (group_name_i.empty())	
  {
    cout << "Unable to add comparison group : group name is empty" << endl;
    return;
  }

  assert (!group_name_i.empty());
  assert (
	  (comparison_groups_.count(group_name_i) == 0)
	  ||
	  (comparison_groups_.count(group_name_i) == 1)
	 );

  if (comparison_groups_.count(group_name_i) == 0)
  {
    comparison_groups_ [group_name_i] = vector<TypeMeasuredFunc*> ();

  }
  assert (comparison_groups_.count(group_name_i) == 1);

} 

//=============================
void ClassManager::add_compared_func (
			const string&		group_name_i, 
			TypeMeasuredFunc*	const func_i
			)
//=============================
{
  if (comparison_groups_.count(group_name_i) == 0)
  {
    add_comparison_group (group_name_i);
  }

  assert (!comparison_groups_.empty());
  assert (comparison_groups_.count(group_name_i) == 1);
  assert (func_i);


  comparison_groups_[group_name_i].push_back(func_i);

  if (find (
	     comparison_groups_[group_name_i].begin(), 
	     comparison_groups_[group_name_i].end(),
	     func_i
	     )
	     == comparison_groups_[group_name_i].end()
     )
  {
    comparison_groups_[group_name_i].push_back(func_i);
  }

  add_measured_func (func_i);
} 


//=============================
void ClassManager::set_tests (ulong tests_i) 
//=============================
{
  total_tests_ = tests_i;
}


//=============================
void ClassManager::set_discarding_threshold (float discarding_threshold_i) 
//=============================
{
  discarding_threshold_ = discarding_threshold_i;
}

//=============================
void ClassManager::set_detailed_measurement_report_flag (bool type_i) 
//=============================
{
  detailed_measurement_report_flag_ = type_i;
}


//=============================
void ClassManager::set_measurement_report_flag (bool type_i) 
//=============================
{
  measurement_report_flag_ = type_i;
} 

//=============================
void ClassManager::set_iterations (ulong iterations_i) 
//=============================
{
  total_iterations_ = iterations_i;
  if (total_iterations_ == 0)
  {
    total_iterations_ = TOTAL_ITERATIONS_PER_TEST_default;
  }

  while (scale_per_calls_ > total_iterations_)
  {
    total_iterations_ *= 10;
  }

  assert (scale_per_calls_ <= total_iterations_);


  total_iterations_ = (total_iterations_/scale_per_calls_)*scale_per_calls_;

  if (iterations_i != total_iterations_)
  {
     string pref0 = "\t ";
     string pref1 = pref0 + ": ";
     size_t the_size = 52;
     ostringstream oss;
     oss << pref0 << string (the_size, '-') << endl;
     oss << pref1 << "\t--- WARNING! ---" << endl;
     oss << pref1 << "Requested number-of-iterations   = " << iterations_i << endl;
     oss << pref1 << "---> Actual number-of-iterations = " << total_iterations_ << endl;
     oss << pref1 << "   Info! scale-per-calls         = " << scale_per_calls_ << endl;
     oss << pref1 << "   Note! Must be : scale-per-calls <= number-of-iterations"<< endl;
     oss << pref1 << "                 : total_iterations_%scale_per_calls_ == 0"<< endl;
     oss << pref0 << string (the_size, '-') << endl;
     notes_.push_back (oss.str());
  }

} // ClassManager::set_iterations



//=============================
void ClassManager::set_per_calls (ulong per_calls_i) 
//=============================
{
  scale_per_calls_ = per_calls_i;
  if (scale_per_calls_ == 0)
  {
    scale_per_calls_ = PER_CALLS_default;
  }
    
} // ClassManager::set_per_calls



//=============================
ulong ClassManager::get_scale_per_calls () const
//=============================
{
  return scale_per_calls_;
} 


//=============================
ulong ClassManager::get_total_tests () const
//=============================
{
  return total_tests_;
} 

//=============================
float ClassManager::get_discarding_threshold ()	const
//=============================
{
  return discarding_threshold_;
}


//=============================
float ClassManager::get_discarding_threshold_S ()
//=============================
{
  assert (inst_counter_s == 1);
  return perf_ins.get_discarding_threshold();
}


//=============================
bool ClassManager::get_tracing_report_flag_S ()
//=============================
{
  assert (inst_counter_s == 1);
  return perf_ins.get_tracing_report_flag();
}



//=============================
ulong ClassManager::get_current_total_iterations () const
//=============================
{
  return total_iterations_;
} 

//=============================
bool ClassManager::get_detailed_measurement_report_flag () const
//=============================
{
  return detailed_measurement_report_flag_;
} 


//=============================
bool ClassManager::get_measurement_report_flag () const
//=============================
{
  return measurement_report_flag_;
} 

//=============================
TypeMeasuredFunc* ClassManager::get_cur_func_ptr () const
//=============================
{
  return cur_func_ptr_s;
} 


//-----------------------------------------
//-
//-  Constants for Parsing command line	
//-
//-----------------------------------------
static const string	help_option_CNS		= "-h";	
static const string	settings_option_CNS	= "-s";	
static const string	version_option_CNS	= "-v";	

static const string	report_option_enabled_CNS	= "+";	
static const string	report_option_disabled_CNS	= "-";	

static const string	full_measuring_report_option_CNS	= "m";	
static const string	detailed_measuring_report_option_CNS	= "d";	
static const string	tracing_report_option_CNS		= "t";	

static const string	set_iterations_option_CNS	= "-i";	
static const string	set_tests_option_CNS		= "-e";	
static const string	set_per_calls_option_CNS	= "-p";	
static const string	set_confidence_threshold_option_CNS	= "-f";	

//=============================
void ClassManager::show_help1(const string& exe_name_i) const
//=============================
{
ostringstream osstr;

  osstr << ""
        << "--- "
        << "USAGE : "
        << exe_name_i
        << " ["
        << help_option_CNS
        << "]"
        << " ["
        << version_option_CNS
        << "]"
        << " ["
        << settings_option_CNS
        << "]"
        << " ["
        << (report_option_disabled_CNS + full_measuring_report_option_CNS)
        << "]"
        << " ["
        << (report_option_enabled_CNS + full_measuring_report_option_CNS)
        << "]"
        << " ["
        << (report_option_disabled_CNS + detailed_measuring_report_option_CNS)
        << "]"
        << " ["
        << (report_option_enabled_CNS + detailed_measuring_report_option_CNS)
        << "]"
        << " ["
        << (report_option_disabled_CNS + tracing_report_option_CNS)
        << "]"
        << " ["
        << (report_option_enabled_CNS + tracing_report_option_CNS)
        << "]"
        << " ["
        << set_iterations_option_CNS
        << " "
        << "<number>"
        << "]"
        << " ["
        << set_tests_option_CNS
        << " "
        << "<number>"
        << "]"
        << " ["
        << set_per_calls_option_CNS
        << " "
        << "<number>"
        << "]"
        << " ["
        << set_confidence_threshold_option_CNS
        << " "
        << "<float-number>"
        << "]";

const string help_msg = osstr.str();
  cout << endl
       << string (help_msg.size(), '-')
       << endl
       << help_msg
       << endl
       << string (help_msg.size(), '-')
       << endl;

} // show_help1


//=============================
void ClassManager::show_help2() const
//=============================
{
ostringstream osstr;
const string shift1 = "   #   ";
const string delim1 = "   : ";
const string text1 = "OPTIONS DESCRIPTION";
const string text2 = " <number>";
const string text_default = "Default";
const string show_delim_CNS = "/";
const string text_enabled = "ENABLED";
const string text_disabled = "DISABLED";

const size_t line1_len = 70;

  osstr.setf (ios::left, ios::adjustfield);

  osstr << ""
        << shift1
        << space9
        << text1
        << endl
        << shift1
        << space9
        << string (text1.size(), '-')
        << endl

        << shift1
        << setw (size12)
        << help_option_CNS.c_str()
        << delim1
        << " get help information"
        << endl

        << shift1
        << setw (size12)
        << version_option_CNS.c_str()
        << delim1
        << " get version information"
        << endl

        << shift1
        << setw (size12)
        << settings_option_CNS.c_str()
        << delim1
        << " get environment variables values"
        << endl


        << shift1
        << setw (size12)
        << (report_option_disabled_CNS + show_delim_CNS + report_option_enabled_CNS + full_measuring_report_option_CNS).c_str()
        << delim1
        << " measurement report "  
        << text_disabled
        << show_delim_CNS
        << text_enabled 
        << " ("
        << text_default
        << " - "
        << text_enabled 
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (report_option_disabled_CNS + show_delim_CNS + report_option_enabled_CNS + detailed_measuring_report_option_CNS).c_str()
        << delim1
        << " detailed measurement report "  
        << text_disabled
        << show_delim_CNS
        << text_enabled 
        << " ("
        << text_default
        << " - "
        << text_disabled 
        << ")"
        << endl

        << shift1
        << setw (size12)
        << ""
        << delim1
        << "   Note! If measurement report " 
        <<  text_disabled 
        << " this option ignored"
        << endl

        << shift1
        << setw (size12)
        << (report_option_disabled_CNS + show_delim_CNS + report_option_enabled_CNS + tracing_report_option_CNS).c_str()
        << delim1
        << " tracing report "  
        << text_disabled
        << show_delim_CNS
        << text_enabled 
        << " ("
        << text_default
        << " - "
        << text_disabled 
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (set_iterations_option_CNS + text2).c_str()
        << delim1
        << " set number of iterations"
        << " ("
        << text_default
        << " = "
        << TOTAL_ITERATIONS_PER_TEST_default
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (set_confidence_threshold_option_CNS + text2).c_str()
        << delim1
        << " set confidence threshold"
        << " ("
        << text_default
        << " = "
        << DISCARDING_THRESHOLD_default
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (set_tests_option_CNS + text2).c_str()
        << delim1
        << " set number of tests"
        << " ("
        << text_default
        << " = "
        << TOTAL_TESTS_default
        << ")"
        << endl

        << shift1
        << setw (size12)
        << (set_per_calls_option_CNS + text2).c_str()
        << delim1
        << " set scale per calls"
        << " ("
        << text_default
        << " = "
        << PER_CALLS_default
        << ")"
        << endl;

const string help_msg = osstr.str();
  cout << endl
       << string (line1_len, '=')
       << endl
       << help_msg
       << string (line1_len, '=')
       << endl;

} // show_help2


//=============================
void ClassManager::show_settings(string msg_i) const
//=============================
{
ostringstream osstr;

const string shift_1 = "   *   ";
const string delim_1 = "   : ";
string text_1 = "ENVIRONMENT VARIABLES";
  text_1 += " ";  
  text_1 += "(";  
  text_1 += msg_i;  
  text_1 += ")";  

const string msg_info1 = "Measurement Report";
const string msg_info2 = "Detailed Measurement Report";
const string msg_info3 = "Tracing Report";

const string msg_info4 = "Total Iteration";
const string msg_info5 = "Total Tests";
const string msg_info6 = "Scale per Calls Value";
const string msg_info7 = "Confidence Threshold";

const string msg_enable = "Enable";
const string msg_disable = "Disable";

const size_t line_1_len = 70;

  osstr.setf (ios::left, ios::adjustfield);

  osstr << ""
        << shift_1
        << space9
        << text_1
        << endl
        << shift_1
        << space9
        << string (text_1.size(), '-')
        << endl

        << shift_1
        << setw (size27)
        << msg_info1.c_str()
        << delim_1
        << (measurement_report_flag_ ? msg_enable : msg_disable)
        << endl;

        
        if (measurement_report_flag_)
        {
          osstr << ""
                << shift_1
                << setw (size27)
                << msg_info2.c_str()
                << delim_1
                << (detailed_measurement_report_flag_ ? msg_enable : msg_disable)
                << endl;
        }

  osstr << ""
        << shift_1
        << setw (size27)
        << msg_info3.c_str()
        << delim_1
        << (tracing_report_flag_ ? msg_enable : msg_disable)
        << endl

        << shift_1
        << setw (size27)
        << msg_info4.c_str()
        << delim_1
        << total_iterations_
        << endl

        << shift_1
        << setw (size27)
        << msg_info5.c_str()
        << delim_1
        << total_tests_
        << endl

        << shift_1
        << setw (size27)
        << msg_info6.c_str()
        << delim_1
        << scale_per_calls_
        << endl

        << shift_1
        << setw (size27)
        << msg_info7.c_str()
        << delim_1
        << discarding_threshold_
        << endl;

const string help_msg = osstr.str();
  cout << endl
       << string (line_1_len, '=')
       << endl
       << help_msg
       << string (line_1_len, '=')
       << endl
       << endl
       << endl;

} // show_settings




//=============================
void ClassManager::show_notes(string msg_i) const
//=============================
{
  cout << endl;
  if (!msg_i.empty())
  {
     cout << "\t" << space3 << msg_i << endl;
     cout << "\t" << string (space3.size() + msg_i.size(), '-') << endl;
  }

  for (count_t i = 0; i < notes_.size(); i++)
  {
    cout << notes_[i] << endl;
  }
  cout << endl;

} // show_notes



//=============================
bool ClassManager::parse_command_line (int argc, char** argv)
//=============================
{
bool ret_value = false;
  assert (command_line_.empty());

  read_command_line (argc, argv);

  ret_value |= parse1_command_line();
  ret_value |= parse5_command_line();
  ret_value |= parse4_command_line();

  //-----------------------------
  //show_settings("Default Values");
  //-----------------------------

  ret_value |= parse2_command_line();
  ret_value |= parse3_command_line();

  //-----------------------------
  show_command_line();
  show_notes ();
  show_settings ("Actual Values");
  //-----------------------------

  return ret_value;
} // ClassManager::parse_command_line()


//=============================
bool ClassManager::parse1_command_line ()
//=============================
{
bool	ret_value = false;
  
 ret_value = (find (
			command_line_.begin(), 
			command_line_.end(), 
			help_option_CNS
		) != command_line_.end()
		);

  if (ret_value)
  {
    show_help1(command_line_[0]);
    show_help2();
  }

  return ret_value;

} // ClassManager::parse1_command_line()



//=============================
bool ClassManager::parse2_command_line ()
//=============================
{
  
  for (count_t i = 0; i < command_line_.size(); i++)
  {
    //--------------------------------
    if (command_line_[i] == (report_option_disabled_CNS + full_measuring_report_option_CNS))
    {
      set_measurement_report_flag (false);
    }

    //--------------------------------
    if (command_line_[i] == (report_option_enabled_CNS + full_measuring_report_option_CNS))
    {
      set_measurement_report_flag (true);
    }


    //--------------------------------
    if (command_line_[i] == (report_option_disabled_CNS + detailed_measuring_report_option_CNS))
    {
      set_detailed_measurement_report_flag (false);
    }

    //--------------------------------
    if (command_line_[i] == (report_option_enabled_CNS + detailed_measuring_report_option_CNS))
    {
      set_detailed_measurement_report_flag (true);
    }


    //--------------------------------
    if (command_line_[i] == (report_option_disabled_CNS + tracing_report_option_CNS))
    {
      set_tracing_report_flag (false);
    }

    //--------------------------------
    if (command_line_[i] == (report_option_enabled_CNS + tracing_report_option_CNS))
    {
      set_tracing_report_flag (true);
    }


    //--------------------------------
    if (command_line_[i] == set_per_calls_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        ulong entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_per_calls (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        cout << "---> Desired Number of Scale-Per-Calls Value is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_per_calls_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_scale_per_calls () 
             << endl;
      }
    }  // if (command_line_[i] == set_per_calls_option_CNS)



    //--------------------------------
    if (command_line_[i] == set_tests_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        ulong entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_tests (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        cout << "---> Desired Number of Tests is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_tests_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_total_tests ()
             << endl;
      }

    } // if (command_line_[i] == set_tests_option_CNS)

    //--------------------------------
    if (command_line_[i] == set_confidence_threshold_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        float entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_discarding_threshold (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        cout << "---> Desired Number of Tests is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_confidence_threshold_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_discarding_threshold ()
             << endl;
      }


    } // if (command_line_[i] == set_confidence_threshold_option_CNS)


  } // for (count_t i = 0; i < command_line_.size(); i++)


  return false;
} // ClassManager::parse2_command_line()



//=============================
bool ClassManager::parse3_command_line ()
//=============================
{
  
  for (count_t i = 0; i < command_line_.size(); i++)
  {
    //--------------------------------
    if (command_line_[i] == set_iterations_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        ulong entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_iterations (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        cout << "---> Desired Number of Iterations is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_iterations_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_current_total_iterations ()
             << endl;
      }
    } // if (command_line_[i] == set_iterations_option_CNS)

  } // for (count_t i = 0; i < command_line_.size(); i++)


  return false;
} // ClassManager::parse3_command_line()


//=============================
bool ClassManager::parse4_command_line () const
//=============================
{
bool	ret_value = false;
  
 ret_value = (find (
			command_line_.begin(), 
			command_line_.end(), 
			settings_option_CNS
		) != command_line_.end()
		);

  return ret_value;

} // ClassManager::parse4_command_line()


//=============================
bool ClassManager::parse5_command_line () const
//=============================
{
bool	ret_value = false;
  
 ret_value = (find (
			command_line_.begin(), 
			command_line_.end(), 
			version_option_CNS
		) != command_line_.end()
		);

  if (ret_value)
  {
    show_version();
  }

  return ret_value;

} // ClassManager::parse5_command_line()



//=============================
void ClassManager::show_command_line()
//=============================
{
  cout << endl;
  cout << endl;

#ifdef OPTIMIZATION_LEVEL
  cout << "\t---> COMPILATION       : " << OPTIMIZATION_LEVEL << endl;
#endif

  cout << "\t---> YOUR COMMAND LINE : ";
  copy (
	command_line_.begin(), 
	command_line_.end(), 
	ostream_iterator<string> (cout, " ")
	);

  cout << endl;
  cout << endl;
} // ClassManager::show_command_line



//=============================
void ClassManager::set_tracing_report_flag (bool format_i)
//=============================
{
  tracing_report_flag_ = format_i;
} 

//=============================
bool ClassManager::get_tracing_report_flag () const
//=============================
{
  return tracing_report_flag_;
} 


//=============================
void ClassManager::read_command_line (int argc, char** argv)
//=============================
{
  assert (command_line_.empty());
  for (int i = 0; i < argc; i++)
  {
    command_line_.push_back (argv[i]);
  }

const string	slash ("/\\");
const string	the_exe_name (command_line_[0]);
string::size_type	pos = the_exe_name.find_last_of (slash);

  if (!(pos == string::npos))
  {
    command_line_[0] = the_exe_name.substr(pos + 1);
  }


} 


//=============================
void ClassManager::show_logo (string msg_i)
//=============================
{
time_t timer;
  timer	= time(NULL);

const string pref1 ("\t#");
const string pref2 = pref1 + " ";
const string logo1 ("C/C++ Program Perfometer");
const string logo2 ("  Alex Vinokur");
const string logo3 ("  http://up.to/alexvn");
//const string logo4 = string (6, ' ') + string (sw_version);
const string logo4 = string ("  --> [") + string (sw_version) + string ("]");

size_t the_width = 0;
  the_width = MAX_VALUE (the_width, logo1.size());
  the_width = MAX_VALUE (the_width, logo2.size());
  the_width = MAX_VALUE (the_width, logo3.size());
  the_width = MAX_VALUE (the_width, logo4.size());

  cout << endl;
  cout << pref1 << string (the_width + 1, '=') << endl;
  cout << pref2 << logo1 << endl;
  cout << pref2 << logo4 << endl;
  cout << pref2 << logo2 << endl;
  cout << pref2 << logo3 << endl;
  cout << pref1 << string (the_width + 1, '-') << endl;
  cout << pref2 << msg_i << endl;
  cout << pref2 << asctime (localtime(&timer));
  cout << pref1 << string (the_width + 1, '=') << endl;
  cout << endl;
} // show_logo



//=============================
void ClassManager::show_version(string msg_i) const
//=============================
{
const string	pref1 ("\t===> ");
size_t		size1 = 0;		
  cout << endl;
  if (!msg_i.empty())
  {
     cout << "\t" << space3 << msg_i << endl;
     cout << "\t" << string (space3.size() + msg_i.size(), '-') << endl;
  }
  
  size1 = MAX_VALUE (size1, string (sw_version).size());
  size1 = MAX_VALUE (size1, string (sw_machine).size());
  size1 = MAX_VALUE (size1, string (sw_compiler).size());

  cout << pref1 << string (size1, '#') << endl;
  cout << pref1 << sw_version << endl;
  cout << pref1 << sw_machine << endl;
  cout << pref1 << sw_compiler << endl;
  cout << pref1 << string (size1, '#') << endl;

  cout << endl;
  cout << endl;

} // show_notes




//=============================
void ClassManager::delete_it ()
//=============================
{
vector<BasicClassMeasurementCollection*> &all_that = 
		BasicClassMeasurementCollection::all_measurement_collections_s;

  for (count_t i = 0; i < all_that.size(); i++)
  {
    delete all_that[i];
  }
  all_that.clear();
  assert (all_that.empty());
}

//#######################################
//############# END OF FILE #############
//#######################################

// ------------------- C++ code : END ----------------------
// === File #8 of 14 : pfmeter.cpp =========================
