#
# webcalng - a web based calendar program.
# Copyright 2003 - webcalng software solutions
#

#
# This file contains perl subroutines used by the main webcal application.
# They are placed in a module so that any of them can be called easily used a 
# seperate script, like the reminder script.
#

package webcalng_subs;
use strict;
use Time::Local;
use HTML::Template;
use CGI::Cookie;

#
# Subroutine to clean the input parameters from nasty, possibly malicious input.
#
sub clean_params {
	my (@args,$key,$value,%cleanq);
	@args = $::q->param;
	for $key (@args) {
		# Since we can not tell if the given parameter is meant to be in 
		# array or scalar format, we just put everything in scalar format 
		# in the cleaned parameter hash.  Parameters with multiple values
		# just use the convention of being comma seperated.
		my @testarray = $::q->param($key);
		if ($#testarray > 0) {
			for (@testarray) {
				$value = clean_scalar($key,$_);
				$cleanq{$key} .= "$value,";
			}
			$cleanq{$key} =~ s/,$//;
		} else {
			$value = clean_scalar($key,$::q->param($key));
			$cleanq{$key} = $value;
		}
	}
	return \%cleanq;
}

#
# Subroutine to replace nasty characters in a scalar.  This is mostly to guard against CSS attacks,
# as the database code guards against sql injection by using placeholders in the sql statements.
#
sub clean_scalar {
	my ($key,$scalar) = (@_);
	$scalar =~ s/\0//g;	# Poison nulls!
	if ($key !~ /password/) {
		$scalar =~ s/\</&lt;/g  unless (($::preferences{'allow_html'}{'value'}) && ($key =~ /notes|description/));
		$scalar =~ s/\>/&gt;/g  unless (($::preferences{'allow_html'}{'value'}) && ($key =~ /notes|description/));
		$scalar =~ s/\&/&amp;/g unless (($key =~ /link/) || (($::preferences{'allow_html'}{'value'}) && ($key =~ /notes|description/)));
	}
	return $scalar;
}

#
# Subroutine to read in the webcalng.conf file
#
sub read_webcalng_conf {
	# Read in the file.
	open FILE, "<$::db_dir/webcalng.conf" or hard_error("Could not read webcalng.conf: $!\n");
	while (<FILE>) {
		next if /^#/;
		next if /^\s*$/;
		chomp;
		my ($key,$value) = split /=/, $_, 2;
		$::webcalng_conf{$key} = $value;
	}
	close FILE;

	return 1;
}

#
# Subroutine to setup language strings in %language hash.
#
sub setup_language {
	my ($language_file,$language_read);
	$language_read = 0;

	# Determine what language to use.
	if ($::preferences{'language'}{'value'}) {
		$language_file = "webcalng." . $::preferences{'language'}{'value'};
	} else {
		$language_file = "webcalng.english";
	}
	$language_file = "webcalng.english" if (! -f "language/$language_file");

	# First, read in contents of English language file, in case the real 
	# language does not have every item defined.
	if (-f "language/webcalng.english") {
		open FILE, "<language/webcalng.english" or hard_error("Could not open webcalng.english: $!");
		while (<FILE>){
			next if /^#/;
			next if /^\s+$/;
			chomp;
			my ($key,$value) = split /=/, $_, 2;
			$::language{$key} = $value;
		}
		close FILE;
		$language_read = 1;
	}

	# Now, override english language keys with real language keys.
	if ($language_file ne "webcalng.english") {
		open FILE, "<language/$language_file" or hard_error("Could not open $language_file: $!");
		while (<FILE>){
			next if /^#/;
			next if /^\s+$/;
			chomp;
			my ($key,$value) = split /=/, $_, 2;
			$::language{$key} = $value;
		}
		close FILE;
		$language_read = 1;
	}

	# Make sure we read in a language file.
	hard_error("Did not find a language file to read.") unless ($language_read);

	# Finally, setup preference key descriptions with newly created language information.
	for my $key (keys %::preferences) {
		my $language_key                    = $key . "_description";
		$::preferences{$key}{'description'} = $::language{$language_key};
	}

	return 1;
}

#
# Subroutine that acts as a filter to parse template files and convert 
# language variables to strings.
#
sub language {
	my $text = shift;
	$$text =~ s/__(.*?)__/$::language{$1}/g;
	return 1;
}

#
# Subroutine to get the preferences from the io module, and
# setup the %preferences hash.  Some extra setup of the hash is
# done so that the data structure can drive the preferences template.
# 
sub read_preferences {
	my ($admin_preferences_ref,$calendar_preferences_ref,$key);
	
	# Get a reference to a hash of the admin preferences, and process them.
	$admin_preferences_ref = webcalng_io::get_admin_preferences();
	for $key (keys %$admin_preferences_ref) {
		my (@options);
		if ($admin_preferences_ref->{$key}{'options'} =~ /:/) {
			if ($admin_preferences_ref->{$key}{'options'} eq "0:1") {
				$::preferences{$key}{'boolean'} = 1;
			} elsif ($admin_preferences_ref->{$key}{'options'} =~ /dir:(.*)/) {
				$::preferences{$key}{'picklist'} = 1;
				my ($dir,@dir,$option);
				$dir = $1;
				opendir DIR, $dir or hard_error("Could not open $dir for reading: $!");
				@dir = readdir DIR;
				closedir DIR;
				for $option (@dir) {
					if ($option =~ /^webcalng\.(.*)/) {
						my %hash;
						$hash{'option'}   = $1;
						$hash{'selected'} = 1 if ($1 eq $admin_preferences_ref->{$key}{'value'});
						push (@options, \%hash);
					}
				}
				$::preferences{$key}{'options'} = \@options;
			} else {
				$::preferences{$key}{'picklist'} = 1;
				my (@tmp,$option);
				@tmp = split /:/, $admin_preferences_ref->{$key}{'options'};
				for $option (@tmp) {
					my %hash;
					$hash{'option'}   = $option;
					$hash{'selected'} = 1 if ($option eq $admin_preferences_ref->{$key}{'value'});
					push (@options, \%hash);
				}
				$::preferences{$key}{'options'} = \@options;
			}
		} else {
			if ($admin_preferences_ref->{$key}{'options'} eq "color") {
				$::preferences{$key}{'color'} = 1;
			} else {
				$::preferences{$key}{'text'} = 1;
			}
		}
		$::preferences{$key}{'key'}         = $key;
		$::preferences{$key}{'value'}       = $admin_preferences_ref->{$key}{'value'};
		$::preferences{$key}{'order'}       = $admin_preferences_ref->{$key}{'order'};
		$::preferences{$key}{'usermod'}     = $admin_preferences_ref->{$key}{'usermod'};
		$::preferences{$key}{'category'}    = $admin_preferences_ref->{$key}{'category'};
	}

	# If we are using a calendar, get those calendar preferences and process them.
	if ($::calendar) {
		$calendar_preferences_ref = webcalng_io::get_calendar_preferences($::calendar);
		for $key (keys %$calendar_preferences_ref) {
			if ($::preferences{$key}{'usermod'}) {
				$::preferences{$key}{'key'}   = $key;
				$::preferences{$key}{'value'} = $calendar_preferences_ref->{$key};
				# If this is a "picklist" type of preference, we need to update the
				# selected key to match the user's setting.
				if ($::preferences{$key}{'picklist'}) {
					my (@tmp,$option_ref,@options);
					@tmp = @{ $::preferences{$key}{'options'} };
					for $option_ref (@tmp) {
						my %hash;
						$hash{'option'}   = $option_ref->{'option'};
						$hash{'selected'} = 1 if ($option_ref->{'option'} eq $calendar_preferences_ref->{$key});
						push (@options, \%hash);
					}
					$::preferences{$key}{'options'} = \@options;
				}
			}
		}
	}

	#
	# Finally, setup the template path, depending on the theme they choose.
	#
	$::template_path = "themes" . "/webcalng." . $::preferences{'theme'}{'value'};
	$::template_path = "themes/webcalng.Basic" if (! -d "$::template_path");
	hard_error('Could not find a valid template path.') if (! -d "$::template_path");
	$ENV{'HTML_TEMPLATE_ROOT'} = $::template_path;

	return 1;
}

# 
# Subroutine to gather preference changes to be written by io module.
#
sub save_preferences {
	my ($adminprefs,$formkey,@form,%updates);

	# First, determine what preference we are writing.
	$adminprefs = $::cleanq->{'adminprefs'};    
	hard_error("\$::calendar should not be set when updating admin preferences.") if (($adminprefs) && ($::calendar));
	hard_error("Neither \$::calendar or \$adminprefs was set for updating preferences.") if ((! $adminprefs) && (! $::calendar));

	# Now, read in all the new preferences to be saved.
	for $formkey (keys %{$::cleanq}) {
		if ($formkey =~ /^PREFERENCE_(\S+)/) {
			my $key = $1;
			if (($key =~ /_USERMOD$/) && ($adminprefs)) {
				$key =~ s/_USERMOD$//;
				$updates{$key}{'usermod'} = $::cleanq->{$formkey} if (defined $::preferences{$key}{'value'});
			} else {
				if ($key eq "import_calendars") {
					# Special checks for the import list.
					my ($list,@list,$newlist,$import_ref);
					$list = $::cleanq->{$formkey};
					if ((! $list) || ($list eq "")) {
						$updates{$key}{'value'} = "";
					} else {
						@list       = split /,/, $list;
						$newlist    = "";
						$import_ref = webcalng_io::get_import_permissions($::calendar);
						for (@list) {
							s/^\s+//;
							s/\s+$//;
							next if ($_ eq $::calendar);
							$newlist .= "$_," if ($import_ref->{$_}{'import'});
						}
						$newlist =~ s/,$//;
						$updates{$key}{'value'} = $newlist;
					}
				} else {
					$updates{$key}{'value'} = $::cleanq->{$formkey} if (defined $::preferences{$key}{'value'});
				}
			}
		}
	}

	# Call the io module to save the data.
	if ($adminprefs) {
		webcalng_io::save_admin_preferences(\%updates);
	} else {
		webcalng_io::save_calendar_preferences(\%updates);
	}

	return 1;
}

#
# Simple subroutine to send the HTTP header.
#
sub http_header {
	if ($::wap) {
		print "Content-Type: text/vnd.wap.wml\n\n";
		print "<?xml version=\"1.0\"?>\n";
	} else {
		# Turn off caching for login screens.
		if ($::nocache) {
			print $::q->header(-Expires       => "Thu, 01 Jan 1970 00:00:00 GMT",
		                   	-Pragma        => "no-cache",
		                   	-Cache_Control => "no-cache",
		                  	);
		} else {
			print $::q->header;
		}
	}

	return 1;
}

#
# Subroutine to output the css code.
#
sub css {
	my ($title,$css_template,$file,$backgroundimage);

	# Setup template.
	if ($::wap) {
		$file = "css.wml.tmpl";
	} else {
		$file = "css.tmpl";
	}
	$css_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Set page title.
	$title = "webcalng";
	$title .= " : $::calendar " if ($::calendar);

	# Some special setup for the background, in case they use an image.
	if ($::preferences{'background'}{'value'} =~ /[\/,\.]/) {
		$backgroundimage = 1;
	} else {
		$backgroundimage = 0;
	}

	# Assign parameters to template.
	$css_template->param(
		title           => $title,
		backgroundimage => $backgroundimage,
		background      => $::preferences{'background'}{'value'},
		hyperlinks      => $::preferences{'hyperlinks'}{'value'},	
		bordercolor     => $::preferences{'bordercolor'}{'value'},	
		textcolor       => $::preferences{'textcolor'}{'value'},
		textbackground  => $::preferences{'textbackground'}{'value'},
		weekdays        => $::preferences{'weekdays'}{'value'},
		weekends        => $::preferences{'weekends'}{'value'},
		current_day     => $::preferences{'current_day'}{'value'},
		not_a_day       => $::preferences{'not_a_day'}{'value'},
	);

	# Write template output.
	$::output .= $css_template->output;
	return 1;
}

#
# Subroutine to see if the user is authorized to access the $op they 
# have requested.
#
sub authorize {
	my ($localop,$adminprefs);
	my (%anyone,%owner,%read,%write,%admin);
	$localop        = $::op || "null";
	$adminprefs     = $::cleanq->{'adminprefs'};

	# Define hashes of operations which require different levels of access.
	%anyone = qw(null 1 start 1 login 1 logout 1 process_login 1 list 1 select_calendar 1 select_calendar2 1 colors 1 changepw 1 changepw2 1 respond 1 respond2 1 userdoc 1 admindoc 1);
	%read   = qw(year 1 month 1 week 1 day 1 notes 1 meeting 1 search 1);
	%write  = qw(additem 1 additem2 1 delitem 1 delitemmulti 1 edititem 1 edititem2 1 edititemmulti 1 address 1 addaddress 1 addaddress2 1 deladdress 1 editaddress 1 editaddress2 1 task 1 addtask 1 addtask2 1 edittask 1 edittask2 1 marktask 1 deltask 1 availability 1);
	%admin  = qw(adduser 1 adduser2 1 deluser 1 deluser2 1 delcal 1 delcal2 1 modify 1 modify2 1);
	if ($::preferences{'create_cal_all'}{'value'}) {
		$anyone{'addcal'}  = 1;
		$anyone{'addcal2'} = 1;
	} else {
		$admin{'addcal'}  = 1;
		$admin{'addcal2'} = 1;
	}
	
	# The following are accessible by anyone.
	return 1 if ($anyone{$localop});

	# Determine if access should be granted for operations that are done on a specific calendar.
	if ($::calendar) {
	 	if ($::permissions_ref->{'type'} eq "open") {
			return 1;
		}
		if ($::permissions_ref->{'type'} eq "public") {
			return 1 if ($read{$localop});
			return 1 if (($write{$localop}) && ($::username) && ($::username =~ /^($::permissions_ref->{'write'})$/));
		}
		if ($::permissions_ref->{'type'} eq "private") {
			return 1 if (($read{$localop}) && ($::username) && (($::username =~ /^($::permissions_ref->{'read'})$/) || ($::username =~ /^($::permissions_ref->{'write'})$/)));
			return 1 if (($write{$localop}) && ($::username) && ($::username =~ /^($::permissions_ref->{'write'})$/));
		}
		return 1 if ((($localop eq "preferences") || ($localop eq "save_preferences")) && (! $adminprefs) && ($::username) && ($::permissions_ref->{'owner'} eq $::username));
	}

	# The following require the user to be logged in as admin.
	return 1 if ((($localop eq "preferences") || ($localop eq "save_preferences")) && ($adminprefs) && ($::username) && ($::username eq "admin"));
	return 1 if (($admin{$localop}) && ($::username) && ($::username eq "admin"));

	# If we get this far, they did not qualify for access, so return 0.
	return 0;
}

#
# Subroutine to redirect the user to a given URL.
# We must use a fully qualified URL for the Location directive.
#
sub redirect {
	my ($url,@cookies) = (@_);
	my ($browser,$not_netscape4,$not_iis);

	# Create URL for redirection.
	$url = $::HTTP . $ENV{'HTTP_HOST'} . $url;

	# Determine if they are not using a netscape level 4 browser.
	$not_netscape4 = 0;
	$browser       = $ENV{'HTTP_USER_AGENT'};
	$not_netscape4 = 1 if (($browser) && ($browser !~ /^Mozilla\/4\./));
	$not_netscape4 = 1 if (($browser) && ($browser =~ /MSIE/));

	# Determine if we are running under IIS.
	$not_iis = 0;
	$not_iis = 1 if (($ENV{'SERVER_SOFTWARE'}) && ($ENV{'SERVER_SOFTWARE'} !~ /iis/i));

	# Wap browsers will require a special redirect.  They should never see
	# the deck that we send, but it needs to be there in case the wap browser
	# does not support the Location redirect for some reason.
	if ($::wap) {
		print "Location: $url\n";
		print "Content-type: text/vnd.wap.wml\n\n";
		print "<?xml version=\"1.0\"?>\n";
		print "<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\">\n";
		print "<wml>\n";
		print "<card title=\"redirect\">\n";
		print "<p>\n";
		print "<b>link:</b>\n";
		print "<br/>\n";
		print "<select name=\"link\" title=\"link\">\n";
		print "<option title=\"go\" onpick=\"$url\">$url</option>\n";
		print "</select>\n";
		print "</p>\n";
		print "</card>\n";
		print "</wml>\n";
	} else {
		# Use a location redirect, unless we are setting cookie(s) *and*
		# redirecting to a url with a query string for a netscape 4 browser.
		# Netscape4 browsers puke when you do this, so we have to do a meta
		# refresh in this case.  Ugh.  Also, try to make the browser not cache
		# any pages that have login cookies being set.  Finally, IIS does not
		# seem to be able to handle a redirect with a cookie setting, so watch
		# out for that.
		if ($#cookies >= 0) {
			if ((($not_netscape4) || ($url !~ /\?/)) && ($not_iis)) {
			print $::q->redirect(-URL           => $url,
			                   	-COOKIE        => \@cookies,
			                   	-Expires       => "Thu, 01 Jan 1970 00:00:00 GMT",
			                   	-Pragma        => "no-cache",
			                   	-Cache_Control => "no-cache",
			                  	);
			} else {
				print $::q->header(-COOKIE        => \@cookies,
			                 	-Expires       => "Thu, 01 Jan 1970 00:00:00 GMT",
			                 	-Pragma        => "no-cache",
			                 	-Cache_Control => "no-cache",
			                  	);
				print "<META HTTP-EQUIV=\"Refresh\" CONTENT=\"1; URL=$url\">\n";
			}
		} else {
			print $::q->redirect(-URL => $url);
		}
	}

	# That is all we need to do.
	goto EXIT;
}

#
# Subroutine to display start page.
#
sub start {
	my ($start_template,$file,$admin,$create);

	# Setup template.
	if ($::wap) {
		$file = "start.wml.tmpl";
	} else {
		$file = "start.tmpl";
	}
	$start_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Assign parameters to template.
	$admin  = $create = 0;
	$admin  = 1 if ($::username eq "admin");
	$create = 1 if (($::preferences{'create_cal_all'}{'value'}) || ($admin));
	$start_template->param(
		username        => $::username,
		webcal          => $::webcalng_conf{'WEBCAL'},
		webcal_version  => $::webcalng_conf{'VERSION'},
		start_image     => $::preferences{'start_image'}{'value'},
		start_link      => $::preferences{'start_link'}{'value'},
		start_link_text => $::preferences{'start_link_text'}{'value'},
		admin           => $admin,
		create          => $create,
		sessionkey      => $::sessionkey,
	);

	# Write template output.
	$::output .= $start_template->output;

	return 1;
}

#
# Subroutine to display common webcalng header.
#
sub header {
	my ($header_template,$file,$next,$prev,$year,$month,$day);
	my ($current_year,$current_month,$current_day);
	my ($next_year,$next_month,$next_day);
	my ($prev_year,$prev_month,$prev_day);
	my ($title,$year_view,$month_view,$week_view,$day_view);
	my (@MONTHS,%WEEK_START);
	my ($return_link,$write,$owner);

	# Setup template.
	if ($::wap) {
		$file = "header.wml.tmpl";
	} else {
		$file = "header.tmpl";
	}
	$header_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Decide what day of the week we start on, Sunday or Monday.
	if ($::preferences{'dow_start_mon'}{'value'}) {
		%WEEK_START = qw (Mon 0 Tue 1 Wed 2 Thu 3 Fri 4 Sat 5 Sun 6);
	} else {
		%WEEK_START = qw (Sun 0 Mon 1 Tue 2 Wed 3 Thu 4 Fri 5 Sat 6);
	}

	# Setup dates for links to different view types and prev/next views.
	($current_day,$current_month,$current_year) = (localtime)[3,4,5];
	$current_month++;
	$current_year += 1900;
	$year  = $::cleanq->{'year'};
	$month = $::cleanq->{'month'};
	$day   = $::cleanq->{'day'};
	($day,$month,$year) = ($current_day,$current_month,$current_year) if (! $year);
	$month = 1 if (! $month);
	$day   = 1 if (! $day);
	$year_view = $month_view = $week_view = $day_view = 0;
	soft_error('baddaterange') if (($year < 1970) || ($year > 2037));
	if ($::op eq "year") {
		$year_view++;
		$title = $year;
		$prev  = timelocal "0","0","12","1","0",($year-1901);
		$next  = timelocal "0","0","12","1","0",($year-1899);
		if ($year == $current_year) {
			$month = $current_month;
			$day   = $current_day;
		} else {
			$month = 1;
			$day   = 1;
		}
	} elsif ($::op eq "month") {
		$month_view++;
		$title = "$::language{ $MONTHS[$month-1] } $year";
		if ($month == 1) {
			$prev_year  = $year - 1;
			$prev_month = 12;	
			$next_year  = $year;
			$next_month = 2;
		} elsif ($month == 12) {
			$next_year  = $year + 1;
			$next_month = 1;
			$prev_year  = $year;
			$prev_month = 11;	
		} else {
			$prev_year  = $year;
			$prev_month = $month - 1;
			$next_year  = $year;
			$next_month = $month + 1;
		}
		$prev = timelocal "0","0","12","1",($prev_month-1),($prev_year-1900);
		$next = timelocal "0","0","12","1",($next_month-1),($next_year-1900);
		if (($month == $current_month) && ($year == $current_year)) {
			$day = $current_day;
		} else {
			$day = 1;
		}
	} elsif ($::op eq "week") {
		$week_view++;
		my $time   = timelocal "0","0","12",$day,($month-1),($year-1900);
		my $dow    = (split /\s+/, localtime($time))[0];
		my $day1   = $time - (86400 * $WEEK_START{$dow});
		my $day7   = $day1 + (86400 * 7);
		my $today  = timelocal "0","0","16",$current_day,($current_month-1),($current_year-1900);
		if (($today > $day1) && ($today < $day7)) {	
			$year  = $current_year;
			$month = $current_month;
			$day   = $current_day;
		}
		my ($dow1,$month1,$year1) = (localtime($day1))[3,4,5];
		$year1 += 1900;
		if ($::preferences{'date_format'}{'value'}) {
			$title = "$::language{'week_of'} $dow1 $::language{ $MONTHS[$month1] } $year1";
		} else {
			$title = "$::language{'week_of'} $::language{ $MONTHS[$month1] } $dow1, $year1";
		}
		$prev  = $day1 - (86400 * 7);
		$next  = $day1 + (86400 * 7);
	} elsif ($::op eq "day") {
		$day_view++;
		if ($::preferences{'date_format'}{'value'}) {
			$title = "$day $::language{ $MONTHS[$month-1] } $year";
		} else {
			$title = "$::language{ $MONTHS[$month-1] } $day, $year";
		}
		$next = $prev = timelocal "0","0","12",$day,($month-1),($year-1900);
		$prev = $prev - 86400;
		$next = $next + 86400;
	} else {
		hard_error("bad op: $::op");
	}
	($next_day,$next_month,$next_year) = (localtime($next))[3,4,5];
	($prev_day,$prev_month,$prev_year) = (localtime($prev))[3,4,5];
	$prev_year += 1900;
	$next_year += 1900;
	$prev_month++;
	$next_month++;
	$next = "op=$::op&year=$next_year&month=$next_month&day=$next_day";
	$prev = "op=$::op&year=$prev_year&month=$prev_month&day=$prev_day";
	$next =~ s/&/&amp;/g if ($::wap);
	$prev =~ s/&/&amp;/g if ($::wap);

	# Determine what the return link will be.
	$return_link = $ENV{'REQUEST_URI'};
	$return_link = $ENV{'SCRIPT_NAME'} . "?" . $ENV{'QUERY_STRING'} if (! $return_link);
	$return_link =~ s/^.*?\?//;	
	$return_link =~ s/=/--/g;
	$return_link =~ s/&/../g;
	$return_link =~ s/\s+/+/g;

	# Setup variable that determines if certain links are shown.
	$write = $owner = 0;
	$write = 1 if (($::username) && ($::username =~ /^($::permissions_ref->{'write'})$/));
	$write = $owner = 1 if ($::permissions_ref->{'type'} eq "open");
	$owner = 1 if ($::permissions_ref->{'owner'} eq $::username);

	# Populate header template parameters and print out template.
	$header_template->param(
		webcal        => $::webcalng_conf{'WEBCAL'},
		calendar      => $::calendar,
		title         => $title,
		header_link   => $::preferences{'header_link'}{'value'},
		header_image  => $::preferences{'header_image'}{'value'},
		next          => $next,
		prev          => $prev,
		year          => $year,
		month         => $month,
		day           => $day,
		current_year  => $current_year,
		current_month => $current_month,
		current_day   => $current_day,
		year_view     => $year_view,
		month_view    => $month_view,
		week_view     => $week_view,
		day_view      => $day_view,
		return_link   => $return_link,
		write         => $write,
		owner         => $owner,
		sessionkey    => $::sessionkey,
	);

	# Write template output.
	$::output .= $header_template->output;

	return 1;
}

#
# Subroutine to display common webcalng footer.
#
sub footer {
	my ($footer_template,$file,$next_url,$return_link,$searchtype,$year,$month,$day,$thisyear,$thismonth,$thisday,$dayview);

	# Setup template.
	if ($::wap) {
		$file = "footer.wml.tmpl";
	} else {
		$file = "footer.tmpl";
	}
	$footer_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup the next_url so that the user can return to this page
	# if the are re-logging in.
	$next_url    = $ENV{'REQUEST_URI'};
	$next_url    = $ENV{'SCRIPT_NAME'} . "?" . $ENV{'QUERY_STRING'} if (! $next_url);
	$next_url    =~ s/=/--/g;
	$next_url    =~ s/&/../g;
	$next_url    =~ s/\s+/+/g;
	$return_link = $next_url;
	$return_link =~ s/^.*?\?//;	
	$return_link =~ s/=/--/g;
	$return_link =~ s/&/../g;
	$return_link =~ s/\s+/+/g;

	# Setup variable used for search form.
	($thisday,$thismonth,$thisyear) = (localtime)[3,4,5];
	$thismonth++;
	$thisyear  += 1900;
	$searchtype = $::op;
	$year       = $::cleanq->{'year'}  || $thisyear;
	$month      = $::cleanq->{'month'} || $thismonth;
	$day        = $::cleanq->{'day'}   || $thisday;
	$dayview    = 0;
	$dayview++ if ($searchtype eq "day");

	# Assign parameters to template.
	$footer_template->param(
		webcal           => $::webcalng_conf{'WEBCAL'},
		calendar         => $::calendar,
		username         => $::username,
		footerlinks      => $::preferences{'show_footer_links'}{'value'},
		start_link       => $::preferences{'start_link'}{'value'},
		start_link_text  => $::preferences{'start_link_text'}{'value'},
		next_url         => $next_url,
		return_link      => $return_link,
		timezone         => $::preferences{'timezone'}{'value'},
		searchtype       => $searchtype,
		year             => $year,
		month            => $month,
		day              => $day,
		dayview          => $dayview,
		sessionkey       => $::sessionkey,
	);

	# Write template output.
	$::output .= $footer_template->output;
	return 1;
}

#
# Subroutine to display webcalng version at bottom of page.
#
sub version {
	my ($version_template,$file);

	# Setup template.
	if ($::wap) {
		$file = "version.wml.tmpl";
	} else {
		$file = "version.tmpl";
	}
	$version_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Assign parameters to template.
	$version_template->param(
		webcal_version => $::webcalng_conf{'VERSION'},
	);

	# Write template output.
	$::output .= $version_template->output;
	return 1;
}

#
# Login subroutine. If using a cookie based approach, display the login
# form.  If using a .htaccess based approach, redirect the user to the
# private url of the application.
#
sub login {
	my ($message) = (@_);
	my ($login_template,$file,%cookies,$username,$next_url);

	# Setup template.
	if ($::wap) {
		$file = "login.wml.tmpl";
	} else {
		$file = "login.tmpl";
	}
	$login_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Determine the authentication type.  If not using cookies, then just redirect to
	# the security URL.  If we are already at the secure URL, then sent 401 and make
	# them authenticate again, as we obviously did not like who they are currently
	# logged in as.
	if (! $::webcalng_conf{'USE_COOKIES'}) {
		if ($ENV{'REQUEST_URI'} =~ /$::webcalng_conf{'PRIVATE'}/) {
			# They made it past the 1st redirect the secure URL, so they must have
			# logged in with a valid user/pass.  Only problem is that user is not
			# authorized to access what they tried to, so make them login again.
			print "Status: 401 Authorization Required.\n";
			print "WWW-authenticate: Basic realm=\"webcalng\"\n";
			print "Content-type: text/plain\n\n";
			goto EXIT;
		} else {
			my ($secure_url,$request_uri);
			$secure_url  = "";
			$request_uri = $ENV{'REQUEST_URI'};
			$request_uri =~ s/.*\?/?/;
			if ($::cleanq->{'next_url'}) {
				my $next_url = $::cleanq->{'next_url'};
				$next_url    =~ s/.*\?/?/;
				$next_url    =~ s/--/=/g;
				$next_url    =~ s/\.\./&/g;
				$next_url    =~ s/\s+/+/g;
				$secure_url  = $::webcalng_conf{'PRIVATE'} . $next_url;
			}
			$secure_url  = $::webcalng_conf{'PRIVATE'} . $request_uri if (! $secure_url);
			$secure_url  = $::webcalng_conf{'PRIVATE'} if ($secure_url =~ /op=login/);
			redirect($secure_url);
		}
	}

	# If we get this far, it must be a cookie based authentication.
	# Figure out where they are going to.
	$next_url = $::cleanq->{'next_url'} if ($::cleanq->{'next_url'});
	$next_url = $ENV{'REQUEST_URI'} if (! $next_url);
	$next_url = $ENV{'SCRIPT_NAME'} . "?" . $ENV{'QUERY_STRING'} if (! $next_url);
	$next_url = "" if ($next_url =~ /op=login/);
	$next_url =~ s/=/--/g;
	$next_url =~ s/&/../g;
	$next_url =~ s/\s+/+/g;

	# See if they have logged in before.
	$username = "";
	%cookies  = fetch CGI::Cookie;
	if ($cookies{$webcalng_auth::user_cookie}) {
		$username = $cookies{$webcalng_auth::user_cookie}->value;
	}

	# Assign parameters to template.
	$login_template->param(
		username => $username,
		webcal   => $::webcalng_conf{'WEBCAL'},
		message  => $message,
		next_url => $next_url,
	);

	# Write template output.
	$::output .= $login_template->output;

	return 1;
}

#
# This subroutine just displays a message telling the user to close their browser
# if they are trying to logout when using basic authentication.  This is not used if
# session based authentication is in use.
#
sub basic_logout {
	my ($basic_logout_template,$file,$admin);

	# Setup template.
	if ($::wap) {
		$file = "basic_logout.wml.tmpl";
	} else {
		$file = "basic_logout.tmpl";
	}
	$basic_logout_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Assign parameters to template.
	$basic_logout_template->param(
		logout_url => $::webcalng_conf{'PUBLIC'},
		username   => $::username,
	);

	# Write template output.
	$::output .= $basic_logout_template->output;

	return 1;
}

#
# Subroutine to display the year view.
#
sub year {
	my ($year_template,$file,@MONTHS,$year,$month,$year_data,@year_data);
	my ($startmark,$endmark,$itemsref,$exceptions_ref,@calendars);
	my ($current_year,$current_month);

	# Setup template.
	if ($::wap) {
		$file = "year.wml.tmpl";
	} else {
		$file = "year.tmpl";
	}
	$year_template = HTML::Template->new(
		filename          => $file, 
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Get year we are looking at.
	$year  = $::cleanq->{'year'};
	if (! $year)  {
		$year  = (localtime)[5];
		$year += 1900;
	}

	# Get current year and month.
	($current_month,$current_year) = (localtime)[4,5];
	$current_month++;
	$current_year += 1900;

	# Get a hash of items that occur during this year.
	$startmark = timelocal "0","0","0","1","0",($year-1900);
	$endmark   = timelocal "0","0","0","1","0",($year-1899);
	@calendars = get_imported_calendars($::calendar);
	($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,undef,undef,undef,@calendars);

	# Loop through the months, populating template data.
	for $month (1..12) {
		my %month_data;

		# Determine if we are starting or ending a row in the year view.
		if ((($month-1) % 4) == 0) {
			$month_data{'start_row'} = 1;
		} else {
			$month_data{'start_row'} = 0;
		}
		if (($month % 4) == 0) {
			$month_data{'end_row'} = 1;
		} else {
			$month_data{'end_row'} = 0;
		}

		# Populate the data for the given month.
		if (($year == $current_year) && ($month == $current_month)) {
			$month_data{'current'} = 1;
		} else {
			$month_data{'current'} = 0;
		}
		$month_data{'year'}       = $year;
		$month_data{'month'}      = $month;
		$month_data{'month_name'} = $::language{ $MONTHS[$month-1] };
		($month_data{'month_rows'},$month_data{'dows'}) = month_data_setup($itemsref,$exceptions_ref,$year,$month);
		push(@year_data,\%month_data);
	}
	$year_data = \@year_data;

	# Assign parameters to template.
	$year_template->param(
		webcal       => $::webcalng_conf{'WEBCAL'},
		calendar     => $::calendar,
		year_data    => $year_data,
		sessionkey   => $::sessionkey,
	);

	# Write template output.
	$::output .= $year_template->output;

	return 1;
}

#
# Subroutine to display the month view.
#
sub month {
	my ($month_template,$file,$return_link,$month,$year,$month_data,$dow_data,$startmark,$endmark,$itemsref,$exceptions_ref,@calendars);

	# Setup template.
	if ($::wap) {
		$file = "month.wml.tmpl";
	} else {
		$file = "month.tmpl";
	}
	$month_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Determine what the return link will be.
	$return_link = $ENV{'REQUEST_URI'};
	$return_link = $ENV{'SCRIPT_NAME'} . "?" . $ENV{'QUERY_STRING'} if (! $return_link);
	$return_link =~ s/^.*?\?//;	
	$return_link =~ s/=/--/g;
	$return_link =~ s/&/../g;
	$return_link =~ s/\s+/+/g;

	# Determine the month and year we are working on.
	$year  = $::cleanq->{'year'};
	$month = $::cleanq->{'month'};
	if ((! $year) || (! $month)) {
		($month,$year) = (localtime)[4,5];
		$month++;
		$year += 1900;
	}

	# Get a hash of items that occur during this month.
	$startmark = timelocal "0","0","0",1,($month-1),($year-1900);
	if ($month == 12) {
		$endmark   = timelocal "0","0","0",1,1,($year-1900+1);
	} else {
		$endmark   = timelocal "0","0","0",1,$month,($year-1900);
	}
	@calendars = get_imported_calendars($::calendar);
	($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,undef,undef,undef,@calendars);

	# Call a subroutine to give us the data for this month.
	($month_data,$dow_data) = month_data_setup($itemsref,$exceptions_ref,$year,$month);

	my @z = webcalng_io::get_calendar_list();
	hard_error("") if ($#z > 1);

	# Assign parameters to template.
	$month_template->param(
		webcal      => $::webcalng_conf{'WEBCAL'},
		return_link => $return_link,
		calendar    => $::calendar,
		month_rows  => $month_data,
		dows        => $dow_data,
		show_added  => $::preferences{'show_added'}{'value'},
		showcalname => $::preferences{'show_cal_name'}{'value'},
		sessionkey  => $::sessionkey,
	);

	# Write template output.
	$::output .= $month_template->output;

	return 1;
}

#
# Subroutine to display the week view.
#
sub week {
	my ($week_template,$file,$return_link,$year,$month,$day,$week_row,@week_row,$dow_data,@dow_data);
	my ($current_day,$current_month,$current_year);
	my (%WEEK_START,$start_time,$time,$dow,$x,$itemsref,$exceptions_ref,$startmark,$endmark,@calendars);
	my ($wday,$yday,$weeknum);

	# Setup template.
	if ($::wap) {
		$file = "week.wml.tmpl";
	} else {
		$file = "week.tmpl";
	}
	$week_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);
	
	# Determine what the return link will be.
	$return_link = $ENV{'REQUEST_URI'};
	$return_link = $ENV{'SCRIPT_NAME'} . "?" . $ENV{'QUERY_STRING'} if (! $return_link);
	$return_link =~ s/^.*?\?//;	
	$return_link =~ s/=/--/g;
	$return_link =~ s/&/../g;
	$return_link =~ s/\s+/+/g;

	# Determine the week we are working on.
	$year  = $::cleanq->{'year'};
	$month = $::cleanq->{'month'};
	$day   = $::cleanq->{'day'};
	if ((! $year) || (! $month)) {
		($day,$month,$year) = (localtime)[3,4,5];
		$month++;
		$year += 1900;
	}

	# Find out what the current date is.
	($current_day,$current_month,$current_year) = (localtime)[3,4,5];
	$current_month++;
	$current_year += 1900;

	# Decide what day of the week we start on, Sunday or Monday.
	if ($::preferences{'dow_start_mon'}{'value'}) {
		%WEEK_START = qw (Mon 0 Tue 1 Wed 2 Thu 3 Fri 4 Sat 5 Sun 6);
	} else {
		%WEEK_START = qw (Sun 0 Mon 1 Tue 2 Wed 3 Thu 4 Fri 5 Sat 6);
	}

	# Determine what week number this is.
	$time = timelocal "0","0","12",$day,($month-1),($year-1900);
	($wday,$yday) = (localtime($time))[6,7];
	$wday++;
	$yday++;
	if ($WEEK_START{'Mon'} == 0) {
		if ($wday == 1) {
			$wday = 7;
		} else {
			$wday = $wday - 1;
		}
	} 
	$yday += (7 - $wday);
	$weeknum = $yday / 7;
	$weeknum =~ s/\..*//;
	$weeknum++ if ($yday % 7);

	# Populate the template variables for the days of the week.
	for $dow (sort { $WEEK_START{$a} <=> $WEEK_START{$b} } keys %WEEK_START) {
		my %hash;
		$hash{'dow'} = $::language{$dow};
		$hash{'short_dow'} = substr($::language{$dow},0,1);
		push(@dow_data,\%hash);	
	}	
	$dow_data = \@dow_data;

	# Determine what the 1st day of this week view is.
	$time = timelocal "0","0","12",$day,($month-1),($year-1900);
	$dow  = (split /\s+/, localtime($time))[0];
	$start_time = $time - (86400 * $WEEK_START{$dow});	

	# Get a hash of items that occur during this week.
	($day,$month,$year) = (localtime($start_time))[3,4,5];
	$startmark          = timelocal "0","0","0",$day,$month,$year;
	($day,$month,$year) = (localtime($start_time+(86400*7)))[3,4,5];
	$endmark            = timelocal "0","0","0",$day,$month,$year;
	@calendars = get_imported_calendars($::calendar);
	($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,undef,undef,undef,@calendars);

	# Populate the template variables for the week data.
	for $x (0..6) {
		my (%hash,$new_itemsref);
		$time = $start_time + ($x * 86400);
		($day,$month,$year) = (localtime($time))[3,4,5];
		$month++;
		$year += 1900;
		$hash{'year'}  = $year;
		$hash{'month'} = $month;
		$hash{'day'}   = $day;
		if (($WEEK_START{'Sun'} == $x) || ($WEEK_START{'Sat'} == $x)) {
			$hash{'weekend'} = 1;
		} else {
			$hash{'weekday'} = 1;
		}
		if (($day == $current_day) && ($month == $current_month) && ($year == $current_year)) {
			$hash{'weekend'}     = 0;
			$hash{'weekday'}     = 0;
			$hash{'current_day'} = 1;
		}
		$new_itemsref  = filter_items($itemsref,$day,$month,$year);
		$new_itemsref  = apply_exceptions($new_itemsref,$exceptions_ref,$day,$month,$year);
		($hash{'day_items'},undef) = format_items_byday($new_itemsref,$day,$month,$year);
		push(@week_row, \%hash);
	}	
	$week_row = \@week_row;

	# The next section of code determines where the days of the given month fall.
	# Assign parameters to template.
	$week_template->param(
		webcal      => $::webcalng_conf{'WEBCAL'},
		return_link => $return_link,
		calendar    => $::calendar,
		dows        => $dow_data,
		week_row    => $week_row,
		show_added  => $::preferences{'show_added'}{'value'},
		showcalname => $::preferences{'show_cal_name'}{'value'},
		sessionkey  => $::sessionkey,
		weeknum     => $weeknum,
	);

	# Write template output.
	$::output .= $week_template->output;

	return 1;
}

#
# Subroutine to display the day view.
#
sub day {
	my ($day_template,$file,$return_link,$year,$month,$day,$month_rows,$time,$dow,$dows,$hour);
	my (@hours,$hours,@MONTHS,$itemsref,$exceptions_ref,$startmark,$endmark,$datetitle,@calendars);
	my ($write,$edit);

	# Setup template.
	if ($::wap) {
		$file = "day.wml.tmpl";
	} else {
		$file = "day.tmpl";
	}
	$day_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Determine what the return link will be.
	$return_link = $ENV{'REQUEST_URI'};
	$return_link = $ENV{'SCRIPT_NAME'} . "?" . $ENV{'QUERY_STRING'} if (! $return_link);
	$return_link =~ s/^.*?\?//;	
	$return_link =~ s/=/--/g;
	$return_link =~ s/&/../g;
	$return_link =~ s/\s+/+/g;

	# Determine the day we are working on.
	$year  = $::cleanq->{'year'};
	$month = $::cleanq->{'month'};
	$day   = $::cleanq->{'day'};
	if ((! $year) || (! $month) || (! $day)) {
		($day,$month,$year) = (localtime)[3,4,5];
		$month++;
		$year += 1900;
	}
	$time = timelocal "0","0","12",$day,($month-1),($year-1900);
	$dow  = (split /\s+/, localtime($time))[0];

	# Get a hash of items that occur in this month.  We get the items for the whole
	# month so that the month box can properly show which days have items in bold.
	# Then, we trim the hash down to just todays items later on.
	$startmark    = timelocal "0","0","0",1,($month-1),($year-1900);
	my ($nextmonth,$prevmonth,$nextyear,$prevyear);
	$nextmonth = $prevmonth = $month;
	$nextyear  = $prevyear  = $year;
	$nextmonth++;
	if ($nextmonth == 13) {
		$nextmonth = 1;
		$nextyear++;
	}
	$prevmonth = $prevmonth - 1;
	if ($prevmonth == 0) {
		$prevmonth = 12;
		$prevyear  = $prevyear - 1;
	}
	$endmark   = timelocal "0","0","0",1,($nextmonth-1),($nextyear-1900);
	@calendars = get_imported_calendars($::calendar);
	($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,undef,undef,undef,@calendars);

	# Populate the data for the given month.
	($month_rows,$dows) = month_data_setup($itemsref,$exceptions_ref,$year,$month);

	# Trim down our list of items and apply exceptions so that we only loop through todays items.
	$itemsref = filter_items($itemsref,$day,$month,$year);
	$itemsref = apply_exceptions($itemsref,$exceptions_ref,$day,$month,$year);

	# Loop through all hours of the day, formatting lists of items to display.
	# First, the group of items that occur before the day_start and day_end hours in the calendar preferences.	
	for $hour (0 .. ($::preferences{'day_start'}{'value'}-1)) {
		my (%hash,$ref);
		$hash{'hour'} = $hour;
		$hour        .= "00";
		$ref          = format_items_byhour($itemsref,$hour,($hour+100),$day,$month,$year);
		# Since we are out of the requested timeframe, only display hours that actually have items.
		if (ref($ref)) {
			$hash{'hour_items'} = $ref;
			$hash{'disp_hour'}  = format_time($hour);
			push(@hours,\%hash);
		}
	}

	# Now, the group of items within the day_start and day_end hours.
	for $hour ($::preferences{'day_start'}{'value'} .. $::preferences{'day_end'}{'value'}) {
		my (%hash,$ref);
		$hash{'hour'} = $hour;
		$hour        .= "00";
		$ref          = format_items_byhour($itemsref,$hour,($hour+100),$day,$month,$year);
		if (ref($ref)) {
			$hash{'hour_items'} = $ref;
		}
		# We want to display at least a blank for all hours within day_start
		# and day_end of the calendar preferences, unless show_empty_hours is not set.
		if ((ref($ref)) || ($::preferences{'show_empty_hours'}{'value'})) {
			$hash{'disp_hour'}  = format_time($hour);
			push(@hours,\%hash);
		}
	}

	# Now, the group of items after the day_start and day_end hours
	for $hour (($::preferences{'day_end'}{'value'}+1) .. 23) {
		my (%hash,$ref);
		$hash{'hour'} = $hour;
		$hour        .= "00";
		$ref          = format_items_byhour($itemsref,,$hour,($hour+100),$day,$month,$year);
		if (ref($ref)) {
			$hash{'hour_items'} = $ref;
			$hash{'disp_hour'}  = format_time($hour);
			push(@hours,\%hash);
		}
	}

	# Finally, the group of items that have no time associated with them.
	my $ref = format_items_byhour($itemsref,undef,undef,$day,$month,$year);
	if (ref($ref)) {
		my %hash;
		$hash{'hour_items'} = $ref;
		$hash{'disp_hour'}  = "&nbsp;";
		push(@hours,\%hash);
	}
	$hours = \@hours;

	# Setup title.
	if ($::preferences{'date_format'}{'value'}) {
		$datetitle = "$::language{$dow} $day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$datetitle = "$::language{$dow} $::language{ $MONTHS[$month-1] } $day, $year";
	}

	# Set some variables that determine what links are shown.
	$write = 0;
	$write = 1 if (($::username) && ($::username =~ /^($::permissions_ref->{'write'})$/));
	$write = 1 if ($::permissions_ref->{'type'} eq "open");

	# Assign parameters to template.
	$day_template->param(
		webcal          => $::webcalng_conf{'WEBCAL'},
		private         => $::webcalng_conf{'PRIVATE'},
		return_link     => $return_link,
		calendar        => $::calendar,
		datetitle       => $datetitle,
		month_name      => $::language{ $MONTHS[$month-1] },
		month_rows      => $month_rows,
		dows            => $dows,
		year            => $year,
		month           => $month,
		day             => $day,
		hours           => $hours,
		write           => $write,
		show_added      => $::preferences{'show_added'}{'value'},
		showcalname     => $::preferences{'show_cal_name'}{'value'},
		confirm_delete  => $::preferences{'confirm_delete'}{'value'},
		timezone        => $::preferences{'timezone'}{'value'},
		username        => $::username,
		sessionkey      => $::sessionkey,
                nextmonth       => $nextmonth,
                nextyear        => $nextyear,
                prevmonth       => $prevmonth,
                prevyear        => $prevyear,
		dayview         => '1',
	);

	# Write template output.
	$::output .= $day_template->output;

	return 1;
}

#
# Subroutine to display a list of items searched for.
#
sub search {
	my ($search_template,$file,$return_link,$search_return_link,$title,$searchtype,$searchtext,$year,$month,$day);
	my (%WEEK_START,@MONTHS,$startmark,$endmark,@calendars,$itemsref,$exceptions_ref,$time);
	my (@results,$results_ref);

	# Setup template.
	if ($::wap) {
		$file = "search.wml.tmpl";
	} else {
		$file = "search.tmpl";
	}
	$search_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form input.
	$return_link = $::cleanq->{'return_link'};
	$searchtype  = $::cleanq->{'searchtype'};
	$searchtext  = $::cleanq->{'searchtext'} || "";
	$year        = $::cleanq->{'year'};
	$month       = $::cleanq->{'month'};
	$day         = $::cleanq->{'day'};
	hard_error("No searchtype given") if (! $searchtype);

	# If no dates given, default to the current day.
	if ((! $day) && (! $month) && (! $year)) {
		($day,$month,$year) = (localtime)[3,4,5];
		$month++;
		$year += 1900;
	}

	# Fix return link.
	if ($return_link) {
		$search_return_link = $return_link;
		$search_return_link =~ s/--/=/g;
		$search_return_link =~ s/\.\./&/g;
		$search_return_link =~ s/\s+/+/g;
	} else {
		$return_link = "op--month..calendar--$::cgi_calendar";
	}

	# Decide what day of the week we start on, Sunday or Monday.
	if ($::preferences{'dow_start_mon'}{'value'}) {
		%WEEK_START = qw (Mon 0 Tue 1 Wed 2 Thu 3 Fri 4 Sat 5 Sun 6);
	} else {
		%WEEK_START = qw (Sun 0 Mon 1 Tue 2 Wed 3 Thu 4 Fri 5 Sat 6);
	}
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Determine start and end marks.
	if ($searchtype eq "year") {
		$title = $year;
		hard_error("Missing year") if (! $year);
		$startmark = timelocal "0","0","0","1","0",($year-1900);
		$endmark   = timelocal "0","0","0","1","0",($year-1900+1);
	} elsif ($searchtype eq "month") {
		my ($nextmonth,$nextyear);
		$title = "$::language{ $MONTHS[$month-1] } $year";
		hard_error("Missing month or year") if ((! $year) || (! $month));
		$nextmonth = $month + 1;
		$nextyear  =  $year;
		if ($nextmonth == 13) {
			$nextmonth = 1;
			$nextyear++;
		}
		$startmark = timelocal "0","0","0","1",($month-1),($year-1900);
		$endmark   = timelocal "0","0","0","1",($nextmonth-1),($nextyear-1900);
	} elsif ($searchtype eq "week") {
		my ($time,$dow,$stime,$syear,$smonth,$sday,$eyear,$emonth,$eday);
		hard_error("Missing day or month or year") if ((! $day) || (! $year) || (! $month));
		$time  = timelocal "0","0","12",$day,($month-1),($year-1900);
		$dow   = (split /\s+/, localtime($time))[0];
		$stime = $time - (86400 * $WEEK_START{$dow});	
		($sday,$smonth,$syear) = (localtime($stime))[3,4,5];
		$startmark             = timelocal "0","0","0",$sday,$smonth,$syear;
		$dow                   = (split /\s+/, localtime($startmark))[0];
		($eday,$emonth,$eyear) = (localtime($stime+(86400*7)))[3,4,5];
		$endmark               = timelocal "0","0","0",$eday,$emonth,$eyear;
		$syear += 1900;
		if ($::preferences{'date_format'}{'value'}) {
			$title = "$::language{'week_of'} $sday $::language{ $MONTHS[$smonth] } $syear";
		} else {
			$title = "$::language{'week_of'} $::language{ $MONTHS[$smonth] } $sday, $syear";
		}
	} elsif ($searchtype eq "day") {
		hard_error("Missing day or month or year") if ((! $day) || (! $year) || (! $month));
		if ($::preferences{'date_format'}{'value'}) {
			$title = "$day $::language{ $MONTHS[$month-1] } $year";
		} else {
			$title = "$::language{ $MONTHS[$month-1] } $day, $year";
		}
		$startmark = timelocal "0","0","0",$day,($month-1),($year-1900);
		$endmark   = $startmark + 86400;
	} else {
		hard_error("Invalid searchtype: $searchtype");
	}

	# Get list of items;
	@calendars = get_imported_calendars($::calendar);
	($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,$searchtext,undef,undef,@calendars);

	# Loop through the days in this search, formatting the data structure for each one 
	# to drive the output template.
	$time = $startmark + (12 * 60 * 60);	
	while ($time < $endmark) {
		my ($new_itemsref,%hash,$dow);
		($day,$month,$year) = (localtime($time))[3,4,5];
		$month++;
		$year += 1900;
		$dow   = (split /\s+/, localtime($time))[0];
		$new_itemsref = filter_items($itemsref,$day,$month,$year);
		$new_itemsref = apply_exceptions($new_itemsref,$exceptions_ref,$day,$month,$year);
		$hash{'year'}  = $year;
		$hash{'month'} = $month;
		$hash{'day'}   = $day;
		if ($::preferences{'date_format'}{'value'}) {
			$hash{'title'} = "$::language{$dow} $day $::language{ $MONTHS[$month-1] } $year";
		} else {
			$hash{'title'} = "$::language{$dow} $::language{ $MONTHS[$month-1] } $day, $year";
		}
		($hash{'items'},$hash{'hasitems'}) = format_items_byday($new_itemsref,$day,$month,$year);
		push(@results,\%hash);
		$time += 86400;
	}
	$results_ref = \@results;

	# Assign parameters to template.
	$search_template->param(
		webcal             => $::webcalng_conf{'WEBCAL'},
		calendar           => $::calendar,
		header_link        => $::preferences{'header_link'}{'value'},
		header_image       => $::preferences{'header_image'}{'value'},
		searchtext         => $searchtext,
		title              => $title,
		return_link        => $return_link,
		search_return_link => $search_return_link,
		results            => $results_ref,
		showcalname        => $::preferences{'show_cal_name'}{'value'},
		sessionkey         => $::sessionkey,
	);

	# Write template output.
	$::output .= $search_template->output;

	return 1;
}

#
# The following subroutine is called by the year(), month(), and day() subroutines.
# It returns a reference to an array of anonymous hashes that represent the weeks
# and days in a month, and the keys in those hashes to populate the template variables
# in the month.tmpl and month_box.tmpl templates.
#
sub month_data_setup {
	my ($itemsref,$exceptions_ref,$year,$month) = (@_);
	my (@month_data,@dow_data,$dow);
	my ($time,$day,@days,$day_counter,$flag,$x,$y,$z);
	my ($current_day,$current_month,$current_year);
	my (@DAYS_IN_MONTH,%MONTH_START);

	# Setup number of days in each month. Add a day to February if this is a leap year.
	@DAYS_IN_MONTH = qw(0 31 28 31 30 31 30 31 31 30 31 30 31);
	if ((($year % 4) == 0) && ((($year % 100) != 0) || (($year % 400) == 0))) {
		$DAYS_IN_MONTH[2]++;
	}

	# Decide what day of the week we start on, Sunday or Monday.
	if ($::preferences{'dow_start_mon'}{'value'}) {
		%MONTH_START = qw (Mon 0 Tue 1 Wed 2 Thu 3 Fri 4 Sat 5 Sun 6);
	} else {
		%MONTH_START = qw (Sun 0 Mon 1 Tue 2 Wed 3 Thu 4 Fri 5 Sat 6);
	}

	# Find out what the current date is.
	($current_day,$current_month,$current_year) = (localtime)[3,4,5];
	$current_month++;
	$current_year += 1900;

	# The next section of code determines where the days of the given month fall.
	$time = timelocal "0","0","12","1",($month-1),($year-1900);
	$day = (split /\s+/, localtime($time))[0];
	$z = 0;
	$day_counter = 1;
	for $x (0..5) {
		for $y (0..6) {
			$days[$x][$y] = 0;
			if ($MONTH_START{$day} == $z) {
				$days[$x][$y] = 1;
				$flag = 1;
				$z++;
				$day_counter++;
				next;
			}
			if ($DAYS_IN_MONTH[$month] == $day_counter) {
				$days[$x][$y] = $day_counter;
				$flag = 0;
				$day_counter++;
			}
			if ($flag) {
				$days[$x][$y] = $day_counter;
				$day_counter++;
			}
			$z++;
		}
	}

	# Populate the template variables for the days of the week.
	for $dow (sort { $MONTH_START{$a} <=> $MONTH_START{$b} } keys %MONTH_START) {
		my %hash;
		$hash{'dow'} = $::language{$dow};
		$hash{'short_dow'} = substr($::language{$dow},0,1);
		push(@dow_data,\%hash);	
	}	

	# Populate the template variables that will fill the day boxes.
	$x=0;
	for (@days) {
		my (%month_rows,@month_columns,$new_itemsref);
		$y = 0;
		for (@{ $days[$x] }) {
			my %month_columns;
			$month_columns{'day'}   = $days[$x][$y];
			$month_columns{'month'} = $month;
			$month_columns{'year'}  = $year;
			if (($MONTH_START{'Sun'} == $y) || ($MONTH_START{'Sat'} == $y)) {
				$month_columns{'weekend'} = 1;
			} else {
				$month_columns{'weekday'} = 1;
			}
			if (($days[$x][$y] == $current_day) && ($month == $current_month) && ($year == $current_year)) {
				$month_columns{'weekend'}     = 0;
				$month_columns{'weekday'}     = 0;
				$month_columns{'current_day'} = 1;
			}
			if ($days[$x][$y]) {
				$new_itemsref = filter_items($itemsref,$days[$x][$y],$month,$year);
				$new_itemsref = apply_exceptions($new_itemsref,$exceptions_ref,$days[$x][$y],$month,$year);
				($month_columns{'day_items'},$month_columns{'hasitems'}) = format_items_byday($new_itemsref,$days[$x][$y],$month,$year);
			}
			push(@month_columns,\%month_columns);
			$y++;
		}
		$month_rows{'month_columns'} = \@month_columns;
		$month_rows{'month_row'} = 0;
		if ((! $days[$x][0]) && (! $days[$x][6])) {
			$month_rows{'row'} = 0;
		} else {
			my ($daynum,$time,$yindex,$wday,$yday,$weeknum);
			$yindex = 0;
			for (@{ $days[$x] }) {
				$daynum = $_ if ($days[$x][$yindex]);
				$yindex++;
			}
			$time = timelocal "0","0","12",$daynum,($month-1),($year-1900);
			($wday,$yday) = (localtime($time))[6,7];
			$wday++;
			$yday++;
			if ($MONTH_START{'Mon'} == 0) {
				if ($wday == 1) {
					$wday = 7;
				} else {
					$wday = $wday - 1;
				}
			} 
			$yday += (7 - $wday);
			$weeknum = $yday / 7;
			$weeknum =~ s/\..*//;
			$weeknum++ if ($yday % 7);
			$month_rows{'row'}      = 1;
			$month_rows{'year'}     = $year;
			$month_rows{'month'}    = $month;
			$month_rows{'day'}      = $daynum; 
			$month_rows{'week_num'} = $weeknum;
		}
		push(@month_data,\%month_rows);
		$x++;
	}
	return (\@month_data,\@dow_data);
}

#
# Filter the list of items, and return a list that only contains items for the given day.
#
sub filter_items {
	my ($itemsref,$day,$month,$year) = (@_);
	my (%new_items,$time,$dow,$startmark,$endmark,$dateregexp,$itemid,$calendar);
	my ($yesterday,$yestday,$yestmonth,$yestyear,$yestdow,$yestregexp,$overnight);

	# Create the dateregexp and start/end marks.
	$time       = timelocal "0","0","12",$day,($month-1),($year-1900);
	$dow        = (split /\s+/, localtime($time))[0];
	$month      = "0" . $month if ($month !~ /\d{2}/);
	$day        = "0" . $day   if ($day !~ /\d{2}/);
	$dateregexp = "$month,$day,$dow";
	$startmark  = timelocal "0","0","0",$day,($month-1),($year-1900);
	$endmark    = $startmark + 86400;

	# Create a regexp to match items that occurred yesterday in case the item continues into today.
	$yesterday = $time - 86400;
	($yestday,$yestmonth,$yestyear) = (localtime($yesterday))[3,4,5];
	$yestmonth++;
	$yestyear+=1900;
	$yestdow    = (split /\s+/, localtime($yesterday))[0];
	$yestmonth  = "0" . $yestmonth if ($yestmonth !~ /\d{2}/);
	$yestday    = "0" . $yestday   if ($yestday !~ /\d{2}/);
	$yestregexp = "$yestmonth,$yestday,$yestdow";

	for $calendar (keys %$itemsref) {
		for $itemid (keys %{ $itemsref->{$calendar} }) {

			# First, see if the item falls in the date range we are looking at.
		   	if ((($itemsref->{$calendar}{$itemid}{'startdate'} >= $startmark) &&
		     	  ($itemsref->{$calendar}{$itemid}{'startdate'} < $endmark)) ||
		    	 (($itemsref->{$calendar}{$itemid}{'startdate'} <= $startmark) &&
		     	  ($itemsref->{$calendar}{$itemid}{'enddate'} >= $startmark))) {

				# See if this is an overnight item.
				$overnight = 0;
				$overnight = 1 if (($itemsref->{$calendar}{$itemid}{'starttime'} ne "") && ($itemsref->{$calendar}{$itemid}{'endtime'} ne "") && ($itemsref->{$calendar}{$itemid}{'starttime'} > $itemsref->{$calendar}{$itemid}{'endtime'}));

				# See if the item falls on today.  Need to do this check for repeating items.
				if ($dateregexp =~ /$itemsref->{$calendar}{$itemid}{'dateregexp'}/) {

					# We need to take some special care for monthlybydaylast or yearlybydaylast items.
					if (($itemsref->{$calendar}{$itemid}{'repeat'} eq "monthlybydaylast") || ($itemsref->{$calendar}{$itemid}{'repeat'} eq "yearlybydaylast")) {
						my ($this_time,$next_time,$next_month);
						$this_time  = timelocal "0","0","12",$day,($month-1),($year-1900);
						$next_time  = $this_time + (86400 * 7);
						$next_month = (localtime($next_time))[4];
						$next_month++;
						next if ($month == $next_month);
					} 
	
					# Also need to take special care for items that repeat every other week.
					if ($itemsref->{$calendar}{$itemid}{'repeat'} eq "weekly2") {
						my $startdow   = (split /\s+/, localtime($itemsref->{$calendar}{$itemid}{'startdate'}))[0];
						my (%dows)     = qw(Sun 0 Mon 1 Tue 2 Wed 3 Thu 4 Fri 5 Sat 6);
						my $weekstart  = $itemsref->{$calendar}{$itemid}{'startdate'} - (86400 * $dows{$startdow});
						my $dayspassed = (split /\./, (($time - $weekstart) / 86400))[0];
						my $weeknum    = (split /\./, ($dayspassed / 7))[0];
						my $weektype   = $weeknum % 2;
						next if $weektype;
					}

					# Here, check to see if the item is a private one, and whether to proceed if it is.
					next if (($::permissions_ref->{'type'} ne "open") && ($itemsref->{$calendar}{$itemid}{'visibility'} == 2) && (($::username ne $::permissions_ref->{'owner'}) || ($calendar ne $::calendar)));

					# If we get here, add the item unless it is the tail end of an overnight.
					if ((! $overnight) || (($itemsref->{$calendar}{$itemid}{'enddate'} + 43200) != $endmark)) {
						for my $key (keys %{ $itemsref->{$calendar}{$itemid} }) {
							$new_items{$calendar}{$itemid}{$key} = $itemsref->{$calendar}{$itemid}{$key};
						}
						# Change the end time if this is an overnight item.
						if ($overnight) {
							$new_items{$calendar}{$itemid}{'endtime'} = "2400";
						}
					}

				} 

				# See if the item falls on yesterday, but continued overnight until today.  We need to do
				# this is a seperate part of the loop because overnight items that repeat daily will match
				# both today (above) and yesterday (right here).
				if (($overnight) && ($yestregexp =~ /$itemsref->{$calendar}{$itemid}{'dateregexp'}/)) {

					# Make sure the item actually has a startdate that includes yesterday.
		   			next if ($itemsref->{$calendar}{$itemid}{'startdate'} >= $startmark);

					# We need to take some special care for monthlybydaylast or yearlybydaylast items.
					if (($itemsref->{$calendar}{$itemid}{'repeat'} eq "monthlybydaylast") || ($itemsref->{$calendar}{$itemid}{'repeat'} eq "yearlybydaylast")) {
						my ($this_time,$next_time,$next_month);
						$this_time  = timelocal "0","0","12",$yestday,($yestmonth-1),($yestyear-1900);
						$next_time  = $this_time + (86400 * 7);
						$next_month = (localtime($next_time))[4];
						$next_month++;
						next if ($month == $next_month);
					} 
	
					# Also need to take special care for items that repeat every other week.
					if ($itemsref->{$calendar}{$itemid}{'repeat'} eq "weekly2") {
						my $startdow   = (split /\s+/, localtime($itemsref->{$calendar}{$itemid}{'startdate'}))[0];
						my (%dows)     = qw(Sun 0 Mon 1 Tue 2 Wed 3 Thu 4 Fri 5 Sat 6);
						my $weekstart  = $itemsref->{$calendar}{$itemid}{'startdate'} - (86400 * $dows{$startdow});
						my $dayspassed = (split /\./, (($time - $weekstart) / 86400))[0];
						my $weeknum    = (split /\./, ($dayspassed / 7))[0];
						my $weektype   = $weeknum % 2;
						next if $weektype;
					}

					# Here, check to see if the item is a private one, and whether to proceed if it is.
					next if (($::permissions_ref->{'type'} ne "open") && ($itemsref->{$calendar}{$itemid}{'visibility'} == 2) && (($::username ne $::permissions_ref->{'owner'}) || ($calendar ne $::calendar)));

					# Since is an overnight one, give it a unique itemid by adding a "overnight" tag to it.  This
					# tag will be stripped later, but we need to do it now as repeating overnight events could 
					# have multiple entries in the same hash using the same itemid key, which would not work.	
					my $newitemid = $itemid;
					$newitemid .= "overnight";

					# If we get here, the item passed all checks and can be copied to the new_items hash.
					for my $key (keys %{ $itemsref->{$calendar}{$itemid} }) {
						$new_items{$calendar}{$newitemid}{$key} = $itemsref->{$calendar}{$itemid}{$key};
					}

					# Alter the starttime.
					$new_items{$calendar}{$newitemid}{'starttime'} = "0000";
				} 

			}
		}
	}

	return \%new_items;
}

#
# Apply exceptions to the list of items.
#
sub apply_exceptions {
	my ($itemsref,$exceptions_ref,$day,$month,$year) = (@_);
	my (%new_items,$time,$itemid,$real_itemid,$exceptionid,$calendar,$found_exception);
	my ($yesterday,$overnight);

	# Figure out timestamp for day we are looking at.
	$time      = timelocal "0","0","12",$day,($month-1),($year-1900);
	$yesterday = $time - 86400;

	# Loop through the list of items.  If an item has a exceptiondate equal to the
	# day we are looking at, apply the exception.
	for $calendar (keys %$itemsref) {
		for $itemid (keys %{ $itemsref->{$calendar} }) {
			$found_exception = 0;
			$overnight       = 0;
			$overnight       = 1 if ($itemid =~ /overnight/);
			$real_itemid     = $itemid;
			$real_itemid     =~ s/overnight//;
			for $exceptionid (keys %{ $exceptions_ref->{$calendar}{$real_itemid} }) {

				# If this is an overnight item, we have to match the exception based on yesterdays date.
				if ($overnight) {
					next unless ($exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{'exceptiondate'} == $yesterday);
				} else {
					next unless ($exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{'exceptiondate'} == $time);
				}
	
				# We found a valid exception if we get here.	
				$found_exception++;

				# Do not copy in items that have no description - these have been removed.
				next unless ($exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{'description'} ne "");

				# Do a special check if it was an overnight item - make sure it is still an overnight item.
				next unless ((! $overnight) || ($exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{'starttime'} ne "") && ($exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{'endtime'} ne "") && ($exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{'starttime'} > $exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{'endtime'}));

				# If there is a description, then copy in the exception over the base of the item.
				for my $key (keys %{ $itemsref->{$calendar}{$itemid} }) {
					$new_items{$calendar}{$itemid}{$key} = $itemsref->{$calendar}{$itemid}{$key};
				}
				for my $key (keys %{ $exceptions_ref->{$calendar}{$real_itemid}{$exceptionid} }) {
					next if (($overnight) && ($key eq "starttime")); # The starttime is set to midnight for an overnight.
					$new_items{$calendar}{$itemid}{$key} = $exceptions_ref->{$calendar}{$real_itemid}{$exceptionid}{$key};
				}

				# Add the exceptionid for the item.
				$new_items{$calendar}{$itemid}{'exceptionid'} = $exceptionid;

			}

			# If no exception was found, copy the item as is into the new items hash.
			if (! $found_exception) {
				for my $key (keys %{ $itemsref->{$calendar}{$itemid} }) {
					$new_items{$calendar}{$itemid}{$key} = $itemsref->{$calendar}{$itemid}{$key};
				}
			}
		}
	}

	return \%new_items;
}

#
# This subroutine creates a sorted data structure of items that
# can drive a loop in a display template.  This is used to create
# a data structure for a whole day's worth of items, for the week
# or month views.
#
sub format_items_byday {
	my ($itemsref,$day,$month,$year) = (@_);
	my (@items,$itemid,$calendar,%sorthash,$sortkey,$hasitems);

	# We have to do some funky stuff here to get all items from all calendars to sort correctly
	# by time of day that they start at.  First, we create a new hash which only contains the
	# starttimes and descriptions (we use both of these in the sort). The key is a combination
	# of the calendar name and the itemid.  This hash is easy to sort, and we then extract
	# the real information from the original hashes as we loop through.
	for $calendar (keys %$itemsref) {
		for $itemid (keys %{ $itemsref->{$calendar} }) {
			$sortkey = $calendar . ";" . $itemid;
			$sorthash{$sortkey}{'starttime'}   = $itemsref->{$calendar}{$itemid}{'starttime'};
			$sorthash{$sortkey}{'description'} = $itemsref->{$calendar}{$itemid}{'description'};
		}
	}

	# Loop through the items, populating the data structure.
	$hasitems = 0;
	for $sortkey (sort { time_sort(\%sorthash) } keys %sorthash) {
		my (%hash);
		($calendar,$itemid)   = split /;/, $sortkey;
		$hash{'overnight'}    = 0;
		$hash{'overnight'}    = 1 if ($itemid =~ /overnight/);
		$hash{'itemid'}       = $itemid;
		$hash{'itemid'}       =~ s/overnight//;
		$hash{'exceptionid'}  = $itemsref->{$calendar}{$itemid}{'exceptionid'};
		$hash{'description'}  = $itemsref->{$calendar}{$itemid}{'description'};
		$hash{'starttime'}    = format_time($itemsref->{$calendar}{$itemid}{'starttime'});
		$hash{'endtime'}      = format_time($itemsref->{$calendar}{$itemid}{'endtime'});
		$hash{'hyperlink'}    = $itemsref->{$calendar}{$itemid}{'hyperlink'};
		$hash{'hyperlink'}    = "http://" . $hash{'hyperlink'} if (($hash{'hyperlink'}) && ($hash{'hyperlink'} !~ /(:\/\/)|(:\\\\)|(^\\)|(^\/)/));
		$hash{'notes'}        = $itemsref->{$calendar}{$itemid}{'notes'};
		$hash{'meetingid'}    = $itemsref->{$calendar}{$itemid}{'meetingid'};
		$hash{'user'}         = $itemsref->{$calendar}{$itemid}{'user'} || "";
		$hash{'origcalendar'} = $itemsref->{$calendar}{$itemid}{'origcalendar'} || "";
		$hash{'notecalendar'} = $itemsref->{$calendar}{$itemid}{'origcalendar'} || $::calendar;
		$hash{'busy'}         = 1 if (($::permissions_ref->{'type'} ne "open") && ($itemsref->{$calendar}{$itemid}{'visibility'}) && ((! $::username) || ($::username ne $itemsref->{$calendar}{$itemid}{'user'})));
		$hash{'lastitem'}     = 0;
		$hash{'day'}          = $day;
		$hash{'month'}        = $month;
		$hash{'year'}         = $year;
		if ($hash{'meetingid'}) {
			$hash{'meetingcal'} = $hash{'meetingid'};
			$hash{'meetingcal'} =~ s/.*?;//;
		}
		push (@items,\%hash);
		$hasitems++;
	}

	# Because loop_context_vars doesn't seem to work in nested loops.
	if ($#items >= 0) {
		$items[$#items]->{'lastitem'}++;
	}

	return (\@items,$hasitems);
}

#
# This subroutine creates a sorted data structure of items that
# can drive a loop in a display template.  This is used to create
# a data structure for an hour's worth of items, for the day view.
#
sub format_items_byhour {
	my ($itemsref,$starthour,$endhour,$day,$month,$year) = (@_);
	my (@items,$itemid,$hasitems,$calendar,$sortkey,%sorthash);
	$hasitems = 0;
	
	# We have to do some funky stuff here to get all items from all calendars to sort correctly
	# by time of day that they start at.  First, we create a new hash which only contains the
	# starttimes and descriptions (we use both of these in the sort). The key is a combination
	# of the calendar name and the itemid.  This hash is easy to sort, and we then extract
	# the real information from the original hashes as we loop through.
	for $calendar (keys %$itemsref) {
		for $itemid (keys %{ $itemsref->{$calendar} }) {
			$sortkey = $calendar . ";" . $itemid;
			$sorthash{$sortkey}{'starttime'}   = $itemsref->{$calendar}{$itemid}{'starttime'};
			$sorthash{$sortkey}{'description'} = $itemsref->{$calendar}{$itemid}{'description'};
		}
	}

	if (! defined $starthour) {
		# If starthour is null, then look for items without starttimes, and return.
		for $sortkey (sort { time_sort(\%sorthash) } keys %sorthash) {
			($calendar,$itemid)   = split /;/, $sortkey;
			if ($itemsref->{$calendar}{$itemid}{'starttime'} eq "") {
				my (%hash,$write);
				$hash{'overnight'}    = 0;
				$hash{'overnight'}    = 1 if ($itemid =~ /overnight/);
				$hash{'itemid'}       = $itemid;
				$hash{'itemid'}       =~ s/overnight//;
				$hash{'exceptionid'}  = $itemsref->{$calendar}{$itemid}{'exceptionid'};
				$hash{'description'}  = $itemsref->{$calendar}{$itemid}{'description'};
				$hash{'starttime'}    = format_time($itemsref->{$calendar}{$itemid}{'starttime'});
				$hash{'endtime'}      = format_time($itemsref->{$calendar}{$itemid}{'endtime'});
				$hash{'hyperlink'}    = $itemsref->{$calendar}{$itemid}{'hyperlink'};
				$hash{'hyperlink'}    = "http://" . $hash{'hyperlink'} if (($hash{'hyperlink'}) && ($hash{'hyperlink'} !~ /(:\/\/)|(:\\\\)|(^\\)|(^\/)/));
				$hash{'notes'}        = $itemsref->{$calendar}{$itemid}{'notes'};
				$hash{'meetingid'}    = $itemsref->{$calendar}{$itemid}{'meetingid'};
				$hash{'user'}         = $itemsref->{$calendar}{$itemid}{'user'};
				$hash{'busy'}         = 1 if (($::permissions_ref->{'type'} ne "open") && ($itemsref->{$calendar}{$itemid}{'visibility'}) && ((! $::username) || ($::username ne $itemsref->{$calendar}{$itemid}{'user'})));
				$hash{'day'}          = $day;
				$hash{'month'}        = $month;
				$hash{'year'}         = $year;
				$hash{'multiple'}     = 1 if ($itemsref->{$calendar}{$itemid}{'repeat'});
				$hash{'origcalendar'} = $itemsref->{$calendar}{$itemid}{'origcalendar'} || "";
				$hash{'notecalendar'} = $itemsref->{$calendar}{$itemid}{'origcalendar'} || $::calendar;
				# The edit key determines if the delete/edit links should be show for this item.
				$write        = 0;
				$write        = 1 if (($::username) && ($::username =~ /^($::permissions_ref->{'write'})$/));
				$write        = 1 if ($::permissions_ref->{'type'} eq "open");
				$write        = 0 if ($hash{'busy'});
				$hash{'edit'} = 1 if (($write) && ((! $::preferences{'only_add_remove'}{'value'}) || ($itemsref->{$calendar}{$itemid}{'user'} eq $::username)));
				$hash{'edit'} = 0 if (($itemsref->{$calendar}{$itemid}{'meetingid'}) && (! $itemsref->{$calendar}{$itemid}{'remindwho'}));
				if ($hash{'meetingid'}) {
					$hash{'meetingcal'} = $hash{'meetingid'};
					$hash{'meetingcal'} =~ s/.*?;//;
				}
				push (@items,\%hash);
				$hasitems++;
			}
		}
	} else {
		# If starttime is not null, loop through the items, populating the data structure
		# with items that match the date regexp and are in the given timeframe.  
		for $sortkey (sort { time_sort(\%sorthash) } keys %sorthash) {
			($calendar,$itemid)   = split /;/, $sortkey;
			if (($itemsref->{$calendar}{$itemid}{'starttime'} =~ /\d+/) &&
	    		($itemsref->{$calendar}{$itemid}{'starttime'} >= $starthour) &&
	    		($itemsref->{$calendar}{$itemid}{'starttime'} < $endhour))
			{
				my (%hash,$write);
				$hash{'overnight'}    = 0;
				$hash{'overnight'}    = 1 if ($itemid =~ /overnight/);
				$hash{'itemid'}       = $itemid;
				$hash{'itemid'}       =~ s/overnight//;
				$hash{'exceptionid'}  = $itemsref->{$calendar}{$itemid}{'exceptionid'};
				$hash{'description'}  = $itemsref->{$calendar}{$itemid}{'description'};
				$hash{'starttime'}    = format_time($itemsref->{$calendar}{$itemid}{'starttime'});
				$hash{'endtime'}      = format_time($itemsref->{$calendar}{$itemid}{'endtime'});
				$hash{'hyperlink'}    = $itemsref->{$calendar}{$itemid}{'hyperlink'};
				$hash{'hyperlink'}    = "http://" . $hash{'hyperlink'} if (($hash{'hyperlink'}) && ($hash{'hyperlink'} !~ /(:\/\/)|(:\\\\)|(^\\)|(^\/)/));
				$hash{'notes'}        = $itemsref->{$calendar}{$itemid}{'notes'};
				$hash{'meetingid'}    = $itemsref->{$calendar}{$itemid}{'meetingid'};
				$hash{'user'}         = $itemsref->{$calendar}{$itemid}{'user'};
				$hash{'busy'}         = 1 if (($::permissions_ref->{'type'} ne "open") && ($itemsref->{$calendar}{$itemid}{'visibility'}) && ((! $::username) || ($::username ne $itemsref->{$calendar}{$itemid}{'user'})));
				$hash{'day'}          = $day;
				$hash{'month'}        = $month;
				$hash{'year'}         = $year;
				$hash{'multiple'}     = 1 if ($itemsref->{$calendar}{$itemid}{'repeat'});
				$hash{'origcalendar'} = $itemsref->{$calendar}{$itemid}{'origcalendar'} || "";
				$hash{'notecalendar'} = $itemsref->{$calendar}{$itemid}{'origcalendar'} || $::calendar;
				# The edit key determines if the delete/edit links should be show for this item.
				$write        = 0;
				$write        = 1 if (($::username) && ($::username =~ /^($::permissions_ref->{'write'})$/));
				$write        = 1 if ($::permissions_ref->{'type'} eq "open");
				$write        = 0 if ($hash{'visibility'});
				$hash{'edit'} = 1 if (($write) && ((! $::preferences{'only_add_remove'}{'value'}) || ($itemsref->{$calendar}{$itemid}{'user'} eq $::username)));
				$hash{'edit'} = 0 if (($itemsref->{$calendar}{$itemid}{'meetingid'}) && (! $itemsref->{$calendar}{$itemid}{'remindwho'}));
				if ($hash{'meetingid'}) {
					$hash{'meetingcal'} = $hash{'meetingid'};
					$hash{'meetingcal'} =~ s/.*?;//;
				}
				push (@items,\%hash);
				$hasitems++;
			}
		}
	}

	if ($hasitems) {
		return \@items;
	} else {
		return "";
	}
}

#
# Subroutine to display preference selection for per calendar prefernces
# and admin preferences.
#
sub preferences {
	my ($preferences_template,$file,$adminprefs,$category,$cgi_return_link,$return_link);
	my ($interaction,$layout_colors,$localization,$reminders,$admin);
	my ($key,$preferences,@preferences,$saved,$width);

	# Setup template.
	if ($::wap) {
		$file = "preferences.wml.tmpl";
	} else {
		$file = "preferences.tmpl";
	}
	$preferences_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);
	
	# Get form input.
	$adminprefs  = $::cleanq->{'adminprefs'};
	$category    = $::cleanq->{'category'};
	$return_link = $::cleanq->{'return_link'};
	$adminprefs  = "" if (! $adminprefs);
	$category    = "interaction" if (! $category);
	$return_link = "op--month..calendar--$::cgi_calendar" if (($::calendar) && (! $return_link));
	$interaction = $layout_colors = $localization = $reminders = $admin = 0;

	# See if we just saved some preferences.
	if ($::op eq "save_preferences") {
		$saved = 1;
	} else {
		$saved = 0;
	}

	# Fix return link.
	if ($return_link) {
		$cgi_return_link = $return_link;
		$cgi_return_link =~ s/\s+/+/g;
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Decide what section we are working on.
	if ($category eq "interaction") {
		$interaction++;
	} elsif ($category eq "layout_colors") {
		$layout_colors++;
	} elsif ($category eq "localization") {
		$localization++;
	} elsif ($category eq "reminders") {
		$reminders++;
	} elsif ($category eq "admin") {
		$admin++;
	} else {
		hard_error("Bad preference category: $category");
	}

	# Build data structure of preference keys suitable for template loop.
	for $key (sort { $::preferences{$a}{'order'} <=> $::preferences{$b}{'order'} } keys %::preferences) {
		if ($::preferences{$key}{'category'} eq $category) {
			if (($adminprefs) || ($::preferences{$key}{'usermod'})) {
				my $hashref = \%{ $::preferences{$key} };
				push(@preferences, $hashref);
			}
		}
	}
	$preferences = \@preferences;

	# Set width for table cells in header.
	if ($adminprefs) {
		$width = "20%";
	} else {
		$width = "25%";
	}
	
	# Assign parameters to template.
	$preferences_template->param(
		webcal          => $::webcalng_conf{'WEBCAL'},
		lang_category   => $::language{$category},
		interaction     => $interaction,
		layout_colors   => $layout_colors,
		localization    => $localization,
		reminders       => $reminders,
		admin           => $admin,
		username        => $::username,
		calendar        => $::calendar,
		return_link     => $return_link,
		preferences     => $preferences,
		adminprefs      => $adminprefs,
		category        => $category,
		saved           => $saved,
		cgi_return_link => $cgi_return_link,
		width           => $width,
		sessionkey      => $::sessionkey,
	);

	# Write template output.
	$::output .= $preferences_template->output;

	return 1;
}

#
# The color browser page, which is only called from the preferences.
# 
sub colors {
	my ($colors_template,$file,$key,@colorlist,$color,@colors,$colors_ref,$counter);

	# Setup template.
	if ($::wap) {
		$file = "colors.wml.tmpl";
	} else {
		$file = "colors.tmpl";
	}
	$colors_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form input.
	$key = $::cleanq->{'key'};

	# Build list of colors to use.
	my (@x,@y,@z,$x,$y,$z);
	@x = qw (00 33 66 99 CC FF);
	@y = qw (00 33 66 99 CC FF);
	@z = qw (00 33 66 99 CC FF);
	for $x (@x)  {
		for $y (@y) {
			for $z (@z) {
				my $hexcode = "#" . $x . $y . $z;
				push(@colorlist,$hexcode);
			}
		}
	}

	# Build data structure to drive template.
	$counter = 0;
	for $color (@colorlist) {
		my %hash;
		$counter++;
		if ($counter == 1) {
			$hash{'startrow'} = 1;
		} else {
			$hash{'startrow'} = 0;
		}
		if ($color eq "#FFFFFF") {
			$hash{'lastcolor'} = 1;
			my (@lasttds,$lasttds_ref);
			for (1..9) {
				my %hash;
				$hash{'color'} = $color;
				push(@lasttds,\%hash);
				$counter++;
			}
			$lasttds_ref     = \@lasttds;
			$hash{'lasttds'} = $lasttds_ref;	
		} else {
			$hash{'color'} = $color;
		}
		if ($counter == 16) {
			$hash{'endrow'} = 1;
			$counter        = 0;
		} else {
			$hash{'endrow'} = 0;
		}
		push(@colors,\%hash);
	}
	$colors_ref = \@colors;

	# Assign parameters to template.
	$colors_template->param(
		colors     => $colors_ref,
		key        => $key,
	);

	# Write template output.
	$::output .= $colors_template->output;

	return 1;
}

#
# Display the list of available calendars.
#
sub list {
	my ($list_template,$file,@list,@list_array,$list_ref,$next,$submit);

	# Setup template.
	if ($::wap) {
		$file = "list.wml.tmpl";
	} else {
		$file = "list.tmpl";
	}
	$list_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form input.
	$next = $::cleanq->{'next'} || "view";
	
	# Set variables based on next action.
	if ($next eq "modify") {
		$submit = $::language{'modify_cal'};
	} else {
		$submit = $::language{'view_cal'};
	}

	# Get list of avaiable calendars from i/o module.
	@list = webcalng_io::get_calendar_list();
	hard_error("Calendar limit exceeded.") if ($#list >= 2);

	# See how many calendars we found, if they were searching.
	if ($::calendar) {
		# If we found a matching calendar, proceed directly to it.
		if (($#list == 0) && ($list[0] eq $::calendar)) {
			my ($calendar,$url);
			$calendar = $list[0];
			$calendar =~ s/\s+/+/g;
			$url      = $::webcalng_conf{'WEBCAL'} . "?op=month&calendar=$calendar";
			redirect($url);
		}
	}

	# Build data structure for template.
	for (sort { lc($a) cmp lc($b) } @list) {
		my %hash;
		$hash{'calendar'} = $_;
		push(@list_array,\%hash);
	}
	$list_ref = \@list_array;

	# Assign parameters to template.
	$list_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		list       => $list_ref,
		submit     => $submit,
		next       => $next,
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $list_template->output;

	return 1;
}

#
# Display the form for adding a new calendar.
#
sub addcal {
	my ($addcal_template,$file,@list,$type,@types,$types);

	# Setup template.
	if ($::wap) {
		$file = "addcal.wml.tmpl";
	} else {
		$file = "addcal.tmpl";
	}
	$addcal_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Populate the template variables for the calendar type options.
	@list = qw(open public private);
	for $type (@list) {
		my %hash;
		$hash{'value'} = $type;
		$hash{'text'}  = $::language{$type};
		push(@types,\%hash);	
	}	
	$types = \@types;

	# Get list of avaiable calendars from i/o module.
	@list = webcalng_io::get_calendar_list();
	hard_error("You can only create two calendars in the trial version.") if ($#list >= 1);

	# Assign parameters to template.
	$addcal_template->param(
		types      => $types,
		webcal     => $::webcalng_conf{'WEBCAL'},
		sessionkey => $::sessionkey,
	);

	# Write template output.
	$::output .= $addcal_template->output;

	return 1;
}

#
# Finish adding a new calendar.
#
sub addcal2 {
	my ($name,$type,$username,$password,$password2,$user_exists,@list);
	
	# Get form input.
	$name      = $::cleanq->{'name'};
	$type      = $::cleanq->{'type'};
	$username  = $::cleanq->{'username'};
	$password  = $::cleanq->{'password'};
	$password2 = $::cleanq->{'password2'};

	# Untaint calendar name.
	if ($name =~ /^([\w,\-,\s,\.]{1,40})$/) {
		$name = $1;
		$name =~ s/^\s+//;
		$name =~ s/\s+$//;
	} else {
		soft_error('invalidcalname');
	}
	soft_error('invalidcalname')         if ($name =~ /^\./);
	soft_error('invalidcalname')         if ($name eq "admin");
	hard_error("Invalid calendar type.") unless ($type =~ /^open|public|private$/);
	soft_error('calendarexists')         if (webcalng_io::calendar_exists($name));
	soft_error('nopasswordgiven')        if (($type ne "open") && ($password =~ /^$/));
	soft_error('invalidusername')        if (($type ne "open") && ($username !~ /^[\w,\d]{1,20}$/));

	# Create a new account, if we need to.
	if ($type ne "open") {
		$user_exists = webcalng_auth::lookup_user($username);
		if ($user_exists) {
			if (! webcalng_auth::validate_login($username,$password)) {
				soft_error('invalidpassword');
			}
		} else {
			soft_error('passwordmismatch') if ($password ne $password2);
			webcalng_auth::create_user($username,$password);
		}
	}

	# Get list of avaiable calendars from i/o module.
	@list = webcalng_io::get_calendar_list();
	hard_error("Too many calendars.") if ($#list > 0);


	# If we actually get this far, call i/o subroutine to create the calendar.
	webcalng_io::create_calendar($name,$type,$username);
	
	return 1;
}

#
# Display a list of calendars for removal.
#
sub delcal {
	my ($delcal_template,$file,@list,@list_array,$list_ref);

	# Setup template.
	if ($::wap) {
		$file = "delcal.wml.tmpl";
	} else {
		$file = "delcal.tmpl";
	}
	$delcal_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get list of avaiable calendars from i/o module.
	@list = webcalng_io::get_calendar_list();

	# Build data structure for template.
	for (sort @list) {
		my %hash;
		$hash{'calendar'} = $_;
		push(@list_array,\%hash);
	}
	$list_ref = \@list_array;

	# Assign parameters to template.
	$delcal_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		list       => $list_ref,
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $delcal_template->output;

	return 1;
}

#
# Remove the given calendar.
#
sub delcal2 {
	my ($delcal2_template,$file,$itemsref,$itemid);

	# Setup template.
	if ($::wap) {
		$file = "delcal2.wml.tmpl";
	} else {
		$file = "delcal2.tmpl";
	}
	$delcal2_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);
	
	# First, delete all occurrences of any meetings from this calendar that may have been added
	# to other calendars.
	($itemsref,undef) = webcalng_io::get_items('0','9999999999',undef,undef,undef,$::calendar);
	for $itemid (keys %{ $itemsref->{$::calendar} }) {
		remove_meeting($itemsref->{$::calendar}{$itemid},'1','0') if ($itemsref->{$::calendar}{$itemid}{'meetingid'});
	}

	# Now just call the delete calendar subroutine from the i/o module.
	# If it returns, that means it succeeded, so display the template.
	webcalng_io::remove_calendar($::calendar);

	# Assign parameters to template.
	$delcal2_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $delcal2_template->output;

	return 1;
}

#
# Display form with calendar parameters that can be changed by the admin.
#
sub modify {
	my ($modify_template,$file,$open,$public,$private);

	# Setup template.
	if ($::wap) {
		$file = "modify.wml.tmpl";
	} else {
		$file = "modify.tmpl";
	}
	$modify_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);
	
	# Set variables for calendar type.
	if ($::permissions_ref->{'type'} eq "open") {
		$open    = 1;
		$public  = 0;
		$private = 0;
	} elsif ($::permissions_ref->{'type'} eq "public") {
		$open    = 0;
		$public  = 1;
		$private = 0;
	} else {
		$open    = 0;
		$public  = 0;
		$private = 1;
	}

	# Assign parameters to template.
	$modify_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		oldcal     => $::calendar,
		oldowner   => $::permissions_ref->{'owner'},
		oldtype    => $::permissions_ref->{'type'},
		open       => $open,
		public     => $public,
		private    => $private,
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $modify_template->output;

	return 1;
}

#
# Make the updates to the calendar.
#
sub modify2 {
	my ($modify2_template,$file,$cal,$oldcal,$owner,$oldowner,$type,$oldtype);

	# Setup template.
	if ($::wap) {
		$file = "modify2.wml.tmpl";
	} else {
		$file = "modify2.tmpl";
	}
	$modify2_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form input.
	$cal      = $::cleanq->{'cal'};
	$oldcal   = $::cleanq->{'oldcal'};	
	$owner    = $::cleanq->{'owner'};	
	$oldowner = $::cleanq->{'oldowner'};	
	$type     = $::cleanq->{'type'};	
	$oldtype  = $::cleanq->{'oldtype'};	

	# Sanity checks.
	soft_error('invalidcalname')         unless ($cal =~ /^([\w,\-,\s,\.]{1,40})$/);
	soft_error('invalidcalname')         if ($cal =~ /^\./);
	soft_error('invalidcalname')         if ($cal eq "admin");
	hard_error("Invalid calendar type.") unless ($type =~ /^open|public|private$/);
	soft_error('calendarexists')         if (($cal ne $oldcal) && (webcalng_io::calendar_exists($cal)));
	soft_error('invalidusername')        if (($type ne "open") && ($owner !~ /^[\w,\d]{1,20}$/));
	soft_error('user_doesnot_exist')     if (($type ne "open") && (! webcalng_auth::lookup_user($owner)));

	# Make changes as needed.
	$owner = "" if ($type eq "open");
	if (($owner ne $oldowner) || ($type ne $oldtype)) {
		webcalng_io::update_calinfo($oldcal,$owner,$type);
	}
	if ($cal ne $oldcal) {
		webcalng_io::update_calname($oldcal,$cal,$type,$owner);
	}
	
	# Assign parameters to template.
	$modify2_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $modify2_template->output;

	return 1;
}

#
# Display form for adding a new user.
#
sub adduser {
	my ($adduser_template,$file);

	# Setup template.
	if ($::wap) {
		$file = "adduser.wml.tmpl";
	} else {
		$file = "adduser.tmpl";
	}
	$adduser_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Assign parameters to template.
	$adduser_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		sessionkey => $::sessionkey,
	);

	# Write template output.
	$::output .= $adduser_template->output;

	return 1;
}

#
# Add the new user.
#
sub adduser2 {
	my ($adduser2_template,$file,$username,$password,$password2,$user_exists);

	# Setup template.
	if ($::wap) {
		$file = "adduser2.wml.tmpl";
	} else {
		$file = "adduser2.tmpl";
	}
	$adduser2_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);
	
	# Get form input.
	$username  = $::cleanq->{'username'};
	$password  = $::cleanq->{'password'};
	$password2 = $::cleanq->{'password2'};

	# Sanity checks.
	$user_exists = webcalng_auth::lookup_user($username);
	soft_error('nopasswordgiven')  if ($password =~ /^$/);
	soft_error('passwordmismatch') if ($password ne $password2);
	soft_error('user_exists')      if ($user_exists);
	soft_error('invalidusername')  if ($username !~ /^[\w,\d]{1,20}$/);

	# Create a new account.
	webcalng_auth::create_user($username,$password);

	# Assign parameters to template.
	$adduser2_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		sessionkey => $::sessionkey,
	);

	# Write template output.
	$::output .= $adduser2_template->output;

	return 1;
}

#
# Display form for removing a user.
#
sub deluser {
	my ($deluser_template,$file,@list,@list_array,$list_ref);

	# Setup template.
	if ($::wap) {
		$file = "deluser.wml.tmpl";
	} else {
		$file = "deluser.tmpl";
	}
	$deluser_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get list of users from auth i/o module.
	@list = webcalng_auth::get_user_list();

	# Build data structure for template.
	for (sort @list) {
		my %hash;
		next if /^admin$/;
		$hash{'user'} = $_;
		push(@list_array,\%hash);
	}
	$list_ref = \@list_array;

	# Assign parameters to template.
	$deluser_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		list       => $list_ref,
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $deluser_template->output;

	return 1;
}

#
# Delete the given user.
#
sub deluser2 {
	my ($deluser2_template,$file,$user);

	# Setup template.
	if ($::wap) {
		$file = "deluser2.wml.tmpl";
	} else {
		$file = "deluser2.tmpl";
	}
	$deluser2_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get user and call subroutine from auth i/o module to remove.
	$user = $::cleanq->{'user'};
	hard_error("Can not remove admin user") if ($user eq "admin");
	webcalng_auth::remove_user($user);

	# Assign parameters to template.
	$deluser2_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $deluser2_template->output;

	return 1;
}

#
# Display form for changing a password.
#
sub changepw {
	my ($changepw_template,$file);

	# Setup template.
	if ($::wap) {
		$file = "changepw.wml.tmpl";
	} else {
		$file = "changepw.tmpl";
	}
	$changepw_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Make sure they are logged in.
	hard_error("You must be logged in to change your password.") if (! $::username);

	# Assign parameters to template.
	$changepw_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		username   => $::username,
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $changepw_template->output;

	return 1;
}

#
# Change the users password.
#
sub changepw2 {
	my ($changepw2_template,$file,$currentpassword,$password,$password2,$user_exists);

	# Setup template.
	if ($::wap) {
		$file = "changepw2.wml.tmpl";
	} else {
		$file = "changepw2.tmpl";
	}
	$changepw2_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form input.
	$currentpassword = $::cleanq->{'currentpassword'};
	$password        = $::cleanq->{'password'};
	$password2       = $::cleanq->{'password2'};

	# Make sure the information they gave is correct.
	$user_exists = webcalng_auth::lookup_user($::username);
	hard_error("$::username does not exist!") if ! ($user_exists);
	soft_error('invalidpassword') if (! webcalng_auth::validate_login($::username,$currentpassword));
	soft_error('passwordmismatch') if ($password ne $password2);
	
	# If we make it this far, change the password.
	webcalng_auth::remove_user($::username);
	webcalng_auth::create_user($::username,$password);

	# Assign parameters to template.
	$changepw2_template->param(
		webcal     => $::webcalng_conf{'WEBCAL'},
		sessionkey => $::sessionkey,
	);	

	# Write template output.
	$::output .= $changepw2_template->output;

	return 1;
}

#
# Display the form to add an item.
#
sub additem {
	my ($additem_template,$file,$year,$month,$day,$time,$dow,$weeknumber,$lastweek);
	my ($address_ref,$addressid,@addresses,$addresses_ref,@endyears,$endyears_ref);
	my (@MONTHS,$time24,$datetitle,%checkday);
	my ($shour,$sampm,$shourvalue,$shourtext,$sampmvalue,$sampmtext);
	my ($endday,$endmonthvalue,$endmonthtext,$endyear);

	# Setup template.
	if ($::wap) {
		$file = "additem.wml.tmpl";
	} else {
		$file = "additem.tmpl";
	}
	$additem_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Determine the day we are working on.
	$year  = $::cleanq->{'year'};
	$month = $::cleanq->{'month'};
	$day   = $::cleanq->{'day'};
        $shour = $::cleanq->{'shour'};
	if ((! $year) || (! $month) || (! $day)) {
		($day,$month,$year) = (localtime)[3,4,5];
		$month++;
		$year += 1900;
	}
	$time = timelocal "0","0","12",$day,($month-1),($year-1900);
	$dow  = (split /\s+/, localtime($time))[0];

	# Setup end date data.
	($endday,$endmonthvalue,$endyear) = (localtime($time))[3,4,5];
	$endmonthvalue++;
	$endyear     += 1900;
	$endmonthtext = $MONTHS[$endmonthvalue-1];
	$endmonthtext = $::language{"short" . $endmonthtext};

	# Determine what week of the month it is.
	if (($day >= 1) && ($day <= 7)) {
		$weeknumber = $::language{'first'};
		$lastweek   = 0;
	} elsif (($day >= 8) && ($day <= 14)) {
		$weeknumber = $::language{'second'};
		$lastweek   = 0;
	} elsif (($day >= 15) && ($day <= 21)) {
		$weeknumber = $::language{'third'};
		$lastweek   = 0;
	} else {
		$weeknumber = $::language{'fourth'};
		$lastweek   = 0;
		# See if we are really on the last dow of the month or not.
		my $this_time  = timelocal "0","0","12",$day,($month-1),($year-1900);
		my $next_time  = $this_time + (86400 * 7);
		my $next_month = (localtime($next_time))[4];
		$next_month++;
		$lastweek   = 1 if ($month != $next_month);
	}

	# Determine what day of week to automatically check.
	$checkday{'Sun'} = $checkday{'Mon'} = $checkday{'Tue'} = $checkday{'Wed'} = $checkday{'Thu'} = $checkday{'Fri'} = $checkday{'Sat'} = 0;
	$checkday{$dow}  = 1;
	
	# Get list of email addresses in our address book.
	# First, add calendar owner address if they have entered one.
	if ($::preferences{'email'}{'value'}) {
		my %hash;
		$hash{'text'}  = $::language{'calendarowner'};
		$hash{'value'} = $::preferences{'email'}{'value'} . ";" . $::calendar;
		push(@addresses,\%hash);
	}
	# Now, add all addresses in our address book.
	$address_ref = webcalng_io::get_address();
	for $addressid (keys %$address_ref) {
		my %hash;
		$hash{'text'}  = $address_ref->{$addressid}{'displayname'};
		$hash{'value'} = $address_ref->{$addressid}{'emailaddress'};
		push(@addresses,\%hash);
	}
	# If the preference is set, add addresses of all available calendars.
	if ($::preferences{'show_cals_in_remind'}{'value'}) {
		my ($ref,$cal);
		$ref = webcalng_io::get_import_permissions($::calendar);
		for $cal (keys %$ref) {
			my %hash;
			next if ($cal eq $::calendar);
			next if ((! $ref->{$cal}{'auto'}) && (! $ref->{$cal}{'email'}));
			$hash{'text'}  = $cal . "*";
			$hash{'value'} = $ref->{$cal}{'email'} . ";" . $cal;
			push(@addresses,\%hash);		
		}
	}
	$addresses_ref = \@addresses;

	# Create list of years to show in end years.
	my $x;
	for ($x=$year;$x<=($year+10);$x++) {
		my %hash;
		$hash{'year'} = $x;
		push(@endyears,\%hash);
	}
	$endyears_ref = \@endyears;

	# Setup a variable that tells the template what time format to use.
	if ($::preferences{'time_format'}{'value'} == 12) {
		$time24 = 0;
	} else {
		$time24 = 1;
	}

	# Setup variables for start hour if they click on an hour to add the item.
	if (defined $shour) {
		$shourvalue = $shour;
		if ($time24) {
			$shourtext  = $shour;
			$shourtext  = "0" . $shourtext if ($shourtext !~ /\d{2}/);
			$sampmvalue = $sampmtext = 0;
		} else {
			if ($shourvalue >= 12) {
				$sampmvalue = 1;
				$sampmtext  = "pm";
				$shourvalue = $shourvalue - 12 unless ($shourvalue == 12);
				$shourtext  = $shourvalue;
			} elsif ($shourvalue == 0) {
				$sampmvalue = 0;
				$sampmtext  = "am";
				$shourvalue = 12;
				$shourtext  = $shourvalue;
			} else {
				$sampmvalue = 0;
				$sampmtext  = "am";
				$shourtext  = $shourvalue;
			}
			$sampm = 1;
		}
		$shour = 1;
	} else {
		$shour = $sampm = $shourvalue = $shourtext = $sampmvalue = $sampmtext = 0;
	}

	# Setup title.
	if ($::preferences{'date_format'}{'value'}) {
		$datetitle = "$::language{$dow} $day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$datetitle = "$::language{$dow} $::language{ $MONTHS[$month-1] } $day, $year";
	}

	# Assign parameters to template.
	$additem_template->param(
		webcal        => $::webcalng_conf{'WEBCAL'},
		year          => $year,
		month         => $month,
		month_name    => $::language{ $MONTHS[$month-1] },
		day           => $day,
		displaydow    => $::language{$dow},
		dow           => $dow,
		weeknumber    => $weeknumber,
		lastweek      => $lastweek,
		calendar      => $::calendar,
		addresses     => $addresses_ref,
		endyears      => $endyears_ref,
		time24        => $time24,
		datetitle     => $datetitle,
		dateformat    => $::preferences{'date_format'}{'value'},
		checksun      => $checkday{'Sun'},
		checkmon      => $checkday{'Mon'},
		checktue      => $checkday{'Tue'},
		checkwed      => $checkday{'Wed'},
		checkthu      => $checkday{'Thu'},
		checkfri      => $checkday{'Fri'},
		checksat      => $checkday{'Sat'},
		sessionkey    => $::sessionkey,
		shour         => $shour,
		sampm         => $sampm,
		shourvalue    => $shourvalue,
		shourtext     => $shourtext,
		sampmvalue    => $sampmvalue,
		sampmtext     => $sampmtext,
		endday        => $endday,
		endmonthvalue => $endmonthvalue,
		endmonthtext  => $endmonthtext,
		endyear       => $endyear,
	);	

	# Write template output.
	$::output .= $additem_template->output;

	return 1;
}

#
# Process the form data from the additem screen.  We send the additem_io
# subroutine an array hash references, with each hash containing key/value
# pairs for each column in a row.
#
sub additem2 {
	my (%data,$year,$month,$day,$dow,$itemid,$exceptionid,$exception,$time24,@remindwhen,$itemref,$key,$datachanged);

	# Get form data that we use frequently.
	$year        = $::cleanq->{'year'};
	$month       = $::cleanq->{'month'};
	$day         = $::cleanq->{'day'};
	$dow         = $::cleanq->{'dow'};
	$itemid      = $::cleanq->{'itemid'} || 0;
	$exceptionid = $::cleanq->{'exceptionid'} || 0;
	$exception   = $::cleanq->{'exception'} || 0;
	$time24      = $::cleanq->{'time24'} || 0;

	# Make sure they did not give us any bogus input from an edit screen.
	soft_error('invalidentrydash') if (($::cleanq->{'year'}) && ($::cleanq->{'year'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'month'}) && ($::cleanq->{'month'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'day'}) && ($::cleanq->{'day'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'shour'}) && ($::cleanq->{'shour'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'smin'}) && ($::cleanq->{'smin'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'sampm'}) && ($::cleanq->{'sampm'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'ehour'}) && ($::cleanq->{'ehour'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'emin'}) && ($::cleanq->{'emin'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'eampm'}) && ($::cleanq->{'eampm'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'repeat'}) && ($::cleanq->{'repeat'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'endyear'}) && ($::cleanq->{'endyear'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'endmonth'}) && ($::cleanq->{'endmonth'} =~ /-/));
	soft_error('invalidentrydash') if (($::cleanq->{'endday'}) && ($::cleanq->{'endday'} =~ /-/));

	# Fix day and month for backward compatibility.
	$month = "0" . $month if ($month !~ /\d{2}/);
	$day   = "0" . $day   if ($day !~ /\d{2}/);
	soft_error('baddaterange') if (($year < 1970) || ($year > 2037));

	# If this item is being edited, get the base information on the item.
	if ($itemid) {
		$itemref = webcalng_io::get_single_item($itemid,$exceptionid);
	}

	# Calculate the startdate.  We just use the time at noon on the first day.
	if ($exception) {
		$data{'startdate'} = $itemref->{'startdate'};
	} else {
		$data{'startdate'} = timelocal "0","0","12",$day,($month-1),($year-1900);
	}

	# If this item is an exception, set the exception date equal to the day which it was edited on.
	if ($exception) {
		$data{'exceptiondate'} = timelocal "0","0","12",$::cleanq->{'editday'},($::cleanq->{'editmonth'}-1),($::cleanq->{'edityear'}-1900);
	} else {
		$data{'exceptiondate'} = 0;
	}

	# Fill in the given description.  It must not be null.
	$data{'description'} = $::cleanq->{'description'};
	soft_error('nodescription') if ((! $data{'description'}) || ($data{'description'} =~ /^\s+$/));

	# Determine the hour/min of the day that this item starts and ends at.
	if ($::cleanq->{'shour'} eq "na") {
		$data{'starttime'} = $data{'endtime'} = "";
	} else {
		if ($time24) {
			$data{'starttime'}  = $::cleanq->{'shour'};
			$data{'starttime'} .= $::cleanq->{'smin'};
			if ($::cleanq->{'ehour'} eq "na") {
				$data{'endtime'} = "";
			} else {
				$data{'endtime'}  = $::cleanq->{'ehour'};
				$data{'endtime'} .= $::cleanq->{'emin'};
			}
		} else {
			$data{'starttime'}  = $::cleanq->{'shour'};
			$data{'starttime'} += 12 if (($::cleanq->{'sampm'}) && ($::cleanq->{'shour'} < 12));
			$data{'starttime'}  = 0 if ((! $::cleanq->{'sampm'}) && ($::cleanq->{'shour'} == 12));
			$data{'starttime'} .= $::cleanq->{'smin'};
			if ($::cleanq->{'ehour'} eq "na") {
				$data{'endtime'} = "";
			} else {
				$data{'endtime'}  = $::cleanq->{'ehour'};
				$data{'endtime'} += 12 if (($::cleanq->{'eampm'}) && ($::cleanq->{'ehour'} < 12));
				$data{'endtime'}  = 0 if ((! $::cleanq->{'eampm'}) && ($::cleanq->{'ehour'} == 12));
				$data{'endtime'} .= $::cleanq->{'emin'};
			}
		}
	}

	# Fill in whether this item is private or busy time.
	$data{'visibility'} = $::cleanq->{'visibility'} || 0;

	# Populate reminder information.
	soft_error('noremindorinviteselected') if (($::cleanq->{'remindwhen'}) && (! $::cleanq->{'remindwho'}));
	if ($exception) {
		$data{'remindwho'}    = $itemref->{'remindwho'};
		$data{'remindwhen'}   = $itemref->{'remindwhen'};
		$data{'remindersent'} = $itemref->{'remindersent'};
	} else {
		$data{'remindwho'}    = "";
		$data{'remindwhen'}   = "";
		$data{'remindersent'} = "";
		$data{'remindwho'}    = $::cleanq->{'remindwho'} if (defined $::cleanq->{'remindwho'});
		if ($data{'remindwho'} ne "") {
			if ($::cleanq->{'remindwhen'}) {
				@remindwhen = split /,/, $::cleanq->{'remindwhen'};
				for (@remindwhen) {
					$data{'remindwhen'}   .= "$_,";
					$data{'remindersent'} .= "0,";
				}
				$data{'remindwhen'}   =~ s/,$//;
				$data{'remindersent'} =~ s/,$//;
			}
		}
		# If this item is to be a meeting, make sure the calendar owner is included automatically in remindwho.
		if ($::cleanq->{'meeting'}) {
			my $owneremail = $::preferences{'email'}{'value'} || 0;
			my $ownerid    = $owneremail . ";" . $::calendar;
			if ($data{'remindwho'} !~ /^$ownerid/) {
				$data{'remindwho'} = $ownerid . "," . $data{'remindwho'};
			}
		}
	}

	# Populate hyperlink or note information if given any.
	if (($::cleanq->{'hyperlink'}) && ($::cleanq->{'hyperlink'} !~ /^\s+$/)) {
		$data{'hyperlink'} = $::cleanq->{'hyperlink'};
	} else {
		$data{'hyperlink'} = "";
	}
	if (($::cleanq->{'notes'}) && ($::cleanq->{'notes'} =~ /\w+/m)) {
		$data{'notes'}     = $::cleanq->{'notes'};
		$data{'notes'}     =~ s/\n/<BR>/g;
		$data{'notes'}     =~ s/
//g;
	} else {
		$data{'notes'} = "";
	}

	# Determine the date regexp to use depending on repeat type.
	if ($exception) {
		$data{'dateregexp'} = $itemref->{'dateregexp'};
		$data{'repeat'}     = $itemref->{'repeat'};
		$data{'repeatdays'} = $itemref->{'repeatdays'};
	} else {
		if (! $dow) {
			my $time = timelocal "0","0","12",$day,($month-1),($year-1900);
			$dow     = (split /\s+/, localtime($time))[0];
		}
		$data{'repeat'} = $::cleanq->{'repeat'} || 0;
		if ($data{'repeat'} eq "daily") {
			$data{'dateregexp'} = ".*,.*,.*";
			$data{'repeatdays'} = 0;
		} elsif (($data{'repeat'} eq "weekly") || ($data{'repeat'} eq "weekly2")) {
			$data{'dateregexp'} = ".*,.*,(";
			if ($::cleanq->{'weekly1'}) {
				$data{'repeatdays'} .= "1,";
				$data{'dateregexp'} .= "Sun|";
			}
			if ($::cleanq->{'weekly2'}) {
				$data{'repeatdays'} .= "2,";
				$data{'dateregexp'} .= "Mon|";
			}
			if ($::cleanq->{'weekly3'}) {
				$data{'repeatdays'} .= "3,";
				$data{'dateregexp'} .= "Tue|";
			}
			if ($::cleanq->{'weekly4'}) {
				$data{'repeatdays'} .= "4,";
				$data{'dateregexp'} .= "Wed|";
			}
			if ($::cleanq->{'weekly5'}) {
				$data{'repeatdays'} .= "5,";
				$data{'dateregexp'} .= "Thu|";
			}
			if ($::cleanq->{'weekly6'}) {
				$data{'repeatdays'} .= "6,";
				$data{'dateregexp'} .= "Fri|";
			}
			if ($::cleanq->{'weekly7'}) {
				$data{'repeatdays'} .= "7,";
				$data{'dateregexp'} .= "Sat|";
			}
			soft_error('selectoneday') if (! $data{'repeatdays'});
			$data{'repeatdays'}  =~ s/,$//;
			$data{'dateregexp'}  =~ s/\|$//;
			$data{'dateregexp'} .= ")";
		} elsif ($data{'repeat'} eq "monthlybydate") {
			$data{'repeatdays'} = 0;
			$data{'dateregexp'} = ".*,$day,.*";
		} elsif (($data{'repeat'} eq "monthlybyday") || ($data{'repeat'} eq "monthlybydaylast")) {
			$data{'repeatdays'} = 0;
			my ($day_regexp);
			if ($data{'repeat'} eq "monthlybyday") {
				if ($day <= 7) {
					$day_regexp = "0[1-7]";
				} elsif ($day <= 14) {
					$day_regexp = "((0[8-9])|(1[0-4]))";
				} elsif ($day <= 21) {
					$day_regexp = "((1[5-9])|(2[0-1]))";
				} else {
					$day_regexp = "2[2-8]";
				}	
			} else {
					$day_regexp = "((2[2-9])|(3[0-1]))";
			}
			$data{'dateregexp'} = ".*,$day_regexp,$dow";
		} elsif ($data{'repeat'} eq "yearlybydate") {
			$data{'repeatdays'} = 0;
			$data{'dateregexp'} = "$month,$day,.*";
		} elsif (($data{'repeat'} eq "yearlybyday") || ($data{'repeat'} eq "yearlybydaylast")) {
			$data{'repeatdays'} = 0;
			my ($day_regexp);
			if ($data{'repeat'} eq "yearlybyday") {
				if ($day <= 7) {
					$day_regexp = "0[1-7]";
				} elsif ($day <= 14) {
					$day_regexp = "((0[8-9])|(1[0-4]))";
				} elsif ($day <= 21) {
					$day_regexp = "((1[5-9])|(2[0-1]))";
				} else {
					$day_regexp = "2[2-8]";
				}	
			} else {
					$day_regexp = "((2[2-9])|(3[0-1]))";
			}
			$data{'dateregexp'} = "$month,$day_regexp,$dow";
		} else {
			$data{'dateregexp'} = "$month,$day,$dow";
			$data{'repeat'}     = 0;
			$data{'repeatdays'} = 0;
		}
	}

	# If this is a repeating item, calculate the end date.
	if ($exception) {
		$data{'repeatend'} = $itemref->{'repeatend'};
		$data{'enddate'}   = $itemref->{'enddate'};
	} else {
		if ($data{'repeat'}) {
			if ($::cleanq->{'repeatend'} eq "date") {
				$data{'repeatend'} = 1;
				my $endyear        = $::cleanq->{'endyear'};
				my $endmonth       = $::cleanq->{'endmonth'};
				my $endday         = $::cleanq->{'endday'};
				$data{'enddate'}   = timelocal "0", "0", "12", $endday, ($endmonth-1), ($endyear-1900);
				$data{'enddate'}  += 86400 if (($data{'starttime'} ne "") && ($data{'endtime'} ne "") && ($data{'starttime'} > $data{'endtime'}));
				soft_error('enddatebeforestartdate') if ($data{'startdate'} > $data{'enddate'});
			} else {
				$data{'repeatend'} = 0;
				$data{'enddate'}   = 9999999999; # Not really a valid date, just bigger than all others.
			}
		} else {
			$data{'repeatend'} = 0;
			$data{'enddate'}   = $data{'startdate'};
			$data{'enddate'}  += 86400 if (($data{'starttime'} ne "") && ($data{'endtime'} ne "") && ($data{'starttime'} > $data{'endtime'}));
		}
	}

	# If this is a meeting, make sure we will be able to add it, and then setup the data for this item.
	if ($::cleanq->{'meeting'}) {
		if ($itemref->{'meetingid'}) {
			$data{'meetingid'} = $itemref->{'meetingid'};
		} else {
			$data{'meetingid'} = time . $$ . ";" . $::calendar;
		}
		check_meeting(\%data);
		# Make sure an immediate reminder is set to send a meeting notice.
		if (($data{'remindwhen'} !~ /^1,/) && ($data{'remindwhen'} ne "1")) {
			$data{'remindwhen'}   = "1," . $data{'remindwhen'};
			$data{'remindersent'} = "0," . $data{'remindersent'};
			$data{'remindwhen'}   =~ s/,$//;
			$data{'remindersent'} =~ s/,$//;
		}	
	} else {
		$data{'meetingid'} = 0;
	}

	# See if the item overlaps, if the prefernces say we should care.
	if (! $::preferences{'overlapping'}{'value'}) {
		my $overlap = check_overlap(\%data,$::calendar);
		if ($overlap) {
			my (@MONTHS,$date,$error,$day,$month,$year);
			@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
			($day,$month,$year) = (localtime($overlap))[3,4,5];
			$month++;
			$year += 1900;
			if ($::preferences{'date_format'}{'value'}) {
				$date = "$day $::language{ $MONTHS[$month-1] } $year";
			} else {
				$date = "$::language{ $MONTHS[$month-1] } $day, $year";
			}
			$error = $::language{'nooverlap'} . " " . $date;
			soft_error($error);
		}
	}
	
	# This item is new, so set it to modified.
	$data{'modified'} = 1;

	# If we get this far, then we passed all the checks on the data.  If we are really editing an
	# item, we can now safely delete it before we re-add it.  But first, we will do a check to see
	# if anything even changed.
	if ($::op eq "edititem2") {
		$datachanged = 0;
		for $key (keys %data) {
			if ($data{$key} ne $itemref->{$key}) {
				next if ($key eq "exceptiondate");
				$datachanged++;
				last;
			}
		}
		return 1 unless ($datachanged);
		if (! $exception) {
			webcalng_io::delitem_io($itemid,'0',$::calendar);
		} else {
			webcalng_io::delitem_io($itemid,$exceptionid,$::calendar) if ($exceptionid);
		}
	}

	# Call io subroutine to actually add the item.
	webcalng_io::additem_io(\%data,$itemid,$exceptionid,$exception,$::calendar);

	# If this is a new meeting, edited meeting, or used to be a meeting and now is not, make the appropriate updates.
	if ($::cleanq->{'meeting'}) {
		$itemid = webcalng_io::get_itemid_for_meetingid($data{'meetingid'},$::calendar);
		if ($itemref->{'meetingid'}) {
			update_meeting(\%data,$itemref,$itemid);
		} else {
			create_meeting(\%data,'0',$itemid);
		}
	} else {
		remove_meeting($itemref,'1','0') if ($itemref->{'meetingid'});
	}

	return 1;
}

#
# Subroutine to check if the meeting the user wants to schedule will be valid.
#
sub check_meeting {
	my ($dataref) = (@_);
	my ($importref,@remindwho,@meeting_calendars,$calendar);

	# Get a list of the calendars that we would have the ability to automatically add items for.
	$importref = webcalng_io::get_import_permissions($::calendar);
	@remindwho = split /,/, $dataref->{'remindwho'};
	for (@remindwho) {
		# Check for calendars that we should automatically add events too.
		if (/(.*?);(.*)/) {
			if ($importref->{$2}{'auto'}) {
				push(@meeting_calendars,$2);
			}
		}
	}

	# Check for overlapping events on auto add calendars.
	for $calendar (@meeting_calendars) {
		my $overlap = check_overlap($dataref,$calendar);
		if ($overlap) {
			my (@MONTHS,$date,$error,$day,$month,$year);
			@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
			($day,$month,$year) = (localtime($overlap))[3,4,5];
			$month++;
			$year += 1900;
			if ($::preferences{'date_format'}{'value'}) {
				$date = "$day $::language{ $MONTHS[$month-1] } $year";
			} else {
				$date = "$::language{ $MONTHS[$month-1] } $day, $year";
			}
			$error = $::language{'nooverlap'} . " " . $date . " " . $::language{'oncalendar'} . " " . $calendar;
			soft_error($error);
		}
	}

	# Make sure they actually invited someone.
	soft_error('no_attendees') if (! $dataref->{'remindwho'});

	# If we get here, everything is a-ok.
	return 1;
}

#
# Add or update meeting information for an item.
#
sub create_meeting {
	my ($dataref,$targetid,$itemid) = (@_);
	my ($importref,@remindwho,@meeting_calendars,$cal,$meetingid,@attendees,%meeting_data,$id,$ownerid,$owneremail);

	# Get a list of the calendars that we would have the ability to automatically add items for.
	$importref = webcalng_io::get_import_permissions($::calendar);
	@remindwho = split /,/, $dataref->{'remindwho'};
	for (@remindwho) {
		next if (($targetid) && ($_ ne $targetid));
		# Check for calendars that we should automatically add events too.
		if (/(.*?);(.*)/) {
			if ($importref->{$2}{'auto'}) {
				push(@meeting_calendars,$2);
			}
		}
	}

	# Add items to calendars that requested we automatically add items to.
	for $cal (@meeting_calendars) {
		copy_meetingitem($::calendar,$cal,$itemid,$dataref);
	}

	# Add the calendar owner as an automatic attendee.
	$meetingid  = $dataref->{'meetingid'};
	$owneremail = $::preferences{'email'}{'value'} || 0;
	$ownerid    = $owneremail . ";" . $::calendar;
	if (! $targetid) {
		$meeting_data{$meetingid}{$ownerid}{'response'} = 3;
		$meeting_data{$meetingid}{$ownerid}{'note'}     = $::language{'meetingorganizer'};
	}

	# Loop through the list of attendees, and setup default response info.
	# Also setup the random password for email only attendees (those without a calendar).
	@attendees = split /,/, $dataref->{'remindwho'};
	for $id (@attendees) {
		next if (($targetid) && ($id ne $targetid));
		next if ($id eq $ownerid);
		my $cal = 0;
		$cal    = $1 if ($id =~ /.+;(.+)/);
		if (($cal) && ($importref->{$cal}{'auto'})) {
			$meeting_data{$meetingid}{$id}{'response'}  = 3;
			$meeting_data{$meetingid}{$id}{'note'}      = $::language{'autoadded'};
		} else {
			$meeting_data{$meetingid}{$id}{'response'} = 0;
			$meeting_data{$meetingid}{$id}{'note'}     = "";
		} 
		if ($cal) {
			$meeting_data{$meetingid}{$id}{'password'} = "";
		} else {
			my @numbers  = ( 'a' .. 'z', 0 .. 9 );
			$meeting_data{$meetingid}{$id}{'password'} = $numbers[rand @numbers] . $numbers[rand @numbers] . $numbers[rand @numbers] . $numbers[rand @numbers];
		}
	}

	# Call io subroutine to actual create entries in the meetings table.
	webcalng_io::create_meeting_io(\%meeting_data);

	return 1;
}

#
# Subroutine to copy a meeting item from one calendar to another.
#
sub copy_meetingitem {
	my ($fromcal,$tocal,$itemid,$dataref) = (@_);
	my ($itemref,$exceptionsref,$tmpexceptionid,$calitemid);

	# Get the data for this item.
	($itemref,$exceptionsref) = webcalng_io::get_items($dataref->{'startdate'},$dataref->{'enddate'},undef,$itemid,undef,$fromcal);

	# Add the item to the calendars that requested auto add.
	$itemref->{$fromcal}{$itemid}{'modified'}     = "1";
	$itemref->{$fromcal}{$itemid}{'remindwho'}    = "";
	$itemref->{$fromcal}{$itemid}{'remindwhen'}   = "";
	$itemref->{$fromcal}{$itemid}{'remindersent'} = "";
	webcalng_io::additem_io($itemref->{$fromcal}{$itemid},0,0,0,$tocal);
	$calitemid = webcalng_io::get_itemid_for_meetingid($dataref->{'meetingid'},$tocal);

	# Now add any exceptions that may exist for the item.
	for $tmpexceptionid (keys %{ $exceptionsref->{$fromcal}{$itemid} }) {
		$exceptionsref->{$fromcal}{$itemid}{$tmpexceptionid}{'modified'}     = "1";
		$exceptionsref->{$fromcal}{$itemid}{$tmpexceptionid}{'remindwho'}    = "";
		$exceptionsref->{$fromcal}{$itemid}{$tmpexceptionid}{'remindwhen'}   = "";
		$exceptionsref->{$fromcal}{$itemid}{$tmpexceptionid}{'remindersent'} = "";
		webcalng_io::additem_io($exceptionsref->{$fromcal}{$itemid}{$tmpexceptionid},$calitemid,$tmpexceptionid,1,$tocal);
	}

	return 1;
}

#
# See if we need to send meeting update notices to anyone.
#
sub update_meeting {
	my ($dataref,$itemref,$itemid) = (@_);
	my (@removed,@added,$bigchange);

	# Compare old meeting information to the new.  If major changes have occurred, remove and
	# re-add the meeting so that people are notified of the changes and force to re-respond.
	# If certain just users have been added or removed, only do what is needed to update those users.
	$bigchange = 0;
	$bigchange++ if ($dataref->{'description'} ne $itemref->{'description'});	
	$bigchange++ if ($dataref->{'dateregexp'}  ne $itemref->{'dateregexp'});	
	$bigchange++ if ($dataref->{'startdate'}   ne $itemref->{'startdate'});	
	$bigchange++ if ($dataref->{'enddate'}     ne $itemref->{'enddate'});	
	$bigchange++ if ($dataref->{'starttime'}   ne $itemref->{'starttime'});	
	$bigchange++ if ($dataref->{'endtime'}     ne $itemref->{'endtime'});	
	$bigchange++ if ($dataref->{'visibility'}  ne $itemref->{'visibility'});	
	$bigchange++ if ($dataref->{'repeat'}      ne $itemref->{'repeat'});	
	$bigchange++ if ($dataref->{'hyperlink'}   ne $itemref->{'hyperlink'});	
	$bigchange++ if ($dataref->{'notes'}       ne $itemref->{'notes'});	
	if ($bigchange) {
		remove_meeting($itemref,'0','0');
		create_meeting($dataref,'0',$itemid);
	} else {
		my @oldremindwho = split /,/, $itemref->{'remindwho'};
		my @newremindwho = split /,/, $dataref->{'remindwho'};
		my ($new,$old);
		for $new (@newremindwho) {
			my $foundold = 0;
			for $old (@oldremindwho) {
				$foundold++, last if ($old eq $new);
			}
			if (! $foundold) {
				# This is a new attendee, so set the 'immediate' reminder to go out again.
				webcalng_io::clear_remindersent($::calendar,$itemid,'0');
				# Add the new attendee to the meeting table.
				create_meeting($dataref,$new,$itemid);
			}
		}
		for $old (@oldremindwho) {
			my $foundnew = 0;
			for $new (@newremindwho) {
				$foundnew++, last if ($old eq $new);
			}
			# If the attendee was removed, call subroutine to tell them and update their calendar if needed.
			remove_meeting($itemref,'1',$old) if (! $foundnew);
		}
	}

	return 1;
}

#
# Remove the meeting from this calendar, and all calendars which had an item added for it.
# 
sub remove_meeting {
	my ($itemref,$sendmessage,$targetid) = (@_);
	my ($meetingid,$meetingdata,$id,$message,$owneremail,$ownerid);
	
	# Get information about this meeting.  Then remove the item from any calendars on which it
	# was added, and schedule a notification about the removal.
	$meetingid   = $itemref->{'meetingid'};
	$owneremail  = $::preferences{'email'}{'value'} || 0;
	$ownerid     = $owneremail . ";" . $::calendar;
	$meetingdata = webcalng_io::get_meeting_data($meetingid,$::calendar);
	for $id (keys %$meetingdata) {
		next if (($targetid) && ($targetid ne $id));
		next if ($id eq $ownerid);
		my $cal = 0;
		$cal    = $1 if ($id =~ /.+;(.+)/);
		if (($meetingdata->{$id}{'response'} >= 2) && ($cal)) {
			webcalng_io::remove_meetingitem($cal,$meetingid);
			my ($emailaddress);
			if ($id =~ /(.+);/) {
				$emailaddress = $1;
			} else {
				$emailaddress = $id;
			}
			$message = $::language{'meetingdeleted'} . " " . $cal . ": " . $itemref->{'description'};
			webcalng_io::schedule_message($id,$message) if (($emailaddress) && ($sendmessage));
		}
	}

	# Now remove the entries in the meeting table for this meeting on this calendar.
	webcalng_io::remove_meeting_io($meetingid,$targetid) unless (($targetid) && ($targetid eq $ownerid));

	return 1;
}

#
# Add an exception item to calendars for a given meeting that was deleted.
#
sub add_meeting_exception {
	my ($itemref,$itemid,$exceptionid,$description) = (@_);
	my ($meetingdata,$id,$owneremail,$ownerid,$message,$date);

	# Loop through the list of meeting lines, and for each calendar which this item was added to, remove the
	# old exception (if needed), then add the new exception.
	$meetingdata = webcalng_io::get_meeting_data($itemref->{'meetingid'},$::calendar);
	$owneremail  = $::preferences{'email'}{'value'} || 0;
	$ownerid     = $owneremail . ";" . $::calendar;
	$date        = localtime($itemref->{'exceptiondate'});
	for $id (keys %$meetingdata) {
		next if ($id eq $ownerid);
		my $cal = 0;
		$cal    = $1 if ($id =~ /.+;(.+)/);
		if (($meetingdata->{$id}{'response'} >= 2) && ($cal)) {
			# Del the exception if it previously existed.
			webcalng_io::delitem_io($itemid,$exceptionid,$cal) if ($exceptionid);

			# Clear out the fields that we do not want to show up on the attendee calendar.
			$itemref->{'modified'}     = "1";
			$itemref->{'remindwho'}    = "";
			$itemref->{'remindwhen'}   = "";
			$itemref->{'remindersent'} = "";
			
			# Add the exception that removes the meeting for that day.	
			my $calitemid = webcalng_io::get_itemid_for_meetingid($itemref->{'meetingid'},$cal);
			webcalng_io::additem_io($itemref,$calitemid,undef,'1',$cal);

			# Setup a notification if needed.
			my ($emailaddress);
			if ($id =~ /(.+);/) {
				$emailaddress = $1;
			} else {
				$emailaddress = $id;
			}
			$message = $::language{'meetingdeleted'} . " " . $cal . " " . $description . ": " . $::language{'on'} . " " . $date;
			webcalng_io::schedule_message($id,$message) if ($emailaddress);
		}
	}

	return 1;
}

#
# This subroutine just calls the io subroutine to delete the item.
#
sub delitem {
	my ($itemid,$exceptionid,$exception,$itemref);

	# Get form input.
	$exception   = $::cleanq->{'exception'} || 0;
	$itemid      = $::cleanq->{'itemid'};
	$exceptionid = $::cleanq->{'exceptionid'} || 0;
	hard_error("No itemid given.") if (! $itemid);

	# Make sure there is no funny business.
	$itemref = webcalng_io::get_single_item($itemid,$exceptionid);
	hard_error('You should not be doing this.') if (($itemref->{'visibility'}) && ($::username ne $::permissions_ref->{'owner'}));
	hard_error('You should not be doing this either.') if (($::preferences{'only_add_remove'}{'value'}) && ($::username ne $itemref->{'user'}));
	hard_error('You should not be removing a meeting added to your calendar from a different calendar.') if (($itemref->{'meetingid'}) && (! $itemref->{'remindwho'}));

	# If we are deleting a single item or all instance of a recurring one, just call the
	# delitem_io subroutine.  If we are removing a single instance of a recurring item, then
	# we really add a exception with no descriptions.
	if (! $exception) {
		webcalng_io::delitem_io($itemid,'0',$::calendar);
		remove_meeting($itemref,'1','0') if ($itemref->{'meetingid'});
	} else {
		my $itemref  = webcalng_io::get_single_item($itemid,$exceptionid);
		webcalng_io::delitem_io($itemid,$exceptionid,$::calendar) if ($exceptionid);
		my $time  = timelocal "0", "0", "12", $::cleanq->{'day'}, ($::cleanq->{'month'}-1), ($::cleanq->{'year'}-1900);
		my $desc = $itemref->{'description'};
		$itemref->{'description'}   = "";
		$itemref->{'exceptiondate'} = $time;
		webcalng_io::additem_io($itemref,$itemid,undef,$exception,$::calendar);
		add_meeting_exception($itemref,$itemid,$exceptionid,$desc) if ($itemref->{'meetingid'});
	}	
	
	return 1;
}

#
# Display a form to let them choose whether to edit all or just a single day in repeating event.
#
sub edititemmulti {
	my ($edititemmulti_template,$file,@MONTHS,$itemid,$exceptionid,$year,$month,$day,$overnight,$datetitle);

	# Setup template.
	if ($::wap) {
		$file = "edititemmulti.wml.tmpl";
	} else {
		$file = "edititemmulti.tmpl";
	}
	$edititemmulti_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Get form input.
	$itemid      = $::cleanq->{'itemid'};
	$exceptionid = $::cleanq->{'exceptionid'} || 0;
	$year        = $::cleanq->{'year'};
	$month       = $::cleanq->{'month'};
	$day         = $::cleanq->{'day'};
	$overnight   = $::cleanq->{'overnight'};
	hard_error("Missing input.") if ((! $itemid) || (! $year) || (! $month) || (! $day));

	# Alter the date if this is the tail end of an overnight item.
	if ($overnight) {
		my $date  = timelocal "0","0","12",$day,($month-1),($year-1900);
		($day,$month,$year) = (localtime($date - 86400))[3,4,5];
		$month++;
		$year += 1900;
	}	

	# Setup title.
	if ($::preferences{'date_format'}{'value'}) {
		$datetitle = "$day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$datetitle = "$::language{ $MONTHS[$month-1] } $day, $year";
	}

	# Assign parameters to template.
	$edititemmulti_template->param(
		webcal       => $::webcalng_conf{'WEBCAL'},
		calendar     => $::calendar,
		itemid       => $itemid,
		exceptionid  => $exceptionid,
		year         => $year,
		month        => $month,
		day          => $day,
		datetitle    => $datetitle,
		sessionkey   => $::sessionkey,
	);

	# Write template output.
 	$::output .= $edititemmulti_template->output;

	return 1;
}

#
# Display a form to let them choose whether to delete all or just a single day in repeating event.
#
sub delitemmulti {
	my ($delitemmulti_template,$file,@MONTHS,$itemid,$exceptionid,$year,$month,$day,$overnight,$datetitle);

	# Setup template.
	if ($::wap) {
		$file = "delitemmulti.wml.tmpl";
	} else {
		$file = "delitemmulti.tmpl";
	}
	$delitemmulti_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Get form input.
	$itemid      = $::cleanq->{'itemid'};
	$exceptionid = $::cleanq->{'exceptionid'} || 0;
	$year        = $::cleanq->{'year'};
	$month       = $::cleanq->{'month'};
	$day         = $::cleanq->{'day'};
	$overnight   = $::cleanq->{'overnight'};
	hard_error("Missing input.") if ((! $itemid) || (! $year) || (! $month) || (! $day));

	# Alter the date if this is the tail end of an overnight item.
	if ($overnight) {
		my $date  = timelocal "0","0","12",$day,($month-1),($year-1900);
		($day,$month,$year) = (localtime($date - 86400))[3,4,5];
		$month++;
		$year += 1900;
	}	

	# Setup title.
	if ($::preferences{'date_format'}{'value'}) {
		$datetitle = "$day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$datetitle = "$::language{ $MONTHS[$month-1] } $day, $year";
	}

	# Assign parameters to template.
	$delitemmulti_template->param(
		webcal       => $::webcalng_conf{'WEBCAL'},
		calendar     => $::calendar,
		itemid       => $itemid,
		exceptionid  => $exceptionid,
		year         => $year,
		month        => $month,
		day          => $day,
		datetitle    => $datetitle,
		sessionkey   => $::sessionkey,
	);

	# Write template output.
	$::output .= $delitemmulti_template->output;

	return 1;
}

#
# Display the edit screen for an item.
#
sub edititem {
	my ($edititem_template,$file,$itemref,$itemid,$exceptionid,$year,$month,$day,$multiple,$x);
	my (@MONTHS,@DOWS,$shortmonth);
	my ($oldstartdate,$oldstartyear,$oldstartmonth,$oldstartday,$oldstartdow);
	my ($sampmtext,$sampmvalue,$shourtext,$shourvalue,$smin,$eampmtext,$eampmvalue,$ehourtext,$ehourvalue,,$emin);
	my ($private,$busy);
	my ($addressid,@addresses,$address_ref,$addresses_ref);
	my (@remindwho,@remindwhen,%remindwho,%remindwhen,%remindwhenlist,@remindtimes,$remindtimes_ref);
	my (@dows,%dows,$date,$dow,$weeknumber,$lastweek,@repeatdays,$repeatdays_ref,$repeatvalue,$repeattext);
	my (@endyears,$endyears_ref,@startyears,$startyears_ref,$endday,$endmonthvalue,$endmonthtext,$endyear,$time);
	my ($notes,$time24,$datetitle,$exception,$edittype);

	# Setup template.
	if ($::wap) {
		$file = "edititem.wml.tmpl";
	} else {
		$file = "edititem.tmpl";
	}
	$edititem_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form data.
	$itemid      = $::cleanq->{'itemid'};
	$exceptionid = $::cleanq->{'exceptionid'} || 0;
	$year        = $::cleanq->{'year'};
	$month       = $::cleanq->{'month'};
	$day         = $::cleanq->{'day'};
	$exception   = $::cleanq->{'exception'} || 0;

	# Get data about this item.
	hard_error('No itemid given.') if (! $itemid);
	$itemref = webcalng_io::get_single_item($itemid,$exceptionid);
	hard_error('You should not be doing this.') if (($itemref->{'visibility'}) && ($::username ne $::permissions_ref->{'owner'}));

	# Setup some constants.
	@MONTHS     = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
	@DOWS       = qw (0 Sun Mon Tue Wed Thu Fri Sat);

	# Determine the day we are working on.
	$oldstartdate = $itemref->{'startdate'};
	$oldstartdow  = (split /\s+/, localtime($oldstartdate))[0];
	($oldstartday,$oldstartmonth,$oldstartyear) = (localtime($oldstartdate))[3,4,5];
	$oldstartmonth++;
	$oldstartyear += 1900;
	if ($::preferences{'date_format'}{'value'}) {
 		$oldstartdate  = "$oldstartday $MONTHS[$oldstartmonth-1] $oldstartyear";
	} else {
		$oldstartdate  = "$MONTHS[$oldstartmonth-1] $oldstartday, $oldstartyear";
	}
	$shortmonth    = $MONTHS[$oldstartmonth-1];
	$shortmonth    = $::language{"short" . $shortmonth};
	$date          = timelocal "0","0","12",$day,($month-1),($year-1900);
	$dow           = (split /\s+/, localtime($date))[0];

	# Setup a variable that tells the template what time format to use.
	if ($::preferences{'time_format'}{'value'} == 12) {
		$time24 = 0;
	} else {
		$time24 = 1;
	}

	# Based on the start and end time, setup some variables used on the template for hours and minutes.
 	if ($itemref->{'starttime'} ne "") {
		if ($itemref->{'starttime'} =~ /(\d+)(\d{2})$/) {
			$shourvalue = $shourtext = $1;
			$smin       = $2;
		}
		if ($time24) {
			$shourtext  = "0" . $shourtext if ($shourtext !~ /\d{2}/);
			$sampmvalue = $sampmtext = "";
		} else {
			if ($shourvalue >= 12) {
				$sampmvalue = 1;
				$sampmtext  = "pm";
				$shourvalue = $shourvalue - 12 unless ($shourvalue == 12);
				$shourtext  = $shourvalue;
			} elsif ($shourvalue == 0) {
				$sampmvalue = 0;
				$sampmtext  = "am";
				$shourvalue = 12;
				$shourtext  = $shourvalue;
			} else {
				$sampmvalue = 0;
				$sampmtext  = "am";
			}
		}
	} else {
		$shourvalue = "na";
		$shourtext  = $::language{'none'};
		$smin       = "00";
		$sampmvalue = 0;
		$sampmtext  = "am";
	}
 	if ($itemref->{'endtime'} ne "") {
		if ($itemref->{'endtime'} =~ /(\d+)(\d{2})$/) {
			$ehourvalue = $ehourtext = $1;
			$emin       = $2;
		}
		if ($time24) {
			$ehourtext  = "0" . $ehourtext if ($ehourtext !~ /\d{2}/);
			$eampmvalue = $eampmtext = "";
		} else {
			if ($ehourvalue >= 12) {
				$eampmvalue = 1;
				$eampmtext  = "pm";
				$ehourvalue = $ehourvalue - 12 unless ($ehourvalue == 12);
				$ehourtext  = $ehourvalue;
			} elsif ($ehourvalue == 0) {
				$eampmvalue = 0;
				$eampmtext  = "am";
				$ehourvalue = 12;
				$ehourtext  = $ehourvalue;
			} else {
				$eampmvalue = 0;
				$eampmtext  = "am";
			}
		}
	} else {
		$ehourvalue = "na";
		$ehourtext  = $::language{'none'};
		$emin       = "00";
		$eampmvalue = 0;
		$eampmtext  = "am";
	}

	# Determine if the item is a private or busy one.
	if ($itemref->{'visibility'}) {
		if ($itemref->{'visibility'} == 1) {
			$busy    = 1;
			$private = 0;
		} else {
			$busy    = 0;
			$private = 1;
		}
	} else {
		$private = $busy = 0;
	}	

	# Build data structures about reminders to help determine who reminders are set for and when.
	@remindwho = split /,/, $itemref->{'remindwho'} if ($itemref->{'remindwho'});
	for (@remindwho) {
		$remindwho{$_}++;
	}
	@remindwhen = split /,/, $itemref->{'remindwhen'} if ($itemref->{'remindwhen'});
	for (@remindwhen) {
		$remindwhen{$_}++;
	}

	# Get list of email addresses in our address book.
	# First, add calendar owner address if they have entered one.
	if ($::preferences{'email'}{'value'}) {
		my %hash;
		$hash{'text'}     = $::language{'calendarowner'};
		$hash{'value'}    = $::preferences{'email'}{'value'} . ";" . $::calendar;
		$hash{'selected'} = 1 if ($remindwho{ $hash{'value'} });
		push(@addresses,\%hash);
	}
	$address_ref = webcalng_io::get_address();
	for $addressid (keys %$address_ref) {
		my %hash;
		$hash{'text'}     = $address_ref->{$addressid}{'displayname'};
		$hash{'value'}    = $address_ref->{$addressid}{'emailaddress'};
		$hash{'selected'} = 1 if ($remindwho{ $address_ref->{$addressid}{'emailaddress'} });
		push(@addresses,\%hash);
	}
	# If the preference is set, add addresses of all available calendars.
	if ($::preferences{'show_cals_in_remind'}{'value'}) {
		my ($ref,$cal);
		$ref = webcalng_io::get_import_permissions($::calendar);
		for $cal (keys %$ref) {
			my %hash;
			next if ($cal eq $::calendar);
			next if ((! $ref->{$cal}{'auto'}) && (! $ref->{$cal}{'email'}));
			$address_ref->{$cal}{'emailaddress'} = $ref->{$cal}{'email'} . ";" . $cal;
			$hash{'text'}     = $cal . "*";
			$hash{'value'}    = $ref->{$cal}{'email'} . ";" . $cal;
			$hash{'selected'} = 1 if ($remindwho{ $hash{'value'} });
			push(@addresses,\%hash);		
		}
	}
	# In case they have since removed an address from their address book.
	for (@remindwho) {
		my $notadded = 1;
		for $addressid (keys %$address_ref) {
			$notadded = 0, last if ($address_ref->{$addressid}{'emailaddress'} eq $_);
		}
		my $ownerid = $::preferences{'email'}{'value'} || 0;
		$ownerid   .= ";" . $::calendar;
		next if ($ownerid eq $_);
		next if /;/;	# Don't keep entries for calendars that have been removed.
		if ($notadded) {
			my %hash;
			$hash{'text'}     = $_;
			$hash{'text'}     =~ s/;.*//;
			$hash{'value'}    = $_;
			$hash{'selected'} = 1;
			push(@addresses,\%hash);
		}
	}
	$addresses_ref = \@addresses;

	# Create data structure to drive remindwhen list in template.
	%remindwhenlist = qw(1 immediately 300 5min 600 10min 900 15min 1800 30min 3600 1hour 7200 2hour 10800 3hour 86400 1day 172800 2day 259200 3day 604800 1week 1209600 2week 2419200 4week);
	for $x (sort {$a <=> $b} keys %remindwhenlist) {
		my %hash;
		$hash{'value'}    = $x;
		$hash{'text'}     = $::language{ $remindwhenlist{$x} };
		$hash{'selected'} = 1 if ($remindwhen{$x});
		push(@remindtimes,\%hash);
	}
	$remindtimes_ref = \@remindtimes;

	# Create list of years to show in start years of new_date. 
	for ($x=($year-10);$x<=($year+10);$x++) {
		my %hash;
		$hash{'year'} = $x;
		push(@startyears,\%hash);
	}
	$startyears_ref = \@startyears;

	# Create list of years to show in end years of end_date.
	for ($x=$year;$x<=($year+10);$x++) {
		my %hash;
		$hash{'year'} = $x;
		push(@endyears,\%hash);
	}
	$endyears_ref = \@endyears;

	# Determine what week of the month it is.
	if (($oldstartday >= 1) && ($oldstartday <= 7)) {
		$weeknumber = $::language{'first'};
		$lastweek   = 0;
	} elsif (($oldstartday >= 8) && ($oldstartday <= 14)) {
		$weeknumber = $::language{'second'};
		$lastweek   = 0;
	} elsif (($oldstartday >= 15) && ($oldstartday <= 21)) {
		$weeknumber = $::language{'third'};
		$lastweek   = 0;
	} else {
		$weeknumber = $::language{'fourth'};
		$lastweek   = 0;
		# See if we are really on the last dow of the month or not.
		my $this_time  = timelocal "0","0","12",$oldstartday,($oldstartmonth-1),($oldstartyear-1900);
		my $next_time  = $this_time + (86400 * 7);
		my $next_month = (localtime($next_time))[4];
		$next_month++;
		$lastweek      = 1 if ($month != $next_month);
	}

	# Setup variables on whether this item repeats.
	if ($itemref->{'repeat'}) {
		$repeatvalue = $itemref->{'repeat'};
		$repeattext  = $::language{$repeatvalue};
		$repeattext .= " " . $oldstartday if ($repeatvalue eq "monthlybydate");
		$repeattext .= " " . $weeknumber . " " . $::language{$oldstartdow} if ($repeatvalue eq "monthlybyday");
		$repeattext .= " " . $::language{$oldstartdow} if ($repeatvalue eq "monthlybydaylast");
		$repeattext .= " " . $::language{ $MONTHS[$oldstartmonth-1] } . " " . $oldstartday if ($repeatvalue eq "yearlybydate");
		$repeattext .= " " . $weeknumber . " " . $::language{$oldstartdow} . " " . $::language{'in'} . " " . $::language{ $MONTHS[$oldstartmonth-1] } if ($repeatvalue eq "yearlybyday");
		$repeattext .= " " . $::language{$oldstartdow} . " " . $::language{'in'} . " " . $::language{ $MONTHS[$oldstartmonth-1] } if ($repeatvalue eq "yearlybydaylast");
	} else {
		$repeatvalue = 0;
		$repeattext  = $::language{'never'};
	}

	# Setup data to drive days of the week that are selected for repeating items in template.
	@dows = split /,/, $itemref->{'repeatdays'};
	for (@dows) {
		$dows{$_}++;
	}
	for $x (1..7) {
		my %hash;
		$hash{'name'}    = "weekly" . $x;
		$hash{'text'}    = $::language{ "short" . $DOWS[$x] };
		$hash{'checked'} = 1 if ($dows{$x});
		push (@repeatdays,\%hash);
	}
	$repeatdays_ref = \@repeatdays;

	# Setup end date on repeating items.
	if ($itemref->{'repeatend'}) {
		($endday,$endmonthvalue,$endyear) = (localtime($itemref->{'enddate'}))[3,4,5];
		$endmonthvalue++;
		$endyear     += 1900;
		$endmonthtext = $MONTHS[$endmonthvalue-1];
		$endmonthtext = $::language{"short" . $endmonthtext};
	} else {
		$time = timelocal "0","0","12",$day,($month-1),($year-1900);
		($endday,$endmonthvalue,$endyear) = (localtime($time))[3,4,5];
		$endmonthvalue++;
		$endyear     += 1900;
		$endmonthtext = $MONTHS[$endmonthvalue-1];
		$endmonthtext = $::language{"short" . $endmonthtext};
	}

	# Fix notes.
	$notes = $itemref->{'notes'};
	$notes =~ s/<BR>/\n/g;

	# Setup title.
	if ($::preferences{'date_format'}{'value'}) {
		$datetitle = "$::language{$dow} $day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$datetitle = "$::language{$dow} $::language{ $MONTHS[$month-1] } $day, $year";
	}

	# Setup message to tell them the edittype.
	if ($exception) {
		$edittype = $::language{'editsingleitem'};
	} else {
		$edittype = $::language{'editallitems'};
	}

	# Assign parameters to template.
	$edititem_template->param(
		webcal          => $::webcalng_conf{'WEBCAL'},
		calendar        => $::calendar,
		itemid          => $itemid,
		exceptionid     => $exceptionid,
		multiple        => $multiple,
		year            => $year,
		month           => $month,
		month_name      => $::language{ $MONTHS[$month-1] },
		day             => $day,
		oldstartyear    => $oldstartyear,
		oldstartmonth   => $oldstartmonth,
		oldstartday     => $oldstartday,
		shortmonth      => $shortmonth,
		oldstartdate    => $oldstartdate,
		description     => $itemref->{'description'},
		shourvalue      => $shourvalue,
		shourtext       => $shourtext,
		smin            => $smin,
		sampmtext       => $sampmtext,
		sampmvalue      => $sampmvalue,
		ehourvalue      => $ehourvalue,
		ehourtext       => $ehourtext,
		emin            => $emin,
		eampmtext       => $eampmtext,
		eampmvalue      => $eampmvalue,
		private         => $private,
		busy            => $busy,
		addresses       => $addresses_ref,
		remindtimes     => $remindtimes_ref,
		meetingid       => $itemref->{'meetingid'} || 0,
		weeknumber      => $weeknumber,
		lastweek        => $lastweek,
		month_name      => $::language{ $MONTHS[$oldstartmonth-1] },
		displaydow      => $::language{$dow},
		repeatvalue     => $repeatvalue,
		repeattext      => $repeattext,
		repeatdays      => $repeatdays_ref,
		endyears        => $endyears_ref,
		startyears      => $startyears_ref,
		endday          => $endday,
		endmonthvalue   => $endmonthvalue,
		endmonthtext    => $endmonthtext,
		endyear         => $endyear,
		repeatend       => $itemref->{'repeatend'},
		hyperlink       => $itemref->{'hyperlink'},
		notes           => $notes,
		time24          => $time24,
		datetitle       => $datetitle,
		dateformat      => $::preferences{'date_format'}{'value'},
		exception       => $exception,
		edittype        => $edittype,
		sessionkey      => $::sessionkey,
	);	

	# Write template output.
	$::output .= $edititem_template->output;

	return 1;
}

#
# Display the address book.
# 
sub address {
	my ($addressid) = (@_);
	my ($address_template,$file,$return_link,$cgi_return_link,$dataref);

	# Get form input.
	$return_link = $::cleanq->{'return_link'};
	$addressid   = $::cleanq->{'addressid'} if (! $addressid);
	$return_link = "op--month..calendar--$::cgi_calendar" if (($::calendar) && (! $return_link));

	# Fix return link.
	if ($return_link) {
		$cgi_return_link = $return_link;
		$cgi_return_link =~ s/\s+/+/g;
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Setup template.
	if ($::wap) {
		$file = "address.wml.tmpl";
	} else {
		$file = "address.tmpl";
	}
	$address_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Build data structure to drive address select input.
	my (@addressids,$addressids_ref,$tmpaddressid);
	$dataref = webcalng_io::get_address();
	for $tmpaddressid (sort keys %$dataref) {
		my %hash;
		$hash{'value'} = $tmpaddressid;
		$hash{'text'}  = $dataref->{$tmpaddressid}{'displayname'};
		if (($addressid) && ($tmpaddressid eq $addressid)) {
			$hash{'selected'} = 1;
		} else {
			$hash{'selected'} = 0;
		}
		push(@addressids,\%hash);
	}
	$addressids_ref = \@addressids;

	# Get specific addressid info if need be.
	$addressid = 0 if ($::op eq "deladdress");
	if (! $addressid) {
		$addressid = 0;
		$dataref->{'displayname'}  = "";
		$dataref->{'emailaddress'} = "";
		$dataref->{'phonenumber'}  = "";
		$dataref->{'phonenumber2'} = "";
	} else {
		$dataref = webcalng_io::get_address($addressid);
	}

	# Assign parameters to template.
	$address_template->param(
		webcal          => $::webcalng_conf{'WEBCAL'},
		return_link     => $return_link,
		cgi_return_link => $cgi_return_link,
		calendar        => $::calendar,
		addressid       => $addressid, 
		displayname     => $dataref->{'displayname'},
		emailaddress    => $dataref->{'emailaddress'},
		phonenumber     => $dataref->{'phonenumber'},
		phonenumber2    => $dataref->{'phonenumber2'},
		addressids      => $addressids_ref,
		sessionkey      => $::sessionkey,
	);

	# Write template output.
	$::output .= $address_template->output;

	return 1;
}

#
# Display form for adding a new address.
#
sub addaddress {
	my ($addaddress_template,$file,$return_link);

	# Get form input.
	$return_link  = $::cleanq->{'return_link'};
	$return_link  = "op--month..calendar--$::cgi_calendar" if (($::calendar) && (! $return_link));

	# Setup template.
	if ($::wap) {
		$file = "addaddress.wml.tmpl";
	} else {
		$file = "addaddress.tmpl";
	}
	$addaddress_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Assign parameters to template.
	$addaddress_template->param(
		webcal      => $::webcalng_conf{'WEBCAL'},
		return_link => $return_link,
		calendar    => $::calendar,
		sessionkey  => $::sessionkey,
	);

	# Write template output.
	$::output .= $addaddress_template->output;

	return 1;
}

#
# Get form data for adding a new address and call i/o subroutine.
#
sub addaddress2 {
	my (%data);

	# Get form data.
	$data{'displayname'}  = $::cleanq->{'displayname'}  || "";
	$data{'emailaddress'} = $::cleanq->{'emailaddress'} || "";
	$data{'phonenumber'}  = $::cleanq->{'phonenumber'}  || "";
	$data{'phonenumber2'} = $::cleanq->{'phonenumber2'} || "";

	# Sanity checks.
	soft_error('nodisplayname')         if (! $data{'displayname'});
	soft_error('noemailaddress')        if (! $data{'emailaddress'});
	soft_error('invaliddisplayname')    if ($data{'displayname'} !~ /^([\w,\s,\-,\.]+)$/);
	soft_error('invalidemailaddress')   if ($data{'emailaddress'} !~ /^([\w,\-,\.,\@]+)$/);
	soft_error('invalidphonenumber')    if (($data{'phonenumber'}) && ($data{'phonenumber'} !~ /^([\d,\s,\-,\.]+)$/));
	soft_error('invalidphonenumber2')  if (($data{'phonenumber2'}) && ($data{'phonenumber2'} !~ /^([\d,\s,\-,\.]+)$/));
	if (webcalng_io::address_exists($data{'displayname'})) {
		soft_error('displaynameinuse');
	}

	# Call i/o subroutine to add the address.
	webcalng_io::addaddress_io(\%data);

	return 1;
}

#
# Display the form for editing an address.
#
sub editaddress {
	my ($editaddress_template,$file,$return_link,$addressid,$dataref);

	# Get form input.
	$return_link  = $::cleanq->{'return_link'};
	$addressid    = $::cleanq->{'addressid'};
	$return_link  = "op--month..calendar--$::cgi_calendar" if (($::calendar) && (! $return_link));

	# Setup template.
	if ($::wap) {
		$file = "editaddress.wml.tmpl";
	} else {
		$file = "editaddress.tmpl";
	}
	$editaddress_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get address data from i/o module.
	$dataref = webcalng_io::get_address($addressid);

	# Assign parameters to template.
	$editaddress_template->param(
		webcal       => $::webcalng_conf{'WEBCAL'},
		return_link  => $return_link,
		calendar     => $::calendar,
		addressid    => $addressid,
		displayname  => $dataref->{'displayname'},
		emailaddress => $dataref->{'emailaddress'},
		phonenumber  => $dataref->{'phonenumber'},
		phonenumber2 => $dataref->{'phonenumber2'},
		sessionkey   => $::sessionkey,
	);

	# Write template output.
	$::output .= $editaddress_template->output;

	return 1;
}

#
# Call the i/o subourtine and update the address.
#
sub editaddress2 {
	my (%data);

	# Get form data.
	$data{'addressid'}       = $::cleanq->{'addressid'}      || "";
	$data{'displayname'}     = $::cleanq->{'displayname'}    || "";
	$data{'olddisplayname'}  = $::cleanq->{'olddisplayname'} || "";
	$data{'emailaddress'}    = $::cleanq->{'emailaddress'}   || "";
	$data{'phonenumber'}     = $::cleanq->{'phonenumber'}    || "";
	$data{'phonenumber2'}    = $::cleanq->{'phonenumber2'}   || "";

	# Sanity checks.
	hard_error("No addressid given.")      if (! $data{'addressid'});
	soft_error('nodisplayname')            if (! $data{'displayname'});
	hard_error("No olddisplayname given.") if (! $data{'olddisplayname'});
	soft_error('noemailaddress')           if (! $data{'emailaddress'});
	if (($data{'olddisplayname'} ne $data{'displayname'}) && (webcalng_io::address_exists($data{'displayname'}))) {
		soft_error('displaynameinuse');
	}
	soft_error('invaliddisplayname')  if ($data{'displayname'} !~ /^([\w,\s,\-,\.]+)$/);
	soft_error('invalidemailaddress') if ($data{'emailaddress'} !~ /^([\w,\-,\.,\@]+)$/);
	soft_error('invalidphonenumber')  if (($data{'phonenumber'}) && ($data{'phonenumber'} !~ /^([\d,\s,\-,\.]+)$/));
	soft_error('invalidphonenumber2') if (($data{'phonenumber2'}) && ($data{'phonenumber2'} !~ /^([\d,\s,\-,\.]+)$/));

	# Call i/o subroutine to update the address.
	webcalng_io::editaddress_io(\%data);

	return 1;
}

#
# Delete a given address.
#
sub deladdress {
	my ($addressid);
	
	# Get form input.
	$addressid = $::cleanq->{'addressid'};
	hard_error("No addressid given.") if (! $addressid);

	# Call i/o subroutine to remove it.
	webcalng_io::deladdress_io($addressid);
	
	return 1;
}

#
# Display the task list.
#
sub task {
	my ($task_template,$file,$return_link,$cgi_return_link,$dataref,$taskid,@tasks,$tasks_ref,$counter);

	# Get form input.
	$return_link = $::cleanq->{'return_link'};
	$return_link = "op--month..calendar--$::cgi_calendar" if (($::calendar) && (! $return_link));

	# Fix return link.
	if ($return_link) {
		$cgi_return_link = $return_link;
		$cgi_return_link =~ s/\s+/+/g;
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Setup template.
	if ($::wap) {
		$file = "task.wml.tmpl";
	} else {
		$file = "task.tmpl";
	}
	$task_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get tasks from i/o module.
	$dataref = webcalng_io::get_task();
	$counter = 0;
	for $taskid (keys %$dataref) {
		my %hash;
		$hash{'taskid'}   = $taskid;
		$hash{'taskdata'} = $dataref->{$taskid}{'taskdata'};
		$hash{'taskdone'} = $dataref->{$taskid}{'taskdone'};
		if ($counter % 2) {
			$hash{'bgcolor'} = $::preferences{'textbackground'}{'value'};		
		} else {
			$hash{'bgcolor'} = "#ffffff";		
		}
		push(@tasks,\%hash);
		$counter++;
	}
	$tasks_ref = \@tasks;

	# Assign parameters to template.
	$task_template->param(
		webcal          => $::webcalng_conf{'WEBCAL'},
		calendar        => $::calendar,
		return_link     => $return_link,
		cgi_return_link => $cgi_return_link,
		tasks           => $tasks_ref,
		counter         => $counter,
		textbackground  => $::preferences{'textbackground'}{'value'},
		sessionkey      => $::sessionkey,
	);

	# Write template output.
	$::output .= $task_template->output;

	return 1;
}

#
# Display form for adding a new task.
#
sub addtask {
	my ($addtask_template,$file,$return_link);

	# Get form input.
	$return_link  = $::cleanq->{'return_link'};
	$return_link  = "op--month..calendar--$::cgi_calendar" if (($::calendar) && (! $return_link));

	# Setup template.
	if ($::wap) {
		$file = "addtask.wml.tmpl";
	} else {
		$file = "addtask.tmpl";
	}
	$addtask_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Assign parameters to template.
	$addtask_template->param(
		webcal      => $::webcalng_conf{'WEBCAL'},
		return_link => $return_link,
		calendar    => $::calendar,
		sessionkey  => $::sessionkey,
	);

	# Write template output.
	$::output .= $addtask_template->output;

	return 1;
}

#
# Get form data for adding a new task and call i/o subroutine.
#
sub addtask2 {
	my (%data);

	# Get form data.
	$data{'taskdata'} = $::cleanq->{'taskdata'}  || "";
	$data{'taskdone'} = 0;

	# Sanity checks.
	soft_error('notaskdata')  if (! $data{'taskdata'});

	# Call i/o subroutine to add the address.
	webcalng_io::addtask_io(\%data);

	return 1;
}

#
# Display the form for editing a task.
#
sub edittask {
	my ($edittask_template,$file,$return_link,$taskid,$dataref);

	# Get form input.
	$return_link = $::cleanq->{'return_link'};
	$taskid      = $::cleanq->{'taskid'};
	$return_link = "op--month..calendar--$::cgi_calendar" if (($::calendar) && (! $return_link));

	# Setup template.
	if ($::wap) {
		$file = "edittask.wml.tmpl";
	} else {
		$file = "edittask.tmpl";
	}
	$edittask_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get address data from i/o module.
	$dataref = webcalng_io::get_task($taskid);

	# Assign parameters to template.
	$edittask_template->param(
		webcal      => $::webcalng_conf{'WEBCAL'},
		return_link => $return_link,
		calendar    => $::calendar,
		taskid      => $taskid,
		taskdata    => $dataref->{'taskdata'},
		taskdone    => $dataref->{'taskdone'},
		sessionkey  => $::sessionkey,
	);

	# Write template output.
	$::output .= $edittask_template->output;

	return 1;
}

#
# Call the i/o subourtine and update the task.
#
sub edittask2 {
	my (%data);

	# Get form data.
	$data{'taskid'}   = $::cleanq->{'taskid'}   || "";
	$data{'taskdata'} = $::cleanq->{'taskdata'} || "";
	$data{'taskdone'} = $::cleanq->{'taskdone'} || "0";

	# Sanity checks.
	hard_error("No taskid given.") if (! $data{'taskid'});
	soft_error('notaskdata')       if (! $data{'taskdata'});

	# Call i/o subroutine to update the address.
	webcalng_io::edittask_io(\%data);

	return 1;
}

#
# Mark a given task complete.
#
sub marktask {
	my ($taskid);
	
	# Get form input.
	$taskid = $::cleanq->{'taskid'};
	hard_error("No taskid given.") if (! $taskid);

	# Call i/o subroutine to remove it.
	webcalng_io::marktask_io($taskid);
	
	return 1;
}

#
# Delete a given task.
#
sub deltask {
	my ($taskid);
	
	# Get form input.
	$taskid = $::cleanq->{'taskid'};
	hard_error("No taskid given.") if (! $taskid);

	# Call i/o subroutine to remove it.
	webcalng_io::deltask_io($taskid);
	
	return 1;
}

#
# Display a note for an item.
#
sub notes {
	my ($notes_template,$file,@MONTHS,$itemid,$exceptionid,$itemref,$return_link,$day,$month,$year,$title,$hyperlink,$calendar);

	# Setup template.
	if ($::wap) {
		$file = "notes.wml.tmpl";
	} else {
		$file = "notes.tmpl";
	}
	$notes_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Get form data.
	$itemid        = $::cleanq->{'itemid'};	
	$exceptionid   = $::cleanq->{'exceptionid'} || 0;
	$return_link   = $::cleanq->{'return_link'};
	$day           = $::cleanq->{'day'};
	$month         = $::cleanq->{'month'};
	$year          = $::cleanq->{'year'};
	hard_error("No itemid given.") if (! $itemid);

	# Fix return link.
	if ($return_link) {
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Set title.
	$title = "$::language{ $MONTHS[$month-1] } $day, $year";

	# Get the information about this itemid.
	$itemref = webcalng_io::get_single_item($itemid,$exceptionid);

	# Fix hyperlink, if needed.
	if ($itemref->{'hyperlink'}) {
		$hyperlink = $itemref->{'hyperlink'};
		$hyperlink = "http://" . $hyperlink if (($hyperlink) && ($hyperlink !~ /(:\/\/)|(:\\\\)|(^\\)|(^\/)/));
	} else {
		$hyperlink = "";
	}

	# Assign template parameters.
	$notes_template->param(
		webcal      => $::webcalng_conf{'WEBCAL'},
		return_link => $return_link,
		calendar    => $::calendar,
		description => $itemref->{'description'},
		starttime   => format_time($itemref->{'starttime'}),
		endtime     => format_time($itemref->{'endtime'}),
		notes       => $itemref->{'notes'},
		hyperlink   => $hyperlink,
		title       => $title,
		sessionkey  => $::sessionkey,
	);

	# Write template output.
	$::output .= $notes_template->output;

	return 1;
}

#
# Display information about a meeting for an item.
#
sub meeting {
	my ($meeting_template,$file,@MONTHS,$itemid,$exceptionid,$itemref,$calsref,$day,$month,$year,$title,$hyperlink,$calendar);
	my ($return_link,$cgi_return_link,$meetingdata,$id,$meetingid,@status,@attendees,$attendees_ref,$meetingcal,$counter);

	# Setup template.
	if ($::wap) {
		$file = "meeting.wml.tmpl";
	} else {
		$file = "meeting.tmpl";
	}
	$meeting_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Setup some constants.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);

	# Get form data.
	$itemid        = $::cleanq->{'itemid'};	
	$exceptionid   = $::cleanq->{'exceptionid'} || 0;
	$return_link   = $::cleanq->{'return_link'};
	$day           = $::cleanq->{'day'};
	$month         = $::cleanq->{'month'};
	$year          = $::cleanq->{'year'};
	$meetingcal    = $::cleanq->{'meetingcal'};
	hard_error("No itemid given.") if (! $itemid);
	hard_error("No meetingcal given.") if (! $meetingcal);

	# Fix return link.
	if ($return_link) {
		$cgi_return_link = $return_link;
		$cgi_return_link =~ s/\s+/+/g;
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Set title.
	$title = "$::language{ $MONTHS[$month-1] } $day, $year";

	# Get the information about this itemid.
	$itemref  = webcalng_io::get_single_item($itemid,$exceptionid);
	$meetingid = $itemref->{'meetingid'};
	hard_error('No meetingid found') if (! $meetingid);

	# Fix hyperlink, if needed.
	if ($itemref->{'hyperlink'}) {
		$hyperlink = $itemref->{'hyperlink'};
		$hyperlink = "http://" . $hyperlink if (($hyperlink) && ($hyperlink !~ /(:\/\/)|(:\\\\)|(^\\)|(^\/)/));
	} else {
		$hyperlink = "";
	}

	# Format data structure for attendees list.
	@status      = qw(noresponse notattend mightattend willattend);
	$meetingdata = webcalng_io::get_meeting_data($meetingid,$meetingcal);
	$calsref     = webcalng_io::get_import_permissions($::calendar);
	hard_error("meeting data not found") if (! $meetingdata);
	$counter     = 0;
	for $id (keys %$meetingdata) {
		my (%hash,$email,$cal);
		if ($id =~ /^(.*?);(.*)/) {
			$email = $1;
			$cal   = $2;
		} else {
			$email = $id;
			$cal   = 0;
		}
		next if (($cal) && (! defined $calsref->{$cal}));
		if ($counter % 2) {
			$hash{'bgcolor'} = $::preferences{'textbackground'}{'value'};		
		} else {
			$hash{'bgcolor'} = "#ffffff";		
		}
		$hash{'organizer'}  = 1 if ($cal eq $meetingcal);
		$hash{'email'}      = $email;
		$hash{'cal'}        = $cal;
		$hash{'status'}     = $::language{ $status[ $meetingdata->{$id}{'response'} ] };
		$hash{'comments'}   = $meetingdata->{$id}{'note'} || "";
		$counter++;
		push(@attendees,\%hash);
	}
	$attendees_ref = \@attendees;

	# Assign template parameters.
	$meetingid  =~ s/;.*//;
	$meeting_template->param(
		webcal          => $::webcalng_conf{'WEBCAL'},
		return_link     => $return_link,
		cgi_return_link => $cgi_return_link,
		calendar        => $::calendar,
		meetingcal      => $meetingcal,
		description     => $itemref->{'description'},
		starttime       => format_time($itemref->{'starttime'}),
		endtime         => format_time($itemref->{'endtime'}),
		notes           => $itemref->{'notes'},
		hyperlink       => $hyperlink,
		title           => $title,
		meetingid       => $meetingid,
		attendees       => $attendees_ref,
		sessionkey      => $::sessionkey,
	);

	# Write template output.
	$::output .= $meeting_template->output;

	return 1;
}

#
# Display meeting response form.
#
sub respond {
	my ($respond_template,$file,$meetingid,$itemid,$exceptionid,$id,$itemref,$email,$usercal);
	my ($date,$startdate,$year,$month,$day,$time,@MONTHS,$repeating,$meetingdata,$user,$status,$statustext,$comments);
	my ($owner,$calendarperms,$return_link,$cgi_return_link,$displaymeetingid,$not_organizer);

	# Setup template.
	if ($::wap) {
		$file = "respond.wml.tmpl";
	} else {
		$file = "respond.tmpl";
	}
	$respond_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form data.
	$meetingid     = $::cleanq->{'meetingid'};	
	$itemid        = $::cleanq->{'itemid'};
	$exceptionid   = $::cleanq->{'exceptionid'} || 0;
	$email         = $::cleanq->{'email'} || 0;
	$usercal       = $::cleanq->{'usercal'} || 0;
	$return_link   = $::cleanq->{'return_link'};
	hard_error("no meetingid given.") if (! $meetingid);	

	# Fix return link.
	if ($return_link) {
		$cgi_return_link = $return_link;
		$cgi_return_link =~ s/\s+/+/g;
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Get information about item.
	$meetingid  .= ";" . $::calendar;
	$itemid      = webcalng_io::get_itemid_for_meetingid($meetingid,$::calendar) if (! $itemid);
	soft_error('couldnotfindmeeting') if (! $itemid);
	$itemref     = webcalng_io::get_single_item($itemid,$exceptionid);
	$meetingdata = webcalng_io::get_meeting_data($meetingid,$::calendar);
	soft_error('couldnotfindmeeting') if (! $meetingdata);

	# Gather the meeting data for this user, if it exists.
	if ($usercal) {
		$id = $email . ";" . $usercal;
	} else {
		$id = $email;
	}
	if ($meetingdata->{$id}{'response'}) {
		$status = $meetingdata->{$id}{'response'};
		if ($status == 3) {
			$statustext = $::language{'willattend'};
		} elsif ($status == 2) {
			$statustext = $::language{'mightattend'};
		} elsif ($status == 1) {
			$statustext = $::language{'notattend'};
		} else {
			$status = $statustext = "";
		}
	} else {
		$status = $statustext = "";
	}
	if ($meetingdata->{$id}{'note'}) {
		$comments = $meetingdata->{$id}{'note'};
	} else {
		$comments = "";
	}

	# Determine the username for who owns this calendar, if needed.
	if ($usercal) {
		$calendarperms = webcalng_io::get_calendar_permissions($usercal);
		$owner         = $calendarperms->{'owner'};
	} else {
		$owner = "";
	}

	# Determine item startdate and time.
	@MONTHS  = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
	($day,$month,$year) = (localtime($itemref->{'startdate'}))[3,4,5];
	$month++;
	$year += 1900;
	if ($::preferences{'date_format'}{'value'}) {
		$startdate = "$day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$startdate = "$::language{ $MONTHS[$month-1] } $day, $year";
	}
	if ($itemref->{'starttime'} ne "") {
		$time = format_time($itemref->{'starttime'});
		if ($itemref->{'endtime'} ne "") {
			$time .= " - " . format_time($itemref->{'endtime'});
		}
	} else {
		$time = $::language{'none'};
	}

	# Get information about whether meeting repeats.
	if ($itemref->{'repeat'}) {
		$repeating  = $::language{ $itemref->{'repeat'} };
		$repeating  =~ s/\s+-.*//; # Get rid of repeat description used when adding events.
		$repeating .= " ";
		if ($itemref->{'repeat'} eq "weekly") {
			$repeating .= $::language{'every'} . " ";
			my @dows    = qw(undef shortSun shortMon shortTue shortWed shortThu shortFri shortSat);
			my @days    = split /,/, $itemref->{'repeatdays'};
			for (@days) {
				$repeating .= "$::language{ $dows[$_] },";
			}
			$repeating  =~ s/,$//;
			$repeating .= " ";
		}
		if ($itemref->{'repeat'} eq "weekly2") {
			$repeating .= $::language{'everyother'} . " ";
			my @dows    = qw(undef shortSun shortMon shortTue shortWed shortThu shortFri shortSat);
			my @days    = split /,/, $itemref->{'repeatdays'};
			for (@days) {
				$repeating .= "$::language{ $dows[$_] },";
			}
			$repeating  =~ s/,$//;
			$repeating .= " ";
		}
		if ($itemref->{'repeatend'}) {
			my ($repeatenddate);
			($day,$month,$year) = (localtime($itemref->{'enddate'}))[3,4,5];
			$month++;
			$year += 1900;
			if ($::preferences{'date_format'}{'value'}) {
				$repeatenddate .= "$day $::language{ $MONTHS[$month-1] } $year";
			} else {
				$repeatenddate .= "$::language{ $MONTHS[$month-1] } $day, $year";
			}
			$repeating .= $::language{'until'} . " " . $repeatenddate;
		}
	} else {
		$repeating = $::language{'no'};
	}

	# The calendar owner should not be able to change their
	# resposne - they must automatically attend.
	if ($usercal ne $::calendar) {
		$not_organizer = 1;
	} else {
		$not_organizer = 0;
	}

	# Assign template parameters.
	$displaymeetingid = $meetingid;
	$displaymeetingid =~ s/;.*//;
	$respond_template->param(
		calendar         => $::calendar,
		meetingid        => $meetingid,
		itemid           => $itemid,
		description      => $itemref->{'description'},
		startdate        => $startdate,
		repeating        => $repeating,
		time             => $time,
		usercal          => $usercal,
		email            => $email,
		owner            => $owner,
		comments         => $comments,
		status           => $status,
		statustext       => $statustext,
		return_link      => $return_link,
		cgi_return_link  => $cgi_return_link,
		displaymeetingid => $displaymeetingid,
		sessionkey       => $::sessionkey,
		not_organizer    => $not_organizer,
	);

	# Write template output.
	$::output .= $respond_template->output;

	return 1;
}

#
# Process meeting response form input.
#
sub respond2 {
	my ($respond2_template,$file,$meetingid,$itemid,$exceptionid,$email,$status,$comments,$itemref);
	my ($index,$targetindex,@remindwho,@responses,$responses,$usercal,$calpassword,$emailpassword);
	my ($meetingdata,%newmeetingdata,$id,$targetid,$message,$return_link);

	# Setup template.
	if ($::wap) {
		$file = "respond2.wml.tmpl";
	} else {
		$file = "respond2.tmpl";
	}
	$respond2_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form data.
	$meetingid     = $::cleanq->{'meetingid'};	
	$itemid        = $::cleanq->{'itemid'};
	$exceptionid   = $::cleanq->{'exceptionid'} || 0;
	$email         = $::cleanq->{'email'} || 0;
	$status        = $::cleanq->{'status'};
	$comments      = $::cleanq->{'comments'};
	$usercal       = $::cleanq->{'usercal'} || 0;
	$calpassword   = $::cleanq->{'calpassword'} || 0;
	$emailpassword = $::cleanq->{'emailpassword'} || 0;
	$return_link   = $::cleanq->{'return_link'};
	hard_error("no meetingid given.") if (! $meetingid);	

	# Check form data.
	hard_error('Invalid status') unless ($status =~ /[0,1,2,3]/);
	hard_error('The calendar owner can not change their status.') if ($usercal eq $::calendar);

	# Fix return link.
	if ($return_link) {
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Get information about item.
	$itemid = webcalng_io::get_itemid_for_meetingid($meetingid,$::calendar) if (! $itemid);
	soft_error('couldnotfindmeeting') if (! $itemid);
	$itemref = webcalng_io::get_single_item($itemid,$exceptionid);

	# Get information about the meeting.
	$meetingdata = webcalng_io::get_meeting_data($meetingid,$::calendar);
	soft_error('couldnotfindmeeting') if (! $meetingdata);
	if ($usercal) {
		$targetid = $email . ";" . $usercal;
	} else {
		$targetid = $email;
	}
	soft_error('usernotinmeeting') unless (defined $meetingdata->{$targetid}{'response'});

	# If they are adding this item to a calendar, authenticate them.  If they pass authentication, update
	# their calendar as needed.  Otherwise, just make sure they gave the password for their email addresses.
	if ($usercal) {
		my ($calendarperms,$owner);
		
		# Find out the real owner of the calendar.	
		$calendarperms   = webcalng_io::get_calendar_permissions($usercal);
		$owner           = $calendarperms->{'owner'};

		# Make sure they gave a valid password for this calendar owner.
		soft_error('invalidpassword') if (! webcalng_auth::validate_login($owner,$calpassword));

		# Untaint $usercal.
		if ($usercal =~ /([\w,\-,\s,\.]+)/) {
			$usercal = $1;
		} else {
			hard_error("Could not untaint usercal");
		}
	
		# Add the item to the users calendar if needed.
		if (($meetingdata->{$targetid}{'response'} <= 1) && ($status >= 2)) {
			copy_meetingitem($::calendar,$usercal,$itemid,$itemref);
			$message = $::language{'meetingadded'};
		}

		# Remove the item from the users calendar if needed.
		if (($meetingdata->{$targetid}{'response'} >= 2) && ($status <= 1)) {
			webcalng_io::remove_meetingitem($usercal,$meetingid);
			$message = $::language{'meetingremoved'};
		}
	} else {
		# Make sure they gave a valid password for this calendar owner.
		soft_error('invalidpassword') if ($meetingdata->{$targetid}{'password'} ne $emailpassword);
	}

	# Update meeting table with new information.
	for $id (keys %$meetingdata) {
		if ($id eq $targetid) {
			$newmeetingdata{$meetingid}{$id}{'response'} = $status;
			$newmeetingdata{$meetingid}{$id}{'note'}     = $comments || "";
			$newmeetingdata{$meetingid}{$id}{'password'} = $meetingdata->{$id}{'password'} || "";
		} else {
			$newmeetingdata{$meetingid}{$id}{'response'} = $meetingdata->{$id}{'response'} || 0;
			$newmeetingdata{$meetingid}{$id}{'note'}     = $meetingdata->{$id}{'note'} || "";
			$newmeetingdata{$meetingid}{$id}{'password'} = $meetingdata->{$id}{'password'} || "";
		}
	}
	webcalng_io::remove_meeting_io($meetingid,undef);
	webcalng_io::create_meeting_io(\%newmeetingdata);

	# If we get here, write template output.
	$respond2_template->param(
		message     => $message,
		usercal     => $usercal,
		return_link => $return_link,
		sessionkey  => $::sessionkey,
	);
	$::output .= $respond2_template->output;

	return 1;
}

#
# Display template for calendar availability for a given day.
#
sub availability {
	my ($availability_template,$file,$return_link,$ref,$cal,@calendars,@calendar_data,$calendars_ref);
	my (@MONTHS,$dow,$datetitle,$year,$month,$day,$startmark,$endmark,$hour,$hours_ref,@hours);
	my (@remindwho,$remindwho);

	# Setup template.
	if ($::wap) {
		$file = "availability.wml.tmpl";
	} else {
		$file = "availability.tmpl";
	}
	$availability_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get form data.
	$return_link = $::cleanq->{'return_link'};
	$year        = $::cleanq->{'year'};
	$month       = $::cleanq->{'month'};
	$day         = $::cleanq->{'day'};
	$remindwho   = $::cleanq->{'remindwho'} || "";
	hard_error("Missing year, month, or day") if ((! $year) || (! $month) || (! $day));

	# Fix return link.
	if ($return_link) {
		$return_link =~ s/--/=/g;
		$return_link =~ s/\.\./&/g;
		$return_link =~ s/\s+/+/g;
	}

	# Generate data structure for list of hours to show.
	for $hour ($::preferences{'day_start'}{'value'} .. $::preferences{'day_end'}{'value'}) {
		my (%hash,$time);
		$time = $hour . "00";
		$hash{'hour'} = format_time($time);	
		push(@hours,\%hash);
	}
	$hours_ref = \@hours;

	# Get list of calendars we are going to check.
	if ($remindwho) {
		push(@calendars,$::calendar);
		@remindwho = split /,/, $remindwho;
		for (@remindwho) {
			if (/;/) {
				$cal = (split /;/, $_)[1];
				next if ($cal eq $::calendar);
				push(@calendars,$cal);
			}
		}
	} else {
		push(@calendars,$::calendar);
		$ref = webcalng_io::get_import_permissions($::calendar);
		for $cal (keys %$ref) {
			next if ((! $ref->{$cal}{'auto'}) && (! $ref->{$cal}{'email'}));
			next if ($cal eq $::calendar);
			push(@calendars,$cal);		
		}
	}	

	# Determine the startmark and endmark for today.
	$startmark = timelocal "0","0","0",$day,($month-1),($year-1900);
	$endmark   = $startmark + 86400;

	# Setup date title.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
	$dow    = (split /\s+/, localtime($startmark))[0];
	if ($::preferences{'date_format'}{'value'}) {
		$datetitle = "$::language{$dow} $day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$datetitle = "$::language{$dow} $::language{ $MONTHS[$month-1] } $day, $year";
	}

	# Loop through list of calendars, creating data structure to drive output template as we go.
	for $cal (@calendars) {
		my (%hash,@subscribed,@availability,$hour,$itemsref,$exceptions_ref);
		@subscribed                 = get_imported_calendars($cal);
		($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,undef,undef,undef,@subscribed);
		$itemsref                   = filter_items($itemsref,$day,$month,$year);
		$itemsref                   = apply_exceptions($itemsref,$exceptions_ref,$day,$month,$year);
		for $hour ($::preferences{'day_start'}{'value'} .. $::preferences{'day_end'}{'value'}) {
			my (%hash,$min);
			for $min (0 .. 3) {
				my ($time);
				$min  = $min * 15;
				$min  = "0" . $min if ($min !~ /\d{2}/);
				$time = $hour . $min;	
				if (hasevent($itemsref,$time,$day,$month,$year)) {
					$hash{'hour'} .= "<TD BGCOLOR=#AAAAAA>&nbsp;</TD>";
				} else {
					$hash{'hour'} .= "<TD BGCOLOR=#FFFFFF>&nbsp;</TD>";
				}
			}
			push(@availability,\%hash);
		}
		$hash{'availability'} = \@availability,
		$hash{'calendar'}     = $cal;
		push (@calendar_data,\%hash);
	}
	$calendars_ref = \@calendar_data;

	# Assign template parameters.
	$availability_template->param(
		webcal      => $::webcal_conf{'WEBCAL'},
		return_link => $return_link,
		hours       => $hours_ref,
		calendars   => $calendars_ref,
		datetitle   => $datetitle,
		sessionkey  => $::sessionkey,
	);

	# Write template output.
	$::output .= $availability_template->output;

	return 1;
}

#
# Determine if a given calendar has items occuring during the given time.
#
sub hasevent {
	my ($itemsref,$time,$day,$month,$year) = (@_);
	my ($hasevent,$calendar,$itemid);

	# Loop through items for today, and see if any of them occur during the given time.
	$hasevent  = 0;
	for $calendar (keys %$itemsref) {
		for $itemid (keys %{ $itemsref->{$calendar} }) {
			my ($start,$end);
			$start = $itemsref->{$calendar}{$itemid}{'starttime'};		
			$end   = $itemsref->{$calendar}{$itemid}{'endtime'};		
			$end  += 2400 if (($end ne "") && ($end < $start));
			next if ($start eq "");
			if ($end eq "") {
				$hasevent++, last if (($start == $time) || (($start - 5) == $time) || (($start - 10) == $time));
			} else {
				$hasevent++ if (($time >= ($start - 10)) && ($time < $end));
			}
		}
	}

	return $hasevent;
}

#
# This just prints the closing tags of the page.
#
sub end {
	my ($end_template,$file,$end_time,$elapsed);

	# Setup template.
	if ($::wap) {
		$file = "end.wml.tmpl";
	} else {
		$file = "end.tmpl";
	}
	$end_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Get timing information if need be.
	if ($::webcalng_conf{'TIMEIT'}) {
		$end_time = [ Time::HiRes::gettimeofday() ];
		$elapsed  = Time::HiRes::tv_interval($::start_time,$end_time);
	} else {
		$elapsed = "";
	}

	# Assign template parameters.
	$end_template->param(
		timeit     => $::webcalng_conf{'TIMEIT'},
		elapsed    => $elapsed,
	);

	# Write template output.
	$::output .= $end_template->output;

	return 1;
}

#
# Subroutine to check if an item that is about to be added overlaps with any other items on the calendar.
# For repeating items, we traverse each day of the repeating item and check for conflicts until the
# date the items ends, or until we have checked at least 5 years out.
#
sub check_overlap {
	my ($dataref,$calendar) = (@_);
	my ($overlap,$itemid,$tmpitemid,$year,$month,$day,$time24,@calendars);
	my ($startmark,$endmark,$itemsref,$exceptions_ref,$newstarttime,$newendtime);
	my ($oneday,$fiveyears,$loopend,$loopstart,$looptime);

	# Get the form input that we need.
	$itemid = $::cleanq->{'itemid'};
	$year   = $::cleanq->{'year'};
	$month  = $::cleanq->{'month'};
	$day    = $::cleanq->{'day'};

	# If there is no start time, there can not be overlap, so just return 0 now.
	$newstarttime = $dataref->{'starttime'};
	$newendtime   = $dataref->{'endtime'};
	return 0 if ($newstarttime eq "");

	# Determine how long we will loop through items on this calendar.
	$oneday    = 86400;
	$fiveyears = $dataref->{'startdate'} + (86400 * 365 * 5);
	if ($fiveyears > $dataref->{'enddate'}) {
		$loopend = $dataref->{'enddate'};
	} else {
		$loopend = $fiveyears;
	}
	$loopstart = timelocal "0","0","0",$day,($month-1),($year-1900);
	$looptime  = $loopstart;

	# Setup some constants.
	$overlap   = 0;
	@calendars = get_imported_calendars($calendar);

	# Loop through the days we will check.  If the item we are adding occurs on the day, then check it
	# against all other items in the calendar for that day.
	while ($looptime <= $loopend) {

		# Create dateregexp for this day in the loop.
		my ($dow,$year,$month,$day,$dateregexp);
		$dow                = (split /\s+/, localtime($looptime))[0];
		($day,$month,$year) = (localtime($looptime))[3,4,5];
		$day                = "0" . $day   if ($day !~ /\d{2}/);
		$month             += 1;
		$month              = "0" . $month if ($month !~ /\d{2}/);
		$year              += 1900;
		$dateregexp         = "$month,$day,$dow";

		# Call the filter_items subroutine on the data we are trying to add, so see if it falls on $looptime.
		my ($key,%tmpitem,$itemcheck,$itemcheckid);
		for $key (keys %$dataref) {
			# Just use dummy calendar name and itemid so our data structure is correct for filter_items().
			$tmpitem{'1'}{'1'}{$key} = $dataref->{$key};
		}
 		$itemcheck = filter_items(\%tmpitem,$day,$month,$year);

		# Check the item being added to see if it falls on this day.  If it matches, check for other
		# items that might overlap.  We actually look for more than 1 itemid in case the item we are
		# adding is an overnight one, in which case filter_items() would have returned an extra itemid.
		for $itemcheckid (keys %{ $itemcheck->{'1'} }) {

			# Get items for this looptime on this calendar.  If we are checking the overnight part
			# of the item, look at the next day's items on the calendars we are checking.
			if ($itemcheckid =~ /overnight/) {
				$startmark = $looptime + $oneday;
				$endmark   = $looptime + $oneday + $oneday; 
			} else {
				$startmark = $looptime;
				$endmark   = $looptime + $oneday; 
			}
			($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,undef,undef,undef,@calendars);
			$itemsref                   = filter_items($itemsref,$day,$month,$year);
			$itemsref                   = apply_exceptions($itemsref,$exceptions_ref,$day,$month,$year);
	
			# Set the hours we are looking at - they will differ if it is an overnight item.
			if ($itemcheckid =~ /overnight/) {
				$newstarttime = $itemcheck->{'1'}{$itemcheckid}{'starttime'};
				$newendtime   = $itemcheck->{'1'}{$itemcheckid}{'endtime'};
			} else {
				$newstarttime = $itemcheck->{'1'}{$itemcheckid}{'starttime'};
				$newendtime   = $itemcheck->{'1'}{$itemcheckid}{'endtime'};
			}

			# Loop through items, looking for conflicts.
			for $calendar (keys %$itemsref) {
				for $tmpitemid (keys %{ $itemsref->{$calendar} }) {
					next if (($itemid) && ($tmpitemid == $itemid));
					next if ($itemsref->{$calendar}{$tmpitemid}{'starttime'} eq "");
					next if (($dataref->{'meetingid'}) && ($itemsref->{$calendar}{$tmpitemid}{'meetingid'}) && ($dataref->{'meetingid'} eq $itemsref->{$calendar}{$tmpitemid}{'meetingid'}));
					my $oldstarttime = $itemsref->{$calendar}{$tmpitemid}{'starttime'};
					my $oldendtime   = $itemsref->{$calendar}{$tmpitemid}{'endtime'};
					$oldendtime     += 2400 if (($oldendtime ne "") && ($oldstarttime > $oldendtime));
					if ($newendtime eq "") {
						if ($oldendtime eq "") {
							$overlap++ if ($newstarttime == $oldstarttime);
						} else {
							$overlap++ if (($newstarttime >= $oldstarttime) && ($newstarttime < $oldendtime));
						}
					} else {
						if ($oldendtime eq "") {
							$overlap++ if (($oldstarttime >= $newstarttime) && ($oldstarttime < $newendtime));
						} else {
							$overlap++ if (($newstarttime >= $oldstarttime) && ($newstarttime < $oldendtime));
							$overlap++ if (($newendtime > $oldstarttime) && ($newendtime < $oldendtime));
							$overlap++ if (($newstarttime <= $oldstarttime) && ($newendtime > $oldstarttime));
						}
					}
					$overlap = $looptime if ($overlap);
					last if ($overlap);
				}
			}
		}
		$looptime += $oneday;
		last if ($overlap);
	}

	return $overlap;
}

#
# This subroutine generates an array of all the calendars we want to see
# items on, including the one we are working on.
#
sub get_imported_calendars {
	my ($calendar) = (@_);
	my (@calendars,$import_ref,@import,@importok);

	# Add the given calendar to the list automatically.
	push(@calendars,$calendar);

	# Get list of calendars that we have permissions to import from within the given calendar.
	$import_ref = webcalng_io::get_import_permissions($calendar);

	# Finding imported calendars for calendars other than the one we are viewing takes a bit more work.
	if ($calendar ne $::calendar) {
		if ($::preferences{'import_calendars'}{'usermod'}) {
			my ($ref,$imports);
			$ref     = webcalng_io::get_calendar_preferences($calendar);
			$imports = $ref->{'import_calendars'};
			@import  = split /,/, $imports if ($imports);
		} 
	} else {
		@import = split /,/, $::preferences{'import_calendars'}{'value'} if ($::preferences{'import_calendars'}{'value'});
	}

	# Loop through the list of calendars we want to import via preferences, and keep the ones
	# that we actually have permissions to import.
	for (@import) {
		push(@importok,$_) if ($import_ref->{$_}{'import'});
	}
	push(@calendars,@importok);

	return @calendars;
}

#
# Simple subroutine to format a given time based on user preferences.
#
sub format_time {
	my ($time) = (@_);
	my ($hour,$min,$ampm);

	# If nothing gets passed in, just return nothing.
	return "" unless ($time =~ /\d+/);

	# Format the time in either 12 hour, or military (24 hour) format.
	if ($::preferences{'time_format'}{'value'} == 12) {
		if ($time =~ /(\d+)(\d{2})$/) {
			$hour = $1;
			$min  = $2;
			if ($hour == 0) {
				$hour = "12";
				$ampm = "am";
			} elsif ($hour < 12) {
				$ampm = "am";
			} elsif ($hour == 12) {
				$ampm = "pm";
			} elsif ($hour == 24) {
				$ampm = "am";
				$hour = 12;
			} else {
				$hour = $hour - 12;
				$ampm = "pm";
			}
			$time = $hour . ":" . $min . $ampm;
		}
	} else {
		if ($time !~ /\d{4}/) {
			$time = "0" . $time;
		}
	}
	
	return $time;
}

#
# This subroutine takes a reference to a hash of items.  The keys should be the
# itemid in $a and $b, and each itemid should contain a subhash of parameters 
# including the starttime and description for the item.  We sort based on starttime,
# followed by description, but items having no starttime end up last.
#
sub time_sort {
	my ($ref) = (@_);
	my ($A,$B);

	if ($ref->{$a}{'starttime'} eq "") {
		$A = 9999;
	} else {
		$A = $ref->{$a}{'starttime'};
	}
	if ($ref->{$b}{'starttime'} eq "") {
		$B = 9999;
	} else {
		$B = $ref->{$b}{'starttime'};
	}

	if ($A == $B) {
		return $ref->{$a}{'description'} cmp $ref->{$b}{'description'};
	} else {
		return $A <=> $B;
	}
}

#
# Subroutine to display webcalng user documentation.
#
sub userdoc {
	my ($userdoc_template);

	# Setup template.
	$userdoc_template = HTML::Template->new(
		filename          => 'userguide.html',
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Write template output.
	$::output .= $userdoc_template->output;

	return 1;
}

#
# Subroutine to display webcalng admin documentation.
#
sub admindoc {
	my ($admindoc_template);

	# Setup template.
	$admindoc_template = HTML::Template->new(
		filename          => 'adminguide.html',
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Write template output.
	$::output .= $admindoc_template->output;

	return 1;
}

#
# Subroutine to display a soft error.  These errors are generally due to
# invalid input from the calendar user, and not an indication of anything
# wrong with the application itself - those would be hard errors.
#
sub soft_error {
	my ($error) = (@_);
	my ($error_template,$file,$error_message);
	
	# Disconnect from database if we are using one.
	$::dbh->disconnect() if ($::webcalng_conf{'DB'});

	# Setup template.
	if ($::wap) {
		$file = "error.wml.tmpl";
	} else {
		$file = "error.tmpl";
	}
	$error_template = HTML::Template->new(
		filename          => $file,
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);

	# Assign template parameters.
	$error_message = $::language{$error} || $error;
	$error_template->param(
		error => $error_message,
	);

	# Write template output and exit the application.
	$::output = "";
	css();
	$::output .= $error_template->output;
	end();
	http_header();
	print $::output;
	goto EXIT;
}

#
# Subroutine to display a hard error.  Hard errors are ones which should not
# be seen unless something is seriously wrong.  They may even occur before 
# all modules are fully loaded, which is why we do not use a template for 
# the HTML.  Since hard errors will hopefully only be seen by administrators
# and not end users, we just use english for all errors.
#
sub hard_error {
	my ($error) = (@_);	
	if ($0 =~ /remind/) {
		# In case this is called by something run via the webcalng_remind.pl script.
		print "Error: $error\n";
		return 1;
	}
	http_header();
	if ($::wap) {
		print <<ERROR;
<!DOCTYPE wml PUBLIC "-//WAPFORUM//DTD WML 1.1//EN" "http://www.wapforum.org/DTD/wml_1.1.xml">
<wml>
<card id="error" title="error">
<p><b>webcalng error:</b></p>
<p>$error</p>
</card>
</wml>
ERROR
	} else {
		print <<ERROR;
<HTML>
<HEAD>
<STYLE TYPE="text/css">
<!--
body { background-color: white }
.border { background: black }
-->
</STYLE>
<TITLE>webcalng Error</TITLE>
</HEAD>
<BODY>

<BR><BR>
<TABLE WIDTH=80% ALIGN=center BORDER=0>
<TR>
  <TD>
    <TABLE ALIGN=center WIDTH=100%>
    <TR>
      <TD ALIGN=center COLSPAN=2 BGCOLOR=white>
        <FONT SIZE=+1><B>webcalng Error</B></FONT>
      </TD>
    </TR>
    </TABLE>
    <BR>
    <TABLE BORDER=0 CELLSPACING=0 CELLPADDING=0 CLASS=border ALIGN=center WIDTH=100%>
    <TR>
      <TD>
        <TABLE BORDER=0 CELLSPACING=1 CELLPADDING=20 CLASS=border ALIGN=center WIDTH=100%>
        <TR>
          <TD WIDTH=60% VALIGN=top ROWSPAN=2 BGCOLOR=white>
           webcalng has encountered an error, and can not continue with your request.
           Please consult your local system administrator for help resolving this error.
           If you are the local administrator, feel free to post a message to the webcalng
           mailing list if you need assistance.
          </TD>
          <TD WIDTH=40% ALIGN=center BGCOLOR=white>
            <U>The error message is displayed below.</U>
          </TD>
        </TR>
        <TR>
          <TD WIDTH=40% VALIGN=top BGCOLOR=white>
            <B>$error</B>
          </TD>
        </TR>
        </TABLE>
      </TD>
    </TR>
    </TABLE>
    <BR>
    <TABLE ALIGN=center WIDTH=100%>
    <TR>
      <TD ALIGN=center COLSPAN=2 BGCOLOR=white>
        <B>Use your browser's BACK button, or right click with your mouse and select Back, to return to the previous page.</B>
      </TD>
    </TR>
    </TABLE>
  </TD>
</TR>
</TABLE>

</BODY>
</HTML>
ERROR
}
	goto EXIT;
}

1;
