{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}


unit pFIBProps;

interface

{$I FIBPlus.inc}
{$J+}
uses
   Classes,SysUtils;
type

  TKindOnOperation     =(koBefore,koAfter);
  TpFIBExistObject =(eoYes,eoNo,eoUnknown);

  TpPrepareOption=
   (pfSetRequiredFields,pfSetReadOnlyFields,pfImportDefaultValues,
    psUseBooleanField,psSQLINT64ToBCD,psApplyRepositary,psGetOrderInfo,psAskRecordCount
   );
  TpPrepareOptions=set of TpPrepareOption;

  TpFIBDsOption=
   (poTrimCharFields,poRefreshAfterPost,
     poRefreshDeletedRecord,
     poStartTransaction,poAutoFormatFields,poProtectedEdit,poKeepSorting,
     poPersistentSorting,poAllowChangeSqls,poVisibleRecno,poNoForceIsNull
   );
  TpFIBDsOptions= set of TpFIBDsOption;


  TpFIBQueryOption =(qoStartTransaction,qoAutoCommit,qoTrimCharFields,qoNoForceIsNull);
  TpFIBQueryOptions=set of TpFIBQueryOption;

  TDetailCondition=(dcForceOpen,dcForceMasterRefresh,
   dcWaitEndMasterScroll
  );

  TDetailConditions= set of TDetailCondition;

  TSQLs = class(TPersistent)
  private
   FOwner     :Tcomponent;
   function  GetSelectSQL:TStrings;
   procedure SetSelectSQL(Value:TStrings);
   function  GetInsertSQL:TStrings;
   procedure SetInsertSQL(Value:TStrings);
   function  GetUpdateSQL:TStrings;
   procedure SetUpdateSQL(Value:TStrings);
   function  GetDeleteSQL:TStrings;
   procedure SetDeleteSQL(Value:TStrings);
   function  GetRefreshSQL:TStrings;
   procedure SetRefreshSQL(Value:TStrings);
  public
   constructor Create(Owner:TComponent);
   property    Owner:TComponent read FOwner;
  published
   property SelectSQL:TStrings read  GetSelectSQL write SetSelectSQL;
   property UpdateSQL:TStrings read  GetUpdateSQL write SetUpdateSQL;
   property DeleteSQL:TStrings read  GetDeleteSQL write SetDeleteSQL;
   property InsertSQL:TStrings read  GetInsertSQL write SetInsertSQL;
   property RefreshSQL:TStrings read GetRefreshSQL write SetRefreshSQL;
  end;

  TFormatFields = class(TPersistent)
  private
     FDisplayFormatDateTime:string;
     FDisplayFormatDate:string;
     FDisplayFormatTime:string;
     FDisplayFormatNumeric :string;
     FEditFormatNumeric    :string;
    function StoreDfDt:boolean;
    function StoreDfN:boolean;
    function StoreEfN:boolean;
    function StoreDfD:boolean;
    function StoreDfT:boolean;
  public
   constructor Create;
  published
    property DateTimeDisplayFormat:string read FDisplayFormatDateTime
                                   write FDisplayFormatDateTime  stored  StoreDfDt ;
    property NumericDisplayFormat :string read FDisplayFormatNumeric
                                   write FDisplayFormatNumeric   stored  StoreDfN ;
    property NumericEditFormat    :string read FEditFormatNumeric
                                   write FEditFormatNumeric     stored  StoreEfN ;
    property DisplayFormatDate:string read FDisplayFormatDate
                               write FDisplayFormatDate  stored StoreDfD;
    property DisplayFormatTime:string read FDisplayFormatTime
                               write FDisplayFormatTime stored StoreDfT;

  end;

  TWhenGetGenID=(wgNever,wgOnNewRecord,wgBeforePost);

  TAutoUpdateOptions= class (TPersistent)
  private
   FOwner     :Tcomponent;
   FUpdateTableName:string;
   FKeyFields      :string; //  
   FAutoReWriteSqls:boolean; // UpdateSQL?
   FCanChangeSQLs:boolean;
   FGeneratorName :string; // 
   FSelectGenID:boolean;//  
   FGenBeforePost:boolean;//   Post,  OnNewRecord
   FUpdateOnlyModifiedFields:boolean;
   FWhenGetGenID:TWhenGetGenID;
   procedure SetUpdateTableName(Value:string);
   procedure SetSelectGenID(Value:boolean);
   function  GetSelectGenID:boolean;
   function  GetGenBeforePost:boolean;
   procedure SetGenBeforePost(Value:boolean);
  protected
  // for compatibility only
    procedure  ReadSelectGenID(Reader: TReader);
    procedure  ReadGenBeforePost(Reader: TReader);

    procedure  DefineProperties(Filer: TFiler); override;
  public
   constructor Create(Owner:TComponent);
   property    Owner:Tcomponent  read    FOwner     ;
   property    SelectGenID:boolean read GetSelectGenID write SetSelectGenID ;
   property    GenBeforePost:boolean read GetGenBeforePost write SetGenBeforePost ;
  published
   property    UpdateTableName:string  read  FUpdateTableName write SetUpdateTableName;
   property    KeyFields  :string read  FKeyFields write FKeyFields;
   property    AutoReWriteSqls:boolean read  FAutoReWriteSqls write FAutoReWriteSqls default false;
   property    CanChangeSQLs:boolean read  FCanChangeSQLs write FCanChangeSQLs default false;
   property    GeneratorName :string read FGeneratorName write FGeneratorName;
   property    UpdateOnlyModifiedFields:boolean read FUpdateOnlyModifiedFields write FUpdateOnlyModifiedFields
    default false
   ;
   property    WhenGetGenID:TWhenGetGenID read FWhenGetGenID write FWhenGetGenID default wgNever;
  end;

  TConnectParams=class(TPersistent)
  private
    FOwner:TComponent;
    function  GetUserNameA: string;
    procedure SetUserName(const Value:string);
    function  GetRoleName: string;
    procedure SetRoleName(const Value:string);
    function  GetPassword: string;
    procedure SetPassword(const Value:string);
    function  GetCharSet: string;
    procedure SetCharSet(const Value:string);

  public
   constructor Create(Owner:TComponent);
  published
    property UserName : string read GetUserNameA write SetUserName stored false;
    property RoleName : string read GetRoleName write SetRoleName stored false;
    property Password : string read GetPassword write SetPassword stored false;
    property CharSet  : string read GetCharSet  write SetCharSet  stored false;
  end;

  TCacheSchemaOptions =class(TPersistent)
  private
   FLocalCacheFile:string;
   FAutoSaveToFile:boolean;
   FAutoLoadFromFile:boolean;
  public
   constructor Create;
  published
   property LocalCacheFile:string read FLocalCacheFile write FLocalCacheFile;
   property AutoSaveToFile:boolean read FAutoSaveToFile write FAutoSaveToFile default false;
   property AutoLoadFromFile:boolean read FAutoLoadFromFile write FAutoLoadFromFile default false;

  end;

  TDBParams =class(TStringList)
  private
    FOwner : TComponent;
    procedure ReadData(Reader: TReader);
    procedure WriteData(Writer: TWriter);
  protected
   procedure DefineProperties(Filer: TFiler); override;
  public
   constructor Create(AOwner:TComponent);
  end;

  TTransactionAction1 =
   (TARollback1, TARollbackRetaining1,TACommit1, TACommitRetaining1);
  TTPBMode=(tpbDefault,tpbReadCommitted,tpbRepeatableRead);

  TConditions =class;

  TCondition =class
  private
   FOwner:TConditions;
   FEnabled:boolean;
   FInDestroy:boolean;
   function  GetName: string;
   function  GetValue: string;
   procedure SetValue(const Value: string);
   procedure SetName(const Name: string);
   procedure SetEnabled(const Value: boolean);
  public
   constructor Create(AOwner:TConditions);
   destructor  Destroy; override;
   property Enabled:boolean read FEnabled write SetEnabled;
   property Name: string read GetName write SetName;
   property Value: string read GetValue write SetValue;
  end;

  TConditionsStateFlag = (csInApply,csInCancel);
  TConditionsState = set of TConditionsStateFlag;

  TConditions= class (TStringList)
  private
    FFIBQuery  :TComponent;
    FApplied   :boolean;
    FPrimarySQL:string;
    FState     :TConditionsState;
    function  GetEnabledText: string;
    function  GetCondition(Index: integer): TCondition;
  protected
    procedure Put(Index: Integer; const S: string); override;
    procedure DefineProperties(Filer: TFiler); override;
    procedure ReadEnabled(Reader: TReader);
    procedure WriteEnabled(Writer: TWriter);
  public
    constructor Create(AOwner:TComponent);
    destructor  Destroy; override;
    function    Add(const S: string): Integer; override;
    function    AddObject(const S: string; AObject: TObject): Integer; override;
    procedure   Clear; override;
    procedure   Delete(Index: Integer); override;
    function    AddCondition(const Name,Condition:string;Enabled:boolean):TCondition;
    function    FindCondition(const Name:string):TCondition;
    function    ByName(const Name:string):TCondition;
    procedure   Remove(const Name:string);
    procedure   Apply;
    procedure   CancelApply;
    procedure   RestorePrimarySQL;
    property    EnabledText: string read GetEnabledText;
    property    Applied :boolean read FApplied;
    property    Condition[Index:integer] :TCondition read  GetCondition ;default;
    property    PrimarySQL:string read FPrimarySQL write FPrimarySQL;
    property    State :TConditionsState read FState;
  end;


//DataSet defaults
const
  {$IFNDEF BOOLEAN_FIELD_SUPPORT}
   DefaultPrepareOptions:TpPrepareOptions =

    [pfImportDefaultValues,psGetOrderInfo
    {$IFDEF  USE_BCD_FOR_SQLINT64}
     ,psSQLINT64ToBCD
    {$ENDIF}
    ];
  {$ELSE}
   DefaultPrepareOptions:TpPrepareOptions =
    [pfImportDefaultValues,psGetOrderInfo,psUseBooleanField
    {$IFDEF  USE_BCD_FOR_SQLINT64}
     ,psSQLINT64ToBCD
    {$ENDIF}
    ];
  {$ENDIF}

  DefaultOptions:TpFIBDsOptions =
  [poTrimCharFields,poAllowChangeSqls,
   poStartTransaction,poAutoFormatFields,  poRefreshAfterPost
  ];

  DefaultDetailConditions:TDetailConditions=[];


//DataBase Defaults
  DefStoreConnected :boolean = true;
  DefSynchronizeTime:boolean = true;
  DefUpperOldNames  :boolean = false;
  DefUseLoginPrompt :boolean = false;
  DefCharSet        :string  = 'WIN1251';
  DefSQLDialect     :integer = 1;

//Transaction Defaults
  DefTimeOutAction :TTransactionAction1=taRollBack1;
  DefTimeOut       :Integer =0;
  DefTPBMode:TTPBMode =tpbReadCommitted;

//FIBQuery Defaults
  DefParamCheck               :boolean = true;
  DefGoToFirstRecordOnExecute :boolean = true;
  DefQueryOptions             :TpFIBQueryOptions =[];
//Registry Keys
  RegFIBRoot   ='FIBC_Software';
  RegFIBTrKinds='Transation Kinds';
  RegPreferences='Preferences';

  DefPrefixGenName     :string  = 'GEN_';
  DefSufixGenName      :string  = '_ID';
  DefEmptyStrToNull    :boolean = true;

implementation
 uses FIBDatabase,FIBDataSet,FIBQuery,fib,ibase,StrUtil;


constructor TAutoUpdateOptions.Create;
begin
 inherited Create;
 FOwner:=Owner;
 FGeneratorName :='';          
 FSelectGenID:=false;
 FGenBeforePost:=true;
 FCanChangeSQLs:=false;
 FUpdateOnlyModifiedFields:=false;
 FWhenGetGenID :=wgNever
end;

procedure  TAutoUpdateOptions.DefineProperties(Filer: TFiler);
begin
 Filer.DefineProperty('SelectGenID',  ReadSelectGenID, nil,   false );
 Filer.DefineProperty('GenBeforePost',  ReadGenBeforePost, nil,   false );
end;

procedure TAutoUpdateOptions.ReadSelectGenID(Reader: TReader);
begin
 FSelectGenID:=Reader.ReadBoolean;
 if FSelectGenID then
    FWhenGetGenID := wgBeforePost
 else
    FWhenGetGenID := wgNever;
end;


procedure  TAutoUpdateOptions.ReadGenBeforePost(Reader: TReader);
begin
 FGenBeforePost:=Reader.ReadBoolean;
 if not FSelectGenID then  exit;
 if FGenBeforePost then
   FWhenGetGenID:=wgBeforePost
 else
   FWhenGetGenID:=wgOnNewRecord
end;

procedure TAutoUpdateOptions.SetUpdateTableName(Value:string);
begin
// FUpdateTableName:=AnsiUpperCase(Value);
 FUpdateTableName:=Value;
 if FGeneratorName='' then
  FGeneratorName:=DefPrefixGenName+FUpdateTableName+DefSufixGenName;
end;


// for compatibility only
function  TAutoUpdateOptions.GetSelectGenID:boolean;
begin
  result:=FWhenGetGenID<>wgNever
end;

procedure TAutoUpdateOptions.SetSelectGenID(Value: boolean);
begin
 if not Value then
   FWhenGetGenID:=wgNever
 else
 if FWhenGetGenID=wgNever then
  FWhenGetGenID:=wgBeforePost;
end;

function TAutoUpdateOptions.GetGenBeforePost: boolean;
begin
 result:=FWhenGetGenID in [wgBeforePost,wgNever]
end;

procedure TAutoUpdateOptions.SetGenBeforePost(Value: boolean);
begin
 if FWhenGetGenID<>wgNever then
 if Value then
   FWhenGetGenID:=wgBeforePost
 else
  FWhenGetGenID :=wgOnNewRecord;
end;

// TFormatFields
constructor TFormatFields.Create;
begin
 FDisplayFormatDateTime:= LongDateFormat {'dd.mm.yyyy'}; // By Wizard
 FDisplayFormatNumeric :='#,##0.' ;
 FEditFormatNumeric    :='0.';
 FDisplayFormatDate:=ShortDateFormat;
 FDisplayFormatTime:=ShortTimeFormat;
end;

function TFormatFields.StoreDfT:boolean;
begin
 Result:= FDisplayFormatTime<>ShortTimeFormat;
end;

function TFormatFields.StoreDfD:boolean;
begin
 Result:= FDisplayFormatDate<>ShortDateFormat;
end;

function TFormatFields.StoreDfDt:boolean;
begin
 Result:= FDisplayFormatDateTime<>ShortDateFormat;
end;

function TFormatFields.StoreDfN:boolean;
begin
 Result:= FDisplayFormatNumeric <>'#,##0.' ;
end;

function TFormatFields.StoreEfN:boolean;
begin
 Result:= FEditFormatNumeric<>'0.';
end;
///TConnectParams

constructor TConnectParams.Create(Owner:TComponent);
begin
 inherited Create;
 FOwner:=Owner
end;

function  TConnectParams.GetUserNameA: string;
begin
 Result:='';
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
 with TFIBDataBase(FOwner) do begin
  Result:=DBParamByDPB[isc_dpb_user_name];
 end;
end;

procedure TConnectParams.SetUserName(const Value:string);
begin
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
  with TFIBDataBase(FOwner) do
   DBParamByDPB[isc_dpb_user_name]:= Value
end;

function  TConnectParams.GetRoleName: string;
begin
 Result:='';
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
 with TFIBDataBase(FOwner) do begin
  Result:=DBParamByDPB[isc_dpb_sql_role_name];
 end;
end;

procedure TConnectParams.SetRoleName(const Value:string);
begin
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
  with TFIBDataBase(FOwner) do
   DBParamByDPB[isc_dpb_sql_role_name]:= Value
end;

function  TConnectParams.GetPassword: string;
begin
 Result:='';
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
 with TFIBDataBase(FOwner) do begin
  Result:=DBParamByDPB[isc_dpb_password];
 end;
end;

procedure TConnectParams.SetPassword(const Value:string);
begin
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
  with TFIBDataBase(FOwner) do
   DBParamByDPB[isc_dpb_password]:= Value
end;

function  TConnectParams.GetCharSet: string;
begin
 Result:='';
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
 with TFIBDataBase(FOwner) do begin
  Result:=DBParamByDPB[isc_dpb_lc_ctype];
 end;
end;

procedure TConnectParams.SetCharSet(const Value:string);
begin
 if Assigned(FOwner) and (FOwner is TFIBDataBase) then
  with TFIBDataBase(FOwner) do
   DBParamByDPB[isc_dpb_lc_ctype]:= Value
end;

{ TSQLs }

constructor TSQLs.Create(Owner: TComponent);
begin
 inherited Create;
 FOwner:=Owner;
end;

function TSQLs.GetDeleteSQL: TStrings;
begin
 Result:=TFIBDataSet(FOwner).DeleteSQL
end;

function TSQLs.GetInsertSQL: TStrings;
begin
 Result:=TFIBDataSet(FOwner).InsertSQL
end;

function TSQLs.GetRefreshSQL: TStrings;
begin
 Result:=TFIBDataSet(FOwner).RefreshSQL
end;

function TSQLs.GetSelectSQL: TStrings;
begin
 Result:=TFIBDataSet(FOwner).SelectSQL
end;

function TSQLs.GetUpdateSQL: TStrings;
begin
 Result:=TFIBDataSet(FOwner).UpdateSQL
end;

procedure TSQLs.SetDeleteSQL(Value: TStrings);
begin
 TFIBDataSet(FOwner).DeleteSQL:=Value
end;

procedure TSQLs.SetInsertSQL(Value: TStrings);
begin
 TFIBDataSet(FOwner).InsertSQL:=Value
end;

procedure TSQLs.SetRefreshSQL(Value: TStrings);
begin
 TFIBDataSet(FOwner).RefreshSQL:=Value
end;

procedure TSQLs.SetSelectSQL(Value: TStrings);
begin
 TFIBDataSet(FOwner).SelectSQL:=Value
end;

procedure TSQLs.SetUpdateSQL(Value: TStrings);
begin
 TFIBDataSet(FOwner).UpdateSQL:=Value
end;




{ TCacheSchemaOptions }

constructor TCacheSchemaOptions.Create;
begin
 inherited Create;
 FLocalCacheFile  :='';
 FAutoSaveToFile  :=false;
 FAutoLoadFromFile:=false;
end;

{ TDBParams }

constructor TDBParams.Create(AOwner: TComponent);
begin
 inherited Create;
 FOwner := AOwner;
end;

procedure TDBParams.DefineProperties(Filer: TFiler);
  function DoWrite: Boolean;
  begin
    if Filer.Ancestor <> nil then
    begin
      Result := True;
      if Filer.Ancestor is TStrings then
        Result := not Equals(TStrings(Filer.Ancestor))
    end
    else Result := Count > 0;
  end;
begin
  Filer.DefineProperty('Strings', ReadData, WriteData, DoWrite); 
end;

procedure TDBParams.ReadData(Reader: TReader);
begin
  Reader.ReadListBegin;
  BeginUpdate;
  try
    Clear;
    while not Reader.EndOfList do Add(Reader.ReadString);
  finally
    EndUpdate;
  end;
  Reader.ReadListEnd;
end;

procedure TDBParams.WriteData(Writer: TWriter);
var
  I: Integer;
  OldPassword:string;
begin
 if (FOwner is TFIBDatabase) and
  (ddoNotSavePassword in  TFIBDatabase(FOwner).DesignDBOptions)
 then
 begin
  OldPassword:=TFIBDatabase(FOwner).ConnectParams.Password;
  TFIBDatabase(FOwner).ConnectParams.Password:='';
 end;
 try
  Writer.WriteListBegin;
  for I := 0 to Count - 1 do Writer.WriteString(Get(I));
  Writer.WriteListEnd;
 finally
  if (FOwner is TFIBDatabase) and
   (ddoNotSavePassword in  TFIBDatabase(FOwner).DesignDBOptions)
  then
  begin
   TFIBDatabase(FOwner).ConnectParams.Password:=OldPassword;
  end;
 end;
end;


{ TConditions }

constructor TConditions.Create(AOwner:TComponent);
begin
 inherited Create;
 FFIBQuery:=AOwner;
 FApplied :=False;
 FPrimarySQL:='';
 FState :=[]
end;

destructor  TConditions.Destroy; 
var i:integer;
begin
 for i := Count - 1 downto 0 do
 begin
  Objects[i].Free;
 end;
 inherited Destroy;
end;

procedure TConditions.Put(Index: Integer; const S: string);
begin
  if Index=Count then AddObject(S,TCondition.Create(Self))
  else
   inherited Put(Index,S)
end;

function TConditions.Add(const S: string): Integer;
begin
  Result:=inherited Add(S);
  Objects[Result]:=TCondition.Create(Self);
end;

function TConditions.AddCondition(const Name,Condition: string;
  Enabled: boolean): TCondition;
var i:integer;
begin
 i:=IndexOfName(Name);
 if i=-1 then
 begin
  i:=Add(Name+'='+Condition);
  Result:=TCondition(Objects[i]);
  Result.FEnabled:=Enabled;
  if Enabled then
   FApplied:=False    
 end
 else
  Result:=TCondition(Objects[i]);
end;



function  TConditions.FindCondition(const Name:string):TCondition;
var i:integer;
begin
 i:=IndexOfName(Name);
 if i<0 then
  Result:=nil
 else
  Result:=TCondition(Objects[i]);
end;

function  TConditions.ByName(const Name:string):TCondition;
begin
  Result:=FindCondition(Name);
  if Result=nil then
    raise Exception.Create ('Can''t find Condition '+Name);
end;


procedure TConditions.Remove(const Name: string);
var Ind:integer;
begin
 Ind:=IndexOfName(Name);
 if Ind<>-1 then
 begin
  Delete(Ind);
  FApplied:=False
 end;
end;

function TConditions.GetEnabledText: string;
var   i,c:integer;
begin
 Result:='';
 c:=Pred(Count);
 if c<0 then Exit;
 for i:=0 to c do
 if TCondition(Objects[i]).Enabled and not IsBlank(Strings[i]) then
 begin
   Result:=Result+'('+ValueFromStr(Strings[i])+')'+CLRF;
   Result:=Result+CLRF+ ' and ';
 end;
 if not IsBlank(Result) then
  SetLength(Result,Length(Result)-7);
end;

procedure TConditions.Apply;
var Wh,Wh1:string;
begin
 with TFIBQuery(FFIBQuery) do
 begin
   BeginModifySQLText;
   Include(FState,csInApply);   
   try
    Wh1:=Conditions.EnabledText;
    if IsBlank(FPrimarySQL) then
      FPrimarySQL:=SQL.Text
    else
     RestorePrimarySQL;
    if not IsBlank(Wh1) then
    begin
     Wh:=GetMainWhereClause;
     if Wh<>'' then
       Wh:=Wh+CLRF+ 'and '+Wh1
     else
       Wh:=Wh1;
     SetMainWhereClause(Wh);
    end;
    FApplied:=True;
   finally
    EndModifySQLText;
    Exclude(FState,csInApply);    
   end;
 end;
end;

procedure TConditions.CancelApply;
begin
  if (FApplied) and (FState=[]) then
  try
    Include(FState,csInCancel);
    RestorePrimarySQL
  finally
    Exclude(FState,csInCancel);
  end;
end;

procedure   TConditions.RestorePrimarySQL;
begin
  FApplied:=False;
  if not IsBlank(FPrimarySQL) then
   TFIBQuery(FFIBQuery).SQL.Text:=FPrimarySQL;
end;

function TConditions.GetCondition(Index: integer): TCondition;
begin
 if (Index<0) or (Index>=Count) then
  Result:=nil
 else
  Result:=TCondition(Objects[Index]);
end;

function TConditions.AddObject(const S: string; AObject: TObject): Integer;
begin
  if (AObject is TCondition) or (AObject=nil) then
   Result:=inherited AddObject(S,AObject)
  else
   Result:=-1
end;

procedure TConditions.Clear;
var i:integer;
begin
 for i := Count - 1 downto 0 do
 begin
  Objects[i].Free;
 end;
 inherited;
end;

procedure TConditions.Delete(Index: Integer);
begin
  if not Condition[Index].FInDestroy then
   Condition[Index].Free
  else
   inherited;
end;

procedure TConditions.DefineProperties(Filer: TFiler);
  function DoWriteCondEnabled: Boolean;
  var i:integer;
  begin
   Result:=False;
   for i:=0 to Pred(Count) do
   if Condition[i].Enabled then
   begin
    Result:=True; Exit;
   end;
  end;
begin
  inherited;
  Filer.DefineProperty('Enabled',  ReadEnabled, WriteEnabled,   DoWriteCondEnabled);
end;

procedure TConditions.ReadEnabled(Reader: TReader);
var s:string;
    i:integer;
begin
 s:=Reader.ReadString;
 i:=2;
 while i<Length(s) do
 begin
  Condition[(i div 2) -1].Enabled:=s[i]='1';
  Inc(i,2)
 end;
end;

procedure TConditions.WriteEnabled(Writer: TWriter);
var s:string;
    i:integer;
begin
 s:='[';
 for i:=0 to Pred(Count) do
  s:=s+IntToStr(Integer(Condition[i].Enabled))+',';
 s[Length(s)]:=']';
 Writer.WriteString(s)
end;

{ TCondition }

constructor TCondition.Create(AOwner: TConditions);
begin
 inherited Create;
 FOwner:=AOwner;
 FInDestroy:=False
end;

destructor TCondition.Destroy;
var i: integer;
begin
  if FInDestroy then  Exit;
  FInDestroy:=True;
  i := FOwner.IndexOfObject(Self);
  if i>=0 then FOwner.Delete(i);
  inherited Destroy;
end;

function TCondition.GetName: string;
var i: integer;
begin
 i := FOwner.IndexOfObject(Self);
 Result := FOwner.Names[i];
end;

function TCondition.GetValue: string;
var i: integer;
begin
 i := FOwner.IndexOfObject(Self);
 Result := ValueFromStr(FOwner.Strings[i]);
end;

procedure TCondition.SetEnabled(const Value: boolean);
begin
  if not FEnabled = Value then
  begin
   FOwner.FApplied:=False;
   FEnabled := Value;
  end;
end;

procedure TCondition.SetName(const Name: string);
var i: integer;
begin
 i := FOwner.IndexOfObject(Self);
 FOwner.Strings[i]:=Name+'='+ValueFromStr(FOwner.Strings[i]);
end;

procedure TCondition.SetValue(const Value: string);
var i: integer;
begin
 i := FOwner.IndexOfObject(Self);
 FOwner.Strings[i]:=FOwner.Names[i]+'='+Value;
end;

end.


