/*
**  MAIL.C
**
**  POP3/SMTP client, uses ASYNC functions.
**
**  This program can:
**
**  (1) Check status of POP3 account.
**  (2) Mail a file [address in first line].
**  (3) Get mail.
**  (4) Delete mail.
**
**  IMPORTANT:
**
**  (1) This program reads MAIL.INI
**  (2) The fist line of any file to be emailed must
**      the destination email address.
**
*/

#include <windows.h>
#include <winsock.h>
#include <ctype.h>

#include "wil.h"
#include "about.h"
#include "async.h"
#include "message.h"
#include "paint.h"
#include "readini.h"
#include "str.h"
#include "unique.h"

#ifdef WIN32
#define USE_INS HINSTANCE
#define USE_PTR PSTR
#else
#define USE_INS HANDLE
#define USE_PTR LPSTR
#endif

LRESULT CALLBACK MainWndProc(HWND, UINT, WPARAM, LPARAM);

/* globals */

HWND hMainWnd;            /* main window handle */

#define BS            8
#define LF           10
#define CR           13
#define ESC          27

#define MAX_BUF     512
#define STR_SIZE     50

#define POP3_PORT   110
#define SMTP_PORT    25

#define MAIL_CMD_FAILURE      100
#define POP3_COMMAND          101
#define POP3_CONN_SUCCESS     102
#define POP3_USER_SUCCESS     103

#define POP3_QUIT_BEGIN       201
#define POP3_QUIT_SUCCESS     202
#define POP3_QUIT_FAILURE     203

#define MAIL_CHK_BEGIN        301
#define MAIL_CHK_MSG_COUNT    302
#define MAIL_CHK_NEXT_HEADER  303
#define MAIL_CHK_TOP_SUCCESS  304
#define MAIL_CHK_READ_SUCCESS 305

#define MAIL_DEL_BEGIN        401

#define MAIL_GET_BEGIN        501
#define MAIL_GET_READ         502

#define MAIL_PUT_BEGIN        601
#define MAIL_PUT_CONN_SUCCESS 602
#define MAIL_PUT_HELO_SUCCESS 603
#define MAIL_PUT_FROM_SUCCESS 604
#define MAIL_PUT_TO_SUCCESS   605
#define MAIL_PUT_DATA_SUCCESS 606
#define MAIL_PUT_SEND         607

#define SMTP_QUIT_BEGIN       701
#define SMTP_QUIT_SUCCESS     702
#define SMTP_QUIT_FAILURE     703

static USE_INS hInstance;
static int WinWidth = 8 * NCOLS;  /* window width */
static int WinHeight = 15 * NROWS;/* window height */
static char Temp[MAX_BUF+8];      /* temprary buffer */
static int  InBufCmd = 0;         /* command */
static int  InBufLen = 0;         /* length of string in InBuffer[] */
static char InBuffer[MAX_BUF+1];  /* input buffer */
static SOCKET Socket = 0;         /* socket */
static int MailID;                /* mail message number */
static int EchoFlag = 0;          /* echo commands of TRUE */
static int PassFlag = 0;          /* replace echoed chars with '*' if TRUE */
static HCURSOR ArrowCursor;       /* arrow cursor */
static HCURSOR WaitCursor;        /* hour glass cursor */
static LPSTR BufferPtr;           /* pointer to buffer in ASYNC */
static int MsgCount;
static int MsgIndex;
static int POP3SuccessMsg;        /* message on success of POP3 command */
static int POP3FailureMsg;        /* message on failure of POP3 command */

static char SmtpString[STR_SIZE] = "";  /* SMTP server */
static char Pop3String[STR_SIZE] = "";  /* POP3 server */
static char UserString[STR_SIZE] = "";  /* POP3 user */
static char PassString[STR_SIZE] = "";  /* POP3 password */
static char FromString[STR_SIZE] = "";  /* our email address */
static char MailFile[STR_SIZE] = "";    /* name of file to mail */
static char ToString[STR_SIZE] = "";    /* destination (user@adress) */

/* disable menu item list */

static void DisableMenuItems(int First, int Last)
{int i;
 HMENU hMenu;
 hMenu = GetMenu(hMainWnd);
 for(i=First;i<=Last;i++)
    EnableMenuItem(hMenu,i,MF_BYCOMMAND | MF_DISABLED | MF_GRAYED);
 DrawMenuBar(hMainWnd);
}

/* enable menu item list */

static void EnableMenuItems(int First, int Last)
{int i;
 HMENU hMenu;
 hMenu = GetMenu(hMainWnd);
 for(i=First;i<=Last;i++)
    EnableMenuItem(hMenu,i,MF_BYCOMMAND | MF_ENABLED);
 DrawMenuBar(hMainWnd);
}

/* disable POP/SMTP menu items */

static void DisableMailMenuItems(void)
{
 DisableMenuItems(MSG_MAIL_CHK, MSG_MAIL_DEL);
}

/* enable POP/SMTP menu items */

static void EnableMailMenuItems(void)
{
 EnableMenuItems(MSG_MAIL_CHK, MSG_MAIL_DEL);
}

/* add character to input buffer */

static void Add2Buffer(char Chr)
{int i;
 /* add char to input buffer */
 switch(Chr)
   {case BS:
      if(InBufLen>0)
        {/* backup on screen */
         DisplayChar(BS);
         /* remove last char from buffer */
         InBufLen--;
        }
      break;
    case ESC:
      for(i=1;i<=InBufLen;i++) DisplayChar(BS);
      InBufLen = 0;
      DestroyCaret();
      break;
    default:
      /* display char */
      if(PassFlag) DisplayChar('*');
      else DisplayChar(Chr);
      /* put into buffer */
      if(InBufLen<MAX_BUF) InBuffer[InBufLen++] = Chr;
      break;
   }
}

/* place the cursor in the display window */

static void SetTheFocus(void)
{/* create client area caret */
 CreateCaret(hMainWnd,NULL,3,10);
 SetCaretPos(PaintGetColPos(),PaintGetRowPos());
 ShowCaret(hMainWnd);
}

/* display error message */

static void DisplayError(int Code, LPSTR Msg)
{wsprintf((LPSTR)Temp,"ERROR %d: ",Code);
 DisplayString((LPSTR)Temp);
 if(Msg) DisplayString(Msg);
 if(Code)
   {wilErrorText(Code,(LPSTR)Temp,50);
    DisplayLine((LPSTR)Temp);
   }
 /* restore arrow cursor */
 SetCursor(ArrowCursor);
}

/* start POP3 command sequence */

void Pop3Command(int OnSuccess, int OnFailure)
{POP3SuccessMsg = OnSuccess;
 POP3FailureMsg = OnFailure;
 POST_MSG(POP3_COMMAND);
}

/* read text line (ending with LF) from disk */

static int ReadTextLine(int Handle, LPSTR Buffer, int BufLen)
{char Ch;
 int  i;
 int  Size = 0;
 static char TextBuffer[128];
 static int  TextLeft = 0;
 static int  TextRight = 0;
 while(1)
   {/* is TextBuffer[] empty ? */
    if(TextLeft>=TextRight)
      {/* read from disk */
       TextLeft = 0;
       TextRight = _lread(Handle,(LPSTR)TextBuffer,125) - 1;
       if(TextRight<0) return -1;
      }
    /* copy till LF or end of buffer */
    for(i=TextLeft;i<=TextRight;i++)
      {Ch = TextBuffer[i];
       Buffer[Size++] = 0x7f & Ch;
       if(Ch=='\n')
         {/* found LF */
          Buffer[Size] = '\0';
          TextLeft = i + 1;
          return Size;
         }
       /* is Buffer[] filled ? */
       if(Size==BufLen-3)
         {/* end with CR/LF */
          Buffer[Size++] = '\r';
          Buffer[Size++] = '\n';
          Buffer[Size] = '\0';
          TextLeft = i + 1;
          return Size;
         }
      }
    /* used up all of TextBuffer[] */
    TextLeft = TextRight;
   } /* end-while */
}

/* WinMain */

#ifdef WIN32
int WINAPI
#else
int PASCAL
#endif
WinMain(USE_INS hInst, USE_INS hPrevInstance,
        USE_PTR szCmdLine,  int nCmdShow)
{WNDCLASS  wc;
 MSG msg;
 BOOL Result;
 if(!hPrevInstance)
   {/* register main window class */
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInst;
    wc.hIcon = LoadIcon(hInst, "HostIcon");
    wc.hCursor = NULL;
    wc.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "HostMenu";
    wc.lpszClassName = "HostWClass";
    Result = RegisterClass(&wc);
    if(!Result) return FALSE;
   }

 /* create main window */
 hInstance = hInst;
 hMainWnd = CreateWindow(
        "HostWClass",   "Mail",    WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,  CW_USEDEFAULT,
        WinWidth,       WinHeight,
        NULL,           NULL,
        hInstance,      NULL);
 ShowWindow(hMainWnd, nCmdShow);
 UpdateWindow(hMainWnd);

 /* window control loop */

 while(GetMessage(&msg,NULL,0,0))
   {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
   }
 return (msg.wParam);
} /* end WinMain */

#ifdef WIN32
LRESULT CALLBACK
#else
long FAR PASCAL
#endif
MainWndProc(HWND hWindow,UINT iMsg,WPARAM wParam,LPARAM lParam)
{int Code;
 HDC hDC;
 PAINTSTRUCT ps;
 int i, n;
 char Chr;
 LPSTR Ptr;
 static int  BufSize;
 static char LastChar;
 static int  Handle = 0;
#ifdef WIN32
#else
 static FARPROC lpfnAboutDlgProc;
#endif
 /* begin */
 hMainWnd = hWindow;
 switch (iMsg)
    {case WM_CREATE:
      /* create cursors */
      ArrowCursor = LoadCursor(NULL, IDC_ARROW);
      WaitCursor = LoadCursor(NULL, IDC_WAIT);
      SetCursor(ArrowCursor);
#ifdef WIN32
#else
       /* create thunk for Win16 */
       lpfnAboutDlgProc = MakeProcInstance(AboutDlgProc,hInstance);
#endif
      /* initialize paint module */
      PaintInit();
      /* attach WINSOCK */
      DisplayString("Attaching WINSOCK...");
      Code = wilAttach();
      DisplayLine("OK");
      if(Code<0) DisplayError(Code,"wilAttach fails:");
      else
        {DisplayString(" Description: ");
         wilGetDescription((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Name: ");
         wilGetMyHostName((LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
         DisplayString("My Host Addr: ");
         wilGetMyHostDotted(0,(LPSTR)Temp, MAX_BUF);
         DisplayLine((LPSTR)Temp);
        }
      /* open MAIL.INI file */
      if(!IniOpen("MAIL.INI"))
        {DisableMailMenuItems();
         break;
        }
      while(1)
        {/* read next line from MAIL.INI */
         if(IniRead((LPSTR)Temp)<0) break;
#if 0
         /* echo strings */
         DisplayLine((LPSTR)Temp);
#endif
         /* test for all legal keywords */
         IniExtract((LPSTR)Temp,"POP3",(LPSTR)Pop3String);
         IniExtract((LPSTR)Temp,"USER",(LPSTR)UserString);
         IniExtract((LPSTR)Temp,"PASS",(LPSTR)PassString);
         IniExtract((LPSTR)Temp,"SMTP",(LPSTR)SmtpString);
         IniExtract((LPSTR)Temp,"FROM",(LPSTR)FromString);
        }
      /* verify that we have all strings read in */
      if(lstrlen((LPSTR)Pop3String)==0) DisplayLine("ERROR: Missing POP3 server name.");
      if(lstrlen((LPSTR)UserString)==0) DisplayLine("ERROR: Missing POP3 user name.");
      if(lstrlen((LPSTR)SmtpString)==0) DisplayLine("ERROR: Missing SMTP user name.");
      if(lstrlen((LPSTR)FromString)==0) DisplayLine("ERROR: Missing POP3 'From' address.");
      if(lstrlen((LPSTR)PassString)==0) DisplayLine("WARNING: Missing POP3 password.");
      break;

     case WM_COMMAND:
         switch(wParam)
           {
            case MSG_ABOUT :
#ifdef WIN32
               DialogBox(hInstance, "AboutBox", hMainWnd, AboutDlgProc);
#else
               DialogBox(hInstance, "AboutBox", hMainWnd, lpfnAboutDlgProc);
#endif
               return 0;

            case MSG_DEBUG:
              wsprintf((LPSTR)Temp,"DEBUG: SmtpString=[%s]", (LPSTR)SmtpString);
              DisplayLine((LPSTR)Temp);
              wsprintf((LPSTR)Temp,"DEBUG: Pop3String=[%s]", (LPSTR)Pop3String);
              DisplayLine((LPSTR)Temp);
              wsprintf((LPSTR)Temp,"DEBUG: UserString=[%s]", (LPSTR)UserString);
              DisplayLine((LPSTR)Temp);
              wsprintf((LPSTR)Temp,"DEBUG: PassString=[%s]", (LPSTR)PassString);
              DisplayLine((LPSTR)Temp);
              break;

            case MSG_ECHO:
              EchoFlag = 1 - EchoFlag;
              DisplayString("Echo flag is now ");
              if(EchoFlag) DisplayLine("ON");
              else DisplayLine("OFF");
              break;

            case MSG_BREAK:
              EnableMailMenuItems();
              SetCursor(ArrowCursor);
              wilCancelBlocking();
              wilCloseSocket(Socket,500);
              break;

            case MSG_EXIT:
              wilRelease();
              DestroyWindow(hMainWnd);
              break;

            case MSG_MAIL_CHK:
              Pop3Command(MAIL_CHK_BEGIN, POP3_QUIT_FAILURE);
              break;

            case MSG_PASS:
              /* set POP3 user password */
              InBufLen = 0;
              InBufCmd = MSG_PASS;
              PassFlag = TRUE;
              DisableMailMenuItems();
              DisplayString("Enter password:");
              SetTheFocus();
              break;

            case MSG_MAIL_DEL:
              /* delete mail message on server */
              InBufLen = 0;
              if(lstrlen((LPSTR)Pop3String)==0)
                {DisplayLine("Missing POP server name.");
                 break;
                }
              InBufCmd = MSG_MAIL_DEL;
              DisableMailMenuItems();
              DisplayString("Enter message #:");
              SetTheFocus();
              break;

            case MSG_MAIL_GET:
              /* get mail */
              InBufLen = 0;
              if(lstrlen((LPSTR)Pop3String)==0)
                {DisplayLine("Missing POP server name.");
                 break;
                }
              InBufCmd = MSG_MAIL_GET;
              DisableMailMenuItems();
              DisplayString("Enter message #:");
              SetTheFocus();
              break;

            case MSG_MAIL_PUT:
              /* send mail */
              InBufLen = 0;
              if(lstrlen((LPSTR)SmtpString)==0)
                {DisplayLine("Missing SMTP server name.");
                 break;
                }
              InBufCmd = MSG_MAIL_PUT;
              DisableMailMenuItems();
              DisplayString("Enter filename:");
              SetTheFocus();
              break;
           }
         break;

    case WM_USER: /* posted by WIL */
      AsyncProcessMsg(lParam);
      break;

    case WM_USER+1:  /* posted by POST_MSG [defined in async.h] */
      /* test response code */
      if(lParam>=500)
        {/* POP3/SMTP server returns fatal error code */
         POST_MSG(MAIL_CMD_FAILURE);
         break;
        }
      /* execute case */
      switch(wParam)
        {
         case MAIL_CMD_FAILURE:
           DisplayLine("SMTP/POP3 command has failed.");
           EnableMailMenuItems();
           wilCloseSocket(Socket,500);
           SetCursor(ArrowCursor);
           break;

         /* finish up POP3 commands with QUIT command */

         case POP3_QUIT_BEGIN:
           AsyncSetEcho(TRUE);
           /* send QUIT command */
           AsyncCommand("QUIT", POP3_QUIT_SUCCESS,
                        POP3_QUIT_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case POP3_QUIT_FAILURE:
           DisplayLine("POP3 command has failed...");
           EnableMailMenuItems();
           wilCloseSocket(Socket,500);
           SetCursor(ArrowCursor);
           break;

         case POP3_QUIT_SUCCESS:
           /* all done */
           DisplayLine("POP3 command succeeds.");
           EnableMailMenuItems();
           wilCloseSocket(Socket,500);
           SetCursor(ArrowCursor);
           break;

         /* The POP3_* series of cases connect to the POP3 server, */
         /* then send the USER and PASS commands. On success,      */
         /* the message POP3SuccessMsg is posted, while on failure */
         /* the message POP3Failure is posted. See Pop3Command().  */

         case POP3_COMMAND:
           /* log onto POP3 server */
           DisableMailMenuItems();
           SetCursor(WaitCursor);
           AsyncSetEcho(TRUE);
           Socket = AsyncConnect(hMainWnd,"POP3",(LPSTR)Pop3String,
                     POP3_PORT, POP3_CONN_SUCCESS,
                     POP3FailureMsg, ASYNC_PLUS_MINUS);
           break;

         case POP3_CONN_SUCCESS:
#ifdef WIN32
           ///Sleep(2000);
#endif
           /* send USER command */
           wsprintf((LPSTR)Temp,"USER %s", (LPSTR)UserString);
           AsyncCommand((LPSTR)Temp, POP3_USER_SUCCESS,
                        POP3FailureMsg, ASYNC_PLUS_MINUS);
           break;

         case POP3_USER_SUCCESS:
#ifdef WIN32
           Sleep(500);
#endif
           /* send PASS (password) command */
           wsprintf((LPSTR)Temp,"PASS %s", (LPSTR)PassString);
           AsyncCommand((LPSTR)Temp, POP3SuccessMsg,
                        POP3FailureMsg, ASYNC_PLUS_MINUS);
           break;

         /***** END - POP3 command cases *****/

         case MAIL_CHK_BEGIN:
           /* send STAT (statistics) command */
           AsyncSetEcho(TRUE);
           AsyncCommand("STAT", MAIL_CHK_MSG_COUNT,
                        POP3_QUIT_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case MAIL_CHK_MSG_COUNT:
           /* get the message count */
           BufferPtr = AsyncGetBufPtr();
           MsgCount = (int) wilParseDecimal(BufferPtr+4);
           wsprintf((LPSTR)Temp,"  %d messages waiting",MsgCount);
           DisplayLine((LPSTR)Temp);
           /* look at each message in turn */
           MsgIndex = 0;
           POST_MSG(MAIL_CHK_NEXT_HEADER);
           break;

         case MAIL_CHK_NEXT_HEADER:
           /* is there another header ? */
           MsgIndex++;
           if(MsgIndex>MsgCount)
             {/* no more headers, so sign off POP server */
              POST_MSG(POP3_QUIT_BEGIN);
              break;
             }
           /* get header for message <MsgIndex> */
           wsprintf((LPSTR)Temp,"TOP %d 0", MsgIndex);
           AsyncCommand((LPSTR)Temp, MAIL_CHK_TOP_SUCCESS,
                        POP3_QUIT_FAILURE, ASYNC_PLUS_MINUS);
           break;

         case MAIL_CHK_TOP_SUCCESS:
           /* display header info */
           wsprintf((LPSTR)Temp,"Msg %d: ",MsgIndex);
           DisplayString((LPSTR)Temp);
           /* read 1st line of header info */
           LastChar = '\n';
           AsyncSetEcho(EchoFlag);
           AsyncRead(MAIL_CHK_READ_SUCCESS, POP3_QUIT_FAILURE,
                     ASYNC_SINGLE_LINE);
           break;

         case MAIL_CHK_READ_SUCCESS:
           /* read was successfull */
           BufferPtr = AsyncGetBufPtr();
           /* check if this is the last line */
           if((LastChar=='\n')&&(*BufferPtr=='.')&&(*(BufferPtr+1)=='\r') )
             {/* end of this header message */
              DisplayLine("<EOM>");
              POST_MSG(MAIL_CHK_NEXT_HEADER);
              break;
             }
           /* got good header line */
           n = lstrlen((LPSTR)BufferPtr);
           LastChar = *(BufferPtr+n-1);
           /* look for "From" or "Subject" */
           Code = IsLeftString(BufferPtr,(LPSTR)"From");
           if(!Code) Code = IsLeftString(BufferPtr,(LPSTR)"Subject");
           if(Code)
             {StripCRLF(BufferPtr);
              wsprintf((LPSTR)Temp,"%s ",BufferPtr);
              DisplayString((LPSTR)Temp);
             }
           /* read next header line */
           AsyncRead(MAIL_CHK_READ_SUCCESS, POP3_QUIT_FAILURE, ASYNC_SINGLE_LINE);
           break;

         case MAIL_DEL_BEGIN:
           AsyncSetEcho(TRUE);
           /* delete message */
           wsprintf((LPSTR)Temp,"DELE %d", MailID);
           AsyncCommand((LPSTR)Temp, POP3_QUIT_BEGIN, POP3_QUIT_FAILURE,
                        ASYNC_PLUS_MINUS);
           break;

         case MAIL_GET_BEGIN:
           AsyncSetEcho(TRUE);
           /* send RETR (retrieve) command, & get 1st line of mail */
           wsprintf((LPSTR)Temp,"RETR %d", MailID);
           AsyncCommand((LPSTR)Temp,MAIL_GET_READ, POP3_QUIT_FAILURE,
                        ASYNC_SINGLE_LINE);
           LastChar = '\n';
           break;

         case MAIL_GET_READ:
           /* got line of text */
           BufferPtr = AsyncGetBufPtr();
           BufSize = lstrlen((LPSTR)BufferPtr);
           /* is there a '.' in CC1 ? */
#if 0
wsprintf((LPSTR)Temp,"<%c>",*BufferPtr);/*DEL*/
DisplayLine((LPSTR)Temp);/*DEL*/
#endif
           if((LastChar=='\n')&&(*BufferPtr=='.')&&(*(BufferPtr+1)=='\r') )
             {/* close file */
              DisplayLine("<EOM>");
              wsprintf((LPSTR)Temp,"Message %d to file %s",
                         MailID,(LPSTR)MailFile);
              DisplayLine((LPSTR)Temp);
              _lclose(Handle);
              DisplayLine("");
              POST_MSG(POP3_QUIT_BEGIN);
              break;
             }
           /* record last char of line */
           if(BufSize) LastChar = BufferPtr[BufSize-1];
           else LastChar = '\n'; /*???*/
           /* write to disk */
           Code = _lwrite(Handle,(LPSTR)BufferPtr,BufSize);
           if(Code<0)
             {DisplayLine("Error writting disk");
              _lclose(Handle);
              POST_MSG(POP3_QUIT_FAILURE);
              break;
             }
           /* wait for next line */
           AsyncRead(MAIL_GET_READ, POP3_QUIT_FAILURE, ASYNC_SINGLE_LINE);
           break;

         case MAIL_PUT_BEGIN:
           AsyncSetEcho(TRUE);
           /* send mail using SMTP */
           DisableMailMenuItems();
           SetCursor(WaitCursor);
           Socket = AsyncConnect(hMainWnd,"SMTP",(LPSTR)SmtpString,
                     SMTP_PORT, MAIL_PUT_CONN_SUCCESS,
                     SMTP_QUIT_FAILURE, ASYNC_SINGLE_LINE);
           break;

         case MAIL_PUT_CONN_SUCCESS:
           /* send HELO (hello) command */
           Ptr = StringChar((LPSTR)FromString,'@');
           wsprintf((LPSTR)Temp,"HELO %s",(LPSTR)(Ptr+1));
           AsyncCommand((LPSTR)Temp, MAIL_PUT_HELO_SUCCESS,
                       SMTP_QUIT_FAILURE, ASYNC_SINGLE_CODED);
           break;

         case MAIL_PUT_HELO_SUCCESS:
           /* send MAIL FROM command */
           wsprintf((LPSTR)Temp,"MAIL FROM:<%s>",(LPSTR)FromString);
           AsyncCommand((LPSTR)Temp,MAIL_PUT_FROM_SUCCESS,
                       SMTP_QUIT_FAILURE, ASYNC_SINGLE_CODED);
           break;

         case MAIL_PUT_FROM_SUCCESS:
           /* send RCPT TO (receipt to) command */
           wsprintf((LPSTR)Temp,"RCPT TO:<%s>",(LPSTR)ToString);
           AsyncCommand((LPSTR)Temp, MAIL_PUT_TO_SUCCESS,
                       SMTP_QUIT_FAILURE, ASYNC_SINGLE_CODED);
           break;

         case MAIL_PUT_TO_SUCCESS:
           /* send DATA command */
           AsyncCommand("DATA", MAIL_PUT_DATA_SUCCESS,
                        SMTP_QUIT_FAILURE, ASYNC_SINGLE_CODED);
           break;

         case MAIL_PUT_DATA_SUCCESS:
           AsyncSetEcho(EchoFlag);
           /* get next line from disk */
           BufSize = ReadTextLine(Handle, (LPSTR)InBuffer, MAX_BUF);
           if(BufSize<=0)
             {/* end of file */
              _lclose(Handle);
              /* append line with only '.' */
              wilWriteString(Socket,".\r\n");
              DisplayLine("<EOF>");
              POST_MSG(SMTP_QUIT_BEGIN);
              break;
             }
           /* ready to send text line */
           POST_MSG(MAIL_PUT_SEND);
           break;

         case MAIL_PUT_SEND:
           /* does buffer consist of lone '.' ? */
           if((InBuffer[0]=='.')&&(InBuffer[1]=='\r')&&(InBuffer[2]=='\n'))
             {/* "quote" the period */
              InBuffer[1] = '.';
              InBuffer[2] = '\r';
              InBuffer[3] = '\n';
              InBuffer[4] = '\0';
              BufSize = 4;
             }
#if 0
           /* display # bytes being sent */
           wsprintf((LPSTR)Temp,"Sending %d bytes.", BufSize);
           DisplayLine((LPSTR)Temp);
#endif
           /* send the buffer */
           Code = wilWriteSocket(Socket,(LPSTR)InBuffer,BufSize);
           if(Code<0)
             {/* socket write error */
              DisplayError(Code, "WriteSocket:");
              POST_MSG(SMTP_QUIT_FAILURE);
              break;
             }
           if(EchoFlag) WriteTheString((LPSTR)InBuffer,BufSize);
           /* go get another buffer to write */
           BufSize = 0;
           POST_MSG(MAIL_PUT_DATA_SUCCESS);
           break;

         case SMTP_QUIT_BEGIN:
           AsyncSetEcho(TRUE);
           /* send QUIT command */
           AsyncCommand("QUIT", SMTP_QUIT_SUCCESS,
                        SMTP_QUIT_FAILURE, ASYNC_SINGLE_CODED);
           break;

         case SMTP_QUIT_FAILURE:
           DisplayLine("SMTP command has failed...");
           EnableMailMenuItems();
           wilCloseSocket(Socket,500);
           SetCursor(ArrowCursor);
           break;
           /* fall thru... */

         case SMTP_QUIT_SUCCESS:
           /* all done */
           DisplayLine("SMTP command succeeds.");
           EnableMailMenuItems();
           wilCloseSocket(Socket,500);
           SetCursor(ArrowCursor);
           break;
        } /* end-switch(wParam) */

    case WM_PAINT:
      HideCaret(hMainWnd);
      hDC = BeginPaint(hMainWnd, &ps);
      SetMapMode(hDC,MM_ANISOTROPIC);
      SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
      PaintMain(hDC,&ps);
      EndPaint(hMainWnd,&ps);
      SetCaretPos(PaintGetColPos(),PaintGetRowPos());
      ShowCaret(hMainWnd);
      break;

    case WM_DESTROY:
      PostQuitMessage(0);
      break;

    case WM_CHAR:
      /* user just typed character */
      if(wParam==CR)
        {/* user has completed input */
         DestroyCaret();
         EnableMailMenuItems();
         PassFlag = FALSE;
         DisplayChar(LF);
         InBuffer[InBufLen] = '\0';
         /* excute command */
         switch(InBufCmd)
            {case MSG_PASS:
               InBufCmd = 0;
               /* save password string */
               lstrcpy((LPSTR)PassString,(LPSTR)InBuffer);
               break;

             case MSG_MAIL_DEL:
               InBufCmd = 0;
               /* get message number to delete */
               MailID = (int) wilParseDecimal((LPSTR)InBuffer);
               wsprintf((LPSTR)Temp,"Deleting message %d",MailID);
               Pop3Command(MAIL_DEL_BEGIN, POP3_QUIT_FAILURE);
               break;

             case MSG_MAIL_GET:
               InBufCmd = 0;
               MailID = (int) wilParseDecimal((LPSTR)InBuffer);
               /* contruct unique mail file name (starting with date) */
               unique((LPSTR)MailFile);
               wsprintf((LPSTR)Temp,"Saving message %d to file %s",
                         MailID,(LPSTR)MailFile);
               DisplayLine((LPSTR)Temp);
               /* create mail file */
               Handle = _lcreat((LPSTR)MailFile,0);
               if(Handle<0)
                 {DisplayString("Cannot create ");
                  DisplayLine((LPSTR)MailFile);
                  break;
                 }
               /* start up mail receive */
               wsprintf((LPSTR)Temp,"File %s is created",(LPSTR)MailFile);
               DisplayLine((LPSTR)Temp);
               Pop3Command(MAIL_GET_BEGIN, POP3_QUIT_FAILURE);
               break;

             case MSG_MAIL_PUT:
               InBufCmd = 0;
               /* open mail file */
               lstrcpy((LPSTR)MailFile, (LPSTR)InBuffer);
               Handle = _lopen((LPSTR)MailFile,OF_READ|OF_SHARE_DENY_WRITE);
               if(Handle<0)
                 {DisplayString("Cannot open ");
                  DisplayLine((LPSTR)MailFile);
                  break;
                 }
               wsprintf((LPSTR)Temp,"File %s opened",(LPSTR)MailFile);
               DisplayLine((LPSTR)Temp);
               /* read email address from 1st line of mail file */
               *ToString = '\0';
               for(i=0;i<STR_SIZE; )
                 {n = _lread(Handle,(LPSTR)&Chr,1);
                  if(n<=0) break;
                  if(Chr=='\n')
                     {ToString[i] = '\0';
                      break;
                     }
                  if(Chr=='\r') ;
                  else ToString[i++] = Chr;
                 }
               /* display IP address */
               wsprintf((LPSTR)Temp,"Address ='%s'", (LPSTR)ToString);
               DisplayLine(Temp);
               /* verify we really have an email address */
               n = lstrlen((LPSTR)ToString);
               for(i=0;i<n;i++)
                 {Chr = ToString[i];
                  if((Chr!='@')&&(Chr!='.')&&(Chr!='_')&&(!isalnum(Chr)))
                    {/* bad ermail address */
                     DisplayLine("Bad email address.");
                     DisplayLine("Email address must be 1st line of file.");
                     n = 0;
                     break;
                    }
                 }
               if(n==0) break;
               /* send mail [SMTP] */
               Ptr = StringChar((LPSTR)FromString,'@');
               if(Ptr==NULL)
                 {wsprintf((LPSTR)Temp,"Missing address in %s",(LPSTR)FromString);
                  DisplayLine((LPSTR)Temp);
                  break;
                 }
               POST_MSG(MAIL_PUT_BEGIN);
               break;

             default:
               break;
            }
        }
      else
        {/* add char to input buffer */
         Add2Buffer((char) wParam);
        }
      break;

    default:
      return (DefWindowProc(hMainWnd, iMsg, wParam, lParam));
   }
 return 0;
} /* end MainWndProc */
