/*  EZGdi v2.00                                                  */
/* A simple interface to the Win32 GDI.							 */
/* This file should be linked with a DrawThread function.	     */
/* This file contains the drawing interface and support code.    */

/* written and copyrighted by Peter Neubauer (1998)              */
/* page flipping, mouse support, and dibsection support:
		written by Mike Morton */
/* page flipping removed and code cleanup by Peter Neubauer (1999)*/

/* CONTACT: Peter Neubauer <peterjn@asu.edu>  */

#define  STRICT
#include <windows.h>
#include <stdlib.h>
#include <string.h>

//if not defined, the KeyCallBack function will not be needed or called
#define USEKEYCALLBACK
#define TITLE "ezGDI v2.00"


//global vars for window and bitmap info
HINSTANCE hInstance;
HDC hBuffer = 0;
HBITMAP hBmpOld = 0;
HPEN hPenOld = 0;
HBRUSH hBrushOld = 0;
HWND hWnd = 0;
HANDLE hThread;
DWORD dwThreadID;
char cKeyPressed;

//mouse information structure
class MouseInfo {
	public:
	int xpos, ypos;
	char mousemove;
	char lbutton, rbutton, mbutton;
	char dclickl;
	char was_dclickl;
};

MouseInfo Mouse;


/*interface to draw*/
DWORD WINAPI DrawThread( LPVOID );
void KeyCallBack(char c);

//current drawing color and fill color
COLORREF crColor     = RGB(255, 255, 255);
COLORREF crFillColor = RGB(0, 0, 0);

//page flipping buffer handles
HBITMAP Page_1;
HBITMAP Page_2;
HBITMAP Page_FAST_1;
HBITMAP Page_FAST_2;

int extern ERROR_BITMAP = 0;

inline void NormalizeRect(RECT* r)
{  //fix a backwards rect so that top <= bottom and left <= right
	LONG temp;

	if (r->left > r->right)  {
		temp = r->left;
		r->left = r->right;
		r->right = temp;
	}
	if (r->top > r->bottom)  {
		temp = r->top;
		r->top = r->bottom;
		r->bottom = temp;
	}
}

inline void GrowRect(RECT* r)
{  // expand rect by 1 unit in all direction
	r->left--;
	r->right++;
	r->top--;
	r->bottom++;
}

void ClearScreen() 
{
    BitBlt(hBuffer,0,0,640,480,NULL,0,0,BLACKNESS);
	InvalidateRect(hWnd, NULL, 0);
}


void DrawLine_(int x1, int y1, int x2, int y2)
{
	MoveToEx(hBuffer, x1, y1, 0);
	LineTo(hBuffer, x2, y2);

	RECT r = {x1, y1, x2, y2};
	NormalizeRect(&r);
	GrowRect(&r);
	InvalidateRect(hWnd, &r, 0);
}


void SetPixel_(int x, int y)
{
	SetPixel(hBuffer, x, y, crColor);

	RECT r = {x-1, y-1, x+1, y+1};
	InvalidateRect(hWnd, &r, 0);
}

void SetColor_(BYTE r, BYTE g, BYTE b)
{
	crColor = RGB(r, g, b);
	DeleteObject(SelectObject(hBuffer, CreatePen(PS_SOLID, 0, crColor)));
	SetTextColor(hBuffer, crColor);
}

void SetLineStyle_( int fnPenStyle, int nWidth)
{
	DeleteObject(SelectObject(hBuffer, CreatePen(fnPenStyle, nWidth, crColor)));
}


void SetFillColor_(BYTE r, BYTE g, BYTE b, UINT style = BS_SOLID, LONG hatch = 0)
{
	LOGBRUSH tmp;
	crFillColor = RGB(r, g, b);
	tmp.lbStyle = style;
	tmp.lbColor = crFillColor;
	tmp.lbHatch = hatch;

	DeleteObject(SelectObject(hBuffer, CreateBrushIndirect(&tmp)));
}


void FloodFill_(int x, int y, COLORREF Boundary = crColor)
{
	ExtFloodFill(hBuffer, x, y, Boundary, FLOODFILLBORDER);

	InvalidateRect(hWnd, NULL, 0);
}


void Ellipse_(int Left, int Top, int Right, int Bottom)
{
	RECT r = {Left, Top, Right, Bottom};
	NormalizeRect(&r);

	Ellipse(hBuffer, r.left, r.top, r.right, r.bottom);

	GrowRect(&r);	
	InvalidateRect(hWnd, &r, 0);
}

void Rectangle_(int Left, int Top, int Right, int Bottom)
{
	RECT r = {Left, Top, Right, Bottom};
	NormalizeRect(&r);

	Rectangle(hBuffer, r.left, r.top, r.right, r.bottom);

	GrowRect(&r);
	InvalidateRect(hWnd, &r, 0);
}

int GetMaxX_()
{
	RECT r;
	GetClientRect(hWnd, &r);
	return r.right;
}

int GetMaxY_()
{
	RECT r;
	GetClientRect(hWnd, &r);
	return r.bottom;
}

HBITMAP GetBitmap_(LPCSTR fname)
{
	return LoadBitmap(hInstance, fname);
}

HBITMAP GetBitmapFile(LPCSTR fname)
{
	return (HBITMAP)LoadImage(hInstance, fname, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE | LR_CREATEDIBSECTION);
}

void PutBitmap_(HBITMAP hBmp, int x, int y)
{
	if (hBmp == NULL)  {
	   	MessageBox(NULL, "Invalid bitmap handle.", "ezGDI - PutBitmap",
      			  MB_OK | MB_ICONEXCLAMATION | MB_SYSTEMMODAL | MB_TOPMOST);
      return;
	}

	HDC hTmp = CreateCompatibleDC(NULL);
	HBITMAP hOld = (HBITMAP)SelectObject(hTmp, hBmp);
	BitBlt(hBuffer, x, y, GetDeviceCaps(hTmp, HORZRES),
   		 GetDeviceCaps(hTmp, VERTRES), hTmp, 0, 0, SRCCOPY);
	SelectObject(hTmp, hOld);
	DeleteDC(hTmp);

	RECT r = {x - 1, y - 1, x+GetDeviceCaps(hTmp, HORZSIZE),
   		    y+GetDeviceCaps(hTmp, VERTSIZE)};
	InvalidateRect(hWnd, &r, 0);
}

void PutText_(LPCSTR str, int x, int y)
{
	SetBkMode(hBuffer, TRANSPARENT);
	TextOut(hBuffer, x, y, str, strlen(str));
	SetBkMode(hBuffer, OPAQUE);

	RECT r;
	GetClientRect(hWnd, &r);
	r.left = x;
	r.top  = y;
	InvalidateRect(hWnd, &r, 0);

}


COLORREF GetPixel_(int x, int y)
{
	return GetPixel(hBuffer, x, y);
}

void SetTitle_(LPCSTR title)
{
	char temp[80];
 	wsprintf(temp, "%s - %s", TITLE, title);
	SetWindowText(hWnd, temp);
}

void ChangeFont_(int nHeight, int fnWeight, bool fdwItalic, bool fdwUnderline,
                bool fdwStrikeOut, LPCSTR lpszFace)
{
	HFONT hFnt = CreateFont(nHeight, 0, 0, 0, fnWeight, fdwItalic,
		fdwUnderline, fdwStrikeOut, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS,
		CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
		DEFAULT_PITCH | FF_DONTCARE, lpszFace);

	DeleteObject(SelectObject(hBuffer, hFnt));
}


HBITMAP CaptureBitmap(int x, int y, int width, int height)
{
	HBITMAP hCapBmp;
	hCapBmp = CreateCompatibleBitmap(hBuffer, width, height);
	if (hCapBmp == NULL) return 0;
	HDC hCapDC = CreateCompatibleDC(hBuffer);
	HBITMAP hBmpOld = (HBITMAP)SelectObject(hCapDC, hCapBmp);

	if (!BitBlt(hCapDC, 0, 0, width, height, hBuffer, x, y, SRCCOPY))  {
		MessageBox(NULL, "Capture failed.", "ezGDI - CaptureBitmap",
      			  MB_OK | MB_ICONEXCLAMATION | MB_SYSTEMMODAL | MB_TOPMOST);
	}

	SelectObject(hCapDC, hBmpOld);
	DeleteDC(hCapDC);
	return hCapBmp;
}





//support code for windows

void InitBuffer()
{  //create the back buffer

	hBuffer = CreateCompatibleDC(NULL);
	HDC hDC = GetDC(hWnd);
	   
	BITMAPINFOHEADER BInf;
	BInf.biSize = sizeof(BITMAPINFOHEADER);
    BInf.biWidth = 640;
    BInf.biHeight = -480;
    BInf.biPlanes = 1;
    BInf.biBitCount = 16;
    BInf.biCompression = BI_RGB;
    BInf.biSizeImage = 0;
    BInf.biXPelsPerMeter = 0;
    BInf.biYPelsPerMeter = 0;
    BInf.biClrUsed = 0;
    BInf.biClrImportant = 0;
	
	BITMAPINFO BPalInf;
	BPalInf.bmiHeader = BInf;

	HBITMAP hDIBSect = (HBITMAP) CreateDIBSection(hBuffer, &BPalInf, DIB_RGB_COLORS, NULL, NULL, 0);
   
	hBmpOld = (HBITMAP) SelectObject(hBuffer, hDIBSect);
	BitBlt(hBuffer,0,0,640,480,NULL,0,0,BLACKNESS);

	hBrushOld = (HBRUSH) SelectObject(hBuffer, GetStockObject(HOLLOW_BRUSH));
	hPenOld   = (HPEN) SelectObject(hBuffer, GetStockObject(WHITE_PEN));

	//init thread
	hThread = CreateThread(0, 0, DrawThread,
   	                    0, CREATE_SUSPENDED, &dwThreadID);

}


void Paint()
{  //update our window as needed in response to WM_PAINT
	PAINTSTRUCT paint;
	HDC hDC = BeginPaint(hWnd, &paint);

	if (hBuffer == NULL)
		InitBuffer();

	int l = paint.rcPaint.left,
		r = paint.rcPaint.right,
		t = paint.rcPaint.top,
		b = paint.rcPaint.bottom;

   BitBlt(hDC, l, t, r, b, hBuffer, l, t, SRCCOPY);
        
   EndPaint(hWnd, &paint);

}


void Draw()
{   
	if (!hBuffer) return;

	SuspendThread(hThread);
	
	InvalidateRect(hWnd, NULL, 1);

	ResumeThread(hThread);
}



void Cleanup()
{
	hBmpOld   = (HBITMAP) SelectObject(hBuffer, hBmpOld);
	hBrushOld = (HBRUSH) SelectObject(hBuffer, hBrushOld);
	hPenOld   = (HPEN) SelectObject(hBuffer, hPenOld);
	DeleteObject(hBmpOld);
	DeleteObject(hBrushOld);
	DeleteObject(hPenOld);
	DeleteObject(hBuffer);
}


LRESULT CALLBACK  WndProc( HWND hWnd, UINT iMessage, WPARAM wParam, LPARAM lParam )
{
	Mouse.dclickl = 0;
	Mouse.mousemove = 0;

	switch (iMessage)
    {
        case WM_CREATE:
            InitBuffer();
            Draw();
            ResumeThread(hThread);
	        break;
        case WM_PAINT:
            Paint();
            break;
        case WM_DESTROY:
       		TerminateThread(hThread, 0);
			hThread = NULL;
			dwThreadID = 0;
       		Cleanup();
            PostQuitMessage( 0 );
            break;
        case WM_ACTIVATE:
        	if (LOWORD(wParam) != WA_INACTIVE)
   				ResumeThread(hThread);
			else
            	SuspendThread(hThread);
			break;

        //case WM_LBUTTONDOWN:
        //		Draw();
        //    ResumeThread(hThread);
        //    break;

        case WM_CHAR:
	        cKeyPressed = (char)wParam;
            #ifdef USEKEYCALLBACK
        		KeyCallBack(cKeyPressed);
            #endif
            break;
        case WM_MOUSEMOVE:
            Mouse.xpos = LOWORD(lParam);
				Mouse.ypos = HIWORD(lParam);
            Mouse.mousemove = 1;

            if ( (wParam & MK_LBUTTON) == MK_LBUTTON )
            	Mouse.lbutton = 1;
            else
               Mouse.lbutton = 0;

            if ( (wParam & MK_RBUTTON) == MK_RBUTTON )
            	Mouse.rbutton = 1;
            else
            	Mouse.rbutton = 0;

            if ( (wParam & MK_MBUTTON) == MK_MBUTTON )
            	Mouse.mbutton = 1;
            else
            	Mouse.mbutton = 0;

            break;

        case WM_LBUTTONDOWN:
        	Mouse.lbutton = 1;
            break;

        case WM_LBUTTONUP:
        	Mouse.lbutton = 0;
            break;

        case WM_LBUTTONDBLCLK:
        	Mouse.dclickl = 1;
            Mouse.was_dclickl = 1;
            break;

        case WM_RBUTTONDOWN:
            Mouse.rbutton = 1;
            break;

        case WM_RBUTTONUP:
        		Mouse.rbutton = 0; 
        	   break;

        default:
            return DefWindowProc( hWnd, iMessage, wParam, lParam );
    }
    return 0;
}




int PASCAL WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpszCmdLine,
                    int nCmdShow )
{
    ::hInstance = hInstance;

    //register/create wndclass
    if ( ! hPrevInstance ) {
		WNDCLASS wndclass;   // Structure used to register Windows class.

	    wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
   		wndclass.lpfnWndProc   = WndProc;
	    wndclass.cbClsExtra    = 0;
	    wndclass.cbWndExtra    = NULL;
	    wndclass.hInstance     = hInstance;
	    wndclass.hIcon         = LoadIcon( hInstance, MAKEINTRESOURCE(1) );
	    wndclass.hCursor       = LoadCursor( NULL, IDC_ARROW );
	    wndclass.hbrBackground = NULL;
	    wndclass.lpszMenuName  = NULL;
	    wndclass.lpszClassName = "EZGDI";

	    if ( ! RegisterClass( &wndclass ) )
	         exit( FALSE );
	}

	hWnd = CreateWindow( "EZGDI",
            TITLE,
            WS_OVERLAPPEDWINDOW,
            CW_USEDEFAULT,
            0,
            640,
            480,
            NULL,
            NULL,
            hInstance,
            0);

	if ( ! hWnd )
        exit( FALSE );

	ShowWindow( hWnd, nCmdShow );
	UpdateWindow( hWnd );


    //message pump
    MSG msg;
    while(1)
    {
        if (PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE))
        {
			if (GetMessage(&msg, NULL, 0, 0) == 0)
           		return msg.wParam;

			TranslateMessage( &msg );
   			DispatchMessage( &msg );
        } else 
			WaitMessage();
    }
}
