/*
 * Copyright (c) 1997,1998
 * Babes-Bolyai University, Cluj-Napoca
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Babes-Bolyai University makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 */

/*
 * NOTE: This is an internal source file, included by other OCL files.
 *   You should not attempt to use it directly.
 */

/////////////////////////////////////////////////////////////////////////////

#include <algorithm>
#include <iterator>

#ifndef __OCL_NO_NAMESPACES
namespace ocl {
#endif

#ifndef __OCL_NO_STD
using namespace std;
#endif

/////////////////////////////////////////////////////////////////////////////
// OCL Collection Types - global functions - implementation

template <class FirstType, class SecondType, class ResultType>
ResultType reunion(
    const FirstType& first_collection, const SecondType& second_collection)
{
    ResultType result;
    set_union(
        first_collection.begin(), first_collection.end(),
        second_collection.begin(), second_collection.end(),
        inserter(result, result.begin()));
    return result;
}

template <class FirstType, class SecondType, class ResultType>
ResultType intersection(
    const FirstType& first_collection, const SecondType& second_collection)
{
    ResultType result;
    set_intersection(
        first_collection.begin(), first_collection.end(),
        second_collection.begin(), second_collection.end(),
        inserter(result, result.begin()));
    return result;
}

template <class FirstType, class SecondType, class ResultType>
ResultType difference(
    const FirstType& first_collection, const SecondType& second_collection)
{
    ResultType result;
    set_difference(
        first_collection.begin(), first_collection.end(),
        second_collection.begin(), second_collection.end(),
        inserter(result, result.begin()));
    return result;
}

template <class FirstType, class SecondType, class ResultType>
ResultType symmetricDifference(
    const FirstType& first_collection, const SecondType& second_collection)
{
    ResultType result;
    set_symmetric_difference(
        first_collection.begin(), first_collection.end(),
        second_collection.begin(), second_collection.end(),
        inserter(result, result.begin()));
    return result;
}

template <class SourceType, class DestinationType>
DestinationType __clone(const SourceType& a_collection)
{
    DestinationType result;
    copy(a_collection.begin(), a_collection.end(),
        inserter(result, result.begin()));
    return result;
}

/////////////////////////////////////////////////////////////////////////////
// OCL Collection - implementation

template <class Type, class Container>
inline Integer Collection<Type, Container>::size() const
{
    return (Integer) Container::size();
}

template <class Type, class Container>
inline Integer Collection<Type, Container>::count(const Type& an_object) const
{
    return (Integer) Container::count(an_object);
}

template <class Type, class Container>
inline Boolean Collection<Type, Container>::includes(const Type& an_object)
    const
{
    return (Boolean) (find(an_object) != end());
}

template <class Type, class Container>
inline Boolean Collection<Type, Container>::includesAll(
    const Collection<Type, Container>& a_collection) const
{
    return (Boolean) (includes(begin(), end(),
        a_collection.begin(), a_collection.end()));
}

template <class Type, class Container>
inline Boolean Collection<Type, Container>::isEmpty() const
{
    return (Boolean) Container::empty();
}

template <class Type, class Container>
inline Boolean Collection<Type, Container>::notEmpty() const
{
    return (Boolean) !Container::empty();
}

template <class Type, class Container>
Type Collection<Type, Container>::sum() const
{
    Type result = 0;

    for (const_iterator i = begin(); i != end(); ++i)
        result = result + *i;  // use 'operator +' instead of 'operator +='
    return result;
}

/////////////////////////////////////////////////////////////////////////////
// OCL Set - implementation

template <class Type, class Pred, class Allocator>
inline Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::reunion(
    const Set<Type, Pred, Allocator>& a_set) const
{
    // Some ugly #ifndefs, because otherwise C++ would get confused...
    return
#ifndef __OCL_NO_NAMESPACES
        ::ocl
#endif
        ::reunion<Set<Type, Pred, Allocator>, Set<Type, Pred, Allocator>,
            Set<Type, Pred, Allocator> >(*this, a_set);
}

template <class Type, class Pred, class Allocator>
inline Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::reunion(
    const Bag<Type, Pred, Allocator>& a_bag) const
{
    return reunion(*this, a_bag);
}

template <class Type, class Pred, class Allocator>
inline Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::intersection(
    const Set<Type, Pred, Allocator>& a_set) const
{
    return
#ifndef __OCL_NO_NAMESPACES
        ::ocl
#endif
        ::intersection<Set<Type, Pred, Allocator>,
            Set<Type, Pred, Allocator>,
            Set<Type, Pred, Allocator> >(*this, a_set);
}

template <class Type, class Pred, class Allocator>
inline Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::intersection(
    const Bag<Type, Pred, Allocator>& a_bag) const
{
    return intersection(*this, a_bag);
}

template <class Type, class Pred, class Allocator>
Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::difference(
    const Set<Type, Pred, Allocator>& a_set) const
{
    Set<Type, Pred, Allocator> result;
    set_difference(
        begin(), end(), a_set.begin(), a_set.end(),
        inserter(result, result.begin()));
    return result;
}

template <class Type, class Pred, class Allocator>
Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::symmetricDifference(
    const Set<Type, Pred, Allocator>& a_set) const
{
    Set<Type, Pred, Allocator> result;
    set_symmetric_difference(
        begin(), end(), a_set.begin(), a_set.end(),
        inserter(result, result.begin()));
    return result;
}

template <class Type, class Pred, class Allocator>
inline Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::including(const Type& an_object) const
{
    Set<Type, Pred, Allocator> result(*this);
    result.insert(an_object);
    return result;
}

template <class Type, class Pred, class Allocator>
inline Set<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::excluding(const Type& an_object) const
{
    Set<Type, Pred, Allocator> result(*this);
    result.erase(an_object);
    return result;
}

template <class Type, class Pred, class Allocator>
inline Bag<Type, Pred, Allocator>
Set<Type, Pred, Allocator>::asBag() const
{
    return __clone<Set<Type, Pred, Allocator>, Bag<Type, Pred, Allocator> >
        (*this);
}

template <class Type, class Pred, class Allocator>
inline Sequence<Type, Allocator>
Set<Type, Pred, Allocator>::asSequence() const
{
    return __clone<Set<Type, Pred, Allocator>, Sequence<Type, Allocator> >
        (*this);
}

/////////////////////////////////////////////////////////////////////////////
// OCL Bag - implementation

template <class Type, class Pred, class Allocator>
inline Bag<Type, Pred, Allocator>
Bag<Type, Pred, Allocator>::reunion(
    const Bag<Type, Pred, Allocator>& a_bag) const
{
    return
#ifndef __OCL_NO_NAMESPACES
        ::ocl
#endif
        ::reunion<Bag<Type, Pred, Allocator>, Bag<Type, Pred, Allocator>,
            Bag<Type, Pred, Allocator> >(*this, a_bag);
}

template <class Type, class Pred, class Allocator>
inline Bag<Type, Pred, Allocator>
Bag<Type, Pred, Allocator>::reunion(
    const Set<Type, Pred, Allocator>& a_set) const
{
    return reunion(*this, a_set);
}

template <class Type, class Pred, class Allocator>
inline Bag<Type, Pred, Allocator>
Bag<Type, Pred, Allocator>::intersection(
    const Bag<Type, Pred, Allocator>& a_bag) const
{
    return
#ifndef __OCL_NO_NAMESPACES
        ::ocl
#endif
        ::intersection<Bag<Type, Pred, Allocator>,
            Bag<Type, Pred, Allocator>,
            Bag<Type, Pred, Allocator> >(*this, a_bag);
}

template <class Type, class Pred, class Allocator>
inline Bag<Type, Pred, Allocator>
Bag<Type, Pred, Allocator>::intersection(
    const Set<Type, Pred, Allocator>& a_set) const
{
    return intersection(*this, a_set);
}

template <class Type, class Pred, class Allocator>
inline Bag<Type, Pred, Allocator>
Bag<Type, Pred, Allocator>::including(const Type& an_object) const
{
    Bag<Type, Pred, Allocator> result(*this);
    result.insert(an_object);
    return result;
}

template <class Type, class Pred, class Allocator>
inline Bag<Type, Pred, Allocator>
Bag<Type, Pred, Allocator>::excluding(const Type& an_object) const
{
    Bag<Type, Pred, Allocator> result(*this);
    result.erase(an_object);
    return result;
}

template <class Type, class Pred, class Allocator>
inline Set<Type, Pred, Allocator>
Bag<Type, Pred, Allocator>::asSet() const
{
    return __clone<Bag<Type, Pred, Allocator>, Set<Type, Pred, Allocator> >
        (*this);
}

template <class Type, class Pred, class Allocator>
inline Sequence<Type, Allocator>
Bag<Type, Pred, Allocator>::asSequence() const
{
    return __clone<Bag<Type, Pred, Allocator>, Sequence<Type, Allocator> >
        (*this);
}

/////////////////////////////////////////////////////////////////////////////
// OCL Sequence - implementation

template <class Type, class Allocator>
inline Sequence<Type, Allocator>
Sequence<Type, Allocator>::append(const Type& an_object) const
{
    Sequence<Type, Allocator> result(*this);
    result.insert(result.end(), an_object);
    return result;
}

template <class Type, class Allocator>
inline Sequence<Type, Allocator>
Sequence<Type, Allocator>::prepend(const Type& an_object) const
{
    Sequence<Type, Allocator> result(*this);
    result.insert(result.begin(), an_object);
    return result;
}

template <class Type, class Allocator>
inline Sequence<Type, Allocator>
Sequence<Type, Allocator>::including(const Type& an_object) const
{
    return append(an_object);
}

template <class Type, class Allocator>
Sequence<Type, Allocator>
Sequence<Type, Allocator>::excluding(const Type& an_object) const
{
    Sequence<Type, Allocator> result(*this);
    for (Sequence<Type, Allocator>::iterator i = result.begin();
            i != result.end(); ++i)
        if (*i == an_object)
            result.erase(i);
    return result;
}

template <class Type, class Allocator>
inline Type&
Sequence<Type, Allocator>::operator [] (Integer position)
{
    return vector::operator [] (position - 1);
}

template <class Type, class Allocator>
inline const Type&
Sequence<Type, Allocator>::operator [] (const Integer position) const
{
    return vector::operator [] (position - 1);
}

template <class Type, class Allocator>
inline Type&
Sequence<Type, Allocator>::at(Integer position)
{
    return vector::at(position - 1);
}

template <class Type, class Allocator>
inline const Type&
Sequence<Type, Allocator>::at(const Integer position) const
{
    return vector::at(position - 1);
}

template <class Type, class Allocator>
Sequence<Type, Allocator>
Sequence<Type, Allocator>::subSequence(
    const Integer lower, const Integer upper) const
{
    Sequence<Type, Allocator> result;
    copy(begin() + lower - 1, begin() + upper,
        inserter(result, result.begin()));
    return result;
}

template <class Type, class Allocator>
inline Type& Sequence<Type, Allocator>::first() const
{
    return front();
}

template <class Type, class Allocator>
inline Type& Sequence<Type, Allocator>::last() const
{
    return back();
}

template <class Type, class Allocator>
inline Set<Type, __OclPred<Type>, Allocator>
Sequence<Type, Allocator>::asSet() const
{
    return __clone<Sequence<Type, Allocator>,
        Set<Type, __OclPred<Type>, Allocator> >(*this);
}

template <class Type, class Allocator>
inline Bag<Type, __OclPred<Type>, Allocator>
Sequence<Type, Allocator>::asBag() const
{
    return __clone<Sequence<Type, Allocator>,
        Bag<Type, __OclPred<Type>, Allocator> >(*this);
}

/////////////////////////////////////////////////////////////////////////////

#ifndef __OCL_NO_NAMESPACES
}  // namespace ocl
#endif
