/** \file DS_Enumeration.h
    Linear representation of data structures.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/

////////////////////////////////////////////////////////////
// Enumeration
//
// Purpose  - Create a serial list reference to another data structure.
// Creation - Enumeration objects are returned by other data structures,
//            upon request for an enumeration of their elements.
//
////////////////////////////////////////////////////////////

#ifndef H_DS_ENUMERATION
#define H_DS_ENUMERATION

class Enumeration;  // Defined here, for use as a return type in Vector.

#include <DS_Vector.h>
#include <DS_Exceptions.h>

typedef long FN_FOR_EACH(Data);  // function prototype for forEach()

/** This class provides a serial list of elements.
    This is especially useful for non-vector containers.
    An instance of this class is returned by many containers' methods.

    If an enumeration is not initialized by construction, opeartor= or
    load(), NotInitialized will be thrown by most of its other methods.

    All enumerations created using the elements() method of a container
    are considered initialized, regardless of whether they have any
    elements.
*/
class Enumeration
{
public:
  /** Default Constructor: Construct an empty (non-initialized) enumeration.
      operator= or load() must be used before using any of this
      enumeration's methods. */
  Enumeration();

  /** Copy Constructor: Construct a copy of an enumeration. */
  Enumeration(const Enumeration& e)       { m_V = NULL; *this = e; }

  /** Copy a non-const enumeration and optionally discard source enumeration 
      if temp is non-zero.  Invalidates e if temp. */
  Enumeration(Enumeration& e, int temp)   { m_V = NULL; load(e, temp); }

  /** Construct an enumeration of any container. */
  Enumeration(const Container& c)         { m_V = NULL; load(c);   }

  /** Construct an enumeration of a vector. */
  Enumeration(const Vector& v)            { m_V = NULL; load(v);   }
  Enumeration(Vector& v)                  { m_V = NULL; load(v);   }
  Enumeration(const Vector* v)            { m_V = NULL; load(*v);  }   // Historical

  /** Destructor */
  virtual ~Enumeration();

  /** Assignment: Copy an enumeration. */
  Enumeration& operator=(const Enumeration&);

  /** Returns non-zero if there are more elements to be read. */
  DS_BOOL hasMoreElements() const;

  /** Returns the next element.
      Throws IndexOutOfBounds if there are no more elements. */
  Data nextElement();

  /** Execute a function on each element from the current element to the last.
      If fn(item) returns non-zero for any element, forEach() stops and returns that value. 
      If, for each element, fn(item) returns 0, forEach() also returns 0. */ 
  long forEach(FN_FOR_EACH* fn);

  /** Reset this enumeration to read the elements again from the start (or end). */
  void reset();

  /** Reverse the order of reading.
      To read from the start (or end), reset() must be applied afterwards. */
  void reverse();

  /** Load this enumeration from any container.
      Returns the number of elements added (c.size()).  Resets this enumeration to the 
      beginning and sets the direction to forward.  Unlike Container::load(), 
      Enumeratiuon::load() is not cumulative (the enumeration is cleared before the load). */
  int  load(const Container& c);
  int  load(const Vector&);          // fast version for vectors
  int  load(Vector& v)                    { return load((const Vector&)v); } // same for non-const

  /** Load this enumeration from another enumeration. */
  int  load(const Enumeration& e)         { *this = e; return m_V ? m_V->size() : 0; }

  /** Load this enumeration from a non-const enum and optionally discard source
      if temp is non-zero. Invaildates e if temp. */
  int load(Enumeration& e, int temp=0);

  /** Returns non-zero if the container or enumeration from which this enumeration was built
      isDestructive(). */
  DS_BOOL isDestructive() const           { return m_isDestructive; }

private:
  Vector*      m_V;
  int          m_pos;
  int          m_dir;
  DS_BOOL      m_isDestructive;
  friend       Vector;
};


/* FOR_EACH
   Use this like a 'for' loop.  It will execute the statement (possibly compound)
   immediately following for each non-NULL remaining element of the enumeration.

      (ncc can be any Container of Number_Containable's)
   Ex.:  Enumeration e = Enumeration(ncc);
         Number_Containable* item;
         FOR_EACH(e,item,Number_Containable*)
            printf("%d\n", (int) item->getValue());
*/

#define FOR_EACH(enumeration,element,cast)                    \
   while (enumeration.hasMoreElements())                      \
      if (NULL==(element = (cast) enumeration.nextElement())) \
         ;                                                    \
      else
         

#endif // H_DS_ENUMERATION
