/** \file CON_Drawable.h
    Definition of a generic 'bitmap' interface.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.  
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_CON_DRAWABLE
#define H_CON_DRAWABLE

#include <iostream.h>
#include <CON_Basic.h>
#include <CON_Math2D.h>


/** This class is a base for all objects that are inherently bitmaps,
    (Currently: Screen & Bitmap)
    It contains methods that allow manipulating the object:
    - Drawing
    - Buffer access
    - Copying
    and more
*/
class Drawable : public Interface
{
public:

//////////////////////////////////////////////////////////////////////
  /**
  Category: Generic information retrieval methods
  */

  /** Returns the width of the drawable */
  virtual int  getWidth()  const = 0;

  /** Returns the height of the drawable */
  virtual int  getHeight() const = 0;

  /** Returns a non-zero value if an error occurred. */
  virtual long status()    const = 0;

  /** Returns non-zero if hareware acceleration is disabled
      for this Drawable.   Return value of 0 does not assure
      hardware acceleration. */
  virtual int  getNoAccel() const = 0;

//////////////////////////////////////////////////////////////////////
  /**
  Category: Surface modifying methods, that require no lock on the surface.
  */

  /** Fill the entire drawable with a specified color. */
  virtual long clear(TColor Color=0) = 0;

  /** Copies (Blits) a rectangle from the source to this drawable.
      Both must be unlocked. 
      x,y are the destination coordinates.
      x1,y1,x2,y2 is the source rectangle, defaulting to the entire area.
      Transparent: (-1 use drawable transparency, 0 no transparency, 1 force transparency.
      Returns >0 if copy was clipped out.
  */
  virtual long copy(Drawable* Source, int x, int y, int x1=-1, int y1=-1, int x2=-1, int y2=-1, int Transparent=-1) = 0;
          long copy(Drawable* Source, const Vector2D& v, const Rect2D& r, int Transparent=-1)
          { return copy(Source, v.x, v.y, r.tl.x, r.tl.y, r.br.x, r.br.y, Transparent); }

  /** This does the same as copy, but it doesn't clip the blit.  
      This can be used for optimization if the application does clipping.
  */
  virtual long rawCopy(Drawable* Source, int x, int y, int x1=-1, int y1=-1, int x2=-1, int y2=-1, int Transparent=-1) = 0;
          long rawCopy(Drawable* Source, const Vector2D& v, const Rect2D& r, int Transparent=-1)
          { return rawCopy(Source, v.x, v.y, r.tl.x, r.tl.y, r.br.x, r.br.y, Transparent); }

  /** Copies an arbitrary src rect to an arbitrary dest rect, stretching as necessary. 
      If a rect is properly empty (br.x < tl.x || br.y < tl.y), then the entire
      source or dest dimensions are used.
  */
  virtual long rawCopy(const Rect2D& destRect, Drawable* Source, const Rect2D& srcRect, int Transparent) = 0;

  /** Copies the a section of the source, scaling it to fit current drawable dimensions.
      The default section takes the entire sources. */
  virtual long scaleCopy(Drawable* Source, const Rect2D& Section=Rect2D()) = 0;

  /** Copies the image from the source, flipping it horizontally */
  virtual long flipHorizontal(Drawable* Source) = 0;

  /** Copies the image from the source, flipping it vertically */
  virtual long flipVertical(Drawable* Source) = 0;

  /** Copies the image from the source, rotating it around its center 
      pixels that cannot be obtained due to being out of the image,
      will be filled with the source current transparent color.
  */
  virtual long rotate(Drawable* Source, int Degrees) = 0;


//////////////////////////////////////////////////////////////////////
  /**
  Category: Internal data access, used for custom special effects.
  */

  /** Return the internal surface for creating special effects. 
      Care must be taken to return the surface to its original state (locks)
      to prevent internal inconsistencies.
      Return type should be cast to: LPDIRECTDRAWSURFACE4
  */
  virtual void* getDDrawSurface() = 0;


//////////////////////////////////////////////////////////////////////
  /**
  Category: Drawable buffer memory access methods.
            The following group of functions work directly with the bitmap memory.
            The Drawable object must be locked before any drawing, or buffer data
            manipulation.  It must be unlocked after all operations are done.
  */

  /** Returns a non-zero value if the Drawable is locked. */
  virtual int  isLocked() const = 0;

  /** Use Optimize=1 if you will do more than a few drawings. 
      This will pre-calculate offsets in the buffer. 
      If only a few single pixels are accessed, it is preferred
      not to use optimize. */
  virtual long lock(int Optimize=1, int WriteOnly=0) = 0;

  /** Unlocks the Drawable.  Must be done to enable other access to 
      the object */
  virtual long unlock() = 0;

  /** Draw a horizontal line at y, from x1 to x2 */
  virtual long drawHLine(int y, int x1, int x2, TColor color) = 0;
          long drawHLine(const Vector2D& p, const int x2, const TColor color)
          { return drawHLine(p.y, p.x, x2, color); }

  /** Draw a vertical line at x, from y1 to y2 */
  virtual long drawVLine(int x, int y1, int y2, TColor color) = 0;
          long drawVLine(const Vector2D& p, const int y2, const TColor color)
          { return drawVLine(p.x, p.y, y2, color); }

  /** Draw any line  */
  virtual long drawLine(int x1, int y1, int x2, int y2, TColor color) = 0;
          long drawLine(const Vector2D& u, const Vector2D& v, const TColor color)
          { return drawLine(u.x, u.y, v.x, v.y, color); }



//////////////////////////////////////////////////////////////////////
  /**
  Category: Clipping area methods.  The area described by the given
            coordinates defines the area where drawing is permitted.
  */

  /** Set an area outside of which nothing can be drawn */
  virtual long setClipArea(int x1, int y1, int x2, int y2) = 0;
          long setClipArea(const Rect2D& r)
          { return setClipArea(r.tl.x, r.tl.y, r.br.x, r.br.y); }

  /** Get the current Clip Area. */
  virtual long getClipArea(int& x1, int& y1, int& x2, int& y2) const = 0;
          long getClipArea(Rect2D& r) const
          { return getClipArea(r.tl.x, r.tl.y, r.br.x, r.br.y); }

  /** Check if a point is within the clipping area. */
  virtual long isInClipArea(int x, int y) const = 0;
          long isInClipArea(const Vector2D p) const
          { return isInClipArea(p.x, p.y); }
  
  
//////////////////////////////////////////////////////////////////////
  /**
  Category: Methods that give access to the object's actual
            memory buffer.   Care must be taken when using
            these methods.
            Drawable object must be locked or these methods
            will fail and return error.
            For more information, refer to the manual.
  */

  /** Returns a pointer to the beginning of the buffer.
      Returns NULL for all errors */
  virtual TColor* getBuffer() const = 0;

  /** Returns the size of a line in bytes.
      This includes pixel data and some extra information
      Only the pixel data should be modified
      For more information, refer to the manual. */
  virtual int     getPitch() const = 0;

  /** Returns a pointer to the beginning of a certain line 
      Returns NULL for all errors */
  virtual TColor* getLine(int y) const = 0;



//////////////////////////////////////////////////////////////////////
  /**
  Category: The following group of methods use the Windows GDI to draw.
            beginDraw must be called before any drawing is done, to 
            obtain a device context endDraw must be called at the 
            end of the drawing.
            Only a few basic operations are provided, for people who
            don't want to learn the win32 API.
            People who are proficient with GDI programming, may get
            the device context, and use any GDI function with it.
  */
  
  /** Locks the buffer and creates a GDI device context.
      endDraw must be called after all operations are done. */
  virtual long beginDraw() = 0;

  /** Unlocks the buffer and releases the GDI device context. */
  virtual long endDraw() = 0;

  /** Returns the device context.  Return value should be cast
      to HDC */
  virtual void* getDeviceContext() = 0;

  /** Sets a single pixel to the specified color */
  virtual long setPixel(int x, int y, int Red, int Green, int Blue) = 0;
          long setPixel(const Vector2D& p, const int Red, const int Green, const int Blue)
          { return setPixel(p.x, p.y, Red, Green, Blue); }

  /** Sets a single pixel to the specified color */
  virtual long setPixel(int x, int y, long Color) = 0;
          long setPixel(const Vector2D& p, const long Color)
          { return setPixel(p.x, p.y, Color); }

  /** Gets the color value of the pixel */
  virtual long getPixel(int x, int y) const = 0;
          long getPixel(const Vector2D& p) const
          { return getPixel(p.x, p.y); }

  /** Draws an ellipse that is bounded by the specified rectangle */
  virtual long ellipse(int x1, int y1, int x2, int y2) = 0;
          long ellipse(const Rect2D& r)
          { return ellipse(r.tl.x, r.tl.y, r.br.x, r.br.y); }

  /** Draw a generic line */
  virtual long line(int x1, int y1, int x2, int y2) = 0;
          long line(const Vector2D& u, const Vector2D& v)
          { return line(u.x, u.y, v.x, v.y); }

  /** Set the current drawing color */
  virtual long setColor(int Red, int Green, int Blue) = 0;

  /** Set the current background color */
  virtual long setBKColor(int Red, int Green, int Blue) = 0;

  /** Print text */
  virtual long printXY(int x, int y, int Color, const char* str) = 0;
          long printXY(const Vector2D& v, const int Color, const char* str)
          { return printXY(v.x, v.y, Color, str); }

  /** Returns the width of a character.  Assumes default monospace font */
  virtual long getTextWidth() = 0;

  /** Returns the height of a character.  Assumes default monospace font */
  virtual long getTextHeight() = 0;

  /** Draws a filled rectangle */
  virtual long fillRect(int x1, int y1, int x2, int y2) = 0;
          long fillRect(const Rect2D& r)
          { return fillRect(r.tl.x, r.tl.y, r.br.x, r.br.y); }


//////////////////////////////////////////////////////////////////////
  /**
  Category: Import/Export methods
  */

  /** Loads data into the Drawable from a 24 bit raw data stream. 
      TopToBottom indicates if image is y inverted.
      Pad32 indicates that each row is padded to 32 bit bounday. (BMP style)
  */
  virtual long loadRaw(istream& is, int TopToBottom=1, int Pad32=0) = 0;

  /** Loads data into the Drawable from a 24 bit raw data memory buffer.
      TopToBottom indicates if image is y inverted.
      Pad32 indicates that each row is padded to 32 bit bounday. (BMP style)
  */
  virtual long loadRaw(void* Buffer, ulong Length, int TopToBottom=1, int Pad32=0) = 0;

  /** Saves the drawable as a bitmap to a stream.
      If BMP is zero this includes a header consisting of:  
      Width (4 bytes) Height (4 Bytes) Bits Per Pixel (4 Bytes) and raw 24 bit data
      If BMP is not 0, the output is a windows BMP file. */
  virtual long save(ostream& os, int BMP=1) = 0;

//////////////////////////////////////////////////////////////////////
  /**
  Category: Drawable objects can have transparent parts.
            A single color is used to indicate which pixels are
            transparent.  When the Drawable is copied, only its
            non-transparent parts will be copied.
  */

  /** Returns a non-zero value if this drawable has a transparent color (for copy ops) */
  virtual int  getTransparency() const = 0;

  /** Set transparency mode.  Still requiring setting a transparent color. */
  virtual long setTransparency(int Transparency) = 0;

  /** Sets the color which will be transparent in copies.  (Default: 1) */
  virtual long setTransparentColor(TColor Color) = 0;

  /** Gets the color which will be transparent in copies. */
  virtual TColor getTransparentColor() const = 0;


//////////////////////////////////////////////////////////////////////
  /**
  Category: Color conversions.  Methods for converting 24<-->16 bit colors.
  */

  /** Convert 24 bit color components into 16 bit. */
  virtual TColor getColor(int Red, int Green, int Blue) const = 0;
  
  /** Convert 24 bit color into 16 bit. */
  virtual TColor getColor(long Color) const = 0;

  /** Convert 16 bit color into 24 bit. */
  virtual long   getRGBColor(TColor Color) const = 0;
};

#endif // H_CON_DRAWABLE