/** \file 2D_Sprite.h
    A simple sprite, bitmap with an outline.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_2D_SPRITE
#define H_2D_SPRITE

#include <2D_View.h>
#include <2D_Collision.h>
#include <2D_Object2D.h>

/** A Sprite is a 2D entity that basically gives a body to a bitmap.
    This means for example, supplying a collision detection between two sprites.
    The first pixel (0,0) is used to determine the transparent color,
    which is considered the 'outside' of the sprite.
*/
class Sprite : public SpriteSelector
{
public:
  /** Construct a sprite from a bitmap.  If Solid is non zero,
      collision detection is enabled for this sprite.
  */
  Sprite(Bitmap* BM, int Solid=0);

  /** Construct a sprite from a section(Rect2D r) of a bitmap. */
  Sprite(Bitmap* BM, Rect2D& r, int Solid=0);

  /** Destruct the Sprite. */
  virtual ~Sprite();

  /** render the sprite to position pos within a 2D view */
  long render(const Vector2D& pos, const View2D& view);

  /** Test for collision with this sprite.  The offset describes
      the location of the other sprite relative to this one.
  */
  int collision(Sprite& S, int dx, int dy) const;

  /** Set the solid (collision detection state) of the Sprite.
      If the sprite was created with Solid=0, it cannot be switched to
      Solid.  This is useful to make Solid sprites transparent.
  */
  void setSolid(int Solid);

  /** Retrieve the Solid (collision detection) state of the sprite. */
  int  getSolid() const;

  /** Retrieve the sprite's bitmap width. */
  int  getWidth() const;

  /** Retrieve the sprite's bitmap height. */
  int  getHeight() const;

  /** Return the offset into the bitmap, where the sprite is located. */
  Vector2D getPos() const;

  /** Return the rectangle in the bitmap, where the sprite is located. */
  Rect2D getRect() const;

  /** Implementation of the abstract SpriteSelector.  Simply returns this sprite. */
  virtual Sprite* getSprite();

  /** Retrieve the sprite's associated bitmap. */
  Bitmap* getBitmap() const; 

  /** Set a new bitmap for this sprite (and optionally new rectangle)
      Recalcuates collision detection info according to Solid parameter.
      if Solid == -1 (default), collision info is calculated if it was already present.
      if Solid == 0, collsion info is not calculated;
      otherwise, collision info is calculated.
  */
  void    setBitmap(Bitmap* BM, Rect2D r=Rect2D(), int Solid=-1);

  /** Set the sprite as the bitmap owner, which means that it will release
      an old bitmap before setting a new one.   Useful for objects not
      using the bitmap caching. */
  void    setBitmapOwner(int Owner=1);
protected:
  Bitmap         *m_Bitmap;
  Collision2D    *m_Coll;
  Rect2D          m_Rect;
  int             m_Solid;
  int             m_Owner;
};

/** AnimatedSprite is the link between a sprite and an Object2D.  
    It associates single/multiple sprites to a world object,
    and is the basis for actual 'sprite' objects in games.
*/
class AnimatedSprite : public Object2D
{
public:
  /** Construct an AnimatedSprite.  Parameters are identical 
      to the ones constructing an Object2D */
  AnimatedSprite();

  /** Object Move/State Change.
      Defaults to switching between sprites to perform animation.
  */
  virtual long advance(float Fraction);

  /** Stop animating between sprites, and set the object to
      one constant sprite.  Default to the first sprite. */
  void stopAnimation(int ConstantSprite=0);

  /** Start animation between sprite frames.
      This is useful to restart the animation after it was stopped.
  */
  void startAnimation(float Rate=1.0f);

  /** Add another sprite to the animation sequence.
      The frame length is in miliseconds.
  */
  void addAnimationFrame(Sprite* S, float FrameLength); 

protected:
  Vector  m_Sprites;
  int     m_Current;
  float   m_Left;
  float   m_Rate;
};

#endif // H_2D_SPRITE