/** \file 2D_Effects.h
    Mechanizm for creating 2D Effects.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_2D_EFFECTS
#define H_2D_EFFECTS

#include <CON_Bitmap.h>
#include <DS_Vector.h>
#include <2D_Types.h>


/** Generic abstract bitmap filter.  Allows creating 2D special effects on a pixel basis. */
class BitmapFilter
{
public:
  /** Constructs a filter that will write its output to the given bitmap. */
  BitmapFilter(Bitmap* Output);

  /** Base virtual destructor. */
  virtual ~BitmapFilter();

  /** Add an input bitmap, that will be used to calculate the output.
      Uses bitmap from offset 0,0
  */
  long addInput(Bitmap* Input);

  /** Add an input bitmap, that will be used to calculate the output.
      The offset is used to use a rectangle within a large bitmap.
  */
  long addInput(Bitmap* Input, Vector2D Offset);

  /** Activate the filter, and calculate the output bitmap. */
  long activate();

  /** Activate the filter, and calculate the output bitmap.
      Process only the given rectangle.
  */
  long activate(Rect2D Region);

  /** Calculates the output pixel value, by considering all or some inputs.
      This pure virtual function must be overridden to supply filter behaviour.
      Input1 = *InputPixels[0];
      Input2 = *InputPixels[1];
      NumberOfInputs specifies how many inputs are available.
  */
  virtual TColor filterPixel(TColor** InputPixels, int NumberOfInputs) = 0;
  
protected:
  /** Separate a 16 bit color to its components (0..255) each. */
  void    getColorComponents(TColor Color, int& Red, int& Green, int& Blue);

  /** Combine the components into a 16 bit color value. */
  TColor  getColor(int Red, int Green, int Blue);

private:
  /** Calculates a common area to all inputs and output. 
      Example: Using a 32x32 output and 16x16 input will only process 16x16 (overlap) area.
  */
  Vector2D commonRegion(Vector2D RegionSize);

  /** Returns the bitmap of the specified input. */
  Bitmap* getInput(int Position=0);

  Vector  m_Inputs;
  Bitmap* m_Output;

public:
  /** Return values. */
  enum ErrorReturnValues { 
    NO_INPUTS=-3001,
    NO_COMMON_REGION=-3002
  };
};

/** Filter implementation.  Bitwise invert each pixel. */
class InverseFilter : public BitmapFilter
{
public:
  InverseFilter(Bitmap* Output) : BitmapFilter(Output) {}
  virtual TColor filterPixel(TColor** InputPixels, int NumberOfInputs);
};

/** Blending Filter, that expects two inputs and adds them with the given ratio.
    Output = Ratio * Input1  +  (1-Ratio) * Input2
*/
class FadeFilter : public BitmapFilter
{
public:
  FadeFilter(Bitmap* Output, float FadeRatio=0.5f) : BitmapFilter(Output), m_Ratio(FadeRatio) {}
  virtual TColor filterPixel(TColor** InputPixels, int NumberOfInputs);
protected:
  float m_Ratio;
};

class GrayFilter : public BitmapFilter
{
public:
  GrayFilter(Bitmap* Output) : BitmapFilter(Output) {}
  virtual TColor filterPixel(TColor** InputPixels, int NumberOfInputs);
};

#endif // H_2D_EFFECTS