
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;
import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;
import java.lang.reflect.*;
// import com.sun.java.util.*;

/**
  * A block represents the text between two {}'s in a template, or else 
  * the text that begins at the start of the template and runs until its
  * end ({}'s around the whole document are not required). It contains 
  * all of the other directives, strings, etc. that can be in a template. 
  */
final public class Block extends Vector implements Macro
{

   static private final boolean _debug = Engine.debug && true;

   // uniqued strings for identifiers block may find while parsing

   static final private String[] _directives = { 
      "org.webmacro.engine.IncludeDirective",
      "org.webmacro.engine.ParseDirective",
      "org.webmacro.engine.SetDirective",
      "org.webmacro.engine.IfDirective",
      "org.webmacro.engine.UseDirective",
      "org.webmacro.engine.ForeachDirective",
      "org.webmacro.engine.ParamDirective" };
   static final private Hashtable _dirTable = new Hashtable();
   static {
     Class[] parseSig = { ParseTool.class };
     Class[] nullSig = { };
      for(int i = 0; i < _directives.length; i++) {
         String dir = _directives[i];
         try {
            if (! dir.endsWith("Directive") ) {
               Engine.log.error(dir + ": not loaded; " +
                     "class names for directives must end with \"Directive\"");
            }
            int end = dir.length() - 9;
            int start = dir.lastIndexOf('.',end) + 1;
            String dirName = dir.substring(start,end);
            if (dir.startsWith("org.webmacro.engine")) {
               dirName = dirName.toLowerCase();
            }
            Class c = Class.forName(_directives[i]);
            Method parseM = c.getMethod("parse",parseSig);
            _dirTable.put(dirName,parseM);
            Engine.log.info("Loaded directive: " + dirName);
         } catch(Exception e) {
            Engine.log.exception(e);
            Engine.log.error("Could not load directive " + dir + ": " + e);
         }
      }
   }

   Block() { 
      super(10); // space for 7 children per block by default 
   }

   /**
     * Attempt to parse the block, or return null if what follows the 
     * current position is not actually a block. 
     * <p>
     * @exception ParseException if the sytax was invalid and we could not recover
     * @exception IOException if we could not successfullly read the parseTool
     */
   public static Object parse(ParseTool in) 
      throws ParseException, IOException
   {
      int tok;
      boolean parens;

      Block b = new Block();

      // eat open parens
      parens = in.parseChar('{');

      if (_debug && parens) Engine.log.debug("Block requires close parens");


      // parse the block

      boolean lineStart = true; // assume block start means start of line
      StringBuffer str = new StringBuffer(512); // start large to save mallocs
      Object child;


      // beware: yp maintains the following loop is unnecessarily complex 

      boolean inBlock = (in.ttype != in.TT_EOF); // to start
      LOOP: while (inBlock)
      {

         // a block can contain strings, blocks, variables, and directives
         child = null;

         if (_debug) {
           Engine.log.debug("in.ttype: " + (char)in.ttype 
                 +  " in.sval: " + in.sval);
         }
	 // each case in this switch will advance to the first unparseable token
	 //   continue==continue building string, jump to the start of the loop
	 //   break==we may have to handle a complex child (child may != null)
SWITCH:  switch(in.ttype) {

            // THESE CASES JUST KEEP APPENDING TO THE STRING
            // they always advance the token

            case in.TT_WORD:
               if (_debug) {
                  Engine.log.debug("tt_word (NOT linestart");
               }
               str.append(in.sval);
               lineStart = false; // saw a word, not a lineStart
	       in.nextToken();
               continue LOOP; // still building string

            case ' ': case '\t': 
               if (_debug) {
                  Engine.log.debug("space");
               }
               // leave lineStart alone
               str.append((char) in.ttype);
	       in.nextToken();
               continue LOOP; // still building string
        
            case '\n': // parseTool fixes portability problems for us here
               if (_debug) {
                  Engine.log.debug("newline -- NOW linestart");
               }
               lineStart = true;
               str.append(in.lineSeparator);
	       in.nextToken();
               continue LOOP; // still building string

            default:
               // not special here, leave child null and append to string
               lineStart = false;
               str.append((char)in.ttype);
	       in.nextToken();
               continue LOOP;

            case '\\': // escape character: next char is literal
               if (_debug) {
                  Engine.log.debug("escape");
               }
               in.nextToken();
               if (in.ttype == in.TT_WORD) {
                  str.append(in.sval);
               } else if (in.ttype == in.TT_EOF) {
                  throw new ParseException(in, 
                        "File ended after escape character");
               } else {
                  str.append((char) in.ttype);
               }
               in.nextToken();
               continue LOOP; // we've handled all the tokens read


            // THESE CASES CAN BREAK THE LOOP
            // they behave like the above appenders if the loop continues

            case '}': // end of block
               if (lineStart) {
                  if (parens) {
                     if (_debug) {
                        Engine.log.debug("closeBlock");
                     }
                     inBlock = false; // breaks the loop
                  } else {
                     throw new ParseException(in, "} unexpected");
                  }
               } else {
                  if (_debug) {
                        Engine.log.debug("not a close block.. not linestart");
                  }
               }
               break SWITCH;

	    case in.TT_EOF:
               if (_debug) {
                  Engine.log.debug("tt_eof");
               }
	       inBlock = false; // breaks the loop
	       break SWITCH; 


            // THESE CASES HANDLE COMPLEX KINDS OF CHILDREN
            // they only advance the token if they parse a child

            case '$': // variable
               if (_debug) {
                  Engine.log.debug("variable -- not linestart");
               }
               child = Variable.parse(in);
               lineStart = false; // Variable doesn't eat the EOL
               break SWITCH; 
 
            case '#': // directive
               if (_debug) {
                  Engine.log.debug("directive");
               }
               if (lineStart) {
                  in.nextToken();
                  if (in.ttype == in.TT_WORD) {
                     String dirName = in.sval;
                     Method parse = (Method) _dirTable.get(dirName);
                     if (parse == null) {
                        throw new ParseException(in, 
                              "Unrecognized directive: " + dirName);
                     }
                     if (_debug) {
                        Engine.log.debug("directive: " + dirName);
                     }
                     try {
                        child = parse.invoke(null,in.asArg);
                     } catch(InvocationTargetException ie) {
                        Throwable e = ie.getTargetException();   
                        if (e instanceof ParseException) {
                           throw (ParseException) e;
                        } else if (e instanceof IOException) {
                           throw (IOException) e;
                        } else {
                           throw new ParseException(in,"Directive " + dirName +
                                 " threw an unusual exception: " + e);
                        }
                     } catch(Exception e) {
                        throw new ParseException(in, "Directive " + dirName +
                              " has implementation errors: " + e);
                     }
                  }
                  else if (in.ttype == '#') 
                  {
                     if (_debug) {
                        Engine.log.debug("comment");
                     }
                     in.parseToEOL(); // ignore rest of line
                     child = ""; // replace with a null
                  }
                  else 
                  {
                     if (_debug) {
                        Engine.log.debug("not a directive:" + (char) in.ttype
                              + " therefore NOT linestart");
                     }
                     lineStart = false; // Directive doesn't eat the EOL
                     if (_debug && (in.ttype == in.TT_WORD)) {
                        Engine.log.debug("actual value = " + in.sval);
                     }
                     in.pushBack(); // we looked ahead an extra word
                  }
               }
               break SWITCH; 

            case '{': // block
               if (_debug) {
                  Engine.log.debug("openBlock");
               }
               if (lineStart && (null != (child = Block.parse(in))) ) {
                  // it was a block, and we parsed it
                  in.parseSpaces(); // skip any spaces
                  in.parseChar('\n'); // parseTool fixes portability problem
                  // after block is like start of line, so leave alone 
               } else {
                  // it wasn't a block, handle it as a single char
                  if (_debug) {
                     Engine.log.debug("{ not block, now NOT linestart");
                  }
                  lineStart = false; 
               }
               break SWITCH; 



         }
    
         if (inBlock && (child == null)) {
            if (_debug) {
               Engine.log.debug("Appending char");
            }
            // failed to read a child, so append the char we read
            str.append((char) in.ttype);
            in.nextToken(); 
         } else {
            if (_debug) {
               Engine.log.debug("Appending child");
            }
            // successfully read some kind of child, or we're quitting
            // so write the string we've built up 
            b.addElement(str.toString());
            str.setLength(0);
	    if (child != null) {
               b.addElement(child);
	    }
         }
      }

      // check for close parens
      if (parens) {
         if (in.ttype == in.TT_EOF) { 
            throw new ParseException(in, "expected } but got EOF");
         }
         if (! in.parseChar('}')) {
            throw new ParseException(in, "expected }");
         }
      }

      return b;
   }


   /**
     * Evaluate the current macro and return it as a string. Same
     * basic operation as calling write.
     * @exception InvalidContextException if required data was missing from context
     */ 
   public Object evaluate(Object context)
      throws InvalidContextException
   {
      try {
         StringWriter sw = new SizedStringWriter(512);
         write(sw,context);
         return sw.toString();
      } catch (IOException e) {
         Engine.log.exception(e);
         Engine.log.error("evaluate got IO exception on write to StringWriter");
         return "";
      }
   }  


   /**
     * Interpret the block  and write it out
     * <p>
     * @exception InvalidContextException if required data was missing from context
     * @exception IOException if we could not successfully write to out
     */
   public void write(Writer out, Object context) 
      throws InvalidContextException, IOException
   {
      Object o;
      if (_debug) {
         Engine.log.debug("Block.write: beginning to write");
      }
      for (int i = 0; i < elementCount; i++) {
         o = elementData[i];
         if (! (o instanceof Macro)) {
            if (_debug) {
               Engine.log.debug("Block.write: scalar");
            }
            out.write(o.toString());
         } else {
            if (_debug) { 
               Engine.log.debug("Block.write: macro");
            }
            ((Macro) o).write(out,context);
         }
      }
   }

    
   /** 
     * test harness
     */
   public static void main(String args[]) {


      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);
      // Build a Map
      Map context = new HashMap();
      context.put("helloworld", "Hello World");
      context.put("hello", "Hello");
      context.put("file", "include.txt");
      
      // testing foreach directive
      Vector customers = new Vector();
      customers.addElement("justin");
      customers.addElement("yuenping");
      customers.addElement("anjli");
      customers.addElement("dave");
  
      Integer zero = new Integer(0);
      Integer numbers[] = { zero };
     
      Vector emptyList = new Vector();
      Integer notExist = new Integer(10);

      context.put("customers", customers);
      context.put("numbers", numbers);
      context.put("emptyList", emptyList);


      // each of these cases maps a case.wm file to a case.html file
      // where "case" is the string in the array
      String cases[] = { 
            "emptyBlock", 
            "variable", "include", 
	    "list"
         };

      for (int i = 0; i < cases.length; i++) {
         File inFile = new File("examples", cases[i] + ".wm");
         File outFile = new File("examples", cases[i] + ".html");
         System.out.println("MAPPING " + inFile + " TO " + outFile);

         try {
            ParseTool in = new ParseTool(inFile);
            in.nextToken();
            Writer out = new FileWriter(outFile);
            Block b = (Block) Block.parse(in);
            b.write(out,context);
            out.close();
         }  catch (Exception e) {
            e.printStackTrace(); 
         }
         System.out.println();
      }
      System.out.println("Done. View the files to confirm the tests.");
   }
  
}




