/*Data Access Library. v. 1.1.
 *Command line interface to code generator.
 *AUTHOR:  Alexander Jaremenko <jarem@altavista.net>
 *RELEASE DATE: 
 */

package JProjects.eab.data.wizzard;

import java.io.*;
import java.util.*;

/**Command line interface to Data Access Builder.
 *Use: <br>java JProjects.eab.data.wizzard.DataWizzard map-parameters-file
 *<p>Map parameter file have a property file format. Map parameter file propeties:<br><dl>
 *<dt><b>BASE_NAME</b> <dd>= Base name for classes to generate.
 *<dt><b>TABLE_NAMES</b><dd>= List of table names. Each table name may be qualified by 
 *schema name.
 *<dt><b>TABLE_ALIASES</b><dd>= List of tables' aliases. Aliases are used in generated
 *SQL to avoid naming conflicts between columns from different tables.
 *<dt><b>COL_NAMES</b><dd>= List of column names. Entries should be grouped in 
 *the order of corresponding table names in the TABLE_NAMES. Columns that constitute
 *a primary key (rather DataId of the PersistentObject) should be placed at head.
 *<dt><b>DATA_TYPES</b><dd>= List of columns' data types. Entries should correspond
 *to ones from COL_NAMES. Names for data types coincide with those from jdbc 
 *specification. See java.sql.Types.
 *<dt><b>NULLABLES</b><dd>= <code>true</code>/<code>false</code> properties for the
 *columns.
 *<dt><b>FIELD_NAMES</b><dd>= List of persistent object field names. Each such field
 * will contain data from the corresponding column of the database table. Optional
 *property. Defaults to COL_NAMES.
 *<dt><b>DATA_IDS</b><dd>=<code>true</code>/<code>false</code> properties for the
 *columns that comprise a data id. Here <code>true</code> means that that column
 *can be modified by application (in contrast to <code>serial</code> data type
 *for Informix).
 *<dt><b>IS_READ_ONLY</b><dd>=<code>true</code>/<code>false</code> property for
 *generated classes. Should be <code>true</code> if it is mapping for SQL join.
 *<dt><b>SQLLINK</b><dd>= String that constitute SQL join clause. It goes into
 *generated SQL "AS IS".
 *</dl></p>
 *@author Alexander Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
 */
public class DataWizzard {
    private String[] tN;
    private String[] taN;
    private String bN;
    private String[] cN;
    private String[] fN;
    private String[] dT;
    private boolean[] nullable;
    private boolean[] dI;
    private String sqlL;
    private String outDir;
    private boolean isReadOnly;

    private static final int TABLE_NAMES =1;
    private static final int TABLE_ALIASES = 2;
    private static final int COL_NAMES = 3;
    private static final int FIELD_NAMES = 4;
    private static final int NULLABLES = 5;
    private static final int DATA_IDS = 6;
    private static final int DATA_TYPES = 7;

    public DataWizzard() {}

    public void run(String cfgFN) {
	try {
	    FileInputStream f = new FileInputStream(cfgFN);
	    parseCfgFile(f);
	    DCGenerator g = new DCGenerator(tN,taN,bN,cN,fN,dT,nullable,dI,sqlL);
	    g.setReadOnly(isReadOnly);
	    g.generate(outDir);
	} catch (IOException ex) {
	    System.out.println("I/O error:");
	    ex.printStackTrace(System.out);
	}
    }

    public static void main(String[] args) {
	if (args.length<1) {
	    System.out.println("USAGE: java JProjects.eab.data.wizzard.DataWizzard paramFileName");
	    return;
	}
	DataWizzard w = new DataWizzard();
	w.run(args[0]);
    }

    protected void parseCfgFile(InputStream f) throws IOException {
	Properties prop = new Properties();
	prop.load(f);
	String vals;
	outDir = prop.getProperty("TARGET_DIR","."+System.getProperty("file.separator"));
	vals = prop.getProperty("IS_READ_ONLY","false");
	if (vals.startsWith("T") || vals.startsWith("t") || vals.trim().equals("1"))
	    isReadOnly = true;
	else
	    isReadOnly = false;
	vals = prop.getProperty("TABLE_NAMES");
	if (vals==null)
	    throw new IOException("Param file format: Table names missed");
	fillStringArray(TABLE_NAMES,vals);
	vals = prop.getProperty("TABLE_ALIASES");
	if (vals == null)
	    taN = null;
	else {
	    fillStringArray(TABLE_ALIASES,vals);
	    if (taN.length!=tN.length)
		throw new IOException("Param file format: Table names array and table aliases array has different length.");
	}
	vals = prop.getProperty("COL_NAMES");
	if (vals==null)
	    throw new IOException("Param file format: Col names missed");
	fillStringArray(COL_NAMES,vals);
	vals = prop.getProperty("FIELD_NAMES");
	if (vals == null)
	    fN = cN;
	else {
	    fillStringArray(FIELD_NAMES,vals);
	    if (fN.length!=cN.length)
		throw new IOException("Param file format: Col names array and field names array has different length.");
	}
	vals = prop.getProperty("NULLABLES");
	if (vals==null)
	    throw new IOException("Param file format: Nullables missed");
	fillBooleanArray(NULLABLES,vals);
	if ( nullable.length != fN.length )
	    throw new IOException("Param file format: Col names array and nullables array has different length.");
	vals = prop.getProperty("DATA_IDS");
	if (vals!=null) {
	    fillBooleanArray(DATA_IDS,vals);
	    if (dI.length > fN.length)
		throw new IOException("Param file format: data id array is greater than field names array.");
	}
	vals = prop.getProperty("DATA_TYPES");
	if (vals==null)
	    throw new IOException("Param file format: Data types missed");
	fillStringArray(DATA_TYPES,vals);
	if (dT.length != fN.length)
	    throw new IOException("Param file format: Col names array and data types array has different length.");

	bN = prop.getProperty("BASE_NAME");
	if (bN==null)
	    throw new IOException("Param file format: Base name missed");
	sqlL = prop.getProperty("SQLLINK","");
    }

    private void fillBooleanArray(int arrId,String vals) {
	StringTokenizer tok = new StringTokenizer(vals);
	boolean[] tgt;
	if (arrId == NULLABLES) {
	    nullable = new boolean[tok.countTokens()];
	    tgt = nullable;
	} else {
	    dI  = new boolean[tok.countTokens()];
	    tgt = dI;
	}
	int i=0;
	while(tok.hasMoreTokens()) {
	    String v = tok.nextToken();
	    if (v.startsWith("T") || v.startsWith("t") || v.trim().equals("1"))
		tgt[i++] = true;
	    else
		tgt[i++] = false;
	}
    }

    private void fillStringArray(int arrId,String vals) {
	StringTokenizer tok = new StringTokenizer(vals);
	String[] tgt=null;
	if (arrId == TABLE_NAMES) {
	    tN = new String[tok.countTokens()];
	    tgt = tN;
	} else if (arrId == TABLE_ALIASES) {
	    taN = new String[tok.countTokens()];
	    tgt = taN;
	} else if (arrId == COL_NAMES) {
	    cN = new String[tok.countTokens()];
	    tgt = cN;
	} else if (arrId == FIELD_NAMES) {
	    fN = new String[tok.countTokens()];
	    tgt = fN;
	} else {
	    dT = new String[tok.countTokens()];
	    tgt = dT;
	}
	int i=0;
	while(tok.hasMoreTokens())
	    tgt[i++] = tok.nextToken();
    }

}
