
package JProjects.eab.data.wizzard;

import java.io.*;
import java.sql.Types;

/**Data Access Builder's code generator.
 *@author Alexandr Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
 *@version Data Access Library. v. 1.1
 */
public class DCGenerator {
    private ClassDescriptor cl;
    private TableDescriptor tbl;
    private ColumnDescriptor[] cols;
    private int diIndex;

    private final static String fs = System.getProperty("file.separator");

    /**This constructor used by the GUI front-end JProjects.wizzard.Database.
     *@param tbls array of table names
     *@param tblAlias array of table aliases
     *@param baseName base name for generated classes
     *@param colNs array of column names
     *@param flds array of field names
     *@param dataTypes 2-dim. array. First element of an row holds JDBC
     *type code and the second one scale value.
     *@param nullable array of nullables properties
     *@param isModDI array of data id modifiable properties
     *@param sqlLink SQLLINK property
     */
    public DCGenerator(String[] tbls,String[] tblAlias,String baseName,String[] colNs,String[] flds,int[][] dataTypes,boolean[] nullable,boolean[] isModDI,String sqlLink) {
	tbl = new TableDescriptor(tbls,tblAlias,sqlLink);
	cl = new ClassDescriptor(baseName,false,(isModDI!=null));
	if (isModDI != null) 
	    diIndex = isModDI.length - 1;
	else
	    diIndex = -1;
	cols = new ColumnDescriptor[flds.length];
	for (int i=0;i< flds.length;i++) {
	    if (i<= diIndex)
		cols[i] = new ColumnDescriptor(colNs[i],flds[i],dataTypes[i],nullable[i],true,isModDI[i]);
	    else
		cols[i] = new ColumnDescriptor(colNs[i],flds[i],dataTypes[i],nullable[i],false,false);
	}

    }

    /**This constructor used by the command line front-end JProjects.eab.data.wizzard.DataWizzard
     *@param tbls array of table names
     *@param tblAlias array of table aliases
     *@param baseName base name for generated classes
     *@param colNs array of column names
     *@param flds array of field names
     *@param dataTypes array of JDBC data types,
     *@param nullable array of nullables properties
     *@param isModDI array of data id modifiable properties
     *@param sqlLink SQLLINK property
     *@see DataWizzard
     */
    public DCGenerator(String[] tbls,String[] tblAlias,String baseName,String[] colNs,String[] flds,String[] dataTypes,boolean[] nullable,boolean[] isModDI,String sqlLink) {
	tbl = new TableDescriptor(tbls,tblAlias,sqlLink);
	cl = new ClassDescriptor(baseName,false,(isModDI!=null));
	if (isModDI != null) 
	    diIndex = isModDI.length - 1;
	else
	    diIndex = -1;
	cols = new ColumnDescriptor[flds.length];
	for (int i=0;i< flds.length;i++) {
	    if (i<= diIndex)
		cols[i] = new ColumnDescriptor(colNs[i],flds[i],dataTypes[i],nullable[i],true,isModDI[i]);
	    else
		cols[i] = new ColumnDescriptor(colNs[i],flds[i],dataTypes[i],nullable[i],false,false);
	}
    }


    /**Sets IS_READ_ONLY property for generated classes.*/
    public void setReadOnly(boolean flag) {
	cl.setReadOnly(flag);
    }

    /**Launches code generation.
     *@param dir target directory
     *@exception java.io.IOException if something is wrong with I/O
     */
    public void generate(String dir) throws IOException {
	String dirName =(dir.endsWith(fs)?dir.substring(0,dir.length()-1):dir);
	generatePO(dirName);
	generateManager(dirName);
	generateMap(dirName);
	if (diIndex>=0) {
	    generatePODataId(dirName);
	    generateDataIdManager(dirName);
	    generateDataIdMap(dirName);
	}
    }

    private void generatePO(String dir) throws IOException {
	Formatter outStr = new Formatter();
	outStr.open(dir+fs+cl.getPOName()+".java");
	genHeader(outStr,cl.getPkgName(),cl.getPOName(),"PersistentObject",null);
	genPOFields(outStr,cl,tbl,cols);
	genPOMethods(outStr,cl,tbl,cols);
	outStr.writeln("\n}",0);
	outStr.flush();
    }

    private void generatePODataId(String dir) throws IOException {
	Formatter outStr = new Formatter();
	outStr.open(dir+fs+cl.getDataIdName()+".java");
	genHeader(outStr,cl.getPkgName(),cl.getDataIdName(),"PODataId",null);
	ColumnDescriptor[] diCols = new ColumnDescriptor[diIndex+1];
	System.arraycopy(cols,0,diCols,0,diIndex+1);
	genPODataIdFields(outStr,cl,diCols);
	genPODataIdMethods(outStr,cl,diCols);
	outStr.writeln("\n}",0);
	outStr.flush();
	//	outStr.close();
    }

    private void  generateManager(String dirName) throws IOException {
	Formatter outStr = new Formatter();
	outStr.open(dirName+fs+cl.getPOManagerName()+".java");
	genHeader(outStr,cl.getPkgName(),cl.getPOManagerName(),"DAManager",null);
	genManagerMethods(outStr,cl,tbl,cols,false);
	outStr.writeln("\n}",0);
	outStr.flush();
	//	outStr.close();
    }


    private void  generateDataIdManager(String dirName) throws IOException {
	Formatter outStr = new Formatter();
	outStr.open(dirName+fs+cl.getDataIdManagerName()+".java");
	ColumnDescriptor[] diCols = new ColumnDescriptor[diIndex+1];
	System.arraycopy(cols,0,diCols,0,diIndex+1);
	genHeader(outStr,cl.getPkgName(),cl.getDataIdManagerName(),"DAManager",null);
	genManagerMethods(outStr,cl,tbl,cols,true);
	outStr.writeln("\n}",0);
	outStr.flush();
    }

    private void generateMap(String dirName) throws IOException {
	Formatter outStr = new Formatter();
	outStr.open(dirName+fs+cl.getPOMapName()+".java");
	genHeader(outStr,cl.getPkgName(),cl.getPOMapName(),null,new String[] {"DAMap"});
	genPutResultSetInto(outStr,cl.getPOName(),cols);
	outStr.writeln("\n}");
	outStr.flush();
    }


    private void generateDataIdMap(String dirName) throws IOException {
	Formatter outStr = new Formatter();
	outStr.open(dirName+fs+cl.getDataIdMapName()+".java");
	genHeader(outStr,cl.getPkgName(),cl.getDataIdMapName(),null,new String[] {"DAMap"});
	ColumnDescriptor[] diCols = new ColumnDescriptor[diIndex+1];
	System.arraycopy(cols,0,diCols,0,diIndex+1);
	genPutResultSetInto(outStr,cl.getDataIdName(),diCols);
	outStr.writeln("\n}");
	outStr.flush();
    }

    protected void genPOFields(Formatter outStr,ClassDescriptor cld,TableDescriptor tbl,ColumnDescriptor[] cols) throws IOException {
	String clName = cld.getPOName();
	String qual ="";
	if (tbl.getNumberTbls() <2)
	    qual = tbl.getQualifier(0);
	outStr.writeln("private static DatastoreJDBC "+cld.getPODatastoreName()+";\n",1);
	if (diIndex>=0) {
	    outStr.writeln("private "+cld.getDataIdName()+" theDataId;");
	    //	    outStr.writeln("private transient PODataIdPropertyChangeAdapter iDataIdAdapter;");
	}
	outStr.writeln("private static String _qualifier ="+"\""+qual+"\";");
	outStr.writeln("/* "+clName+" Attributes and methods\n*/\n");
	for(int i=diIndex+1;i<cols.length;i++) {
	    outStr.writeln("// This attribute is mapped to "+cols[i].getDBFieldName()+".");
	    outStr.writeln("private "+cols[i].getAttrDeclaration()+"\n");
	}
	if (tbl.getNumberTbls()==1) {
	//DASQLGenerator's constructor
	    outStr.write("static DASQLGenerator generator = new DASQLGenerator(");
	    outStr.writeln(null,2);
	    outStr.write("\""+(tbl.getTableNames())[0]+"\",");
	    if (diIndex>=0) {
		outStr.write("new String[] {\""+cols[0].getDBFieldName()+"\"");
		for(int i=1;i<=diIndex;i++)
		    outStr.write(", \""+cols[i].getDBFieldName()+"\"");
		outStr.write("},");
		if (diIndex<cols.length-1) {
		    outStr.writeln("new String[] {\""+cols[diIndex+1].getDBFieldName()+"\"");
		    for (int i=diIndex+2;i<cols.length;i++)
			outStr.write(", \""+cols[i].getDBFieldName()+"\"");
		    outStr.write("},");
		} else
		    outStr.write("null,");
		if (hasModifDataId(cols)) {
		    boolean firstCol=true;
		    for(int i=0;i<=diIndex;i++) {
			if (cols[i].isModifiable()) {
			    if (firstCol) {
				firstCol=false;
				outStr.write("new String[] {\""+cols[i].getDBFieldName()+"\"");
			    } else 
				outStr.write(", \""+cols[i].getDBFieldName()+"\"");
			}
		    }
		    outStr.write("});");
		}
		else
		    outStr.write("null);");
	    } else {
		outStr.write("null,");
		outStr.write("new String[] {\""+cols[0].getDBFieldName()+"\"");
		for (int i=1;i<cols.length;i++)
		    outStr.write(", \""+cols[i].getDBFieldName()+"\"");
		outStr.write("},null);");
	    }
	}
	outStr.writeln("\n",1);
	if (!diOnly()) {
	    outStr.write("\tboolean modified[] = { false");
	    for (int i=diIndex+2;i<cols.length;i++)
		outStr.write(", false");
	    outStr.write("};");
	}
	outStr.writeln("\n\n");

    }

    protected void genPOMethods(Formatter outStr,ClassDescriptor cld,TableDescriptor tbl,ColumnDescriptor[] cols) throws IOException {

	String idName = cld.getDataIdName();

	// Constructors
	genPOConstructors(outStr,cld);

	// _executeAction
	genPOExecuteAction(outStr,cld);

	// small methods
	outStr.writeln("public static String getQualifier() { return _qualifier; }\n",1);
	outStr.writeln("protected POCompleteSupport _getPOCompleteSupport() { return _poSupport; }\n");
	
	outStr.writeln("protected static DatastoreJDBC _getStaticDatastore() {");
	outStr.writeln("\tif ("+cld.getPODatastoreName()+" != null) return "+cld.getPODatastoreName()+";");
	outStr.writeln("\telse return DatastoreJDBC.getApplicationDatastore();\n}\n");

	// protected setters
	for (int i=0;i<cols.length;i++)
	    outStr.writeln(cols[i].getPSetterDef()+"\n");

	// must be methods
	genClone(outStr,cl,cols,false);
	genEquals(outStr,cl,cols,false);
	genGetAttributes(outStr,cols);
	genGetAttributeStrings(outStr,cols);
	genToString(outStr,cols);

	// getCurrentDatastore
	outStr.writeln("public DatastoreJDBC getCurrentDatastore() {",1);
	outStr.writeln("\tif (getObjectsDatastore() != null) return getObjectsDatastore();");
	outStr.writeln("\telse if (getDefaultDatastore() != null) return getDefaultDatastore();");
	outStr.writeln("\telse return DatastoreJDBC.getApplicationDatastore();\n}\n");
	// getDefaultDatastore()
	outStr.writeln(commBlock("Default Datastore."));
	outStr.writeln("public static DatastoreJDBC getDefaultDatastore() { return "+cld.getPODatastoreName()+"; }\n",1);

	//getters
	for (int i=0;i<cols.length;i++)
	    outStr.writeln(cols[i].getGetterDef()+"\n");
	
	//public isNullable methods
	for (int i=0;i<cols.length;i++)
	    outStr.writeln(cols[i].getNullableDef()+"\n");

	//public setters
	if (diIndex >=0)
	    for (int i=0;i<cols.length;i++)
		outStr.writeln(cols[i].getSetterDef(i-diIndex-1)+"\n");
	else
	    for (int i=0;i<cols.length;i++)
		outStr.writeln(cols[i].getSetterDef(i)+"\n");

	if (diIndex >=0) {
	    // getDataId()
	    outStr.writeln(commBlock(idName+"."));
	    outStr.writeln("public "+idName+" getDataId() { return theDataId; }\n",1);
	    // getObjectDataId()
	    outStr.writeln("public PODataId getObjectDataId() { return theDataId; }\n");
	    //hasDataId()
	    outStr.writeln("public boolean hasDataId() { return true; }\n");
	    // setObjectDataId()
	    outStr.writeln("public void setObjectDataId(PODataId aDataId) throws DAException {");
	    outStr.writeln("\tsetDataId(("+idName+")aDataId);\n}\n");
	    // setDataId()
	    genSetDataId(outStr,cld);
	} else 
	    outStr.writeln("public boolean hasDataId() { return false; }\n",1);
	//isReadOnly()
	outStr.writeln("public boolean isReadOnly() { return "+((cld.isReadOnly())?"true":"false")+";}\n");
	// retrieve()
	if (diIndex>=0)
	    genRetrieve(outStr,cld,tbl,cols);
	if (!cld.isReadOnly() && diIndex >=0) {
	    genAdd(outStr,cld,tbl,cols);
	    genDelete(outStr,cld,tbl,cols);
	    if (diIndex < cols.length -1)
		genUpdate(outStr,cld,tbl,cols);
	    genUpdateId(outStr,cld,tbl,cols);
	}
	// isModified - family
	genSetAllModified(outStr);
	genSetModified(outStr,false);
	genIsModified(outStr);

    }

    protected void genPOConstructors(Formatter outStr,ClassDescriptor cld) throws IOException {
	String clName = cld.getPOName();
	String idName = cld.getDataIdName();
	outStr.writeln(commBlock("Construcors."));
	if (diIndex>=0) {
	outStr.writeln("public "+clName+"() {",1);
	outStr.writeln("\tthis(null,null);\n}\n");
	outStr.writeln("public "+clName+"(DatastoreJDBC aDatastore) {");
	outStr.writeln("\tthis(null,aDatastore);\n}\n");
	outStr.writeln("public "+clName+"("+idName+" aDataId) {");
	outStr.writeln("\tthis(aDataId,null);\n}\n");
	outStr.writeln("public "+clName+"("+idName+" aDataId, DatastoreJDBC aDatastore) {");
	outStr.writeln("\tsuper(aDatastore);\n\tif ( aDataId == null ) theDataId = new "+idName+"();");
	outStr.writeln("\telse theDataId = aDataId;\n}");
	//	outStr.writeln("iDataIdAdapter = new PODataIdPropertyChangeAdapter(theDataId,this);\n}");
	} else {
	outStr.writeln("public "+clName+"() {",1);
	outStr.writeln("\tthis(null);\n}\n");
	outStr.writeln("public "+clName+"(DatastoreJDBC aDatastore) {");
	outStr.writeln("\tsuper(aDatastore);\n}");
	}
	outStr.writeln("",0);
    }

    protected void genPOExecuteAction(Formatter outStr,ClassDescriptor cld) throws IOException {
	String idName = cld.getDataIdName();
	outStr.writeln("public void _executeAction( String _methodName, Object[] _params ) throws Exception {",1);
	if (diIndex>=0) {
	    outStr.writeln("if ( _methodName.equals(\"retrieve()\") ) {",2);
	    outStr.writeln("\tretrieve();\n}");
	}
	if (!cld.isReadOnly()) {
	    if (diIndex>=0) {
		outStr.writeln("else if ( _methodName.equals(\"add()\") ) {\n\tadd();\n}");
		outStr.writeln("else if ( _methodName.equals(\"delete()\") ) {\n\tdelete();\n}");
		if ( hasModifDataId(cols) )
		    outStr.writeln("else if ( _methodName.equals(\"update(dataId)\") ) {\n\tupdate( ("+idName+")_params[0] );\n}");
		if (!diOnly())
		    outStr.writeln("else if ( _methodName.equals(\"update()\") ) {\n\tupdate();\n}");
	    } 
	    else 
		outStr.writeln("if ( _methodName.equals(\"add()\") ) {\n\tadd();\n}",2);
	}
	outStr.writeln("}",1);
	outStr.writeln("",0);
    }

    protected void genClone(Formatter outStr,ClassDescriptor cl,ColumnDescriptor[] cols,boolean isDI) throws IOException {
	outStr.writeln(commBlock("clone - returns a copy of the object."));
	outStr.writeln("public Object clone() {",1);
	String clName;
	if (isDI) {
	    clName = cl.getDataIdName();
	    outStr.writeln("\t"+clName+" obj = new "+clName+"();\n");
	    outStr.writeln("\tif (getObjectsDatastore()!=null) ");
	    outStr.writeln("\t\tobj.setObjectsDatastore(getObjectsDatastore());");
	    for (int i=0;i<cols.length;i++)
		outStr.writeln("\tobj."+cols[i].getAttrName()+" = "+cols[i].getAttrName()+";");
	} else {
	    clName = cl.getPOName();
	    String idName = cl.getDataIdName();
	    if (cl.hasDataId())
		outStr.writeln("\t"+clName+" obj = new "+clName+"(("+idName+")theDataId.clone(),getObjectsDatastore());");
	    else
		outStr.writeln("\t"+clName+" obj = new "+clName+"(getObjectsDatastore());");
	    for (int i=diIndex+1;i<cols.length;i++)
		outStr.writeln("\tobj."+cols[i].getAttrName()+" = "+cols[i].getAttrName()+";");
	    outStr.writeln("\tobj.setAsynchronous(isAsynchronous());");
	}
	outStr.writeln("\tobj.setModified(isModified());");
	if (!diOnly())
	    outStr.writeln("\tSystem.arraycopy(modified,0,obj.modified,0,modified.length);");
	outStr.writeln("\treturn obj;\n}");
	outStr.writeln("",0);
    }

    protected void genEquals(Formatter outStr,ClassDescriptor cl,ColumnDescriptor[] cols,boolean isDI) throws IOException {
	String clName = ((isDI)?cl.getDataIdName():cl.getPOName());
	String oName = "a"+clName;
	outStr.writeln(commBlock("equals - compares itself with another object."),0);
	outStr.writeln("public boolean equals(Object anObject) {",1);
	outStr.writeln("\tif ( !(anObject instanceof "+clName+") ) return false;");
	outStr.writeln(clName+" "+oName+" = ("+clName+")anObject;");
	if (diIndex >=0 && !isDI)
	    outStr.writeln("\tif ( !theDataId.equals("+oName+".theDataId) ) return false;");
	int i = ((isDI)?0:diIndex+1);
	for (;i<cols.length;i++)
	    outStr.writeln(cols[i].genEquals(oName,"\t"));
	outStr.writeln("\treturn true;\n}");
	outStr.writeln("",0);
    }

    protected void genGetAttributes(Formatter outStr,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln("public Object[] getAttributes() {",1);
	outStr.writeln("\tObject[] o = { "+cols[0].getAsCtr(null));
	for (int i=1;i<cols.length;i++)
	    outStr.writeln("\t,"+cols[i].getAsCtr(null));
	outStr.writeln("\t};\n\treturn (o);\n}");
	outStr.writeln("",0);
    }

    protected void genGetAttributeStrings(Formatter outStr,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln("public String[] getAttributeStrings() {",1);
	outStr.writeln("\tString[] s = { "+cols[0].getAttributeStrings());
	for (int i=1;i<cols.length;i++)
	    outStr.writeln("\t,"+cols[i].getAttributeStrings());
	outStr.writeln("\t};\n\treturn (s);\n}");
	outStr.writeln("",0);
    }

    protected void genToString(Formatter outStr,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln(commBlock("Display methods"));
	outStr.writeln("public String toString(String aSeparator) {",1);
	outStr.writeln("\treturn ( "+cols[0].getToString());
	for (int i=1;i<cols.length;i++)
	    outStr.writeln("\t+aSeparator + "+cols[i].getToString());
	outStr.writeln("\t);\n}");
	outStr.writeln("",0);
    }

    protected void genSetDataId(Formatter outStr,ClassDescriptor cld) throws IOException {
	String clName = cld.getPOName();
	String idName = cld.getDataIdName();
	outStr.writeln("public void setDataId("+idName+" aDataId) throws DAException {",1);
	outStr.writeln("\tif (aDataId == null) throw new DAException(DAResource.CANT_SET_DATAID_NULL);");
	outStr.writeln("\ttheDataId = aDataId;");
	//	outStr.writeln("\ttheDataId = aDataId;\n\tiDataIdAdapter.setSource(theDataId);");
	outStr.writeln("\tsetModified(true);");
	outStr.writeln("\tfor(int i=0; i < theDataId.modified.length; i++) theDataId.modified[i] = true;\n}");
	outStr.writeln("",0);
    }

    protected void genRetrieve(Formatter outStr,ClassDescriptor cld,TableDescriptor tbl,ColumnDescriptor[] cols) throws IOException {
	String dmName = cld.getPOManagerName();
	String mapName = cld.getPOMapName();

	outStr.writeln(commBlock("Retrieve a row from the database."),0);
	outStr.writeln("public void retrieve() throws DAException {",1);
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {   };");
	outStr.writeln("\t\t_putOnBackgroundThread( \"retrieve()\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("try {",2);
	outStr.writeln("\tif (getCurrentDatastore() == null)\n\t\t throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\t_stmt = getCurrentDatastore().getConnection().prepareStatement(");
	outStr.writeln("\t\t"+dmName+".getSQLBaseQuery()");
	if (diIndex >=0) {
	    if (tbl.getNumberTbls()>1)
		outStr.write("\t\t+\" and "+cols[0].getPrefixedName()+" = ?\"");
	    else
		outStr.write("\t\t+\" where "+cols[0].getPrefixedName()+" = ?\"");
	    for (int i=1;i<cols.length && cols[i].isDataId();i++)
		outStr.write("\t+\" and "+cols[i].getDBFieldName()+" = ?\"");
	    outStr.write("\t);");
	    outStr.writeln("\t// put parameters");
	    outStr.writeln("\tint index=1;");
	    for (int i=0;i<cols.length && cols[i].isDataId();i++)
		outStr.writeln(cols[i].genSetParamRs("_stmt","index",null,"\t"));
	    
	} else
	    outStr.writeln("\t);");
	outStr.writeln("\tResultSet rs = _stmt.executeQuery();");
	outStr.writeln("\tif (!rs.next()) throw new DAException(DAResource.METHOD_NO_RESULTS);");
	outStr.writeln("\t// get output from result set");
	outStr.writeln("\t"+mapName+" map = new "+mapName+"();\n");
	outStr.writeln("\tmap.putResultSetInto(this,rs,false);\n");
	outStr.writeln("\t_stmt.close(); _stmt = null;");
	outStr.writeln("\t} catch(SQLException exc) {",1);
	outStr.writeln("\t\tthrow new DAException (DAResource.ERROR_IN_METHOD,\"retrieve()\", exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\tsetModified(false);");
	outStr.writeln("\tif (_poSupport!=null)");
	outStr.writeln("\t\t_poSupport.fireRetrieveComplete(\"retrieve()\", null);\n}");
	outStr.writeln("",0);
    }

    protected void genAdd(Formatter outStr,ClassDescriptor cld,TableDescriptor tbl,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln(commBlock("Add a row to the database."),0);
	outStr.writeln("public void add() throws DAException {",1);
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {   };");
	outStr.writeln("\t\t_putOnBackgroundThread( \"add()\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("try {",2);
	outStr.writeln("\tif (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\t_stmt = getCurrentDatastore().getConnection().prepareStatement(");
	outStr.writeln("\tgenerator.genInsert(_qualifier));");
	outStr.writeln("\t//put parameters into statement");
	outStr.writeln("\tint index=1;");
	for (int i=0;i<cols.length;i++)
	    if (!cols[i].isDataId() || cols[i].isModifiable())
		outStr.writeln(cols[i].genSetParamRs("_stmt","index",null,"\t"));
	outStr.writeln("\tint rowsAffected = _stmt.executeUpdate();");
	outStr.writeln("\tif (rowsAffected == 0) throw new DAException(DAResource.METHOD_AFFCTD_NOROWS);");
	outStr.writeln("\tif (rowsAffected > 1) throw new DAException(DAResource.METHOD_AFFCTD_MULTROWS);");
	outStr.writeln("\t_stmt.close(); _stmt = null;");
	outStr.writeln("\t} catch(SQLException exc) {",1);
	outStr.writeln("\t\tthrow new DAException (DAResource.ERROR_IN_METHOD,\"add() \", exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\tsetModified(false);");
	outStr.writeln("\tif (_poSupport!=null)");
	outStr.writeln("\t\t_poSupport.fireUpdateComplete(\"add()\", null);\n}");
	outStr.writeln("",0);
    }

  
    protected void genDelete(Formatter outStr,ClassDescriptor cld,TableDescriptor tbl,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln(commBlock("Delete a row from the database."),0);
	outStr.writeln("public void delete() throws DAException {",1);
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {   };");
	outStr.writeln("\t\t_putOnBackgroundThread( \"delete()\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("try {",2);
	outStr.writeln("\tif (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\t_stmt = getCurrentDatastore().getConnection().prepareStatement(");
	outStr.writeln("\tgenerator.genDelete(_qualifier));");
	outStr.writeln("\t//put parameters into statement");
	outStr.writeln("\tint index=1;");
	for (int i=0;i<cols.length && cols[i].isDataId();i++)
	    outStr.writeln(cols[i].genSetParamRs("_stmt","index",null,"\t"));
	outStr.writeln("\tint rowsAffected = _stmt.executeUpdate();");
	outStr.writeln("\tif (rowsAffected == 0) throw new DAException(DAResource.METHOD_AFFCTD_NOROWS);");
	outStr.writeln("\tif (rowsAffected > 1) throw new DAException(DAResource.METHOD_AFFCTD_MULTROWS);");
	outStr.writeln("\t_stmt.close(); _stmt = null;");
	outStr.writeln("\t} catch(SQLException exc) {",1);
	outStr.writeln("\t\tthrow new DAException (DAResource.ERROR_IN_METHOD,\"delete()\", exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\tsetModified(false);");
	outStr.writeln("\tif (_poSupport!=null)");
	outStr.writeln("\t\t_poSupport.fireUpdateComplete(\"delete()\", null);\n}");
	outStr.writeln("",0);
    }

    protected void genUpdate(Formatter outStr,ClassDescriptor cld,TableDescriptor tbl,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln(commBlock("Update a row in the database. PK remains the same."),0);
	outStr.writeln("public void update() throws DAException {",1);
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {   };");
	outStr.writeln("\t\t_putOnBackgroundThread( \"update()\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("try {",2);
	outStr.writeln("\tif (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\tif (!isModified()) throw new DAException(DAResource.NO_CHANGES_MADE);");
	outStr.writeln("\tif (theDataId.isModified()) setAllModified();");
	outStr.writeln("\t_stmt = getCurrentDatastore().getConnection().prepareStatement(");
	outStr.writeln("\t\tgenerator.genUpdate(getQualifier(), modified)\n\t);");
	outStr.writeln("\tint index=1;");
	for (int i=((diIndex>=0)?diIndex+1:0),j=0;i< cols.length;i++,j++) {
	    outStr.writeln("\tif (modified["+j+"]) {");
	    outStr.writeln(cols[i].genSetParamRs("_stmt","index",null,"\t\t")+"\n\t}");
	}
	for (int i=0; i<=diIndex; i++)
	    outStr.writeln(cols[i].genSetParamRs("_stmt","index",null,"\t"));
	outStr.writeln("\tint rowsAffected = _stmt.executeUpdate();");
	outStr.writeln("\tif (rowsAffected == 0) throw new DAException(DAResource.METHOD_AFFCTD_NOROWS);");
	outStr.writeln("\tif (rowsAffected > 1) throw new DAException(DAResource.METHOD_AFFCTD_MULTROWS);");
	outStr.writeln("\t_stmt.close(); _stmt = null;");
	outStr.writeln("\t} catch(SQLException exc) {",1);
	outStr.writeln("\t\tthrow new DAException (DAResource.ERROR_IN_METHOD,\"update()\", exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\tsetModified(false);");
	outStr.writeln("\tif (_poSupport!=null)");
	outStr.writeln("\t\t_poSupport.fireUpdateComplete(\"update()\", null);\n}");
	outStr.writeln("",0);
    }

    protected void genUpdateId(Formatter outStr,ClassDescriptor cld,TableDescriptor tbl,ColumnDescriptor[] cols) throws IOException {
	String idName=cld.getDataIdName();
	if (diIndex < 0)
	    return;
	for(int i=0;i<=diIndex && !cols[i].isModifiable();i++)
	    if ( i == diIndex )
		return;
	outStr.writeln(commBlock("Update a row in the database. PK changes."),0);
	outStr.writeln("public void update("+idName+" aDataId) throws DAException {",1);
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {aDataId};");
	outStr.writeln("\t\t_putOnBackgroundThread( \"update(dataId)\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("try {",2);
	outStr.writeln("\tif (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\tif (aDataId == null) throw new DAException(DAResource.CANT_SET_DATAID_NULL);");
	outStr.writeln("\tif (theDataId.isModified()) setAllModified();");
	outStr.writeln("\t_stmt = getCurrentDatastore().getConnection().prepareStatement(");
	if (diOnly())
	    outStr.writeln("\t\tgenerator.genUpdateDataId(getQualifier(), null)\n\t\t);");
	else
	    outStr.writeln("\t\tgenerator.genUpdateDataId(getQualifier(), modified)\n\t\t);");
	outStr.writeln("\tint index=1;");
	for (int i=0;i<=diIndex;i++)
	    if (cols[i].isModifiable())
		outStr.writeln(cols[i].genSetParamRs("_stmt","index","aDataId","\t"));
	for (int i=diIndex+1,j=0;i< cols.length;i++,j++) {
	    outStr.writeln("\tif (modified["+j+"]) {");
	    outStr.writeln(cols[i].genSetParamRs("_stmt","index",null,"\t\t")+"\n\t}");
	}
	for (int i=0; i<=diIndex; i++)
	    outStr.writeln(cols[i].genSetParamRs("_stmt","index",null,"\t"));
	outStr.writeln("\tint rowsAffected = _stmt.executeUpdate();");
	outStr.writeln("\tif (rowsAffected == 0) throw new DAException(DAResource.METHOD_AFFCTD_NOROWS);");
	outStr.writeln("\tif (rowsAffected > 1) throw new DAException(DAResource.METHOD_AFFCTD_MULTROWS);");
	outStr.writeln("\t_stmt.close(); _stmt = null;\n\tsetDataId(aDataId);");
	outStr.writeln("\t} catch(SQLException exc) {",1);
	outStr.writeln("\t\tthrow new DAException (DAResource.ERROR_IN_METHOD,\"update(dataId)\", exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\tsetModified(false);");
	outStr.writeln("\tif (_poSupport!=null)");
	outStr.writeln("\t\t_poSupport.fireUpdateComplete(\"update(dataId)\", null);\n}");
	outStr.writeln("",0);
    }

    protected void genSetAllModified(Formatter outStr) throws IOException {
	outStr.writeln("void setAllModified() {",1);
	if (!diOnly())
	    outStr.writeln("\tfor(int i=0; i < modified.length; i++) modified[i] = true;");
	outStr.writeln("}");
	outStr.writeln("",0);
    }

    protected void genSetModified(Formatter outStr,boolean isDI) throws IOException {
	outStr.writeln("public void setModified(boolean enable) {",1);
	outStr.writeln("\tif ( !enable ) {");
	if (diIndex >=0 && !isDI)
	    outStr.writeln("\t\ttheDataId.setModified(false);");
	if (!diOnly())
	    outStr.writeln("\t\tfor(int i=0; i < modified.length; i++) modified[i] = false;\n\t}");
	else
	    outStr.writeln("\t}");
	outStr.writeln("\tsuper.setModified(enable);\n}");
	outStr.writeln("",0);
    }

    protected void genIsModified(Formatter outStr) throws IOException {
	if (diIndex>=0) {
	    outStr.writeln("public boolean isModified() {",1);
	    outStr.writeln("\treturn super.isModified() || theDataId.isModified();\n}");
	    outStr.writeln("",0);
	}
    }

    protected void genPODataIdFields(Formatter outStr,ClassDescriptor cl,ColumnDescriptor[] diCols) throws IOException {
	String clName = cl.getDataIdName();
	outStr.writeln("/* "+clName+" Attributes and methods\n*/\n");
	for(int i=0;i<diCols.length;i++) {
	    outStr.writeln("// This attribute is mapped to "+diCols[i].getDBFieldName()+".");
	    outStr.writeln("private "+diCols[i].getAttrDeclaration()+"\n");
	}
	outStr.write("\tboolean modified[] = { false");
	for (int i=1;i<diCols.length;i++)
	    outStr.write(", false");
	outStr.write("};");
	outStr.writeln("\n");
    }

    protected void genPODataIdMethods(Formatter outStr,ClassDescriptor cl,ColumnDescriptor[] diCols)  throws IOException {
	genDataIdConstructors(outStr,cl,diCols);

	// protected setters
	for (int i=0;i<diCols.length;i++)
	    outStr.writeln(diCols[i].getDIPSetterDef());

	// must be methods
	genClone(outStr,cl,diCols,true);
	genEquals(outStr,cl,diCols,true);
	genGetAttributes(outStr,diCols);
	genGetAttributeStrings(outStr,diCols);
	genToString(outStr,diCols);

	//getters
	for (int i=0;i<diCols.length;i++)
	    outStr.writeln(diCols[i].getDIGetterDef());
	
	//public isNullable methods
	for (int i=0;i<diCols.length;i++)
	    outStr.writeln(diCols[i].getNullableDef());

	//public setters
	for (int i=0;i<diCols.length;i++)
	    outStr.writeln(diCols[i].getDISetterDef(i));

	genSetModified(outStr,true);
	
    }

    protected void genDataIdConstructors(Formatter outStr,ClassDescriptor cl,ColumnDescriptor[] cols) throws IOException {
	String clName = cl.getDataIdName();
	outStr.writeln(commBlock("Construcors"));
	outStr.writeln("public "+clName+"() {}");
	outStr.write("public "+clName+"("+cols[0].getAttrParam());
	for (int i=1;i<cols.length;i++)
	    outStr.write(", "+cols[i].getAttrParam());
	outStr.write(") {");
	for (int i=0;i<cols.length;i++)
	    outStr.writeln("\t"+cols[i].getSetAttr(cols[i].getAttrParamName(),";"));
	outStr.writeln("}");

    }


    protected void genManagerMethods(Formatter outStr,ClassDescriptor cl,TableDescriptor t,ColumnDescriptor[] cols,boolean isDI) throws IOException {
	String clName = ((isDI)?cl.getDataIdManagerName():cl.getPOManagerName());
	String bclName = cl.getPOName();
	String idName = cl.getDataIdName();

	ColumnDescriptor[] columns;

	if (isDI) {
	    columns = new ColumnDescriptor[diIndex+1];
	    System.arraycopy(cols,0,columns,0,diIndex+1);
	} else
	    columns = cols;
	//constructors
	outStr.writeln(commBlock("Constructors"));
	outStr.writeln("public "+clName+"() {}");
	outStr.writeln("public "+clName+"(DatastoreJDBC aDatastore) {");
	outStr.writeln("\tsuper(aDatastore);\n}");

	//getCurrentDatastore
	outStr.writeln("public DatastoreJDBC getCurrentDatastore() {");
	outStr.writeln("\tif (getObjectsDatastore() != null) return getObjectsDatastore();");
	outStr.writeln("\telse if ("+bclName+".getDefaultDatastore() != null) return "+bclName+".getDefaultDatastore();");
	outStr.writeln("\telse return DatastoreJDBC.getApplicationDatastore();\n}");
	genPrepareOpen(outStr,cl,isDI);
	genOpen(outStr,cl,isDI);
	genSelect(outStr,cl,isDI);
	genSelectCount(outStr,t);

	//newElement, element
	if (isDI) {
	    outStr.writeln("protected DataAccessObject newElement() { return new "+idName+"();}");
	    outStr.writeln("public "+idName+" element() { return ("+idName+")_cachedObject;}");
	}
	else {
	    outStr.writeln("protected DataAccessObject newElement() { return new "+bclName+"();}");
	    outStr.writeln("public "+bclName+" element() { return ("+bclName+")_cachedObject;}");
	}

	genGetSQLBaseQuery(outStr,t,columns);
	genMExecuteAction(outStr,cl.isReadOnly());
	if (!cl.isReadOnly()) {
	    genDeleteFetched(outStr,t,cl);
	    genUpdateFetched(outStr,cl,columns,isDI);
	}
    }

    protected void genPrepareOpen(Formatter outStr,ClassDescriptor cl,boolean isDI) throws IOException {
	String mapName = ((isDI)?cl.getDataIdMapName():cl.getPOMapName());
	outStr.writeln(commBlock("Open cursor with parameters"),0);
	outStr.writeln("public void prepareOpen(String sqlSuffix,Object[] vals,int[] types) throws DAException {",1);
	outStr.writeln("\tif (sqlSuffix==null|| vals==null|| types==null)");
	outStr.writeln("\t\tthrow new IllegalArgumentException();");
	outStr.writeln("\tif (vals.length != types.length)");
	outStr.writeln("\t\tthrow new IllegalArgumentException();");
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {sqlSuffix,vals,types};");
	outStr.writeln("\t\t_putOnBackgroundThread( \"prepareOpen(..)\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("try{",2);
	outStr.writeln("if (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);",3);
	outStr.writeln("if (isOpen()) close();");
	outStr.writeln("_stmt = getCurrentDatastore().getConnection().prepareStatement(getSQLBaseQuery() + sqlSuffix);");
	outStr.writeln("_setStmtOptions();");
	outStr.writeln("for (int i=0;i<types.length;i++)");
	outStr.writeln("\t_stmt.setObject(i+1,vals[i],types[i]);");
	outStr.writeln("_rs = _stmt.executeQuery();");
	outStr.writeln("_map = new "+mapName+"();\n_setOpen(true);");
	outStr.writeln("\t\t_setFetched(false);\n");
	outStr.writeln("} catch(SQLException exc) {",2);
	outStr.writeln("\tthrow new DAException(DAResource.PREP_OPEN_FAILED, exc);");
	outStr.writeln("} finally { _setBusy(false); }");
	outStr.writeln("_managerSupport.firePrepareOpenComplete(\"prepareOpen(..)\");");
	outStr.writeln("}",1);
    }

    protected void genOpen(Formatter outStr,ClassDescriptor cl,boolean isDI) throws IOException {
	String mapName = ((isDI)?cl.getDataIdMapName():cl.getPOMapName());
	outStr.writeln(commBlock("Open cursor"),0);
	outStr.writeln("public void open(String sqlSuffix) throws DAException {\n",1);
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {sqlSuffix};");
	outStr.writeln("\t\t_putOnBackgroundThread( \"open(String)\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("\ttry {");
	outStr.writeln("\tif (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\t\tif (isOpen()) close();\n\t\tif (sqlSuffix == null) sqlSuffix = \"\";");
	outStr.writeln("\t\t_stmt = getCurrentDatastore().getConnection().prepareStatement(getSQLBaseQuery() + sqlSuffix);");
	outStr.writeln("\t\t_setStmtOptions();\n\t\t_rs = _stmt.executeQuery();");
	outStr.writeln("\t\t_map = new "+mapName+"();\n\t\t_setOpen(true);\n");
	outStr.writeln("\t\t_setFetched(false);\n");
	outStr.writeln("\t} catch(SQLException exc) {");
	outStr.writeln("\t\tthrow new DAException(DAResource.OPEN_FAILED, exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\t_managerSupport.fireOpenComplete(\"open(String)\");\n}");
    }

    protected void genSelect(Formatter outStr,ClassDescriptor cl,boolean isDI) throws IOException {
	String mapName = ((isDI)?cl.getDataIdMapName():cl.getPOMapName());
	outStr.writeln(commBlock("Select rows and fill internal sequence."));
	outStr.writeln("public void select(String sqlSuffix) throws DAException {\n");
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\tObject[] params = {sqlSuffix};");
	outStr.writeln("\t\t_putOnBackgroundThread( \"select(String)\", params );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("\ttry {");
	outStr.writeln("\tif (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\t\tif (isOpen()) close();\n\t\tif (sqlSuffix == null) sqlSuffix = \"\";");
	outStr.writeln("\t\t_stmt = getCurrentDatastore().getConnection().prepareStatement(getSQLBaseQuery() + sqlSuffix);");
	outStr.writeln("\t\t_setStmtOptions();\n\t\t_rs = _stmt.executeQuery();");
	outStr.writeln("\t\t_map = new "+mapName+"();");
	outStr.writeln("\t\t_fillInternalSequence();\n\t\t_dbclose();\n");
	outStr.writeln("\t} catch(SQLException exc) {");
	outStr.writeln("\t\tthrow new DAException(DAResource.SELECT_FAILED, exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\t_managerSupport.fireSelectComplete(\"select(String)\");\n}");

    }

    protected void genSelectCount(Formatter outStr,TableDescriptor t) throws IOException {
	outStr.writeln(commBlock("Get count of the rows."));
	outStr.writeln("public int selectCount(String sqlSuffix) throws DAException {\n");
	outStr.writeln("try {",2);
	outStr.writeln("\tif (getCurrentDatastore() == null) throw new DAException(DAResource.NO_CONNECT_EXIST);");
	outStr.writeln("\tif (isOpen()) close();\n\tif (sqlSuffix == null) sqlSuffix = \"\";");
	String wc = t.getWhereClause("\t");
	if (wc.length()>0)
	    outStr.writeln("\tString stm= \"select count(*)\"+"+t.getFromClause("")+"+\n"+wc+";");
	else
	    outStr.writeln("\tString stm= \"select count(*)\"+"+t.getFromClause("\t")+";");

	outStr.writeln("\t_stmt = getCurrentDatastore().getConnection().prepareStatement(stm + sqlSuffix);");
	outStr.writeln("\t_setStmtOptions();\n\t_rs = _stmt.executeQuery();");
	outStr.writeln("\tint res = _rs.getInt(1);");
	outStr.writeln("\t_rs.close(); _stmt.close();");
	outStr.writeln("\treturn res;");
	outStr.writeln("} catch(SQLException exc) {");
	outStr.writeln("\tthrow new DAException(DAResource.SELECT_FAILED, exc);\n}");
	outStr.writeln("}\n",1);
    }

    protected void genMExecuteAction(Formatter outStr,boolean isReadOnly) throws IOException {
	outStr.writeln("public void _executeAction( String _methodName, Object[] _params ) throws Exception {",1);
	if (!isReadOnly) {
	    outStr.writeln("if ( _methodName.equals(\"deleteFetched()\") ) {",2);
	    outStr.writeln("\tdeleteFetched();\n}");
	    outStr.writeln("else if ( _methodName.equals(\"updateFetched()\") ) {");
	    outStr.writeln("\tupdateFetched();\n}");
	    outStr.writeln("else if (_methodName.equals(\"prepareOpen(..)\")) {");
	} else
	    outStr.writeln("if (_methodName.equals(\"prepareOpen(..)\")) {",2);
	outStr.writeln("\tprepareOpen((String)_params[0],(Object[])_params[1],(int[])_params[2]);\n}");
	outStr.writeln("\telse super._executeAction(_methodName,_params);\n}",1);
    }

    protected void genGetSQLBaseQuery(Formatter outStr,TableDescriptor t,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln("public static String getSQLBaseQuery() {");
	outStr.writeln("\treturn \"select "+cols[0].getPrefixedName()+"\" +");
	for (int i=1;i<cols.length;i++)
	    outStr.writeln("\t\","+cols[i].getPrefixedName()+"\" +");
	String wc = t.getWhereClause("\t");
	if (wc.length()>0)
	    outStr.writeln("\t"+t.getFromClause("")+"+\n"+wc+";");
	else
	    outStr.writeln(t.getFromClause("\t")+";");
	outStr.writeln("}");
    }

    protected void genDeleteFetched(Formatter outStr,TableDescriptor t,ClassDescriptor cl) throws IOException {
	outStr.writeln("public void deleteFetched() throws DAException {\n");
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\t_putOnBackgroundThread( \"deleteFetched()\", null );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("\ttry {");
	outStr.writeln("\t\tif (!isOpen()) throw new DAException(DAResource.CURSOR_NOT_OPEN);");
	outStr.writeln("\t\tif (!isFetched()) throw new DAException(DAResource.ROW_NOT_FETCHED);");
	if (t.getQualifier(0).length()>0)
	    outStr.writeln("\t\tString stmStr = \"delete from \" +" + cl.getPOName()+".getQualifier() + \"."+(t.getTableNames())[0]+" where current of \" + _rs.getCursorName();");
	else
	    outStr.writeln("\t\tString stmStr = \"delete from " +(t.getTableNames())[0]+" where current of \" + _rs.getCursorName();");
	outStr.writeln("\t\tPreparedStatement deleteStmt = getCurrentDatastore().getConnection().prepareStatement( stmStr );");
	outStr.writeln("\t\tdeleteStmt.executeUpdate();");
	outStr.writeln("\t\tdeleteStmt.close();");
	outStr.writeln("\t} catch(SQLException exc) {");
	outStr.writeln("\t\tthrow new DAException(DAResource.ERROR_IN_METHOD,\"deleteFetched()\", exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\t_setFetched(false);");
	outStr.writeln("\telement().setModified(false);");
	outStr.writeln("\t_managerSupport.fireDeleteFetchedComplete();\n}");

    }

    protected void genUpdateFetched(Formatter outStr,ClassDescriptor cl,ColumnDescriptor[] cols,boolean isDI) throws IOException {
	String clName = ((isDI)?cl.getDataIdName():cl.getPOName());
	outStr.writeln("public void updateFetched() throws DAException {\n");
	outStr.writeln("\tif ( !_onBackground() && isAsynchronous() ) {");
	outStr.writeln("\t\t_putOnBackgroundThread( \"updateFetched()\", null );");
	outStr.writeln("\t\treturn ;\n\t}");
	outStr.writeln("\t"+clName+" obj = element();");
	outStr.writeln("\ttry {");
	outStr.writeln("\t\tif (!isOpen()) throw new DAException(DAResource.CURSOR_NOT_OPEN);");
	outStr.writeln("\t\tif (!isFetched()) throw new DAException(DAResource.ROW_NOT_FETCHED);");
	outStr.writeln("\t\tif (!obj.isModified()) throw new DAException(DAResource.NO_CHANGES_MADE);");
	if (isDI) 
	    outStr.writeln("\t\tString stmStr = "+cl.getPOName()+".generator.genUpdateDataIdFetched("+cl.getPOName()+".getQualifier(), obj.modified) + _rs.getCursorName();");
	else {
	    outStr.writeln("\t\tString stmStr = "+clName+".generator.genUpdateFetched("+clName+".getQualifier(),");
	    if (diIndex >= 0) {
		if (diOnly())
		    outStr.writeln("\t\tobj.getDataId().modified, null) + _rs.getCursorName();");
		else
		    outStr.writeln("\t\tobj.getDataId().modified, obj.modified) + _rs.getCursorName();");
	    }
	    else
		outStr.writeln("\t\tnull,obj.modified) + _rs.getCursorName();");
	}
	outStr.writeln("\t\tPreparedStatement updateStmt = getCurrentDatastore().getConnection().prepareStatement( stmStr );");
	outStr.writeln("\t\tint index = 1;");
	int i=0;
	if (!isDI && diIndex >=0) 
	    for (; i<= diIndex;i++) {
		if (cols[i].isModifiable()) {
		    outStr.writeln("\t\tif (obj.getDataId().modified["+i+"]) {");
		    outStr.writeln(cols[i].genSetParamRs("updateStmt","index","obj","\t\t\t"));
		    outStr.writeln("\t\t}");
		}
	    }
	for(int j=0;i<cols.length;i++,j++) {
	    outStr.writeln("\t\tif (obj.modified["+j+"]) {");
	    outStr.writeln(cols[i].genSetParamRs("updateStmt","index","obj","\t\t\t"));
	    outStr.writeln("\t\t}");
	}
	outStr.writeln("\t\tupdateStmt.executeUpdate();\n\t\tupdateStmt.close();");
	outStr.writeln("\t} catch(SQLException exc) {");
	outStr.writeln("\t\tthrow new DAException(DAResource.ERROR_IN_METHOD,\"updateFetched()\", exc);");
	outStr.writeln("\t} finally {\n\t\t_setBusy(false);\n\t}");
	outStr.writeln("\tobj.setModified(false);");
	outStr.writeln("\t_managerSupport.fireUpdateFetchedComplete();\n}");

    }

    protected void genPutResultSetInto(Formatter outStr,String clName,ColumnDescriptor[] cols) throws IOException {
	outStr.writeln("public void putResultSetInto(DataAccessObject obj, ResultSet rs, boolean saveStream) throws DAException, SQLException {");
	outStr.writeln("\t"+clName+" theObj = ("+clName+")obj;\n");
	for (int i=0;i<cols.length;i++)
	    if (cols[i].isStream())
		outStr.writeln(cols[i].genSetFromRsStream("rs",i+1,"theObj","saveStream","\t"));
	    else
		outStr.writeln(cols[i].genSetFromRs("rs",i+1,"theObj","\t"));
	outStr.writeln("\n}");
    }

    private void genHeader(Formatter outStr,String pkg,String cl,String par,String[] infs) throws IOException {
	if (pkg != null)
	    outStr.writeln("package "+pkg+";\n",0);
	outStr.writeln("import java.sql.*;",0);
	outStr.writeln("import java.util.*;");
	outStr.writeln("import JProjects.eab.data.*;");
	if (hasBigDecimal())
	    outStr.writeln("import java.math.*;");
	outStr.writeln("import java.io.Serializable;\n\n");
	outStr.write("class "+cl);
	if (par != null)
	    outStr.write(" extends "+par);
	if (infs != null) {
	    outStr.write(" implements ");
	    outStr.write(infs[0]);
	    for (int i=1;i<infs.length;i++)
		outStr.write(", "+infs[i]);
	}
	outStr.write(" {");
	outStr.writeln("\n",1);
    }
    
    protected boolean hasModifDataId(ColumnDescriptor[] cols) {
	int i=0;
	boolean res = false;
	for(;i<cols.length && !res;i++)
	    res = cols[i].isDataId() && cols[i].isModifiable();
	return res;
    }
    
    private String commBlock(String comm) {
	String commL = "//----------------------------------------------------------------";
	return (commL+"\n// "+comm+"\n"+commL);
    }

    private boolean hasBigDecimal() {
	boolean res = false;
	for (int i=0;i<cols.length && !res;i++)
	    res = cols[i].getAttrType().equals("BigDecimal");
	return res;
    }

    private boolean diOnly() {
	return (diIndex == cols.length - 1);
    }
	
}
