/* BaseController
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package rogatkin;

import java.util.Hashtable;
import java.util.Enumeration;
import java.awt.Cursor;
import java.awt.Component;
import java.awt.Container;
import java.awt.Image;
import java.awt.Toolkit;
import java.io.File;
import java.io.IOException;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;

import javax.swing.*;

public abstract class BaseController implements Persistentable {

    public final static int LOW_MEMORY = 0;

    public final static Cursor WAIT_CURSOR = new Cursor(Cursor.WAIT_CURSOR);
    public final static Cursor DEFAULT_CURSOR = new Cursor(Cursor.DEFAULT_CURSOR);

    public static final String ISO_8859_1 = "iso-8859-1";

    private static final String WIN_ID = "Windows";
    private static final String WIN_PATH = "rundll32";
    private static final String WIN_FLAG = "url.dll,FileProtocolHandler";
    private static final String UNIX_PATH = "netscape";
    private static final String UNIX_FLAG = "-remote openURL";
    private static final String[] HEX_DIGITS = {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "a", "b", "c", "d", "e", "f" };

    public BaseController(Program main) {
	this.main = main;
	HASHHASH = main.getVersion();
        components = new Hashtable();
        serializer = new Serializer(main.getName());
        try {
            ms_vm  = System.getProperty("java.vendor").indexOf("Microsoft") >= 0;
        } catch(NullPointerException e) {
        }
        screen_height = Toolkit.getDefaultToolkit().getScreenSize().height;
    }

    public abstract Image getMainIcon();

    public void save() {
        Enumeration ee = components.elements();
        Object o;
        while(ee.hasMoreElements()) {
            o = ee.nextElement();
            if (o instanceof Persistentable)
                ((Persistentable)o).save();
        }
    }
    
    public void load() {
        Enumeration ee = components.elements();
        Object o;
        while(ee.hasMoreElements()) {
            o = ee.nextElement();
            if (o instanceof Persistentable)
                ((Persistentable)o).load();
        }
    }

    public boolean showUrl(String url) {
        String cmd = "";
        try {
            if (isWindowsPlatform()) {
                // cmd = 'rundll32 url.dll,FileProtocolHandler http://...'
                cmd = WIN_PATH + " " + WIN_FLAG + " " + url;
                Process p = Runtime.getRuntime().exec(cmd);
            } else {
                // Under Unix, Netscape has to be running for the "-remote"
                // command to work.  So, we try sending the command and
                // check for an exit value.  If the exit command is 0,
                // it worked, otherwise we need to start the browser.

                // cmd = 'netscape -remote openURL(http://www.javaworld.com)'
                cmd = UNIX_PATH + " " + UNIX_FLAG + "(" + url + ")";
                Process p = Runtime.getRuntime().exec(cmd);

                try {
                    // wait for exit code -- if it's 0, command worked,
                    // otherwise we need to start the browser up.
                    int exitCode = p.waitFor();

                    if (exitCode != 0){
                        // Command failed, start up the browser
                        // cmd = 'netscape http://www.javaworld.com'
                        cmd = UNIX_PATH + " "  + url;
                        p = Runtime.getRuntime().exec(cmd);
                    }
                } catch(InterruptedException x) {
                    System.err.println("Error bringing up browser, cmd='" + cmd + "'\nCaught: " + x);
                    return false;
                }
            }
        } catch(IOException x) {
            // couldn't exec browser
            System.err.println("Could not invoke browser, using "+cmd+"\n"+x);
            return false;
        }
        return true;
    }

    public static boolean isWindowsPlatform(){
        String os = System.getProperty("os.name");
        if ( os != null && os.startsWith(WIN_ID))
            return true;
        else
            return false;
    }
    
    public static boolean isLowMemory() {
	if(java2)
	    return (Runtime.getRuntime().freeMemory()/1024/1024) < LOW_MEMORY;
	return false;
    }

    public static ImageIcon getResourceIcon(String name) {
        try {
            return new ImageIcon(ClassLoader.getSystemResource((java2?"":"/")+name));
        } catch(Exception e) {
            System.err.println("Couldn't load graphical resource "+name+", 'cause "+e);
        }
        return new ImageIcon();
    }

    public void add(Component component, Object constraint) {
        components.put(constraint, component);
    }

    public Component component(Object constraint) {
        return (Component)components.get(constraint);
    }

    public Enumeration components() {
	return components.elements();
    }

    public Enumeration names() {
	return components.keys();
    }

    public static String encryptXor(String s) { // not trusted
        if (s == null)
            return s;
        StringBuffer result = new StringBuffer(s);
        for (int i = 0; i < result.length(); i++) {
            result.setCharAt(i, (char)(result.charAt(i) ^ HASHHASH.charAt(i % HASHHASH.length())));
        }
        return result.toString();
    }

    public void setWaitCursor(Container comp, boolean on) {
        if (comp == null) {
	    if (main instanceof Container)
		comp = (Container)main;
	    else
		return;
        } else if (ms_vm && comp instanceof JComponent)
            comp = ((JComponent)comp).getTopLevelAncestor();
        if (on)
            comp.setCursor(WAIT_CURSOR);
        else
            comp.setCursor(DEFAULT_CURSOR);
    }

    public JFrame getMainFrame() {
	if (main instanceof JFrame)
	    return (JFrame)main;
	return null;
    }
    
    public Component getMainGlassPane() {
	if (main instanceof JFrame)
	    return ((JFrame)main).getGlassPane();
	return null;
    }
    
    public Serializer getSerializer() {
        return serializer;
    }

    static final int BUF_SIZE = 8192;
    public static void copyFile(File in, File out) throws IOException {
	copyFile(in, new BufferedOutputStream(new FileOutputStream(out), BUF_SIZE));
    }

    public static void copyFile(File in, OutputStream os) throws IOException {
	BufferedInputStream is = new BufferedInputStream(new FileInputStream(in), BUF_SIZE);
	byte [] buffer = new byte[BUF_SIZE];
	int len;
	while((len = is.read(buffer)) > 0)
	    os.write(buffer, 0, len);
	is.close();
	os.close();
    }

    public static String toHexString(String src) {
        StringBuffer result = new StringBuffer(src.length()*4);
        for (int i=0; i<src.length(); i++)
            result.append("0x0").append(Integer.toHexString(src.charAt(i)));
        return result.toString();
    }

    public static String bytesToHex(byte[] ba) {
        StringBuffer result = new StringBuffer(ba.length);
        for (int i = 0; i<ba.length; i++) {
            result.append(HEX_DIGITS[(ba[i]>>4) & 15]).append(HEX_DIGITS[ba[i] & 15]);
        }
        return result.toString();
    }

    public static byte[] hexToBytes(String s) {
        char []ca = s.toCharArray();
	byte[] result = new byte[(ca.length+1) /2];
        for (int i=0; i<(ca.length+1)/2; i++) {
            int r = ca[i*2]-'0';
            if (r>9)
                r = ca[i*2]-'a' + 10;
	    if (r>15)
		r = ca[i*2]-'A' + 10;
	    if (r>15)
		r = 0;
            result[i]=(byte)(r<<4);
	    if (i*2+1 < ca.length) {
		r = ca[i*2+1]-'0';
		if (r>9)
		    r = ca[i*2+1]-'a' + 10;
		if (r>15)
		    r = ca[i*2+1]-'A' + 10;
		if (r<16)
		    result[i]+=(byte)r;
	    }
        }
        return result;
    }

    private static String HASHHASH = "something";
    private Hashtable components;
    protected Program main;
    protected Serializer serializer;
    protected boolean ms_vm;
    protected static boolean java2 = System.getProperty("java.specification.name") != null;
    protected int screen_height;
    static {
        try {
            System.class.getDeclaredMethod("setProperty", new Class[] {String.class, String.class});
            java2 = true;
        } catch(NoSuchMethodException nsme) {
            java2 = false;
        } catch(SecurityException se) {
        }
    }
}
