// WinPlot3D.h
// Base plotting class provides drawing primitives for Windows 95/NT

#include "plot3d.h"

#ifndef	_WINPLOT3D_H_
#define	_WINPLOT3D_H_	1	// avoid re-including


/////////////////////////////////////////////////////////////////////////////
// Base plotting class provides drawing primitives for Windows 95/NT

class WinPlot3D
{
public:
	WinPlot3D()
	{
		m_pWnd=0;
		m_pApp=0;
		m_pDC=0;
		m_pMemDC=0;
		m_hBitmap=0;
		m_monochrome_color=RGB(0,0,0);	// black
		m_draw_rate=0;
		m_draw_counts=0;
		m_is_rendering=0;
		m_is_color=0;
		m_background_color = RGB(255,255,255);	// white
		m_rect.right = m_rect.left = m_rect.bottom = m_rect.top = 0;

	}
	~WinPlot3D()
	{
		if (m_hBitmap)
			DeleteObject(m_hBitmap);
		if (m_pMemDC)
			delete m_pMemDC;
		if (m_pDC && m_pWnd)
			m_pWnd->ReleaseDC(m_pDC);	// be sure to release any device context
		m_pDC = 0;
		m_pMemDC = 0;
		m_pWnd = 0;
		m_pApp = 0;
	}

	// setup routines
	void SetWnd(CWnd* pWnd,CWinApp* pApp) {	m_pWnd = pWnd; m_pApp = pApp; }
	void SetDrawRate(long draw_rate=0) { m_draw_rate = draw_rate; }
	void WinSetColorMode(int ShowColor=0) { m_is_color = ShowColor; }
	void WinSetMonochromePen(long color=RGB(0,0,0)) { m_monochrome_color = color; }
	void WinSetBackgroundColor(long bk_color=RGB(255,255,255)) { m_background_color = bk_color; }

	// supply primitive move/draw for Windows
	void WinMoveTo(int xv,int yv)
	{
		m_pDC->MoveTo(xv,yv);	// issue a move to windows
		m_pMemDC->MoveTo(xv,yv);	// issue a move to windows
	}
	void WinDrawTo(int xv,int yv)
	{
		if (m_is_color)
		{
			CPen* pOldPen1;
			CPen* pOldPen2;
			CPen Pen(PS_SOLID,1,m_monochrome_color);	// construct a pen with the desired color

			pOldPen1 = m_pDC->SelectObject(&Pen);	// select the pen into the device context
			m_pDC->LineTo(xv,yv);		   	// draw the colored line
	        m_pDC->SelectObject(pOldPen1);	// restore the original pen

			pOldPen2 = m_pMemDC->SelectObject(&Pen);	// select the pen into the device context
			m_pMemDC->LineTo(xv,yv);		   	// draw the colored line
	        m_pMemDC->SelectObject(pOldPen1);	// restore the original pen
		}
		else
		{
			// just draw the line if in monochrome mode
			m_pDC->LineTo(xv,yv);
			m_pMemDC->LineTo(xv,yv);
		}
	
		// check for slow draw
		if (!m_draw_rate) return;
		m_draw_counts++;	// one more line drawn
		if (m_draw_counts < m_draw_rate) return;
		Sleep(1);	// one millisecond delay
		m_draw_counts=0;	// reset counter
	}
	void WinColorTo(long color)
	{
		m_monochrome_color = color;	// just save it
	}
	void WinSetPixelColor(int xv,int yv,long color)
	{
		m_pDC->SetPixel(xv,yv,color);
		m_pMemDC->SetPixel(xv,yv,color);
	}
	void WinPlotText(AXIS_PLOT_INFO* pAxisInfo,double degrees,int xv,int yv,CString text)
	{
		BOOL bRC;
		double radians;
	
		if (m_pDC)
		{
			// setup font info
			LOGFONT logFont;
			memset((char*)&logFont,0,sizeof(logFont));
			strcpy(logFont.lfFaceName,pAxisInfo->font_name);
			logFont.lfHeight = pAxisInfo->font_height;
			logFont.lfWidth  = pAxisInfo->font_width;
	
			// check if need to invert text (don't draw text upside down)
			radians = Deg2Rad(degrees);
			if (cos(radians)<0.)
			{
				// need to invert text
				CFont extFont;
	
				// determine width and height of text at zero degrees
				bRC = extFont.CreateFontIndirect(&logFont);
				m_pDC->SelectObject(extFont);
				CSize textExt = m_pDC->GetTextExtent(text);
				
				// calculate opposite corner
				xv = (int)((double)xv + (double)(textExt.cx)*cos(radians) - (double)(textExt.cy)*sin(radians));
				yv = (int)((double)yv - (double)(textExt.cx)*sin(radians) - (double)(textExt.cy)*cos(radians));
				degrees += 180.; // add 180 degrees
				if (degrees > 360.) degrees -= 360.;
			}
	
			// create a rotated font
			CFont rotFont;
			logFont.lfEscapement = (long)(degrees*10.);	// tenths of degree
			bRC = rotFont.CreateFontIndirect(&logFont);
	
			// output the text
			m_pDC->SelectObject(rotFont);
			m_pDC->TextOut(xv,yv,text);
			m_pMemDC->SelectObject(rotFont);
			m_pMemDC->TextOut(xv,yv,text);
		}
	}

	// supply "call-back" routines for Windows
	void WinBegPlot()
	{
		int nWidth, nHeight;

		if (!m_pWnd) return;
	
		// get window size
		m_pWnd->GetClientRect(&m_rect);
		nWidth  = m_rect.right  - m_rect.left;
		nHeight = m_rect.bottom - m_rect.top ;
		CSize wsize = CSize(nWidth,nHeight);
	
		// get device context
		if (m_pWnd && m_pDC) m_pWnd->ReleaseDC(m_pDC);
		m_pDC = m_pWnd->GetDC();
		if (m_pDC==0) return;
		m_pDC->SetBkColor(m_background_color);
		m_pDC->SetMapMode(MM_ISOTROPIC);
		m_pDC->SetWindowOrg(0,wsize.cy);
		m_pDC->SetWindowExt(wsize);
		m_pDC->SetViewportExt(wsize);
		m_pDC->ScaleViewportExt(1,1,-1,1);	// invert y coordinate

		// clear the screen
		m_pDC->FillSolidRect(&m_rect,m_background_color);

		// create a memory device context
		BOOL bval;
		int rc2; 
		if (m_hBitmap)
			DeleteObject(m_hBitmap);
		m_hBitmap = 0;
		if (m_pMemDC)
			delete m_pMemDC;
		m_pMemDC = new CDC;
		if (m_pMemDC==0) return;
		nWidth  = m_rect.right  - m_rect.left;
		nHeight = m_rect.bottom - m_rect.top ;
		bval = m_pMemDC->CreateCompatibleDC(NULL);
		m_hBitmap = CreateCompatibleBitmap(m_pDC->m_hDC, nWidth, nHeight);
		SelectObject(m_pMemDC->m_hDC, m_hBitmap);
		rc2 = m_pMemDC->SetStretchBltMode(STRETCH_DELETESCANS); 
		bval = m_pMemDC->StretchBlt(0,0,nWidth,nHeight,m_pDC,0,0,nWidth,nHeight,SRCCOPY);

		m_pMemDC->SetBkColor(m_background_color);
		m_pMemDC->SetMapMode(MM_ISOTROPIC);
		m_pMemDC->SetWindowOrg(0,wsize.cy);
		m_pMemDC->SetWindowExt(wsize);
		m_pMemDC->SetViewportExt(wsize);
		m_pMemDC->ScaleViewportExt(1,1,-1,1);	// invert y coordinate
	
		// clear the screen
		m_pMemDC->FillSolidRect(&m_rect,m_background_color);

		// select pen for monochrome mode
		if (!m_is_color)
		{
			CPen Pen(PS_SOLID,1,m_monochrome_color);	// construct a pen with the desired color
			m_pDC->SelectObject(&Pen);					// select the pen into the device context
			m_pMemDC->SelectObject(&Pen);					// select the pen into the device context
		}
		m_is_rendering=1;
	}
	void WinEndPlot()
	{
		// release device context
		if (m_pWnd && m_pDC)
			m_pWnd->ReleaseDC(m_pDC);
		m_pDC = 0;
		m_is_rendering=0;
	}

	// returns: 0=ok, 1=no copy
	int CopyDCToClipBoard()
	{
		int rc = 0;
		CDC memdc;
		CDC* dc;
		HBITMAP hBitmap=0;
		BITMAP bm;
		BOOL bval;
		int	nWidth, nHeight;
		int rc2; 

		dc = m_pWnd->GetDC();
		
	 	// create a bitmap from the window contents
		nWidth  = m_rect.right  - m_rect.left;
		nHeight = m_rect.bottom - m_rect.top;
	  	bval = memdc.CreateCompatibleDC(NULL);
	  	hBitmap = CreateCompatibleBitmap(dc->m_hDC, nWidth, nHeight);
		SelectObject(memdc.m_hDC,hBitmap);
		rc2 = memdc.SetStretchBltMode(STRETCH_DELETESCANS); 
		bval = memdc.StretchBlt(0,0,nWidth,nHeight,dc,0,0,nWidth,nHeight,SRCCOPY);
		rc2 = GetObject(hBitmap,sizeof(bm),&bm);
		
		// Open the clipboard
		if (OpenClipboard(NULL))
		{
			// Clean the contents of the clipboard
			bval = EmptyClipboard();
	
		 	// ----- put data to the clipboard data -----
		 	SetClipboardData(CF_BITMAP, hBitmap);
	
	  		// Close the clipboard
		 	bval = CloseClipboard();
		}
		else
		{
			AfxMessageBox( "Clipboard unavailable" );
			rc = 1;
		}
		if (hBitmap)
			DeleteObject(hBitmap);
		return(rc);
	}
	int  IsAbortRequested()
	{
		if (m_pApp==0) return(0);

		MSG Msg;
		unsigned key;
//		ProcessSystemEvents(); // keep pumping windows messages
		
		if (PeekMessage(&Msg,NULL,WM_KEYFIRST,WM_KEYLAST,PM_NOREMOVE))
		{
			// wait for a kestrokes from current thrad's windows message queue
			if (GetMessage(&Msg,NULL,WM_KEYFIRST,WM_KEYLAST) > 0)
			{
			    key = Msg.wParam;
				if (key == VK_ESCAPE) // ESC
					return(1);
			}
		}
		

		// keep window messages pumping to allow plot cancellation
		/*
		MSG msg;
		if ( ::PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE ) ) 
		{ 
			if ( !m_pApp->PumpMessage( ) ) 
			{ 
				::PostQuitMessage(0); 
			} 
		}
		*/
		return(0);
	}
	int IsRendering() { return(m_is_rendering); }

public: // data
	CDC*    m_pMemDC;	// memory Device Context
	RECT	m_rect;		// window rectangle size
	HBITMAP m_hBitmap;

protected: // data
	CDC* m_pDC;			// device context
	CWinApp* m_pApp;	// pointer to the application (we don't own this resource)
	CWnd* m_pWnd;		// pointer to our window (we don't own this resource)
	COLORREF m_monochrome_color;	// pen color for monochrome plotting
	COLORREF m_background_color;	// background window color
	BOOL m_is_rendering;// 0=no, 1=rendering image
	long m_draw_rate;	// 0=normal speed, else draw rate
	long m_draw_counts;	// increments after every line is drawn
	int	 m_is_color;	// 0=monochrome, 1=color
};

#endif	// _WINPLOT3D_H_

// ------------------------------  end WinPlot3D.H  ----------------------------

