/** \file DS_Vector.h
    Vector data structure

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_DS_VECTOR
#define H_DS_VECTOR

#include <DS_Container.h>

#ifndef BASE_VECTOR
  #define BASE_VECTOR void
#endif

class Vector : public Container
{
public:
  /** Default Constructor: Construct an empty vector. */
  Vector(const unsigned flags=0);

  /** Copy Constructor: Construct a copy of a vector. */
  Vector(const Vector&);

  /** Construct a vector from any container. */
  Vector(const Container&);

  /** Construct a vector from an enumeration. */
  Vector(const Enumeration&);

  /** Construct a vector from a non-const enumeration and optionally discard source
       if temp is non-zero. Invaildates e if temp. */
  Vector(Enumeration& e, int temp=0);


  /** Destructor. */
  ~Vector();

  /** Assignment: Copy a vector. */
   Vector& operator=(const Vector&);


// Element Addition, set  

   /** Add an element at the end of this vector. */
   void    addElement(const Data item);

   /** Insert an element at a specified position.
       This method pushes all following elements one forward.
       Throws IndexOutOfBounds if !isValidIndex(index). */
   void    insertElementAt(const Data item, const int index);

   /** Change the data value of the element, without adding elements 
       Throws IndexOutOfBounds if !isValidIndex(index). */
   Data    setElementAt(const Data item, const int index);

   /** Relocate the item to the end of the vector. 
       Throws IndexOutOfBounds if !isValidIndex(index). */
   void    pushToEnd(const int index);

   /** Relocate the item to the beginning of the vector. 
       Throws IndexOutOfBounds if !isValidIndex(index). */
   void    pullToFront(const int index);

   /** Swap the elements in the two positions.
       Throws IndexOutOfBounds if either index is not valid. */
   void    swapElements(const int index1, const int index2);


// Element Retrieval, get

   /** Retrieve an element at the specified index.
       Throws IndexOutOfBounds if !isValidIndex(index). */
   Data    elementAt(const int index) const;

   /** Retrieve the item at position 0. 
       Throws IndexOutOfBounds if !isValidIndex(0). */
   Data    firstElement() const                          { return elementAt(0); }

   /** Retrieve the item at position size()-1. 
       Throws IndexOutOfBounds if size() == 0. */
   Data    lastElement() const                           { return elementAt(size()-1); }


// Element Removal

   /** Remove all items from this vector. */
   void    removeAllElements()                           { clear(); }

   /** Remove the element at position size()-1
       Throws IndexOutOfBounds if size() == 0. */
   void    removeLastElement()                           { removeElement(size()-1); }

   /** Remove the element at the specified position
       Throws IndexOutOfBounds if !isValidIndex(index). */
   void     removeElement(const int index);
 
   /** Remove the element, without destroying it.
       Same as removeElement for non-destructive vectors.
       Returns the item. Throws IndexOutOfBounds if !isValidIndex(index). */
   Data     unlinkElement(const int index);

   
// Sort, Search

   /** Searches serially (from the end), returns -1 if not found.
       Otherwise, returns the index of the item. */
   int      lastIndexOf(const Data item) const;

   /** Sort all items in this vector.  Uses Containable::lessThan() to compare items. */
   void     Sort();

   /** Binary search, Assumes this vector is sorted. */
   DS_BOOL  Find(Data& item) const;


// Implement pure and overridden functions of Container.
// See <Container.h> for complete details.

   /** Add a data item to this vector. */
   virtual int         put(const Data item, const index=-1);

   /** Retrieve an item by index. */
   virtual Data        get(const int index=-1) const;

   /** Pop an item from the end of the vector. */
   virtual Data        pop();

   /** Remove a data item from this vector by key. */
   virtual DS_BOOL     remove(const Data item);

   /** Remove a data item from this vector by index. */
   virtual void        remove(const int index);

   /** Number of elements in this vector. */
   virtual int         size() const;

   /** Remove all elements from this vector. */
   virtual void        clear();

   /** Load this vector with an enumeration. */
           int         load(const Enumeration& e)     { return Vector::load((Enumeration&)e, 0); }

   /** Load this vector with a non-const enumeration and optionally discard source
       if temp is non-zero. Invaildates e if temp. */
   virtual int         load(Enumeration& e, int temp=0);

   /** Load this vector with another vector. */
   virtual int         load(const Vector&);

   /** Searches serially for the item, returns -1 if not found. 
       Otherwise, returns the index of the item. */
   virtual int         indexOf(const Data item) const;


protected:
  BASE_VECTOR *v;
  friend class Enumeration;
};


#include <DS_Enumeration.h>

#endif // H_DS_VECTOR
