// Search.cpp : for HEX/ASCII/EBCDIC searches - implements CHexFindDialog
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include <MultiMon.h>
#include "HexEdit.h"
#include <assert.h>

#include "HexEditDoc.h"
#include "HexEditView.h"
#include "MainFrm.h"

#include "Dlgs.h"
#include "resource.hm"          // For control help IDs
#include "HelpID.hm"            // User defined help IDs

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CHexFindDialog dialog

CHexFindDialog::CHexFindDialog()
{
    //{{AFX_DATA_INIT(CHexFindDialog)
    find_type_ = -1;
    //}}AFX_DATA_INIT

    m_fr.lpTemplateName = MAKEINTRESOURCE(IDD_FIND);
    m_fr.Flags |= FR_ENABLETEMPLATE | FR_DOWN | FR_SHOWHELP;

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    if (!Create(TRUE, "", NULL, 0, mm))
    {
        TRACE0("Failed to create Find Dialog\n");
        ASSERT(0);
        return;
    }

    if (aa->find_y_ != -30000)
    {
        CRect rr;               // Rectangle where we will put the dialog
        GetWindowRect(&rr);

        // Move to where it was when it was last closed
        rr.OffsetRect(aa->find_x_ - rr.left, aa->find_y_ - rr.top);

        CRect scr_rect;         // Rectangle that we want to make sure the window is within

        // Get the rectangle that contains the screen work area (excluding system bars etc)
        if (aa->mult_monitor_)
        {
            HMONITOR hh = MonitorFromRect(&rr, MONITOR_DEFAULTTONEAREST);
            MONITORINFO mi;
            mi.cbSize = sizeof(mi);
            if (hh != 0 && GetMonitorInfo(hh, &mi))
                scr_rect = mi.rcWork;  // work area of nearest monitor
            else
            {
                // Shouldn't happen but if it does use the whole virtual screen
                ASSERT(0);
                scr_rect = CRect(::GetSystemMetrics(SM_XVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_YVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_XVIRTUALSCREEN) + ::GetSystemMetrics(SM_CXVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_YVIRTUALSCREEN) + ::GetSystemMetrics(SM_CYVIRTUALSCREEN));
            }
        }
        else if (!::SystemParametersInfo(SPI_GETWORKAREA, 0, &scr_rect, 0))
        {
            // I don't know if this will ever happen since the Windows documentation
            // is pathetic and does not say when or why SystemParametersInfo might fail.
            scr_rect = CRect(0, 0, ::GetSystemMetrics(SM_CXFULLSCREEN),
                                   ::GetSystemMetrics(SM_CYFULLSCREEN));
        }

        if (rr.left > scr_rect.right - 20)              // off right edge?
            rr.OffsetRect(scr_rect.right - (rr.left+rr.right)/2, 0);
        if (rr.right < scr_rect.left + 20)              // off left edge?
            rr.OffsetRect(scr_rect.left - (rr.left+rr.right)/2, 0);
        if (rr.top > scr_rect.bottom - 20)              // off bottom?
            rr.OffsetRect(0, scr_rect.bottom - (rr.top+rr.bottom)/2);
        // This is not analogous to the prev. 3 since we don't want the window
        // off the top at all, otherwise you can get to the drag bar to move it.
        if (rr.top < scr_rect.top)                      // off top at all?
            rr.OffsetRect(0, scr_rect.top - rr.top);

        MoveWindow(&rr);
    }

    from_edit_ = false;
    if (!radio_hex_.SubclassDlgItem(IDC_FIND_HEX, this) ||
        !radio_ascii_.SubclassDlgItem(IDC_FIND_ASCII, this) ||
        !radio_unicode_.SubclassDlgItem(IDC_FIND_UNICODE, this) ||
        !radio_ebcdic_.SubclassDlgItem(IDC_FIND_EBCDIC, this) )
    {
        TRACE0("Failed to get radio buttons\n");
        ASSERT(0);
        return;
    }
//    if (!edit_.SubclassDlgItem(edt1, this) )
//    {
//      TRACE0("Failed to get edit control\n");
//      ASSERT(0);
//      return;
//    }

    CComboBox *pcombo;          // Ptr to combo box control
    CWnd *pedit;                // Ptr to edit control of combo box

    if ((pcombo = (CComboBox *)GetDlgItem(IDC_COMBO)) == NULL ||
        (pedit = pcombo->ChildWindowFromPoint(CPoint(25,5))) == NULL ||
        pedit == pcombo ||
        !edit_.SubclassWindow(pedit->m_hWnd) )
    {
            TRACE0("Failed to get edit control from combo\n");
    }

    // Get search history from the find tool on the Edit Bar
    CString ss;
    CComboBox *pfindtool = (CComboBox *)mm->m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(pfindtool != NULL);
    int last = pfindtool->GetCount();

    for (int ii = 0; ii < last; ++ii)
    {
        pfindtool->GetLBText(ii, ss);
        if (!ss.IsEmpty())
        {
            char cc = ss[0];
            if (cc == CSearchEditControl::sflag_char ||
                cc == CSearchEditControl::iflag_char)
            {
                ss = ss.Mid(1);
            }
            pcombo->AddString(ss);
            pcombo->SetItemData(pcombo->GetCount()-1, (DWORD)cc);
        }
    }

    update_controls();
    update_controls(1);
    edit_.SetSel(0, -1, FALSE);
}

// Change state of controls in find dialog based on current view and
// text in the edit bar search combo box
// 'down' determines direction of search (0=up, 1=down, -1=no change)
// 'keep_hex' forces the current search to be hex (if FALSE then the type
//     depends on the current search string & the EBCDIC mode of the view).
void CHexFindDialog::update_controls(int down, BOOL keep_hex)
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    // Get current search string in edit bar find combo
    CString ss;                         // String to hold text of combo box
    mm->sec_search_.GetWindowText(ss);

    // Get match case check box window
    CButton *wcase = (CButton *)GetDlgItem(chx2); // xxx dynamic_cast fails (returns NULL)
    ASSERT(wcase != NULL);

    if ((!keep_hex || find_type_ != 0) && !ss.IsEmpty() &&
        (ss[0] == CSearchEditControl::sflag_char ||
         ss[0] == CSearchEditControl::iflag_char) )
    {
        // Enable match case check box and set appropriately
        wcase->EnableWindow(TRUE);
        wcase->SetCheck(ss[0] == CSearchEditControl::sflag_char);

        // Remove flag char (char at start of string)
        ss = ss.Mid(1);

        // Get current view to work out if we want an ASCII or EBCDIC search
        find_type_ = 1;                 // default to ASCII search
        ASSERT(GetView() != NULL);
        if (GetView()->EbcdicMode())
            find_type_ = 3;             // Use EBCDIC search
    }
    else
    {
        // Set to hex type search
        find_type_ = 0;
        // Disable match case check box
        wcase->EnableWindow(FALSE);
    }

    UpdateData(FALSE);

    if (down != -1)
    {
        // Set up/down radio buttons
        CButton *w1 = (CButton *)GetDlgItem(rad1);
        ASSERT(w1 != NULL);
        w1->SetCheck(!down);
        CButton *w2 = (CButton *)GetDlgItem(rad2);
        ASSERT(w2 != NULL);
        w2->SetCheck(down);
    }

    // Set text string
    edit_.SetWindowText(ss);
    CWnd *findnext = GetDlgItem(IDOK);
    findnext->EnableWindow(!ss.IsEmpty());
}

// Force a hex search
void CHexFindDialog::set_hex() 
{
    // Select hex radio button
    CButton *bb;
    bb = (CButton *)GetDlgItem(IDC_FIND_HEX);
    ASSERT(bb != NULL);
    bb->SetCheck(TRUE);
    bb = (CButton *)GetDlgItem(IDC_FIND_ASCII);
    ASSERT(bb != NULL);
    bb->SetCheck(FALSE);
    bb = (CButton *)GetDlgItem(IDC_FIND_UNICODE);
    ASSERT(bb != NULL);
    bb->SetCheck(FALSE);
    bb = (CButton *)GetDlgItem(IDC_FIND_EBCDIC);
    ASSERT(bb != NULL);
    bb->SetCheck(FALSE);

    // Disable match case check box
    bb = (CButton *)GetDlgItem(chx2);
    ASSERT(bb != NULL);
    bb->EnableWindow(FALSE);

    // If we were in a text mode convert the bytes to hex
    if (find_type_ != 0)
        edit_.text2hex(find_type_);

    // Remember that we are now in hex mode and set focus to allow entry of hex digits
    find_type_ = 0;
    edit_.SetFocus();
    edit_.SetSel(0, -1, FALSE);
}

BOOL CHexFindDialog::OnHelpInfo(HELPINFO *pHelpInfo) 
{
    if (GetDlgItem(IDC_COMBO)->m_hWnd == (HWND)pHelpInfo->hItemHandle ||
        GetDlgItem(IDC_FIND_GROUP)->m_hWnd == (HWND)pHelpInfo->hItemHandle ||
        GetDlgItem(IDC_FIND_HEX)->m_hWnd == (HWND)pHelpInfo->hItemHandle ||
        GetDlgItem(IDC_FIND_ASCII)->m_hWnd == (HWND)pHelpInfo->hItemHandle ||
        GetDlgItem(IDC_FIND_EBCDIC)->m_hWnd == (HWND)pHelpInfo->hItemHandle ||
        GetDlgItem(IDC_FIND_UNICODE)->m_hWnd == (HWND)pHelpInfo->hItemHandle  )
    {
        static DWORD id_pairs[] = { 
            IDC_COMBO, HIDC_COMBO,
            IDC_FIND_GROUP, HIDC_FIND_GROUP,
            IDC_FIND_HEX, HIDC_FIND_HEX,
            IDC_FIND_ASCII, HIDC_FIND_ASCII,
            IDC_FIND_UNICODE, HIDC_FIND_UNICODE,
            IDC_FIND_EBCDIC, HIDC_FIND_EBCDIC,
            psh15, HIDC_HELP_BUTTON,
            0,0 
        }; 
 
        CWinApp* pApp = AfxGetApp();
        ASSERT_VALID(pApp);
        ASSERT(pApp->m_pszHelpFilePath != NULL);

        CWaitCursor wait;

        if (!::WinHelp((HWND)pHelpInfo->hItemHandle, pApp->m_pszHelpFilePath, HELP_WM_HELP, (DWORD) (LPSTR) id_pairs))
            return CFindReplaceDialog::OnHelpInfo(pHelpInfo);
        return TRUE;
    }
    else
        return CFindReplaceDialog::OnHelpInfo(pHelpInfo);
}

void CHexFindDialog::OnOK() 
{
    // Copy the current edit text from the combo to the "real" hidden edit control
    CString ss;
    edit_.GetWindowText(ss);
    CWnd *hidden = GetDlgItem(edt1);
    hidden->SetWindowText(ss);

    // Call base class implementation to start the search
    CFindReplaceDialog::OnOK();
}

void CHexFindDialog::DoDataExchange(CDataExchange* pDX)
{
        CDialog::DoDataExchange(pDX);
        //{{AFX_DATA_MAP(CHexFindDialog)
        DDX_Radio(pDX, IDC_FIND_HEX, find_type_);
        //}}AFX_DATA_MAP
}

void CHexFindDialog::OnDestroy() 
{
    CFindReplaceDialog::OnDestroy();
        
    // Save current page to restore when reopened
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CRect rr;
    GetWindowRect(&rr);
    aa->find_x_ = rr.left;
    aa->find_y_ = rr.top;
}

BEGIN_MESSAGE_MAP(CHexFindDialog, CFindReplaceDialog)
        //{{AFX_MSG_MAP(CHexFindDialog)
        ON_BN_CLICKED(IDC_FIND_HEX, OnFindHex)
        ON_BN_CLICKED(IDC_FIND_ASCII, OnFindAscii)
        ON_BN_CLICKED(IDC_FIND_EBCDIC, OnFindEbcdic)
        ON_BN_CLICKED(IDC_FIND_UNICODE, OnFindUnicode)
        ON_CBN_EDITCHANGE(IDC_COMBO, OnEditchangeCombo)
        ON_CBN_SELCHANGE(IDC_COMBO, OnSelchangeCombo)
        ON_WM_DESTROY()
        ON_WM_HELPINFO()
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CHexFindDialog message handlers

void CHexFindDialog::OnFindHex() 
{
    CButton *wcase = (CButton *)GetDlgItem(chx2); // xxx dynamic_cast fails (returns NULL)
    ASSERT(wcase != NULL);

    // Disable match case check box
    wcase->EnableWindow(FALSE);

    if (find_type_ != 0)
        edit_.text2hex(find_type_);

    find_type_ = 0;

    if (from_edit_)
        edit_.SetFocus();
}

void CHexFindDialog::OnFindAscii() 
{
    CButton *wcase = (CButton *)GetDlgItem(chx2); // xxx dynamic_cast fails (returns NULL)
    ASSERT(wcase != NULL);

    // Enable match case check box and set appropriately
    wcase->EnableWindow(TRUE);

    if (find_type_ == 0)
        edit_.hex2text(1);

    find_type_ = 1;

    if (from_edit_)
        edit_.SetFocus();
}

void CHexFindDialog::OnFindUnicode() 
{
    CButton *wcase = (CButton *)GetDlgItem(chx2); // xxx dynamic_cast fails (returns NULL)
    ASSERT(wcase != NULL);

    // Enable match case check box and set appropriately
    wcase->EnableWindow(TRUE);

    if (find_type_ == 0)
        edit_.hex2text(2);
    else if (find_type_ == 1)
        edit_.ascii2unicode();

    find_type_ = 2;

    if (from_edit_)
        edit_.SetFocus();
}

void CHexFindDialog::OnFindEbcdic() 
{
    CButton *wcase = (CButton *)GetDlgItem(chx2); // xxx dynamic_cast fails (returns NULL)
    ASSERT(wcase != NULL);

    // Enable match case check box and set appropriately
    wcase->EnableWindow(TRUE);

    if (find_type_ == 0)
        edit_.hex2text(3);
    else if (find_type_ == 1 || find_type_ == 2)
        // Changed from ASCII/Unicode to EBCDIC - check for invalid EBCDIC chars
        edit_.ascii2ebcdic();

    find_type_ = 3;

    if (from_edit_)
        edit_.SetFocus();
}


void CHexFindDialog::OnEditchangeCombo() 
{
    CButton *findnext = (CButton *)GetDlgItem(IDOK);
    ASSERT(findnext != NULL);
    CString ss;
    edit_.GetWindowText(ss);
    findnext->EnableWindow(!ss.IsEmpty());
}

void CHexFindDialog::OnSelchangeCombo() 
{
    CComboBox *pcombo = (CComboBox *)GetDlgItem(IDC_COMBO);
    char cc = (char)pcombo->GetItemData(pcombo->GetCurSel());

    CButton *bb;
    if (cc == CSearchEditControl::sflag_char ||
        cc == CSearchEditControl::iflag_char)
    {
        find_type_ = 1;

        // Work out if current view is in EBCDIC mode
        BOOL bAsc = TRUE;                       // ASCII (or EBCDIC)?
        ASSERT(GetView() != NULL);
        if (GetView()->EbcdicMode())
        {
            find_type_ = 3;
            bAsc = FALSE;
        }

        // Enable ASCII (or EBCDIC) radio button
        bb = (CButton *)GetDlgItem(IDC_FIND_ASCII);
        ASSERT(bb != NULL);
        bb->SetCheck(bAsc);
        bb = (CButton *)GetDlgItem(IDC_FIND_EBCDIC);
        ASSERT(bb != NULL);
        bb->SetCheck(!bAsc);

        bb = (CButton *)GetDlgItem(IDC_FIND_HEX);
        ASSERT(bb != NULL);
        bb->SetCheck(FALSE);
        bb = (CButton *)GetDlgItem(IDC_FIND_UNICODE);
        ASSERT(bb != NULL);
        bb->SetCheck(FALSE);

        // Enable match case check box and set appropriately
        bb = (CButton *)GetDlgItem(chx2);
        ASSERT(bb != NULL);
        bb->EnableWindow(TRUE);
        bb->SetCheck(cc == CSearchEditControl::sflag_char);
    }
    else
    {
        find_type_ = 0;

        // Enable HEX radio button
        bb = (CButton *)GetDlgItem(IDC_FIND_HEX);
        ASSERT(bb != NULL);
        bb->SetCheck(TRUE);
        bb = (CButton *)GetDlgItem(IDC_FIND_ASCII);
        ASSERT(bb != NULL);
        bb->SetCheck(FALSE);
        bb = (CButton *)GetDlgItem(IDC_FIND_UNICODE);
        ASSERT(bb != NULL);
        bb->SetCheck(FALSE);
        bb = (CButton *)GetDlgItem(IDC_FIND_EBCDIC);
        ASSERT(bb != NULL);
        bb->SetCheck(FALSE);

        // Disable match case check box
        bb = (CButton *)GetDlgItem(chx2);
        ASSERT(bb != NULL);
        bb->EnableWindow(FALSE);
    }
    bb = (CButton *)GetDlgItem(IDOK);
    ASSERT(bb != NULL);

    CString ss;
    pcombo->GetLBText(pcombo->GetCurSel(), ss);
    bb->EnableWindow(!ss.IsEmpty());
}

/////////////////////////////////////////////////////////////////////////////
// CCustEdit

CCustEdit::CCustEdit()
{
}

CCustEdit::~CCustEdit()
{
}

void CCustEdit::normalize_hex()
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    const char *pp;                     // Ptr into orig. (hex digit) string
    int newstart, newend;               // New caret position/selection
    newstart = newend = 0;              // In case of empty text string

    // Allocate enough space (allowing for space padding)
    char *out = new char[(ss.GetLength()*3)/2+2]; // Where chars are stored
    size_t ii, jj;                      // Number of hex nybbles written/read
    char *dd = out;                     // Ptr to current output char

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    for (pp = ss.GetBuffer(0), ii = jj = 0; /*forever*/; ++pp, ++ii)
    {
        if (ii == start)
            newstart = dd - out;        // save new caret position
        if (ii == end)
            newend = dd - out;

        if (*pp == '\0')
            break;

        // Ignore spaces (and anything else)
        if (!isxdigit(*pp))
            continue;           // ignore all but hex digits

        if (aa->hex_ucase_)
            *dd++ = toupper(*pp);       // convert all to upper case
        else
            *dd++ = tolower(*pp);       // convert all to lower case

        if ((++jj % 2) == 0)
            *dd++ = ' ';                // pad with space after 2 nybbles
    }
    if ((jj % 2) == 0 && jj > 0)
        dd--;                   // Remove trailing space
    *dd = '\0';                 // Terminate string

    SetWindowText(out);
    SetSel(newstart, newend);
    delete[] out;
}

/* Convert ASCII string to hex bytes.  The hex bytes correspond to different
 * character sets depending on the value of tt (1 = ASCII, 2 = Unicode, 3 = EBCDIC).
 */
void CCustEdit::text2hex(int tt)
{
    ASSERT(tt == 1 || tt == 2 || tt == 3);
    BOOL bad_chars = FALSE;
    CString ss;

    GetWindowText(ss);
    if (ss.GetLength() == 0)
        return;

    const char *pp;                     // Ptr into input (ASCII char) string
    char *out = new char[ss.GetLength()*6+1]; // Where output chars are stored
    char *dd = out;                     // Ptr to current output char

    // Display hex in upper or lower case?
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    const char *hex_fmt;
    if (aa->hex_ucase_)
        hex_fmt = "%02X ";
    else
        hex_fmt = "%02x ";

    for (pp = ss.GetBuffer(0); *pp != '\0'; ++pp)
    {
        if (tt == 1)
        {
            sprintf(dd, hex_fmt, (unsigned char)*pp);
            dd += 3;
        }
        else if (tt == 2)
        {
            wchar_t ww;

            // Convert ASCII character to wide char then to hex digits
            if (mbtowc(&ww, pp, 1) == 1)
            {
                sprintf(dd, hex_fmt, ww & 0xFF);
                dd += 3;
                sprintf(dd, hex_fmt, (ww>>8) & 0xFF);
                dd += 3;
            }
            else
                bad_chars = TRUE;
        }
        else if (tt == 3)
        {
            if (*pp < 128 && *pp >= 0 && a2e_tab[*pp] != '\0')
            {
                sprintf(dd, hex_fmt, a2e_tab[(unsigned char)*pp]);
                dd += 3;
            }
            else
                bad_chars = TRUE;
        }
    }
    if (dd > out) dd--;                 // Forget trailing space
    *dd = '\0';                         // Terminate string

    SetWindowText(out);
    delete[] out;

    if (bad_chars && tt == 2)
        ::HMessageBox("Invalid Unicode character(s) ignored");
    else if (bad_chars && tt == 3)
        ::HMessageBox("Invalid EBCDIC character(s) ignored");
}

/* Convert hex bytes to text, interpreting the bytes depending on tt
 * (1 = ASCII, 2 = Unicode, 3 = EBCDIC)
 */
void CCustEdit::hex2text(int tt)
{
    ASSERT(tt == 1 || tt == 2 || tt == 3);
    BOOL bad_chars = FALSE;             // Any invalid ASCII/EBCDIC/Unicode chars?
    CString ss;

    GetWindowText(ss);
    if (ss.GetLength() == 0)
        return;

    const char *pp;                     // Ptr to input (hex digit) string

    char *out = new char[ss.GetLength()/2+3]; // Where output chars are stored
    size_t length;                      // Number of hex output nybbles generated
    unsigned int curr;                  // Output character value
    char *dd = out;                     // Ptr to current output char

    for (pp = ss.GetBuffer(0), length = 0, curr = 0; *pp != '\0'; ++pp)
    {
        // Ignore spaces (and anything else)
        if (!isxdigit(*pp))
            continue;

        curr = (curr<<4) + (isdigit(*pp) ? *pp - '0' : toupper(*pp) - 'A' + 10);

        length++;                       // Got one more hex digit
        if (tt == 1 && (length % 2) == 0)
        {
            // Allow any byte value above CR
            if (curr > '\r')
            {
                *dd++ = curr;           // Printable so move to next byte
            }
            else
            {
                length -= 2;    // Forget byte (2 hex digits) just seen
                bad_chars = TRUE;
            }
            curr = 0;                   // Start next char
        }
        else if (tt == 2 && (length % 4) == 0)  // Get 2 bytes for Unicode
        {
            // Swap bytes and convert wide character to MBCS
            if (wctomb(dd, (wchar_t)((curr>>8)&0xFF|(curr<<8)&0xFF00)) == 1)
                ++dd;                   // Keep 1 byte MBCS (= ASCII char)
            else
            {
                length -= 4;
                bad_chars = TRUE;       // Not translateable or > 1 byte MBCS char
            }
            curr = 0;                   // Start next wide char
        }
        else if (tt == 3 && (length % 2) == 0)
        {
            // Only allow valid EBCDIC characters
            if ((*dd = e2a_tab[curr]) != '\0') 
                ++dd;
            else
            {
                length -= 2;    // Forget byte (2 hex digits) just seen
                bad_chars = TRUE;
            }
            curr = 0;                   // Start next char
        }
    }
    if (tt == 2)
    {
        if ((length%4) != 0) bad_chars = TRUE;
        length = length/4;
    }
    else
    {
        if ((length%2) != 0) bad_chars = TRUE;
        length = length/2;
    }
    out[length] = '\0';

    SetWindowText(out);
    delete[] out;

    if (bad_chars && tt == 1)
        ::HMessageBox("ASCII control character(s) ignored");
    else if (bad_chars && tt == 2)
        ::HMessageBox("Undisplayable Unicode character(s) ignored");
    else if (bad_chars && tt == 3)
        ::HMessageBox("Invalid EBCDIC character(s) ignored");
}

void CCustEdit::ascii2ebcdic()
{
    BOOL bad_chars = FALSE;             // Any invalid EBCDIC chars?
    CString ss;

    GetWindowText(ss);
    if (ss.GetLength() == 0)
        return;

    const char *pp;                     // Ptr to input (ASCII) string

    char *out = new char[ss.GetLength()+1]; // Where output chars are stored
    char *dd = out;                     // Ptr to current output char

    // Copy string ignoring invalid EBCDIC chars
    for (pp = ss.GetBuffer(0); *pp != '\0'; ++pp)
    {
        // If valid EBCDIC char keep it
        if (*pp >= 0 && *pp < 128 && a2e_tab[(unsigned char)*pp] != '\0')
            *dd++ = *pp;
        else
            bad_chars = TRUE;
    }
    *dd = '\0';

    SetWindowText(out);
    delete[] out;

    if (bad_chars)
        ::HMessageBox("Invalid EBCDIC character(s) deleted");
}

void CCustEdit::ascii2unicode()
{
    BOOL bad_chars = FALSE;             // Any invalid Unicode chars?
    CString ss;

    GetWindowText(ss);
    if (ss.GetLength() == 0)
        return;

    const char *pp;                     // Ptr to input (ASCII) string

    char *out = new char[ss.GetLength()+1]; // Where output chars are stored
    char *dd = out;                     // Ptr to current output char

    // Copy string ignoring invalid EBCDIC chars
    for (pp = ss.GetBuffer(0); *pp != '\0'; ++pp)
    {
        wchar_t ww;
        // If char can be converted to Unicode then keep it
        if (mbtowc(&ww, pp, 1) == 1)
            *dd++ = *pp;
        else
            bad_chars = TRUE;
    }
    *dd = '\0';

    SetWindowText(out);
    delete[] out;

    if (bad_chars)
        ::HMessageBox("Invalid Unicode character(s) deleted");
}

BEGIN_MESSAGE_MAP(CCustEdit, CEdit)
        //{{AFX_MSG_MAP(CCustEdit)
        ON_WM_CHAR()
        ON_WM_KEYDOWN()
        ON_WM_KILLFOCUS()
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CCustEdit message handlers

void CCustEdit::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;             // Range of current selection
    GetSel(start, end);         // (start == end means no selection just caret)
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    ASSERT(mm->pfind_ != NULL);

    if (nChar == '\r')
    {
        // Ignore Enter which we seem to get if drop down list is visible and user
        // presses enter to select an item from the drop down list.
        return;
    }
    else if (mm->pfind_->find_type_ == 0 && nChar == '\b' &&
        start > 1 && start == end && ss[start-1] == ' ')
    {
        // If deleting 1st nybble (hex mode) then also delete preceding space
        SetSel(start-2, end);
    }
    else if (mm->pfind_->find_type_ == 0 && nChar == ' ')
    {
        // Ignore spaces if in hex mode (tend to type them because of display)
        return;
    }
    else if (mm->pfind_->find_type_ == 0 && nChar != '\b' && !isxdigit(nChar))
    {
        // Non-hex digit entered when in hex mode
        if (::HMessageBox ( "The hex find button is selected and\n"
                            "you have entered an invalid hex digit.\n"
                            "Do you want to switch to a text find?", MB_YESNO) == IDYES)
        {
            int new_start, new_end;
            new_start = new_end = 0;            // Just in case

            // Convert the string to text (remove inserted spaces) and 
            char *out = new char[ss.GetLength()+1]; // Where chars are stored
            size_t ii;                          // Number of hex digits read
            char *dd = out;                     // Ptr to current output char
            const char *pp;                     // Ptr to current input char

            for (pp = ss, ii = 0; /*forever*/; ++pp, ++ii)
            {
                if (ii == start)
                    new_start = dd - out;        // save new caret position
                if (ii == end)
                    new_end = dd - out;

                if (*pp == '\0')
                    break;

                // Ignore spaces (and anything else)
                if (isxdigit(*pp))
                    *dd++ = *pp;
            }
            *dd = '\0';
            SetWindowText(out);
            delete[] out;

            // Work out if we want an ASCII or EBCDIC search
            CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
            ASSERT(mm->pfind_ != NULL);
            BOOL bAsc;
            ASSERT(GetView() != NULL);
            if (GetView()->EbcdicMode())
            {
                mm->pfind_->find_type_ = 3;
                bAsc = FALSE;
            }
            else
            {
                mm->pfind_->find_type_ = 1;
                bAsc = TRUE;                       // ASCII (or EBCDIC)?
            }

            // Enable ASCII (or EBCDIC) radio button
            CButton *bb;
            bb = (CButton *)mm->pfind_->GetDlgItem(IDC_FIND_ASCII);
            ASSERT(bb != NULL);
            bb->SetCheck(bAsc);
            bb = (CButton *)mm->pfind_->GetDlgItem(IDC_FIND_EBCDIC);
            ASSERT(bb != NULL);
            bb->SetCheck(!bAsc);

            bb = (CButton *)mm->pfind_->GetDlgItem(IDC_FIND_HEX);
            ASSERT(bb != NULL);
            bb->SetCheck(FALSE);
            bb = (CButton *)mm->pfind_->GetDlgItem(IDC_FIND_UNICODE);
            ASSERT(bb != NULL);
            bb->SetCheck(FALSE);

            // Enable match case check box and set appropriately
            bb = (CButton *)mm->pfind_->GetDlgItem(chx2);
            ASSERT(bb != NULL);
            bb->EnableWindow(TRUE);
            bb->SetCheck(FALSE);

            (void)SetFocus();
            SetSel(new_start, new_end);
        }
        else
        {
            (void)SetFocus();
            SetSel(start, end);
            return;
        }
    }
    else if ((mm->pfind_->find_type_ == 1 || mm->pfind_->find_type_ == 2) &&
                nChar != '\b' && nChar <= '\r')
    {
        int start, end;
        GetSel(start, end);
        ::HMessageBox("Disallowed control character");
        (void)SetFocus();
        SetSel(start, end);
        return;
    }
    else if (mm->pfind_->find_type_ == 3 && nChar != '\b' && (nChar > 127 || a2e_tab[nChar] == '\0'))
    {
        int start, end;
        GetSel(start, end);
        ::HMessageBox("Invalid EBCDIC character");
        (void)SetFocus();
        SetSel(start, end);
        return;
    }

    CEdit::OnChar(nChar, nRepCnt, nFlags);

    // If in hex mode make sure it looks neat
    GetWindowText(ss);
    if (ss.GetLength() > 0 && mm->pfind_->find_type_ == 0)
        normalize_hex();
}

void CCustEdit::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    ASSERT(mm->pfind_ != NULL);

    if (mm->pfind_->find_type_ == 0 && nChar == VK_DELETE &&
        start == end && ss.GetLength() > start+1 && ss[start+1] == ' ')
        SetSel(start, start+2);
    else if (mm->pfind_->find_type_ == 0 && nChar == VK_LEFT &&
             start == end && start > 0 && ss[start-1] == ' ')
        SetSel(start-1, start-1);

    CEdit::OnKeyDown(nChar, nRepCnt, nFlags);

    // Tidy display if hex and char(s) deleted or caret moved
    GetWindowText(ss);
    if (mm->pfind_->find_type_ == 0 &&
        (nChar == VK_DELETE || nChar == VK_RIGHT || nChar == VK_LEFT) &&
        ::GetKeyState(VK_SHIFT) >= 0 && ss.GetLength() > 0)
    {
        normalize_hex();
    }
}

void CCustEdit::OnKillFocus(CWnd* pNewWnd) 
{
        CEdit::OnKillFocus(pNewWnd);
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
        if (mm->pfind_ != NULL)
            mm->pfind_->from_edit_ = true;
}
/////////////////////////////////////////////////////////////////////////////
// CCustRadio

CCustRadio::CCustRadio()
{
}

CCustRadio::~CCustRadio()
{
}


BEGIN_MESSAGE_MAP(CCustRadio, CButton)
        //{{AFX_MSG_MAP(CCustRadio)
        ON_WM_KILLFOCUS()
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CCustRadio message handlers

void CCustRadio::OnKillFocus(CWnd* pNewWnd) 
{
        CButton::OnKillFocus(pNewWnd);
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
        if (mm->pfind_ != NULL)
            mm->pfind_->from_edit_ = false;
}
