// CalcDlg.cpp : implements the Goto/Calculator dialog
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include <MultiMon.h>
#include <afxpriv.h>  // For WM_KICKIDLE
#include <cmath>
#include "HexEdit.h"
#include "MainFrm.h"
#include "HexEditDoc.h"
#include "HexEditView.h"
#include "CalcDlg.h"
#include "resource.hm"      // Help IDs

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CCalcDlg dialog

CCalcDlg::CCalcDlg(CWnd* pParent /*=NULL*/)
    : CDialog(), purple_pen(PS_SOLID, 0, RGB(0x80, 0, 0x80))
{
    //{{AFX_DATA_INIT(CCalcDlg)
    bits_index_ = -1;
    base_index_ = -1;
	big_endian_ = FALSE;
	//}}AFX_DATA_INIT

    aa_ = dynamic_cast<CHexEditApp *>(AfxGetApp());
    mm_ = dynamic_cast<CMainFrame *>(AfxGetMainWnd());

    op_ = binop_none;
    current_ = previous_ = 0;
    in_edit_ = FALSE;
    visible_ = FALSE;

    // Get last used settings from ini file/registry
    change_base(aa_->GetProfileInt("Calculator", "Base", 16));
    change_bits(aa_->GetProfileInt("Calculator", "Bits", 32));
    current_ = _atoi64(aa_->GetProfileString("Calculator", "Current"));
    memory_ = _atoi64(aa_->GetProfileString("Calculator", "Memory"));

    edit_.pp_ = this;                   // Set parent of edit control
}

BOOL CCalcDlg::Create(CWnd* pParentWnd /*=NULL*/) 
{
    BOOL retval = CDialog::Create(CCalcDlg::IDD, pParentWnd);
    if (aa_->calc_y_ != -30000)
    {
        CRect rr;               // Rectangle where we will put the dialog
        GetWindowRect(&rr);

        // Move to where it was when it was last closed
        rr.OffsetRect(aa_->calc_x_ - rr.left, aa_->calc_y_ - rr.top);

        CRect scr_rect;         // Rectangle that we want to make sure the window is within

        // Get the rectangle that contains the screen work area (excluding system bars etc)
        if (aa_->mult_monitor_)
        {
            HMONITOR hh = MonitorFromRect(&rr, MONITOR_DEFAULTTONEAREST);
            MONITORINFO mi;
            mi.cbSize = sizeof(mi);
            if (hh != 0 && GetMonitorInfo(hh, &mi))
                scr_rect = mi.rcWork;  // work area of nearest monitor
            else
            {
                // Shouldn't happen but if it does use the whole virtual screen
                ASSERT(0);
                scr_rect = CRect(::GetSystemMetrics(SM_XVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_YVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_XVIRTUALSCREEN) + ::GetSystemMetrics(SM_CXVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_YVIRTUALSCREEN) + ::GetSystemMetrics(SM_CYVIRTUALSCREEN));
            }
        }
        else if (!::SystemParametersInfo(SPI_GETWORKAREA, 0, &scr_rect, 0))
        {
            // I don't know if this will ever happen since the Windows documentation
            // is pathetic and does not say when or why SystemParametersInfo might fail.
            scr_rect = CRect(0, 0, ::GetSystemMetrics(SM_CXFULLSCREEN),
                                   ::GetSystemMetrics(SM_CYFULLSCREEN));
        }

        if (rr.left > scr_rect.right - 20)              // off right edge?
            rr.OffsetRect(scr_rect.right - (rr.left+rr.right)/2, 0);
        if (rr.right < scr_rect.left + 20)              // off left edge?
            rr.OffsetRect(scr_rect.left - (rr.left+rr.right)/2, 0);
        if (rr.top > scr_rect.bottom - 20)              // off bottom?
            rr.OffsetRect(0, scr_rect.bottom - (rr.top+rr.bottom)/2);
        // This is not analogous to the prev. 3 since we don't want the window
        // off the top at all, otherwise you can get to the drag bar to move it.
        if (rr.top < scr_rect.top)                      // off top at all?
            rr.OffsetRect(0, scr_rect.top - rr.top);

        MoveWindow(&rr);
    }
    return retval;
}

BOOL CCalcDlg::Create(LPCTSTR lpszClassName, LPCTSTR lpszWindowName, DWORD dwStyle, const RECT& rect, CWnd* pParentWnd, UINT nID, CCreateContext* pContext) 
{
    return CDialog::Create(IDD, pParentWnd);
}

void CCalcDlg::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
    //{{AFX_DATA_MAP(CCalcDlg)
    DDX_Radio(pDX, IDC_64BIT, bits_index_);
    DDX_Radio(pDX, IDC_HEX, base_index_);
	DDX_Check(pDX, IDC_BIG_ENDIAN_FILE_ACCESS, big_endian_);
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CCalcDlg, CDialog)
    //{{AFX_MSG_MAP(CCalcDlg)
    ON_BN_CLICKED(IDC_DIGIT_0, OnDigit0)
    ON_BN_CLICKED(IDC_DIGIT_1, OnDigit1)
    ON_BN_CLICKED(IDC_DIGIT_2, OnDigit2)
    ON_BN_CLICKED(IDC_DIGIT_3, OnDigit3)
    ON_BN_CLICKED(IDC_DIGIT_4, OnDigit4)
    ON_BN_CLICKED(IDC_DIGIT_5, OnDigit5)
    ON_BN_CLICKED(IDC_DIGIT_6, OnDigit6)
    ON_BN_CLICKED(IDC_DIGIT_7, OnDigit7)
    ON_BN_CLICKED(IDC_DIGIT_8, OnDigit8)
    ON_BN_CLICKED(IDC_DIGIT_9, OnDigit9)
    ON_BN_CLICKED(IDC_DIGIT_A, OnDigitA)
    ON_BN_CLICKED(IDC_DIGIT_B, OnDigitB)
    ON_BN_CLICKED(IDC_DIGIT_C, OnDigitC)
    ON_BN_CLICKED(IDC_DIGIT_D, OnDigitD)
    ON_BN_CLICKED(IDC_DIGIT_E, OnDigitE)
    ON_BN_CLICKED(IDC_DIGIT_F, OnDigitF)
    ON_BN_CLICKED(IDC_BACKSPACE, OnBackspace)
    ON_BN_CLICKED(IDC_CLEAR_ENTRY, OnClearEntry)
    ON_BN_CLICKED(IDC_CLEAR, OnClear)
    ON_BN_CLICKED(IDC_EQUALS, OnEquals)
    ON_BN_CLICKED(IDC_AND, OnAnd)
    ON_BN_CLICKED(IDC_ASR, OnAsr)
    ON_BN_CLICKED(IDC_DIVIDE, OnDivide)
    ON_BN_CLICKED(IDC_LSL, OnLsl)
    ON_BN_CLICKED(IDC_LSR, OnLsr)
    ON_BN_CLICKED(IDC_ROL, OnRol)
    ON_BN_CLICKED(IDC_ROR, OnRor)
    ON_BN_CLICKED(IDC_XOR, OnXor)
    ON_BN_CLICKED(IDC_MOD, OnMod)
    ON_BN_CLICKED(IDC_MULTIPLY, OnMultiply)
    ON_BN_CLICKED(IDC_OR, OnOr)
    ON_BN_CLICKED(IDC_UNARY_DEC, OnUnaryDec)
    ON_BN_CLICKED(IDC_UNARY_FACTORIAL, OnUnaryFactorial)
    ON_BN_CLICKED(IDC_UNARY_FLIP, OnUnaryFlip)
    ON_BN_CLICKED(IDC_UNARY_REV, OnUnaryRev)
    ON_BN_CLICKED(IDC_UNARY_INC, OnUnaryInc)
    ON_BN_CLICKED(IDC_UNARY_NOT, OnUnaryNot)
    ON_BN_CLICKED(IDC_UNARY_SIGN, OnUnarySign)
    ON_BN_CLICKED(IDC_UNARY_SQUARE, OnUnarySquare)
    ON_BN_CLICKED(IDC_SUBTRACT, OnSubtract)
    ON_BN_CLICKED(IDC_GO, OnGo)
    ON_BN_CLICKED(IDC_MEM_GET, OnMemGet)
    ON_BN_CLICKED(IDC_MEM_CLEAR, OnMemClear)
    ON_BN_CLICKED(IDC_MEM_ADD, OnMemAdd)
    ON_BN_CLICKED(IDC_MEM_SUBTRACT, OnMemSubtract)
    ON_BN_CLICKED(IDC_MARK_GET, OnMarkGet)
    ON_BN_CLICKED(IDC_MARK_AT, OnMarkAt)
    ON_BN_CLICKED(IDC_MARK_ADD, OnMarkAdd)
    ON_BN_CLICKED(IDC_MARK_SUBTRACT, OnMarkSubtract)
    ON_BN_CLICKED(IDC_SEL_GET, OnSelGet)
    ON_BN_CLICKED(IDC_SEL_AT, OnSelAt)
    ON_BN_CLICKED(IDC_EOF_GET, OnEofGet)
    ON_BN_CLICKED(IDC_GTR, OnGtr)
    ON_BN_CLICKED(IDC_LESS, OnLess)
    ON_BN_CLICKED(IDC_POW, OnPow)
    ON_BN_CLICKED(IDC_MARK_AT_STORE, OnMarkAtStore)
    ON_BN_CLICKED(IDC_MARK_CLEAR, OnMarkClear)
    ON_BN_CLICKED(IDC_MARK_STORE, OnMarkStore)
    ON_BN_CLICKED(IDC_MEM_STORE, OnMemStore)
    ON_BN_CLICKED(IDC_SEL_AT_STORE, OnSelAtStore)
    ON_BN_CLICKED(IDC_SEL_STORE, OnSelStore)
    ON_BN_CLICKED(IDC_UNARY_ROL, OnUnaryRol)
    ON_BN_CLICKED(IDC_UNARY_ROR, OnUnaryRor)
    ON_BN_CLICKED(IDC_UNARY_LSL, OnUnaryLsl)
    ON_BN_CLICKED(IDC_UNARY_LSR, OnUnaryLsr)
    ON_BN_CLICKED(IDC_UNARY_ASR, OnUnaryAsr)
    ON_BN_CLICKED(IDC_UNARY_CUBE, OnUnaryCube)
    ON_BN_CLICKED(IDC_UNARY_AT, OnUnaryAt)
    ON_BN_CLICKED(IDC_UNARY_SQUARE_ROOT, OnUnarySquareRoot)
    ON_WM_DRAWITEM()
    ON_BN_CLICKED(IDC_8BIT, On8bit)
    ON_BN_CLICKED(IDC_16BIT, On16bit)
    ON_BN_CLICKED(IDC_32BIT, On32bit)
    ON_BN_CLICKED(IDC_64BIT, On64bit)
    ON_BN_CLICKED(IDC_BINARY, OnBinary)
    ON_BN_CLICKED(IDC_OCTAL, OnOctal)
    ON_BN_CLICKED(IDC_DECIMAL, OnDecimal)
    ON_BN_CLICKED(IDC_HEX, OnHex)
    ON_BN_CLICKED(IDC_SEL_LEN, OnSelLen)
    ON_BN_CLICKED(IDC_SEL_LEN_STORE, OnSelLenStore)
    ON_WM_CLOSE()
    ON_BN_CLICKED(IDC_ADDOP, OnAdd)
    ON_WM_DESTROY()
	ON_WM_HELPINFO()
    ON_BN_DOUBLECLICKED(IDC_DIGIT_D, OnDigitD)
    ON_BN_DOUBLECLICKED(IDC_BACKSPACE, OnBackspace)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_0, OnDigit0)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_1, OnDigit1)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_2, OnDigit2)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_3, OnDigit3)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_4, OnDigit4)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_5, OnDigit5)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_6, OnDigit6)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_7, OnDigit7)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_8, OnDigit8)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_9, OnDigit9)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_A, OnDigitA)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_B, OnDigitB)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_C, OnDigitC)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_E, OnDigitE)
    ON_BN_DOUBLECLICKED(IDC_DIGIT_F, OnDigitF)
    ON_BN_DOUBLECLICKED(IDC_EQUALS, OnEquals)
    ON_BN_DOUBLECLICKED(IDC_MARK_ADD, OnMarkAdd)
    ON_BN_DOUBLECLICKED(IDC_MARK_SUBTRACT, OnMarkSubtract)
    ON_BN_DOUBLECLICKED(IDC_MEM_ADD, OnMemAdd)
    ON_BN_DOUBLECLICKED(IDC_MEM_SUBTRACT, OnMemSubtract)
    ON_BN_DOUBLECLICKED(IDC_UNARY_ASR, OnUnaryAsr)
    ON_BN_DOUBLECLICKED(IDC_UNARY_AT, OnUnaryAt)
    ON_BN_DOUBLECLICKED(IDC_UNARY_CUBE, OnUnaryCube)
    ON_BN_DOUBLECLICKED(IDC_UNARY_DEC, OnUnaryDec)
    ON_BN_DOUBLECLICKED(IDC_UNARY_FACTORIAL, OnUnaryFactorial)
    ON_BN_DOUBLECLICKED(IDC_UNARY_FLIP, OnUnaryFlip)
    ON_BN_DOUBLECLICKED(IDC_UNARY_INC, OnUnaryInc)
    ON_BN_DOUBLECLICKED(IDC_UNARY_LSL, OnUnaryLsl)
    ON_BN_DOUBLECLICKED(IDC_UNARY_LSR, OnUnaryLsr)
    ON_BN_DOUBLECLICKED(IDC_UNARY_NOT, OnUnaryNot)
    ON_BN_DOUBLECLICKED(IDC_UNARY_REV, OnUnaryRev)
    ON_BN_DOUBLECLICKED(IDC_UNARY_ROL, OnUnaryRol)
    ON_BN_DOUBLECLICKED(IDC_UNARY_ROR, OnUnaryRor)
    ON_BN_DOUBLECLICKED(IDC_UNARY_SIGN, OnUnarySign)
    ON_BN_DOUBLECLICKED(IDC_UNARY_SQUARE, OnUnarySquare)
    ON_BN_DOUBLECLICKED(IDC_UNARY_SQUARE_ROOT, OnUnarySquareRoot)
	ON_BN_CLICKED(IDC_BIG_ENDIAN_FILE_ACCESS, OnBigEndian)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

void CCalcDlg::Redisplay()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CString ss;

    DWORD sel = edit_.GetSel();
    edit_.GetWindowText(ss);
    if (aa->hex_ucase_)
        ss.MakeUpper();
    else
        ss.MakeLower();
    edit_.SetWindowText(ss);
    edit_.SetSel(sel);
}

void CCalcDlg::do_binop(binop_type binop) 
{
    if (in_edit_ || op_ == binop_none)
    {
        calc_previous();
        if (!aa_->refresh_off_ && visible_) edit_.Put();

        // If the value in current_ is not there as a result of a calculation then
        // save operand (current_) and it's source (user-entered, cursor position etc)
        // Also don't save the value if this is the very first thing in the macro.
        if (source_ != km_result)
            aa_->SaveToMacro(source_, current_);

        // Get ready for new value to be entered
        previous_ = current_;
        current_ = 0;
        source_ = aa_->recording_ ? km_user : km_result;
    }
    op_ = binop;
    in_edit_ = FALSE;

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        FixFileButtons();
        if (!overflow_ && !error_)
            ShowBinop(binop);
    }
}

void CCalcDlg::ShowStatus()
{
    if (!aa_->refresh_off_ && visible_)
    {
        ASSERT(GetDlgItem(IDC_OP_DISPLAY) != NULL);
        if (overflow_)
        {
            TRACE0("OVERFLOW!!");
            GetDlgItem(IDC_OP_DISPLAY)->SetWindowText("O");
            ::Beep(3000,400);
        }
        else if (error_)
        {
            TRACE0("ERROR!!");
            GetDlgItem(IDC_OP_DISPLAY)->SetWindowText("E");
            ::Beep(3000,400);
        }
        else
            GetDlgItem(IDC_OP_DISPLAY)->SetWindowText("");
    }
}

void CCalcDlg::ShowBinop(int ii /*=-1*/)
{
    binop_type binop;

    if (ii == -1)
        binop = op_;
    else
        binop = binop_type(ii);

    const char *op_disp = "";
    switch (binop)
    {
    case binop_add:
        op_disp = "+";
        break;
    case binop_subtract:
        op_disp = "-";
        break;
    case binop_multiply:
        op_disp = "*";
        break;
    case binop_divide:
        op_disp = "/";
        break;
    case binop_mod:
        op_disp = "%";
        break;
    case binop_pow:
        op_disp = "**";
        break;
    case binop_gtr:
        op_disp = ">";
        break;
    case binop_less:
        op_disp = "<";
        break;
    case binop_ror:
        op_disp = "=>";
        break;
    case binop_rol:
        op_disp = "<=";
        break;
    case binop_lsl:
        op_disp = "<-";
        break;
    case binop_lsr:
        op_disp = "->";
        break;
    case binop_asr:
        op_disp = "+>";
        break;

    case binop_and:
        op_disp = "&&";
        break;
    case binop_or:
        op_disp = "|";
        break;
    case binop_xor:
        op_disp = "^";
        break;
    }

    ASSERT(visible_);
    ASSERT(GetDlgItem(IDC_OP_DISPLAY) != NULL);
    GetDlgItem(IDC_OP_DISPLAY)->SetWindowText(op_disp);
}

// Values for detecting overflows in some unary operators
static __int64 fact_max[4] = { 5, 8, 12, 20};
static __int64 square_max[4] = { 0xF, 0xFF, 0xFFFF, 0xFFFFFFFF };
static __int64 cube_max[4] = { 6, 40, 1625, 2642244 };
static __int64 signed_fact_max[4] = { 5, 7, 12, 20};
static __int64 signed_square_max[4] = { 11, 181, 46340, 3037000499 };
static __int64 signed_cube_max[4] = { 5, 31, 1290, 2097151 };

void CCalcDlg::do_unary(unary_type unary)
{
    unsigned char *byte;                // Pointer to bytes of current value
    unsigned char cc;
    int ii;                             // Loop variable
    __int64 temp;                       // Temp variables for calcs
    signed char s8;
    signed short s16;
    signed long s32;
    signed __int64 s64;
    CHexEditView *pview;                // Active view (or NULL if no views)

    // Save the value to macro before we do anything with it (unless it's a result of a previous op).
    // Also don't save the value if this (unary op) is the very first thing in the macro.
    if (source_ != km_result)
        aa_->SaveToMacro(source_, current_);

    overflow_ = error_ = FALSE;

    if (unary >= unary_mask)
    {
        // Mask off any high bits that might interfere with the calculations
        current_ &= mask_;
    }
    switch (unary)
    {
    case unary_inc:
        ++current_;
        if ((radix_ == 10 && current_ == ~(mask_>>1)) ||
            (radix_ != 10 && (current_ & mask_) == 0))
        {
            overflow_ = TRUE;
        }
        break;
    case unary_dec:
        if ((radix_ == 10 && current_ == ~(mask_>>1)) ||
            (radix_ != 10 && (current_ & mask_) == 0))
        {
            overflow_ = TRUE;
        }
        current_ --;
        break;
    case unary_sign:
        if (radix_ == 10 && current_ == ~(mask_>>1))
            overflow_ = TRUE;  // eg in 8 bit mode -128 => 128 (overflow)
        current_ = -current_;
        break;
    case unary_square:
        ASSERT(bits_index_ > -1 && bits_index_ < 4);
        if ((radix_ == 10 && current_ > signed_square_max[3-bits_index_]) ||
            (radix_ != 10 && current_ > square_max[3-bits_index_]))
        {
            overflow_ = TRUE;
        }
        current_ = current_ * current_;
        break;
    case unary_squareroot:
        // Rounds down to the nearest int so before we take root add 0.5 to avoid
        // problems. eg. sqrt(4) -> 1.99999999... -> 1 BUT sqrt(4.5) -> 2.12 -> 2.
        if (radix_ == 10 && (current_&sign_mask_) != 0)
        {
            mm_->StatusBarText("Square root of -ve value is illegal");
            error_ = TRUE;
        }
        current_ = unsigned __int64(sqrt(double(__int64(current_)) + 0.5));
        break;
    case unary_cube:
        ASSERT(bits_index_ > -1 && bits_index_ < 4);
        if ((radix_ == 10 && current_ > signed_cube_max[3-bits_index_]) ||
            (radix_ != 10 && current_ > cube_max[3-bits_index_]))
        {
            overflow_ = TRUE;
        }
        current_ = current_ * current_ * current_;
        break;
    case unary_factorial:
        ASSERT(bits_index_ > -1 && bits_index_ < 4);
        if ((radix_ == 10 && current_ > signed_fact_max[3-bits_index_]) ||
            (radix_ != 10 && current_ > fact_max[3-bits_index_]))
        {
            overflow_ = TRUE;
        }
        // Only do the calcs if current_ is not too big (else calcs may take a long time)
        if (current_ < 100)
        {
            temp = 1;
            for (ii = 1; ii < current_; ++ii)
                temp *= ii + 1;
            current_ = temp;
        }
        else
        {
            ASSERT(overflow_);        // (Must have overflowed if current_ is this big.)
            current_ = 0;
        }
        break;
    case unary_not:
        current_ = ~current_;
        break;
    case unary_rol:
        current_ = ((current_ << 1) & mask_) | (current_ >> (bits_ - 1));
        break;
    case unary_ror:
        current_ = (current_ >> 1) | ((current_ << (bits_ - 1)) & mask_);
        break;
    case unary_lsl:
        current_ = (current_ & mask_) << 1;
        break;
    case unary_lsr:
        current_ = (current_ & mask_) >> 1;
        break;
    case unary_asr:
        // Use Compiler signed types to do sign extension
        if (bits_ == 8)
        {
            s8 = current_;
            s8 >>= 1;
            current_ = s8;
        }
        else if (bits_ == 16)
        {
            s16 = current_;
            s16 >>= 1;
            current_ = s16;
        }
        else if (bits_ == 32)
        {
            s32 = current_;
            s32 >>= 1;
            current_ = s32;
        }
        else if (bits_ == 64)
        {
            s64 = current_;
            s64 >>= 1;
            current_ = s64;
        }
        else
            ASSERT(0);
        break;
    case unary_rev:  // Reverse all bits
        // This algorithm tests the current bits by shifting right and testing
        // bottom bit.  Any bits on sets the correspoding bit of the result as
        // it is shifted left.
        // This could be sped up using a table lookup but is probably OK for now.
        temp = 0;
        for (ii = 0; ii < bits_; ++ii)
        {
            temp <<= 1;                  // Make room for the next bit
            if (current_ & 0x1)
                temp |= 0x1;
            current_ >>= 1;              // Move bits down to test the next
        }
        current_ = temp;
        break;
    case unary_flip: // Flip byte order
        // This assumes (of course) that byte order is little-endian in memory
        byte = (unsigned char *)&current_;
        switch (bits_)
        {
        case 8:
            mm_->StatusBarText("Can't flip bytes in 8 bit mode");
            aa_->mac_error_ = 2;
            break;
        case 16:
            cc = byte[0]; byte[0] = byte[1]; byte[1] = cc;
            break;
        case 32:
            cc = byte[0]; byte[0] = byte[3]; byte[3] = cc;
            cc = byte[1]; byte[1] = byte[2]; byte[2] = cc;
            break;
        case 64:
            cc = byte[0]; byte[0] = byte[7]; byte[7] = cc;
            cc = byte[1]; byte[1] = byte[6]; byte[6] = cc;
            cc = byte[2]; byte[2] = byte[5]; byte[5] = cc;
            cc = byte[3]; byte[3] = byte[4]; byte[4] = cc;
            break;
        default:
            ASSERT(0);
        }
        break;
    case unary_at:
        UpdateData();                       // Make sure we know what endianness to use
        pview = GetView();
        if (pview == NULL || (current_&mask_) + bits_/8 > pview->GetDocument()->length())
        {
            if (pview == NULL)
                mm_->StatusBarText("No window open");
            else
                mm_->StatusBarText("@ n: Not enough bytes to end of file to read");
            aa_->mac_error_ = 10;
            return;
        }

        temp = 0;
        pview->GetDocument()->GetData((unsigned char *)&temp, bits_/8, current_);
        if (big_endian_)
        {
            // Reverse the byte order to match that used internally (Intel=little-endian)
            byte = (unsigned char *)&temp;
            switch (bits_)
            {
            case 8:
                /* nothing */
                break;
            case 16:
                cc = byte[0]; byte[0] = byte[1]; byte[1] = cc;
                break;
            case 32:
                cc = byte[0]; byte[0] = byte[3]; byte[3] = cc;
                cc = byte[1]; byte[1] = byte[2]; byte[2] = cc;
                break;
            case 64:
                cc = byte[0]; byte[0] = byte[7]; byte[7] = cc;
                cc = byte[1]; byte[1] = byte[6]; byte[6] = cc;
                cc = byte[2]; byte[2] = byte[5]; byte[5] = cc;
                cc = byte[3]; byte[3] = byte[4]; byte[4] = cc;
                break;
            default:
                ASSERT(0);
            }
        }
        current_ = temp;
        break;

    case unary_none:
    default:
        ASSERT(0);
    }

    ShowStatus();                       // Indicate overflow etc

    if (overflow_ || error_)
    {
        if (overflow_)
            mm_->StatusBarText("Result overflowed data size");
        else
            mm_->StatusBarText("Arithmetic error (root of -ve, etc)");
        aa_->mac_error_ = 2;
    }

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    // Save the unary operation to the current macro (if any)
    aa_->SaveToMacro(km_unaryop, long(unary));

    // The user can't edit the result of a unary operation by pressing digits
    in_edit_ = FALSE;
    source_ = km_result;
}

void CCalcDlg::do_digit(char digit)
{
    if (!aa_->refresh_off_ && visible_)  // Always true but left in for consistency
    {
        edit_.SetFocus();
        if (!in_edit_)
        {
            edit_.SetWindowText("");
            ShowBinop(op_);
        }
    }
    edit_.SendMessage(WM_CHAR, digit, 1);
    in_edit_ = TRUE;
    source_ = aa_->recording_ ? km_user : km_result;
}

void CCalcDlg::calc_previous()
{
    bool same_sign;                     // Used in signed (radix 10 only) tests
    int ii;
    __int64 temp;                       // Temp used in calcs
    signed char s8;                     // Signed int types used for ASR
    signed short s16;
    signed long s32;
    signed __int64 s64;

    overflow_ = error_ = FALSE;

    if (op_ >= binop_mask)
    {
        // Mask off any high bits that might interfere with the calculations
        current_ &= mask_;
        previous_ &= mask_;
    }

    switch (op_)
    {
    case binop_add:
        if (radix_ == 10)
            same_sign = (current_ & sign_mask_) == (previous_ & sign_mask_);
        else if ((previous_&mask_) > mask_ - (current_&mask_))
            overflow_ = TRUE;
        current_ += previous_;

        // For signed numbers overflow if operands have same sign which is diff to sign of result
        if (radix_ == 10 && same_sign && (current_ & sign_mask_) != (previous_ & sign_mask_))
            overflow_ = TRUE;
        break;
    case binop_subtract:
        if (radix_ == 10)
            same_sign = (current_ & sign_mask_) == (previous_ & sign_mask_);
        else if ((previous_&mask_) < (current_&mask_))
            overflow_ = TRUE;
        current_ = previous_ - current_;

        if (radix_ == 10 && !same_sign && (current_ & sign_mask_) != (previous_ & sign_mask_))
            overflow_ = TRUE;
        break;
    case binop_multiply:
        if (current_ != 0 && radix_ == 10)
        {
            if ((bits_ ==  8 && abs((signed char)previous_) > abs((signed char)(mask_>>1)/(signed char)current_)) ||
                (bits_ == 16 && abs((signed short)previous_) > abs((signed short)(mask_>>1)/(signed short)current_)) ||
                (bits_ == 32 && abs((signed long)previous_) > abs((signed long)(mask_>>1)/(signed long)current_)) ||
                (bits_ == 64 && abs((signed __int64)previous_) > abs((signed __int64)(mask_>>1)/(signed __int64)current_)))
            {
                overflow_ = TRUE;
            }
        }
        else if (current_ != 0 && (previous_&mask_) > mask_/(current_&mask_))
            overflow_ = TRUE;
        current_ *= previous_;
        break;
    case binop_divide:
        if ((current_&mask_) == 0)
            current_ = 0, error_ = TRUE;
        else
            current_ = previous_ / current_;
        break;
    case binop_mod:
        if ((current_&mask_) == 0)
            current_ = 0, error_ = TRUE;
        else
            current_ = previous_ % current_;
        break;
    case binop_pow:
        if (previous_ == 1 || current_ == 0)
            current_ = 1;               // One to anything or anything to zero is one (assume 0^0 == 1)
        else if (previous_ == 0)
            current_ = 0;               // Zero to anything is zero
        else
        {
            double logmax = log(pow(2.0, double(bits_)) - 1.0);
            if ((signed __int64)current_ > logmax/log((signed __int64)previous_))
                overflow_ = TRUE;
            // Only do the calcs if current_ is not too big (else calcs may take a long time)
            if (current_ < 128)
            {
                temp = 1;
                // Using a short loop should not be too slow but could possibly be sped up if necessary.
                for (ii = 0; ii < current_; ++ii)
                    temp *= previous_;
                current_ = temp;
            }
            else
            {
                ASSERT(overflow_);        // (Must have overflowed if current_ is this big.)
                current_ = 0;
            }
        }
        break;
    case binop_gtr:
    case binop_gtr_old:
        if ((radix_ == 10 && bits_ ==  8 && (signed char)current_ <= (signed char)previous_) ||
            (radix_ == 10 && bits_ == 16 && (signed short)current_ <= (signed short)previous_) ||
            (radix_ == 10 && bits_ == 32 && (signed long)current_ <= (signed long)previous_) ||
            (radix_ == 10 && bits_ == 64 && (signed __int64)current_ <= (signed __int64)previous_) ||
            (radix_ != 10 && bits_ ==  8 && (unsigned char)current_ <= (unsigned char)previous_) ||
            (radix_ != 10 && bits_ == 16 && (unsigned short)current_ <= (unsigned short)previous_) ||
            (radix_ != 10 && bits_ == 32 && (unsigned long)current_ <= (unsigned long)previous_) ||
            (radix_ != 10 && bits_ == 64 && (unsigned __int64)current_ <= (unsigned __int64)previous_) )
        {
            current_ = previous_;
        }
        else
            aa_->mac_error_ = 1;        // Allow detection of max value
        break;
    case binop_less:
    case binop_less_old:
        if ((radix_ == 10 && bits_ ==  8 && (signed char)current_ >= (signed char)previous_) ||
            (radix_ == 10 && bits_ == 16 && (signed short)current_ >= (signed short)previous_) ||
            (radix_ == 10 && bits_ == 32 && (signed long)current_ >= (signed long)previous_) ||
            (radix_ == 10 && bits_ == 64 && (signed __int64)current_ >= (signed __int64)previous_) ||
            (radix_ != 10 && bits_ ==  8 && (unsigned char)current_ >= (unsigned char)previous_) ||
            (radix_ != 10 && bits_ == 16 && (unsigned short)current_ >= (unsigned short)previous_) ||
            (radix_ != 10 && bits_ == 32 && (unsigned long)current_ >= (unsigned long)previous_) ||
            (radix_ != 10 && bits_ == 64 && (unsigned __int64)current_ >= (unsigned __int64)previous_) )
        {
            current_ = previous_;
        }
        else
            aa_->mac_error_ = 1;        // Allow detection of min value
        break;
    case binop_ror:
        temp = current_ % bits_;
        current_ = ((previous_ << temp) & mask_) | (previous_ >> (bits_ - temp));
        break;
    case binop_rol:
        temp = current_ % bits_;
        current_ = (previous_ >> temp) | ((previous_ << (bits_ - temp)) & mask_);
        break;
    case binop_lsl:
        current_ = (previous_ << current_) & mask_;
        break;
    case binop_lsr:
        current_ = previous_ >> current_;
        break;
    case binop_asr:
        // Use Compiler signed types to do sign extension
        if (bits_ == 8)
        {
            s8 = previous_;
            s8 >>= current_;
            current_ = s8;
        }
        else if (bits_ == 16)
        {
            s16 = previous_;
            s16 >>= current_;
            current_ = s16;
        }
        else if (bits_ == 32)
        {
            s32 = previous_;
            s32 >>= current_;
            current_ = s32;
        }
        else if (bits_ == 64)
        {
            s64 = previous_;
            s64 >>= current_;
            current_ = s64;
        }
        else
            ASSERT(0);
        break;

    case binop_and:
        current_ &= previous_;
        break;
    case binop_or:
        current_ |= previous_;
        break;
    case binop_xor:
        current_ ^= previous_;
        break;

    default:
        ASSERT(0);
        /* fall through in release versions */
    case binop_none:
        // No operation but convert a +ve decimal that is too big to be
        // represented to the equivalent -ve value.
        if (radix_ == 10 && (current_ & ((mask_ + 1)>>1)) != 0)
        {
            current_ |= ~mask_;
        }
        break;
    }

    // Check if the result overflowed the number of bits in use.
    // This will not detect overflow for bits_ == 64, in which case
    // overflow should have been detected above.

    if (op_ >= binop_mask)
    {
        // If we are using decimals check if the value is -ve (has high bit set)
        if (radix_ == 10 && (current_ & ((mask_ + 1)>>1)) != 0)
        {
            // Make sure all the high bits are on
            if (~(current_ | mask_) != 0)
                overflow_ = TRUE;
        }
        else
        {
            // Make sure all the high bits are off
            if ((current_ & ~mask_) != 0)
                overflow_ = TRUE;
        }
    }

    ShowStatus();                       // Indicate overflow/error (if any)

    if (overflow_ || error_)
    {
        if (overflow_)
            mm_->StatusBarText("Result overflowed data size");
        else
            mm_->StatusBarText("Arithmetic error (divide by 0, etc)");
        aa_->mac_error_ = 2;
    }

    // The operation has actually been used so save it to macro
    if (op_ != binop_none)
        aa_->SaveToMacro(km_binop, long(op_));
}

void CCalcDlg::toggle_endian()
{
    big_endian_ = !big_endian_;
    if (!aa_->refresh_off_ && visible_)
    {
        // Update check box
        UpdateData(FALSE);
    }
}

void CCalcDlg::change_base(int base)
{
    radix_ = base;                      // Store new radix (affects edit control display)

    // Set up index for next radio button update
    switch (radix_)
    {
    case 2:
        base_index_ = 3;
        break;
    case 8:
        base_index_ = 2;
        break;
    case 10:
        base_index_ = 1;
        break;
    case 16:
        base_index_ = 0;
        break;
    default:
        ASSERT(0);
    }

    if (!aa_->refresh_off_ && visible_)
    {
        // Update radio buttons
        UpdateData(FALSE);

        // Enable/disable digits keys depending on base
        ASSERT(GetDlgItem(IDC_DIGIT_2) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_3) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_4) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_5) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_6) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_7) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_8) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_9) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_A) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_B) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_C) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_D) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_E) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_F) != NULL);
        GetDlgItem(IDC_DIGIT_2)->EnableWindow(radix_ > 2);
        GetDlgItem(IDC_DIGIT_3)->EnableWindow(radix_ > 3);
        GetDlgItem(IDC_DIGIT_4)->EnableWindow(radix_ > 4);
        GetDlgItem(IDC_DIGIT_5)->EnableWindow(radix_ > 5);
        GetDlgItem(IDC_DIGIT_6)->EnableWindow(radix_ > 6);
        GetDlgItem(IDC_DIGIT_7)->EnableWindow(radix_ > 7);
        GetDlgItem(IDC_DIGIT_8)->EnableWindow(radix_ > 8);
        GetDlgItem(IDC_DIGIT_9)->EnableWindow(radix_ > 9);
        GetDlgItem(IDC_DIGIT_A)->EnableWindow(radix_ > 10);
        GetDlgItem(IDC_DIGIT_B)->EnableWindow(radix_ > 11);
        GetDlgItem(IDC_DIGIT_C)->EnableWindow(radix_ > 12);
        GetDlgItem(IDC_DIGIT_D)->EnableWindow(radix_ > 13);
        GetDlgItem(IDC_DIGIT_E)->EnableWindow(radix_ > 14);
        GetDlgItem(IDC_DIGIT_F)->EnableWindow(radix_ > 15);

        // edit_.SetLimitText(???);

        edit_.SetFocus();
        edit_.Put();                    // Show (same) number in new base
    }

    source_ = km_result;
    aa_->SaveToMacro(km_change_base, long(radix_));
}

void CCalcDlg::change_bits(int bits)
{
    __int64 one = 1;

    bits_ = bits;
    mask_ = (one << bits_) - 1;
    sign_mask_ = one << (bits - 1);

    // Set up index for next radio button update
    switch (bits_)
    {
    case 8:
        bits_index_ = 3;
        break;
    case 16:
        bits_index_ = 2;
        break;
    case 32:
        bits_index_ = 1;
        break;
    case 64:
        bits_index_ = 0;
        break;
    default:
        ASSERT(0);
    }

    overflow_ = error_ = FALSE;
    if (radix_ == 10 &&
        ((current_&sign_mask_) == 0 && (current_&~mask_) != 0 ||
         (current_&sign_mask_) != 0 && ~(current_|mask_) != 0) ||
        radix_ != 10 && (current_ & ~mask_) != 0)
    {
        overflow_ = TRUE;
        mm_->StatusBarText("Current value overflowed new data size");
        aa_->mac_error_ = 1;
    }
    ShowStatus();                       // Set/clear overflow indicator

    if (!aa_->refresh_off_ && visible_)
    {
        UpdateData(FALSE);              // Update base radio buttons

        // Disable flip bytes button if there is only one byte
        ASSERT(GetDlgItem(IDC_UNARY_FLIP) != NULL);
        GetDlgItem(IDC_UNARY_FLIP)->EnableWindow(bits_ > 8);
        ASSERT(GetDlgItem(IDC_BIG_ENDIAN_FILE_ACCESS) != NULL);
        GetDlgItem(IDC_BIG_ENDIAN_FILE_ACCESS)->EnableWindow(bits_ > 8);

        //    edit_.SetLimitText(???);

        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    source_ = km_result;
    aa_->SaveToMacro(km_change_bits, long(bits_));
}

void CCalcDlg::FixFileButtons()
{
    ASSERT(visible_);      // Should only be called if dlg is visible
    ASSERT(GetDlgItem(IDC_MARK_GET) != NULL);
    ASSERT(GetDlgItem(IDC_MARK_STORE) != NULL);
    ASSERT(GetDlgItem(IDC_MARK_CLEAR) != NULL);
    ASSERT(GetDlgItem(IDC_MARK_ADD) != NULL);
    ASSERT(GetDlgItem(IDC_MARK_SUBTRACT) != NULL);
    ASSERT(GetDlgItem(IDC_MARK_AT) != NULL);
    ASSERT(GetDlgItem(IDC_MARK_AT_STORE) != NULL);
    ASSERT(GetDlgItem(IDC_SEL_GET) != NULL);
    ASSERT(GetDlgItem(IDC_SEL_STORE) != NULL);
    ASSERT(GetDlgItem(IDC_SEL_AT) != NULL);
    ASSERT(GetDlgItem(IDC_SEL_AT_STORE) != NULL);
    ASSERT(GetDlgItem(IDC_SEL_LEN) != NULL);
    ASSERT(GetDlgItem(IDC_SEL_LEN_STORE) != NULL);
    ASSERT(GetDlgItem(IDC_EOF_GET) != NULL);
    ASSERT(GetDlgItem(IDC_GO) != NULL);
    ASSERT(GetDlgItem(IDC_UNARY_AT) != NULL);
    ASSERT(GetDlgItem(IDC_BIG_ENDIAN_FILE_ACCESS) != NULL);

    CHexEditView *pview = GetView();
    FILE_ADDRESS start, end;                    // Current selection
    FILE_ADDRESS eof, mark;                     // Length of file and posn of mark

    if (pview != NULL)
    {
        pview->GetSelAddr(start, end);
        eof = pview->GetDocument()->length();
        mark = pview->GetMark();
    }

    GetDlgItem(IDC_MARK_GET)->EnableWindow(pview != NULL);
    GetDlgItem(IDC_MARK_STORE)->EnableWindow(pview != NULL && (current_&mask_) <= eof);
    GetDlgItem(IDC_MARK_CLEAR)->EnableWindow(pview != NULL);
    if (pview != NULL && radix_ == 10)
    {
        signed __int64 mark_minus, mark_plus;
        switch (bits_)
        {
        case 8:
            mark_minus = mark - (signed char)current_;
            mark_plus  = mark + (signed char)current_;
            break;
        case 16:
            mark_minus = mark - (signed short)current_;
            mark_plus  = mark + (signed short)current_;
            break;
        case 32:
            mark_minus = mark - (signed long)current_;
            mark_plus  = mark + (signed long)current_;
            break;
        case 64:
            mark_minus = mark - (signed __int64)current_;
            mark_plus  = mark + (signed __int64)current_;
            break;
        }
        GetDlgItem(IDC_MARK_SUBTRACT)->EnableWindow(pview != NULL && mark_minus > 0 && mark_minus <= eof);
        GetDlgItem(IDC_MARK_ADD)->EnableWindow(pview != NULL && mark_plus > 0 && mark_plus <= eof);
    }
    else
    {
        GetDlgItem(IDC_MARK_SUBTRACT)->EnableWindow(pview != NULL && mark - (current_&mask_) <= eof);
        GetDlgItem(IDC_MARK_ADD)->EnableWindow(pview != NULL && mark + (current_&mask_) <= eof);
    }
    GetDlgItem(IDC_MARK_AT)->EnableWindow(pview != NULL && mark + bits_/8 <= eof);
    GetDlgItem(IDC_MARK_AT_STORE)->EnableWindow(pview != NULL && !pview->ReadOnly() && mark <= eof &&
                                                (mark + bits_/8 <= eof || !pview->OverType()));
    GetDlgItem(IDC_SEL_GET)->EnableWindow(pview != NULL);
    GetDlgItem(IDC_SEL_STORE)->EnableWindow(pview != NULL && (current_&mask_) <= eof);
    GetDlgItem(IDC_SEL_AT)->EnableWindow(pview != NULL && start + bits_/8 <= eof);
    GetDlgItem(IDC_SEL_AT_STORE)->EnableWindow(pview != NULL && !pview->ReadOnly() && start <= eof &&
                                               (start + bits_/8 <= eof || !pview->OverType()));
    GetDlgItem(IDC_SEL_LEN)->EnableWindow(pview != NULL);
    GetDlgItem(IDC_SEL_LEN_STORE)->EnableWindow(pview != NULL && start + (current_&mask_) <= eof);
    GetDlgItem(IDC_EOF_GET)->EnableWindow(pview != NULL);
    GetDlgItem(IDC_GO)->EnableWindow(pview != NULL && (current_&mask_) <= eof);
    GetDlgItem(IDC_UNARY_AT)->EnableWindow(pview != NULL && (current_&mask_) + bits_/8 <= eof);
    GetDlgItem(IDC_BIG_ENDIAN_FILE_ACCESS)->EnableWindow(pview != NULL);
}

// Button drawing funcs for OnDrawItem below
static void Draw3DFrame(CDC *pDC, CRect rcBox, COLORREF colBottomRight, COLORREF colTopLeft)
{
    CPen *pPen2, *pPen, *pOldPen;

    pPen = new CPen(PS_SOLID, 1, colBottomRight);
    pOldPen = pDC->SelectObject(pPen);
    pDC->MoveTo(rcBox.right-1, rcBox.top);
    pDC->LineTo(rcBox.right-1, rcBox.bottom-1);
    pDC->LineTo(rcBox.left-1, rcBox.bottom-1);

    pPen2 = new CPen(PS_SOLID, 1, colTopLeft);
    pDC->SelectObject(pPen2);
    delete pPen;

    pDC->MoveTo(rcBox.left, rcBox.bottom-2);
    pDC->LineTo(rcBox.left, rcBox.top);
    pDC->LineTo(rcBox.right-1, rcBox.top);

    pDC->SelectObject(pOldPen);
    delete pPen2;
}

static void Draw3DButtonFrame(CDC *pDC, CRect rcButton, BOOL bFocus)
{
    CPen *pPen, *pOldPen;
    CBrush GrayBrush(GetSysColor(COLOR_BTNSHADOW));
    CBrush BlackBrush(GetSysColor(COLOR_BTNFACE));

    pPen = new CPen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME));
    pOldPen = pDC->SelectObject(pPen);

    // Draw gray outside
    pDC->FrameRect(&rcButton, &BlackBrush);
    rcButton.InflateRect(-1, -1);

    pDC->MoveTo(rcButton.left+1, rcButton.top);
    pDC->LineTo(rcButton.right-1, rcButton.top);
    pDC->MoveTo(rcButton.left+1, rcButton.bottom-1);
    pDC->LineTo(rcButton.right-1, rcButton.bottom-1);
    pDC->MoveTo(rcButton.left, rcButton.top+1);
    pDC->LineTo(rcButton.left, rcButton.bottom-1);
    pDC->MoveTo(rcButton.right-1, rcButton.top+1);
    pDC->LineTo(rcButton.right-1, rcButton.bottom-1);

    if (bFocus)
    {
        rcButton.InflateRect(-3, -3);
        pDC->FrameRect(&rcButton, &GrayBrush);
    }

    pDC->SelectObject(pOldPen);
    delete pPen;
}

/////////////////////////////////////////////////////////////////////////////
// CCalcDlg message handlers

BOOL CCalcDlg::OnInitDialog() 
{
    CDialog::OnInitDialog();

    ASSERT(GetDlgItem(IDC_EDIT) != NULL);
    if (!edit_.SubclassDlgItem(IDC_EDIT, this))
    {
        ::HMessageBox("Could not subclass edit control.");
        return -1;
    }

    // If saved memory value was restored then make it available
    if (memory_ != 0)
        GetDlgItem(IDC_MEM_GET)->EnableWindow(TRUE);

    // Set up tool tips
    if (ttc_.Create(this))
    {
        ASSERT(GetDlgItem(IDC_DIGIT_0) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_1) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_2) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_3) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_4) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_5) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_6) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_7) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_8) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_9) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_A) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_B) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_C) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_D) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_E) != NULL);
        ASSERT(GetDlgItem(IDC_DIGIT_F) != NULL);
        ASSERT(GetDlgItem(IDC_BACKSPACE) != NULL);
        ASSERT(GetDlgItem(IDC_CLEAR_ENTRY) != NULL);
        ASSERT(GetDlgItem(IDC_CLEAR) != NULL);
        ASSERT(GetDlgItem(IDC_EQUALS) != NULL);
        ASSERT(GetDlgItem(IDC_AND) != NULL);
        ASSERT(GetDlgItem(IDC_ASR) != NULL);
        ASSERT(GetDlgItem(IDC_DIVIDE) != NULL);
        ASSERT(GetDlgItem(IDC_LSL) != NULL);
        ASSERT(GetDlgItem(IDC_LSR) != NULL);
        ASSERT(GetDlgItem(IDC_ROL) != NULL);
        ASSERT(GetDlgItem(IDC_ROR) != NULL);
        ASSERT(GetDlgItem(IDC_XOR) != NULL);
        ASSERT(GetDlgItem(IDC_MOD) != NULL);
        ASSERT(GetDlgItem(IDC_MULTIPLY) != NULL);
        ASSERT(GetDlgItem(IDC_OR) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_DEC) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_FACTORIAL) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_FLIP) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_REV) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_INC) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_NOT) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_SIGN) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_SQUARE) != NULL);
        ASSERT(GetDlgItem(IDC_SUBTRACT) != NULL);
        ASSERT(GetDlgItem(IDC_GO) != NULL);
        ASSERT(GetDlgItem(IDC_MEM_GET) != NULL);
        ASSERT(GetDlgItem(IDC_MEM_CLEAR) != NULL);
        ASSERT(GetDlgItem(IDC_MEM_ADD) != NULL);
        ASSERT(GetDlgItem(IDC_MEM_SUBTRACT) != NULL);
        ASSERT(GetDlgItem(IDC_MARK_GET) != NULL);
        ASSERT(GetDlgItem(IDC_MARK_AT) != NULL);
        ASSERT(GetDlgItem(IDC_MARK_ADD) != NULL);
        ASSERT(GetDlgItem(IDC_MARK_SUBTRACT) != NULL);
        ASSERT(GetDlgItem(IDC_SEL_GET) != NULL);
        ASSERT(GetDlgItem(IDC_SEL_AT) != NULL);
        ASSERT(GetDlgItem(IDC_EOF_GET) != NULL);
        ASSERT(GetDlgItem(IDC_GTR) != NULL);
        ASSERT(GetDlgItem(IDC_LESS) != NULL);
        ASSERT(GetDlgItem(IDC_POW) != NULL);
        ASSERT(GetDlgItem(IDC_MARK_AT_STORE) != NULL);
        ASSERT(GetDlgItem(IDC_MARK_CLEAR) != NULL);
        ASSERT(GetDlgItem(IDC_MARK_STORE) != NULL);
        ASSERT(GetDlgItem(IDC_MEM_STORE) != NULL);
        ASSERT(GetDlgItem(IDC_SEL_AT_STORE) != NULL);
        ASSERT(GetDlgItem(IDC_SEL_STORE) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_ROL) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_ROR) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_LSL) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_LSR) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_ASR) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_CUBE) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_AT) != NULL);
        ASSERT(GetDlgItem(IDC_UNARY_SQUARE_ROOT) != NULL);
        ASSERT(GetDlgItem(IDC_8BIT) != NULL);
        ASSERT(GetDlgItem(IDC_16BIT) != NULL);
        ASSERT(GetDlgItem(IDC_32BIT) != NULL);
        ASSERT(GetDlgItem(IDC_64BIT) != NULL);
        ASSERT(GetDlgItem(IDC_BINARY) != NULL);
        ASSERT(GetDlgItem(IDC_OCTAL) != NULL);
        ASSERT(GetDlgItem(IDC_DECIMAL) != NULL);
        ASSERT(GetDlgItem(IDC_HEX) != NULL);
        ASSERT(GetDlgItem(IDC_SEL_LEN) != NULL);
        ASSERT(GetDlgItem(IDC_SEL_LEN_STORE) != NULL);
        ASSERT(GetDlgItem(IDC_ADDOP) != NULL);
        ASSERT(GetDlgItem(IDC_BIG_ENDIAN_FILE_ACCESS) != NULL);

        ttc_.AddTool(GetDlgItem(IDC_BACKSPACE), "Delete Last Digit");
        ttc_.AddTool(GetDlgItem(IDC_CLEAR_ENTRY), "Clear Entry");
        ttc_.AddTool(GetDlgItem(IDC_CLEAR), "Clear All");
        ttc_.AddTool(GetDlgItem(IDC_EQUALS), "Calculate Result (=)");
        ttc_.AddTool(GetDlgItem(IDC_AND), "AND (&&)");
        ttc_.AddTool(GetDlgItem(IDC_ASR), "Arithmetic Shift Right");
        ttc_.AddTool(GetDlgItem(IDC_DIVIDE), "Integer Divide (/)");
        ttc_.AddTool(GetDlgItem(IDC_LSL), "Logical Shift Left");
        ttc_.AddTool(GetDlgItem(IDC_LSR), "Logical Shift Right");
        ttc_.AddTool(GetDlgItem(IDC_ROL), "Rotate Left");
        ttc_.AddTool(GetDlgItem(IDC_ROR), "Rotate Right");
        ttc_.AddTool(GetDlgItem(IDC_XOR), "Exclusive Or (^)");
        ttc_.AddTool(GetDlgItem(IDC_MOD), "Remainder (%)");
        ttc_.AddTool(GetDlgItem(IDC_MULTIPLY), "Multiply (*)");
        ttc_.AddTool(GetDlgItem(IDC_OR), "OR (|)");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_DEC), "Subtract One");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_FACTORIAL), "Factorial (!)");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_FLIP), "Reverse Byte Order");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_REV), "Reverse Bit Order");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_INC), "Add One");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_NOT), "NOT (~)");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_SIGN), "Change Sign");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_SQUARE), "Square");
        ttc_.AddTool(GetDlgItem(IDC_SUBTRACT), "Subtract (-)");
        ttc_.AddTool(GetDlgItem(IDC_GO), "Jump To Resultant Address");
        ttc_.AddTool(GetDlgItem(IDC_MEM_GET), "Memory Recall (Ctrl+R)");
        ttc_.AddTool(GetDlgItem(IDC_MEM_CLEAR), "Memory Clear (Ctrl+L)");
        ttc_.AddTool(GetDlgItem(IDC_MEM_ADD), "Memory Add (Ctrl+P)");
        ttc_.AddTool(GetDlgItem(IDC_MEM_SUBTRACT), "Memory Subtract");
        ttc_.AddTool(GetDlgItem(IDC_MARK_GET), "Address of Mark");
        ttc_.AddTool(GetDlgItem(IDC_MARK_AT), "Get the Value at the Mark");
        ttc_.AddTool(GetDlgItem(IDC_MARK_ADD), "Add to Mark");
        ttc_.AddTool(GetDlgItem(IDC_MARK_SUBTRACT), "Subtract from Mark");
        ttc_.AddTool(GetDlgItem(IDC_SEL_GET), "Cursor Location");
        ttc_.AddTool(GetDlgItem(IDC_SEL_AT), "Get the Value at the Cursor");
        ttc_.AddTool(GetDlgItem(IDC_EOF_GET), "Get the Length of File");
        ttc_.AddTool(GetDlgItem(IDC_GTR), "Greater of Two Values");
        ttc_.AddTool(GetDlgItem(IDC_LESS), "Smaller of Two Values");
        ttc_.AddTool(GetDlgItem(IDC_POW), "Power");
        ttc_.AddTool(GetDlgItem(IDC_MARK_AT_STORE), "Store at Mark");
        ttc_.AddTool(GetDlgItem(IDC_MARK_CLEAR), "Set the Mark to Start of File");
        ttc_.AddTool(GetDlgItem(IDC_MARK_STORE), "Move the Mark");
        ttc_.AddTool(GetDlgItem(IDC_MEM_STORE), "Memory Store (Ctrl+M)");
        ttc_.AddTool(GetDlgItem(IDC_SEL_AT_STORE), "Store to File at Cursor");
        ttc_.AddTool(GetDlgItem(IDC_SEL_STORE), "Move the Cursor");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_ROL), "Rotate Left by One Bit");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_ROR), "Rotate Right by One Bit");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_LSL), "Logical Shift Left by One Bit");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_LSR), "Logical Shift Right by One Bit");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_ASR), "Arithmetic Shift Right by One Bit");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_CUBE), "Cube");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_AT), "Retrieve the Value from File");
        ttc_.AddTool(GetDlgItem(IDC_UNARY_SQUARE_ROOT), "Square Root");
        ttc_.AddTool(GetDlgItem(IDC_8BIT), "Use Bytes (F12)");
        ttc_.AddTool(GetDlgItem(IDC_16BIT), "Use Words (F11)");
        ttc_.AddTool(GetDlgItem(IDC_32BIT), "Use Double Words (F10)");
        ttc_.AddTool(GetDlgItem(IDC_64BIT), "Use Quad Words (F9)");
        ttc_.AddTool(GetDlgItem(IDC_BINARY), "Use Binary Numbers (F8)");
        ttc_.AddTool(GetDlgItem(IDC_OCTAL), "USe Octal Numbers (F7)");
        ttc_.AddTool(GetDlgItem(IDC_DECIMAL), "Use (Signed) Decimal Numbers (F6)");
        ttc_.AddTool(GetDlgItem(IDC_HEX), "Use Hexadecimal Numbers (F5)");
        ttc_.AddTool(GetDlgItem(IDC_SEL_LEN), "Get the Length of the Selection");
        ttc_.AddTool(GetDlgItem(IDC_SEL_LEN_STORE), "Change the Length of the Selection");
        ttc_.AddTool(GetDlgItem(IDC_ADDOP), "Add");
        ttc_.AddTool(GetDlgItem(IDC_BIG_ENDIAN_FILE_ACCESS), "Read/Write to File in Big-Endian Format");
        ttc_.Activate(TRUE);
    }
    return TRUE;
}

void CCalcDlg::OnClose() 
{
    ASSERT(visible_);                   // Should only be called if dlg is visible
    ShowWindow(SW_HIDE);
    visible_ = FALSE;
    aa_->SaveToMacro(km_calc_close);
}

void CCalcDlg::OnDestroy() 
{
    CDialog::OnDestroy();

    // Save some settings to the in file/registry
    aa_->WriteProfileInt("Calculator", "Base", radix_);
    aa_->WriteProfileInt("Calculator", "Bits", bits_);

    char buf[30];
    _i64toa(memory_, buf, 10);
    aa_->WriteProfileString("Calculator", "Memory", buf);
    calc_previous();
    _i64toa((current_&mask_), buf, 10);
    aa_->WriteProfileString("Calculator", "Current", buf);

    CRect rr;
    GetWindowRect(&rr);
    aa_->calc_x_ = rr.left;
    aa_->calc_y_ = rr.top;
}

BOOL CCalcDlg::PreTranslateMessage(MSG* pMsg) 
{
    ttc_.RelayEvent(pMsg);

    if (pMsg->message == WM_SYSKEYDOWN && pMsg->wParam == VK_F10)
    {
        // Handle F10 press
        On32bit();
        return TRUE;
    }
    else if (pMsg->message == WM_CHAR)
    {
        static const char *bo_char = "+-*/%&|^";
        static binop_type bo_op[] =
        {
            binop_add,
            binop_subtract,
            binop_multiply,
            binop_divide,
            binop_mod,
            binop_and,
            binop_or,
            binop_xor,
        };
        const char *pbo;                    // Pointer into bo_char string

        if ((pbo = strchr(bo_char, pMsg->wParam)) != NULL)
        {
            // Binary operator character typed
            ASSERT(pbo - bo_char < sizeof(bo_op)/sizeof(*bo_op)); // Bounds check
            do_binop(bo_op[pbo - bo_char]);
            return TRUE;
        }
        else if (pMsg->wParam == '!')
        {
            do_unary(unary_factorial);
            return TRUE;
        }
        else if (pMsg->wParam == '~')
        {
            do_unary(unary_not);
            return TRUE;
        }
        else if (pMsg->wParam == '\x0C')
        {
            OnMemClear();  // ^L
            return TRUE;
        }
        else if (pMsg->wParam == '\x12')
        {
            OnMemGet();   // ^R
            return TRUE;
        }
        else if (pMsg->wParam == '\x0D')
        {
            OnMemStore(); // ^M
            return TRUE;
        }
        else if (pMsg->wParam == '\x10')
        {
            OnMemAdd();  // ^P
            return TRUE;
        }
        else if (pMsg->wParam == '=')
        {
            OnEquals();
            return TRUE;
        }
    }
    else if (pMsg->message == WM_KEYDOWN)
    {
        if (pMsg->wParam > VK_F1 && pMsg->wParam <= VK_F12)
        {
            switch(pMsg->wParam)
            {
            case VK_F5:
                OnHex();
                break;
            case VK_F6:
                OnDecimal();
                break;
            case VK_F7:
                OnOctal();
                break;
            case VK_F8:
                OnBinary();
                break;
            case VK_F9:
                On64bit();
                break;
// F10 is a system key (see above)
//            case VK_F10:
//                On32bit();
//                break;
            case VK_F11:
                On16bit();
                break;
            case VK_F12:
                On8bit();
                break;
           }
           return TRUE;
        }
    }

    return CDialog::PreTranslateMessage(pMsg);
}

BOOL CCalcDlg::OnHelpInfo(HELPINFO* pHelpInfo) 
{
    static DWORD id_pairs[] = { 
        IDC_EDIT, HIDC_EDIT,
        IDC_OP_DISPLAY, HIDC_OP_DISPLAY,
        IDC_BIG_ENDIAN_FILE_ACCESS, HIDC_BIG_ENDIAN_FILE_ACCESS,
        IDC_HEX, HIDC_HEX,
        IDC_DECIMAL, HIDC_DECIMAL,
        IDC_OCTAL, HIDC_OCTAL,
        IDC_BINARY, HIDC_BINARY,
        IDC_64BIT, HIDC_64BIT,
        IDC_32BIT, HIDC_32BIT,
        IDC_16BIT, HIDC_16BIT,
        IDC_8BIT, HIDC_8BIT,
        IDC_MEM_GET, HIDC_MEM_GET,
        IDC_MEM_STORE, HIDC_MEM_STORE,
        IDC_MEM_CLEAR, HIDC_MEM_CLEAR,
        IDC_MEM_ADD, HIDC_MEM_ADD,
        IDC_MEM_SUBTRACT, HIDC_MEM_SUBTRACT,
        IDC_BACKSPACE, HIDC_BACKSPACE,
        IDC_CLEAR_ENTRY, HIDC_CLEAR_ENTRY,
        IDC_CLEAR, HIDC_CLEAR,
        IDC_MARK_GET, HIDC_MARK_GET,
        IDC_MARK_STORE, HIDC_MARK_STORE,
        IDC_MARK_CLEAR, HIDC_MARK_CLEAR,
        IDC_MARK_ADD, HIDC_MARK_ADD,
        IDC_MARK_SUBTRACT, HIDC_MARK_SUBTRACT,
        IDC_MARK_AT, HIDC_MARK_AT,
        IDC_MARK_AT_STORE, HIDC_MARK_AT_STORE,
        IDC_SEL_GET, HIDC_SEL_GET,
        IDC_SEL_STORE, HIDC_SEL_STORE,
        IDC_SEL_AT, HIDC_SEL_AT,
        IDC_SEL_AT_STORE, HIDC_SEL_AT_STORE,
        IDC_SEL_LEN, HIDC_SEL_LEN,
        IDC_SEL_LEN_STORE, HIDC_SEL_LEN_STORE,
        IDC_EOF_GET, HIDC_EOF_GET,
        IDC_UNARY_SQUARE, HIDC_UNARY_SQUARE,
        IDC_UNARY_SQUARE_ROOT, HIDC_UNARY_SQUARE_ROOT,
        IDC_UNARY_CUBE, HIDC_UNARY_CUBE,
        IDC_UNARY_FACTORIAL, HIDC_UNARY_FACTORIAL,
        IDC_UNARY_AT, HIDC_UNARY_AT,
        IDC_UNARY_ROL, HIDC_UNARY_ROL,
        IDC_UNARY_ROR, HIDC_UNARY_ROR,
        IDC_UNARY_LSL, HIDC_UNARY_LSL,
        IDC_UNARY_LSR, HIDC_UNARY_LSR,
        IDC_UNARY_ASR, HIDC_UNARY_ASR,
        IDC_UNARY_REV, HIDC_UNARY_REV,
        IDC_UNARY_NOT, HIDC_UNARY_NOT,
        IDC_UNARY_INC, HIDC_UNARY_INC,
        IDC_UNARY_DEC, HIDC_UNARY_DEC,
        IDC_UNARY_FLIP, HIDC_UNARY_FLIP,
        IDC_UNARY_SIGN, HIDC_UNARY_SIGN,
        IDC_EQUALS, HIDC_EQUALS,
        IDC_POW, HIDC_POW,
        IDC_MOD, HIDC_MOD,
        IDC_DIVIDE, HIDC_DIVIDE,
        IDC_MULTIPLY, HIDC_MULTIPLY,
        IDC_SUBTRACT, HIDC_SUBTRACT,
        IDC_ADDOP, HIDC_ADDOP,
        IDC_GTR, HIDC_GTR,
        IDC_LESS, HIDC_LESS,
        IDC_AND, HIDC_AND,
        IDC_OR, HIDC_OR,
        IDC_XOR, HIDC_XOR,
        IDC_ROL, HIDC_ROL,
        IDC_ROR, HIDC_ROR,
        IDC_LSL, HIDC_LSL,
        IDC_LSR, HIDC_LSR,
        IDC_ASR, HIDC_ASR,
        IDC_GO, HIDC_GO,
        IDC_DIGIT_0, HIDC_DIGIT_0,
        IDC_DIGIT_1, HIDC_DIGIT_0,
        IDC_DIGIT_2, HIDC_DIGIT_0,
        IDC_DIGIT_3, HIDC_DIGIT_0,
        IDC_DIGIT_4, HIDC_DIGIT_0,
        IDC_DIGIT_5, HIDC_DIGIT_0,
        IDC_DIGIT_6, HIDC_DIGIT_0,
        IDC_DIGIT_7, HIDC_DIGIT_0,
        IDC_DIGIT_8, HIDC_DIGIT_0,
        IDC_DIGIT_9, HIDC_DIGIT_0,
        IDC_DIGIT_A, HIDC_DIGIT_0,
        IDC_DIGIT_B, HIDC_DIGIT_0,
        IDC_DIGIT_C, HIDC_DIGIT_0,
        IDC_DIGIT_D, HIDC_DIGIT_0,
        IDC_DIGIT_E, HIDC_DIGIT_0,
        IDC_DIGIT_F, HIDC_DIGIT_0,
        0,0 
    }; 
 
    CWinApp* pApp = AfxGetApp();
    ASSERT_VALID(pApp);
    ASSERT(pApp->m_pszHelpFilePath != NULL);

    CWaitCursor wait;

    if (!::WinHelp((HWND)pHelpInfo->hItemHandle, pApp->m_pszHelpFilePath, 
                   HELP_WM_HELP, (DWORD) (LPSTR) id_pairs))
        ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return TRUE;
}

#define IDC_FIRST_BLUE   IDC_DIGIT_0
#define IDC_FIRST_RED    IDC_BACKSPACE
#define IDC_FIRST_BLACK  IDC_MARK_STORE
#define IDC_FIRST_PURPLE IDC_POW

void CCalcDlg::OnDrawItem(int nIDCtl, LPDRAWITEMSTRUCT lpDrawItemStruct) 
{
    ASSERT(visible_);                   // Should only be called if dlg is visible
    ASSERT(lpDrawItemStruct->CtlType == ODT_BUTTON);

    CDC *pDC = CDC::FromHandle(lpDrawItemStruct->hDC);
    COLORREF cr = pDC->GetTextColor();          // Save text colour to restore later

    if (lpDrawItemStruct->itemState & ODS_DISABLED)
        pDC->SetTextColor(::GetSysColor(COLOR_GRAYTEXT));
    else if (nIDCtl >= IDC_FIRST_BLUE && nIDCtl < IDC_FIRST_RED)
        pDC->SetTextColor(RGB(0, 0, 0xC0));
    else if (nIDCtl >= IDC_FIRST_RED && nIDCtl < IDC_FIRST_BLACK)
        pDC->SetTextColor(RGB(0xC0, 0, 0));
    else if (nIDCtl >= IDC_FIRST_BLACK && nIDCtl < IDC_FIRST_PURPLE)
        pDC->SetTextColor(RGB(0x0, 0, 0));
    else if (nIDCtl >= IDC_FIRST_PURPLE)
        pDC->SetTextColor(RGB(0x80, 0, 0x80));

    CRect rcButton(lpDrawItemStruct->rcItem);
    rcButton.InflateRect(-2, -2);
    if ((lpDrawItemStruct->itemState & ODS_SELECTED) == 0)
    {
        ::Draw3DFrame(pDC, rcButton, GetSysColor(COLOR_BTNSHADOW), GetSysColor(COLOR_BTNHIGHLIGHT));
        rcButton.InflateRect(-1, -1);
        ::Draw3DFrame(pDC, rcButton, GetSysColor(COLOR_BTNSHADOW), GetSysColor(COLOR_BTNHIGHLIGHT));
        rcButton.InflateRect(-1, -1);
    }

//    CBrush *pBrush = new CBrush(GetSysColor(COLOR_BTNFACE));
//    pDC->FillRect(&rcButton, pBrush);
//    delete pBrush;
    pDC->FillSolidRect(&rcButton, GetSysColor(COLOR_BTNFACE));

    if (nIDCtl == IDC_UNARY_SQUARE)
    {
        RECT rct = lpDrawItemStruct->rcItem;
        rct.right -= (rct.right - rct.left)/5;
        rct.top += (rct.bottom - rct.top)/10;
        pDC->DrawText("n", &rct,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);
        rct = lpDrawItemStruct->rcItem;
        rct.left += (rct.right - rct.left)/5;
        rct.bottom -= (rct.bottom - rct.top)/5;
        pDC->DrawText("2", &rct,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);
    }
    else if (nIDCtl == IDC_UNARY_SQUARE_ROOT)
    {
        RECT rct = lpDrawItemStruct->rcItem;
        rct.left += (rct.right - rct.left)/5;
        rct.top += (rct.bottom - rct.top)/10;
        pDC->DrawText("n", &rct,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);

        // Now draw the square root symbol
        CPen *ppen = pDC->SelectObject(&purple_pen);
        rct = lpDrawItemStruct->rcItem;
        pDC->MoveTo(rct.left + 3*(rct.right - rct.left)/10, rct.top + (rct.bottom - rct.top)/2);
        pDC->LineTo(rct.left + (rct.right - rct.left)/3, rct.top + 2*(rct.bottom - rct.top)/3);
        pDC->LineTo(rct.left + (rct.right - rct.left)/2, rct.top + (rct.bottom - rct.top)/3);
        pDC->LineTo(rct.left + 3*(rct.right - rct.left)/4, rct.top + (rct.bottom - rct.top)/3);
        pDC->SelectObject(ppen);
    }
    else if (nIDCtl == IDC_UNARY_CUBE)
    {
        RECT rct = lpDrawItemStruct->rcItem;
        rct.right -= (rct.right - rct.left)/5;
        rct.top += (rct.bottom - rct.top)/10;
        pDC->DrawText("n", &rct,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);
        rct = lpDrawItemStruct->rcItem;
        rct.left += (rct.right - rct.left)/5;
        rct.bottom -= (rct.bottom - rct.top)/5;
        pDC->DrawText("3", &rct,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);
    }
    else if (nIDCtl == IDC_POW)
    {
        RECT rct = lpDrawItemStruct->rcItem;
        rct.right -= (rct.right - rct.left)/4;
        pDC->DrawText("m", &rct,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);
        rct = lpDrawItemStruct->rcItem;
        rct.left += (rct.right - rct.left)/4;
        rct.bottom -= (rct.bottom - rct.top)/4;
        pDC->DrawText("n", &rct,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);
    }
    else
    {
        CString str;
        GetDlgItem(nIDCtl)->GetWindowText(str);
        pDC->DrawText(str, &lpDrawItemStruct->rcItem,
                      DT_SINGLELINE | DT_CENTER | DT_VCENTER | DT_NOPREFIX);
    }

    TRACE3("Draw button %ld, action %d, state %d\n",
           long(nIDCtl), 
           lpDrawItemStruct->itemAction, 
           lpDrawItemStruct->itemState);

    rcButton = lpDrawItemStruct->rcItem;
    ::Draw3DButtonFrame(pDC, rcButton,
                        (lpDrawItemStruct->itemState & ODS_FOCUS) != 0);

    // Restore text colour
    ::SetTextColor(lpDrawItemStruct->hDC, cr);
    CDialog::OnDrawItem(nIDCtl, lpDrawItemStruct);
}

// This actually goes to a calculated address.  Unlike the "Store Cursor" button,
// the result is calculated (saves pressing "=" button) and the view is given focus
// after the current cursor position has been moved.
void CCalcDlg::OnGo()                   // Move cursor to current value
{
    unsigned __int64 temp = current_;
    calc_previous();

    // If the value in current_ is not there as a result of a calculation then
    // save it before it is lost.
    if (source_ != km_result)
        aa_->SaveToMacro(source_, temp);

    op_ = binop_none;

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
        if (!overflow_ && !error_)
            ShowBinop(binop_none);
    }
    in_edit_ = FALSE;
    source_ = aa_->recording_ ? km_user : km_result;

    // Make sure we can go to the address
    CHexEditView *pview = GetView();
    if (pview == NULL || (current_&mask_) > pview->GetDocument()->length())
    {
        mm_->StatusBarText("Error: new cursor address past end of file");
        aa_->mac_error_ = 10;
        return;
    }

    pview->MoveToAddress(current_);

    // Give view the focus
    if (pview != pview->GetFocus())
        pview->SetFocus();

    aa_->SaveToMacro(km_go);
}

void CCalcDlg::OnBigEndian() 
{
    UpdateData();
    aa_->SaveToMacro(km_endian);
}

void CCalcDlg::On8bit() 
{
    change_bits(8);
}

void CCalcDlg::On16bit() 
{
    change_bits(16);
}

void CCalcDlg::On32bit() 
{
    change_bits(32);
}

void CCalcDlg::On64bit() 
{
    change_bits(64);
}

void CCalcDlg::OnBinary() 
{
    change_base(2);
}

void CCalcDlg::OnOctal() 
{
    change_base(8);
}

void CCalcDlg::OnDecimal() 
{
    change_base(10);
}

void CCalcDlg::OnHex() 
{
    change_base(16);
}

void CCalcDlg::OnBackspace()            // Delete back one digit
{
    ASSERT(visible_);
    edit_.SetFocus();

    edit_.SendMessage(WM_CHAR, '\b', 1);
    in_edit_ = TRUE;                    // Allow edit of current value
    source_ = aa_->recording_ ? km_user : km_result;
}

void CCalcDlg::OnClearEntry()           // Zero current value
{
    current_ = 0;

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    in_edit_ = FALSE;                   // Necessary?
    source_ = aa_->recording_ ? km_user : km_result;
    aa_->SaveToMacro(km_clear_entry);
}

void CCalcDlg::OnClear()                // Zero current and remove any operators/brackets
{
    op_ = binop_none;
    current_ = 0;

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
        ShowBinop(binop_none);
    }
    
    in_edit_ = FALSE;
    source_ = aa_->recording_ ? km_user : km_result;
    aa_->SaveToMacro(km_clear);
}

void CCalcDlg::OnEquals()               // Calculate result
{
    unsigned __int64 temp = current_;
    calc_previous();

    // If the value in current_ is not there as a result of a calculation then
    // save it before it is lost.
    if (source_ != km_result)
        aa_->SaveToMacro(source_, temp);

    op_ = binop_none;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
        if (!overflow_ && !error_)
            ShowBinop(binop_none);
    }

    in_edit_ = FALSE;
    source_ = aa_->recording_ ? km_user : km_result;
    aa_->SaveToMacro(km_equals);
}

// ---------- Digits for entering numbers ----------
void CCalcDlg::OnDigit0() 
{
    do_digit('0');
}

void CCalcDlg::OnDigit1() 
{
    do_digit('1');
}

void CCalcDlg::OnDigit2() 
{
    do_digit('2');
}

void CCalcDlg::OnDigit3() 
{
    do_digit('3');
}

void CCalcDlg::OnDigit4() 
{
    do_digit('4');
}

void CCalcDlg::OnDigit5() 
{
    do_digit('5');
}

void CCalcDlg::OnDigit6() 
{
    do_digit('6');
}

void CCalcDlg::OnDigit7() 
{
    do_digit('7');
}

void CCalcDlg::OnDigit8() 
{
    do_digit('8');
}

void CCalcDlg::OnDigit9() 
{
    do_digit('9');
}

void CCalcDlg::OnDigitA() 
{
    do_digit('A');
}

void CCalcDlg::OnDigitB() 
{
    do_digit('B');
}

void CCalcDlg::OnDigitC() 
{
    do_digit('C');
}

void CCalcDlg::OnDigitD() 
{
    do_digit('D');
}

void CCalcDlg::OnDigitE() 
{
    do_digit('E');
}

void CCalcDlg::OnDigitF() 
{
    do_digit('F');
}

// ----- Binary operators ----------
void CCalcDlg::OnAdd() 
{
    do_binop(binop_add);
}

void CCalcDlg::OnSubtract() 
{
    do_binop(binop_subtract);
}

void CCalcDlg::OnMultiply() 
{
    do_binop(binop_multiply);
}

void CCalcDlg::OnDivide() 
{
    do_binop(binop_divide);
}

void CCalcDlg::OnMod() 
{
    do_binop(binop_mod);
}

void CCalcDlg::OnPow() 
{
    do_binop(binop_pow);
}

void CCalcDlg::OnAnd() 
{
    do_binop(binop_and);
}

void CCalcDlg::OnOr() 
{
    do_binop(binop_or);
}

void CCalcDlg::OnXor() 
{
    do_binop(binop_xor);
}

void CCalcDlg::OnLsl() 
{
    do_binop(binop_lsl);
}

void CCalcDlg::OnLsr() 
{
    do_binop(binop_lsr);
}

void CCalcDlg::OnAsr() 
{
    do_binop(binop_asr);
}

void CCalcDlg::OnRol() 
{
    do_binop(binop_rol);
}

void CCalcDlg::OnRor() 
{
    do_binop(binop_ror);
}

void CCalcDlg::OnGtr()                  // Returns the larger of it's 2 operands
{
    do_binop(binop_gtr);
}

void CCalcDlg::OnLess()                 // Returns the smaller of it's 2 operands
{
    do_binop(binop_less);
}

// --------- Unary operators -----------

void CCalcDlg::OnUnaryAt() 
{
    do_unary(unary_at);
}

void CCalcDlg::OnUnaryRol() 
{
    do_unary(unary_rol);
}

void CCalcDlg::OnUnaryRor() 
{
    do_unary(unary_ror);
}

void CCalcDlg::OnUnaryLsl() 
{
    do_unary(unary_lsl);
}

void CCalcDlg::OnUnaryLsr() 
{
    do_unary(unary_lsr);
}

void CCalcDlg::OnUnaryAsr() 
{
    do_unary(unary_asr);
}

void CCalcDlg::OnUnaryNot() 
{
    do_unary(unary_not);
}

void CCalcDlg::OnUnaryFlip()            // Flip bytes
{
    do_unary(unary_flip);
}

void CCalcDlg::OnUnaryRev()             // Reverse bits
{
    do_unary(unary_rev);
}

void CCalcDlg::OnUnarySign() 
{
    do_unary(unary_sign);
}

void CCalcDlg::OnUnaryInc() 
{
    do_unary(unary_inc);
}

void CCalcDlg::OnUnaryDec() 
{
    do_unary(unary_dec);
}

void CCalcDlg::OnUnarySquare() 
{
    do_unary(unary_square);
}

void CCalcDlg::OnUnarySquareRoot() 
{
    do_unary(unary_squareroot);
}

void CCalcDlg::OnUnaryCube() 
{
    do_unary(unary_cube);
}

void CCalcDlg::OnUnaryFactorial() 
{
    do_unary(unary_factorial);
}

// ------- Calculator memory funcs ----------
void CCalcDlg::OnMemGet() 
{
    current_ = memory_;

    overflow_ = error_ = FALSE;
    if ((current_ & ~mask_) != 0)
    {
        overflow_ = TRUE;
        mm_->StatusBarText("Memory overflowed data size");
        aa_->mac_error_ = 2;
    }
    ShowStatus();                       // Clear/set overflow indicator

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    in_edit_ = TRUE;                    // The user can edit the value edit control
    source_ = aa_->recording_ ? km_memget : km_result;
}

void CCalcDlg::OnMemStore() 
{
    memory_ = current_&mask_;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        GetDlgItem(IDC_MEM_GET)->EnableWindow(TRUE);
    }

    aa_->SaveToMacro(km_memstore);
}

void CCalcDlg::OnMemClear() 
{
    memory_ = 0;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        GetDlgItem(IDC_MEM_GET)->EnableWindow(FALSE);
    }

    aa_->SaveToMacro(km_memclear);
}

void CCalcDlg::OnMemAdd() 
{
    memory_ += current_;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        GetDlgItem(IDC_MEM_GET)->EnableWindow(TRUE);
    }

    aa_->SaveToMacro(km_memadd);
}

void CCalcDlg::OnMemSubtract() 
{
    memory_ -= current_;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        GetDlgItem(IDC_MEM_GET)->EnableWindow(TRUE);
    }

    aa_->SaveToMacro(km_memsubtract);
}

// ----------- Mark functions --------------
void CCalcDlg::OnMarkGet()              // Position of mark in the file
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }

    current_ = pview->GetMark();

    overflow_ = error_ = FALSE;
    if ((current_ & ~mask_) != 0)
    {
        overflow_ = TRUE;
        mm_->StatusBarText("Mark overflowed data size");
        aa_->mac_error_ = 2;
    }
    ShowStatus();                       // Clear/set overflow indicator

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    in_edit_ = TRUE;                    // The user can edit the new value in edit control
    source_ = aa_->recording_ ? km_markget : km_result;
}

void CCalcDlg::OnMarkStore() 
{
    CHexEditView *pview = GetView();
    if (pview == NULL || (current_&mask_) > pview->GetDocument()->length())
    {
        mm_->StatusBarText("New mark address past end of file");
        aa_->mac_error_ = 10;
        return;
    }

    pview->SetMark(current_&mask_);
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        FixFileButtons();
    }

    aa_->SaveToMacro(km_markstore);
}

void CCalcDlg::OnMarkClear() 
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }

    pview->SetMark(0);
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        FixFileButtons();
    }

    aa_->SaveToMacro(km_markclear);
}

void CCalcDlg::OnMarkAdd()              // Add current value to mark
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }
    FILE_ADDRESS eof = pview->GetDocument()->length();
    FILE_ADDRESS mark = pview->GetMark();

    if (radix_ == 10)
    {
        signed __int64 new_mark;
        switch (bits_)
        {
        case 8:
            new_mark = mark + (signed char)current_;
            break;
        case 16:
            new_mark = mark + (signed short)current_;
            break;
        case 32:
            new_mark = mark + (signed long)current_;
            break;
        case 64:
            new_mark = mark + (signed __int64)current_;
            break;
        }
        if (new_mark < 0 || new_mark > eof)
        {
            if (new_mark < 0)
                mm_->StatusBarText("New mark address is -ve");
            else
                mm_->StatusBarText("New mark address past end of file");
            aa_->mac_error_ = 10;
            return;
        }
        pview->SetMark(new_mark);
    }
    else
    {
        if (mark + (current_&mask_) > eof)
        {
            mm_->StatusBarText("New mark address past end of file");
            aa_->mac_error_ = 10;
            return;
        }
        pview->SetMark(mark + (current_&mask_));
    }

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        FixFileButtons();
    }

    aa_->SaveToMacro(km_markadd);
}

void CCalcDlg::OnMarkSubtract() 
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }
    FILE_ADDRESS eof = pview->GetDocument()->length();
    FILE_ADDRESS mark = pview->GetMark();

    if (radix_ == 10)
    {
        signed __int64 new_mark;
        switch (bits_)
        {
        case 8:
            new_mark = mark - (signed char)current_;
            break;
        case 16:
            new_mark = mark - (signed short)current_;
            break;
        case 32:
            new_mark = mark - (signed long)current_;
            break;
        case 64:
            new_mark = mark - (signed __int64)current_;
            break;
        }
        if (new_mark < 0 || new_mark > eof)
        {
            if (new_mark < 0)
                mm_->StatusBarText("New mark address is -ve");
            else
                mm_->StatusBarText("New mark address past end of file");
            aa_->mac_error_ = 10;
            return;
        }
        pview->SetMark(new_mark);
    }
    else
    {
        if (mark - (current_&mask_) > eof)
        {
            mm_->StatusBarText("New mark address past end of file");
            aa_->mac_error_ = 10;
            return;
        }
        pview->SetMark(mark - (current_&mask_));
    }
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        FixFileButtons();
    }

    aa_->SaveToMacro(km_marksubtract);
}

// ----------- Other file get funcs -------------
void CCalcDlg::OnMarkAt()               // Get value from file at mark
{
    UpdateData();                       // Make sure we know what endianness to use

    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }
    FILE_ADDRESS eof = pview->GetDocument()->length();
    FILE_ADDRESS mark = pview->GetMark();
    if (mark + bits_/8 > eof)
    {
        mm_->StatusBarText("Mark address too close to end of file to get data");
        aa_->mac_error_ = 10;
        return;
    }

    __int64 temp = 0;
    pview->GetDocument()->GetData((unsigned char *)&temp, bits_/8, mark);
    if (big_endian_)
    {
        // Reverse the byte order to match that used internally (Intel=little-endian)
        unsigned char cc, *byte = (unsigned char *)&temp;
        switch (bits_)
        {
        case 8:
            /* nothing */
            break;
        case 16:
            cc = byte[0]; byte[0] = byte[1]; byte[1] = cc;
            break;
        case 32:
            cc = byte[0]; byte[0] = byte[3]; byte[3] = cc;
            cc = byte[1]; byte[1] = byte[2]; byte[2] = cc;
            break;
        case 64:
            cc = byte[0]; byte[0] = byte[7]; byte[7] = cc;
            cc = byte[1]; byte[1] = byte[6]; byte[6] = cc;
            cc = byte[2]; byte[2] = byte[5]; byte[5] = cc;
            cc = byte[3]; byte[3] = byte[4]; byte[4] = cc;
            break;
        default:
            ASSERT(0);
        }
    }
    current_ = temp;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }
    
    in_edit_ = TRUE;                    // The user can edit the new value in edit control
    source_ = aa_->recording_ ? km_markat : km_result;
}

void CCalcDlg::OnSelGet()              // Position of cursor (or start of selection)
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }
    FILE_ADDRESS start, end;
    pview->GetSelAddr(start, end);

    current_ = start;

    overflow_ = error_ = FALSE;
    if ((current_ & ~mask_) != 0)
    {
        overflow_ = TRUE;
        mm_->StatusBarText("Cursor position overflowed data size");
        aa_->mac_error_ = 2;
    }
    ShowStatus();                       // Clear/set overflow indicator

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    in_edit_ = TRUE;                    // The user can edit the new value in edit control
    source_ = aa_->recording_ ? km_selget : km_result;
}

void CCalcDlg::OnSelAt()                // Value in file at cursor
{
    UpdateData();                       // Make sure we know what endianness to use

    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }

    FILE_ADDRESS eof = pview->GetDocument()->length();
    FILE_ADDRESS start, end;
    pview->GetSelAddr(start, end);

    if (start + bits_/8 > eof)
    {
        mm_->StatusBarText("Cursor too close to end of file to get data");
        aa_->mac_error_ = 10;
        return;
    }

    __int64 temp = 0;
    pview->GetDocument()->GetData((unsigned char *)&temp, bits_/8, start);
    if (big_endian_)
    {
        // Reverse the byte order to match that used internally (Intel=little-endian)
        unsigned char cc, *byte = (unsigned char *)&temp;
        switch (bits_)
        {
        case 8:
            /* nothing */
            break;
        case 16:
            cc = byte[0]; byte[0] = byte[1]; byte[1] = cc;
            break;
        case 32:
            cc = byte[0]; byte[0] = byte[3]; byte[3] = cc;
            cc = byte[1]; byte[1] = byte[2]; byte[2] = cc;
            break;
        case 64:
            cc = byte[0]; byte[0] = byte[7]; byte[7] = cc;
            cc = byte[1]; byte[1] = byte[6]; byte[6] = cc;
            cc = byte[2]; byte[2] = byte[5]; byte[5] = cc;
            cc = byte[3]; byte[3] = byte[4]; byte[4] = cc;
            break;
        default:
            ASSERT(0);
        }
    }
    current_ = temp;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    in_edit_ = TRUE;                    // The user can edit the new value in edit control
    source_ = aa_->recording_ ? km_selat : km_result;
}

void CCalcDlg::OnSelLen() 
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }

    FILE_ADDRESS start, end;
    pview->GetSelAddr(start, end);

    current_ = end - start;
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    in_edit_ = TRUE;                    // The user can edit the new value in edit control
    source_ = aa_->recording_ ? km_sellen : km_result;
}

void CCalcDlg::OnEofGet()               // Length of file
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }

    current_ = pview->GetDocument()->length();
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        edit_.Put();
        FixFileButtons();
    }

    in_edit_ = TRUE;                    // The user can edit the new value in edit control
    source_ = aa_->recording_ ? km_eofget : km_result;
}

// ----------- File change funcs -------------

void CCalcDlg::OnMarkAtStore() 
{
    UpdateData();                       // Make sure we know what endianness to use

    CHexEditView *pview = GetView();
    if (pview == NULL || pview->ReadOnly())
    {
        mm_->StatusBarText("Can't write at mark: file is read only");
        aa_->mac_error_ = 10;
        return;
    }

    FILE_ADDRESS eof = pview->GetDocument()->length();
    FILE_ADDRESS mark = pview->GetMark();
    if (mark > eof || (mark + bits_/8 > eof && pview->OverType()))
    {
        mm_->StatusBarText("Can't write at mark: too close to EOF (OVR mode) or past EOF");
        aa_->mac_error_ = 10;
        return;
    }

    __int64 temp = (current_&mask_);
    if (big_endian_)
    {
        // Reverse the byte order to match that used internally (Intel=little-endian)
        unsigned char cc, *byte = (unsigned char *)&temp;
        switch (bits_)
        {
        case 8:
            /* nothing */
            break;
        case 16:
            cc = byte[0]; byte[0] = byte[1]; byte[1] = cc;
            break;
        case 32:
            cc = byte[0]; byte[0] = byte[3]; byte[3] = cc;
            cc = byte[1]; byte[1] = byte[2]; byte[2] = cc;
            break;
        case 64:
            cc = byte[0]; byte[0] = byte[7]; byte[7] = cc;
            cc = byte[1]; byte[1] = byte[6]; byte[6] = cc;
            cc = byte[2]; byte[2] = byte[5]; byte[5] = cc;
            cc = byte[3]; byte[3] = byte[4]; byte[4] = cc;
            break;
        default:
            ASSERT(0);
        }
    }
    pview->GetDocument()->Change(pview->OverType() ? mod_replace : mod_insert,
                                mark, bits_/8, (unsigned char *)&temp, 0, pview);

    if (!pview->OverType())
        pview->SetMark(mark);   // Inserting at mark would have move it forward

    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        FixFileButtons();
    }

    aa_->SaveToMacro(km_markatstore);
}

void CCalcDlg::OnSelStore() 
{
    CHexEditView *pview = GetView();
    if (pview == NULL || (current_&mask_) > pview->GetDocument()->length())
    {
        mm_->StatusBarText("Can't move cursor past end of file");
        aa_->mac_error_ = 10;
        return;
    }
    pview->MoveToAddress(current_);
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        // FixFileButtons(); // done indirectly through MoveToAddress
    }

    aa_->SaveToMacro(km_selstore);
}

void CCalcDlg::OnSelAtStore() 
{
    UpdateData();                       // Make sure we know what endianness to use

    CHexEditView *pview = GetView();
    if (pview == NULL || pview->ReadOnly())
    {
        mm_->StatusBarText("Can't store at cursor: file is read-only");
        aa_->mac_error_ = 10;
        return;
    }

    FILE_ADDRESS eof = pview->GetDocument()->length();
    FILE_ADDRESS start, end;
    pview->GetSelAddr(start, end);

    if (start > eof || (start + bits_/8 > eof && pview->OverType()))
    {
        mm_->StatusBarText("Can't store at cursor: too close to EOF (OVR mode) or past EOF");
        aa_->mac_error_ = 10;
        return;
    }

    __int64 temp = (current_&mask_);
    if (big_endian_)
    {
        // Reverse the byte order to match that used internally (Intel=little-endian)
        unsigned char cc, *byte = (unsigned char *)&temp;
        switch (bits_)
        {
        case 8:
            /* nothing */
            break;
        case 16:
            cc = byte[0]; byte[0] = byte[1]; byte[1] = cc;
            break;
        case 32:
            cc = byte[0]; byte[0] = byte[3]; byte[3] = cc;
            cc = byte[1]; byte[1] = byte[2]; byte[2] = cc;
            break;
        case 64:
            cc = byte[0]; byte[0] = byte[7]; byte[7] = cc;
            cc = byte[1]; byte[1] = byte[6]; byte[6] = cc;
            cc = byte[2]; byte[2] = byte[5]; byte[5] = cc;
            cc = byte[3]; byte[3] = byte[4]; byte[4] = cc;
            break;
        default:
            ASSERT(0);
        }
    }
    pview->GetDocument()->Change(pview->OverType() ? mod_replace : mod_insert,
                                start, bits_/8, (unsigned char *)&temp, 0, pview);
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        FixFileButtons();
    }

    aa_->SaveToMacro(km_selatstore);
}

void CCalcDlg::OnSelLenStore() 
{
    CHexEditView *pview = GetView();
    if (pview == NULL)
    {
        aa_->mac_error_ = 10;
        return;
    }

    FILE_ADDRESS eof = pview->GetDocument()->length();
    FILE_ADDRESS start, end;
    pview->GetSelAddr(start, end);
    if (start + (current_ & mask_) > eof)
    {
        mm_->StatusBarText("New selection length would be past EOF");
        aa_->mac_error_ = 10;
        return;
    }

    pview->MoveToAddress(start, start + current_);
    if (!aa_->refresh_off_ && visible_)
    {
        edit_.SetFocus();
        // FixFileButtons(); // done indirectly through MoveToAddress
    }

    aa_->SaveToMacro(km_sellenstore);
}

