//PasScript interpreter of the Object Pascal language, version 6.51
//Copyright (c) by Alexander Baranovsky, 1999-2002
//File 'fmMain.pas'

unit fmMain;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, StdCtrls, ComCtrls, ExtCtrls, Buttons, Printers, ShellAPI, passcr,

// use imp_gen.dpr to generate these units:

TPasScript_IMP,
TForm_IMP,
TPanel_IMP,
TSpeedButton_IMP,
TMainMenu_IMP,
TPopUpMenu_IMP,
TButton_IMP,
TLabel_IMP,
TMenuItem_IMP,
TOpenDialog_IMP,
TSaveDialog_IMP,
TPrintDialog_IMP,
TMemo_IMP,
TStringList_IMP,
TListBox_IMP,
TTimer_IMP,
TCanvas_IMP;

const
  Version = '6.51';

type
  TForm1 = class(TForm)
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Script1: TMenuItem;
    Help1: TMenuItem;
    New1: TMenuItem;
    N1: TMenuItem;
    Open1: TMenuItem;
    N2: TMenuItem;
    Save1: TMenuItem;
    SaveAs1: TMenuItem;
    N3: TMenuItem;
    Exit1: TMenuItem;
    Compile1: TMenuItem;
    Run1: TMenuItem;
    VIRTLaboratoryHomePage1: TMenuItem;
    N5: TMenuItem;
    About1: TMenuItem;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    PanelTop: TPanel;
    PanelBottom: TPanel;
    PanelLeft: TPanel;
    PanelRight: TPanel;
    LabelLeft: TLabel;
    LabelRight: TLabel;
    SpeedButtonNew: TSpeedButton;
    SpeedButtonOpen: TSpeedButton;
    SpeedButtonSave: TSpeedButton;
    SpeedButtonRun: TSpeedButton;
    Print1: TMenuItem;
    N4: TMenuItem;
    PrintDialog1: TPrintDialog;
    OpenDemo1: TMenuItem;
    OLEAutomation1: TMenuItem;
    AIapplications1: TMenuItem;
    Characterdifferentiation1: TMenuItem;
    Theoremproving1: TMenuItem;
    LeeAlgorithm1: TMenuItem;
    OpenArrays1: TMenuItem;
    PasScriptmanual1: TMenuItem;
    N6: TMenuItem;
    PasScriptHomePage1: TMenuItem;
    WhatsNewSetTypes1: TMenuItem;
    ButtonSuspend: TButton;
    ButtonResume: TButton;
    ButtonTerminate: TButton;
    PanelHelp: TPanel;
    SpeedButtonHelp: TSpeedButton;
    EventHandlers1: TMenuItem;
    Classreferencetypes1: TMenuItem;
    Memo1: TMemo;
    DefaultArrayProperties1: TMenuItem;
    Polymorphicbinaytrees1: TMenuItem;
    imp_dfm: TMenuItem;
    Typedexceptions1: TMenuItem;
    N7: TMenuItem;
    Tetrisgame1: TMenuItem;
    N8: TMenuItem;
    N9: TMenuItem;
    ImportDfmfile1: TMenuItem;
    WhatsNew: TMenuItem;
    procedure New1Click(Sender: TObject);
    procedure Open1Click(Sender: TObject);
    procedure SaveAs1Click(Sender: TObject);
    procedure Save1Click(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Compile1Click(Sender: TObject);
    procedure Run1Click(Sender: TObject);
    procedure SpeedButtonNewClick(Sender: TObject);
    procedure SpeedButtonOpenClick(Sender: TObject);
    procedure SpeedButtonSaveClick(Sender: TObject);
    procedure SpeedButtonRunClick(Sender: TObject);
    procedure Memo1KeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormShow(Sender: TObject);
    procedure Memo1MouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Memo1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Print1Click(Sender: TObject);
    procedure VIRTLaboratoryHomePage1Click(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure ClassesandObjects1Click(Sender: TObject);
    procedure OpenArrays1Click(Sender: TObject);
    procedure OLEAutomation1Click(Sender: TObject);
    procedure Characterdifferentiation1Click(Sender: TObject);
    procedure Theoremproving1Click(Sender: TObject);
    procedure LeeAlgorithm1Click(Sender: TObject);
    procedure PasScriptmanual1Click(Sender: TObject);
    procedure KnownDelphiTypesandRoutines1Click(Sender: TObject);
    procedure SourceCodeWhichEnsuresImport1Click(Sender: TObject);
    procedure ImportedDelphiclasses1Click(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure PasScriptHomePage1Click(Sender: TObject);
    procedure UsingTListclass1Click(Sender: TObject);
    procedure WhatsNewSetTypes1Click(Sender: TObject);
    procedure ButtonSuspendClick(Sender: TObject);
    procedure ButtonResumeClick(Sender: TObject);
    procedure ButtonTerminateClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure EventHandlers1Click(Sender: TObject);
    procedure Classreferencetypes1Click(Sender: TObject);
    procedure DefaultArrayProperties1Click(Sender: TObject);
    procedure Polymorphicbinaytrees1Click(Sender: TObject);
    procedure PasScriptCompileProgress(Sender: TObject);
    procedure imp_dfmClick(Sender: TObject);
    procedure Typedexceptions1Click(Sender: TObject);
    procedure Tetrisgame1Click(Sender: TObject);
    procedure ImportDfmfile1Click(Sender: TObject);
    procedure WhatsNewClick(Sender: TObject);
  private
    fCompileStage, fRunStage, WasClosed: boolean;
    fScriptName: string;
    CompileIsOK: boolean;
    CompileAndRun: boolean;
    fScriptAge: Integer;
    procedure SetCompileStage(Value: boolean);
    procedure SetRunStage(Value: boolean);
    procedure SetScriptName(const Value: string);
    function  GetBinaryProgName: String;
    procedure ShowError;
    procedure PasScriptCompileError(Sender: TObject);
    procedure PasScriptRunError(Sender: TObject);
    procedure PasScriptSaveBinaryProg(Sender: TObject);
    procedure PasScriptLoadBinaryProg(Sender: TObject);
    procedure OpenDemo(const FileName: String);
    procedure ShowText(const FileName, Title: String);
    { Private declarations }
  public
    PasScript: TPasScript;
    FF1, FF2: Integer;
    
    { Public declarations }
    property ScriptName: String read fScriptName write SetScriptName;
    property BinaryProgName: String read GetBinaryProgName;
    property ScriptAge: Integer read fScriptAge write fScriptAge;
    property CompileStage: boolean read fCompileStage write SetCompileStage;
    property RunStage: boolean read fRunStage write SetRunStage;
  end;

var
  Form1: TForm1;

implementation

uses fmCompiling, typinfo;

{$R *.DFM}

procedure ErrMessageBox(const Msg, Err:  String);
begin
  MessageBox( GetActiveWindow(), PChar(Msg), PChar(Err), MB_ICONEXCLAMATION or MB_OK );
end;

function StrEql(Const S1, S2: String): Boolean;
begin
  Result := CompareText(S1, S2) = 0;
end;

const
  PasScriptFilter = 'PasScript file (*.pas)|*.PAS|Any file (*.*)|*.*';
  DFMFilter = 'Exe files (*.exe)|*.exe|Dfm files (*.dfm)|*.dfm';

procedure TForm1.SetScriptName(const Value: string);
begin
   Caption := 'PasScript ' + Version;
   if Length(Value) > 0 then
     Caption := Caption + ' - ' + Value;
   fScriptName := Value;
end;

procedure TForm1.SetCompileStage(Value: boolean);
begin
  if Assigned(PasScript) then
    PasScript.OnCancelParsing := nil;

  fCompileStage := Value;
  if Value then
  begin
    if Assigned(Compiling) then
      Compiling.Button1.Caption := 'Cancel';
    LabelRight.Caption := 'Compiling ' + ScriptName + '...';
  end
  else
  begin
    LabelRight.Caption := '';
    if Assigned(Compiling) then
      Compiling.Button1.Caption := 'OK';
  end;
end;

procedure TForm1.SetRunStage(Value: boolean);
begin
  fRunStage := Value;
  if Value then
  begin
    LabelRight.Caption := 'Running ' + ScriptName + '...';
    ButtonSuspend.Enabled := true;
    ButtonSuspend.Visible := true;

    ButtonResume.Enabled := true;
    ButtonResume.Visible := true;

    ButtonTerminate.Enabled := true;
    ButtonTerminate.Visible := true;

    File1.Enabled := false;
    Script1.Enabled := false;
    Help1.Enabled := false;

    SpeedButtonRun.Enabled := false;
    SpeedButtonOpen.Enabled := false;
    SpeedButtonSave.Enabled := false;
    SpeedButtonNew.Enabled := false;
  end
  else
  begin
    ButtonSuspend.Enabled := false;
    ButtonSuspend.Visible := false;

    ButtonResume.Enabled := false;
    ButtonResume.Visible := false;

    ButtonTerminate.Enabled := false;
    ButtonTerminate.Visible := false;

    File1.Enabled := true;
    Script1.Enabled := true;
    Help1.Enabled := true;

    SpeedButtonRun.Enabled := true;
    SpeedButtonOpen.Enabled := true;
    SpeedButtonSave.Enabled := true;
    SpeedButtonNew.Enabled := true;

    LabelRight.Caption := '';
  end;
end;

function TForm1.GetBinaryProgName: String;
begin
  result := ChangeFileExt(ScriptName, '.PB');
end;

procedure TForm1.New1Click(Sender: TObject);
begin
  Memo1.Lines.Clear;
  ScriptName := '';
end;

procedure TForm1.ImportDfmfile1Click(Sender: TObject);
var
  I, DfmCount: Integer;
  S: String;
begin
  OpenDialog1.Filter := DFMFilter;
  with OpenDialog1 do
  if Execute then
  begin
    DfmCount := RegisterDFMs(FileName);
    if DfmCount = 0 then
      ErrMessageBox('No DFM RCDATA found!', 'Importing dfm')
    else
    begin
      if DfmCount = 1 then
        S := 'The following PasScript unit was generated:'#13#10
      else
        S := 'The following PasScript units were generated:'#13#10;
      for I:=1 to DfmCount do
        S := S + GetDfmUnitName(I) + #13#10;
      ShowMessage(S);  
    end;
  end;
  OpenDialog1.Filter := PasScriptFilter;
end;

procedure TForm1.Open1Click(Sender: TObject);
begin
  with OpenDialog1 do
  if Execute then
  begin
    ScriptName := FileName;
    Memo1.Lines.LoadFromFile(ScriptName);
    Memo1.Modified := false;

    ScriptAge := FileAge(ScriptName);
  end;
end;

procedure TForm1.SaveAs1Click(Sender: TObject);
begin
  with SaveDialog1 do
  if Execute then
  begin
    if Pos('.', FileName) = 0 then
      FileName := FileName + '.PAS';

    ScriptName := FileName;
    Memo1.Lines.SaveToFile(ScriptName);
    Memo1.Modified := false;

    ScriptAge := FileAge(ScriptName);
  end;
end;

procedure TForm1.Save1Click(Sender: TObject);
begin
  if ScriptName = '' then
    SaveAs1Click(Sender)
  else
    Memo1.Lines.SaveToFile(ScriptName);
  Memo1.Modified := false;
  ScriptAge := FileAge(ScriptName);
end;

procedure TForm1.Exit1Click(Sender: TObject);
begin
  Close;
end;

procedure TForm1.FormCreate(Sender: TObject);
var
  S: String;
begin
  S := ExtractFileDir(Application.ExeName);
  SetCurrentDir(S);

  Memo1.Font.Name := 'Courier new';
  Memo1.Font.Size := 12;

  OpenDialog1.InitialDir := S;
  OpenDialog1.Filter := PasScriptFilter;

  SaveDialog1.InitialDir := S;
  SaveDialog1.Filter := PasScriptFilter;

  ScriptName := '';
  PasScript := nil;
  Compiling := nil;
  CompileAndRun := false;
  CompileStage := false;
  RunStage := false;

  ScriptAge := 0;

  WasClosed := false;
end;

procedure TForm1.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  WasClosed := true;

  if Assigned(PasScript) then
    PasScript.Terminate(False);

  if Memo1.Modified  then
    if ScriptName = '' then
      SaveAs1Click(nil)
    else
    if MessageBox(GetActiveWindow(),
       PChar(Format('Save changes to %s ?', [ScriptName])),
         PChar('Confirm'), MB_ICONQUESTION or MB_YESNO) = IDYES then
        begin
          Save1Click(nil);
        end;
end;

procedure TForm1.FormDestroy(Sender: TObject);
begin
  if Assigned(PasScript) then
    PasScript.Free;
end;

procedure TForm1.ShowError;
var
  Form: TForm;
  Memo: TMemo;
begin
  if StrEql(ExtractFileName(ScriptName), ExtractFileName(PasScript.ErrorFileName)) then
  begin
    Memo1.Modified := false;
    Memo1.SetFocus;
    Memo1.SelStart := PasScript.TextErrorPos;
    Memo1.SelLength := 0;
    FormShow(nil);

    Exit;
  end;

  Form := TForm.Create(nil);
  Form.Left  := 50;
  Form.Top   := 200;
  Form.Width := 580;
  Memo := TMemo.Create(Form);
  Form.InsertControl(Memo);

  with PasScript do
    Form.Caption := Format('Error : %s, File : %s', [ErrorMessage, ErrorFileName]);
  Memo.Align := alClient;
  Memo.Font.Name := Memo1.Font.Name;
  Memo.Font.Size := Memo1.Font.Size;

  try
    Memo.Lines.Text := PasScript.ErrorScript;
    Form.Show;

    Memo.SetFocus;
    Memo.SelStart  := PasScript.TextErrorPos;
    Memo.SelLength := 0;

    Memo.Modified := false;

    Form.Hide;
    Form.ShowModal;

    if Memo.Modified then
    begin
      if MessageBox(GetActiveWindow(),
          PChar(Format('Save changes to %s?', [PasScript.ErrorFileName])),
         PChar('Confirm'), MB_ICONQUESTION or MB_YESNO) = IDYES then
        Memo.Lines.SaveToFile(PasScript.ErrorFileName);
    end;
  finally
    Form.Free;
  end;
end;

procedure TForm1.PasScriptCompileError(Sender: TObject);
begin
  CompileStage := false;

  with Compiling do
  begin
    LabelStatus.Caption   := 'Done:         There is an error';
    LabelCurrLineNumber.Caption := IntToStr(PasScript.ErrorLine);
    LabelAllLinesNumber.Caption := '0';
    LabelError.Caption    := 'Error:        ' + PasScript.ErrorMessage;

    WaitForEnter;
  end;

  ShowError;
end;

procedure TForm1.PasScriptRunError(Sender: TObject);
const
  Fmt = '%s'#13#10'File : %s'#13#10'Line : %d';
begin
  RunStage := false;

  with PasScript do
    ErrMessageBox(Format(Fmt, [ErrorMessage, ErrorFileName, ErrorLine]), 'Run-time error');
    
  ShowError;
end;

procedure TForm1.PasScriptSaveBinaryProg(Sender: TObject);
begin
  with Compiling do
  begin
    LabelStatus.Caption   := 'Done:        There is an error';
    LabelError.Caption    := 'Error:       Cannot create file ' + BinaryProgName;

    WaitForEnter;
  end;
end;

procedure TForm1.PasScriptLoadBinaryProg(Sender: TObject);
begin
  ShowMessage('Cannot load file' + BinaryProgName);
end;

procedure TForm1.PasScriptCompileProgress(Sender: TObject);
var
  S: String;
begin
  with Compiling do
  begin
    LabelCurrLineNumber.Caption := IntToStr(Form1.PasScript.CurrentLine);
    LabelAllLinesNumber.Caption := IntToStr(Form1.PasScript.LineCount);
    S := 'Compiling:    ' + ExtractFileName(Form1.PasScript.CurrentFile);
    if LabelStatus.Caption <> S then
      LabelStatus.Caption := S;
  end;
end;

procedure TForm1.Compile1Click(Sender: TObject);
var
  res: boolean;
  f: TFileStream;
begin
  CompileIsOK := false;

  if Memo1.Modified or (ScriptName = '') then
  begin
    Save1Click(Sender);
    if ScriptName = '' then
      Exit;
  end;

  if not FileExists(ScriptName) then
  begin
    ShowMessage('Could not find file ' + ScriptName );
    Exit;
  end;

  if Assigned(PasScript) then
    PasScript.Free;
  PasScript := TPasScript.Create(Self);

  PasScript.ScriptFileName := ScriptName;

  PasScript.OnCompileError := Form1.PasScriptCompileError;
{
  M.Code := @TForm1.PasScriptCompileError;
  M.Data := Pointer(Form1);

  PropInfo := GetPropInfo(PTypeInfo(PasScript.ClassType.ClassInfo), 'OnCompileError');
  SetMethodProp(PasScript, PropInfo, M);
}
  Application.CreateForm(TCompiling, Compiling);
  f := nil;
  try
    with Compiling do
    begin
      LabelProject.Caption  := 'Project:      ' + ScriptName;
      LabelStatus.Caption   := 'Compiling:    ' + ExtractFileName(ScriptName);
      LabelCurrLineNumber.Caption := '0';
      LabelAllLinesNumber.Caption := '0';
      LabelError.Caption    := 'Error:        None';
    end;
    Compiling.Show;

    CompileStage := true;

    PasScript.OnCompileProgress := Form1.PasScriptCompileProgress;

    res := PasScript.Parse;
    CompileStage := false;

    if not res then
      Exit;

    PasScript.OnSaveBinProgError := PasScriptSaveBinaryProg;

    f := TFileStream.Create(BinaryProgName, fmCreate);
    PasScript.SaveBinProg(f);

    with Compiling do
    begin
      LabelStatus.Caption   := 'Done:       Compiled';
      LabelCurrLineNumber.Caption := IntToStr(PasScript.CurrentLine);
      LabelAllLinesNumber.Caption := IntToStr(PasScript.LineCount);
      LabelError.Caption    := 'Error:        None';

      if not CompileAndRun then
        WaitForEnter;
    end;

    CompileIsOK := true;

  finally
    if Assigned(f) then
      f.Free;

    if Assigned(Compiling) then
      Compiling.Free;
    if Assigned(PasScript) then
      PasScript.Free;
    Compiling := nil;
    PasScript := nil;
  end;
end;

procedure TForm1.Run1Click(Sender: TObject);
var
  f: TFileStream;
begin
  if Memo1.Modified or (ScriptName = '' ) then
  begin
    Save1Click(Sender);
    if ScriptName = '' then
      Exit;
  end;

  if not FileExists(ScriptName) then
  begin
    ShowMessage('Could not find file ' + ScriptName );
    Exit;
  end;

  if (not FileExists(BinaryProgName)) or
     (FileAge(BinaryProgName) < FileAge(ScriptName)) then
  begin
    CompileAndRun := true;
    Compile1Click(Sender);
    CompileAndRun := false;
    if not CompileIsOK then Exit;
  end;

  if Assigned(PasScript) then
    PasScript.Free;
  PasScript := TPasScript.Create(Self);
  f := nil;
  try
    RunStage := true;
    PasScript.ScriptFileName := ScriptName;
    PasScript.OnRunError := PasScriptRunError;
    PasScript.OnLoadBinProgError := PasScriptLoadBinaryProg;

    f := TFileStream.Create(BinaryProgName, fmOpenRead);
    PasScript.LoadBinProg(f);
    PasScript.Run;
  finally
    if Assigned(f) then
      f.Free;
    if Assigned(PasScript) then
      if not PasScript.IsSuspended then
      begin
        RunStage := false;
        PasScript.Free;
        PasScript := nil;
      end;
  end;

  if WasClosed then
    Close;
end;

procedure TForm1.SpeedButtonNewClick(Sender: TObject);
begin
  New1Click(Sender);
end;

procedure TForm1.SpeedButtonOpenClick(Sender: TObject);
begin
  Open1Click(Sender);
end;

procedure TForm1.SpeedButtonSaveClick(Sender: TObject);
begin
  Save1Click(Sender);
end;

procedure TForm1.SpeedButtonRunClick(Sender: TObject);
begin
  Run1Click(Sender);
end;

procedure TForm1.Memo1KeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  FormShow(nil);
  if not (CompileStage or RunStage) then
    LabelRight.Caption := '';
end;

procedure TForm1.FormShow(Sender: TObject);
var
  X,Y : Integer;
begin
  with Memo1 do
  begin
    Y:=SendMessage(Handle, EM_LINEFROMCHAR, SelStart, 0);
    X:=SelStart-SendMessage(Handle, EM_LINEINDEX, Y, 0);
  end;
  LabelLeft.Caption :=Format('%d : %d', [Y+1, X+1]);
end;

procedure TForm1.Memo1MouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  FormShow(nil);
  if not (CompileStage or RunStage) then
    LabelRight.Caption := '';
end;

procedure TForm1.Memo1MouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  FormShow(nil);
  if not (CompileStage or RunStage) then
    LabelRight.Caption := '';
end;

procedure TForm1.Print1Click(Sender: TObject);
var
  FPrn: System.Text;
  I: Integer;
begin
  if PrintDialog1.Execute then
  begin
    AssignPrn(FPrn);
    Rewrite(FPrn);
    try
      Printer.Canvas.Font := Memo1.Font;
      for I:=0 to Memo1.Lines.Count - 1 do
        writeln(FPrn, Memo1.Lines[I]);
    finally
      CloseFile(FPrn);
    end;
  end;
end;

procedure TForm1.VIRTLaboratoryHomePage1Click(Sender: TObject);
begin
  ShellExecute(Handle , 'open', PChar('http://users.ints.net/virtlabor/'), nil, nil, SW_MAXIMIZE);
end;

procedure TForm1.PasScriptHomePage1Click(Sender: TObject);
begin
  ShellExecute(Handle , 'open', PChar('http://www.passcript.com/'), nil, nil, SW_MAXIMIZE);
end;

procedure TForm1.About1Click(Sender: TObject);
begin
  ShowMessage('                       PasScript v' + Version + #13#10 + #13#10 +
              'Copyright 1999-2001 by Alexander Baranovsky' + #13#10 +
              '                       VIRT Laboratory');
end;

procedure TForm1.OpenDemo(const FileName: String);
var
  S: String;
begin
  S := ExtractFilePath(Application.ExeName)+FileName;
  if FileExists(S) then
  begin
    ScriptName := S;
    Memo1.Lines.LoadFromFile(ScriptName);
    Memo1.Modified := false;
  end
  else
    ErrMessageBox('File ' + S + ' not found!', 'Error');
end;

procedure TForm1.ClassesandObjects1Click(Sender: TObject);
begin
  OpenDemo('democls1.pas');
end;

procedure TForm1.OpenArrays1Click(Sender: TObject);
begin
  OpenDemo('qsort.pas');
end;

procedure TForm1.OLEAutomation1Click(Sender: TObject);
begin
  OpenDemo('demoword.pas');
end;

procedure TForm1.Characterdifferentiation1Click(Sender: TObject);
begin
  OpenDemo('diff.pas');
end;

procedure TForm1.Theoremproving1Click(Sender: TObject);
begin
  OpenDemo('chang.pas');
end;

procedure TForm1.LeeAlgorithm1Click(Sender: TObject);
begin
  OpenDemo('lee.pas');
end;

procedure TForm1.ImportedDelphiclasses1Click(Sender: TObject);
begin
  OpenDemo('demoimport.pas');
end;

procedure TForm1.UsingTListclass1Click(Sender: TObject);
begin
  OpenDemo('DemoTList.pas');
end;

procedure TForm1.WhatsNewSetTypes1Click(Sender: TObject);
begin
  OpenDemo('demomsg.pas');
end;

procedure TForm1.EventHandlers1Click(Sender: TObject);
begin
  OpenDemo('EventHandlers1.pas');
end;

procedure TForm1.Classreferencetypes1Click(Sender: TObject);
begin
  OpenDemo('classref.pas');
end;

procedure TForm1.DefaultArrayProperties1Click(Sender: TObject);
begin
  OpenDemo('democls3.pas');
end;

procedure TForm1.Polymorphicbinaytrees1Click(Sender: TObject);
begin
  OpenDemo('bintree1.pas');
end;

procedure TForm1.Typedexceptions1Click(Sender: TObject);
begin
  OpenDemo('exceptions.pas');
end;

procedure TForm1.Tetrisgame1Click(Sender: TObject);
begin
  OpenDemo('tetris.pas');
end;

procedure TForm1.imp_dfmClick(Sender: TObject);
begin
  OpenDemo('imp_dfm.pas');
end;

procedure TForm1.WhatsNewClick(Sender: TObject);
begin
  OpenDemo('WhatsNew.pas');
end;

procedure TForm1.ShowText(const FileName, Title: String);
var
  Form: TForm;
  Memo: TMemo;
begin
  Form := TForm.Create(nil);
  Form.Left  := 50;
  Form.Top   := 200;
  Form.Width := 580;
  Memo := TMemo.Create(Form);
  Form.InsertControl(Memo);

  with PasScript do
    Form.Caption := Title;
  Memo.Align := alClient;
  Memo.Font.Name := Memo1.Font.Name;
  Memo.Font.Size := 8;
  Memo.ReadOnly  := true;

  if not FileExists(FileName) then
  begin
    ShowMessage('Could not find file ' + FileName);
    Exit;
  end;
  Memo.Lines.LoadFromFile(FileName);
  Form.Show;
end;

procedure TForm1.PasScriptmanual1Click(Sender: TObject);
begin
  Application.HelpFile := 'passcript.hlp';
  Application.HelpJump('TPasScript');
end;

procedure TForm1.KnownDelphiTypesandRoutines1Click(Sender: TObject);
var
  Form: TForm;
  TreeView: TTreeView;
begin
  Form := TForm.Create(nil);
  Form.Left  := 50;
  Form.Top   := 200;
  Form.Width := 580;
  TreeView := TTreeView.Create(Form);
  Form.InsertControl(TreeView);

  Form.Caption := 'Imported Delphi Types';
  TreeView.Align := alClient;

  GetKnownDelphiTypes(TreeView);
  Form.Show;
end;

procedure TForm1.SourceCodeWhichEnsuresImport1Click(Sender: TObject);
begin
  ShowText('import.dpr', 'Source Code Providing Import');
end;

procedure TForm1.ButtonSuspendClick(Sender: TObject);
begin
  PasScript.Suspend;
end;

procedure TForm1.ButtonResumeClick(Sender: TObject);
begin
  if PasScript.IsRunning then Exit;

  PasScript.Resume;
  if not PasScript.IsSuspended then
    RunStage := false;
  if WasClosed then
    Close;
end;

procedure TForm1.ButtonTerminateClick(Sender: TObject);
begin
  RunStage := false;
  PasScript.Terminate(True);
end;

{ ------------  Callback tests ---------------------}
{
type
  TCallBackProc = procedure (I, J: Integer); stdcall;
procedure TestCallBackProc(P: TCallBackProc);
begin
  P(8,9);
end;

type
  TCallBackFuncInteger = function (I, J:Integer): Integer; stdcall;
function TestCallBackFuncInteger(P: TCallBackFuncInteger): Integer;
begin
  result := P(8,9);
end;

type
  TCallBackFuncString = function (I, J:Integer): String; stdcall;
function TestCallBackFuncString(P: TCallBackFuncString): String;
begin
  result := P(8,9);
end;

type
  TCallBackFuncSingle = function (I, J:Integer): Single; stdcall;
function TestCallBackFuncSingle(P: TCallBackFuncSingle): Single;
begin
  result := P(8,9);
end;

type
  TSmallRec = record P1, P2: Byte; end;
  TCallBackFuncSmallRec = function (I, J:Integer; R: TSmallRec): TSmallRec; stdcall;
function TestCallBackFuncSmallRec(P: TCallBackFuncSmallRec): TSmallRec;
var
  R: TSmallRec;
begin
  R.P1 := 17;
  R.P2 := 18;
  result := P(8,9, R);
end;

type
  TBigRec = record P1, P2: Integer; end;
  TCallBackFuncBigRec = function (I, J:Integer; R: TBigRec): TBigRec; stdcall;
function TestCallBackFuncBigRec(P: TCallBackFuncBigRec): TBigRec;
var
  R: TBigRec;
begin
  R.P1 := 55;
  R.P2 := 77;
  result := P(8,9, R);
end;

type
  TCallBackFuncVariant = function (var I:Integer; const J:String; U: Variant): Variant; stdcall;
function TestCallBackFuncVariant(P: TCallBackFuncVariant): Variant;
var
  I: Integer;
  S: String;
  U: Variant;
begin
  I := 8;
  S := 'abc';
  U := 44;
  result := P(I, S, U);
  I := I + 1;
  S := 'd';
end;
}

initialization
  RegisterDfms('passcript.exe');

{ ------------  Delphi handler tests ---------------------}
{
  RegisterRTTIType(TypeInfo(TDelphiHandler));
  RegisterMethod(TDelphiHandler, 'procedure DelphiHandlerEvent(I, J: Integer);',
                 @TDelphiHandler.DelphiHandlerEvent);
  RegisterMethod(TDelphiHandler, 'procedure DelphiHandlerNotifyEvent(Sender: TObject);',
                 @TDelphiHandler.DelphiHandlerNotifyEvent);
}
{ ------------  Callback tests ---------------------}

{
  RegisterType('TCallBackProc = procedure (I, J: Integer); stdcall;');
  RegisterRoutine('procedure TestCallBackProc(P: TCallBackProc);',
                   @TestCallBackProc);

  RegisterType('TCallBackFuncInteger = function (I, J:Integer): Integer; stdcall;');
  RegisterRoutine('function TestCallBackFuncInteger(P: TCallBackFuncInteger): Integer;',
                   @TestCallBackFuncInteger);

  RegisterType('TCallBackFuncString = function (I, J:Integer): String; stdcall;');
  RegisterRoutine('function TestCallBackFuncString(P: TCallBackFuncString): String;',
                   @TestCallBackFuncString);

  RegisterType('TCallBackFuncSingle = function (I, J:Integer): Single; stdcall;');
  RegisterRoutine('function TestCallBackFuncSingle(P: TCallBackFuncSingle): Single;',
                   @TestCallBackFuncSingle);

  RegisterType('TSmallRec = record P1, P2: Byte; end;');
  RegisterType('TCallBackFuncSmallRec = function (I, J:Integer; R: TSmallRec): TSmallRec; stdcall;');
  RegisterRoutine('function TestCallBackFuncSmallRec(P: TCallBackFuncSmallRec): TSmallRec;',
                   @TestCallBackFuncSmallRec);

  RegisterType('TBigRec = record P1, P2: Integer; end;');
  RegisterType('TCallBackFuncBigRec = function (I, J:Integer; R: TBigRec): TBigRec; stdcall;');
  RegisterRoutine('function TestCallBackFuncBigRec(P: TCallBackFuncBigRec): TBigRec;',
                   @TestCallBackFuncBigRec);

  RegisterType('TCallBackFuncVariant = function (var I:Integer; const J: String; U: Variant): Variant; stdcall;');
  RegisterRoutine('function TestCallBackFuncVariant(P: TCallBackFuncVariant): Variant;',
                   @TestCallBackFuncVariant);
}
end.


