unit ESBMaths2;

{
	ESBMaths - contains useful Mathematical routines for Delphi 3 & 4<p>
<p>
	 1999 ESB Consultancy<p>
<p>
	These routines are used by ESB Consultancy within the
	development of their Customised Applications, and have been
	under Development since the early Turbo Pascal days.
	Many of the routines were developed for specific needs<p>
<p>
	ESB Consultancy retains full copyright.<p>
<p>
	ESB Consultancy grants users of this code royalty free rights
	to do with this code as they wish.<p>
<p>
	ESB Consultancy makes no guarantees nor excepts any liabilities
	due to the use of these routines<p>
<p>
	We does ask that if this code helps you in you development
	that you send as an email mailto:glenn@esbconsult.com.au or even
	a local postcard. It would also be nice if you gave us a
	mention in your About Box or Help File.<p>
<p>
	ESB Consultancy Home Page: http://www.esbconsult.com.au<p>
<p>
	Mail Address: PO Box 2259, Boulder, WA 6432 AUSTRALIA<p>
<p>
	v2.1 13 July 1999<p>
		- Now over 110 Routines<p>
		- Added ESBDigits and ESBBitsNeeded for Cardinals to ESBMaths<p>
		- Added ESBMagnitude for Reals to ESBMaths<p>
		- Added GrandMean to ESBMath2<p>
		- Added Matrix Operations to ESBMath2, just the start<p>
<p>
	v2.0 16 Apr 1999<p>
		- Delphi 4 specific routines moved to ESBMaths2<p>
		- Added Vector operations to ESBMath2<p>
<p>
	v1.1 22 Jan 1999 - Improved Delphi 4 support<p>
		- Added: SumLWArray, SumSqEArray, SumSqDiffEArray, SumXYEArray<p>
		- Added: IsPositiveEArray<p>
		- Added: Geometric and Harmonic Means<p>
		- Added ESBMean, Median and Mode<p>
		- Added Routines for Variances and Means for Populations & Samples<p>
		- Added Quartile Calculations<p>
		- Fixed a couple of minor bugs<p>
		- Fixed bug in XY2Polar reported by Wolfgang Werner<p>
		- Added Help File<p>
<p>
	v1.01 17 July 1998 - Added some improvements suggested by Rory Daulton<p>
		- Added Factorials using Extendeds<p>
		- Added Permutations using Extendeds<p>
		- Added Combinations using Extendeds<p>
		- Added SwapI32 to Swap Integers<p>
		- Added SwapC to Swap Cardinals<p>
		- Added Factorials computed using Extendeds<p>
		- Added Permutations computed using Extendeds<p>
		- Added Binomial Coefficients (Combinations) computed using Extendeds<p>
<p>
	v1.00 17 April 1998 - first public/freeware release<p>
}

interface

{$DEFINE UseMath} 	{Comment out this line if you don't want to use the Delphi Math Library}
				{ Desktop versions of Delphi do not include the Math Library }

{$IFNDEF Ver120}
Routines designed for Delphi 4 only!
{$ENDIF}

uses
	ESBMath;

type
	TDynFloatArray = array of Extended;
	TDynLWordArray = array of LongWord;
	TDynLIntArray = array of LongInt;

type
	TDynFloatMatrix = array of TDynFloatArray;
	TDynLWordMatrix = array of TDynLWordArray;
	TDynLIntMatrix = array of TDynLIntArray;

{--- Vector Operations ---}

{: Returns Vector X with all its elements squared }
function SquareAll (const X: TDynFloatArray): TDynFloatArray;

{: Returns Vector X with all its elements inversed , i.e 1 / X [i].
An exception is raised if any element is zero }
function InverseAll (const X: TDynFloatArray): TDynFloatArray;

{: Returns Vector X with all the Natural Log of all its elements .
An exception is raised if any element is not Positive }
function LnAll (const X: TDynFloatArray): TDynFloatArray;

{: Returns Vector X with all the Log to Base 10 of all its elements .
An exception is raised if any element is not Positive }
function Log10All (const X: TDynFloatArray): TDynFloatArray;

{: Returns Vector X with all elements Linearly transformed.
	NewX [i] = Offset + Scale * X [i] }
function LinearTransform (const X: TDynFloatArray;
	Offset, Scale: Extended): TDynFloatArray;

{: Returns a vector where each element is the corresponding elements
of X and Y added together. The Length of the resultant vector is that
of the smaller of X and Y. }
function AddVectors (const X, Y: TDynFloatArray): TDynFloatArray;

{: Returns a vector where each element is the corresponding elements
of Y subtracted from X added together. The Length of the resultant vector is
that of the smaller of X and Y. }
function SubVectors (const X, Y: TDynFloatArray): TDynFloatArray;

{: Returns a vector where each element is the corresponding elements
of X and Y multiplied together. The Length of the resultant vector is that
of the smaller of X and Y. }
function MultVectors (const X, Y: TDynFloatArray): TDynFloatArray;

{: Returns the Dot Product of the two vectors, i.e. the sum of the pairwise
products of the elements. If Vectors are not of equal length then only
the shorter length is used.}
function DotProduct (const X, Y: TDynFloatArray): Extended;

{: Returns the Norm of a vector, i.e. the square root of the sum of the
squares of the elements. }
function Norm (const X: TDynFloatArray): Extended;

{--- Matrix Operations ---}

{: Returns true if the Matrix is not nil and all the "columns" are
the same length - Delphi allows a 2 Dimensional Dynamic Array with
different length "columns" - this can cause problems in some operations }
function MatrixIsRectangular (const X: TDynFloatMatrix): Boolean;

{: Returns Rectangular as true if the Matrix is not nil and all the "columns"
are the same length - Delphi allows a 2 Dimensional Dynamic Array with
different length "columns" - this can cause problems in some operations.
M and N are the dimensions which really only make sense if the Matrix
is Rectangular }
procedure MatrixDimensions (const X: TDynFloatMatrix;
	var M, N: Integer; var Rectangular: Boolean);

{: For a Matrix to be Square it must be Rectangular and have the
same number of "rows" and "columns" }
function MatrixIsSquare (const X: TDynFloatMatrix): Boolean;

{: Matrices have the same dimensions if they are both Rectangular and
they have the same number of "rows" and the same number of "columns"}
function MatricesSameDimensions (const X, Y: TDynFloatMatrix): Boolean;

{: Returns a Dynamic Matrix that is the result of Adding the two supplied
Matrices. Both X and Y must be truly Rectangular and must be of the same
dimension otherwise an Exception is raised. }
function AddMatrices (const X, Y: TDynFloatMatrix): TDynFloatMatrix;

{: Returns a Dynamic Matrix that is the result of Subtracting the two supplied
Matrices. Both X and Y must be truly Rectangular and must be of the same
dimension otherwise an Exception is raised. }
function SubtractMatrices (const X, Y: TDynFloatMatrix): TDynFloatMatrix;

{: Returns a Dynamic Matrix that is the result of multiplying each element
of X by the constant K. Will handle non-Rectangular Matrices }
function MultiplyMatrixByConst (const X: TDynFloatMatrix; const K: Extended): TDynFloatMatrix;

{: Calculates the Grand Mean of a Matrix: Sum of all the values
divided by no of values. Will handle non-Rectangular Matrices.
Also returns N the number of Values since the Matrix may not be Rectangular }
function GrandMean (const X: TDynFloatMatrix; var N: LongWord): Extended;

implementation

uses
{$IFDEF UseMath}
	Math,
{$ENDIF}
	SysUtils;

function SquareAll (const X: TDynFloatArray): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, High (X) + 1);
	for I := 0 to High (X) do
		Result [I] := Sqr (X [I]);
end;

function InverseAll (const X: TDynFloatArray): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, High (X) + 1);
	for I := 0 to High (X) do
	begin
		if X [I] = 0 then
			raise Exception.Create ('Inverse of Zero');
		Result [I] := 1 / (X [I]);
	end;
end;

function LnAll (const X: TDynFloatArray): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, High (X) + 1);
	for I := 0 to High (X) do
	begin
		if X [I] <= 0 then
			raise Exception.Create ('Logarithm on non-Positive');
		Result [I] := Ln (X [I]);
	end;
end;

function Log10All (const X: TDynFloatArray): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, High (X) + 1);
	for I := 0 to High (X) do
	begin
		if X [I] <= 0 then
			raise Exception.Create ('Logarithm on non-Positive');
		Result [I] := Log10 (X [I]);
	end;
end;

function LinearTransform (const X: TDynFloatArray;
	Offset, Scale: Extended): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, High (X) + 1);
	for I := 0 to High (X) do
		Result [I] := OffSet + Scale * X [I];
end;

function AddVectors (const X, Y: TDynFloatArray): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, min (High (X), High (Y)) + 1);
	for I := 0 to High (Result) do
		Result [I] := X [I] + Y [I];
end;

function SubVectors (const X, Y: TDynFloatArray): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, min (High (X), High (Y)) + 1);
	for I := 0 to High (Result) do
		Result [I] := X [I] - Y [I];
end;

function MultVectors (const X, Y: TDynFloatArray): TDynFloatArray;
var
	I: LongWord;
begin
	SetLength (Result, min (High (X), High (Y)) + 1);
	for I := 0 to High (Result) do
		Result [I] := X [I] * Y [I];
end;

function DotProduct (const X, Y: TDynFloatArray): Extended;
var
	I, N: Longword;
begin
	Result := 0.0;
	N := min (High (X), High (Y));
	for I := 0 to N do
		Result := Result + X [I] * Y [I];
end;

function Norm (const X: TDynFloatArray): Extended;
begin
	Result := Sqrt (DotProduct (X, X));
end;

function GrandMean (const X: TDynFloatMatrix; var N: LongWord): Extended;
var
	I, J: Integer;
begin
	Result := 0;
	if (High (X) < 0) or (High (X [0]) < 0) then
		raise Exception.Create ('Matrix is Empty!');

	N := 0;
	for I := 0 to High (X) do
	begin
		N := N + Longword (High (X [I])) + 1;
		for J := 0 to High (X [I]) do
			Result := Result + X [I, J];
	end;
	if N > 0 then
		Result := Result / N
	else
		raise Exception.Create ('Matrix is Empty!');
end;

function AddMatrices (const X, Y: TDynFloatMatrix): TDynFloatMatrix;
var
	I, J, N: Integer;
begin
	Result := nil;
	if (High (X) < 0) or (High (Y) < 0) then
		raise Exception.Create ('Matrix is Empty!');

	if (High (X) <> High (Y)) then
		raise Exception.Create ('Matrices must be the same Dimension to Add!');

	N := High (X [0]);
	SetLength (Result, High (X) + 1, N + 1);
	for I := 0 to High (X) do
	begin
		if (High (X [I]) <> N) then
		begin
			Result := nil;
			raise Exception.Create ('Matrices must be truly rectangular to Add!');
		end;
		if (High (Y [I]) <> N) then
		begin
			Result := nil;
			raise Exception.Create ('Matrices must be the same Dimension to Add!');
		end;

		for J := 0 to N do
			Result [I, J] := X [I, J] + Y [I, J];
	end;
end;

function SubtractMatrices (const X, Y: TDynFloatMatrix): TDynFloatMatrix;
var
	I, J, N: Integer;
begin
	Result := nil;
	if (High (X) < 0) or (High (Y) < 0) then
		raise Exception.Create ('Matrix is Empty!');

	if (High (X) <> High (Y)) then
		raise Exception.Create ('Matrices must be the same Dimension to Subtract!');

	N := High (X [0]);
	SetLength (Result, High (X) + 1, N + 1);
	for I := 0 to High (X) do
	begin
		if (High (X [I]) <> N) then
		begin
			Result := nil;
			raise Exception.Create ('Matrices must be truly rectangular to Subtract!');
		end;
		if (High (Y [I]) <> N) then
		begin
			Result := nil;
			raise Exception.Create ('Matrices must be the same Dimension to Subtract!');
		end;

		for J := 0 to N do
			Result [I, J] := X [I, J] - Y [I, J];
	end;
end;

function MultiplyMatrixByConst (const X: TDynFloatMatrix; const K: Extended): TDynFloatMatrix;
var
	I, J: Integer;
begin
	Result := nil;
	if (High (X) < 0) then
		raise Exception.Create ('Matrix is Empty!');

	SetLength (Result, High (X) + 1);
	for I := 0 to High (X) do
	begin
		SetLength (Result [I], High (X [I]) + 1);
		for J := 0 to High (X [I]) do
			Result [I, J] := X [I, J] * K;
	end;
end;

function MatrixIsRectangular (const X: TDynFloatMatrix): Boolean;
var
	I, N: Integer;
begin
	Result := False;
	if (High (X) < 0) then
		Exit;

	N := High (X [0]);
	for I := 0 to High (X) do
		if (High (X [I]) <> N) then
			Exit;

	Result := True;
end;

procedure MatrixDimensions (const X: TDynFloatMatrix;
	var M, N: Integer; var Rectangular: Boolean);
var
	I: Integer;
begin
	M := 0;
	N := 0;
	Rectangular := False;
	if (High (X) < 0) then
		Exit;

	M := High (X) + 1;
	N := High (X [0]);
	for I := 0 to High (X) do
		if (High (X [I]) <> N) then
		begin
			N := 0;
			Exit;
		end;

	Inc (N);
	Rectangular := True;
end;

function MatrixIsSquare (const X: TDynFloatMatrix): Boolean;
var
	M, N: Integer;
	Rectangular: Boolean;
begin
	MatrixDimensions (X, M, N, Rectangular);
	Result := Rectangular and (M = N);
end;

function MatricesSameDimensions (const X, Y: TDynFloatMatrix): Boolean;
var
	M1, N1: Integer;
	Rectangular1: Boolean;
	M2, N2: Integer;
	Rectangular2: Boolean;
begin
	MatrixDimensions (X, M1, N1, Rectangular1);
	MatrixDimensions (Y, M2, N2, Rectangular2);
	Result := Rectangular1 and Rectangular2 and (M1 = M2) and (N1 = N2);
end;

end.
