{$F+} { Compiler Directive: Generate far procedure calls: On }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  Pick
    version 2.3

  This unit holds procedures designed to allow system file picks.

  Purpose:
    To provide support procedures that allow a program to offer a list of
      directories.

  How it works:
    This unit uses the menu unit and the string unit to offer a selection of
      files.

  Features:
    This unit uses the Menu unit which may link with the Windows unit.
    Allocates heap space to hold the directory list.
    No limit to the amount of file names.

  Versions
    1.1 - Added the Pick_Okay flag.
    2.0 - Removes the limit to the amount of file names and reduced the amount
          of memory necessary to support this unit by using the menu itself to
          supply the names.
    2.1 - Added option to remove backup file from the pick list display.
    2.2 - Makes it possible that the selection is not necessarily limited to
          the current directory and it's child branches.
    2.3 - Adds the option of moving to another drive.

  Copyright 1989, 1995 All rights reserved.
    Paul R. Renaud

   Compilers:
     Turbo Pascal versions 4.0 to 6.0
     Speed Pascal/2 version 1.5

   Systems:
     MS-DOS, MDOS, OS/2

*****************************************************************************)

Unit Pick;

  Interface

    Uses
      DOS,
      CRT,
      Menu,
     {$IFNDEF OS2}
      String_Utilities;
     {$ELSE}
      String_U;
     {$ENDIF}

(***********************************************************

  Exclude_Backups - This switch, when set to true, will
    force the files with the .Bak extension to be excluded
    from the file list.

***********************************************************)

    Const
      Exclude_BackUps: Boolean = False;

(***********************************************************

  Allow_Drive_Switch -  This switch, when set to true, gives
    the user the opportunity to switch to another disk drive
    on the system.

***********************************************************)

      Allow_Drive_Switch: Boolean = True;

(***********************************************************

  Pick_Okay - This flag is set to true if the Pick_File
    routine worked, otherwise it will be set to false.

***********************************************************)

    Var
      Pick_Okay: Boolean;

(***********************************************************

  Function: Pick file in the path.

    This function allows the user to search through all the
    directories in order to pick a file.  The file name is
    returned as a full path name.
    Current_Path is a path supplied to begin the search.
      The path should look something like these ..
             'A:\'
             'C:\MENU\'

    If there isn't enough memory to allocate a pick list,
    then this function sets Pick_Okay to false and exits.

***********************************************************)

    Function Pick_File( Current_Path: String ): String;

{-----------------------------------------------------------------------------}

  Implementation

    Const
     { Exclusions holds the files it won't offer as a choice. }
     {$IFNDEF VER40}
      Exclusions: Word = ( VolumeID + Hidden );
     {$ELSE}
      Exclusions: Word = 0;
     {$ENDIF}
      Root_Constant = '< root >    ';
      Drive_Constant = '< drives >';

    Type
     { This is returned by Directory to determine what the selection was. }
      Directory_Type = ( D_Root, D_File, D_Branch, D_Drive );



{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Stretch.
    This procedure takes the name and stretches
    it in the Long_Name variable.  Spaces are
    added normally before the extension.

*************************************************)

  Procedure Stretch( Name: String; Var Long_Name: String );
    Var
      Where: Byte;
    Begin
      Long_Name := Name;
      Where := Pos( '.', Long_Name );
      If ( Where <> 0 )
        then
          Begin
            While ( ( Length( Long_Name ) < 12 ) and ( Where < 10 ) ) do
              Begin
                Insert( ' ', Long_Name, Where );
                Where := Pos( '.', Long_Name );
              End
          End;
      While ( Length( Long_Name ) < 12 ) do
        Long_Name := Long_Name + ' ';
    End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Expand.
    This procedure expands the given string with
    blanks until it is the specified size.

*************************************************)

    Procedure Expand( Var Data: String; Size: Byte );
      Begin
        While ( Length( Data ) < Size ) do
          Data := ' ' + Data;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Zero.
    This procedure replaces the leading blank with
    a zero.

*************************************************)

    Procedure Zero( Var Data );
      Var
        Data2: String Absolute Data;
      Begin
        If Data2[ 1 ] = ' '
          then
            Data2[ 1 ] := '0';
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Convert.
    This procedure converts the information in
    Information into a readable format for the
    menu.

*************************************************)

    Procedure Convert( Var Information: SearchRec; Var Result: String );
      Var
        Data_String: String;
        Day_String,
        Hour_String,
        Month_String,
        Minute_String: String[ 2 ];
        Extension: String[ 3 ];
        Year_String: String[ 4 ];
        Date_Time: DateTime;
      Begin
        With Information do
          Begin
            If ( Name <> '..' )
              then
                Stretch( Name, Data_String )
              else
                Data_String := Root_Constant;
            Result := Data_String;
            If ( ( Attr and Directory ) <> 0 )
              then
                Begin
                  Result := Result + '  ';
                  Result[ 10 ] := '[';
                  Result[ 11 ] := 'D';
                  Result[ 12 ] := 'I';
                  Result[ 13 ] := 'R';
                  Result[ 14 ] := ']';
                End
              else
                Result := Result + '  ';
            Str( Size:7, Data_String );
            Comma_The_String( Data_String );
            Expand( Data_String, 9 );
            Result := Result + Data_String + ' ';
            UnpackTime( Time, Date_Time );
            Str( Date_Time.Year:4, Year_String );
            Str( Date_Time.Month:2, Month_String );
            Str( Date_Time.Day:2, Day_String );
            Zero( Day_String );
            Result := Result + Month_String + '-' + Day_String + '-' + Year_String + '  ';
            Extension := 'pm ';
            If ( Date_Time.Hour > 12 )
              then
                Dec( Date_Time.Hour, 12 )
              else
                Extension := 'am ';
            If ( Date_Time.Hour = 0 )
              then
                Date_Time.Hour := 12;
            Str( Date_Time.Hour:2, Hour_String );
            Str( Date_Time.Min:2, Minute_String );
            Zero( Minute_String );
            Result := Result + Hour_String + ':' + Minute_String + Extension;
            If ( ( Attr and Hidden ) <> 0 )
              then
                Result := Result + 'Hidden ';
            If ( ( Attr and VolumeID ) <> 0 )
              then
                Result := Result + 'Label ';
            If ( ( Attr and ReadOnly ) <> 0 )
              then
                Result := Result + 'Protected ';
            If ( ( Attr and SysFile ) <> 0 )
              then
                Result := Result + 'System ';
            If ( ( Attr and Archive ) <> 0 )
              then
                Result := Result + 'Archive ';
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Valid name.
    This function takes the data and returns true
    if the name is valid for making up the
    directory list.

*************************************************)

    Function Valid_Name( Data: String ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := ( Data <> '.' );
        If Okay
          then
            Okay := ( Pos( '$$$', Data ) = 0 );
        If ( Okay and Exclude_BackUps )
          then
            Okay := ( Pos( 'BAK', Data ) = 0 );
        Valid_Name := Okay;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Fill file menu.
    This function fills the menu with the list of
    files and subdirectories in the directory.  It
    returns false if it fails.

*************************************************)

    Function Fill_File_Menu( Var Menu: Box_Menu_Type; Current_Path: String ): Boolean;
      Var
        Okay: Boolean;
        Result: String;
        Point,
        Attribute: Word;
        Information: SearchRec;
      Begin
        Okay := True;
        Point := 1;
        Attribute := ( AnyFile - Exclusions );
        If Allow_Drive_Switch and ( Length( Current_Path ) < 4 )
          then
            Begin
              Okay := Insert_Top_Box_Menu( Menu, Drive_Constant, Point );
              Inc( Point );
            End;
        FindFirst( ( Current_Path + '*.*' ), Attribute, Information );
        While ( ( DOSError = 0 ) and Okay ) do
          Begin
            If Valid_Name( Information.Name )
              then
                Begin
                  Convert( Information, Result );
                  Okay := Insert_Bottom_Box_Menu( Menu, Result, Point );
                  Inc( Point );
                End;
            FindNext( Information );
          End;
       {$IFDEF OS2}
        FindClose( Information );
       {$ENDIF}
        Fill_File_Menu := Okay;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Fill drive menu.
    This function fills the menu with the list of
    drives on the system.  It returns false if it
    fails.

*************************************************)

    Function Fill_Drive_Menu( Var Menu: Box_Menu_Type ): Boolean;
      Var
        Okay: Boolean;
        Current_Path: String;
        Character: Char;
        Point: Word;
      Begin
        GetDir( 0, Current_Path );
        Okay := True;
        Point := 1;
        For Character := 'A' to 'Z' do
          If Okay
            then
              Begin
               {$I-}
                ChDir( Character + ':\' );
                If ( IOResult = 0 )
                  then
                    Begin
                      Okay := Insert_Bottom_Box_Menu( Menu, Character + ':\', Point );
                      Inc( Point );
                    End;
               {$I+}
              End;
        Fill_Drive_Menu := Okay;
        ChDir( Current_Path );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Strip.
    This procedure strips the trailing characters
    of the string until the last character is the
    directory switching character.

*************************************************)

    Procedure Strip( Var Data: String );
      Var
        Data_Length: Byte absolute Data;
      Begin
        If ( Data_Length > 3 )
          then
            Repeat
              Dec( Data_Length );
            Until ( ( Data[ Data_Length ] = '\' ) or ( Data_Length = 3 ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Get choice.
    This function gets the name of the file or
    branch from the menu.  It destroys part of the
    menu as it searches for the name.

*************************************************)

    Function Get_Choice( Var Menu: Box_Menu_Type; Where: Byte ): String;
      Var
        Data: String;
      Begin
        Data := '';
        While ( Where > 0 ) do
          Begin
            Data := Get_Top_Box_Menu( Menu );
            Remove_Top_Box_Menu( Menu );
            Dec( Where );
          End;
        Get_Choice := Copy( Data, 1, 12 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Directory.
    This function returns the part of the
    directory structure that the name is.

*************************************************)

    Function Directory( Data: String ): Directory_Type;
      Begin
        If ( Data = Drive_Constant )
          then
            Directory := D_Drive
          else
            If ( Pos( '[', Data ) = 0 )
              then
                Directory := D_File
              else
                If ( Pos( '<', Data ) = 0 )
                  then
                    Directory := D_Branch
                  else
                    Directory := D_Root;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Pick drive internal.
    This function allows the user to pick a drive
    from a given choice of drives from the system.

*************************************************)

    Function Pick_Drive_Internal( Var Path: String ): Boolean;
      Var
        Okay: Boolean;
        Result: Word;
        Box_Menu: Box_Menu_Type;
      Begin
        Okay := True;
        Initialize_Box_Menu( Box_Menu );
        Pick_Okay := Fill_Drive_Menu( Box_Menu );
        If Pick_Okay
          then
            Begin
              Offer_Box_Menu( Box_Menu, Result );
              Okay := ( Menu.Command = Enter_Key );
              If Okay
                then
                  Path := Get_Choice( Box_Menu, Result )
            End;
        Dispose_Box_Menu( Box_Menu );
        Pick_Drive_Internal := ( Pick_Okay and Okay );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Pick file internal.
    This function allows the user to pick a file
    from a given choice of files from the given
    directory.

*************************************************)

    Function Pick_File_Internal( Current_Path: String ): String;
      Var
        Done: Boolean;
        Result: Word;
        Command,
        Save_Attribute: Byte;
        Box_Menu: Box_Menu_Type;
        Name,
        Directory_Path: String;
      Begin
        Done := False;
        Directory_Path := FExpand( Current_Path );
        Save_Attribute := TextAttr;
        Blend_Attribute := False;
        Repeat
          Initialize_Box_Menu( Box_Menu );
          Pick_Okay := Fill_File_Menu( Box_Menu, Directory_Path );
          If Pick_Okay
            then
              Begin
                TextAttr := Save_Attribute;
               {$IFDEF OS2}
                TextBackground( TextAttr shr 4 );
               {$ENDIF}
                Offer_Box_Menu( Box_Menu, Result );
                Command := Menu.Command;
                If ( Command = Enter_Key )
                  then
                    Begin
                      Name := Get_Choice( Box_Menu, Result );
                      Case Directory( Name ) of
                        D_Root: Strip( Directory_Path );
                        D_Branch: Directory_Path := Directory_Path + Copy( Name, 1, Pred( Pos( ' ', Name ) ) ) + '\';
                        D_File:
                          Begin
                            Done := True;
                            Directory_Path := Directory_Path + Name;
                          End;
                        D_Drive:
                          Begin
                            Done := not Pick_Drive_Internal( Directory_Path );
                          End
                      End; { Case }
                    End
                  else
                    Done := True;
              End;
          Dispose_Box_Menu( Box_Menu );
        Until ( Done = True ) or ( not Pick_Okay );
        Pick_File_Internal := Directory_Path;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Pick file.
    As previously defined.

*************************************************)

    Function Pick_File( Current_Path: String ): String;
      Var
        Data: String;
      Begin
        Data := Pick_File_Internal( Current_Path );
        Remove_All_Blanks( Data );
        Pick_File := Data;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Main initializing section.
    Initialize the exclusion variable if
      necessary.

*************************************************)

   {$IFDEF VER40}
    Begin
      Exclusions := ( VolumeID + Hidden );
     {$ENDIF}
    End.
