{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Screen move.
    This procedure provides the core data moving
    routine between the screen and normal memory.

*************************************************)

   {$IFNDEF OS2}
    Procedure Screen_Move( Var Source, Destination; Length: Word );
      External;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get screen.
    This procedure provides the core screen
    accessing routine.

*************************************************)

    Procedure Get_Screen( Row, Column: Byte; Var Character: Char; Var Attribute: Byte; Current_Page: Byte );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Var
        Value: Cell_Type;
      Begin
       {$IFDEF Debug}
        Value.Character := ' ';
        Value.Attribute := 0;
       {$ENDIF}
        VideoReadCell( Value, 2, Pred( Row ), Pred( Column ) );
        Character := Value.Character;
        Attribute := Value.Attribute;
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get screen plus.
    This procedure provides the screen multiple
    character reading routine.

*************************************************)

   {$IFDEF OS2}
    Procedure Get_Screen_Plus( Row, Column: Byte; Var Data; Length: Byte );
      Begin
        VideoReadCell( Data, ( Length * 2 ), Pred( Row ), Pred( Column ) );
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put screen.
    This procedure provides the core screen
    writing routine.

*************************************************)

    Procedure Put_Screen( Row, Column: Byte; Character: Char; Attribute, Current_Page: Byte );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Var
        Value: Cell_Type;
      Begin
        Value.Character := Character;
        Value.Attribute := Attribute;
        VideoWriteCell( Value, 2, Pred( Row ), Pred( Column ) );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put screen plus.
    This procedure provides the screen multiple
    character writing routine.

*************************************************)

   {$IFDEF OS2}
    Procedure Put_Screen_Plus( Row, Column: Byte; Var Data; Length: Byte );
      Begin
        VideoWriteCell( Data, ( Length * 2 ), Pred( Row ), Pred( Column ) );
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System scroll up.
    This procedure provides the standard system
    scrolling routines going up.

*************************************************)

    Procedure System_Scroll_Up( Left, Top, Right, Bottom: Byte );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Var
        Fill: Cell_Type;
      Begin
        Fill.Character := ' ';
        Fill.Attribute := TextAttr;
        VideoScrollUp( Pred( Left ), Pred( Top ), Pred( Right ), Pred( Bottom ), 1, Fill );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System scroll down.
    This procedure provides the standard system
    scrolling routines going down.

*************************************************)

    Procedure System_Scroll_Down( Left, Top, Right, Bottom: Byte );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Var
        Fill: Cell_Type;
      Begin
        Fill.Character := ' ';
        Fill.Attribute := TextAttr;
        VideoScrollDown( Pred( Left ), Pred( Top ), Pred( Right ), Pred( Bottom ), 1, Fill );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Change attribute.
    This procedure changes the specified screen
    attribute.

*************************************************)

    Procedure Change_Attribute( Row, Column, New_Attribute, Page_Number: Byte );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Begin
        VideoWriteAttribute( New_Attribute, 1, Pred( Row ), Pred( Column ) );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dim attribute.
    This procedure alters the current screen
    attribute at the given location.

*************************************************)

    Procedure Dim_Attribute( Row, Column, Attribute, Page_Number: Byte );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Var
        Value: Cell_Type;
      Begin
        Dec( Row );
        Dec( Column );
        VideoReadCell( Value, 2, Row, Column );
        Value.Attribute := ( ( Value.Attribute and $80 ) or Attribute );
        VideoWriteCell( Value, 2, Row, Column );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Fill word.
    This procedure fills the specified memory
    region with the given word.  Length is in
    words.

*************************************************)

    Procedure Fill_Word( Var Destination; Length, Source: Word );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Type
        Memory_Type = packed array[ 1 .. 66000 ] of Word;
      Var
        Count: Word;
        Destination_Data: Memory_Type absolute Destination;
      Begin
        For Count := 1 to Length do
          Destination_Data[ Count ] := Source;
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Exchange.
    This procedure exchanges the two given data
    structures.  Length is in bytes.

*************************************************)

    Procedure Exchange( Var Source, Destination; Length: Word );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Type
        Memory_Type = packed array[ 1 .. 66000 ] of Byte;
      Var
        Source_Data: Memory_Type absolute Source;
        Destination_Data: Memory_Type absolute Destination;
        Hold: Byte;
        Count: Word;
      Begin
        For Count := 1 to Length do
          Begin
            Hold := Source_Data[ Count ];
            Source_Data[ Count ] := Destination_Data[ Count ];
            Destination_Data[ Count ] := Hold;
          End;
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put screen long.
    This procedure puts the specified character
    and attribute on the screen then allows a
    repeat factor.

*************************************************)

    Procedure Put_Screen_Long( Row, Column: Byte; Character: Char; Attribute, Current_Page, Amount: Byte );
     {$IFNDEF OS2}
      External;
     {$ELSE}
      Var
        Count: Byte;
        Data: Area_Type;
        Cell: Cell_Type;
      Begin
        Cell.Character := Character;
        Cell.Attribute := Attribute;
        For Count := 1 to Amount do
          Data[ Count ] := Cell;
        VideoWriteCell( Data, Amount, Pred( Row ), Pred( Column ) );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get the mode.
    As previously defined.

*************************************************)

    Procedure Get_The_Mode;
     {$IFNDEF OS2}
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ax := $0F00;
        Intr( $10, The_Registers );
        The_Mode := The_Registers.Al;
        Screen_Column_Limit := The_Registers.Ah;
        Screen_Row_Limit := Original_Row_Limit;
        Page_Number := The_Registers.Bh;
        Case The_Mode of
           00,
           01: Case Page_Number of
                 00:  Video_Segment := $B800;
                 01:  Video_Segment := $B880;
                 02:  Video_Segment := $B900;
                 03:  Video_Segment := $B980;
                 04:  Video_Segment := $BA00;
                 05:  Video_Segment := $BA80;
                 06:  Video_Segment := $BB00;
                 07:  Video_Segment := $BB80;
                 ELSE Video_Segment := $B800;
               End; { Case }
           02,
           03: Case Page_Number of
                 00:  Video_Segment := $B800;
                 01:  Video_Segment := $B900;
                 02:  Video_Segment := $BA00;
                 03:  Video_Segment := $BB00;
                 ELSE Video_Segment := $B800;
               End; { Case }
           07: Video_Segment := $B000;
           else DirectVideo := False;
         End; { Case }
        Video_Offset := $0000;
      End;
     {$ELSE}
      Var
        VideoMode: Mode_Information_Type;
      Begin
        VideoGetMode( VideoMode );
        Page_Number := 0;
        Screen_Row_Limit := VideoMode.Text_Rows;
        Screen_Column_Limit := VideoMode.Text_Columns;
        If ( Screen_Column_Limit = 40 )
          then
            The_Mode := 0
          else
            The_Mode := 2;
        If ( ( VideoMode.Type_Mask and 4 ) = 0 )
          then
            If ( The_Mode = 0 )
              then
                The_Mode := 1
              else
                The_Mode := 3;
        If ( VideoMode.Bits_Per_Pixel = 0 )
          then
            The_Mode := 7;
        If ( Screen_Row_Limit > 25 )
          then
            The_Mode := The_Mode or Font8x8;
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write data.
    As previously defined.

*************************************************)

   {$IFNDEF OS2}
    Procedure Write_Data( Row, Column: Byte; Var Data; Amount: Word );
      Var
        Where: Byte;
        Counter: Word;
        The_Screen: Pointer;
        Memory: Area_Type absolute Data;
      Begin
        If ( Amount <> 0 ) and ( Row <= Screen_Row_Limit ) and ( Column <= Screen_Column_Limit ) and ( Row > 0 )
          then
            If ( not DirectVideo )
              then
                Begin
                  Where := Column;
                  For Counter := 1 to Amount do
                    Begin
                      With Memory[ Counter ] do
                        Put_Screen( Row, Where, Character, Attribute, Page_Number );
                      Inc( Where );
                    End;
                End
              else
                Begin
                  Counter := 1;
                  While ( Column < 1 ) do
                    Begin
                      Inc( Column );
                      Dec( Amount );
                      Inc( Counter );
                    End;
                  While ( Pred( Column + Amount ) > Screen_Column_Limit ) do
                    Dec( Amount );
                  If ( Amount > 0 )
                    then
                      Begin
                        The_Screen := Get_Screen_Address( Column, Row );
                        Screen_Move( Memory[ Counter ], The_Screen^, ( Amount * 2 ) );
                      End;
                End;
      End;
   {$ELSE}
    Procedure Write_Data( Row, Column: Byte; Data: Pointer; Amount: Word );
      Var
        Memory: Area_Type_Pointer;
      Begin
        Memory := Data;
        If ( Amount <> 0 ) and ( Row <= Screen_Row_Limit ) and ( Column <= Screen_Column_Limit ) and ( Row > 0 )
          then
            Put_Screen_Plus( Row, Column, Memory^[ 1 ], Amount );
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read data.
    As previously defined.

*************************************************)

   {$IFNDEF OS2}
    Procedure Read_Data( Row, Column: Byte; Var Data; Amount: Word );
      Var
        Where: Byte;
        Counter: Word;
        The_Screen: Pointer;
        Memory: Area_Type absolute Data;
      Begin
        If ( Amount <> 0 )
          then
            If ( not DirectVideo )
              then
                Begin
                  Where := Column;
                  For Counter := 1 to Amount do
                    Begin
                      With Memory[ Counter ] do
                        Get_Screen( Row, Where, Character, Attribute, Page_Number );
                      Inc( Where );
                    End;
                End
              else
                Begin
                  The_Screen := Get_Screen_Address( Column, Row );
                  Screen_Move( The_Screen^, Memory, ( Amount * 2 ) );
                End;
      End;
   {$ELSE}
    Procedure Read_Data( Row, Column: Byte; Data: Pointer; Amount: Word );
      Var
        Memory: Area_Type_Pointer;
      Begin
        Memory := Data;
        If ( Amount <> 0 )
          then
            Get_Screen_Plus( Row, Column, Memory^[ 1 ], Amount );
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Internal screen address.
    This procedure calculates the location of the
    internal screen address and then gives the
    value as an offset of the current screen
    segment.

*************************************************)

    Procedure Internal_Screen_Address( Row, Column: Byte; Var Offset: Word );
      Begin
        Dec( Column );
        Dec( Row );
        Offset := ( Video_Offset + ( ( ( Row * Screen_Column_Limit ) + Column ) * 2 ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Invalid.
    This function returns true only if the given
    row and column values are off the current
    screen.

*************************************************)

    Function Invalid( Row, Column: Byte ): Boolean;
      Begin
        Invalid := ( Column < 1 ) or ( Column > Screen_Column_Limit ) or ( Row > Screen_Row_Limit ) or ( Row < 1 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get character from screen.
    As previously defined.

*************************************************)

    Procedure Get_Character_From_Screen( Column, Row: Byte; Var Character: Char; Var Attribute: Byte );
      Var
        Data: Cell_Type;
        Offset: Word;
      Begin
       {$IFNDEF OS2}
        If Invalid( Row, Column )
          then
            Begin
              Character := #0;
              Attribute := $0;
            End
          else
            If ( not DirectVideo )
              then
                Get_Screen( Row, Column, Character, Attribute, Page_Number )
              else
                Begin
                  Internal_Screen_Address( Row, Column, Offset );
                  Screen_Move(  Mem[ Video_Segment: Offset ], Data, 2 );
                  Character := Data.Character;
                  Attribute := Data.Attribute;
                End;
       {$ELSE}
        If Invalid( Row, Column )
          then
            Begin
              Character := #0;
              Attribute := $0;
            End
          else
            Get_Screen( Row, Column, Character, Attribute, Page_Number );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put character on screen.
    As previously defined.

*************************************************)

    Procedure Put_Character_On_Screen( Column, Row: Byte; Character: Char; Attribute: Byte );
      Var
        Data: Cell_Type;
        Offset: Word;
      Begin
       {$IFNDEF OS2}
        If not Invalid( Row, Column )
          then
            If ( not DirectVideo )
              then
                Put_Screen( Row, Column, Character, Attribute, Page_Number )
              else
                Begin
                  Data.Character := Character;
                  Data.Attribute := Attribute;
                  Internal_Screen_Address( Row, Column, Offset );
                  Screen_Move( Data, Mem[ Video_Segment: Offset ], 2 );
               End;
       {$ELSE}
        If not Invalid( Row, Column )
          then
            Put_Screen( Row, Column, Character, Attribute, Page_Number );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write error.
    As previously defined.

*************************************************)

    Procedure Write_Error( Result: Word; Sentence: String );
      Begin
        If ( Result <> 0 )
          then
            Begin
              WriteLn( 'Error ', Result, ' in ', Sentence, '.' );
             {$IFNDEF VER40}
              RunError( Result );
             {$ELSE}
              Halt( Result );
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get screen address.
    As previously defined.

*************************************************)

    Function Get_Screen_Address( Column, Row: Byte ): Pointer;
      Var
        Offset: Word;
      Begin
        If Invalid( Row, Column )
          then
            Write_Error( 201, 'Get_Screen_Address' );
        Internal_Screen_Address( Row, Column, Offset );
       {$IFNDEF OS2}
        Get_Screen_Address := Addr( Mem[ Video_Segment: Offset ] );
       {$ELSE}
        Get_Screen_Address := Nil;
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Fix ranges.
    This procedure takes the given ranges and
    tries to truncate them to the current screen
    size.

*************************************************)

    Procedure Fix_Ranges( Var The_Left, The_Top, The_Right, The_Bottom: Byte );
      Begin
        If ( The_Right > Screen_Column_Limit )
          then
            The_Right := Screen_Column_Limit;
        If ( The_Bottom > Screen_Row_Limit )
          then
            The_Bottom := Screen_Row_Limit;
        If ( The_Left < 1 )
          then
            The_Left := 1;
        If ( The_Top < 1 )
          then
            The_Top := 1;
        If ( The_Left >= The_Right )
          then
            Write_Error( 201, 'Fix_Ranges: Left is greater than Right' );
        If ( The_Top >= The_Bottom )
          then
            Write_Error( 201, 'Fix_Ranges: Top is greater than Bottom' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick scroll up.
    This procedure scrolls the screen region up
    using the quick memory access method.

*************************************************)

   {$IFNDEF OS2}
    Procedure Quick_Scroll_Up( Left, Top, Right, Bottom: Byte );
      Var
        Count,
        Value1: Byte;
        Value2,
        Offset: Word;
      Begin
        Value1 := ( 2 * ( Succ( Right ) - Left ) );
        Value2 := ( 2 * Screen_Column_Limit );
        Internal_Screen_Address( Succ( Top ), Left, Offset );
        For Count := Succ( Top ) to Bottom do
          Begin
            Screen_Move( Mem[ Video_Segment: Offset ], Mem[ Video_Segment: ( Offset - Value2 ) ], Value1 );
            Offset := ( Offset + Value2 );
          End;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick scroll down.
    This procedure scrolls the screen region down
    using the quick memory access method.

*************************************************)

   {$IFNDEF OS2}
    Procedure Quick_Scroll_Down( Left, Top, Right, Bottom: Byte );
      Var
        Count,
        Value1: Byte;
        Value2,
        Offset: Word;
      Begin
        Value1 := ( 2 * ( Succ( Right ) - Left ) );
        Value2 := ( 2 * Screen_Column_Limit );
        Internal_Screen_Address( Pred( Bottom ), Left, Offset );
        For Count := Pred( Bottom ) downto Top do
          Begin
            Screen_Move( Mem[ Video_Segment: Offset ], Mem[ Video_Segment: ( Offset + Value2 ) ], Value1 );
            Offset := ( Offset - Value2 );
           End;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System scroll left.
    This procedure scrolls the screen region left
    using the standard system routines.

*************************************************)

    Procedure System_Scroll_Left( Left, Top, Right, Bottom: Byte );
      Var
        Row,
        Column,
        Attribute: Byte;
        Character: Char;
      Begin
        For Row := Top to Bottom do
          For Column := Succ( Left ) to Right do
            Begin
              Get_Screen( Row, Column, Character, Attribute, Page_Number );
              Put_Screen( Row, Pred( Column ), Character, Attribute, Page_Number );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick scroll left.
    This procedure scrolls the screen region left
    using the quick memory access method.

*************************************************)

    Procedure Quick_Scroll_Left( Left, Top, Right, Bottom: Byte );
     {$IFNDEF OS2}
      Var
        Row,
        Value1: Byte;
        Value2,
        Offset: Word;
      Begin
        Value1 := ( 2 * ( Right - Left ) );
        Value2 := ( Screen_Column_Limit * 2 );
        Internal_Screen_Address( Top, Succ( Left ), Offset );
        For Row := Top to Bottom do
          Begin
            Screen_Move( Mem[ Video_Segment: Offset ], Mem[ Video_Segment: Pred( Pred( Offset ) ) ], Value1 );
            Offset :=( Offset + Value2 );
          End;
      End;
     {$ELSE}
      Var
        Fill: Cell_Type;
      Begin
        Fill.Character := ' ';
        Fill.Attribute := TextAttr;
        VideoScrollLeft( Pred( Left ), Pred( Top ), Pred( Right ), Pred( Bottom ), 1, Fill );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System scroll right.
    This procedure scrolls the screen region right
    using the standard system routines.

*************************************************)

    Procedure System_Scroll_Right( Left, Top, Right, Bottom: Byte );
      Var
        Row,
        Column,
        Attribute: Byte;
        Character: Char;
      Begin
        For Row := Top to Bottom do
          For Column := Pred( Right ) downto Left do
            Begin
              Get_Screen( Row, Column, Character, Attribute, Page_Number );
              Put_Screen( Row, Succ( Column ), Character, Attribute, Page_Number );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick scroll right.
    This procedure scrolls the screen region right
    using the quick memory access method.

*************************************************)

    Procedure Quick_Scroll_Right( Left, Top, Right, Bottom: Byte );
     {$IFNDEF OS2}
      Var
        Row,
        Value1: Byte;
        Value2,
        Offset: Word;
      Begin
        Value1 := ( 2 * ( Right - Left ) );
        Value2 := ( Screen_Column_Limit * 2 );
        Internal_Screen_Address( Top, Left, Offset );
        For Row := Top to Bottom do
          Begin
            Screen_Move( Mem[ Video_Segment: Offset ], Work_Area, Value1 );
            Screen_Move( Work_Area, Mem[ Video_Segment: Succ( Succ( Offset ) ) ], Value1 );
            Offset :=( Offset + Value2 );
          End;
      End;
     {$ELSE}
      Var
        Fill: Cell_Type;
      Begin
        Fill.Character := ' ';
        Fill.Attribute := TextAttr;
        VideoScrollRight( Pred( Left ), Pred( Top ), Pred( Right ), Pred( Bottom ), 1, Fill );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll region up.
    As previously defined.

*************************************************)

    Procedure Scroll_Region_Up( Left, Top, Right, Bottom: Byte );
      Begin
        Fix_Ranges( Left, Top, Right, Bottom );
       {$IFNDEF OS2}
        If DirectVideo
          then
            Quick_Scroll_Up( Left, Top, Right, Bottom )
          else
            System_Scroll_Up( Left, Top, Right, Bottom );
       {$ELSE}
        System_Scroll_Up( Left, Top, Right, Bottom );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll region down.
    As previously defined.

*************************************************)

    Procedure Scroll_Region_Down( Left, Top, Right, Bottom: Byte );
      Begin
        Fix_Ranges( Left, Top, Right, Bottom );
       {$IFNDEF OS2}
        If DirectVideo
          then
            Quick_Scroll_Down( Left, Top, Right, Bottom )
          else
            System_Scroll_Down( Left, Top, Right, Bottom );
       {$ELSE}
        System_Scroll_Down( Left, Top, Right, Bottom );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll region left.
    As previously defined.

*************************************************)

    Procedure Scroll_Region_Left( Left, Top, Right, Bottom: Byte );
      Begin
        Fix_Ranges( Left, Top, Right, Bottom );
       {$IFNDEF OS2}
        If DirectVideo
          then
            Quick_Scroll_Left( Left, Top, Right, Bottom )
          else
            System_Scroll_Left( Left, Top, Right, Bottom )
       {$ELSE}
        {$IFDEF Quick}
        Quick_Scroll_Left( Left, Top, Right, Bottom );
        {$ELSE}
        System_Scroll_Left( Left, Top, Right, Bottom );
        {$ENDIF}
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll region right.
    As previously defined.

*************************************************)

    Procedure Scroll_Region_Right( Left, Top, Right, Bottom: Byte );
      Begin
        Fix_Ranges( Left, Top, Right, Bottom );
       {$IFNDEF OS2}
        If DirectVideo
          then
            Quick_Scroll_Right( Left, Top, Right, Bottom )
          else
            System_Scroll_Right( Left, Top, Right, Bottom );
       {$ELSE}
        {$IFDEF Quick}
        Quick_Scroll_Right( Left, Top, Right, Bottom );
        {$ELSE}
        System_Scroll_Right( Left, Top, Right, Bottom );
        {$ENDIF}
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Blank row.
    As previously defined.

*************************************************)

    Procedure Blank_Row( Row, Left, Right: Byte );
     {$IFNDEF OS2}
      Var
        Offset: Word;
        Data: Cell_Type;
        Column: Byte;
      Begin
        If DirectVideo
          then
            Begin
              Internal_Screen_Address( Row, Left, Offset );
              Data.Character := ' ';
              Data.Attribute := TextAttr;
              For Column := Left to Right do
                Begin
                  Screen_Move( Data, Mem[ Video_Segment: Offset ], 2 );
                  Inc( Offset, 2 );
                End;
            End
          else
            Put_Screen_Long( Row, Left, ' ', TextAttr, Page_Number, Succ( Right - Left ) );
      End;
     {$ELSE}
      Begin
        Put_Screen_Long( Row, Left, ' ', TextAttr, Page_Number, Succ( Right - Left ) );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up.
    This procedure sets up the variables for use
    by the procedure.

*************************************************)

    Procedure Set_Up( Var Left, Top, Right, Bottom: Byte );
      Begin
        Get_The_Mode;
        Left := Succ( Left_Of_Window^ );
        Top := Succ( Top_Of_Window^ );
        Right := Succ( Right_Of_Window^ );
        Bottom := Succ( Bottom_Of_Window^ );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll window up.
    As previously defined.

*************************************************)

    Procedure Scroll_Window_Up;
      Var
        Top,
        Left,
        Right,
        Bottom: Byte;
      Begin
        Set_Up( Left, Top, Right, Bottom );
        Scroll_Region_Up( Left, Top, Right, Bottom );
        If ( DirectVideo )
          then
            Blank_Row( Bottom, Left, Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll window down.
    As previously defined.

*************************************************)

    Procedure Scroll_Window_Down;
      Var
        Top,
        Left,
        Right,
        Bottom: Byte;
      Begin
        Set_Up( Left, Top, Right, Bottom );
        Scroll_Region_Down( Left, Top, Right, Bottom );
        If ( DirectVideo )
          then
            Blank_Row( Top, Left, Right );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Blank column.
    As previously defined.

*************************************************)

    Procedure Blank_Column( Column, Top, Bottom: Byte );
     {$IFNDEF OS2}
      Var
        Row: Byte;
        Value,
        Offset: Word;
        Data: Cell_Type;
      Begin
        If ( DirectVideo )
          then
            Begin
              Internal_Screen_Address( Top, Column, Offset );
              Data.Character := ' ';
              Data.Attribute := TextAttr;
              Value := ( 2 * Screen_Column_Limit );
              For Row := Top to Bottom do
                Begin
                  Screen_Move( Data, Mem[ Video_Segment: Offset ], 2 );
                  Value := ( 2 * Screen_Column_Limit );
                  Offset := ( Offset + Value );
                End;
            End
          else
            For Row := Top to Bottom do
              Put_Screen( Row, Column, ' ', TextAttr, Page_Number );
      End;
     {$ELSE}
      Var
        Row: Byte;
      Begin
        For Row := Top to Bottom do
          Put_Screen( Row, Column, ' ', TextAttr, Page_Number );
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System scroll window left.
    This procedure scrolls the window left by
    using the system routines.

*************************************************)

    Procedure System_Scroll_Window_Left( Top, Left, Bottom, Right: Byte );
      Var
        Row,
        Column,
        Attribute: Byte;
        Character: Char;
      Begin
        For Row := Top to Bottom do
          Begin
            For Column := Succ( Left ) to Right do
              Begin
                Get_Screen( Row, Column, Character, Attribute, Page_Number );
                Put_Screen( Row, Pred( Column ), Character, Attribute, Page_Number );
              End;
            Put_Screen( Row, Right, ' ', TextAttr, Page_Number );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick scroll window left.
    This procedure scrolls the window left by
    using the memory accessing procedures.

*************************************************)

   {$IFNDEF OS2}
    Procedure Quick_Scroll_Window_Left( Top, Left, Bottom, Right: Byte );
      Var
        Row,
        Value1: Byte;
        Value2,
        Offset1,
        Offset2: Word;
        Data: Cell_Type;
      Begin
        Value1 := ( 2 * ( Right - Left ) );
        Value2 := ( 2 * Screen_Column_Limit );
        Data.Character := ' ';
        Data.Attribute := TextAttr;
        Internal_Screen_Address( Top, Succ( Left ), Offset1 );
        Internal_Screen_Address( Top, Right, Offset2 );
        For Row := Top to Bottom do
          Begin
            Screen_Move( Mem[ Video_Segment: Offset1 ], Mem[ Video_Segment: Pred( Pred( Offset1 ) ) ], Value1 );
            Offset1 := ( Offset1 + Value2 );
            Screen_Move( Data, Mem[ Video_Segment: Offset2 ], 2 );
            Offset2 := ( Offset2 + Value2 );
          End;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll window left.
    As previously defined.

*************************************************)

    Procedure Scroll_Window_Left;
      Var
        Top,
        Left,
        Right,
        Bottom: Byte;
      Begin
        Set_Up( Left, Top, Right, Bottom );
       {$IFNDEF OS2}
        If DirectVideo
          then
            Quick_Scroll_Window_Left( Top, Left, Bottom, Right )
          else
            System_Scroll_Window_Left( Top, Left, Bottom, Right );
       {$ELSE}
        System_Scroll_Window_Left( Top, Left, Bottom, Right );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System scroll window right.
    This procedure scrolls the window right by
    using the system routines.

*************************************************)

    Procedure System_Scroll_Window_Right( Top, Left, Bottom, Right: Byte );
      Var
        Row,
        Column,
        Attribute: Byte;
        Character: Char;
      Begin
        For Row := Top to Bottom do
          Begin
            For Column := Pred( Right ) downto Left do
              Begin
                Get_Screen( Row, Column, Character, Attribute, Page_Number );
                Put_Screen( Row, Succ( Column ), Character, Attribute, Page_Number );
              End;
            Put_Screen( Row, Left, ' ', TextAttr, Page_Number );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick scroll window right.
    This procedure scrolls the window right by
    using the memory accessing procedures.

*************************************************)

   {$IFNDEF OS2}
    Procedure Quick_Scroll_Window_Right( Top, Left, Bottom, Right: Byte );
      Var
        Row,
        Value1: Byte;
        Value2,
        Offset: Word;
        Data: Cell_Type;
      Begin
        Value1 := ( 2 * ( Right - Left ) );
        Value2 := ( 2 * Screen_Column_Limit );
        Data.Character := ' ';
        Data.Attribute := TextAttr;
        Internal_Screen_Address( Top, Left, Offset );
        For Row := Top to Bottom do
          Begin
            Screen_Move( Mem[ Video_Segment: Offset ], Work_Area, Value1 );
            Screen_Move( Work_Area, Mem[ Video_Segment: Succ( Succ( Offset ) ) ], Value1 );
            Screen_Move( Data, Mem[ Video_Segment: Offset ], 2 );
            Offset := ( Offset + Value2 );
          End;
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll window right.
    As previously defined.

*************************************************)

    Procedure Scroll_Window_Right;
      Var
        Top,
        Left,
        Right,
        Bottom: Byte;
      Begin
        Set_Up( Left, Top, Right, Bottom );
       {$IFNDEF OS2}
        If DirectVideo
          then
            Quick_Scroll_Window_Right( Top, Left, Bottom, Right )
          else
            System_Scroll_Window_Right( Top, Left, Bottom, Right );
       {$ELSE}
        System_Scroll_Window_Right( Top, Left, Bottom, Right );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System change attributes.
    This procedure changes the specified screen
    attributes by using the system routines.

*************************************************)

    Procedure System_Change_Attributes( Row, Column, Length, New_Attribute: Byte );
      Var
        Counter: Byte;
      Begin
       {$IFNDEF OS2}
        For Counter := 1 to Length do
          Begin
            Change_Attribute( Row, Column, New_Attribute, Page_Number );
            Inc( Column );
          End;
       {$ELSE}
        VideoWriteAttribute( New_Attribute, Length, Pred( Row ), Pred( Column ) );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick change attributes.
    This procedure changes the specified screen
    attributes by using memory access routines.

*************************************************)

   {$IFNDEF OS2}
    Procedure Quick_Change_Attributes( Row, Column, Length, New_Attribute: Byte );
      Var
        Counter,
        Direct_Length: Byte;
        The_Screen: Pointer;
      Begin
        Direct_Length := ( 2 * Length );
        The_Screen := Get_Screen_Address( Column, Row );
        Screen_Move( The_Screen^, Work_Area, Direct_Length );
        For Counter := 1 to Length do
          Work_Area[ Counter ].Attribute := New_Attribute;
        Screen_Move( Work_Area, The_Screen^, Direct_Length );
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Change screen attributes.
    As previously defined.

*************************************************)

    Procedure Change_Screen_Attribute( Row, Column, Length, Attribute: Byte );
      Begin
        Get_The_Mode;
        While ( Column < 1 ) do
          Begin
            Inc( Column );
            Dec( Length );
          End;
        If ( ( Row <= Screen_Row_Limit ) and ( Column <= Screen_Column_Limit ) and
             ( Row > 0 ) and ( Column > 0 ) and ( Length > 0 ) )
          then
            Begin
              While ( Pred( Column + Length ) > Screen_Column_Limit ) do
                Dec( Length );
             {$IFNDEF OS2}
              If DirectVideo
                then
                  Quick_Change_Attributes( Row, Column, Length, Attribute )
                else
                  System_Change_Attributes( Row, Column, Length, Attribute );
             {$ELSE}
              System_Change_Attributes( Row, Column, Length, Attribute )
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System dim attributes.
    This procedure alters the specified screen
    attributes by using the system routines.

*************************************************)

    Procedure System_Dim_Attributes( Row, Column, Length, Attribute: Byte );
      Var
        Counter: Byte;
      Begin
        Dec( Row );
        Dec( Column );
        For Counter := 1 to Length do
          Begin
            Dim_Attribute( Row, Column, Attribute, Page_Number );
            Inc( Column );
          End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick dim attributes.
    This procedure alters the specified screen
    attributes by using the memory accessing
    routines.

*************************************************)

   {$IFNDEF OS2}
    Procedure Quick_Dim_Attributes( Row, Column, Length, New_Attribute: Byte );
      Var
        Counter,
        Direct_Length: Byte;
        The_Screen: Pointer;
      Begin
        Direct_Length := ( 2 * Length );
        The_Screen := Get_Screen_Address(  Column , Row );
        Screen_Move( The_Screen^, Work_Area, Direct_Length );
        For Counter := 1 to Length do
          With Work_Area[ Counter ] do
            Attribute := ( ( Attribute and $80 ) or New_Attribute );
        Screen_Move( Work_Area, The_Screen^, Direct_Length );
      End;
   {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dim screen attributes.
    As previously defined.

*************************************************)

    Procedure Dim_Screen_Attribute( Row, Column, Length, Attribute: Byte );
      Begin
        Get_The_Mode;
        While ( Column < 1 ) do
          Begin
            Inc( Column );
            Dec( Length );
          End;
        If ( not Invalid( Row, Column ) and ( Length > 0 ) )
          then
            Begin
              While ( Pred( Column + Length ) > Screen_Column_Limit ) do
                Dec( Length );
             {$IFNDEF OS2}
              If DirectVideo
                then
                  Quick_Dim_Attributes( Row, Column, Length, Attribute )
                else
                  System_Dim_Attributes( Row, Column, Length, Attribute );
             {$ELSE}
              System_Dim_Attributes( Row, Column, Length, Attribute )
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Change window attributes.
    As previously defined.

*************************************************)

    Procedure Change_Window_Attribute( Row, Column, Length, Attribute: Byte );
      Var
        Screen_Width,
        Screen_Height: Byte;
      Begin
        Screen_Width := Succ( Right_Of_Window^ - Left_Of_Window^ );
        Screen_Height := Succ( Bottom_Of_Window^ - Top_Of_Window^ );
        If ( ( Column <= Screen_Width ) and ( Row <= Screen_Height ) and ( Row > 0 ) and ( Column > 0 ) )
          then
            Begin
              While ( Pred( Column + Length ) > Screen_Width ) do
                Dec( Length );
              Change_Screen_Attribute( ( Row + Top_Of_Window^ ), ( Column + Left_Of_Window^ ), Length, Attribute );
            End;
      End;

