{////////////////////////////////////////////////////////////////////////////}

(*************************************************

  Procedure: Write error.
    This procedure writes out the error message
    if something went wrong and ends the program.
    Otherwise, it returns.

*************************************************)

    Procedure Write_Error( Error_Code: Word; Sentence: String );
      Begin
        If ( Error_Code <> 0 )
          then
            Begin
              WriteLn( 'Error ', Error_Code,' in ', Sentence, '.' );
             {$IFNDEF Ver40}
              RunError( Error_Code );
             {$ELSE}
              Halt( Error_Code );
             {$ENDIF}
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read Info.
    This procedure reads the info data in the file
    into the Info record.  It is a low level
    procedure.

*************************************************)

    Procedure Read_Info( Var Tree: Tree_Type );
      Begin
       {$I-}
        Seek( Tree.The_File, 0 );
        Write_Error( IOResult, 'Read_Info: Seek' );
        BlockRead( Tree.The_File, Tree.Info, SizeOf( Info_Type ) );
        Write_Error( IOResult, 'Read_Info: Read' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write info.
    This procedure writes the info record out to
    the disk file.  It is a low level procedure.

*************************************************)

    Procedure Write_Info( Var Tree: Tree_Type );
      Begin
       {$I-}
        Seek( Tree.The_File, 0 );
        Write_Error( IOResult, 'Write_Info: Seek' );
        BlockWrite( Tree.The_File, Tree.Info, SizeOf( Info_Type ) );
        Write_Error( IOResult, 'Write_Info: Write' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Perform read.
    This procedure reads the data into the given
    buffer.  It is a low level procedure.

*************************************************)

    Procedure Perform_Read( Var Tree: Tree_Type; Address: Point_Type; Var Buffer: Node_Type );
      Begin
       {$I-}
        Seek( Tree.The_File, Address );
        Write_Error( IOResult, 'Perform_Read: Seek' );
        BlockRead( Tree.The_File, Buffer, Tree.Info.Node_Size );
        Write_Error( IOResult, 'Perform_Read: Read' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Perform write.
    This procedure writes the buffer data into the
    disk file.  It is a low level procedure.

*************************************************)

    Procedure Perform_Write( Var Tree: Tree_Type; Address: Point_Type; Var Buffer: Node_Type );
      Begin
       {$I-}
        Seek( Tree.The_File, Address );
        Write_Error( IOResult, 'Perform_Write: Seek' );
        BlockWrite( Tree.The_File, Buffer, Tree.Info.Node_Size );
        Write_Error( IOResult, 'Perform_Write: Write' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Perform reset.
    This procedure performs a reset on the tree
    file.  It essentially flushes out the file
    and commits it to the disk.

*************************************************)

    Procedure Perform_Reset( Var Tree: Tree_Type );
      Begin
       {$I-}
        Write_Info( Tree );
        Tree.Pointer := Null;
        Reset( Tree.The_File, 1 );
        Write_Error( IOResult, 'Perform_Reset' );
       {$I+}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get the offset.
    This function calculates the offset in the
    tree node data part where the particular data
    is destined to reside.  This calculation must
    be precise, or one data record will corrupt
    another.

*************************************************)

    Function Get_Offset( Var Tree: Tree_Type; Location: Position ): Word;
      Var
        Count: Word;
      Begin
        { Both assumed to begin at 1 }
        Count := Succ( Pred( Location ) * Tree.Info.Record_Size );
        Get_Offset := ( ( SizeOf( Point_Type ) * Succ( Tree.Info.Maximum ) ) + Count );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get the pointer.
    This function gets the pointer for the
    requested location in the address record.
    This is a mid level function.

*************************************************)

    Function Get_Pointer( Var Tree: Tree_Type; Address: Point_Type; Location: Position ): Point_Type;
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Get_Previous_Node_Pointer: Invalid address for pointer' );
        If ( Address <> Tree.Pointer )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Get_Pointer := Tree.Buffer.Branch[ Location ];
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get the previous pointer.
    This function returns the pointer to the
    previous node of the node of address.
    This is a mid level function.

*************************************************)

    Function Get_Previous( Var Tree: Tree_Type; Address: Point_Type ): Point_Type;
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Get_Previous_Node_Pointer: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Get_Previous := Tree.Buffer.Previous;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get the data.
    This procedure returns the data that is stored
    in the record of address at the given
    location.
    This is a mid level function.

*************************************************)

    Procedure Get_Data( Var Tree: Tree_Type; Address: Point_Type; Location: Position; Var Data: Data_Type );
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Get_Data: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Move( Tree.Buffer.Data[ Get_Offset( Tree, Location ) ], Data, Tree.Info.Record_Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get the count.
    This function returns the count of records in
    the requested node.
    This is a mid level function.

*************************************************)

    Function Get_Count( Var Tree: Tree_Type; Address: Point_Type ): Position;
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Get_Count: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Get_Count := Tree.Buffer.Count;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get the data and pointer.
    This procedure returns the data and pointer
    that is stored in the record of address at the
    given location.
    This is a mid level function.

*************************************************)

    Procedure Get_Data_And_Pointer( Var Tree: Tree_Type; Address: Point_Type; Location: Position; Var Data: Data_Type;
                                    Var Old_Pointer: Point_Type );
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Get_Data: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Move( Tree.Buffer.Data[ Get_Offset( Tree, Location ) ], Data, Tree.Info.Record_Size );
        Old_Pointer := Tree.Buffer.Branch[ Location ];
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in pointer.
    This procedure changes the requested pointer.
    This is a mid level procedure.

*************************************************)

    Procedure Put_Pointer( Var Tree: Tree_Type; Address: Point_Type; Location: Position; New_Value: Point_Type );
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Put_Previous_Node_Pointer: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Tree.Buffer.Branch[ Location ] := New_Value;
        Perform_Write( Tree, Address, Tree.Buffer );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in previous.
    This procedure changes the requested previous
    pointer.
    This is a mid level procedure.

*************************************************)

    Procedure Put_Previous( Var Tree: Tree_Type; Address, New_Value: Point_Type );
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Put_Previous_Node_Pointer: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Tree.Buffer.Previous := New_Value;
        Perform_Write( Tree, Address, Tree.Buffer );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in data.
    This procedure changes the data in the
    requested location.
    This is a mid level procedure.

*************************************************)

    Procedure Put_Data( Var Tree: Tree_Type; Address: Point_Type; Location: Position; Var Data: Data_Type );
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Put_Data: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Move( Data, Tree.Buffer.Data[ Get_Offset( Tree, Location ) ], Tree.Info.Record_Size );
        Perform_Write( Tree, Address, Tree.Buffer );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in count.
    This procedure changes the count in the
    specified procedure.
    This is a mid level procedure.

*************************************************)

    Procedure Put_Count( Var Tree: Tree_Type; Address: Point_Type; New_Count: Position );
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Put_Data: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Tree.Buffer.Count := New_Count;
        Perform_Write( Tree, Address, Tree.Buffer );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in data and pointer.
    This procedure changes the data and pointer
    in the requested location.
    This is a mid level procedure.

*************************************************)

    Procedure Put_Data_And_Pointer( Var Tree: Tree_Type; Address: Point_Type; Location: Position;
                                    Var Data: Data_Type; New_Pointer: Point_Type );
      Begin
        If ( Address = Null )
          then
            Write_Error( 204, 'Put_Data_And_Pointer: Invalid address for pointer' );
        If ( Tree.Pointer <> Address )
          then
            Begin
              Perform_Read( Tree, Address, Tree.Buffer );
              Tree.Pointer := Address;
            End;
        Move( Data, Tree.Buffer.Data[ Get_Offset( Tree, Location ) ], Tree.Info.Record_Size );
        Tree.Buffer.Branch[ Location ] := New_Pointer;
        Perform_Write( Tree, Address, Tree.Buffer );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scoot everything back.
    This procedure moves everything to the left
    from the specified starting point.
    This is a mid level procedure.

*************************************************)

  Procedure Scoot_Back( Var Tree: Tree_Type; Where: Point_Type; Start: Position );
    Var
      Index: Position;
    Begin
      If ( Where = Null )
        then
          Write_Error( 204, 'Scoot_Back: Invalid address for pointer' );
      If ( Tree.Pointer <> Where )
        then
          Begin;
            Perform_Read( Tree, Where, Tree.Buffer );
            Tree.Pointer := Where;
          End;
      Dec( Tree.Buffer.Count );
     {$IFNDEF Quick}
      For Index := Start to Tree.Buffer.Count do
        Begin
          Move( Tree.Buffer.Data[ Get_Offset( Tree, Succ( Index ) ) ],
                Tree.Buffer.Data[ Get_Offset( Tree, Index ) ],
                Tree.Info.Record_Size );
          Tree.Buffer.Branch[ Index ] := Tree.Buffer.Branch[ Succ( Index ) ];
        End;
     {$ELSE}
      Index := Succ( Tree.Buffer.Count - Start );
      Move( Tree.Buffer.Branch[ Succ( Start ) ], Tree.Buffer.Branch[ Start ],
            ( Index * SizeOf( Point_Type ) ) );
      Move( Tree.Buffer.Data[ Get_Offset( Tree, Succ( Start ) ) ],
            Tree.Buffer.Data[ Get_Offset( Tree, Start ) ],
            ( Index * Tree.Info.Record_Size ) );
     {$ENDIF}
      Perform_Write( Tree, Where, Tree.Buffer );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scoot everything up.
    This procedure moves everything to the right
    from the specified starting point to the
    specified ending point.
    This is a mid level procedure.

*************************************************)

  Procedure Scoot_Up( Var Tree: Tree_Type; Where: Point_Type; Finish: Position; New_Pointer: Point_Type;
                      Var Data: Data_Type );
    Var
      Index: Position;
    Begin
      If ( Where = Null )
        then
          Write_Error( 204, 'Scoot_Up: Invalid address for pointer' );
      If ( Tree.Pointer <> Where )
        then
          Begin;
            Perform_Read( Tree, Where, Tree.Buffer );
            Tree.Pointer := Where;
          End;
     {$IFNDEF Quick}
      For Index := Tree.Buffer.Count downto Finish do
        Begin
          Move( Tree.Buffer.Data[ Get_Offset( Tree, Index ) ],
                Tree.Buffer.Data[ Get_Offset( Tree, Succ( Index ) ) ],
                Tree.Info.Record_Size );
          Tree.Buffer.Branch[ Succ( Index ) ] := Tree.Buffer.Branch[ Index ];
        End;
     {$ELSE}
      Index := Succ( Tree.Buffer.Count - Finish );
      Move( Tree.Buffer.Branch[ Finish ],
            Tree.Buffer.Branch[ Succ( Finish ) ],
            ( Index * SizeOf( Point_Type ) ) );
      Move( Tree.Buffer.Data[ Get_Offset( Tree, Finish ) ],
            Tree.Buffer.Data[ Get_Offset( Tree, Succ( Finish ) ) ],
            ( Index * Tree.Info.Record_Size ) );
     {$ENDIF}
      Tree.Buffer.Branch[ Finish ] := New_Pointer;
      Move( Data, Tree.Buffer.Data[ Get_Offset( Tree, Finish ) ], Tree.Info.Record_Size );
      Inc( Tree.Buffer.Count );
      Perform_Write( Tree, Where, Tree.Buffer );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Increment Count.
    This procedure increments the count of the
    given node.
    This is a mid level procedure.

*************************************************)

  Procedure Increment_Count( Var Tree: Tree_Type; Where: Point_Type );
    Begin
      If ( Where = Null )
        then
          Write_Error( 204, 'Increment_Count: Invalid address for pointer' );
      If ( Tree.Pointer <> Where )
        then
          Begin;
            Perform_Read( Tree, Where, Tree.Buffer );
            Tree.Pointer := Where;
          End;
      Inc( Tree.Buffer.Count );
      Perform_Write( Tree, Where, Tree.Buffer );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Decrement Count.
    This procedure Decrements the count of the
    given node.
    This is a mid level procedure.

*************************************************)

  Procedure Decrement_Count( Var Tree: Tree_Type; Where: Point_Type );
    Begin
      If ( Where = Null )
        then
          Write_Error( 204, 'Decrement_Count: Invalid address for pointer' );
      If ( Tree.Pointer <> Where )
        then
          Begin;
            Perform_Read( Tree, Where, Tree.Buffer );
            Tree.Pointer := Where;
          End;
      Dec( Tree.Buffer.Count );
      Perform_Write( Tree, Where, Tree.Buffer );
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Make new pointer.
    This procedure creates a new pointer
    for the calling routine.

*************************************************)

    Procedure Make_New_Pointer( Var Tree: Tree_Type; Var Pointer: Point_Type );
      Var
        Data_Node: Node_Type;
      Begin
        If ( DiskFree( Tree.Default_Drive ) >= SizeOf( Tree_Type ) )
          then
            Begin
              Pointer := FileSize( Tree.The_File );
              Perform_Write( Tree, Pointer, Data_Node );
            End
          else
            Begin
              WriteLn( 'Error in New_Pointer: Out of disk space.' );
              WriteLn( 'Current data will be saved.' );
              Close( Tree.The_File );
              Halt;
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: New pointer.
    This procedure finds or creates a new pointer
    for the calling routine.  It will take the
    pointer out of the free list, if it exists,
    otherwise, it tries to create one.

*************************************************)

    Procedure New_Pointer( Var Tree: Tree_Type; Var Pointer: Point_Type );
      Begin
        If ( Tree.Info.Free = Null )
          then
            Make_New_Pointer( Tree, Pointer )
          else
            Begin
              Pointer := Tree.Info.Free;
              Tree.Info.Free := Get_Previous( Tree, Tree.Info.Free );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dispose of the pointer.
    This procedure takes the pointer and puts it
    in the free list for later possible reuse.

*************************************************)

    Procedure Dispose_Pointer( Var Tree: Tree_Type; Var Pointer: Point_Type );
      Begin
        Put_Count( Tree, Pointer, 0 );
        Put_Previous( Tree, Pointer, Tree.Info.Free );
        Tree.Info.Free := Pointer;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Equal default.
    This function returns true only if data1's
    key is equal to data2's key.

*************************************************)

    Function Equal_Default( Var Data_1, Data_2; Start, Finish: Word ): Boolean;
      Var
        Counter: Word;
        Data1: Data_Type absolute Data_1;
        Data2: Data_Type absolute Data_2;
      Begin
        Counter := Start;
        While ( ( Counter <= Finish ) and ( Data1[ Counter ] = Data2[ Counter ] ) ) do
          Inc( Counter );
        Equal_Default := ( Counter > Finish );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Less than default.
    This function compares data1 to data2 and
    returns true only if data1's key is less than
    data2's key.

*************************************************)

    Function Less_Than_Default( Var Data_1, Data_2; Start, Finish: Word ): Boolean;
      Var
        Counter: Word;
        Data1: Data_Type absolute Data_1;
        Data2: Data_Type absolute Data_2;
      Begin
        Counter := Start;
        While ( ( Counter <= Finish ) and ( Data1[ Counter ] = Data2[ Counter ] ) ) do
          Inc( Counter );
        Less_Than_Default := ( Counter <= Finish ) and ( Data1[ Counter ] < Data2[ Counter ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize tree.
    This procedure performs the actual
    initializing of the tree record.

*************************************************)

    Procedure Initialize_Tree( Var Tree: Tree_Type; Record_Size, Node_Size: Word );
      Begin
        If ( Node_Size > SizeOf( Node_Type ) )
          then
            Write_Error( 201, 'Initialize_Tree: Record size too large' );
        With Tree do
          Begin
            Info.Top := Null;
            Info.Free := Null;
            Info.Identity := Identity;
            Info.Record_Size := Record_Size;
            Info.Node_Size := Node_Size;
            Info.Start := 1;
            Info.Finish := Record_Size;
            Pointer := Null;
           {$IFNDEF VER40}
            Equal := Equal_Default;
            Less_Than := Less_Than_Default;
           {$ENDIF}
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Open tree file.
    As previously defined.

*************************************************)

    Procedure Open_Tree_File( Var Tree: Tree_Type; File_Name: String; Record_Size, Key_Offset, Key_Length: Word );
      Var
        Node_Size,
        Records_Per_Node: Word;
      Begin
        Records_Per_Node := Succ( Maximum_Records_Per_Node );
        Repeat
          Dec( Records_Per_Node );
          Node_Size := ( ( Record_Size * Records_Per_Node ) + { Data }
                         ( SizeOf( Point_Type ) * Succ( Records_Per_Node ) ) + { Branches }
                         SizeOf( Point_Type ) + { Previous pointer }
                         SizeOf( Position ) { Count } );
        Until ( Node_Size < Node_Maximum_Length );
        If ( Records_Per_Node < Minimum_Records_Per_Node )
          then
            Write_Error( 201, 'Open_Tree_File_Automatic: Records_Size to large.' );
        { Minimum allowable value for Maximum }
        Tree.Info.Maximum := Records_Per_Node;
        Tree.Info.Minimum := ( Records_Per_Node div 2 );
        Initialize_Tree( Tree, Record_Size, Node_Size );
        Tree.Info.Start := Key_Offset;
        Tree.Info.Finish := Pred( Key_Length + Key_Offset );
        If ( Key_Length < Minimum_Key_Size )
          then
            Write_Error( 201, 'Open_Tree_File: Key size is too small for system' );
        If ( Tree.Info.Start > Record_Size )
          then
            Write_Error( 201, 'Open_Tree_File: Key offset beyond record size' );
        If ( Tree.Info.Finish > Record_Size )
          then
            Write_Error( 201, 'Open_Tree_File: Key size beyond end of record' );
        If ( Tree.Info.Start > Tree.Info.Finish )
          then
            Write_Error( 201, 'Open_Tree_File: Key size invalid' );
        If ( Record_Size > Maximum_Record_Size )
          then
            Write_Error( 201, 'Open_Tree_File: Record size too large' );
        {$I-}
        If ( File_Name[ 2 ] = ':' )
          then
            Tree.Default_Drive := Succ( Ord( UpCase( File_Name[ 1 ] ) ) - Ord( 'A' ) )
          else
            Tree.Default_Drive := 0;
        Assign( Tree.The_File, File_Name );
        Reset( Tree.The_File, 1 );
        Case IOResult of
          0: Begin
               Read_Info( Tree );
               Tree.Pointer := Null;
             End;
          2: Begin
               Rewrite( Tree.The_File, 1 );
               Write_Info( Tree );
               Tree.Pointer := Null;
               Close( Tree.The_File );
               Reset( Tree.The_File, 1 );
             End;
         {$IFDEF OS2}
          110: Begin
                 Rewrite( Tree.The_File, 1 );
                 Write_Info( Tree );
                 Tree.Pointer := Null;
                 Close( Tree.The_File );
                 Reset( Tree.The_File, 1 );
               End;
         {$ENDIF}
          else Write_Error( IOResult, 'Open_Tree_File' );
        End; { Case }
        {$I+}
        If ( Tree.Info.Identity <> Identity )
          then
            Write_Error( 5, 'Open_Tree_File: File not of tree type' );
        If ( Tree.Info.Record_Size <> Record_Size )
          then
            Write_Error( 5, 'Open_Tree_File: Record sizes are unmatched' );
        If ( Tree.Info.Node_Size <> Node_Size )
          then
            Write_Error( 5, 'Open_Tree_File: Node sizes are unmatched' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Close tree file.
    As previously defined.

*************************************************)

    Procedure Close_Tree_File( Var Tree: Tree_Type );
      Begin
        Perform_Reset( Tree );
        Close( Tree.The_File );
      End;

{////////////////////////////////////////////////////////////////////////////}
