{ STFLIB.PAS -- String function library

  Title   : STFLIB
  Version : 2.1
  Language: Borland Turbo Pascal 4.0 through 7.0 (all targets)
            Borland Delphi 1.0 for Windows
  Date    : Jan 08,1997
  Author  : J.R. Ferguson
  Usage   : Unit

  Derived from StpLib with some enhancements.
  Function arguments are left unchanged, unless stated otherwise.
  The function result always holds the new string.
  Declarations of MaxStp, StpTyp, StpPtr and StpInd are in DEFLIB.PAS.
  Character positions are numbered starting with 1.
}

UNIT StfLib;

INTERFACE
uses DefLib, StpLib;

function StfAfter(src, pat: StpTyp): StpTyp;
{ Extracts the part from string src that comes after pattern pat.
  If pat is not a part of src, an empty string is returned.

  See also: <StfBefore>, <StfRight>, <StfSub>
}

function StfBefore(src, pat: StpTyp): StpTyp;
{ Extracts the part from string src that comes before pattern pat.
  If pat is not a part of src, string src is returned unaltered.

  See also: <StfAfter>, <StfRight>, <StfSub>
}

function StfCat(s1, s2: StpTyp): StpTyp;
{ Concatenates strings s1 and s2 (appends s2 after s1).

  See also: <StfNCat>, <StfcCat>
}

function StfCenter(s: StpTyp; n: StpInd): StpTyp;
{ Centers string s to be printed on a line of n characters wide by
  inserting spaces in front of it.
}

function StfcCat(s: StpTyp; c: char): StpTyp;
{ Appends character c to string s.

  See also: <StfCat>, <StfNCat>
}

function StfcCpy(c: char): StpTyp;
{ Returns a string consisting of 1 character c.

  See also: <StfNCpy>, <StfStzCpy>
}

function StfcIns(s:StpTyp; c:char; i:StpInd): StpTyp;
{ Inserts character c into string s at position i.
  If i=0 the character will be inserted at the front.
  If i>StpLen(s) the character is appended to string s.

  See also: <StfIns>, <StfNIns>
}

function StfcUpd(s: StpTyp; c: char; i: StpInd): StpTyp;
{ Replaces the character at position i in string s with character c.
  If i=0 or i>StpLen(s), string s is returned unaltered.

  See also: <StfRepl>, <StfNRepl>
}

function StfDel(s: StpTyp; i,n : StpInd): StpTyp;
{ Deletes n characters from string s, starting at position i.
  If i=0, the result is the same as if i=1. }

function StfDetab(src: StpTyp; n: StpInd): StpTyp;
{ Expands tabs in string src into space groups, using n as the tab field
  width.

  This function recognizes the following control characters:
    HT : Horizontal tab : expand to spaces.
    BS : Back space     : decrement column position by 1.
    CR : Carriage return: reset column position to 1.
  Other characters with an ordinal value in the range 0..31 are considered
  as non-printable. They are copied without change, but don't alter the
  current column position.
  Characters with an ordinal value in the range 128..255 are considered to
  be printable, so they are copied and increment the column position by 1.

  Remarks:
  - The column positioning may be at fault when string src containts BS
    characters immediately following HT or other control characters.
  - If n=0 string src is returned unaltered.

  See also: <StfEntab> 
}

function StfEntab(src: StpTyp; n: StpInd): StpTyp;
{ Replaces space groups in src by horizontal tab characters, using
  multiples of n as tab columns. Single spaces are never replaced by tabs.

  This function recognizes the following control characters:
    HT : Horizontal tab : expand to spaces.
    BS : Back space     : decrement column position by 1.
    CR : Carriage return: reset column position to 1.
  Other characters with an ordinal value in the range 0..31 are considered
  as non-printable. They are copied without change, but don't alter the
  current column position.
  Characters with an ordinal value in the range 128..255 are considered to
  be printable, so they are copied and increment the column position by 1.

  Remarks:
  - The column positioning may be at fault when string src containts BS
    characters immediately following a space group, a HT or another control
    character.
  - If n=0 the string src is returned unaltered.

  See also: <StfDetab> }

function StfFill(s: StpTyp; c: char; n: StpInd): StpTyp;
{ Fills (lengthens) string s to a length of n, by appending characters c.
  If n < StpLen(s) string s is returned unaltered.

  See also: <StfMake>
}

function StfGtw(var s: StpTyp): StpTyp;
{ Returns the first word from string s and removes this word from s.
  If string s does not contain a word, an empty string is returned, and
  s itself is also made empty.

  The folowing ASCII characters are seen as word separators:
  carriage return, line feed, form feed, horizontal tab, vertical tab
  and space.
}

function StfIns(dst, src: StpTyp; i: StpInd): StpTyp;
{ Returns a copy of string dst, after inserting string src at position i
  of dst.
  If i=0 string src will be inserted in front.
  If i>StpLen(src), string src will be appended to dst.

  See also: <StfNIns>, <StfcIns>
}

function StfLexSrt(s: StpTyp; n: StpInd): StpTyp;
{ Like StfSrt, using the ChrLib.LexOrder character order.

  See also: <StfSrt>, <StfUppSrt>
}

function StfLow(s: StpTyp): StpTyp;
{ Returns string s after converting all upper case letters to lower case.

  See also: <StfUpp>
}

function StfMake(c: char; n: StpInd): StpTyp;
{ Returns a string consisting of n chacarters c.

  See also: <StfFill>
}

function StfNCat(s1, s2: StpTyp; n: StpInd): StpTyp;
{ Appends a maximum of n characters from string s2 to string s1.

  See also: <StfCat>, <StfcCat>
}

function StfNCpy(s: StpTyp; n: StpInd): StpTyp;
{ Copies a maximum of n characters from string s.

  See also: <StfcCpy>, <StfStzCpy>
}

function StfNIns(dst, src: StpTyp; i,n: StpInd): StpTyp;
{ Returns a copy of string dst, after inserting a maximum of n characters
  from string src at position i of dst.
  If i=0 the characters are inserted in front.
  If i>StpLen(src) the characters are appended to dst.

  See also: <StfIns>, <StfcIns>
}

function StfNRepl(dst: StpTyp; src: StpTyp; i,n: StpInd): StpTyp;
{ Like StfRepl, replacing a maximum of n characters of string dst.

  See also: <StfRepl>, <StfcUpd>
}

function StfRAS(s: StpTyp): StpTyp;
{ Remove All Spaces: Removes all carriage return, line feed, form feed,
  horizontal tab, vertical tab and space characters from string s.

  See also: <StfRLS>, <StfRTS>
}

function StfRepl(dst: StpTyp; src: StpTyp; i: StpInd): StpTyp;
{ Returns a copy of string dst after replacing characters from dst by
  those of string src, starting at position i in dst. The returned string
  may be longer than string dst.
  If i=0 the result is the same as if i=1.
  If i>StpLen(dst) spaces are added to dst until its length is i, and then
  string src is appended to it.

  See also: <StfNRepl>, <StfcUpd>
}

function StfRev(s: StpTyp): StpTyp;
{ Reverses the character order of string s. }

function StfRight(s: StpTyp; n: StpInd): StpTyp;
{ Returns a string consisting of the n rightmost characters of string s.
  If n=0 an empty string is returned.

  See also: <StfAfter>, <StfBefore>, <StfSub>
}

function StfRLS(s: StpTyp): StpTyp;
{ Remove Leading Spaces: Removes all leading carriage return, line feed,
  form feed, horizontal tab, vertical tab and space characters from
  string s.

  See also: <StfRAS>, <StfRTS>
}

function StfRTS(s: StpTyp): StpTyp;
{ Remove Trailing Spaces: Removes all trailing carriage return, line feed,
  form feed, horizontal tab, vertical tab and space characters from
  string s.

  See also: <StfRAS>, <StfRLS>
}

function StfSoundEx(src: StpTyp; i: StpInd): StpTyp;
{ Sound Expression: Returns a string that is derived from string src by
  copying the first i characters, translating lower case letters to upper
  case, and then adding the 'sound' of the remaining characters. Two
  strings that yield the same StfSoundEx result will probably sound alike
  in English. This function can therefore be used when searching a name or
  another string value in a database where the correct spelling is not
  certain.

  The exact algorithm for deriving the function result is as follows:
  1. Translate the first i characters to upper case.
  2. Translate the remaining characters to upper case, then code them
     as follows:
       'B','F','P','V'                 become '1'
       'C','G','J','K','Q','S','X','Z' become '2'
       'D','T'                         become '3'
       'L'                             become '4'
       'M','N'                         become '5'
       'R'                             become '6'
       alle other characters           are skipped.
     Moreover, never append the same code digit twice in this coding
     process. }

function StfSrt(s: StpTyp; n: StpInd): StpTyp;
{ Sorts substring fields of string s, having a fixed field length n.
  If n=0, string s is returned unaltered.
  Example:    s = 'IF    THEN  BEGIN END   WHILE  REPEATDO    '
    StfSrt(s,6) = 'BEGIN DO    END   IF    REPEAT THEN  WHILE '

  See also: <StfLexSrt>, <StfUppSrt>
}

function StfStzCpy(const src: StzPtr): StpTyp;
{ Converts the type StzTyp string pointed to by src into type StpTyp.

  See also: <StfNCpy>, <StfcCpy>
}

function StfSub(s: StpTyp; i,n: StpInd): StpTyp;
{ Returns a substring from s, consisting of a maximum of n characters
  starting at position i.
  If i=0, the result is the same as if i=1.

  See also: <StfAfter>, <StfBefore>, <StfRight>
}

function StfTrunc(s: StpTyp; i: StpInd): StpTyp;
{ Removes all characters in string s after position i.
  If n=0 an empty string is returned.
  If n>StpLen(s) string s is returned unaltered.
}

function StfUpp(s: StpTyp): StpTyp;
{ Returns string s after converting all lower case letters to upper case.

  See also: <StfLow>
}


function StfUppSrt(s: StpTyp; n: StpInd): StpTyp;
{ Like StfSrt, sorting in UppOrder character order, as defined im unit
  ChrLib.

  See also: <StfSrt>, <StfLexSrt>
}

IMPLEMENTATION

function StfAfter(src, pat: StpTyp): StpTyp;
var l1: StpInd absolute src;
    l2: StpInd absolute pat;
    p : StpInd;
begin
  p:= Pos(pat,src);
  if p=0 then StfAfter:= '' else StfAfter:= Copy(src,p+l2,l1)
end;

function StfBefore(src, pat: StpTyp): StpTyp;
var p: StpInd;
begin
  p:= Pos(pat,src);
  if p=0 then StfBefore:= src else StfBefore:= Copy(src,1,p-1)
end;

function StfCat(s1, s2: StpTyp): StpTyp; begin StfCat:=s1+s2 end;

function StfCenter(s: StpTyp; n: StpInd): StpTyp;
begin
  StpCenter(s,n);
  StfCenter:= s
end;

function StfcCat(s: StpTyp; c: char): StpTyp;
begin StfcCat:=s+c end;

function StfcCpy(c: char): StpTyp;
begin StfcCpy:= c; end;

function StfcIns(s:StpTyp; c:char; i:StpInd): StpTyp;
begin StfcIns:= StfIns(s,c,i) end;

function StfcUpd(s: StpTyp; c: char; i: StpInd): StpTyp;
begin StpcUpd(s,c,i); StfcUpd:= s end;

function StfDel(s: StpTyp; i,n : StpInd): StpTyp;
begin StpDel(s,i,n); StfDel:= s end;

function StfDetab(src: StpTyp; n: StpInd): StpTyp;
var tmp: StpTyp;
begin StpDetab(tmp,src,n); StfDetab:= tmp end;

function StfEntab(src: StpTyp; n: StpInd): StpTyp;
var tmp: StpTyp;
begin StpEntab(tmp,src,n); StfEntab:= tmp end;

function StfFill(s: StpTyp; c: char; n: StpInd): StpTyp;
begin StpFill(s,c,n); StfFill:= s; end;

function StfGtw(var s: StpTyp): StpTyp;
var tmp: StpTyp;
begin StpGtw(tmp,s); StfGtw:= tmp; end;

function StfIns(dst, src: StpTyp; i: StpInd): StpTyp;
begin StpIns(dst,src,i); StfIns:= dst end;

function StfLexSrt(s: StpTyp; n: StpInd): StpTyp;
var tmp: StpTyp;
begin StpLexSrt(tmp,s,n); StfLexSrt:= tmp end;

function StfLow(s: StpTyp): StpTyp;
begin StpLow(s); StfLow:= s end;

function StfMake(c: char; n: StpInd): StpTyp;
begin StfMake:= StfFill('',c,n) end;

function StfNCat(s1, s2: StpTyp; n: StpInd): StpTyp;
begin StfNCat:=s1+Copy(s2,1,n) end;

function StfNCpy(s: StpTyp; n: StpInd): StpTyp;
begin StfNCpy:=Copy(s,1,n) end;

function StfNIns(dst, src: StpTyp; i,n: StpInd): StpTyp;
begin StfNIns:= StfIns(dst,Copy(src,1,n),i) end;

function StfNRepl(dst: StpTyp; src: StpTyp; i,n: StpInd): StpTyp;
begin StpNRepl(dst,src,i,n); StfNRepl:= dst end;

function StfRAS(s: StpTyp): StpTyp;
begin StpRAS(s); StfRAS:= s end;

function StfRepl(dst: StpTyp; src: StpTyp; i: StpInd): StpTyp;
var n: StpInd absolute src;
begin StfRepl:= StfNRepl(dst,src,i,n) end;

function StfRev(s: StpTyp): StpTyp;
begin StpRev(s); StfRev:= s end;

function StfRight(s: StpTyp; n: StpInd): StpTyp;
var i: StpInd; l: StpInd absolute s;
begin if n>=l then i:=1 else i:=l-n+1; StfRight:=Copy(s,i,n) end;

function StfRLS(s: StpTyp): StpTyp;
begin StpRLS(s); StfRLS:= s end;

function StfRTS(s: StpTyp): StpTyp;
begin StpRTS(s); StfRTS:= s end;

function StfSoundEx(src: StpTyp; i: StpInd): StpTyp;
var tmp: StpTyp;
begin StpSoundEx(tmp,src,i); StfSoundEx:= tmp end;

function StfSrt(s: StpTyp; n: StpInd): StpTyp;
var tmp: StpTyp;
begin StpSrt(tmp,s,n); StfSrt:= tmp end;

function StfStzCpy(const src: StzPtr): StpTyp;
var tmp: StpTyp;
begin StpStzCpy(tmp,src); StfStzCpy:= tmp; end;

function StfSub(s: StpTyp; i,n: StpInd): StpTyp;
begin if i=0 then i:=1; StfSub:=Copy(s,i,n) end;

function StfTrunc(s: StpTyp; i: StpInd): StpTyp;
begin StfTrunc:= Copy(s,1,i) end;

function StfUpp(s: StpTyp): StpTyp;
begin StpUpp(s); StfUpp:= s end;

function StfUppSrt(s: StpTyp; n: StpInd): StpTyp;
var tmp: StpTyp;
begin StpUppSrt(tmp,s,n); StfUppSrt:= tmp end;

END.
