{ RRGLIB.PAS : Real range handling routine library

  Title   : RRGLIB
  Version : 4.0
  Date    : Nov 11,1996
  Author  : J R Ferguson
  Language: Borland Turbo Pascal 4.0 through 7.0 (all targets)
            Borland Delphi 1.0 for Windows
  Usage   : Unit

  The internal representation of a (possibly broken) real range is an
  ordered list of non-overlapping low-high interval boundaries.
}

UNIT RrgLib;

INTERFACE
uses DefLib;

type
  RrgTyp = ^RrgRec;                                { public }

  RrgRec = record                                  { for internal use }
             lb,hb: real;     { low/high boundary }
             li,hi: boolean;  { low/high included }
             nxt  : RrgTyp
           end;


procedure RrgCreate(var r: RrgTyp);
{ Create an empty range set }

procedure RrgDispose(var r: RrgTyp);
{ Dispose a possibly non-empty range set }

procedure RrgClear(var r: RrgTyp);
{ Clear an existing range set (making it empty) }

function RrgEmpty(r: RrgTyp): boolean;
{ Test if range set r is empty (contains no values) }

function RrgInside(r: RrgTyp; i: real): boolean;
{ Test if value i is inside range set r }

procedure RrgInsert(var r : RrgTyp;     { empty or existing range }
                        lb: real;       { low boundary value }
                        li: boolean;    { low boundary included }
                        hb: real;       { high boundary value }
                        hi: boolean);   { high boundary included }
{ Insert a single range into range set r }

procedure RrgDelete(var r : RrgTyp;     { empty or existing range }
                        lb: real;       { low boundary value }
                        li: boolean;    { low boundary included }
                        hb: real;       { high boundary value }
                        hi: boolean);   { high boundary included }
{ Delete a single range from range set r }

procedure RrgMerge(var r1: RrgTyp; r2: RrgTyp);
{ Merge range set r2 into range set r1 (result in r1) }

procedure RrgRemove(var r1: RrgTyp; r2: RrgTyp);
{ Remove range set r2 from range set r1 (result in r1) }

procedure RrgCopy(var dst: RrgTyp; src: RrgTyp);
{ Create a copy from an existing range set }

procedure RrgInvert(var r: RrgTyp);
{ Invert range set r }

procedure RrgIntersect(var r: RrgTyp; r1,r2: RrgTyp);
{ Intersect range sets r1 and 2, creating result set r }

procedure RrgUnite(var r: RrgTyp; r1,r2: RrgTyp);
{ Unite range sets r1 and r2, creating result set r }


IMPLEMENTATION


{--- Local routines ---}

procedure order(var lb,hb: real; var li,hi: boolean);
var tb: real; ti: boolean;
begin
  if lb>hb then begin
    tb:= lb; lb:= hb; hb:= tb;
    ti:= li; li:= hi; hi:= ti;
  end;
end;

function below(r: RrgTyp; i: real): boolean;
begin
  if r=nil then below:= false
  else with r^ do case li of
    false : below:= i <= lb;
    true  : below:= i <  lb;
  end;
end;

function BndBelow(r: RrgTyp; hb: real; hi: boolean): boolean;
begin
  if r=nil then BndBelow:= false
  else begin
    if hi and r^.li then BndBelow:= hb <  r^.lb
                    else BndBelow:= hb <= r^.lb;
  end;
end;

function above(r: RrgTyp; i: real): boolean;
begin
  if r=nil then above:= false
  else with r^ do case hi of
    false : above:= i >= hb;
    true  : above:= i >  hb;
  end;
end;

function BndAbove(r: RrgTyp; lb: real; li: boolean): boolean;
begin
  if r=nil then BndAbove:= false
  else begin
    if li and r^.hi then BndAbove:= lb >  r^.hb
                    else BndAbove:= lb >= r^.hb;
  end;
end;

function LbndInside(r: RrgTyp; lb: real; li: boolean): boolean;
begin
  if li and not r^.li then LbndInside:= lb >  r^.lb
                      else LbndInside:= lb >= r^.lb;
end;

function HbndInside(r: RrgTyp; hb: real; hi: boolean): boolean;
begin
  if hi and not r^.hi then HbndInside:= hb <  r^.hb
                      else HbndInside:= hb <= r^.hb;
end;

function inside(r: RrgTyp; i: real): boolean;
begin
  if r=nil then inside:= false
  else inside:= not ( below(r,i) or above(r,i) );
end;

function BndInside(r: RrgTyp; lb,hb: real; li,hi: boolean): boolean;
begin
  if r=nil then BndInside:= false
  else BndInside:= LbndInside(r,lb,li) and HbndInside(r,hb,hi);
end;

function connected(r1,r2: RrgTyp): boolean;
begin
  if       r2=nil            then connected:= false
  else if  r1^.hi or r2^.li  then connected:= r1^.hb >= r2^.lb
  else                            connected:= r1^.hb >  r2^.lb;
end;

function empty(r: RrgTyp): boolean;
begin with r^ do begin
  if li and hi then empty:= lb > hb else empty:= lb >= hb;
end end;

procedure cleanup(var r: RrgTyp);
{ clear empty subranges, merge connected subranges }
var r0: RrgTyp;
begin if r<>nil then begin
  if empty(r) then begin
    r0:= r;
    r:= r^.nxt;
    dispose(r0);
    cleanup(r);
  end
  else if connected(r,r^.nxt) then begin
    r0:= r^.nxt;
    r^.hb:= r0^.hb; r^.hi:= r0^.hi; r^.nxt:= r0^.nxt;
    dispose(r0);
    cleanup(r);
  end
  else cleanup(r^.nxt);
end end;

procedure getminimum(    b1: real;     i1: boolean;
                         b2: real;     i2: boolean;
                     var b : real; var i : boolean);
begin
  if       b1 < b2  then begin b:= b1; i:= i1;       end
  else if  b1 > b2  then begin b:= b2; i:= i2;       end
  else    {b1 = b2}      begin b:= b1; i:= i1 or i2; end;
end;

procedure getmaximum(    b1: real;     i1: boolean;
                         b2: real;     i2: boolean;
                     var b : real; var i : boolean);
begin
  if       b1 > b2  then begin b:= b1; i:= i1;       end
  else if  b1 < b2  then begin b:= b2; i:= i2;       end
  else    {b1 = b2}      begin b:= b1; i:= i1 or i2; end;
end;


{--- Interfaced routines ---}

procedure RrgCreate(var r: RrgTyp);
begin r:= nil end;

procedure RrgDispose(var r: RrgTyp);
begin if r<>nil then begin RrgDispose(r^.nxt); dispose(r); end; end;

procedure RrgClear(var r: RrgTyp);
begin RrgDispose(r); RrgCreate(r) end;

function RrgEmpty(r: RrgTyp): boolean;
begin RrgEmpty := r=nil end;

function RrgInside(r: RrgTyp; i: real): boolean;
begin
  if      r=nil      then RrgInside:= false
  else if above(r,i) then RrgInside:= RrgInside(r^.nxt,i)
  else                    RrgInside:= inside(r,i);
end;

procedure RrgInsert(var r : RrgTyp;     { empty or existing range }
                        lb: real;       { low boundary value }
                        li: boolean;    { low boundary included }
                        hb: real;       { high boundary value }
                        hi: boolean);   { high boundary included }
var r0: RrgTyp;
begin
  order(lb,hb,li,hi);
  if above(r,lb) then RrgInsert(r^.nxt,lb,li,hb,hi)
  else if (r=nil) or below(r,hb) then begin
    new(r0);
    r0^.lb:= lb; r0^.li:= li;
    r0^.hb:= hb; r0^.hi:= hi;
    r0^.nxt:= r;
    r:= r0;
  end
  else if not BndInside(r,lb,hb,li,hi) then begin
    getminimum(lb,li, r^.lb,r^.li, r^.lb,r^.li);
    getmaximum(hb,hi, r^.hb,r^.hi, r^.hb,r^.hi);
  end;
  cleanup(r);
end;

procedure RrgDelete(var r : RrgTyp;     { empty or existing range }
                        lb: real;       { low boundary value }
                        li: boolean;    { low boundary included }
                        hb: real;       { high boundary value }
                        hi: boolean);   { high boundary included }
var r0: RrgTyp;
begin if r<>nil then begin
  order(lb,hb,li,hi);
  if BndAbove(r,lb,li) then RrgDelete(r^.nxt,lb,li,hb,hi)
  else if not BndBelow(r,hb,hi) then begin
    if LbndInside(r,lb,li) then begin
      if HbndInside(r,hb,hi) then begin
        new(r0);
        r0^.lb := hb    ; r0^.li:= not hi;
        r0^.hb := r^.hb ; r0^.hi:= r^.hi;
        r0^.nxt:= r^.nxt;
        r^.hb  := lb    ; r^.hi := not li;
        r^.nxt := r0;
      end
      else begin
        r^.hb:= lb; r^.hi:= not li;
        RrgDelete(r^.nxt,lb,li,hb,hi);
      end
    end
    else begin
      if HbndInside(r,hb,hi) then begin
        r^.lb:= hb; r^.li:= not hi
      end
      else begin
        r0:= r; r:= r^.nxt; dispose(r0);
        RrgDelete(r,lb,li,hb,hi);
      end
    end
  end;
  cleanup(r);
end end;

procedure RrgMerge(var r1: RrgTyp; r2: RrgTyp);
begin with r2^ do begin
  if r2<>nil then begin RrgInsert(r1,lb,li,hb,hi); RrgMerge(r1,nxt) end;
end end;

procedure RrgRemove(var r1: RrgTyp; r2: RrgTyp);
begin with r2^ do begin
  if r2<>nil then begin RrgDelete(r1,lb,li,hb,hi); RrgRemove(r1,nxt) end;
end end;

procedure RrgCopy(var dst: RrgTyp; src: RrgTyp);
begin RrgCreate(dst); RrgMerge(dst,src) end;

procedure RrgInvert(var r: RrgTyp);
var r0,t: RrgTyp;
begin
  RrgCreate(r0); RrgInsert(r0,MinReal,true,MaxReal,true);
  RrgRemove(r0,r);
  t:= r; r:= r0; r0:= t;
  RrgDispose(r0);
end;

procedure RrgUnite(var r: RrgTyp; r1,r2: RrgTyp);
begin RrgCopy(r,r1); RrgMerge(r,r2) end;

procedure RrgIntersect(var r: RrgTyp; r1,r2: RrgTyp);
var t1,t2: RrgTyp;
begin
  RrgCopy(t1,r1);  RrgRemove(t1,r2);
  RrgCopy(t2,r2);  RrgRemove(t2,r1);
  RrgUnite(r,r1,r2);
  RrgRemove(r,t1); RrgDispose(t1);
  RrgRemove(r,t2); RrgDispose(t2);
end;

END.
