/* MKTAR.C - Create Un*x format TAR file on disk or DOS device
   $Author:   Phlash  $
   $Date:   04 Nov 1994 21:47:54  $
   $Revision:   1.0  $
 */
#include <stdio.h>
#include <io.h>
#include <dos.h>
#include <time.h>
#include <malloc.h>
#include <fcntl.h>
#include <string.h>
#include <direct.h>
#include "tar.h"
#include "ioctl.h"
#include "qic_drv.h"

/* check for memory model */
#ifndef M_I86LM
#error Must use large model!
#endif

#define MAXPATH  128
#define WILDCHR '*'

int verbose=0;

#define formsize(str, l)   sprintf(str, "%10lo", l)
#define formdate(str, l)   sprintf(str, "%11lo", l)
#define formint(str, i)    sprintf(str, "%7o", i)

static int qic_tape=0;
static qicDrvIoctl_t ioc;
static int bnum=0;
static unsigned int blen=0;
static char *tb[2];

char *getbuffer(void)
{
char *buffer;
unsigned int seg;

   _dos_allocmem(0xFFFF, &seg);
   if(_dos_allocmem(seg, &seg))
   {
      fprintf(stderr, "cannot allocate transfer buffer\n");
      return NULL;
   }
   FP_SEG(buffer) = (seg + 0x0FFF) & ~0x0FFF;
   FP_OFF(buffer) = 0;
   tb[1] = tb[0] = buffer;
   FP_SEG(tb[0]) += 0x1000;
   FP_SEG(tb[1]) += 0x2000;
   return buffer;
}

int devicecheck(int tf)
{
int i;

   if(ioctl(tf, FIOGETINFO, &i))
   {
      fprintf(stderr, "cannot determine type of output file\n");
      return 1;
   }
   if(i & 0x80)
   {
      i = (i & 0xFF) | 0x20;
      ioctl(tf, CIOSETINFO, &i);
      if(ioctl(tf, CIORDCNTRL(sizeof(ioc)), &ioc))
         return 0;
      if(strncmp(ioc.chk, QIC_CHK_VALUE, 4))
         return 0;
      ioc.async = 1;
      if(ioctl(tf, CIOWRCNTRL(sizeof(ioc)), &ioc))
         return 0;
      puts("(AshbySoft *) QIC_TAPE device detected, using asyncronous mode");
      qic_tape = 1;
   }
   return 0;
}

int writebuffer(int tf, char *buf, unsigned int len)
{
unsigned int bleft = 0xF000-blen;

/* collate data into transfer buffer, write when full */
   if(len < bleft)
   {
      memcpy(&tb[bnum][blen], buf, len);
      blen += len;
   }
   else
   {
      memcpy(&tb[bnum][blen], buf, bleft);
#ifdef SLOW_IO
/*
 * This code waits for the previous write to complete and checks the status
 * for errors before initiating the next one....
 * Unfortunately DOS is not quick enough to keep the tape streaming if we
 * do this, so we default to not checking :-(
 */
      if(qic_tape)
      {
         ioc.command = QIC_COMPLETE;
         if(ioctl(tf, CIOWRCNTRL(sizeof(ioc)), &ioc))
            return -1;
         if(ioc.command == -1)
            return -1;
      }
#endif
      write(tf, tb[bnum], 0xF000);
      bnum = 1-bnum;
      memcpy(tb[bnum], &buf[bleft], len-bleft);
      blen = len-bleft;
   }
   return len;
}

int closebuffer(int tf)
{
/* flush remaining data */
   if(blen)
      write(tf, tb[bnum], blen);
   bnum=0;
   blen=0;
   return close(tf);
}

void dostounix(char *filename)
{
int i;

/* just change all '\' to '/' & lowercase all letters */
   for(i=0; i<strlen(filename); i++)
   {
      if(filename[i] == '\\')
         filename[i] = '/';
      else if(filename[i] >= 'A' && filename[i] <= 'Z')
         filename[i] = filename[i] + 'a' - 'A';
   }
}

long calclong(char *f_size)
{
long size;

   sscanf(f_size, " %lo", &size);
   return size;
}

int calcint(char *pint)
{
int val;

    sscanf(pint, " %o", &val);
    return val;
}

void displayfile(union TAR_BLOCK *ptb)
{
int i, j, perms, gid, uid;
long size, tim;
struct tm *ptm;
char permstr[12], *pchrs="xwr";

/* extract data as binary */
   perms = calcint(ptb->direct.f_perm);
   uid =   calcint(ptb->direct.f_uid);
   gid =   calcint(ptb->direct.f_gid);
   size =  calclong(ptb->direct.f_size);
   tim =   calclong(ptb->direct.f_date);

/* convert permissions to string representation (eg: '-rw-r--r--') */
   if(perms & 0xFE00)
      permstr[0] = 'd';
   else
      permstr[0] = '-';
   for(i=0; i<3; i++)
   {
      for(j=0; j<3; j++)
      {
         if(perms & (1 << (3*i+j)))
            permstr[9-(3*i+j)] = pchrs[j];
         else
            permstr[9-(3*i+j)] = '-';
      }
   }
   permstr[10] = '\0';

/* obtain file date */
   ptm = gmtime(&tim);

/* print infomation */
   printf("%s %5d %5d %02d/%02d/%02d %8ld bytes %.100s\n",
      permstr, uid, gid,
      ptm->tm_mday, ptm->tm_mon, ptm->tm_year,
      size, ptb->direct.f_name);
}

int putdir(int tf, char *filename, struct find_t *pfi, union TAR_BLOCK *ptb)
{
int i;

/* format the directory block */
   for(i=0; i<BLOCKSIZE; i++)
      ptb->raw[i] = 0;
   if(filename)
   {
      dostounix(filename);
      strcpy(ptb->direct.f_name, filename);
      if(pfi->attrib & _A_RDONLY)
         formint(ptb->direct.f_perm, 0444);
      else
         formint(ptb->direct.f_perm, 0666);
      formint(ptb->direct.f_uid, 1);
      formint(ptb->direct.f_gid, 0);
      formsize(ptb->direct.f_size, pfi->size);      
      formdate(ptb->direct.f_date, time(NULL));
      formint(ptb->direct.f_csum, 0);
      formint(ptb->direct.f_flags, 0);
      if(verbose)
         displayfile(ptb);
   }

/* write directory block */
   if(writebuffer(tf, ptb->raw, BLOCKSIZE) != BLOCKSIZE)
   {
      fprintf(stderr, "cannot write directory block\n");
      return 1;
   }
   return 0;
}

int putfile(int tf, char *filename, union TAR_BLOCK *ptb)
{
int rf;
unsigned int n;

/* open source file */
   rf = open(filename, O_RDONLY | O_BINARY);
   if(rf < 0)
   {
      fprintf(stderr, "cannot open file %s\n", filename);
      return 1;
   }

/* copy file in big blocks */
   n = 1;
   while(n)
   {
      switch(n=read(rf, ptb->raw, 0xF000))
      {
      case 0:
         continue;
      case 0xFFFF:
         fprintf(stderr, "cannot read data file %s\n", filename);
         close(rf);
         return 1;
      }
      while(n%BLOCKSIZE)
         ptb->raw[n++] = 0;
      if(writebuffer(tf, ptb->raw, n) == -1)
      {
         fprintf(stderr, "cannot write tar file\n");
         close(rf);
         return 1;
      }
   }
   close(rf);
   return 0;
}

int doFiles(int tf, char *rootpath, char *wildpath, union TAR_BLOCK *ptb)
{
struct find_t srch_blk;
char pathBuf[128];

/* search specified path for files / sub-dirs */
   strcpy(pathBuf, rootpath);
   if(rootpath[0])
      strcat(pathBuf, "\\");
   strcat(pathBuf, wildpath);
   if(_dos_findfirst(pathBuf, _A_NORMAL | _A_HIDDEN | _A_SYSTEM | _A_SUBDIR,
      &srch_blk) == 0)
   {
      do
      {
      /* ignore '.' & '..' entries */
         if(srch_blk.name[0] == '.')
            continue;

      /* See if we have a directory to process */
         if(srch_blk.attrib & _A_SUBDIR)
         {
         /* if so, write entry then recurse */
            strcpy(pathBuf, rootpath);
            if(rootpath[0])
               strcat(pathBuf, "\\");
            strcat(pathBuf, srch_blk.name);
            strcat(pathBuf, "\\");
            if(putdir(tf, pathBuf, &srch_blk, ptb))
               return 1;
            strcpy(pathBuf, rootpath);
            if(rootpath[0])
               strcat(pathBuf, "\\");
            strcat(pathBuf, srch_blk.name);
            if(doFiles(tf, pathBuf, "*.*", ptb))
               return 1;
         }
         else
         {
         /* otherwise, add to the tar file */
            strcpy(pathBuf, rootpath);
            if(rootpath[0])
               strcat(pathBuf, "\\");
            strcat(pathBuf, srch_blk.name);
            if(putdir(tf, pathBuf, &srch_blk, ptb))
               return 1;
            if(putfile(tf, pathBuf, ptb))
               return 1;
         }
      } while(_dos_findnext(&srch_blk) == 0);
   }
   else
   {
      fprintf(stderr, "Invalid file/path %s\n", pathBuf);
      return 1;
   }
   return 0;
}

int main(int argc, char *argv[])
{
int tf, i, j;
union TAR_BLOCK *ptb;
static char wild[128];

/* check args */
   if(argc < 2)
   {
      puts("usage: mktar [-v] <tar file> [file(s)]");
      puts("where -v displays file names as they are processed");
      return 0;
   }
   i=0;
   while(!i)
   {
      if(strcmp(argv[1], "-v") == 0)
      {
         verbose++;
         argc--;
         argv++;
      }
      else
         i++;
   }

/* allocate transfer buffer */
   ptb = (union TAR_BLOCK *)getbuffer();
   if(!ptb)
      return 1;

/* open output file */
   tf = open(argv[1], O_WRONLY | O_BINARY | O_CREAT | O_TRUNC, 0666);
   if(tf < 0)
   {
      fprintf(stderr, "cannot open tar file %s\n", argv[1]);
      return 1;
   }
   if(devicecheck(tf))
      return 1;

/* now write tar file */
   if(argc < 3)
   {
      if(doFiles(tf, "", "*.*", ptb))
         return 1;
   }
   else
   {
      for(i=2; i<argc; i++)
      {
      /* parse wildcard string off end of path/file name */
         for(j=strlen(argv[i])-1; j >= 0; j--)
            if(argv[i][j] == '\\')
               break;
         if(j >= 0)
         {
            strcpy(wild, &(argv[i][j+1]));
            argv[i][j] = 0;
         }
         else
         {
            strcpy(wild, argv[i]);
            argv[i][0] = 0;
         }
         if(doFiles(tf, argv[i], wild, ptb))
            return 1;
      }
   }

/* write EOF marker */
   putdir(tf, NULL, NULL, ptb);
   closebuffer(tf);
   return 0;
}
