{**********************************************************************
 Unit   : ADCFUNCS
 Version: 1.0
 Purpose: This unit contains procedures for accessing the Stanford
          Reasearch Systems, SR245 analog to digital NIM board over the
          GPIB interface.
 Author : Roger Carlson (Feb. 1991)
 Notes  : 1. A boolean error flag is returned by most of these procedures.
             Not that if the flag is already set, these procedures will
             not alter that flag.
 Changes:
***********************************************************************}
UNIT ADCFUNCS;

INTERFACE

PROCEDURE ADCINIT(VAR ADUD:INTEGER; VAR ERR:BOOLEAN);
  {Initializes the A/D board.  Returns the the unit descriptor in ADUD.}
PROCEDURE ADCRESET(ADUD:INTEGER; VAR ERR:BOOLEAN);
  {Perform master reset.}
PROCEDURE ADCSYNC(ADUD:INTEGER; VAR ERR:BOOLEAN);
  {Sets A/D to synchronous trigger mode.}
FUNCTION ADOVERFLOW(ADUD:INTEGER; VAR ABORT:BOOLEAN):BOOLEAN;
  {Returns true if missed data or voltage out of range.  ABORT is
   set true if any other a/d board error is detected.}
FUNCTION ADCREAD(ADUD,CH:INTEGER; VAR ERR,OVERFLOW:BOOLEAN):REAL;
  {Reads channel CH of SR245 A/D board.  ERR should be initialized
   externally.  OVERFLOW flags out of range or missed data.}

IMPLEMENTATION

USES TPDECL,       {National Instruments GPIB interface}
     GPIBFUNC,     {version 1.0}
     GRAPH,        {screen graphics driver}
     IOFUNCS;      {version 1.4}

{*********** ADCINIT(VAR ADUD:INTEGER; VAR ERR:BOOLEAN) *******************}
PROCEDURE ADCINIT;
  {Initializes the A/D board.  Returns the the unit descriptor in ADUD.}
VAR ADNAME:NBUF; {7 character string GPIB name for A/D converter}
BEGIN
  ADNAME:='AD     '; ADUD:=IBFIND(ADNAME); {open A/D over GPIB}
  IF ADUD<0 THEN BEGIN
    RESTORECRTMODE; BEEP(200); ERR:=TRUE;
    WRITELN('GPIB interface to A/D ibfind error!');
    WRITE('Hit <ENTER> to continue. '); READLN;
    END {IF ADUD<0}
  ELSE BEGIN {clear the GPIB device}
    IBCLR(ADUD);
    ERR:=ERR OR GPIBERR('Procedure ADCINIT');
    END;
END; {ADCINIT}

{*************** ADCRESET(ADUD:INTEGER; VAR ERR:BOOLEAN) *******************}
PROCEDURE ADCRESET;  {Perform master reset.}
VAR  WRT:ARRAY[1..4] OF CHAR;
BEGIN
  WRT[1]:='M'; WRT[2]:='R'; WRT[3]:=CHR($0D); WRT[4]:=CHR($0A);
  IBWRT(ADUD,WRT,4); ERR:=ERR OR GPIBERR('Procedure ADCRESET');
END; {PROCEDURE ADCRESET}

{************** ADCSYNC(ADUD:INTEGER; VAR ERR:BOOLEAN) *********************}
PROCEDURE ADCSYNC;  {Sets A/D to synchronous trigger mode.}
VAR WRT:ARRAY[1..4] OF CHAR;
BEGIN
  WRT[1]:='M'; WRT[2]:='S'; WRT[3]:=CHR($0D); WRT[4]:=CHR($0A);
  IBWRT(ADUD,WRT,4); ERR:=ERR OR GPIBERR('Procedure ADCSYNC');
END; {PROCEDURE ADCSYNCH}

{********* ADOVERFLOW(ADUD:INTEGER; VAR ABORT:BOOLEAN):BOOLEAN *************}
FUNCTION ADOVERFLOW;
  {Returns true if missed data or voltage out of range.  ABORT is
   set true if any other a/d board error is detected.}
VAR  RD : BYTE;
BEGIN
  IBRSP(ADUD,RD);              {serial poll of status byte}
  ADOVERFLOW:=(RD AND 10)<>0;         {bit 1 or 3 set}
  ABORT:=ABORT OR ((RD AND 5)<>0);    {bit 0 or 2 set}
END; {FUNCTION ADOVERFLOW}

{******* ADCREAD(ADUD,CH:INTEGER; VAR ERR,OVERFLOW:BOOLEAN):REAL ***********}
FUNCTION ADCREAD;
  {Reads channel CH of SR245 A/D board.  ERR should be initialized
   externally.  OVERFLOW flags out of range or missed data.}
VAR
  ANS           : INTEGER;               {A/D value}
  WRT           : ARRAY[1..7] OF CHAR;   {write string for a/d}
  RD            : ARRAY[1..2] OF BYTE;   {read string for a/d}
BEGIN
  WRT[1]:='S'; WRT[2]:='S'; WRT[3]:=CHR(CH+48); WRT[4]:=':'; WRT[5]:='1';
  WRT[6]:=CHR($0D); WRT[7]:=CHR($0A);
  IBWRT(ADUD,WRT,7);  ERR:=ERR OR GPIBERR('Function ADCREAD');
  IBRD(ADUD,RD,2);    ERR:=ERR OR GPIBERR('Function ADCREAD');
  OVERFLOW:=ADOVERFLOW(ADUD,ERR);
  ANS:=(RD[1] SHL 8) OR RD[2];  {convert 2 bytes to integer}
  IF (ANS AND $1000)<>0 THEN ANS:=-(ANS-$1000); {get correct sign}
  ADCREAD:=0.0025*ANS;
END; {FUNCTION ADCREAD}

END.