(******************************************************************************
 *                                                                            *
 *  File        : XRAY.MOD                                                    *
 *  Description : Filter utility for finding text strings in binary files     *
 *  Language    : JPI TopSpeed Modula-2                                       *
 *  Author      : Scott McIntosh                                              *
 *  History     : 1.00 (03/20/90) : Original version                          *
 *                1.01 (03/21/90) : Filter routine speeded up                 *
 *              # 1.02 (03/24/90) : Escape key abort added                    *
 *                1.03 (03/31/90) : Break processing excluded                 *
 *                1.04 (04/07/90) : Code cleaned up                           *
 *              # 1.05 (04/10/90) : Pause routine relocated for speed         *
 *                                                                            *
 *              # = public versions                                           *                    *
 *                                                                            *
 ******************************************************************************)

MODULE XRay;

  (*$B-*)  (* Breaks not processed *)

  FROM FIO     IMPORT Close, File, IOcheck, Open, RdBin, Seek;
  FROM IO      IMPORT KeyPressed, RdKey, RdStr, WrLn, WrStr;
  FROM IO2     IMPORT ClearKeyBuffer;
  FROM Lib     IMPORT CommandLine, FatalError;
  FROM Storage IMPORT ALLOCATE, Available, DEALLOCATE;
  FROM Str     IMPORT CHARSET, Append, Copy, Length;

  VAR
    F : File;

  PROCEDURE FileGet (VAR F : File);

    VAR
      Len      : CARDINAL;
      FileName : ARRAY [0..79] OF CHAR;

  BEGIN  (* FileGet *)

    (* Get file name *)
    Copy (FileName, CommandLine^);
    IF FileName[0] = 0C THEN
      WrStr ("File name? ");
      RdStr (FileName)
    END;

    (* Get handle for file I/O; abort if file not found *)
    IOcheck := FALSE;
    F := Open (FileName);             (* If file not found then F = 65535 *)
    IF F = MAX (CARDINAL) THEN FatalError ("File not found" + 15C + 12C) END

  END FileGet;

  PROCEDURE FileScan (F : File);

    CONST
      BufSize = 65535;

    TYPE
      BufType = ARRAY [0..BufSize] OF BYTE;         (* 64K buffer *)
      BufPtr  = POINTER TO BufType;

    VAR
      Buffer   : BufPtr;
      BufChars : CARDINAL;          (* Number of chars fetched into buffer *)
      i        : LONGCARD;

    PROCEDURE KeyPause ();
    (*
       Used to pause output.
    *)

      VAR
        Dummy : CHAR;

    BEGIN (* KeyPause *)

      IF KeyPressed () THEN
        IF RdKey () = 33C THEN HALT END;      (* Quit if Escape key pressed *)
        ClearKeyBuffer;
        Dummy := RdKey ()                     (* Otherwise, wait for next key *)
      END

    END KeyPause;

    PROCEDURE FileScan_BufProcess (VAR B    : BufType;
                                   BufChars : CARDINAL);
    (*
      The real work of finding ASCII strings is done here.
    *)
      CONST
        MaxStrLen  = 80;
        MinStrLen  = 5;
        ValidChars = CHARSET{" ".."?", "A".."Z", "a".."z"};


      VAR
        i      : CARDINAL;
        String : ARRAY [0..80] OF CHAR;

    BEGIN (* FileScan_BufProcess *)

      String[0] := 0C;                          (* Start with empty string *)
      DEC (BufChars);                           (* Buffer is zero-indexed *)
      FOR i := 0 TO BufChars DO                 (* For each char in buff ... *)
        IF CHAR (B[i]) IN ValidChars THEN       (* if char valid ... *)
          Append (String, CHAR (B[i]));         (* add to string *)
          IF Length (String) = MaxStrLen THEN   (* if string full ... *)
            WrStr (String);                     (* show string ... *)
            String[0] := 0C;                    (* and clear *)
            KeyPause ()
          END
        ELSIF String[0] # 0C THEN               (* if char not valid ... *)
          IF Length (String) >= MinStrLen THEN  (* and str long enough ... *)
            WrStr (String); WrLn;               (* show string *)
            KeyPause ()
          END;
          String[0] := 0C                       (* clear string *)
        END
      END;

      (* If buffer done and string not empty, show string *)
      IF String[0] # 0C THEN WrStr (String) END

    END FileScan_BufProcess;

  BEGIN (* FileScan *)

    (* Create buffer *)
    IF Available (SIZE (BufType)) THEN
      ALLOCATE (Buffer, SIZE (BufType))
    ELSE
      FatalError ("Insufficient memory" + 15C + 12C)
    END;

    (* Feed file into buffer and scan buffer for ASCII strings *)
    i := 0;                                         (* 1st 64K gulp *)
    REPEAT
      INC (i);
      Seek (F, ((i-1) * BufSize) + 0);              (* Start at 64K boundary *)
      BufChars := RdBin (F, Buffer^, BufSize);      (* Copy (up to) next 64K *)
      FileScan_BufProcess (Buffer^, BufChars)       (* Scan for ASCII string *)
    UNTIL BufChars < BufSize;    (* If last gulp was full 64K, file not done *)

    (* Close buffer *)
    DEALLOCATE (Buffer, SIZE (BufType))

  END FileScan;

BEGIN (* XRay *)

  FileGet (F);
  FileScan (F);
  Close (F)

END XRay.