/*
  xd.c - File dump utility.

  Display a file (or standard input) as a series of hex bytes and/or
  characters, or records. Write a dump to a file.

  Jason Hood, 5 to 7 July, 1998. (Originally?)

  980921: made offset display optional.
  981030: added ignore byte(s) record format.
  010811: fixed bug with 'z' record format (wasn't reading NUL properly);
	  added dump of multiple files (with the one format);
	  fixed bug in calculating record length (wasn't accumulating).
  021005: really fixed bug with 'z' record format;
	  added variable length strings, if no length (or 1) is given;
	   currently limited to 1040 characters (will crash if more).
  030629: recognise the 'z' length, in case no NUL is present (prevents crash).
  030706: increased MAX_ITEMS to 20 (from 10);
    to	  added floating point numbers;
  030710  added width specifier (which includes separating space) and '0' fill;
	  allow a comma to separate record fields;
	  added a blank line between multiple records;
	  use a semi-colon to add separating space between fields;
	  added PRINT macro.

  030912: width no longer includes separating space (no good with '0' fill);
    to	  ';' is space, ',' is tab and '+' is now no separator (not unsigned);
  031007  separating space is part of the record;
  v1.00   added "triplet" type (three bytes);
	  added "Unicode" flag for characters/strings (two-byte chars);
	  added "-~" option and '~' type flag to reverse byte order;
	  ability to convert a text file into a binary file;
	  added character dump filters.

  Acknowledgement:
    The real format was posted to comp.compilers.lcc by Andreas
     <Andi.Martin@freenet.de>, "Re: Floating points in lcc", 9 July 2003.

  Note: I don't think it will work as-is on a big-endian machine. In part-
	icular: the real/double conversion routines and writing integers.
*/

#define PVERS "1.00"
#define PDATE "7 October, 2003"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>

#ifdef __DJGPP__
#include <unistd.h>
#include <crt0.h>
void __crt0_load_environment_file( char* filler ) { }
#endif


typedef signed char    INT8;
typedef signed short   INT16;
typedef signed long    INT32;
typedef unsigned char  UINT8;
typedef unsigned short UINT16;
typedef unsigned long  UINT32;

enum
{
  T_IGN,					// Ignore bytes
  T_INT8,  T_INT16,  T_INT24,  T_INT32, 	// Signed decimal numbers
  T_UINT8, T_UINT16, T_UINT24, T_UINT32,	// Unsigned decimal numbers
  T_HEX8,  T_HEX16,  T_HEX24,  T_HEX32, 	// Hexadecimal numbers
  T_FLOAT, T_REAL,   T_DOUBLE, T_LDOUBLE,	// Floating point numbers
  T_CHAR,  T_ZERO,				// Characters and ASCIZ strings
  T_LEN8,  T_LEN16,  T_LEN32			// Length-prefixed strings
};

const char* const t_name[] =
{
  "",				// T_IGN
  "decimal byte (signed)",	// T_INT8
  "decimal word (signed)",	// T_INT16
  "decimal triplet (signed)",   // T_INT24
  "decimal long (signed)",      // T_INT32
  "decimal byte (unsigned)",    // T_UINT8
  "decimal word (unsigned)",    // T_UINT16
  "decimal triplet (unsigned)", // T_UINT24
  "decimal long (unsigned)",    // T_UINT32
  "hexadecimal byte",           // T_HEX8
  "hexadecimal word",           // T_HEX16
  "hexadecimal triplet",        // T_HEX24
  "hexadecimal long",           // T_HEX32
  "float",                      // T_FLOAT
  "real",                       // T_REAL
  "double",                     // T_DOUBLE
  "long double",                // T_LDOUBLE
  "character"                   // T_CHAR
};

const char* const t_fmt[][2] =	// printf format specifier
{
  { "",       ""         },     // T_IGN
  { "%*d",    "% 0*d"    },     // T_INT8
  { "%*hd",   "% 0*hd"   },     // T_INT16
  { "%*ld",   "% 0*ld"   },     // T_INT24
  { "%*ld",   "% 0*ld"   },     // T_INT32
  { "%*u",    "%0*u"     },     // T_UINT8
  { "%*hu",   "%0*hu"    },     // T_UINT16
  { "%*lu",   "%0*lu"    },     // T_UINT24
  { "%*lu",   "%0*lu"    },     // T_UINT32
  { "%*x",    "%0*x"     },     // T_HEX8
  { "%*hx",   "%0*hx"    },     // T_HEX16
  { "%*lx",   "%0*lx"    },     // T_HEX24
  { "%*lx",   "%0*lx"    },     // T_HEX32
  { "%*.*f",  "% 0*.*f"  },     // T_FLOAT
  { "%*.*f",  "% 0*.*f"  },     // T_REAL
  { "%*.*f",  "% 0*.*f"  },     // T_DOUBLE
  { "%*.*Lf", "% 0*.*Lf" }      // T_LDOUBLE
};

const char* const t_flt[][2] =	// Alternative printf format for floats
{
  { "%*.*e",  "% 0*.*e"  },     // T_FLOAT
  { "%*.*e",  "% 0*.*e"  },     // T_REAL
  { "%*.*e",  "% 0*.*e"  },     // T_DOUBLE
  { "%*.*Le", "% 0*.*Le" }      // T_LDOUBLE
};

const int t_width[] =		// String length
{
   0,				// T_IGN	123456789012345678
   4,				// T_INT8	-128
   6,				// T_INT16	-32768
   8,				// T_INT24	-8388608
  11,				// T_INT32	-2147483648
   3,				// T_UINT8	255
   5,				// T_UINT16	65535
   8,				// T_UINT24	16777215
  10,				// T_UINT32	4294967295
   2,				// T_HEX8	ff
   4,				// T_HEX16	ffff
   6,				// T_HEX24	ffffff
   8,				// T_HEX32	ffffffff
  11,				// T_FLOAT	-123456.789
  13,				// T_REAL	-12345678.901
  14,				// T_DOUBLE	-123456789.123
  17,				// T_LDOUBLE	-123456789123.456
   1,				// T_CHAR
};

#define MAXSTR 1040		// Maximum length of variable-length strings

const char* const t_scan[] =	// scanf format specifier
{
  "%f",                         // T_FLOAT
  "%lf",                        // T_REAL
  "%lf",                        // T_DOUBLE
  "%Lf",                        // T_LDOUBLE
  " %c",                        // T_CHAR
};

int t_bytes[] = 		// Byte length
{
   1,				// T_IGN
   1,				// T_INT8
   2,				// T_INT16
   3,				// T_INT24
   4,				// T_INT32
   1,				// T_UINT8
   2,				// T_UINT16
   3,				// T_UINT24
   4,				// T_UINT32
   1,				// T_HEX8
   2,				// T_HEX16
   3,				// T_HEX24
   4,				// T_HEX32
   4,				// T_FLOAT
   6,				// T_REAL
   8,				// T_DOUBLE
  10,				// T_LDOUBLE
   1,				// T_CHAR (or 2 if Unicode)
};

const INT32 t_irange[][2] =	// Range of signed integers
{
  {	-128,	   127	},	// T_INT8
  {   -32768L,	 32767	},	// T_INT16
  { -8388608L, 8388607L }	// T_INT24
};

const UINT32 t_umax[] = 	// Maximum value of unsigned integers
{
       255,			// T_UINT8,  T_HEX8
     65535u,			// T_UINT16, T_HEX16
  16777215uL,			// T_UINT24, T_HEX24
};


#define MAX_ITEMS 20

typedef struct
{
  int count;			// Number of times the item occurs or
  int type;			//  maximum length of strings
  int width;			// Minimum field width
  int prec;			// Floating point precision (# of decimals)
  int sep;			// Separating spaces/tabs
  int flag;
} Item;

#define f_zero	1		// Pad with zeros, not spaces
#define f_padD	2		// Long doubles are 12 bytes (10 + 2 pad)
#define f_tab	4		// Use tabs instead of spaces for separation
#define f_uni	8		// Unicode characters
#define f_end  16		// Use non-native endian format
#define f_exp  32		// Floats use exponent

typedef struct
{
  int  count;			// Number of times the record is repeated
  int  items;			// Number of items comprising the record
  Item item[MAX_ITEMS];
  int  len;			// The string length of the record
} Record;

typedef union			// Used to convert between real & double
{
  double d;
  UINT32 L[2];
  UINT16 w[4];
  UINT8  b[8];
} Double;


int    hex   = 0,		// Hex dump
       ascii = 0,		// Character dump
       num   = 16,		// Number of bytes per line
       ofs,			// Display offset?
       format = 0,		// Output a "formatted" file
       endian = 0;		// Use non-native endian format
int    line_num;		// Possible line number for record errors
UINT8* byte;			// Array to hold the bytes

// Value for filter.
#define ALNUM 1 		// Default for dump
#define ASCII 2 		// -a; default for records
#define ALL   3 		// -A (except control and Unicode characters)

#if 1				// PCs can display all characters, but isprint
#define isprnt(ch) ((ch) >= 32) //  is not configured to do so
#else
#define isprnt(ch) isprint(ch)
#endif

// Display invalid characters as dots.
#define PRINT( ch, filt ) \
  ((((filt) == ALNUM) ? isalnum(ch) : \
    ((filt) == ASCII) ? (isascii(ch) && !iscntrl(ch)) : \
			isprnt(ch)) ? (ch) : '.')


/*
  Display a hex and/or character dump of size bytes,
  or to end of file if size is negative.
*/
void dump( FILE* bin, FILE* txt, long size )
{
  long offset;
  int  n, j;

  while (size != 0)
  {
    offset = ftell( bin );
    n = (int)fread( byte, 1, num, bin );
    if (n == 0)
      break;
    if (size > 0 && n > size)
      n = (int)size;
    if (ofs)
      fprintf( txt, "%06lx  ", offset );
    if (hex)
    {
      for (j = 0; j < n; ++j)
	fprintf( txt, "%02x ", byte[j] );
    }
    if (ascii)
    {
      if (hex)
	fprintf( txt, "%*c", 3 + 3 * (num - n), ' ' );
      for (j = 0; j < n; ++j)
	putc( PRINT( byte[j], ascii ), txt );
    }
    putc( '\n', txt );
    size -= n;
  }
}


/*
  Parse the record format string fmt into the record rec.
  Returns 1 for success; otherwise 0 with an error message displayed.
*/
int process_record( Record* rec, char* fmt )
{
  int	n, t;
  int	sign;
  char* end = strchr( fmt, 0 );
  char* bp;
  int	i, j, k;
  int	rc = 0;
  char* err = fmt;
  static const char* const err_str[] =
  {
    "Count must be greater than zero",
    "Unknown type",
    "Repeat must be last",
    "Too many formats",
    "Expecting width"
  };

  if (fmt == end)
  {
    fputs( "Missing record format.\n", stderr );
    return 0;
  }

  rec->count = 1;
  i = 0;
  j = 1;					// The NUL

  while (fmt < end)
  {
    err = fmt;					// Where the error occurs

    if (*fmt == '*')                            // Repeat count
    {
      n = (int)strtol( fmt+1, &bp, 10 );
      if (bp != end)				// Only valid if it's the
	rc = 3; 				//  last item
      else if (n <= 0)
	rc = 1;
      else
	rec->count = n;
      break;
    }

    if (i == MAX_ITEMS) 			// Too many items
    {
      rc = 4;
      break;
    }

    n = (int)strtol( fmt, &bp, 10 );
    if (bp == fmt)				// No count is the same as one
      n = 1;
    else if (n <= 0)
    {
      rc = 1;
      break;
    }
    else
      fmt = bp; 				// Point past the number

    sign = 0;
    rec->item[i].flag = endian;
    for (;;)
    {
      if (*fmt == '-')
	sign = 1;
      else if (*fmt == '~')
	rec->item[i].flag ^= f_end;
      else if (*fmt == 'u')
	rec->item[i].flag |= f_uni;
      else
	break;
      ++fmt;
    }

    t = -1;
    switch (*fmt)
    {
      case 'i': t = T_IGN;     break;

      case 'b': t = (sign) ? T_INT8  : T_UINT8;  break;
      case 'w': t = (sign) ? T_INT16 : T_UINT16; break;
      case 't': t = (sign) ? T_INT24 : T_UINT24; break;
      case 'l': t = (sign) ? T_INT32 : T_UINT32; break;

      case 'B': t = T_HEX8;    break;
      case 'W': t = T_HEX16;   break;
      case 'T': t = T_HEX24;   break;
      case 'L': t = T_HEX32;   break;

      case 'f': t = T_FLOAT;   break;
      case 'r': t = T_REAL;    break;
      case 'd': t = T_DOUBLE;  break;
      case 'G': rec->item[i].flag |= f_padD;
      case 'D': t = T_LDOUBLE; break;

      case 'c': t = T_CHAR;    break;
      case 'z': t = T_ZERO;    break;
      case 'p': t = T_LEN8;    break;
      case 's': t = T_LEN16;   break;
      case 'S': t = T_LEN32;   break;
    }
    if (t == -1)		// Unknown type
    {
      rc = 2;
      break;
    }
    ++fmt;

    rec->item[i].count = n;
    rec->item[i].type  = t;
    rec->item[i].width = (t >= T_ZERO) ? n : t_width[t];
    rec->item[i].prec  = 3;

    // Check for width (and zero-padding, precision)
    if (*fmt == ':')
    {
      ++fmt;
      if ((*fmt < '0' || *fmt > '9') && *fmt != '.' && *fmt != 'e')
      {
	err = fmt - 1;
	rc  = 5;
	break;
      }
      if (*fmt == '0')
	rec->item[i].flag |= f_zero, ++fmt;
      if (*fmt >= '1' && *fmt <= '9')
	rec->item[i].width = (int)strtol( fmt,	 &fmt, 10 );
      if (*fmt == '.')
	rec->item[i].prec  = (int)strtol( fmt+1, &fmt, 10 );
      if (*fmt == 'e')
	rec->item[i].flag |= f_exp, ++fmt;
    }
    // Zero-pad hex by default (specify a width to blank-pad)
    else if (t >= T_HEX8 && t <= T_HEX32)
      rec->item[i].flag |= f_zero;

    // Check for separator(s)
    if (*fmt == '+')
      rec->item[i].sep = 0, ++fmt;
    else
    {
      rec->item[i].sep = 1;
      if (*fmt == ';' || *fmt == ',')
      {
	if (*fmt == ',')
	  rec->item[i].flag |= f_tab;
	while (*++fmt == ';' || *fmt == ',')
	  ++rec->item[i].sep;
      }
      else if (t == T_IGN || t == T_CHAR)
	rec->item[i].sep = 0;
    }

    // Determine character length of the type
    k = rec->item[i].width;
    if (t >= T_ZERO)
    {
      j += k + rec->item[i].sep + 2;	// +2 for the quotes
      if (n == 1)			// Unknown string length,
	j += MAXSTR;			//  so allow lots of room
    }
    else
    {
      if (t_width[t] > k)		// Use maximum in case of
	k = t_width[t]; 		//  unexpected data
      j += n * (k + rec->item[i].sep);
    }
    ++i;
  }
  rec->items = i;
  rec->len   = j;

  if (rc)
    fprintf( stderr, "%s: %s\n", err_str[rc-1], err );

  return (rc == 0);
}


/*
  Read a number of bytes from a file. If rev is true, the bytes are reversed.
  Returns 0 if EOF occurred.
*/
int oread( void* buf, int num, int rev, FILE* file )
{
  if (rev)
  {
    UINT8* p = buf;
    p += num;
    do
    {
      *--p = getc( file );
    } while (--num);
  }
  else
    fread( buf, num, 1, file );

  return !feof( file );
}


/*
  Write a number of bytes to a file. If rev is true, the bytes are reversed.
*/
void owrite( void* buf, int num, int rev, FILE* file )
{
  if (rev)
  {
    UINT8* p = buf;
    p += num;
    do
    {
      putc( *--p, file );
    } while (--num);
  }
  else
    fwrite( buf, num, 1, file );
}


/*
  Functions to convert between real and double (in-place).

  The real format (1 sign, 39 mantissa, 8 exponent):
		       SMMMMMMM MMMMMMMM MMMMMMMM MMMMMMMM MMMMMMMM EEEEEEEE
		       5	4	 3	  2	   1	    0

  The double format (1 sign, 52 mantissa, 11 exponent):
     SEEEEEEE EEEEMMMM MMMMMMMM MMMMMMMM MMMMMMMM MMMMMMMM MMMMMMMM MMMMMMMM
     7	      6        5	4	 3	  2	   1	    0
*/
void rtod( double* real )
{
  Double* dbl = (Double*)real;

  dbl->b[6] = 0;
  dbl->L[1] <<= 5;
  dbl->b[7] = (dbl->b[6] & 0x10) << 3;
  dbl->b[6] &= ~0x10;
  dbl->L[1] |= ((UINT32)(dbl->b[0] + 1023 - 129) << 20) | (dbl->b[3] >> 3);
  dbl->b[0] = 0;
  dbl->L[0] <<= 5;
}

void dtor( double* dbl )
{
  Double* real = (Double*)dbl;
  int	  rnd  = real->b[1] & 0x10;

  real->L[0] >>= 5;
  real->b[3] |= real->b[4] << 3;
  real->b[0] = ((real->w[3] & 0x7fff) >> 4) - 1023 + 129;
  real->b[6] &= ~0x10;
  real->b[6] |= (real->b[7] & 0x80) >> 3;
  real->L[1] >>= 5;

  if (rnd)
  {
    real->L[0] += 256;
    if ((real->L[0] & 0xffffff00uL) == 0)
      ++real->L[1];		// Just hope it doesn't overflow
  }
}


/*
  Display a record. If end-of-file occurs before a complete (individual)
  record is displayed, display a dump instead.
*/
void display_record( Record* rec, FILE* bin, FILE* txt )
{
  int	 i, j, k, r, s;
  int	 t, z, w, p, u, e;
  UINT32 n, len;
  int	 ch = 0;
  long	 offset;
  int	 partial = 0;
  char*  fmt;
  const char* flt;
  char	 sep;
  union
  {
    INT8	i8;
    INT16	i16;
    INT32	i32;
    UINT8	u8;
    UINT16	u16;
    UINT32	u32;
    float	f;
    double	d;
    long double ld;
  } v;

  fmt = malloc( rec->len );
  if (fmt == NULL)
  {
    fputs( "Not enough memory to store record.\n", stderr );
    exit( 3 );
  }

  for (k = rec->count; k > 0; --k)
  {
    offset = ftell( bin );
    i = 0;
    for (j = 0; j < rec->items; ++j)
    {
      sep = (rec->item[j].flag & f_tab) ? '\t' : ' ';
      u   = rec->item[j].flag & f_uni;
      e   = rec->item[j].flag & f_end;
      t   = rec->item[j].type;
      if (t >= T_ZERO)			// String types
      {
	fmt[i++] = '"';
	r = 0;
	if (t == T_ZERO)
	{
	  len = rec->item[j].count;
	  n   = (len == 1) ? MAXSTR : len;
	}
	else
	{
	  switch (t)
	  {
	    case T_LEN8:
	      len = getc( bin );
	    break;
	    case T_LEN16:
	      len = 0;
	      oread( &len, 2, e, bin );
	    break;
	    case T_LEN32:
	      oread( &len, 4, e, bin );
	    break;
	  }
	  if (len > rec->item[j].count && rec->item[j].count != 1)
	  {
	    partial = 1;
	    break;
	  }
	  n = len;
	}
	for (; n > 0; --n)
	{
	  if (u)
	  {
	    oread( &ch, 2, e, bin );
	    if (ch > 255)
	      ch = '.';
	  }
	  else
	    ch = getc( bin );
	  if (feof( bin ) || (t == T_ZERO && ch == '\0'))
	    break;
	  fmt[i++] = PRINT( ch, ALL );
	  ++r;
	}
	if (t == T_ZERO)
	{
	  if (ch != '\0' && ch != EOF)
	  {
	    if (u)
	      oread( &ch, 2, e, bin );
	    else
	      ch = getc( bin );
	    if (feof( bin ) || ch != '\0')
	    {
	      partial = 1;
	      break;
	    }
	  }
	  if (len == 1)
	    n = 0;
	}
	else if (rec->item[j].count != 1)
	  n = rec->item[j].count - len;
	if (!feof( bin ))
	{
	  fmt[i++] = '"';
	  for (; r < rec->item[j].width; ++r)
	    fmt[i++] = ' ';
	  for (; n > 0 && !feof( bin ); --n)
	  {
	    fgetc( bin );
	    if (u)
	      fgetc( bin );
	  }
	  for (s = rec->item[j].sep; s > 0; --s)
	    fmt[i++] = sep;
	}
      }
      else
      {
	w = rec->item[j].width;
	p = rec->item[j].prec;
	z = rec->item[j].flag & f_zero;
	flt = t_fmt[t][z];
	if ((rec->item[j].flag & f_exp) && t >= T_FLOAT && t <= T_LDOUBLE)
	  flt = t_flt[t - T_FLOAT][z];
	if (t_bytes[t] == 3) // T_INT24, T_UINT24, T_HEX24
	  v.u32 = 0;
	for (r = rec->item[j].count; r > 0 && !feof( bin ); --r)
	{
	  if (oread( &v, t_bytes[t], e, bin ))
	  switch (t)
	  {
	    case T_IGN:
	      if (w && r == 1)
		i += sprintf( fmt+i, "%*c", w, ' ' );
	    break;

	    case T_INT8:
	      i += sprintf( fmt+i, t_fmt[t][z], w, v.i8 );
	    break;

	    case T_INT16:
	      i += sprintf( fmt+i, t_fmt[t][z], w, v.i16 );
	    break;

	    case T_INT24:
	      if (v.i32 > 0x7fffffL)
		v.i32 -= 0x1000000L;
	    case T_INT32:
	      i += sprintf( fmt+i, t_fmt[t][z], w, v.i32 );
	    break;

	    case T_UINT8:
	    case T_HEX8:
	      i += sprintf( fmt+i, t_fmt[t][z], w, v.u8 );
	    break;

	    case T_UINT16:
	    case T_HEX16:
	      i += sprintf( fmt+i, t_fmt[t][z], w, v.u16 );
	    break;

	    case T_UINT24:
	    case T_HEX24:
	    case T_UINT32:
	    case T_HEX32:
	      i += sprintf( fmt+i, t_fmt[t][z], w, v.u32 );
	    break;

	    case T_FLOAT:
	      i += sprintf( fmt+i, flt, w, p, v.f );
	    break;

	    case T_REAL:
	      rtod( &v.d );
	    case T_DOUBLE:
	      i += sprintf( fmt+i, flt, w, p, v.d );
	    break;

	    case T_LDOUBLE:
	      if (rec->item[j].flag & f_padD)
		fgetc( bin ), fgetc( bin );
	      i += sprintf( fmt+i, flt, w, p, v.ld );
	    break;

	    case T_CHAR:
	      if (u)
	      {
		ch = v.u16;
		if (ch > 255)
		  ch = '.';
	      }
	      else
		ch = v.u8;
	      i += sprintf( fmt+i, "%*c", w, PRINT( ch, ALL ) );
	    break;
	  }
	  // Remove trailing zeros after the point.
	  if (t >= T_FLOAT && t <= T_LDOUBLE && p > 0 && !z &&
	      !(rec->item[j].flag & f_exp))
	  {
	    s = i;
	    while (fmt[--s] == '0')
	      fmt[s] = ' ';
	    if (fmt[s] == '.')
	      fmt[s] = ' ';
	  }
	  for (s = rec->item[j].sep; s > 0; --s)
	    fmt[i++] = sep;
	}
      }
      if (i >= rec->len)
      {
	fprintf( stderr, "Buffer overflow; try increasing width by %d.\n",
			 i - rec->len + 1 );
	_exit( 3 );
      }
    }
    if (feof( bin ) || partial)
    {
      if (offset != ftell( bin ))	// EOF occurred before reading anything
      {
	long size = ftell( bin ) - offset;
	fputs( "\b\n", txt );
	fseek( bin, offset, SEEK_SET );
	dump( bin, txt, size );
      }
      break;
    }
    else
    {
      while (fmt[i-1] == ' ' || fmt[i-1] == '\t')
	--i;
      fmt[i] = '\0';
      if (ofs)
	fprintf( txt, "%06lx  ", offset );
      fprintf( txt, "%s\n", fmt );
    }
  }
  free( fmt );
}


/*
  Read a text representation of a record and write the binary equivalent.
*/
void write_record( Record* rec, FILE* txt, FILE* bin )
{
  int	 j, k, r;
  int	 t, u, e, p;
  int	 n;
  int	 ch;
  char	 fmt[16];
  char*  str;
  union
  {
    int 	i;
    INT32	i32;
    UINT32	u32;
    double	d;
    long double ld;
  } v;

  str = malloc( rec->len );
  if (str == NULL)
  {
    fputs( "Not enough memory to store record.\n", stderr );
    exit( 3 );
  }

  for (k = rec->count; k > 0; --k)
  {
    for (j = 0; j < rec->items; ++j)
    {
      u = rec->item[j].flag & f_uni;
      e = rec->item[j].flag & f_end;
      r = rec->item[j].count;
      t = rec->item[j].type;
      if (t == T_IGN)
      {
	fseek( bin, r, SEEK_CUR );
      }
      else if (t >= T_ZERO)		// String types
      {
	sprintf( fmt, " \"%%%d[^\"]", (r == 1) ? MAXSTR : r );
	if (fscanf( txt, fmt, str ) != 1)
	{
	  if (getc( txt ) != '"')       // scanf won't read an empty string
	  {
	    t = -1;			// Expecting "string"
	    goto error;
	  }
	  *str = '\0';
	}
	else if (getc( txt ) != '"')
	{
	  t = -2;			// String too long
	  goto error;
	}
	// Todo: recognise quotes (embedded in dump; use '\x22'?)
	//	 control characters ('.' in dump; use '\xnn'?)
	//	 Unicode characters ('.' in dump; use '\unnnn'?)
	n = strlen( str );
	if (t >= T_LEN8)
	{
	  v.u32 = n;
	  owrite( &v.u32, 1 << (t - T_LEN8), e, bin );
	}
	if (u)
	{
	  UINT8* s;
	  for (s = (UINT8*)str; *s; ++s)
	  {
	    ch = *s;
	    owrite( &ch, 2, e, bin );
	  }
	}
	else
	  fwrite( str, 1, n, bin );
	n = (r == 1) ? 0 : r - n;
	if (t == T_ZERO)
	  ++n;
	if (u)
	  n <<= 1;
	for (; n > 0; --n)
	  putc( 0, bin );
      }
      else
      {
	if (t >= T_FLOAT)		// Floating point and character
	{
	  strcpy( fmt, t_scan[t - T_FLOAT] );
	  if (t == T_CHAR)
	  {
	    t_bytes[T_CHAR] = (u) ? 2 : 1;
	    v.i = 0;
	  }
	}
	else				// Integers
	{
	  sprintf( fmt, "%%%dl%c", rec->item[j].width,
				   (t >= T_HEX8)  ? 'x' :
				   (t >= T_UINT8) ? 'u' : 'd' );
	}
	p = rec->item[j].flag & f_padD;
	for (; r > 0; --r)
	{
	  if (fscanf( txt, fmt, &v ) != 1)
	    goto error;
	  if (t == T_REAL)
	    dtor( &v.d );
	  else if (t == T_CHAR) 	// Todo: same as string,
	  {				//	  as well as space ('\x20'?)
	    if (v.i == '\b')
	      goto hex;
	  }
	  else if (t <= T_HEX24)	// Check integers for overflow
	  {
	    if (t < T_INT32)
	    {
	      if (v.i32 < t_irange[t - T_INT8][0] ||
		  v.i32 > t_irange[t - T_INT8][1])
		goto error;
	    }
	    else if (t != T_UINT32)
	    {
	      if (v.u32 > t_umax[(t - 1) & 3])
		goto error;
	    }
	  }
	  owrite( &v, t_bytes[t], e, bin );
	  if (p)
	    putc( 0, bin ), putc( 0, bin );
	}
      }
    }
    ++line_num;
  }
  free( str );
  return;

error:
  // Check if EOF occurred before reading a record.
  if (feof( txt ) && k == rec->count && j == 0 && r == rec->item[0].count)
  {
    free( str );
    return;
  }

  if (getc( txt ) == '\b')
  {
  hex:
    while (fscanf( txt, "%2x", &ch ) == 1)
      putc( ch, bin );
    free( str );
    return;
  }

  fprintf( stderr, "Error at item %d", j + 1 );
  if (t > 0 && rec->item[j].count > 1)
    fprintf( stderr, " (number %d)", rec->item[j].count - r + 1 );
  if (rec->count > 1)
    fprintf( stderr, ", repeat %d", rec->count - k + 1 );
  fprintf( stderr, " (line %d?): ", line_num );
  if (t == -1)
    fputs( "expecting \"string\".\n", stderr );
  else if (t == -2)
    fputs( "string too long.\n", stderr );
  else
    fprintf( stderr, "expecting %s.\n", t_name[t] );
  exit( 4 );
}


/*
  Copy the old filename to new, replacing the extension with ext (which
  should begin with a dot).
*/
void replace_ext( char* new, const char* old, const char* ext )
{
  char* dot = NULL;

  while (*old)
  {
    if (*old == '.')
      dot = new;
    *new++ = *old++;
  }
  strcpy( (dot == NULL) ? new : dot, ext );
}


void help( char* program );


int main( int argc, char* argv[] )
{
  FILE*  in;				// File to read
  FILE*  out;				// File to write
  int	 records = 0,			// Record dump
	 write	 = 0,			// Read text file, write binary
	 filter  = ASCII,		// Type of character dump
	 ofsarg  = 0;			// Seen -o
  long	 offset  = 0,			// Initial offset into file
	 size	 = -1;			// Number of bytes/records to display
  Record rec[MAX_ITEMS];		// Record formats
  char	 arg, recarg[64];
  char	 line[128], name[260];
  int	 filearg = 0,			// Position of file in arg. list
	 fileend = 0,			// Position of last file
	 multi; 			// More than one file
  char*  bp;
  int	 j, k;

  if (argc > 1 && (argv[1][0] == '?' || argv[1][1] == '?' ||
		   strcmp( argv[1], "--help" ) == 0))
  {
    help( argv[0] );
    return 0;
  }

  *line = '\0';                         // String containing all records
  for (j = 1; j < argc; ++j)
  {
    arg = argv[j][0];
    if (arg == '-')
    {
      for (k = 1; argv[j][k]; ++k)
      {
	switch (argv[j][k])
	{
	  case '~':
	    endian ^= f_end;
	    strcat( line, " -~" );
	  break;

	  case 'r':
	    if (records == MAX_ITEMS)
	    {
	      fputs( "Too many records.\n", stderr );
	      return 1;
	    }
	    if (!process_record( &rec[records], argv[j] + k + 1 ))
	      return 1;
	    ++records;
	    strcat( line, " -r" );
	    strcat( line, argv[j] + k + 1 );
	    k = strlen( argv[j] ) - 1; // break for
	  break;

	  case 'f': format = 1;     break;
	  case 'w': write  = 1;     break;
	  case 'o': ofsarg = 1;     break;
	  case 'x': hex    = 1;     break;
	  case 'c': ascii  = 1;     break;
	  case 'a': filter = ALNUM; break;
	  case 'A': filter = ALL;   break;

	  default:
	    num = (int)strtol( argv[j] + k, &bp, 0 );
	    if (num <= 0)
	    {
	      fprintf(stderr, "Bad width or unknown option: %c\n", argv[j][k]);
	      return 1;
	    }
	    k = bp - argv[j] - 1;
	  break;
	}
      }
    }
    else if (arg == '+')
    {
      offset = strtol( argv[j] + 1, &bp, 0 );
      if (offset < 0)
      {
	size   = -offset + 1;
	offset = 0;
      }
      else if (*bp == '-')
      {
	size = strtol( bp + 1, NULL, 0 );
	if (size == 0 || size < offset)
	{
	  fprintf( stderr, "Bad offset: %s\n", argv[j] );
	  return 1;
	}
	size -= offset - 1;
      }
    }
    else if (arg == '=')
    {
      size = strtol( argv[j] + 1, NULL, 0 );
    }
    else
    {
      if (filearg == 0)
	filearg = fileend = j;
      else if (++fileend != j)		// Keep all the files together
      {
	char* temp = argv[j];
	argv[j] = argv[fileend];
	argv[fileend] = temp;
      }
    }
  }
  if (write && format && records)
  {
    fputs( "Cannot have both format and records with write.\n", stderr );
    return 1;
  }

  if (format)				// Ensure formatted files have no
  {					//  offset and no character display
    ofs = ascii = 0;			//  in the dump
    hex = 1;
  }
  else
  {
    if (ascii)
      ascii = filter;
    else if (!hex && !ascii)		// Not specifying either means both
    {
      hex   = 1;
      ascii = filter;
    }
    ofs = (records == 0);		// Display offset for dump,
    if (ofsarg) 			//  but not for records
      ofs = !ofs;			// But invert that if -o is given
  }

  byte = malloc( num );
  if (byte == NULL)
  {
    fputs( "Not enough memory.\n", stderr );
    return 3;
  }

  multi = (filearg != fileend);
  for (; filearg <= fileend; ++filearg)
  {
    if (write)	// text to binary
    {
      if (filearg == 0)
	in = stdin;
      else
      {
	if ((in = fopen( argv[filearg], "r" )) == NULL)
	{
	  fprintf( stderr, "%s: unable to open.\n", argv[filearg] );
	  if (!multi)
	    return 2;
	  continue;
	}
      }
      if (format)
      {
	int bad = 1;
	if (fscanf( in, "\"%[^\"]\" +%li", name, &offset ) == 2)
	{
	  fgets( line, sizeof(line), in );
	  bad = 0;
	  j = 0;
	  while (sscanf( line + j, " -%s%n", recarg, &k ) == 1)
	  {
	    j += k;
	    if (recarg[0] == '~')
	      endian ^= f_end;
	    else if (recarg[0] != 'r'
		     || !process_record( &rec[records], recarg + 1 ))
	    {
	      bad = 1;
	      break;
	    }
	    ++records;
	  }
	}
	if (bad)
	{
	  fprintf( stderr, "%s: not a formatted XD file.\n",
			   (filearg == 0) ? "stdin" : argv[filearg] );
	  if (!multi)
	    return 2;
	  fclose( in );
	  continue;
	}
	if (*name == '-' && name[1] == '\0')
	  out = stdout;
	else
	{
	  out = fopen( name, "rb+" );
	  if (out == NULL)
	    out = fopen( name, "wb" );
	}
      }
      else
      {
	if (filearg == 0)
	  out = stdout;
	else
	{
	  replace_ext( name, argv[filearg], ".xdb" );
	  out = fopen( name, "wb" );
	}
      }
      if (out == stdout)
      {
// djgpp doesn't like stdout being binary if reading from the keyboard
#ifdef __DJGPP__
	if (!isatty( 1 ) || !isatty( 0 ))
#endif
	setmode( 1, O_BINARY );
      }
      else if (out == NULL)
      {
	fprintf( stderr, "%s: unable to open/create.\n", name );
	if (!multi)
	  return 2;
	fclose( in );
	continue;
      }

      fseek( out, offset, SEEK_SET );

      if (records == 0)
      {
	while (fscanf( in, "%2x", &j ) == 1)
	  putc( j, out );
      }
      else
      {
	line_num = (format) ? 3 : 1;
	do
	{
	  for (j = 0; j < records && !feof( in ); ++j)
	    write_record( &rec[j], in, out );
	  if (records > 1)
	    ++line_num;
	} while (!feof( in ));
      }

      fclose( in );
      fclose( out );
    }
    else	// binary to text
    {
      if (filearg == 0)
      {
	in = stdin;
	if (!isatty( 0 ))
	  setmode( 0, O_BINARY );
      }
      else
      {
	if ((in = fopen( argv[filearg], "rb" )) == NULL)
	{
	  fprintf( stderr, "%s: unable to open.\n", argv[filearg] );
	  if (!multi)
	    return 2;
	  if (!format && isatty( 1 ) && filearg != fileend)
	    putchar( '\n' );
	  continue;
	}
	if (!format && multi)
	  printf( "%s:\n\n", argv[filearg] );
      }
      if (!format || filearg == 0)
	out = stdout;
      else
      {
	replace_ext( name, argv[filearg], ".xdt" );
	if ((out = fopen( name, "w" )) == NULL)
	{
	  fprintf( stderr, "%s: unable to create.\n", name );
	  if (!multi)
	    return 2;
	  fclose( in );
	  continue;
	}
      }
      if (format)
      {
	fprintf( out, "\"%s\" +%ld%s\n\n",
		      (filearg == 0) ? "-" : argv[filearg], offset, line );
      }

      fseek( in, offset, SEEK_SET );

      if (records == 0)
	dump( in, out, size );
      else
      {
	long s = size;
	while (!feof( in ) && s != 0)
	{
	  if (records > 1 && s != size)
	    putc( '\n', out );
	  for (j = 0; j < records && !feof( in ); ++j)
	    display_record( &rec[j], in, out );
	  --s;
	}
      }
      if (!format)
      {
	if (records > 1 && !feof( in ))
	  putchar( '\n' );
	if (multi && filearg != fileend)
	  putchar( '\n' );
      }

      fclose( in );
      if (format)
	fclose( out );
    }
  }

  return 0;
}


void help( char* program )
{
printf(
"XD by Jason Hood <jadoxa@yahoo.com.au>.\n"
"Version "PVERS" ("PDATE"). Freeware.\n"
"http://xd.adoxa.cjb.net/\n"
"\n"
"Dump files (or standard input if none) as a series of hexadecimal bytes\n"
"and/or printable characters, or as a set of records. Conversely, turn a\n"
"dump into a file.\n"
"\n"
"Usage: %s [options] [files] [options]\n"
"\n"
"-x                      Hex dump\n"
"-c                      Character dump (default is ASCII)\n"
"-a                      Use an alphanumeric character dump\n"
"-A                      Dump all printable characters\n"
"-<num>                  Num bytes per line (default is 16)\n"
"-r<record>              Use a record format (up to %d allowed)\n"
"-~                      Reverse the byte order of all types (must precede -r)\n"
"-f                      Use a \"formatted\" record file\n"
"-w                      Write (or update) a binary file from a text file\n"
"-o                      Turn offset display off (dump) or on (record)\n"
"+[[<offset>]-]<offset>  Start and finish file offsets (inclusive)\n"
"=<count>                Number of bytes or records to display\n"
"\n"
"Numbers can be prefixed with 0x for hexadecimal or 0 for octal.\n"
"\n"
"Record format:\n"
"       [<count>][-][~][u]<type>[:[0][<width>][.<prec>][e]][;|,|+]...[*<repeat>]\n"
"\n"
"<count>   number of times to repeat the type or the maximum length\n"
"           of strings. If omitted (or one) for a string, the length\n"
"           of the string itself will be used (variable-length record)\n"
"\n"
"-         use signed decimal numbers\n"
"~         reverse the byte order\n"
"u         characters are in Unicode (two bytes)\n"
"\n"
"<type>    b - Decimal byte                B - Hexadecimal byte\n"
"          w - Decimal word (two bytes)    W - Hexadecimal word\n"
"          t - Decimal triplet (3 bytes)   T - Hexadecimal triplet\n"
"          l - Decimal long (four bytes)   L - Hexadecimal long\n"
"          f - Float (four bytes)          c - Character\n"
"          r - Real (six bytes)            z - Zero-terminated string (C, ASCIZ)\n"
"          d - Double (eight bytes)        p - Byte-length string (Pascal)\n"
"          D - Long double (ten bytes)     s - Word-length string\n"
"          G - Long double (ten+two pad)   S - Long-length string\n"
"          i - Ignore byte\n"
"\n"
"0         display numbers with leading zeros (default for hex without width)\n"
"<width>   minimum number of characters to display the type (default is max.)\n"
"<prec>    number of decimals to display (default is three)\n"
"e         display floating point numbers with an exponent\n"
"\n"
";         separate types with a space (can accumulate; default)\n"
",         separate types with a tab (can accumulate)\n"
"+         types will not be separated\n"
"\n"
"<repeat>  number of times to repeat the entire record\n"
, program, MAX_ITEMS );
}
