Program SortDemo;
(* SortDemo Version 2.4: 19 Mar 1995                                        *)

(* Copyright Tapirsoft (C) 1995, Harald Selke                               *)
(* Based on a programme by K.L. Noell.                                      *)
(* Demonstration programme for sorting algorithms, integrated version       *)
(* See SortDemo.Doc for full documentation.                                 *)
(*                                                                          *)
(* This programme is free software; you can redistribute it and/or modify   *)
(* it under the terms of the GNU General Public License (version 1) as      *)
(* published by the Free Software Foundation.                               *)
(*                                                                          *)
(* This programme is distributed in the hope that it will be useful,        *)
(* but WITHOUT ANY WARRANTY; without even the implied warranty of           *)
(* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *)
(* GNU General Public License for more details.                             *)
(*                                                                          *)
(* You should have received a copy of the GNU General Public License        *)
(* along with this programme; if not, write to the Free Software            *)
(* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                *)


{$A+}                        (* Word alignment on                   *)
{$B-}                        (* Boolean complete evaluation off     *)
{$D+}                        (* Debug information on                *)
{$E+}                        (* Emulate coprocessor if necessary    *)
{$F-}                        (* Don't force far calls               *)
{$G-}                        (* Don't generate 80286 code           *)
{$I+}                        (* I/O checking on                     *)
{$L+}                        (* Generate local symbol information   *)
{$N-}                        (* No numeric processing               *)
{$O-}                        (* No overlaying                       *)
{$R+}                        (* Range checking on                   *)
{$S+}                        (* Stack checking on                   *)
{$V-}                        (* No var-string checking              *)
{$X-}                        (* No extended syntax allowed          *)
{$M 65520,0,655360}          (* Stack and heap size                 *)

Uses Crt, Graph, Menu, SortAlgs, SortInps;

Const NrAlgs      = 8;              (* Number of implemented algorithms     *)
      MaxDelay    = 1000;           (* Maximum delay between two operations *)

      Logo        : MenuLine = '(C) Tapirsoft, H. Selke';
      AlgMenu     : MenuType = ('SortDemo 2.4', '1. Bubblesort',
                                '2. Shakersort', '3. Selectionsort',
                                '4. Insertionsort', '5. Shellsort',
                                '6. Quicksort', '7. Heapsort',
                                '8. Mergesort', '9. Options',
                                '0. End SortDemo');
      AlgSelect : HotkeyType = ('BSCILQHMOE');     (* Hotkeys for selecting *)
      InputChoice : MenuType = ('', 'Random array', 'Reverse order',
                                'Sorted array','Almost sorted','File input',
                                'Cancel','','','','');

Type  String40 = String [40];

Var   achoice, ichoice, i,
      n, range, perc, WaitTime : Integer;
      OldTextAttr     : Byte;
      quit, StoreData : Boolean;
      s               : String40;
      IFName, LFName  : String12;
      ListFile        : Text;


Function li2s (l : longint) : String40;
(* Converts a long integer to a string. *)
Var s : String40;
Begin                                                               (* li2s *)
  Str (l:0,s);
  li2s := s;
End;   (* li2s *)

Function State : String40;
(* Returns a string consisting of the number of keys to be sorted and the   *)
(* current speed mode.                                                      *)
Var s : String40;
Begin                                                              (* State *)
  Str (n, s);
  s := s + ' keys, ';
  Case speed Of
    fast : s := s + 'real time mode';
    slow : s := s + 'slow motion';
    manual : s := s + 'manual mode'
  End;
  State := s
End;   (* State *)

Function Str2Int (str : String40; MinVal, MaxVal : Integer) : Integer;
(* Converts a string to an integer. If the resulting integer is greater     *)
(* than MaxVal or less than MinVal, -1 is returned.                         *)
Var value, code : Integer;
Begin                                                            (* Str2Int *)
  Val (str, value, code);
  If (code <> 0) Or (value > MaxVal) Or (value < MinVal) Then Str2Int := -1
  Else Str2Int := value
End;    (* Str2Int *)

Procedure Lower (Var s : String); Assembler;
(* Converts a string to small letters.                                      *)
(* Taken from A. Schaepers, Turbo Pascal 6.0.                               *)
asm                                                                (* Lower *)
       push ds
       lds si, s                      (* ; Addr (s) to DS:SI                *)
       push ds
       pop es                         (* ; address of return string (ES:DI) *)
       mov di,si                      (* ; is the same                      *)
       cld
       lodsb                          (* ; lengthbyte of s                  *)
       stosb                          (* ; lengthbyte of return string      *)
       mov cl, al                     (* ; this is the lengthbyte           *)
       xor ch, ch
       jcxz @LoDone                   (* ; length is 0                      *)
@LLoop:lodsb                          (* ; load one character of s into al  *)
       cmp al, 'A'                    (* ; less than 'A'?                   *)
       jb @islower                    (* ; then, don't convert              *)
       cmp al, 'Z'                    (* ; less than or equal to 'Z'?       *)
       jb @islow                      (* ; then, direct conversion          *)
       cmp al, ''                    (* ; else                             *)
       jnz @L2
       mov al, ''
       jmp @islower
@L2:   cmp al, ''
       jnz @L3
       mov al, ''
       jmp @islower
@L3:   cmp al, ''
       jnz @islower
       mov al, ''
       jmp @islower
@islow:add al, 'a'-'A'                (* ; convert the character            *)
@islower: stosb                       (* ; and store it in the return string*)
       loop @LLoop                    (* ; until cx = 0                     *)
@LoDone: pop ds
End;   (* Lower *)

Procedure ReadConfigFile;
Var ConfigFile : Text;
    Buf : Array [1..2000] Of Char;
    i, t, IORes : Integer;
    s, v : String40;

Procedure ReadConfigLine;
Var c : Char;
Begin                                                     (* ReadConfigLine *)
  s := ''; v := '';
  While (Not EoF (ConfigFile)) And (EoLn (ConfigFile)) Do
    ReadLn (ConfigFile);
  If Not EoF (ConfigFile) Then
  Begin
    Read (ConfigFile, c);
    If c <> '#' Then
    Begin
      Repeat
        If (c <> ' ') Then s := s + c;
        Read (ConfigFile, c);
      Until (EoLn (ConfigFile)) Or (c = ':');
      If Not EoLn (ConfigFile) Then
      Begin
        Repeat
          Read (ConfigFile, c);
          If (c <> '#') And (c <> ' ') Then v := v + c
        Until (EoLn (ConfigFile)) Or (c = '#');
      End
    End;
    ReadLn (ConfigFile)
  End;
End;   (* ReadConfigLine *)

Function Colour (v : String40; OldColour : Byte) : Byte;
(* Returns the number of the colour named v, or the value of OldColour if v *)
(* isn't a valid name.                                                      *)
Type PaletteType = Array [0..15] Of String [12];
Const Palette : PaletteType = ('black','blue','green','cyan','red','magenta',
                               'brown','lightgray','darkgray','lightblue',
                               'lightgreen','lightcyan','lightred',
                               'lightmagenta','yellow','white');
Var i : Integer;
Begin                                                             (* Colour *)
  i := 0;
  While (i < 16) And (v <> Palette [i]) Do Inc (i);
  If i < 16 Then Colour := i
  Else Colour := OldColour
End;   (* Colour *)

Begin                                                     (* ReadConfigFile *)
  {$I-}
  Assign (ConfigFile, 'SortDemo.Cfg');
  SetTextBuf (ConfigFile, Buf);
  Reset (ConfigFile);
  {$I+}
  IORes := IOResult;
  If IORes = 0 Then
  Begin
    While Not EoF (ConfigFile) Do
    Begin
      ReadConfigLine;
      If s <> '' Then
      Begin
        Lower (s); Lower (v);
        If s = 'dotcolour' Then DotColour := Colour (v, DotColour)
        Else If s = 'backcolour' Then BackColour  := Colour (v, BackColour)
        Else If s = 'lowcolour' Then LowColour    := Colour (v, LowColour)
        Else If s = 'highcolour' Then HighColour  := Colour (v, HighColour)
        Else If s = 'menubackcolour' Then
          MenuBackColour := Colour (v, MenuBackColour)
        Else If s = 'messcolour' Then MessColour  := Colour (v, MessColour)
        Else If s = 'framecolour' Then FrameColour := Colour (v, FrameColour)
        Else If s = 'headcolour' Then HeadColour  := Colour (v, HeadColour)
        Else If s = 'headbackcolour' Then
          HeadBackColour := Colour (v, HeadBackColour)
        Else If s = 'numberofkeys' Then
        Begin
          t := Str2Int (v,2,XMax);
          If t <> -1 Then
          Begin
            n := t; XDist := XMax Div n
          End
        End
        Else If s = 'range' Then
        Begin
          t := Str2Int (v,1,YMax);
          If t <> -1 Then
          Begin
            range := t; YDist := YMax Div range
          End
        End
        Else If s = 'input' Then
        Begin
          If v = 'random' Then ichoice := 1
          Else If v = 'reverse' Then ichoice := 2
          Else If v = 'sorted' Then ichoice := 3
          Else If v = 'almost' Then ichoice := 4
          Else If v = 'file' Then ichoice := 5
        End
        Else If s = 'percentage' Then
        Begin
          t := Str2Int (v,1,30); If t <> -1 Then perc := t
        End
        Else If s = 'inputfile' Then IFName := v
        Else If s = 'speed' Then
        Begin
          If v = 'fast' Then speed := fast
          Else If v = 'slow' Then speed := slow
          Else If v = 'manual' Then speed := manual
        End
        Else If s = 'delaytime' Then
        Begin
          t := Str2Int (v,0,MaxDelay); If t <> -1 Then DelayTime := t
        End
        Else If s = 'waittime' Then
        Begin
          t := Str2Int (v,0,10000); If t <> -1 Then WaitTime := t
        End
      End
    End
  End
End;   (* ReadConfigFile *)

Procedure Prepare;
(* Initializes the graphics card and initializes the variables. *)
Var GrMode, GrDriver, ErrorCode : Integer;
Begin                                                            (* Prepare *)
  GrDriver := Detect;
  InitGraph (GrDriver,GrMode,'');
  ErrorCode := GraphResult;
  If ErrorCode <> 0 Then
  Begin
    ErrorMess (GraphErrorMsg (ErrorCode));
    Halt (1);
  End;
  DotColour := GetMaxColor;
  BackColour := Black;
  XMax := GetMaxX + 1;
  YMax := GetMaxY + 1;
  If XMax > NMax Then XMax := NMax;
  n := XMax;
  range := YMax;
  XDist := 1; YDist := 1;
  achoice := 1; ichoice := 1;
  speed := fast; perc := 10;
  quit := False; StoreData := False;
  DelayTime := 10; WaitTime := 1000;
  IFName := 'SortDemo.Inp';
  LFName := 'SortDemo.Lst';
  ReadConfigFile;
  OldTextAttr := TextAttr;
  TextAttr := LowColour + 16 * MenuBackColour;
  SetColor (DotColour);
  SetBkColor (BackColour);
  SetViewPort (0,0,XMax,YMax,Clipon);
  ClearDevice
End;   (* Prepare *)

Function PrepareLF : Boolean;
(* Prepares the ListFile for storing the performance data, returns true if  *)
(* successful. The name of the ListFile is LFName.                          *)
Var i, IORes : Integer;
Begin                                                          (* PrepareLF *)
  LFName := GetFileName ('Name of the list file: ', LFName, 'Lst');
  {$I-}
  Assign (ListFile, LFName);
  Rewrite (ListFile);
  {$I+}
  IORes := IOResult;
  ClearMessLine;
  If IORes <> 0 Then DisplayMess ('I can''t open the list file!')
  Else
  Begin
    WriteLn (ListFile);
    Write (ListFile, 'Algorithm':12, ' ':7, 'Input length');
    Write (ListFile, ' ':7, 'Input', ' ':10);
    WriteLn (ListFile, 'Comparisons', ' ':5, 'Swaps');
    For i := 1 To 78 Do Write (ListFile, '-'); WriteLn (ListFile);
  End;
  PrepareLF := IORes = 0
End;   (* PrepareLF *)

Procedure Options;
Const OptMenu : MenuType     = ('SortDemo Options',
                                '1. Number of keys',
                                '2. Range of values',
                                '3. Keys in wrong place (in %)',
                                '4. Fast mode (real time)',
                                '5. Slow motion',
                                '6. Manual mode',
                                '7. Delay time',
                                '8. Waittime before starting',
                                '9. Performance Data',
                                '0. Quit options menu');
      OptSelect : HotkeyType = ('NRKFSMDWPQ');
Var ochoice : Integer;
    quit : Boolean;
Begin                                                            (* Options *)
  quit := False;
  ochoice := 1;
  Repeat
    ochoice := StdMenu (State, OptMenu, OptSelect, ochoice);
    Case ochoice Of
      1 : n     := GetInt ('Number of keys: ', n, 2, XMax);
      2 : range := GetInt ('Range of values: ', range, 1, YMax);
      3 : perc  := GetInt ('Percentage: ', perc, 1, 30);
      4 : speed := fast;
      5 : speed := slow;
      6 : speed := manual;
      7 : Begin
            DelayTime := GetInt ('Delay between operations: ',
                                 DelayTime, 2, MaxDelay);
            speed := slow
          End;
      8 : WaitTime := GetInt ('Waittime before starting: ',
                              WaitTime, 0, 10000);
      9 : Begin
            If StoreData Then
            Begin
              Close (ListFile); StoreData := False;
              DisplayMess ('Storing is now turned off.')
            End
            Else StoreData := PrepareLF;
            If StoreData Then DisplayMess ('Storing data in ' + LFName)
          End;
      10: quit := True;
    End
  Until quit;
  XDist := XMax Div n;
  YDist := YMax Div range
End;   (* Options *)

Function CallFileInit (Var n, range : Integer) : Boolean;
(* Initializes the array A with numbers read from the text file IFName      *)
(* which must contain at least 2 nonnegative numbers. Only the first XMax   *)
(* numbers will be read, even if the file is longer. If the maximal value   *)
(* is greater than YMax, the whole input is scaled down to fit to the       *)
(* screen. A [0] is initialized with -1 as sentinel.                        *)
Var FIResult : Integer;
Begin                                                       (* CallFileInit *)
  IFName := GetFileName ('Name of the input file: ', IFName, 'Inp');
  FIResult := FileInit (IFName, n, range, False);
  ClearMessLine;
  Case FIResult Of
    10 : DisplayMess ('I can''t open that input file!');
    11 : DisplayMess ('The input file must not contain negative numbers.');
    12 : DisplayMess ('The input file must contain at least two numbers.');
  End;
  CallFileInit := FIResult = 0
End;   (* CallFileInit *)

Procedure PlotArray;
(* The dots are plotted on the screen. Needed only for FileInput. *)
Var i : Integer;
Begin                                                          (* PlotArray *)
  For i := 1 To n Do DrawDot (i, A[i])
End;   (* PlotArray *)


(* ----------------------------------------- *)

Begin                                                           (* SortDemo *)
  Randomize;
  Prepare;
  RestoreCrtMode;
  ClrScr;
  ShowHeadline (Logo, AlgMenu [0]);
  WriteLn; WriteLn;
  WriteLn ('This is free software, and you are welcome to redistribute it ',
           'under');
  WriteLn ('certain conditions; however, it comes with ABSOLUTELY NO ',
           'WARRANTY.');
  WriteLn ('See the file COPYLEFT for full details.');
  WriteLn;
  WriteLn ('For a documentation see the file SortDemo.Doc.');
  WriteLn;
  WriteLn ('The execution of any of the algorithms can be stopped after ',
           'each');
  WriteLn ('round of sorting by pressing <ESC> or paused by ',
           'any other key.');
  WriteLn;
  WriteLn ('Press <Return> to start the programme.');
  ReadLn;
  Repeat
    achoice := StdMenu (State, AlgMenu, AlgSelect, achoice);
    If (achoice > 0) And (achoice <= NrAlgs) Then
    Begin
      ichoice := PopMenu (InputChoice, '', ichoice, 50, achoice*2+2);
      If (ichoice = 5) And (Not CallFileInit (n, range)) Then ichoice := 6;
      If ichoice < 6 Then
      Begin
        SetGraphMode (GetGraphMode);
        SetColor (DotColour);
        SetBkColor (BackColour);
        ClearDevice;
        Case ichoice Of
          1 : RandomInit (n, range);
          2 : ReversInit (n, range);
          3 : SortedInit (n, range);
          4 : AlmostInit (n, range, perc);
          5 : PlotArray;
        End;
        Delay (WaitTime);
        Case achoice Of
          1 : BubbleSort (n);
          2 : ShakerSort (n);
          3 : SelectSort (n);
          4 : InsertSort (n);
          5 : ShellSort  (n);
          6 : QuickSort  (n);
          7 : HeapSort   (n);
          8 : MergeSort  (n);
        End;
        OutTextXY (30, 10, 'Comparisons: ' + li2s (comps));
        OutTextXY (30, 20, 'Swaps      : ' + li2s (swaps));
        OutTextXY (30, 30, 'Press <Return> to continue.');
        If StoreData Then
        Begin
          s := AlgMenu [achoice];
          Delete (s, 1, 3);
          Write (ListFile, '  ', s, ' ':(15-Length(s)));
          Write (ListFile, n:9, InputChoice [ichoice]:20);
          If ichoice = 4 Then Write (ListFile, ' (', perc:2, '%)')
          Else Write (ListFile, ' ':6);
          WriteLn (ListFile, comps:11, ' ', swaps:10);
          For i := 1 To 78 Do Write (ListFile, '-'); WriteLn (ListFile)
        End;
        While Not (ReadKey In [#3, #13, #27]) Do;
      End
      Else ichoice := 1
    End
    Else If achoice = NrAlgs + 1 Then Options
    Else If achoice = NrAlgs + 2 Then quit := True;
    RestoreCrtMode;
  Until quit;
  TextMode (LastMode);
  If StoreData Then
  Begin
    Close (ListFile);
    WriteLn ('Performance data have been stored in ', LFName)
  End;
  CloseGraph;
  TextAttr := OldTextAttr
End.   (* SortDemo *)
