/*
 * GNU m4 -- A simple macro processor
 * Copyright (C) 1989, 1990, 1991 Free Software Foundation, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "m4.h"
#include "version.h"

#include <signal.h>

void cmd_error();


/* Operate interactively (-e) */
int interactive = 0;

/* Enable sync output for /lib/cpp (-s) */
int sync_output = 0;

/* Debug (-d[flags]) */
int debug_level = 0;

/* Hash table size (should be a prime) (-Hsize) */
int hash_table_size = HASHMAX;

/* Number of diversions allocated */
int ndivertion = NDIVERTIONS;

/* Disable GNU extensions (-G) */
int no_gnu_extensions = 0;

/* Max length of arguments in trace output (-lsize) */
int max_debug_argument_length = 0;

/* Program name */
char *progname = nil;


/*
 * usage --- Print usage message on stderr.
 */
char *name;

void
usage()
{
    printf("\n\
GNU m4 %s, Copyright (C) 1989, 1990 Free Software Foundation, Inc.\n\
There is ABSOLUTELY NO WARRANTY for GNU m4.  See the file\n\
COPYING in the source distribution for more details.\n", version);

    printf("\nUsage: %s [options] file ....\n", name);

    printf("\n  -e           make m4 interactive (output unbuffered)"
           "\n  -s           generate sync line for C preprocessor"
           "\n  -Hn          make symbol lookup hash table n entries big (n should be prime)"
           "\n  -dn          set debug level to n"
           "\n  -G           suppress GNU extensions"
           "\n  -B, -S, -T   present for System V compatibility, no-ops"
           "\n  -Idir        specify include file search path"
           "\n  -Dname=val   define macro"
           "\n  -Uname       delete predefined macro\n");

    exit(1);
}



#define NEXTARG --argc, ++argv

int
main(argc, argv)
    int argc;

    char **argv;
{
    boolean no_more_options = false;
    char *macro_value;
    FILE *fp;

    progname = rindex(argv[0], '/');
    if (progname == nil)
	progname = argv[0];
    else
	progname++;

    include_init();
    debug_init();

    name = argv[0];

    /*
     * First, we decode the basic arguments, to size up tables and stuff.
     */
    for (NEXTARG; argc && argv[0][0] == '-'; NEXTARG) {
	switch (argv[0][1]) {
	case 'e':
	    interactive = 1;
	    break;

	case 'V':
	    fprintf(stderr,
		    "\
GNU m4 %s, Copyright (C) 1989, 1990, 1991 Free Software Foundation, Inc.\n\
There is ABSOLUTELY NO WARRANTY for GNU m4.  See the file\n\
COPYING in the source distribution for more details.\n",
		    version);
	    break;

	case 's':
	    sync_output = 1;
	    break;

	case 'd':
	    debug_level = debug_decode(&argv[0][2]);
	    if (debug_level < 0) {
		cmd_error("bad debug flags: `%s'", &argv[0][2]);
		debug_level = 0;
	    }
	    break;

	case 'l':
	    max_debug_argument_length = atoi(&argv[0][2]);
	    if (max_debug_argument_length <= 0)
		max_debug_argument_length = 0;
	    break;

	case 'o':
	    if (!debug_set_output(&argv[0][2]))
		cmd_error("cannot set error file %s: %s", &argv[0][2], syserr());
	    break;

	case 'H':
	    hash_table_size = atoi(&argv[0][2]);
	    if (hash_table_size <= 0)
		hash_table_size = HASHMAX;
	    break;

	case 'N':
	    ndivertion = atoi(&argv[0][2]);
	    if (ndivertion <= 0)
		ndivertion = NDIVERTIONS;
	    break;

	case 'I':
	    add_include_directory(&argv[0][2]);
	    break;

	case 'G':
	    no_gnu_extensions = 1;
	    break;

	case 'B':			/* compatibility junk */
	case 'S':
	case 'T':
	    break;

	default:
	    usage();

	    /* These are handled later */
	case '\0':			/* `-' meaning standard input */
	case 'D':			/* define macro */
	case 'U':			/* undefine macro */
	case 't':			/* trace name */
	    no_more_options = true;
	    break;
	}
	if (no_more_options)
	    break;
    }

    input_init();
    output_init();
    symtab_init();
    builtin_init();
    include_env_init();

    /*
     * Define command line macro definitions.  Must come after
     * initialisation of the symbol table.
     */
    no_more_options = false;
    for (; argc && argv[0][0] == '-'; NEXTARG) {
	switch (argv[0][1]) {
	case '\0':
	    no_more_options = true;
	    break;

	case 'D':
	    macro_value = index(&argv[0][2], '=');
	    if (macro_value == nil)
		macro_value = "";
	    else
		*macro_value++ = '\0';
	    define_user_macro(&argv[0][2], macro_value, SYMBOL_INSERT);
	    break;

	case 'U':
	    lookup_symbol(&argv[0][2], SYMBOL_DELETE);
	    break;

	case 't':
	    ;{
		symbol *sym;

		sym = lookup_symbol(&argv[0][2], SYMBOL_INSERT);
		SYMBOL_TRACED(sym) = true;
	    }
	    break;

	default:
	    usage();
	    break;
	}
	if (no_more_options)
	    break;
    }

    /*
     * Interactive mode means unbuffered output, and interrupts ignored.
     */
    if (interactive) {
	signal(SIGINT, SIG_IGN);
	setbuf(stdout, (char *)NULL);
    }

    /*
     * Handle the various input files.  Each file is pushed on the
     * input, and the input read.  Wrapup text is handled separately
     * later.
     */
    if (argc == 0) {
        if ( isatty(fileno(stdin)) )
          usage();
	push_file(stdin, "stdin");
	expand_input();
    } else {
	for ( ; argc > 0; NEXTARG) {
	    if (strcmp(argv[0], "-") == 0) {
		push_file(stdin, "stdin");
	    } else {
		fp = path_search(argv[0]);
		if (fp == nil) {
		    cmd_error("can't open %s: %s", argv[0], syserr());
		    continue;
		} else
		    push_file(fp, argv[0]);
	    }
	    expand_input();
	}
    }
#undef NEXTARG

    /* Now handle wrapup text. */
    while (pop_wrapup())
	expand_input();

    undivert_all();

    return 0;
}


/*
 * The rest of this file contains error handling functions, and memory
 * allocation.
 */

/* Non m4 specific error -- just complain */
/* VARARGS */
void
cmd_error(va_alist)
    va_dcl
{
    va_list args;
    char *fmt;

    fprintf(stderr, "%s: ", progname);

    va_start(args);
    fmt = va_arg(args, char*);
    vfprintf(stderr, fmt, args);
    va_end(args);

    putc('\n', stderr);
}

/* Basic varargs function for all error output */
void
vmesg(level, args)
    char *level;
    va_list args;
{
    char *fmt;

    fflush(stdout);
    fmt = va_arg(args, char*);
    fprintf(stderr, "%s: %s: %d: ", progname, current_file, current_line);
    if (level != nil)
	fprintf(stderr, "%s: ", level);
    vfprintf(stderr, fmt, args);
    putc('\n', stderr);
}

/* Internal errors -- print and dump core */
/* VARARGS */
void
internal_error(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg("internal error", args);
    va_end(args);

    abort();
}

/* Fatal error -- print and exit */
/* VARARGS */
void
fatal(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg("fatal error", args);
    va_end(args);

    exit(1);
}

/* "Normal" error -- just complain */
/* VARARGS */
void
error(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg((char *)nil, args);
    va_end(args);
}

/* Warning --- for potential trouble */
/* VARARGS */
void
warning(va_alist)
    va_dcl
{
    va_list args;
    va_start(args);
    vmesg("warning", args);
    va_end(args);
}


/*
 * Memory allocation functions
 */

/* Out ofmemory error -- die */
void
no_memory()
{
    fatal("Out of memory");
}

/* Free previously allocated memory */
void
xfree(p)
    char *p;
{
    if (p != nil)
	free(p);
}

/* Semi-safe malloc -- never returns nil */
char *
xmalloc(size)
    unsigned int size;
{
    register char *cp = malloc(size);
    if (cp == nil)
	no_memory();
    return cp;
}

#if 0
/* Ditto realloc */
char *
xrealloc(p, size)
    char *p;
    unsigned int size;
{
    register char *cp = realloc(p, size);
    if (cp == nil)
	no_memory();
    return cp;
}
#endif

/* and strdup */
char *
xstrdup(s)
    char *s;
{
    return strcpy(xmalloc((unsigned int)strlen(s)+1), s);
}
