/* find -- search for files in a directory hierarchy
   Copyright (C) 1987, 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* GNU find was written by Eric Decker (cire@cisco.com),
   with enhancements by David MacKenzie (djm@ai.mit.edu).  */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#if defined(USG) || defined(STDC_HEADERS)
#include <string.h>
#define index strchr
#define rindex strrchr
#else
#include <strings.h>
#endif
#include "defs.h"
#include "modetype.h"

#ifndef S_IFLNK
#define lstat stat
#endif

int lstat ();
int stat ();

#define apply_predicate(pathname, stat_buf_ptr, node)	\
  (*(node)->pred_func)((pathname), (stat_buf_ptr), (node))

boolean parse_open ();
boolean parse_close ();
char *savedir ();
void error ();

/* Name this program was run with. */
char *program_name;

/* All predicates for each path to process. */
struct predicate *predicates;

/* The last predicate allocated. */
struct predicate *last_pred;

/* The root of the evaluation tree. */
struct predicate *eval_tree;

/* If true, process directory before contents.  True unless -depth given. */
boolean do_dir_first;

/* If >0, don't descend more than this many levels of subdirectories. */
int maxdepth;

/* If >0, don't process files above this level. */
int mindepth;

/* Current depth; 0 means current path is a command line arg. */
int curdepth;

/* Seconds between 00:00 1/1/70 and either one day before now
   (the default), or the start of today (if -daystart is given). */
time_t cur_day_start;

/* If true, cur_day_start has been adjusted to the start of the day. */
boolean full_days;

/* If true, don't cross filesystem boundaries. */
boolean stay_on_filesystem;

/* If true, don't descend past current directory.
   Can be set by -prune, -maxdepth, and -xdev. */
boolean stop_at_current_level;

/* Status value to return to system. */
int exit_status;

/* Length of current path. */
int path_length;

/* Pointer to the function used to stat files. */
int (*xstat) ();

void
main (argc, argv)
     int argc;
     char *argv[];
{
  int i;
  PFB parse_function;		/* Pointer to who is to do the parsing. */
  struct predicate *cur_pred;
  char *predicate_name;		/* Name of predicate being parsed. */

  program_name = argv[0];

  predicates = NULL;
  last_pred = NULL;
  do_dir_first = true;
  maxdepth = mindepth = -1;
  cur_day_start = time ((time_t *) 0) - DAYSECS;
  full_days = false;
  stay_on_filesystem = false;
  exit_status = 0;
  xstat = lstat;

#ifdef	DEBUG
  printf ("%ld %s", cur_day_start, ctime (&cur_day_start));
#endif /* DEBUG */

  if ( argc == 1 )
    usage(NULL);

  /* Find where in ARGV the predicates begin. */
  for (i = 1; i < argc && index ("-!(),", argv[i][0]) == 0; i++)
    ;

  /* Enclose the expression in `( ... )' so a default -print will
     apply to the whole expression. */
  parse_open (argv, &argc);
  /* Build the input order list. */
  while (i < argc)
    {
      if (index ("-!(),", argv[i][0]) == 0)
	usage ("paths must precede expression");
      predicate_name = argv[i];
      parse_function = find_parser (predicate_name);
      if (parse_function == NULL)
	error (1, 0, "invalid predicate `%s'", predicate_name);
      i++;
      if (!(*parse_function) (argv, &i))
	{
	  if (argv[i] == NULL)
	    error (1, 0, "missing argument to `%s'", predicate_name);
	  else
	    error (1, 0, "invalid argument to `%s'", predicate_name);
	}
    }
  if (predicates->pred_next == NULL)
    {
      /* No predicates that do something other than set a global variable
	 were given; remove the unneeded initial `(' and add `-print'. */
      cur_pred = predicates;
      predicates = last_pred = predicates->pred_next;
      free ((char *) cur_pred);
      parse_print (argv, &argc);
    }
  else if (!no_side_effects (predicates->pred_next))
    {
      /* One or more predicates that produce output were given;
	 remove the unneeded initial `('. */
      cur_pred = predicates;
      predicates = predicates->pred_next;
      free ((char *) cur_pred);
    }
  else
    {
      /* `( user-supplied-expression ) -print'. */
      parse_close (argv, &argc);
      parse_print (argv, &argc);
    }

#ifdef	DEBUG
  print_list (predicates);
#endif /* DEBUG */

  /* Done parsing the predicates.  Build the evaluation tree. */
  cur_pred = predicates;
  eval_tree = get_expr (&cur_pred, NO_PREC);
#ifdef	DEBUG
  print_tree (eval_tree, 0);
#endif /* DEBUG */

  for (i = 1; i < argc && index ("-!(),", argv[i][0]) == 0; i++)
    {
      path_length = strlen (argv[i]);
      process_path (argv[i], 1);
    }
  if (i == 1)
    {
      path_length = 1;
      process_path (".", 1);
    }

  exit (exit_status);
}

/* Recursively descend path PATHNAME, applying the predicates.
   DEPTH is the number of levels of directories that it is down
   from a comand-line argument, with 1 meaning it is a command line
   argument. */

void
process_path (pathname, depth)
     char *pathname;
     int depth;
{
  struct stat stat_buf;
  char *name_space;
  static dev_t root_dev;

  if ((*xstat) (pathname, &stat_buf) != 0)
    {
      fflush (stdout);
      error (0, errno, "%s", pathname);
      exit_status = 1;
      return;
    }

  curdepth = depth - 1;

  if (!S_ISDIR (stat_buf.st_mode))
    {
      if (depth > mindepth)
	apply_predicate (pathname, &stat_buf, eval_tree);
      return;
    }

  stop_at_current_level = maxdepth >= 0 && depth > maxdepth;

  if (stay_on_filesystem)
    {
      if (depth == 1)
	root_dev = stat_buf.st_dev;
      else if (stat_buf.st_dev != root_dev)
	stop_at_current_level = true;
    }

  if (do_dir_first && depth > mindepth)
    apply_predicate (pathname, &stat_buf, eval_tree);

  if (stop_at_current_level == false)
    {
      errno = 0;
#ifdef OS2
      name_space = savedir (pathname, 512L);
#else
      name_space = savedir (pathname, stat_buf.st_size);
#endif
      if (name_space == NULL)
	{
	  if (errno)
	    {
	      fflush (stdout);
	      error (0, errno, "%s", pathname);
	      exit_status = 1;
	    }
	  else
	    {
	      fflush (stdout);
	      error (1, 0, "virtual memory exhausted");
	    }
	}
      else
	{
	  char *namep;		/* Current point in `name_space'. */
	  char *cur_path;	/* Full path of each file to process. */
	  unsigned cur_path_size; /* Bytes allocated for `cur_path'. */
	  unsigned file_len;	/* Length of each path to process. */
	  unsigned pathname_len; /* Length of `pathname' + 2. */

#ifdef OS2
          if (!strcmp (pathname+1, ":/") || !strcmp (pathname+1, ":\\"))
	    pathname_len = 4;
          else if (!strcmp (pathname, "/") || !strcmp (pathname, "\\"))
#else
	  if (!strcmp (pathname, "/"))
#endif
	    pathname_len = 2;	/* Won't add a slash to this. */
	  else
	    pathname_len = strlen (pathname) + 2; /* For '/' and '\0'. */
	  cur_path_size = 0;
	  cur_path = NULL;

	  for (namep = name_space; *namep;
	       namep += file_len - pathname_len + 1)
	    {
	      file_len = pathname_len + strlen (namep);
	      if (file_len > cur_path_size)
		{
		  while (file_len > cur_path_size)
		    cur_path_size += 1024;
		  if (cur_path)
		    free (cur_path);
		  cur_path = xmalloc (cur_path_size);
		  strcpy (cur_path, pathname);
		  cur_path[pathname_len - 2] = '/';
		}
	      strcpy (cur_path + pathname_len - 1, namep);
	      process_path (cur_path, depth + 1);
	      curdepth = depth - 1;
	    }
	  if (cur_path)
	    free (cur_path);
	  free (name_space);
	}
    }

  if (do_dir_first == false && depth > mindepth)
    apply_predicate (pathname, &stat_buf, eval_tree);
}

/* Return true if there are no side effects in any of the predicates in
   predicate list PRED, false if there are any. */

boolean
no_side_effects (pred)
     struct predicate *pred;
{
  while (pred != NULL)
    {
      if (pred->side_effects)
	return (false);
      pred = pred->pred_next;
    }
  return (true);
}
