// PACK8.C
// This program reads 8 bit PCM sound data from a file.
// The 8 bit sound data is converted to 3 bit with silence 
// encoding using the packFirst and packNext routines.
// The 3 bit data is then written to a new file.
//
// The 2 buffers that are used in this program are allocated 
// using cvxBufferAlloc.  The size of both buffers is set to 32K. 

#include <stdio.h>
#include <bios.h>
#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include "cvxdigi.h"
#include "cvxutil.h"

#define  _BUFFER_SIZE    0x8000
#define  _PCM_FILE       "TEST1.V8"
#define  _3_BIT_FILE     "TEST1.V3S"

VOID main( VOID )
{
   LONG            recordedLength;
   HANDLE          pcmFileHandle;
   HANDLE          packedFileHandle;
   LONG            bytesWritten;
   LONG            bytesAvailable;
   LONG            bytesRead;
   _CONVERT_DATA   cnvrtData;
   LPSTR           pcmBuffer;
   LPSTR           conversionBuffer;


   // Allocate memory for buffers used to read 8 bit PCM
   // data from file.
   pcmBuffer = cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );

   if( pcmBuffer == _NULL )
   {
      printf( "ERROR : Cannot allocate memory for PCM buffer!\n" );
      exit( 0 );
   }

   // Allocate memory for buffer used for packing 8 bit PCM to
   // 3 bit with silence encoding.
   //
   conversionBuffer = cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );

   if( conversionBuffer == _NULL )
   {
      printf( "ERROR : Cannot allocate memory for pack buffer!\n" );
      exit( 0 );
   }

   // Open a file containg 8 bit PCM data.
   if( cvxFileOpen( _PCM_FILE, _OPEN_R_ONLY, &pcmFileHandle ) )
      printf( "ERROR : %s not found.\n", _PCM_FILE );
   else
      // Read header information from file. 
      cvxFileRead( pcmFileHandle, pcmBuffer,
                   ( LONG )_HEADER_LENGTH, &bytesRead );

   // Point _CONVERT_DATA element to same location that will contain
   // 8 bit PCM data read from _PCM_FILE
   cnvrtData.sourcePointer = ( LPSTR )pcmBuffer;

   // Point _CONVERT_DATA structure element to same location as
   // conversionBuffer. This buffer will contain the data
   // that has been packed.
   cnvrtData.destinationPointer = ( LPVOID )conversionBuffer;

   // Open a new file to write 3 bit sound data.
   if( cvxFileCreate( _3_BIT_FILE, _CREATE_NORMAL, &packedFileHandle ) )
      printf( "ERROR : Cannot create file %s.\n", _3_BIT_FILE );
   else
   {
      // Put header information into structure for call
      // to packFirst.
      cnvrtData.sampleRate        = ( BYTE )*( pcmBuffer + _HEADER_RATE_OFFSET );
      cnvrtData.bufferFormat      = _FORMAT_V3S;
      cnvrtData.silenceRange      = _TRUE;
      cnvrtData.sourceLength      = (LONG)_HEADER_LENGTH;
      cnvrtData.destinationLength = (LONG)_HEADER_LENGTH;

      // Features not needed during pack.
      cnvrtData.trebleFlag        = 0; 
      cnvrtData.noiseFlag         = 0; 


      // Call the function to set up the sound file header.
      if( packFirst( ( _CONVERT_DATA far * )&cnvrtData ) )
         printf( "ERROR : Function packFirst failed.\n" );
      else
      {
         // Write the 16 bit header returned by packFirst()
         // to file.
         cvxFileWrite( packedFileHandle, cnvrtData.destinationPointer, 
                       ( LONG )_HEADER_LENGTH, &bytesWritten );

         // Increment pcmBuffer past the 16 byte header
         pcmBuffer = pcmBuffer + (WORD)_HEADER_LENGTH;

         // Setup loop to process contents of pcmBuffer
         while( !eof( pcmFileHandle ) )
         {
            // Read 8 bit PCM data from file. 
            cvxFileRead( pcmFileHandle, pcmBuffer,
                         ( LONG )_BUFFER_SIZE, &bytesRead );

            // Put information into _CONVERT_DATA structure for 
            // packNext.
            cnvrtData.sourceLength      = bytesRead;
            cnvrtData.destinationLength = _BUFFER_SIZE;

            // Call the function to set up the next buffer.
            if( packNext( ( _CONVERT_DATA far * ) &cnvrtData ) )
               printf( "ERROR : Data packing failure.\n" );
            else
               cvxFileWrite( packedFileHandle,
                             cnvrtData.destinationPointer,
                             cnvrtData.destinationFilled,
                             &bytesWritten );
         }
      }

      // Close files.
      cvxFileClose( pcmFileHandle );
      cvxFileClose( packedFileHandle );
   }

   // Free memory used for 3 bit sound data.
   if( cvxBufferFree( conversionBuffer ) )
      printf( "ERROR : Cannot free memory used for pack buffer!\n" );

   // Free memory used by 8 bit PCM buffer.
   if( cvxBufferFree( pcmBuffer ) )
      printf( "ERROR : Cannot free memory used for PCM buffer!\n" );
}



