/*
  lxmta.c -- Message Tranfer Agent for the HP200LX palmtop.

  Copyright (C) 1997-1999  Rod Whitby

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be 
  useful, but WITHOUT ANY WARRANTY; without even the implied
  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  USA.

  $Source: A:/SRC/TCP/MTA/RCS/LXMTA.C $
  $Id: LXMTA.C 1.13 2000/04/09 07:02:50 rwhitby Exp $
*/

#include <alloc.h>
#include <conio.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <errno.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "socklib.h"

int do_smtp(void);
int do_pop(void);
int do_imap(void);
int do_nntp(void);

/* This must match the value in pnrti.h */
#define BREAKLEN 4

static char *smtpserver = 0;
static int   smtpport = SMTP_PORT;
static char *smtpproxy = 0;

static char *popserver = 0;
static int   popport = POP_PORT;
static char *popproxy = 0;
static char *popusername = 0;
static char *poppassword = 0;

static char *imapserver = 0;
static int   imapport = IMAP_PORT;
static char *imapproxy = 0;
static char *imapusername = 0;
static char *imappassword = 0;
static char *imapselect = 0;

static char *nntpserver = 0;
static int   nntpport = NNTP_PORT;
static char *nntpproxy = 0;
static char *nntpusername = 0;
static char *nntppassword = 0;
static char *newsrc = "incoming.jn";

static char *spooldir = 0;
static char *domain = 0;

static int catchup = 0;
static int delete_smtp = 1;
static int delete_pop  = 1;
static int delete_imap = 1;
static int delete_nntp = 1;
static int save_sent = 0;

static long min_free  = -1;
static long max_lines = -1;
static long max_bytes = -1;
static long max_msgs  = -1;

static int scroll_info = 0;
static int show_header = 0;
static char eol_status = '\r';
static char eol_header = 0;

static char temp[MAXPATH];
static char save[MAXPATH];

void usage(char * name)
{
  fprintf(stderr, "Usage: %s [<directive>=<value>] ...\n", name);
  exit(1);
}

void (*old_init)(char *directive, char *value);
void new_init   (char *directive, char *value)
{
  dbug_printf("Setting %s to %s.", directive, value);

  if (!strnicmp(directive, "lxmta.", 6)) {
    strcpy(directive, directive+6);
  }

  if (!stricmp(directive, "server")) {
    if (*value) smtpserver = strdup(value); else smtpserver = 0;
    if (*value) popserver  = strdup(value); else popserver  = 0;
    if (*value) imapserver = strdup(value); else imapserver = 0;
    if (*value) nntpserver = strdup(value); else nntpserver = 0;
  }
  else if (!stricmp(directive, "smtp.server")) {
    if (*value) smtpserver = strdup(value); else smtpserver = 0;
  }
  else if (!stricmp(directive, "pop.server")) {
    if (*value) popserver  = strdup(value); else popserver  = 0;
  }
  else if (!stricmp(directive, "imap.server")) {
    if (*value) imapserver = strdup(value); else imapserver = 0;
  }
  else if (!stricmp(directive, "nntp.server")) {
    if (*value) nntpserver = strdup(value); else nntpserver = 0;
  }

  if (!stricmp(directive, "port")) {
    if (*value) smtpport = atoi(value);
    if (*value) popport  = atoi(value);
    if (*value) imapport = atoi(value);
    if (*value) nntpport = atoi(value);
  }
  else if (!stricmp(directive, "smtp.port")) {
    if (*value) smtpport = atoi(value);
  }
  else if (!stricmp(directive, "pop.port")) {
    if (*value) popport  = atoi(value);
  }
  else if (!stricmp(directive, "imap.port")) {
    if (*value) imapport = atoi(value);
  }
  else if (!stricmp(directive, "nntp.port")) {
    if (*value) nntpport = atoi(value);
  }

  if (!stricmp(directive, "proxy")) {
    if (*value) smtpproxy = strdup(value); else smtpproxy = 0;
    if (*value) popproxy  = strdup(value); else popproxy  = 0;
    if (*value) imapproxy = strdup(value); else imapproxy = 0;
    if (*value) nntpproxy = strdup(value); else nntpproxy = 0;
  }
  else if (!stricmp(directive, "smtp.proxy")) {
    if (*value) smtpproxy = strdup(value); else smtpproxy = 0;
  }
  else if (!stricmp(directive, "pop.proxy")) {
    if (*value) popproxy  = strdup(value); else popproxy  = 0;
  }
  else if (!stricmp(directive, "imap.proxy")) {
    if (*value) imapproxy = strdup(value); else imapproxy = 0;
  }
  else if (!stricmp(directive, "nntp.proxy")) {
    if (*value) nntpproxy = strdup(value); else nntpproxy = 0;
  }

  if (!stricmp(directive, "username")) {
    if (*value) popusername  = strdup(value); else popusername  = 0;
    if (*value) imapusername = strdup(value); else imapusername = 0;
    if (*value) nntpusername = strdup(value); else nntpusername = 0;
  }
  else if (!stricmp(directive, "pop.username")) {
    if (*value) popusername  = strdup(value); else popusername  = 0;
  }
  else if (!stricmp(directive, "imap.username")) {
    if (*value) imapusername = strdup(value); else imapusername = 0;
  }
  else if (!stricmp(directive, "nntp.username")) {
    if (*value) nntpusername = strdup(value); else nntpusername = 0;
  }

  if (!stricmp(directive, "password")) {
    if (*value) poppassword  = strdup(value); else poppassword  = 0;
    if (*value) imappassword = strdup(value); else imappassword = 0;
    if (*value) nntppassword = strdup(value); else nntppassword = 0;
  }
  else if (!stricmp(directive, "pop.password")) {
    if (*value) poppassword  = strdup(value); else poppassword  = 0;
  }
  else if (!stricmp(directive, "imap.password")) {
    if (*value) imappassword = strdup(value); else imappassword = 0;
  }
  else if (!stricmp(directive, "nntp.password")) {
    if (*value) nntppassword = strdup(value); else nntppassword = 0;
  }

  if (!stricmp(directive, "delete")) {
    delete_smtp = atoi(value);
    delete_pop  = atoi(value);
    delete_imap = atoi(value);
    delete_nntp = atoi(value);
  }
  else if (!stricmp(directive, "smtp.delete")) {
    delete_smtp = atoi(value);
  }
  else if (!stricmp(directive, "pop.delete")) {
    delete_pop  = atoi(value);
  }
  else if (!stricmp(directive, "imap.delete")) {
    delete_imap = atoi(value);
  }
  else if (!stricmp(directive, "nntp.delete")) {
    delete_nntp = atoi(value);
  }

  if (!stricmp(directive, "imap.select")) {
    if (*value) imapselect = strdup(value); else imapselect = 0;
  }

  if (!stricmp(directive, "spooldir")) {
    if (*value) spooldir = strdup(value); else spooldir = 0;
  }
  else if (!stricmp(directive, "newsrc")) {
    if (*value) newsrc = strdup(value); else newsrc = 0;
  }
  else if (!stricmp(directive, "catchup")) {
    catchup = atoi(value);
  }
  else if (!stricmp(directive, "savesent")) {
    save_sent = atoi(value);
  }
  else if (!stricmp(directive, "minfree")) {
    min_free = atol(value);
  }
  else if (!stricmp(directive, "maxlines")) {
    max_lines = atol(value);
  }
  else if (!stricmp(directive, "maxbytes")) {
    max_bytes = atol(value);
  }
  else if (!stricmp(directive, "maxmsgs")) {
    max_msgs = atol(value);
  }
  else if (!stricmp(directive, "showheader")) {
    show_header = atoi(value);
  }
  else if (!stricmp(directive, "scrollinfo")) {
    scroll_info = atoi(value);
  }

  if (!stricmp(directive, "debug")) {
    socklib_debug(atoi(value));
  }

  if (old_init) (*old_init)(directive, value);
}

void main(int argc, char *argv[])
{
  int i;
  char *directive, *value;
  int order = 0;
  int smtp = 0;
  int pop = 0;
  int imap = 0;
  int nntp = 0;

  fprintf(stderr, "LXMTA %d.%d", VERSION/10, VERSION%10);

  if (RELEASE > 0) {
    fprintf(stderr, "b%d", RELEASE);
  }

  fprintf(stderr, " - %s <%s>\n",
          "HP200LX TCP/IP Suite",
          "http://lxtcp.hplx.net/");

  /* Install our configuration handler */
  old_init = usr_init;
  usr_init = new_init;

  if (getenv("LXTCP.CFG")) {
    tcp_config_file(getenv("LXTCP.CFG"));
  }

  /* Initialise the TCP/IP stack */
  socklib_init();

  (void)getdomainname(temp, MAXPATH);
  domain = strdup(temp);

  /* Parse the command line arguments */
  for (i = 1; i < argc; i++) {
    directive = argv[i];
    if ((value = strchr(directive, '=')) != 0) {
      *value = 0; value++; strupr(directive);
      usr_init(directive, value);
    }

    else if (!strcmpi(directive, "smtp")) smtp = ++order;
    else if (!strcmpi(directive, "pop" )) pop  = ++order;
    else if (!strcmpi(directive, "imap")) imap = ++order;
    else if (!strcmpi(directive, "nntp")) nntp = ++order;
     
    else {
      usage(argv[0]);
    }
  }

  if (order == 0) {
    smtp = ++order; pop = ++order; imap = 0; nntp = ++order;
  }

  /* Complain if spooldir is not defined */
  if (!spooldir) {
    fprintf(stderr, "No definition for lxmta.spooldir\n");
    exit(0);
  }

  if (pop && popserver && popusername && !poppassword) {
    sprintf(temp, "Password for POP account %s@%s:", popusername, popserver);
    (void)getnpass(temp, save, MAXPATH);
    poppassword = strdup(save);
  }

  if (imap && imapserver && imapusername && !imappassword) {
    sprintf(temp, "Password for IMAP account %s@%s:", imapusername, imapserver);
    (void)getnpass(temp, save, MAXPATH);
    imappassword = strdup(save);
  }

  if (nntp && nntpserver && nntpusername && !nntppassword) {
    sprintf(temp, "Password for NNTP account %s@%s:", nntpusername, nntpserver);
    (void)getnpass(temp, save, MAXPATH);
    nntppassword = strdup(save);
  }

  if (scroll_info) {
    eol_status = '\n';
  }
  if (show_header) {
    eol_header = eol_status;
  }

  for (i = 1; i <= order; i++) {
    if (i == smtp) do_smtp();
    if (i == pop)  do_pop();
    if (i == imap) do_imap();
    if (i == nntp) do_nntp();
  }

  exit(0);
}

int do_smtp(void)
{
  int ch, done, success;
  FILE * f;
  struct ffblk ffblk;
  tcp_Socket* smtp_sock = 0;

  /* Complain if smtpserver is not defined */
  if (!smtpserver) {
    fprintf(stderr, "No definition for smtp.server\n");
    return 0;
  }

  /* Check if there is anything to send. */
  sprintf(temp, "%s\\outgoing.m??", spooldir);
  if ((done = findfirst(temp, &ffblk, 0)) != 0) {
    return 1;
  }

  /* Make a connection to the SMTP server. */
  if (!(smtp_sock = smtp_init(smtpproxy, smtpserver, smtpport, domain))) {
    fprintf(stderr, "Unable to connect to port %d at %s\n",
            smtpport, smtpserver);
    return 0;
  }
    
  fprintf(stderr, "Connected to SMTP at %s\n", smtpserver);

  while (!done) {

    /* Check for user interrupt */
    if (kbhit()) {
      if ((ch = getch()) == 27) {
        fprintf(stderr, "Interrupted by user\n");
        done = 1;
        break;
      }
      ungetch(ch);
    }

    sprintf(temp, "%s\\%s", spooldir, ffblk.ff_name);
    if ((f = fopen(temp, "r")) != 0) {
      fprintf(stderr, "Sending %s\n", temp);
      success = smtp_sendf(smtp_sock, f, eol_header);
      fclose(f);
      if (success && delete_smtp) {
        if (save_sent) {
          fprintf(stderr, "Saving %s\n", temp);
          strcpy(save, temp);
          save[strlen(temp)-3] = 's';
          if (rename(temp, save)) {
            fprintf(stderr, "rename(%s,%s): %s",
                    temp, save, strerror(errno));
          }
        }
        else {
          fprintf(stderr, "Removing %s\n", temp);
          if (remove(temp)) {
            fprintf(stderr, "remove(%s): %s",
                    temp, strerror(errno));
          }
        }
      }
    }
    done = findnext(&ffblk);
  }

smtp_done:

  /* Close down the connection */
  if (smtp_sock) {
    smtp_shutdown(smtp_sock);
    fprintf(stderr, "Disconnected from SMTP at %s\n", smtpserver);
  }

  return 1;
}

int do_pop()
{
  long count, size, freespace, prev, lines;
  int i, ch, done, success;
  FILE * f;
  struct dfree dfree;
  tcp_Socket* pop_sock = 0;

  /* Complain if popserver is not defined */
  if (!popserver) {
    fprintf(stderr, "No definition for pop.server\n");
    return 0;
  }

  /* Complain if username is not defined */
  if (!popusername) {
    fprintf(stderr, "No definition for pop.username\n");
    return 0;
  }

  done = 0;

  /* Make a connection to the POP server. */
  if (!(pop_sock = pop_init(popproxy, popserver, popport))) {
    fprintf(stderr, "Unable to connect to port %d at %s\n",
            popport, popserver);
    return 0;
  }
    
  fprintf(stderr, "Connected to POP at %s\n", popserver);

  if (pop_login(pop_sock, popusername, poppassword)) {
    fprintf(stderr, "Authenticated %s@%s\n", popusername, popserver);
  }
  else {
    fprintf(stderr, "Unable to authenticate %s@%s\n", popusername, popserver);
    goto pop_close;
  }

  if (pop_status(pop_sock, &count, &size)) {
    fprintf(stderr, "%s@%s has %ld messages (%ld bytes)\n",
            popusername, popserver, count, size);
    if (count) {
      sprintf(temp, "%s\\incoming.mai", spooldir);
      f=fopen(temp, "a");
      for (i=1; !done && (i <= count); i++) {

        /* Reset the line count for this message */
        lines = max_lines;

        /* Check for user interrupt */
        if (kbhit()) {
          if ((ch = getch()) == 27) {
            fprintf(stderr, "\nDownloading stopped - user interrupt");
            done = 1;
            break;
          }
          ungetch(ch);
        }

        getdfree(toupper(*temp)-'A'+1, &dfree);
        freespace = ((long)dfree.df_avail *
                     (long)dfree.df_bsec *
                     (long)dfree.df_sclus);

        if ((min_free >= 0) && (freespace < min_free)) {
          fprintf(stderr, "\nDownloading stopped - out of space");
          done = 1;
          break;
        }

        pop_length(pop_sock, i, &size);

        if ((min_free >= 0) && (freespace - size < min_free)) {
          fprintf(stderr,
                  "\nPreviewing message %d (%ld bytes) - no room\n",
                  i, size);
          lines = 0;
        }

        if ((max_bytes >= 0) && (size > max_bytes)) {
          fprintf(stderr,
                  "\nPreviewing message %d (%ld bytes) - too large\n",
                  i, size);
          lines = 0;
        }

        fprintf(stderr,
                "%cReceiving message %d (%ld bytes) (%d remaining)",
                eol_status, i, size, count - i); clreol();
        fflush(f); prev = ftell(f);

        success = pop_retrieve(pop_sock, f, i, lines, eol_header);

        /* Exit if the socket has been closed. */
        if (!sock_established(pop_sock)) {
          done = 1;
        }        

        if (success) {
          if (delete_pop && (lines == max_lines)) {
              fprintf(stderr, "%cDeleting message %d",
                      eol_status, i); clreol();
              pop_delete(pop_sock, i);
          }
        }
        else {
          /* Discard erroneous message. */
          fprintf(stderr, "%cError during message %d\n",
                  eol_status, i);
          if (ftell(f) > prev) {
            chsize(fileno(f), prev);
            fseek(f, 0, SEEK_END);
          }
        }
      }
      fprintf(stderr, "\n");
      if (fclose(f)) {
        fprintf(stderr, "write(%s): Disk full\n", temp);
      }
    }
  }

pop_close:

  /* Close down the connection */
  if (pop_sock) {
    pop_shutdown(pop_sock);
    fprintf(stderr, "Disconnected from POP at %s\n", popserver);
  }

  return 1;
}

int do_imap()
{
  long count, size, freespace, prev, lines;
  char *matches, *current;
  int i, message, ch, done, success;
  FILE * f;
  struct dfree dfree;
  tcp_Socket* imap_sock = 0;

  /* Complain if imapserver is not defined */
  if (!imapserver) {
    fprintf(stderr, "No definition for imap.server\n");
    return 0;
  }

  /* Complain if username is not defined */
  if (!imapusername) {
    fprintf(stderr, "No definition for imap.username\n");
    return 0;
  }

  done = 0;

  /* Make a connection to the IMAP server. */
  if (!(imap_sock = imap_init(imapproxy, imapserver, imapport))) {
    fprintf(stderr, "Unable to connect to port %d at %s\n",
            imapport, imapserver);
    return 0;
  }
    
  fprintf(stderr, "Connected to IMAP at %s\n", imapserver);

  if (imap_login(imap_sock, imapusername, imappassword)) {
    fprintf(stderr, "Authenticated %s@%s\n", imapusername, imapserver);
  }
  else {
    fprintf(stderr, "Unable to authenticate %s@%s\n", imapusername, imapserver);
    goto imap_close;
  }

  if (imap_select(imap_sock, "INBOX", &count)) {

    fprintf(stderr, "%s@%s has %ld messages\n",
            imapusername, imapserver, count);

    if (count) {

      if (imapselect) {
        imap_search(imap_sock, imapselect, &matches);
        if (matches) {
          fprintf(stderr, "Messages selected: %s\n", matches);
          current = strtok(matches, " ");
        }
        else {
          fprintf(stderr, "No messages selected");
          count = 0;
        }
      }
      else {
        fprintf(stderr, "All messages selected\n");
      }

      sprintf(temp, "%s\\incoming.mai", spooldir);
      f=fopen(temp, "a");

      for (i=1; !done && (i <= count); i++) {

        /* Calculate the target message number */
        if (imapselect) {
          message = atoi(current);
        }
        else {
          message = i;
        }

        /* Reset the line count for this message */
        lines = max_lines;

        /* Check for user interrupt */
        if (kbhit()) {
          if ((ch = getch()) == 27) {
            fprintf(stderr, "\nDownloading stopped - user interrupt");
            done = 1;
            break;
          }
          ungetch(ch);
        }

        getdfree(toupper(*temp)-'A'+1, &dfree);
        freespace = ((long)dfree.df_avail *
                     (long)dfree.df_bsec *
                     (long)dfree.df_sclus);

        if ((min_free >= 0) && (freespace < min_free)) {
          fprintf(stderr, "\nDownloading stopped - out of space");
          done = 1;
          break;
        }

        imap_length(imap_sock, message, &size);

        if ((min_free >= 0) && (freespace - size < min_free)) {
          fprintf(stderr,
                  "\nPreviewing message %d (%ld bytes) - no room\n",
                  message, size);
          lines = 0;
        }

        if ((max_bytes >= 0) && (size > max_bytes)) {
          fprintf(stderr,
                  "\nPreviewing message %d (%ld bytes) - too large\n",
                  message, size);
          lines = 0;
        }

        fprintf(stderr,
                "%cReceiving message %d (%ld bytes) (%d remaining)",
                eol_status, message, size, count - message); clreol();
        fflush(f); prev = ftell(f);

        success = imap_retrieve(imap_sock, f, message, lines, eol_header);

        /* Exit if the socket has been closed. */
        if (!sock_established(imap_sock)) {
          done = 1;
        }        

        if (success) {
          if (delete_imap && (lines == max_lines)) {
              fprintf(stderr, "%cDeleting message %d",
                      eol_status, message); clreol();
              imap_delete(imap_sock, message);
          }
        }
        else {
          /* Discard erroneous message. */
          fprintf(stderr, "%cError during message %d\n",
                  eol_status,  message);
          if (ftell(f) > prev) {
            chsize(fileno(f), prev);
            fseek(f, 0, SEEK_END);
          }
        }

        /* Skip along the matches string */
        if (imapselect) {
          current = strtok(NULL, " ");
          if (!current) done = 1;
        }
      }
      fprintf(stderr, "\n");
      if (fclose(f)) {
        fprintf(stderr, "write(%s): Disk full\n", temp);
      }
      if (imapselect) {
        free(matches);
      }
    }
  }

imap_close:

  /* Close down the connection */
  if (imap_sock) {
    imap_shutdown(imap_sock);
    fprintf(stderr, "Disconnected from IMAP at %s\n", imapserver);
  }

  return 1;
}

int do_nntp()
{
  long dummy, msg_num, count, first, last;
  long update, download, total;
  long grp_offset, art_offset, freespace;
  int ch, done, success;
  char line[MAXPATH];
  char group[MAXPATH];
  char *oldrc = 0;
  char *newrc = 0;
  char *ifile = 0;
  char *index = 0;
  FILE *f, *idx, *irc, *orc;
  struct dfree dfree;
  struct ffblk ffblk;
  tcp_Socket* nntp_sock = 0;

  /* Complain if nntpserver is not defined */
  if (!nntpserver) {
    fprintf(stderr, "No definition for nntp.server\n");
    return 0;
  }

  /* Complain if newsrc is not defined */
  if (!newsrc) {
    fprintf(stderr, "No definition for lxmta.newsrc\n");
    return 0;
  }

  /* Make a connection to the NNTP server. */
  if (!(nntp_sock = nntp_init(nntpproxy, nntpserver, nntpport))) {
    fprintf(stderr, "Unable to connect to port %d at %s\n",
            nntpport, nntpserver);
    return 0;
  }
    
  fprintf(stderr, "Connected to NNTP at %s\n", nntpserver);

  if (nntpusername) {
    if (nntp_auth(nntp_sock, nntpusername, nntppassword)) {
      fprintf(stderr, "Authenticated %s@%s\n", nntpusername, nntpserver);
    }
    else {
      fprintf(stderr, "Unable to authenticate %s@%s\n", nntpusername, nntpserver);
      goto nntp_close;
    }
  }

  /* Look for outgoing news messages */
  sprintf(temp, "%s\\outgoing.n??", spooldir);

  if ((done = findfirst(temp, &ffblk, 0)) == 0) {
    while (!done) {

      /* Check for user interrupt */
      if (kbhit()) {
        if ((ch = getch()) == 27) {
          fprintf(stderr, "Interrupted by user\n");
          done = 1;
          break;
        }
        ungetch(ch);
      }

      sprintf(temp, "%s\\%s", spooldir, ffblk.ff_name);
      if ((f = fopen(temp, "r")) != 0) {
        fprintf(stderr, "Sending %s\n", temp);
        success = nntp_post(nntp_sock, f, eol_header);
        fclose(f);

        /* Exit if the socket has been closed. */
        if (!sock_established(nntp_sock)) {
          done = 1; break;
        }        

        if (success && delete_nntp) {
          if (save_sent) {
            fprintf(stderr, "Saving %s\n", temp);
            strcpy(save, temp);
            save[strlen(temp)-3] = 's';
            if (rename(temp, save)) {
              fprintf(stderr, "rename(%s,%s): %s",
                      temp, save, strerror(errno));
            }
          }
          else {
            fprintf(stderr, "Removing %s\n", temp);
            if (remove(temp)) {
              fprintf(stderr, "remove(%s): %s",
                      temp, strerror(errno));
            }
          }
        }
      }
      done = findnext(&ffblk);
    }
  }

  if (done == 1) {
    goto nntp_close;
  }

  /* Open the newsrc file */
  sprintf(temp, "%s\\%s", spooldir, newsrc);
  oldrc = strdup((const char *)temp);
  irc = fopen(oldrc, "r");

  /* Complain if the newsrc file cannot be read */
  if (!irc) {
    fprintf(stderr, "open(%s, read): %s",
            oldrc, strerror(errno));
  }

  /* Open a temporary file for the new newsrc file */
  sprintf(temp, "%s\\incoming.tmp", spooldir);
  newrc = strdup((const char *)temp);
  orc = fopen(newrc, "w");

  /* Complain if the temp file cannot be written */
  if (!orc) {
     fprintf(stderr, "open(%s, write): %s",
             newrc, strerror(errno));
  }

  /* Open the news file */
  sprintf(temp, "%s\\incoming.nws", spooldir);
  ifile = strdup((const char *)temp);
  f = fopen(ifile, "a");

  /* Complain if the news file cannot be written */
  if (!f) {
    fprintf(stderr, "open(%s, append): %s",
            ifile, strerror(errno));
  }

  /* Open the index file */
  sprintf(temp, "%s\\incoming.idx", spooldir);
  index = strdup((const char *)temp);
  idx = fopen(index, "a");

  /* Complain if the index file cannot be written */
  if (!idx) {
    fprintf(stderr, "open(%s, append): %s",
            index, strerror(errno));
  }

  /* Download any new news articles */
  if (irc && orc && ifile && idx) {

    /* Always reserve enough space to update the newsrc file */
    freespace = filelength(fileno(irc)) * 2 + 1024;
    if (min_free < freespace) min_free = freespace;

    /* Reset the total articles counter */
    total = 0;
    done = 0;

    /* Read each line of the newsrc file */
    while (fgets(line, MAXPATH, irc)) {

      rip(line);

      /* Extract the group and last message read */
      if (sscanf(line, "%[^:]: %ld-%ld",
                 group, &dummy, &msg_num) == 3) {

        if (!done && (*group != '!') &&

            /* Request the group from the server */
            nntp_group(nntp_sock, group,
                       &count, &first, &last)) {

          /* Save the location of this group for the index */
          fseek(f, 0, SEEK_END);
          grp_offset = ftell(f) + BREAKLEN + 2;

          /* Check if there are any new messages */
          if ((count > 0) && (last > msg_num)) {

            /* Catch up when the server is ahead of us */
            if (first > msg_num) msg_num = first - 1;

            fprintf(stderr, "%s has %ld new articles (%ld-%ld)\n",
                    group, last - msg_num, msg_num + 1, last);

            /* Catchup this group if requested */
            if (catchup) {
              fprintf(stderr, "%s is now up to date\n", group);
              msg_num = last;
            }

            /* Reset the flags */
            download = 0; success = 1; update = msg_num;

            /* Loop through the new articles in this group */
            while (msg_num < last) {

              /* Check for user interrupt */
              if (kbhit()) {
                if ((ch = getch()) == 27) {
                  fprintf(stderr, "\nInterrupted by user");
                  done = 1; break;
                }
                ungetch(ch);
              }

              /* Ensure we don't download too many articles */
              if ((max_msgs >= 0) && (total >= max_msgs)) {
                fprintf(stderr, "\nTotal articles limit reached");
                done = 1; break;
              }

              /* Save the offset of this article */
              art_offset = ftell(f);

              fprintf(stderr,
                      "%cReceiving article %ld (%ld remaining)",
                      eol_status, msg_num + 1, last - msg_num - 1); clreol();

              /* Download the article */
              success = nntp_article(nntp_sock, f, msg_num + 1,
                                     max_lines, eol_header);

              /* Flush the output buffer */
              fflush(f);

              /* Check the free space left on disk */
              getdfree(toupper(*ifile)-'A'+1, &dfree);
              freespace = ((long)dfree.df_avail *
                           (long)dfree.df_bsec *
                           (long)dfree.df_sclus);

              /* Don't store if no space left */
              if (freespace < min_free) {
                fprintf(stderr,
                        "\nIgnoring article %ld - no free space\n",
                        msg_num + 1);
                success = 0; done = 1;
              }

              /* Don't store if an error occurred */
              else if (!success) {
                fprintf(stderr, "%cError during article %ld\n",
                        eol_status, msg_num + 1);
              }

              /* Increment the message number */
              msg_num++;

              if (success) {
                /* Increment the download count */ 
                download++; total++;
                /* Update the newsrc counter if successful */
                update = msg_num;
              }
              else {
                /* Rewind if we started storing */
                if (ftell(f) > art_offset) {
                  chsize(fileno(f), art_offset);
                }
              }

              /* Check if the socket has been closed. */
              if (!sock_established(nntp_sock)) {
                done = 1; break;
              }        

            }

            /* Update the index if we stored any articles */
            if (download > 0) {
              if (fprintf(idx, "N %08ld %s\n",
                          grp_offset, group) < 0) {
                fprintf(stderr, "write(%s): Disk full\n", idx);
              }
            }

            /* Update the newsrc information */
            sprintf(line, "%s: 0-%ld", group, update);

            fprintf(stderr, "\n");
          }
          else if (msg_num > last) {
            fprintf(stderr, "%s lost %ld articles (%ld-%ld)\n",
                    group, msg_num - last, last + 1, msg_num);

            /* Update the newsrc information */
            sprintf(line, "%s: 0-%ld", group, last);
          }
          else {
            fprintf(stderr, "%s has no new articles\n", group);
          }
        }
      }

      /* Allow blank lines */
      else if (*line) {
        fprintf(stderr, "Invalid newsrc syntax: %s\n", line);
      }

      /* Store the newsrc information, removing blank lines */
      if (*line) {
        if (fprintf(orc, "%s\n", line) < 0) {
          fprintf(stderr, "write(%s): Disk full\n", newrc);
        }
      }
    }

    /* Close all the open files */
    success = 1;
    fclose(irc);

    if (fclose(orc)) {
      fprintf(stderr, "write(%s): Disk full\n", newrc);
      success = 0;
    }

    if (fclose(f)) {
      fprintf(stderr, "write(%s): Disk full\n", "incoming.nws");
      success = 0;
    }

    if (fclose(idx)) {
      fprintf(stderr, "write(%s): Disk full\n", "incoming.idx");
      success = 0;
    }

    if (success) {
      /* Remove the old newsrc file */
      if (remove(oldrc)) {
        fprintf(stderr, "remove(%s): %s\n",
                oldrc, strerror(errno));
      }
      else {
        /* Update the newsrc file */
        if (rename(newrc, oldrc)) {
          fprintf(stderr, "rename(%s, %s): %s\n",
                  newrc, oldrc, strerror(errno));
        }
      }
    }
  }

nntp_close:

  /* Close down the connection */
  if (nntp_sock) {
    nntp_shutdown(nntp_sock);
    fprintf(stderr, "Disconnected from NNTP at %s\n", nntpserver);
  }

  return 1;
}

/* End of lxmta.c */
