/*
  imaplib.c -- Waterloo TCP IMAP client libraries.

  Copyright (C) 1999  Rod Whitby

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be 
  useful, but WITHOUT ANY WARRANTY; without even the implied
  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  USA.

  $Source: A:/SRC/TCP/MTA/RCS/IMAPLIB.C $
  $Id: IMAPLIB.C 1.9 2000/04/09 07:02:50 rwhitby Exp $
*/

#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "socklib.h"

#define SOCK_CHECK_OK(PROTOCOL) \
  if (strncmp(Sock_buffer, "IMAP OK", 7)) { \
    SOCK_LOGOUT(PROTOCOL); \
  }

#define SOCK_COMMAND(PROTOCOL) \
  strcpy(Sock_last, Sock_buffer); \
  SOCK_PUTS(PROTOCOL); \
  SOCK_GETS(PROTOCOL); \
  if (!strcmp(Sock_buffer, Sock_last)) { \
    SOCK_GETS(PROTOCOL); \
  } \

#define SOCK_LOGOUT(PROTOCOL) \
  fprintf(stderr, "\n" #PROTOCOL "> Logout: %s\n", \
          Sock_buffer); \
  sprintf(Sock_buffer, "IMAP LOGOUT"); \
  SOCK_PUTS(PROTOCOL); \
  SOCK_GETS(PROTOCOL); \
  sock_close(PROTOCOL##_sock); \
  return 0;

/* Global Macros */

static int IMAP_stat;

/*-------------------------------------------------------------
 -                                                            <
 -  IMAP Stuff                                                 <
 -                                                            <
/------------------------------------------------------------*/

tcp_Socket * imap_init(char *proxy, char *host, int port)
{
  tcp_Socket *IMAP_sock=NULL;
  char *p;

  IMAP_sock=(tcp_Socket *)malloc(sizeof(tcp_Socket));

  if (!socklib_connect( "IMAP", IMAP_sock, &IMAP_stat, proxy, host, port)) {
    fprintf(stderr, "IMAP> Unable to connect to %s", host);
    return 0;
  }

  if (proxy) {
    if ((p = strstr(Sock_buffer, "* OK ")) != NULL) {
      strcpy(Sock_buffer, p);
    }
    else {
      SOCK_GETS(IMAP);
    }
  }
  else {
    SOCK_GETS( IMAP );
  }

  if ( strncmp(Sock_buffer, "* OK", 4 ) ) {
    SOCK_LOGOUT(IMAP);
  }

  SOCK_READ_ERR(IMAP, );

  return IMAP_sock;
}

int imap_login(tcp_Socket *IMAP_sock, char *userid, char *password)
{

  sprintf( Sock_buffer, "IMAP LOGIN %s %s", userid, password);
  SOCK_COMMAND ( IMAP );
  while (Sock_buffer[0] == '*') {
    SOCK_GETS(IMAP);
  }
  SOCK_CHECK_OK( IMAP );

  SOCK_READ_ERR(IMAP, );

  return 1;
}

int imap_select(tcp_Socket *IMAP_sock, char *mailbox, long *count)
{
  long result;
  char string[7];

  sprintf( Sock_buffer, "IMAP SELECT %s", mailbox);
  SOCK_COMMAND ( IMAP );
  while (*Sock_buffer == '*') {
    if ((sscanf(Sock_buffer, "* %lu %7s", &result, &string) == 2) &&
        !strcmp(string, "EXISTS")) {
      *count = result;
    }
    SOCK_GETS ( IMAP );
  }
  SOCK_CHECK_OK(IMAP);

  SOCK_READ_ERR(IMAP, );

  return 1;
}

int imap_search(tcp_Socket *IMAP_sock, char *criteria, char **result)
{

  sprintf( Sock_buffer, "IMAP SEARCH %s", criteria);
  SOCK_COMMAND ( IMAP );
  *result = 0;
  while (*Sock_buffer == '*') {
    if (!strncmp(Sock_buffer, "* SEARCH", 8) &&
        (strlen(Sock_buffer) > 8)) {
      *result = strdup(trim(Sock_buffer+8));
    }
    SOCK_GETS ( IMAP );
  }
  SOCK_CHECK_OK(IMAP);

  SOCK_READ_ERR(IMAP, );

  if (*result && strlen(*result)) return 1;

  return 0;
}

long imap_length(tcp_Socket *IMAP_sock, long msg_num, long *size)
{
  long number;
  long result;

  sprintf(Sock_buffer, "IMAP FETCH %lu RFC822.SIZE", msg_num);
  SOCK_COMMAND ( IMAP );
  while (*Sock_buffer == '*') {
    if ((sscanf(Sock_buffer, "* %lu FETCH (RFC822.SIZE %lu)",
                &number, &result) == 2) &&
        (number == msg_num)) {
      *size = result;
    }
    SOCK_GETS ( IMAP );
  }
  SOCK_CHECK_OK(IMAP);

  SOCK_READ_ERR(IMAP, );

  return *size;
}

static int imap_start_article(FILE *f)
{
  /* MMDF format */
  fprintf(f, "\001\001\001\001\n");
  return;
}

static int imap_end_article(FILE *f)
{
  /* MMDF format */
  fprintf(f, "\001\001\001\001\n");
  return;
}

int imap_retrieve(tcp_Socket *IMAP_sock, FILE *f, long msg_num,
                  long body_lines, char eol)
{
  int in_body = 0;
  int flush = 0;
  int status = 1;
  long number, result;
  long characters = 0;

  if (body_lines == 0) { 
    sprintf( Sock_buffer, "IMAP FETCH %lu RFC822.HEADER", msg_num);
  }
  else {
    sprintf( Sock_buffer, "IMAP FETCH %lu RFC822", msg_num);
  }

  SOCK_COMMAND ( IMAP );
  if ((sscanf(Sock_buffer, "* %lu FETCH (%*s {%lu}",
              &number, &result) == 2) &&
      (number == msg_num)) {
    characters = result;
  }
  else {
    SOCK_LOGOUT(IMAP);
  }    

  imap_start_article(f);

  while (characters > 0) {
    SOCK_GETS( IMAP );
    characters -= strlen(Sock_buffer) + 2;
    if (flush) continue;
    if (in_body) {
      if (body_lines == 0) {
        fprintf(f, "[Truncated]\n");
        flush = 1;
        continue;
      }
      body_lines--;
    }
    else if (*Sock_buffer == 0) {
      in_body = 1;
    }
    else if (eol && !strncmp(Sock_buffer, "From:", 5)) {
      fprintf(stderr, "%c%s", eol, Sock_buffer); clreol();
    }
    else if (eol && !strncmp(Sock_buffer, "Subject:", 8)) {
      fprintf(stderr, "%c%s", eol, Sock_buffer); clreol();
    }
    if (fprintf(f, "%s\n", Sock_buffer) < 0) {
      flush = 1;
      status = 0;
    }
  }

  SOCK_GETS( IMAP );

  SOCK_GETS( IMAP );
  SOCK_CHECK_OK(IMAP);

  SOCK_READ_ERR(IMAP, imap_end_article(f));

  imap_end_article(f);

  return status;
}

int imap_delete(tcp_Socket *IMAP_sock, long msg_num)
{
  sprintf( Sock_buffer, "IMAP STORE %lu +FLAGS.SILENT (\\Deleted)", msg_num );
  SOCK_COMMAND ( IMAP );
  while (Sock_buffer[0] == '*') {
    SOCK_GETS(IMAP);
  }
  SOCK_CHECK_OK( IMAP );

  SOCK_READ_ERR(IMAP, );

  return 1;
}

int imap_shutdown(tcp_Socket *IMAP_sock)
{
  if (sock_established(IMAP_sock)) {
    sprintf(Sock_buffer, "IMAP CLOSE");
    SOCK_COMMAND( IMAP );
    SOCK_CHECK_OK(IMAP);
    sprintf(Sock_buffer, "IMAP LOGOUT");
    SOCK_COMMAND( IMAP );
    sock_close( IMAP_sock );
  }
  else {
    sock_abort( IMAP_sock );
  }
  SOCK_READ_ERR(IMAP, sock_abort(IMAP_sock));
  return 1;
}

/* End of imaplib.c */
