/*
  http.c -- HTTPD server for the HP200LX palmtop.

  Copyright (C) 1999  Rod Whitby
  Copyright (C) 1999  Steven Lawson

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied
  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  USA.

  $Source: A:/SRC/TCP/INETD/RCS/HTTP.C $
  $Id: HTTP.C 1.9 2000/04/09 07:15:52 rwhitby Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <conio.h>
#include <sys/stat.h>

#include "lxinetd.h"
#include "http.h"

// the domain configuration file
static const char *domain_file="http.dom";
static const char *mime_file  ="http.typ";

// http status strings (sock_puts adds the final \n)
static const char *http_document  ="HTTP/1.0 200 OK\r\n";
static const char *http_uselocal  ="HTTP/1.0 304 Not Modified\r\n";
static const char *http_nopath    ="HTTP/1.0 403 Forbidden\r\n\r\n"
    "<HEAD><TITLE>Access Forbidden</TITLE></HEAD>\r\n"
    "<BODY><H1>Access Forbidden</H1>\r\n"
    "You are not authorised to access the requested URL /%s on this server.<P>\r\n"
    "</BODY>\r\n\r";
static const char *http_notfound  ="HTTP/1.0 404 Not Found\r\n\r\n"
    "<HEAD><TITLE>File Not Found</TITLE></HEAD>\r\n"
    "<BODY><H1>File Not Found</H1>\r\n"
    "The requested URL /%s was not found on this server.<P>\r\n"
    "</BODY>\r\n\r";
static const char *http_nomemory  ="HTTP/1.0 500 Internal Server Error\r\n\r\n"
    "<HEAD><TITLE>Internal Server Error</TITLE></HEAD>\r\n"
    "<BODY><H1>Internal Server Error</H1>\r\n"
    "The server encountered an internal error while processing your request.<P>\r\n"
    "</BODY>\r\n\r";
static const char *http_badrequest="HTTP/1.0 501 Not Implemented\r\n\r\n"
    "<HEAD><TITLE>Method Not Implemented</TITLE></HEAD>\r\n"
    "<BODY><H1>Method Not Implemented</H1>\r\n"
    "The requested method is not implemented on this server.<P>\r\n"
    "</BODY>\r\n\r";

// structure to hold domain information
struct domain {
   struct domain *next;			// link to next domain
   char *host_name;			// virtual hostname
   char *base_path;			// base path for this domain
};

// structure to hold content type information
struct content {
   struct content *next;		// link to next content
   char *content_type;			// content type
   char *extension_list;		// file extension
};

// structure to handle a connection
struct connection {
   char state;				// state of connection
   char *host;				// virtual hostname
   char *path;				// requested path
   char *when;				// browsers local copy date/time
   FILE *f;				// in-transit file
};

// connection structure states
#define CS_METHOD    1
#define CS_GETHEADER 2
#define CS_PUTHEADER 3
#define CS_PUTDATA   4

// globals
static struct domain *domains=NULL;
static char *default_path=NULL;
static struct content *contents=NULL;
static char *default_type=NULL;

// prototypes
static void resolve_request(struct connection *, char *);
static char *date_string(time_t);
static int modified(char *, struct tm *);

int http_boot(void)
{
  FILE *f;
  char *line, *cpa, *cpb;
  struct domain *dom;
  struct content *typ;

  // get the timezone via the DOS "SET TZ=ZZZnZZZ" environment setting
  if (!getenv("TZ")) {
    fprintf(stderr, "http: Error: DOS timezone not set, *very* important to set it right\n");
    return 0;
  }
  tzset();

  line = (char *)buffer;

  // read in the domains (format: hostname=path)
  if ((f = fopen(domain_file, "r")) == NULL) {
    fprintf(stderr, "http: Warning: domain file (%s) not found\n", domain_file);
  }
  else {
    line[sizeof(buffer)-1] = 0;
    while (fgets(line, sizeof(buffer)-2, f)) {
      rip(line);
      if (line[0] == 0 || line[0] == ';') continue;
      cpa = strchr(line, '=');
      if (cpa == NULL) continue;
      *cpa++ = 0;
      cpb = strchr(cpa, 0); cpb--;
      if (*cpb != '\\') {
        *(++cpb) = '\\';
        *(++cpb) = 0;
      }
      if (line[0]) {
        // hostname was specified, link it to the list
        dom = (struct domain *) malloc(sizeof(struct domain));
        if (dom == NULL) {
          fprintf(stderr, "http: Error: out of memory\n");
          exit(1);
        }
        dom->host_name = strdup(line);
        dom->base_path = strdup(cpa);
        dom->next = domains;
        domains = dom;
        fprintf(stderr, "http: Info: \"%s\" maps to \"%s\"\n",
                dom->host_name, dom->base_path);
      } else {
        // hostname missing, this is the default path for pre-1.1 browsers
        default_path = strdup(cpa);
        fprintf(stderr, "http: Info: default path is \"%s\"\n", default_path);
      }
    }
    fclose(f);
  }
  if ((domains == NULL) && (default_path == NULL)) {
    fprintf(stderr, "http: Warning: no domains in %s were found\n", domain_file);
  }

  // verify the mime types file is there and can be accessed
  if ((f = fopen(mime_file, "r")) == NULL) {
    fprintf(stderr, "http: Warning: mime types file (%s) not found\n", mime_file);
  }
  else {
    line[sizeof(buffer)-1] = 0;
    while (fgets(line, sizeof(buffer)-3, f)) {
      rip(line);
      if (line[0] == 0 || line[0] == ';') continue;
      cpa = strchr(line, '=');
      if (cpa == NULL) continue;
      *cpa++ = 0;
      if (cpa[0]) {
        // extensions were specified, link it to the list
        typ = (struct content *) malloc(sizeof(struct content));
        if (typ == NULL) {
          fprintf(stderr, "http: Error: out of memory\n");
          exit(1);
        }
        typ->content_type = strdup(line);
        *(--cpa) = ' '; strcpy(strchr(cpa, 0), " ");
        typ->extension_list = strdup(cpa);
        typ->next = contents;
        contents = typ;
        fprintf(stderr, "http: Info: \"%s\" maps to \"%s\"\n",
                typ->content_type, typ->extension_list);
      } else {
        // extensions missing, this is the default content type
        default_type = strdup(line);
        fprintf(stderr, "http: Info: default content type is \"%s\"\n", default_type);
      }
    }
    fclose(f);
  }
  if ((contents == NULL) && (default_type == NULL)) {
    fprintf(stderr, "http: Warning: no contents in %s were found\n", mime_file);
  }

  return 1;
}

int http_conn(tcp_Socket *socket, void **datap)
{
  struct connection *con;

  sock_mode(socket, TCP_MODE_ASCII);

  con=(struct connection *) malloc(sizeof(struct connection));
  if (con==NULL) return 0;

  con->state=CS_METHOD;

  con->host=con->path=con->when=NULL;

  con->f=NULL;

  *datap = con;

  return 1;
}

int http_tick(tcp_Socket *socket, void **datap)
{
  int cc;
  char *line, *cpa;
  struct connection *con = *datap;

  switch(con->state) {
    case CS_METHOD:                     // waiting for request
      cc=sock_dataready(socket);
      if (cc <= 0) return (cc == 0);
      cc=sock_gets(socket, buffer, sizeof(buffer));
      if (cc <= 0) return 0;
      line = (char *)buffer;
      if (strncmpi(line, "GET ", 4)) {
        // bad method, return error and close connection
        sock_puts(socket, (byte *)http_badrequest);
        return 0;
      }
      // pluck the path for later processing
      for (cpa=&line[4] ; *cpa && *cpa!=' ' ; cpa++);
      *cpa='\0';
      // skip leading slashes (virtual path starts with one)
      for (cpa=&line[4] ; *cpa==' ' || *cpa=='/' ; cpa++);
      con->path=strdup(cpa);
      if (con->path==NULL) {
        // no memory, return error and close connection
        sock_puts(socket, (byte *)http_nomemory);
        return 0;
      }
      con->state++;
      break;
    case CS_GETHEADER:                  // reading in header
      cc=sock_dataready(socket);
      if (cc <= 0) return (cc == 0);
      cc=sock_gets(socket, buffer, sizeof(buffer));
      if (cc < 0) return 0;
      line = (char *)buffer;
      // just look for Host:, If-Modified-Since:, or blank line
      if (line[0] == 0) {
        con->state++;
        return 1;
      }
      if (!strncmpi(line, "Host:", 5)) {
        for (cpa=&line[5] ; *cpa==' ' ; cpa++);
        if (con->host==NULL) con->host=strdup(cpa);
        if (con->host==NULL) {
          // no memory, return error and close connection
          sock_puts(socket, (byte *)http_nomemory);
          return 0;
        }
        break;
      }
      if (!strncmpi(line, "If-Modified-Since:", 18)) {
        for (cpa=&line[18] ; *cpa==' ' ; cpa++);
        // don't worry if this strdup fails, no biggie..
        if (con->when==NULL) con->when=strdup(cpa);
      }
      break;
    case CS_PUTHEADER:                  // writing out header
      resolve_request(con, (char *)buffer);
      sock_puts(socket, buffer);
      if (con->f==NULL) return 0;
      // set socket back to raw for file transfer
      sock_mode(socket, TCP_MODE_BINARY);
      con->state++;
      break;
    case CS_PUTDATA:                    // writing out file
      cc=sock_tbleft(socket);
      if (cc > 100) {
        if (cc>sizeof(buffer)) cc=sizeof(buffer);
        cc=fread((char *)buffer,1,cc,con->f);
        if (cc <= 0) {
           fclose(con->f); con->f=NULL;
           return 0;
        } else {
          sock_write(socket, buffer, cc);
        }
      }
      break;
  }
  return 1;
}

int http_quit(tcp_Socket *socket, void **datap)
{
  struct connection *con = *datap;

  if (con->host) free(con->host);
  if (con->path) free(con->path);
  if (con->when) free(con->when);

  if (con->f) fclose(con->f);

  free(con);

  *datap = NULL;

  (void)socket;

  return 1;
}

// parse all the junk we collected and do one of the following:
//    write an error to buffer
//    write a "uselocal" status to buffer
//    open the file and write a full status/mime header to buffer
static void resolve_request(struct connection *con, char *buffer) {
  const char *mheader1="Date: %s\r\nServer: %s/%d.%d%s\r\n";
  const char *mheader2="Content-Type: %s\r\n";
  const char *mheader3="Last-Modified: %s\r\n\r";
  char *cp, *pp=NULL;
  char *ctype=NULL;
  char search[6];
  char found;
  struct domain *dom;
  struct content *typ;
  struct stat st;

  if (con->host==NULL) {
    pp=default_path;
  } else {
    for (dom=domains ; dom ; dom=dom->next) {
      if (!strcmp(con->host, dom->host_name)) {
        pp=dom->base_path;
        break;
      }
    }
    if (pp==NULL) {
       fprintf(stderr, "Warning: no map for domain '%s'\n", con->host);
       pp=default_path;
    }
  }
  if (pp==NULL) {
    // return an error, we didn't find a base path
    strcpy(buffer, http_nopath);
  } else {
    for (cp=con->path ; *cp ; cp++)
      if (*cp=='/') *cp='\\';
    sprintf(buffer,"%s%s", pp, con->path);
    cp=buffer+strlen(buffer)-1;
    if (*cp == '\\') *cp='\0';
    found=1;
    if ((stat(buffer, &st)==-1)) found=0;
    if (found && (st.st_mode & S_IFDIR)) {
      cp=buffer+strlen(buffer)-1;
      if (*cp++ != '\\') *cp++='\\';
      strcpy(cp, "index.htm");
      if ((stat(buffer, &st)==-1)) found=0;
    }
    if (found && (con->f=fopen(buffer,"rb"))==NULL) found=0;
    if (!found) {
      fprintf(stderr, "Error: \"%s\" not found\n", buffer);
      // return an error, file can't be opened
      sprintf(buffer, http_notfound, con->path);
    }
    else {

      /* determine the proper mime type for the file */
      if (default_type) ctype=default_type;
      if ((cp=strchr(buffer, (int) '.'))!=NULL) {
        sprintf(search," %.3s ", ++cp);
        strupr(search);
        for (typ=contents ; typ ; typ=typ->next) {
          if (strstr(typ->extension_list, search)) {
            ctype=typ->content_type;
            break;
          }
        }
      }

      if (con->when &&
          !modified(con->when, gmtime(&st.st_mtime))) {
        fprintf(stderr, "Info: \"%s\" (%s) not modified\n", buffer,
                ctype ? ctype : "unknown/unknown");
        strcpy(buffer, http_uselocal);
        fclose(con->f);
        con->f=NULL;
      } else {
        fprintf(stderr, "Info: \"%s\" (%s) requested\n", buffer,
                ctype ? ctype : "unknown/unknown");
        strcpy(buffer, http_document);
      }

      sprintf(buffer+strlen(buffer), mheader1, date_string(time(NULL)),
              "LXVWS", VERSION/10, VERSION%10, (RELEASE > 0) ? "beta" : "");

      if (ctype) {
        sprintf(buffer+strlen(buffer), mheader2, ctype);
      }

      sprintf(buffer+strlen(buffer), mheader3, date_string(st.st_mtime));
    }
  }
}

static char *date_string(time_t time)
{
  static char date[30];

  /*  012345678901234567890123456789 */
  /* "Fri, 13 Sep 1986 00:00:00 GMT" */

  strftime(date, 30, "%a, %d %b %Y %H:%M:%S GMT",
           gmtime(&time));

  return date;
}

static int modified(char *remote, struct tm *file)
{
  char daystr[3], monstr[3], tzstr[3];
  int mday, mon, year, hour, min, sec;
  char *months[] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

  /* "Fri, 13 Sep 1986 00:00:00 GMT" */
  if ((sscanf(remote, "%3s, %2d %3s %4d %2d:%2d:%2d %3s",
              daystr, &mday, monstr, &year,
              &hour, &min, &sec, tzstr) != 8) ||
      strcmp(tzstr, "GMT")) {
    return 1;
  }
  year -= 1900;

  for (mon = 0; mon < 12; mon++) {
    if (!strcmp(months[mon], monstr)) break;
  }
  if (mon >= 12) return 1;

  if (file->tm_year > year) return 1;
  if (file->tm_year < year) return 0;
  if (file->tm_mon  > mon)  return 1;
  if (file->tm_mon  < mon)  return 0;
  if (file->tm_mday > mday) return 1;
  if (file->tm_mday < mday) return 0;
  if (file->tm_hour > hour) return 1;
  if (file->tm_hour < hour) return 0;
  if (file->tm_min  > min)  return 1;
  if (file->tm_min  < min)  return 0;
  if (file->tm_sec  > sec)  return 1;
  return 0;
}

/* End of lxvws.c */
